/****************************************************************************\
 * TCPStandaloneProcessing.pjt                                              *
 *                                                                          *
 * Written by David Bell                                                    *
 *               03/12/2001                                                 *
 *      Modified 06/13/2003  by Chad Courtney                               *
 *                                                                          *
 *		Modified 06/29/2005  by Harry Thompson                              *
 *                                                                          *
 *                           csl_timerhal.h in CCS 3.0 was modified and a   *
 *                           parameter was added to control CTL bit. Added  *
 *                           parameter TIMER_CTL_SPND_EMURUN to             *
 *                           TimerControl. The lines:                       *
 *                                                                          * 
 *   static Uint32 TimerControl = TIMER_CTL_RMK(                            *
 *   TIMER_CTL_INVINP_NO,                                                   *
 *                                                                          *
 *                           were changed to:                               *
 *                                                                          *
 *  static Uint32 TimerControl = TIMER_CTL_RMK(                             *
 *   TIMER_CTL_SPND_EMURUN,                                                 *
 *   TIMER_CTL_INVINP_NO,                                                   *
 *                                                                          *
 *                                                                          *
 *                                                                          *
 * The purpose of this project is to demonstrate how to use the Turbo Co-   *
 * Processor (TCP) using the Chip Support Library (CSL).                    *
 *                                                                          *
 * User data, residing in the tcpuserx.asm files included with this project *
 * is processed through the TCP in Standalone Processing Mode. The TCP is   *
 * configured based on the channel parameters set in tcp_parameters.h. This *
 * software handles the TCP, EDMA, and interrupt configuration necessary to *
 * process each user channel.                                               *
\****************************************************************************/

#include <csl.h>
#include <csl_tcp.h>
#include <csl_edma.h>
#include <csl_emifa.h>
#include <csl_timer.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tcp_parameters.h"

/* Prototype declarations */
void intConfig(void);
Int32 checkResults(Uint32 *actual, Uint32 *reference, Uint32 size,
                   Uint32 nbValidBitsInLastWord,
                   Uint32 actualNbIterations, Uint32 refNbIterations);
interrupt void edmaIsr(void);
void submitEdma(TCP_UserData *userData, TCP_Params *tcpParameters,
                   Uint16 *interleaverTable, Uint32 *hardDecisions,
                   Uint32 *outParms);

/* Global variable declarations */
Int32 tcpDone;                   /* TCP processing completion flag          */ 
int tcc = -1;                    /* Transfer completion code used by EDMA   */
EDMA_Handle hEdmaRevt,           /* EDMA channel used for Hard Decisions    */
            hEdmaXevt,           /* EDMA channel used for IC Values         */
            hEdmaSysPar,         /* EDMA channel used for Systematics and   */
                                 /*      Parities data                      */
            hEdmaInter,          /* EDMA channel used for Interleaver data  */
            hEdmaOutPar,         /* EDMA channel used for Output Parameters */
            hEdmaNull;           /* EDMA NULL channel used for termination  */
            
static TIMER_Handle hTimer;

static Uint32 TimerControl = TIMER_CTL_RMK(
  TIMER_CTL_SPND_EMURUN,			/* added 6/29/05 by HHT */
  TIMER_CTL_INVINP_NO,
  TIMER_CTL_CLKSRC_CPUOVR8,
  TIMER_CTL_CP_PULSE,
  TIMER_CTL_HLD_YES,
  TIMER_CTL_GO_NO,
  TIMER_CTL_PWID_ONE,
  TIMER_CTL_DATOUT_0,
  TIMER_CTL_INVOUT_NO,
  TIMER_CTL_FUNC_GPIO
); 

int cnts[NUM_USERS];

/****************************************************************************\
 * main function                                                            *
\****************************************************************************/
void
main(void)
{
volatile Uint32 error = 0;
Uint32 *hardDecisions, *outParms, numHd, validHdLast;
TCP_Params tcpParms;
int i,cnt; 

    /* Initialize Chip Support Library */
    CSL_init();
    
    /* Initialize the EDMA interrupt */
 	intConfig();

	printf("running TCP test \n");
      /* TIMER Put in for Profiling of TCP Proc */
      hTimer = TIMER_open(TIMER_DEV1, TIMER_OPEN_RESET);
      TIMER_configArgs(hTimer,
    	TimerControl, /* use predefined control value  */
    	0xffffffff,   /* set period                    */
    	0x00000000    /* start count value at zero     */
  	  );
  	  
    /* Cycle through user channels */
    for(i = 0; i < NUM_USERS; i++){
      /* Clear TCP completion flag */
      tcpDone = 0;

      /* Get parameters for current user channel */
      tcpParms = tcpParameters[i];
      
      /* Calculate all necessary parameters */
      TCP_calcSubBlocksSA(&tcpParms);         /* Calculate parameters       */
      TCP_calcCountsSA(&tcpParms);            /* Calculate counts for the   */
                                              /*   Systematics and Parities,*/
                                              /*   Interleaver Table, and   */
                                              /*   Hard Decision data       */
                                             
      /* Allocate hard decisions and output parameters memory, aligned on   */
      /* double word boundaries                                             */
      numHd = tcpParms.numHd;
      validHdLast = numHd % 4;
      hardDecisions = (Uint32 *)memalign(2*sizeof(Uint32),
                                                        numHd*sizeof(Uint32));
      outParms = (Uint32 *)memalign(2*sizeof(Uint32),
                                                   TCP_NUM_OP*sizeof(Uint32));

      /* Program EDMA to service User Channel */
      submitEdma(userData[i], &tcpParms,
                 interleaverTable[i], hardDecisions, outParms);

      TIMER_setCount(hTimer,0x00000000);        
      TIMER_start(hTimer);
      
      /* Start the TCP to begin the EDMA transfers */
      TCP_start();

      /* Wait on turbo decode to complete. The task pends on a semaphore    */
      /* that is set in the EDMA interrupt service routine.                 */
	  while(!tcpDone);
	  
      /* Stop Timer for Profiling TCP done and HD + Out transfer completed  */     
      cnt = TIMER_getCount(hTimer);
      cnts[i] = cnt;
      
      printf("Timer for FL = %d, CPU cycles = %d (8 cycle accuracy due to timer) \n", tcpParms.frameLen, cnt*8);                            

      /* Verify the coprocessing results by comparing received decisions and*/
      /* output parameters to programmed reference data. A value of "0"     */
      /* indicates that no error was present. A non-zero value indicates an */
      /* error occurred.                                                    */

      error = checkResults(hardDecisions, referenceHd[i], numHd,
                     validHdLast, *outParms, *referenceOutParms[i]) ;
	  if (error) break;

      /* Free memory spaces back to the system */
	  free(hardDecisions);
	  free(outParms);
	} /* end for i */
	
	while(1);
	
} /* end main */ 


/****************************************************************************\
 * intConfig: Configure the CPU interrupt controller to receive interrupts  *
 * from the EDMA.                                                           *
\****************************************************************************/
void
intConfig(void)
{
    IRQ_resetAll();                       /* Reset all maskable interrupts  */
  	IRQ_enable(IRQ_EVT_EDMAINT);          /* Enable EDMA -> CPU interrupt   */
    IRQ_nmiEnable();                      /* Enable non-maskable interrupt  */
    IRQ_globalEnable();                   /* Globally enable all interrupts */
} /* end intConfig() */


/****************************************************************************\
 * checkResults: Verify that the hard decisions and output parameters sent  *
 * by the TCP are correct. If the hard decisions are not correct, the index *
 * of the first incorrect value is returned. If the output parameters (the  *
 * number of iterations) is not correct, then a -1 is returned. If all      *
 * values are correct then a 0 is returned.                                 *
\****************************************************************************/
Int32
checkResults(Uint32 *actual ,Uint32 *reference, Uint32 size, 
             Uint32 numValidBitsLast, Uint32 numIter,
             Uint32 refNumIter) 
{
Uint32 i;
Uint32 mismatch=0;
Uint32 lastActualWord = actual[size-1];
Uint32 lastRefWord = reference[size-1];
    
    /* Directly compare received values to reference values for 32 valid    */
    /* bits.                                                                */
    for (i=0; i<size-1; i++) {
      if (actual[i]!=reference[i]) {
        mismatch++;
        break;
      } /* end if actual != reference */
    } /* end for i */
    
    /* Compare received values to reference values on a bit-by-bit basis    */
    /* for the last decision values, if < 32.                               */
    if (!mismatch) {    
      for (i=0; i<numValidBitsLast; i++) {
        if (((lastActualWord>>i)& 0x1) != ((lastRefWord>>i)&0x1)) {
          mismatch++;
          i=size-1;
          break;
        } /* end if */
      } /* end for i */
    } /* end if !mismatch */

    if (mismatch)
      return(i+1);              /* return index for 1st error that occured. */
                                /* Index will range from 1 to framelength   */

    /* Verify the number of iterations performed by the TCP, which is       */
    /* reported by the output parameters                                    */
    else {/* (!mismatch) */
//      if (numIter!=refNumIter)
//        return(-1);              /* number of iterations run does not match */
//      else /* (numIter == refNumIter) */
        return(0); /* no error */
    } /* end else if !mismatch */
    
} /* end checkResults() */


/****************************************************************************\
 * edmaIsr: EDMA interrupt service routine. The tcpDone flag is set to one  *
 * and all EDMA handles are closed, freeing them for use on the next        *
 * iteration.                                                               *
\****************************************************************************/
interrupt void
edmaIsr(void)  
{ 
    tcpDone = EDMA_intTest(tcc); /* Signal that processing is completed     */
    EDMA_intClear(tcc);          /* Clear the interrupt in the CIPR         */
    EDMA_intFree(tcc);           /* Free the TCC value to the system        */
    EDMA_close(hEdmaRevt);       /* Close the hard decisions EDMA channel   */
    EDMA_close(hEdmaXevt);       /* Close the IC values EDMA channel        */
    EDMA_freeTable(hEdmaSysPar); /* Close the systematics and parities EDMA */
                                 /*   channel                               */
    EDMA_freeTable(hEdmaInter);  /* Close the interleaver table EDMA channel*/
    EDMA_freeTable(hEdmaOutPar); /* Close the output parameters EDMA channel*/
    EDMA_freeTable(hEdmaNull);   /* Close the NULL EDMA channel             */
 
} /* end edmaIsr */


/****************************************************************************\
 * submitEdma: The EDMA is programmed to service the user channel. The IC   *
 * values are configured and all EDMA parameters are calculated here.       *
\****************************************************************************/
void
submitEdma(TCP_UserData *userData, TCP_Params *tcpParameters,
           Uint16 *interleaverTable, Uint32 *hardDecisions,
           Uint32 *outParms)
{
TCP_UserData *xabData;
Uint16 index;
Uint16 frameCount;
Uint16 elementCount;
Uint32 frameLen = tcpParameters->frameLen;
Uint32 rate = tcpParameters->rate;
EDMA_Config edmaConfig;
TCP_ConfigIc configIc = {0,0,0,0,0,0,0,0,0,0,0,0};

    /* Calculate the TCP IC values */
    index = frameLen * rate;           /* Location of the X, A, and B data */
    xabData = &userData[index];        /* Pointer to the X, A, and B data  */
    TCP_genIc(tcpParameters, xabData, &configIc);    /* Generate IC values */
      
    /* Open handles to the EDMA channels */
    hEdmaRevt = EDMA_open(EDMA_CHA_TCPREVT, EDMA_OPEN_RESET);
    hEdmaXevt = EDMA_open(EDMA_CHA_TCPXEVT, EDMA_OPEN_RESET);
    EDMA_allocTableEx(1, &hEdmaSysPar);
    EDMA_allocTableEx(1, &hEdmaInter);
    EDMA_allocTableEx(1, &hEdmaOutPar);
    EDMA_allocTableEx(1, &hEdmaNull);
    EDMA_reset(hEdmaNull);

    /* Configure channel parameters for IC registers */
    edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                     EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_INC,
                     EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_INC,
                     EDMA_OPT_TCINT_NO,             EDMA_OPT_TCC_DEFAULT,
                     EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                     EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                     EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                     EDMA_OPT_FS_YES);

    edmaConfig.src = (Uint32)&configIc;
    edmaConfig.cnt = (Uint32)TCP_NUM_IC;
    edmaConfig.dst = (Uint32)TCP_ICMEM_ADDR;
    edmaConfig.idx = 0;
    edmaConfig.rld = 0;
    EDMA_config(hEdmaXevt, &edmaConfig);
    EDMA_link(hEdmaXevt, hEdmaSysPar);

    /* Configure channel for systematics and parities data */
    frameCount     = TCP_normalCeil((frameLen * rate), 4 * tcpParameters->numSysPar);
    elementCount   = 2 * TCP_normalCeil((frameLen * rate), (8 * frameCount));
    edmaConfig.src = (Uint32)userData;
    edmaConfig.cnt = EDMA_CNT_RMK(frameCount - 1, elementCount);
    edmaConfig.dst = (Uint32)TCP_SPMEM_ADDR;
    EDMA_config(hEdmaSysPar, &edmaConfig);
    EDMA_link(hEdmaSysPar, hEdmaInter);

    /* Configure channel parameters for interleaver data */
    frameCount     = TCP_normalCeil(frameLen, (2 * tcpParameters->numInter));
    elementCount   = 2 * TCP_normalCeil(frameLen, 4 * frameCount);
    edmaConfig.src = (Uint32)interleaverTable;
    edmaConfig.cnt = EDMA_CNT_RMK(frameCount - 1, elementCount);
    edmaConfig.dst = (Uint32)TCP_ILMEM_ADDR;
    EDMA_config(hEdmaInter, &edmaConfig);
    EDMA_link(hEdmaInter, hEdmaNull);

    /* Configure channel parameters for hard decision data */
    frameCount     = TCP_normalCeil(frameLen, (32 * tcpParameters->numHd));
    elementCount   = 2 * TCP_normalCeil(frameLen, 64 * frameCount);
    edmaConfig.src = (Uint32)TCP_HDMEM_ADDR;
    edmaConfig.cnt = EDMA_CNT_RMK(frameCount - 1, elementCount);
    edmaConfig.dst = (Uint32)hardDecisions;
    EDMA_config(hEdmaRevt, &edmaConfig);
    EDMA_link(hEdmaRevt, hEdmaOutPar);

    tcc = EDMA_intAlloc(-1);

    /* Configure channel parameters for TCP output parameters */
    edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                     EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_INC,
                     EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_INC,
                     EDMA_OPT_TCINT_YES,            EDMA_OPT_TCC_OF(tcc),
                     EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                     EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                     EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                     EDMA_OPT_FS_YES);
    edmaConfig.src = (Uint32)TCP_OPMEM_ADDR;
    edmaConfig.cnt = (Uint32)TCP_NUM_OP;
    edmaConfig.dst = (Uint32)outParms;
    EDMA_config(hEdmaOutPar, &edmaConfig);
    EDMA_link(hEdmaOutPar, hEdmaNull);
    
    /* Enable the TCC to generate a CPU interrupt */
    EDMA_intEnable(tcc);
    
    /* Enable the transmit and receive channels */
    EDMA_enableChannel(hEdmaRevt);
    EDMA_enableChannel(hEdmaXevt);

} /* end submitEdma() */
