/*
Keystone DDR configuration
Author: Brighton
Created on 2010-12-6
Updated on 2011-4-17 for TCI6608/C6678 EVM
Updated on 2011-7-12 for dual TCI6618 EVM
Updated on 2012-5-8  for C6670/TCI6618 EVM and TCI6614 EVM
*/
#include <stdio.h>
#include <csl_bootcfgaux.h>
#include "KeyStone_DDR_Init.h"
#include "KeyStone_common.h"

CSL_Emif4fRegs * DDR_Regs= (CSL_Emif4fRegs *)CSL_DDR3_EMIF_CONFIG_REGS;

void KeyStone_DDR_PLL_init (unsigned int DDR_PLLM, unsigned int DDR_PLLD)
{
	int i;
	
	if(0<DDR_PLLM&&DDR_PLLM<=8192&&0<DDR_PLLD&&DDR_PLLD<=64)
	{
		CSL_BootCfgUnlockKicker();

		boot_cfg_regs->DDR3_PLL_CTL1 = 0x00000040      /*Set ENSAT bit = 1*/
			|((DDR_PLLM-1)/2)>>8;	/*BWADJ[11:8]*/

		/*Set Bypass = 1*/
		boot_cfg_regs->DDR3_PLL_CTL0 |= (1<<23) ;

		boot_cfg_regs->DDR3_PLL_CTL1 |= 0x00002000;      //Set RESET bit = 1

		boot_cfg_regs->DDR3_PLL_CTL0 = (boot_cfg_regs->DDR3_PLL_CTL0&0x00780000)|
			(((DDR_PLLM-1)/2)<<24)|((DDR_PLLM-1)<<6)|(DDR_PLLD-1);
		for(i=0;i<1000;i++)
			asm(" nop");            //Wait >1000ns for reset to complete

		boot_cfg_regs->DDR3_PLL_CTL1 &= ~(0x00002000);   //Clear RESET bit
		for(i=0;i<10000;i++)
			asm(" nop");          //Wait >2000 clocks (>30us) for PLL lock

		/*Set Bypass = 0 (PLL Mode)*/
		boot_cfg_regs->DDR3_PLL_CTL0 &= ~(1<<23) ;
	}
}


/*setup DDR incremental leveling, can only be executed after full leveling*/
void Keystone_DDR_incremental_leveling(Uint32 incremental_interval_ms)
{
	Uint32 uiIncemental_inteval;

	uiIncemental_inteval= incremental_interval_ms;
	if(0xFF<uiIncemental_inteval)
		uiIncemental_inteval= 0xFF;

	/*set normal incremental leveling inteval*/
	DDR_Regs->RDWR_LVL_CTRL = 
		(0x7F<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLINC_PRE_SHIFT)|
		(uiIncemental_inteval<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLINC_INT_SHIFT)|
		(uiIncemental_inteval<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLGATEINC_INT_SHIFT)|
		(uiIncemental_inteval<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_WRLVLINC_INT_SHIFT);
}

/*execute DDR full leveling*/
int Keystone_DDR_full_leveling()
{
	Uint32 i;
	
	/*enable full leveling*/
	DDR_Regs->RDWR_LVL_RMP_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVL_EN_SHIFT);

	/*start full leveling*/
	DDR_Regs->RDWR_LVL_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLFULL_START_SHIFT);

	/*Read back any of the DDR3 controller registers.
	This ensures full leveling is complete because this step is executed 
	only after full	leveling completes.*/
	i= DDR_Regs->RDWR_LVL_RMP_CTRL; 	//dummy read

	//Wait 3ms for leveling to complete
	TSC_delay_ms(3); 	

	if(DDR_Regs->STATUS&(CSL_EMIF4F_STATUS_REG_RDLVLGATETO_MASK
		|CSL_EMIF4F_STATUS_REG_RDLVLTO_MASK
		|CSL_EMIF4F_STATUS_REG_WRLVLTO_MASK))
	{
		printf("DDR3 leveling has failed, STATUS = 0x%x\n", DDR_Regs->STATUS);
		return 1;
	}
	return 0;
}

/*configure DDR according to the clock speed
please note, clock_MHz is the clock speed in MHz, not data rate. 
For example, clock speed for 1333.333M data rate is 666.667MHz*/
void C6678_TCI6608_EVM_DDR_Init(float clock_MHz)
{
	int i;
	
	CSL_BootCfgUnlockKicker();

	//initial vale for leveling
	/*WRLVL_INIT_RATIO*/
	boot_cfg_regs->DDR3_CONFIG_REG[2] = 0x20;
	boot_cfg_regs->DDR3_CONFIG_REG[3] = 0x24;
	boot_cfg_regs->DDR3_CONFIG_REG[4] = 0x3A;
	boot_cfg_regs->DDR3_CONFIG_REG[5] = 0x38;
	boot_cfg_regs->DDR3_CONFIG_REG[6] = 0x51;
	boot_cfg_regs->DDR3_CONFIG_REG[7] = 0x5E;
	boot_cfg_regs->DDR3_CONFIG_REG[8] = 0x5E;
	boot_cfg_regs->DDR3_CONFIG_REG[9] = 0x5E;
	boot_cfg_regs->DDR3_CONFIG_REG[10] = 0x44;

	/*GTLVL_INIT_RATIO*/
	boot_cfg_regs->DDR3_CONFIG_REG[14] = 0xA1;
	boot_cfg_regs->DDR3_CONFIG_REG[15] = 0x9E;
	boot_cfg_regs->DDR3_CONFIG_REG[16] = 0xA7;
	boot_cfg_regs->DDR3_CONFIG_REG[17] = 0xA9;
	boot_cfg_regs->DDR3_CONFIG_REG[18] = 0xCA;
	boot_cfg_regs->DDR3_CONFIG_REG[19] = 0xBE;
	boot_cfg_regs->DDR3_CONFIG_REG[20] = 0xDD;
	boot_cfg_regs->DDR3_CONFIG_REG[21] = 0xDD;
	boot_cfg_regs->DDR3_CONFIG_REG[22] = 0xBA;
	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0xF; 	// set dll_lock_diff to 15

	/*Invert Clock Out*/
	boot_cfg_regs->DDR3_CONFIG_REG[0] &= ~(0x007FE000);  // clear ctrl_slave_ratio field
	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0x00200000;     // set ctrl_slave_ratio to 0x100
	boot_cfg_regs->DDR3_CONFIG_REG[12] |= 0x08000000;    // Set invert_clkout = 1
	
	boot_cfg_regs->DDR3_CONFIG_REG[23] |= 0x00000200; //Set bit 9 = 1 to use forced ratio leveling for read DQS

	/*the PHY_RESET is pulsed (0 -> 1 -> 0) to latch these 
	leveling configuration values into the PHY logic.*/
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 |= (0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);


	/*Drives CKE low.
	This is a JEDEC requirement that we have 500us delay between reset de-assert 
	and cke assert and then program the correct refresh rate
	The DDR internal clock is divide by 16 before SDCFG write*/
	DDR_Regs->SDRAM_REF_CTRL = 0x80000000|(unsigned int)(500.f*clock_MHz/16.f);

	DDR_Regs->SDRAM_TIM_1 =
		((unsigned int)(13.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RP_SHIFT)|
		((unsigned int)(13.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RCD_SHIFT)|
		((unsigned int)(15*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WR_SHIFT)|
		((unsigned int)(36*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RAS_SHIFT)|
		((unsigned int)(49.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RC_SHIFT)|
		((unsigned int)(45*clock_MHz/4000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RRD_SHIFT)| 	/*T_RRD = (tFAW/(4*tCK)) C 1*/
		((unsigned int)(7.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WTR_SHIFT);
	DDR_Regs->SDRAM_TIM_2   = 
		((unsigned int)(6*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XP_SHIFT)|
		((unsigned int)(120*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSNR_SHIFT)| 	/*T_XSNR = (tXS /tCK)C 1*/
		((512-1)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSRD_SHIFT)| 	/*T_XSRD =tXSDLLC 1*/
		((unsigned int)(7.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_RTP_SHIFT)|
		((unsigned int)(5.625*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_CKE_SHIFT);
	DDR_Regs->SDRAM_TIM_3   = 
		(5<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_PDLL_UL_SHIFT)| 	/*This field must always be programmed to 0x5.*/
		((5)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CSTA_SHIFT)| 	/*This field should be set according to PHY requirements as 0x5.*/
		((unsigned int)(5.625*clock_MHz/1000.f+0.9999f)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CKESR_SHIFT)|
		((64-1)<<CSL_EMIF4F_SDRAM_TIM_3_REG_ZQ_ZQCS_SHIFT)|
		((unsigned int)(110*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RFC_SHIFT)|
		(15<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RAS_MAX_SHIFT); 	/*This field must always be programmed to 0xF.*/

	DDR_Regs->DDR_PHY_CTRL_1  = 0x00100100|
		(12<<CSL_EMIF4F_DDR_PHY_CTRL_1_REG_READ_LATENCY_SHIFT); 	/*between CAS Latency + 1 and CAS Latency + 7*/

	DDR_Regs->ZQ_CONFIG = 
		((0)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS1EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS0EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_DUALCALEN_SHIFT)| 	/*This bit should always be set to 1.*/
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_SFEXITEN_SHIFT)|
		((512/256-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQINIT_MULT_SHIFT)| 	/*T_ZQ_ZQINIT_MULT = (tZQinit/tZQoper C 1)*/
		((256/64-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQCL_MULT_SHIFT)| 	/*T_ZQ_ZQCL_MULT = (tZQoper/tZQCS C 1)*/
		/*interval between ZQCS commands = 0.5%/((TSens x Tdriftrate) + (VSens x Vdriftrate))
		=0.5%/((max (dRTTdT, dRONdTM) x Tdriftrate in C/second) + (max(dRTTdV, dRONdVM) x Vdriftrate in mV/second))
		this time need be converted to refresh period number*/
		((151515151/(64000000/8192))<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_REFINTERVAL_SHIFT);

	/*map priority 0,1,2,3 to COS0,
	map priority 3,5,6,7 to COS1*/
	DDR_Regs->PRI_COS_MAP = 
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_COS_MAP_EN_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_7_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_6_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_5_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_4_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_3_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_2_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_1_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_0_COS_SHIFT);

	/*master based COS map is disabled*/
	DDR_Regs->MSTID_COS_1_MAP= 0;
	DDR_Regs->MSTID_COS_2_MAP= 0;

	/*LAT_CONFIG*/
	DDR_Regs->VBUSM_CONFIG= 
		(8<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_1_SHIFT)|
		(16<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_2_SHIFT)|
		(32<<CSL_EMIF4F_VBUSM_CONFIG_REG_PR_OLD_COUNT_SHIFT);

	DDR_Regs->ECC_CTRL = 
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_PROT_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_2_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_1_EN_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_1= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_STRT_ADDR_1_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_END_ADDR_1_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_2= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_STRT_ADDR_2_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_END_ADDR_2_SHIFT);

	/* enables DRAM configuration.  It still has the refresh interval 
	programmed to the longer number needed during DRAM initialization.*/
	DDR_Regs->SDRAM_REF_CTRL = (unsigned int)(500.f*clock_MHz/16.f); 

	DDR_Regs->SDRAM_CONFIG = 
		(3<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_TYPE_SHIFT)| 	/*Set to 3 for DDR3. All other values reserved.*/
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_POS_SHIFT)|
		(DDR_TERM_RZQ_OVER_6<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_TERM_SHIFT)|
		(DDR_DYN_ODT_DISABLED<<CSL_EMIF4F_SDRAM_CONFIG_REG_DYN_ODT_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_DISABLE_DLL_SHIFT)|
		(SDRAM_DRIVE_RZQ_OVER_7<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_DRIVE_SHIFT)|
		(DDR_CWL_7<<CSL_EMIF4F_SDRAM_CONFIG_REG_CWL_SHIFT)|
		(DDR_BUS_WIDTH_64<<CSL_EMIF4F_SDRAM_CONFIG_REG_NARROW_MODE_SHIFT)|
		(DDR_CL_9<<CSL_EMIF4F_SDRAM_CONFIG_REG_CL_SHIFT)|
		(DDR_ROW_SIZE_13_BIT<<CSL_EMIF4F_SDRAM_CONFIG_REG_ROWSIZE_SHIFT)|
		(DDR_BANK_NUM_8<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_EBANK_SHIFT)|
		(DDR_PAGE_SIZE_10_BIT_1024_WORD<<CSL_EMIF4F_SDRAM_CONFIG_REG_PAGESIZE_SHIFT);

	for(i=0;i<100000;i++)
		asm(" nop"); 	//Wait 600us for HW init to complete

//	DDR_Regs->SDRAM_REF_CTRL    = 64000000/8192/(1000/clock_MHz);
	DDR_Regs->SDRAM_REF_CTRL    = (unsigned int)64000.f*clock_MHz/8192.f;

	//enable full leveling, no incremental leveling
	/*Typically program a higher rate of incremental leveling in the ramp window*/
	DDR_Regs->RDWR_LVL_RMP_WIN = 10000/7.8; 	/*10000us*/
	DDR_Regs->RDWR_LVL_RMP_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVL_EN_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVLINC_RMP_PRE_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLINC_RMP_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLGATEINC_RMP_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_WRLVLINC_RMP_INT_SHIFT);
	DDR_Regs->RDWR_LVL_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLFULL_START_SHIFT)|
		(0x7F<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLINC_PRE_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLINC_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLGATEINC_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_WRLVLINC_INT_SHIFT);

	TSC_delay_ms(3); 	//Wait 3ms for leveling to complete

	if(DDR_Regs->STATUS&(CSL_EMIF4F_STATUS_REG_RDLVLGATETO_MASK
		|CSL_EMIF4F_STATUS_REG_RDLVLTO_MASK
		|CSL_EMIF4F_STATUS_REG_WRLVLTO_MASK))
	{
		printf("DDR3 leveling has failed, STATUS = 0x%x\n", DDR_Regs->STATUS);
		return;
	}
}

/*configure DDR according to the clock speed
please note, clock_MHz is the clock speed in MHz, not data rate. 
For example, clock speed for 1333.333M data rate is 666.667MHz*/
void Dual_TCI6618_EVM_DDR_Init(float clock_MHz)
{
	int i;
	
	Uint32 uiDSP_Num= Get_DSP_Number();

	if(0!=uiDSP_Num&&clock_MHz>533.333)
		clock_MHz= 533.333;
		
	CSL_BootCfgUnlockKicker();

	/*different DSP on the board has different trace length to the DDR device
	so, the initial leveling value is different*/
	if(0==uiDSP_Num)
	{
		//initial vale for leveling                
		/*WRLVL_INIT_RATIO*/                       
		boot_cfg_regs->DDR3_CONFIG_REG[2]  = 0x0F; 
		boot_cfg_regs->DDR3_CONFIG_REG[3]  = 0x0F; 
		boot_cfg_regs->DDR3_CONFIG_REG[4]  = 0x20; 
		boot_cfg_regs->DDR3_CONFIG_REG[5]  = 0x24; 
		boot_cfg_regs->DDR3_CONFIG_REG[6]  = 0x33; 
		boot_cfg_regs->DDR3_CONFIG_REG[7]  = 0x38; 
		boot_cfg_regs->DDR3_CONFIG_REG[8]  = 0x21; 
		boot_cfg_regs->DDR3_CONFIG_REG[9]  = 0x29; 
		boot_cfg_regs->DDR3_CONFIG_REG[10] = 0x0 ; 
                                                           
		/*GTLVL_INIT_RATIO*/                       
		boot_cfg_regs->DDR3_CONFIG_REG[14] = 0x7B; 
		boot_cfg_regs->DDR3_CONFIG_REG[15] = 0x7B; 
		boot_cfg_regs->DDR3_CONFIG_REG[16] = 0x8D; 
		boot_cfg_regs->DDR3_CONFIG_REG[17] = 0x89; 
		boot_cfg_regs->DDR3_CONFIG_REG[18] = 0xC0; 
		boot_cfg_regs->DDR3_CONFIG_REG[19] = 0xBC; 
		boot_cfg_regs->DDR3_CONFIG_REG[20] = 0xC3; 
		boot_cfg_regs->DDR3_CONFIG_REG[21] = 0xBB; 
		boot_cfg_regs->DDR3_CONFIG_REG[22] = 0x0 ; 
	}
	else
	{
		//initial vale for leveling
		/*GTLVL_INIT_RATIO*/
		boot_cfg_regs->DDR3_CONFIG_REG[22] = 0;
		boot_cfg_regs->DDR3_CONFIG_REG[21] = 0xB8;
		boot_cfg_regs->DDR3_CONFIG_REG[20] = 0xC0;
		boot_cfg_regs->DDR3_CONFIG_REG[19] = 0xBE;
		boot_cfg_regs->DDR3_CONFIG_REG[18] = 0xC2;
		boot_cfg_regs->DDR3_CONFIG_REG[17] = 0x89;
		boot_cfg_regs->DDR3_CONFIG_REG[16] = 0x8C;
		boot_cfg_regs->DDR3_CONFIG_REG[15] = 0x7B;
		boot_cfg_regs->DDR3_CONFIG_REG[14] = 0x7E;

		/*WRLVL_INIT_RATIO*/
		boot_cfg_regs->DDR3_CONFIG_REG[10] = 0;
		boot_cfg_regs->DDR3_CONFIG_REG[9]  = 0x25;
		boot_cfg_regs->DDR3_CONFIG_REG[8]  = 0x1D;
		boot_cfg_regs->DDR3_CONFIG_REG[7]  = 0x38;
		boot_cfg_regs->DDR3_CONFIG_REG[6]  = 0x34;
		boot_cfg_regs->DDR3_CONFIG_REG[5]  = 0x21;
		boot_cfg_regs->DDR3_CONFIG_REG[4]  = 0x1E;
		boot_cfg_regs->DDR3_CONFIG_REG[3]  = 0x0E;
		boot_cfg_regs->DDR3_CONFIG_REG[2]  = 0x0B;
	}
	
	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0xF; 	// set dll_lock_diff to 15

	/*Invert Clock Out*/
	boot_cfg_regs->DDR3_CONFIG_REG[0] &= ~(0x007FE000);  // clear ctrl_slave_ratio field
	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0x00200000;     // set ctrl_slave_ratio to 0x100
	boot_cfg_regs->DDR3_CONFIG_REG[12] |= 0x08000000;    // Set invert_clkout = 1
	
	boot_cfg_regs->DDR3_CONFIG_REG[23] |= 0x00000200; //Set bit 9 = 1 to use forced ratio leveling for read DQS

	/*the PHY_RESET is pulsed (0 -> 1 -> 0) to latch these 
	leveling configuration values into the PHY logic.*/
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 |= (0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);


	/*Drives CKE low.
	This is a JEDEC requirement that we have 500us delay between reset de-assert 
	and cke assert and then program the correct refresh rate
	The DDR internal clock is divide by 16 before SDCFG write*/
	DDR_Regs->SDRAM_REF_CTRL = 0x80000000|(unsigned int)(500.f*clock_MHz/16.f);

	DDR_Regs->SDRAM_TIM_1 =
		((unsigned int)(13.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RP_SHIFT)|
		((unsigned int)(13.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RCD_SHIFT)|
		((unsigned int)(15*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WR_SHIFT)|
		((unsigned int)(36*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RAS_SHIFT)|
		((unsigned int)(49.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RC_SHIFT)|
		((unsigned int)(45*clock_MHz/4000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RRD_SHIFT)| 	/*T_RRD = (tFAW/(4*tCK)) C 1*/
		(_max2(4-1, (unsigned int)(7.5*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WTR_SHIFT);
	DDR_Regs->SDRAM_TIM_2   = 
		(_max2(3-1, (unsigned int)(6*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XP_SHIFT)|
		(_max2(5-1, (unsigned int)((160+10)*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSNR_SHIFT)| 	/*T_XSNR = (tXS /tCK)C 1*/
		((512-1)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSRD_SHIFT)| 	/*T_XSRD =tXSDLLC 1*/
		(_max2(4-1, (unsigned int)(7.5*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_RTP_SHIFT)|
		(_max2(3-1, (unsigned int)(5.625*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_CKE_SHIFT);
	DDR_Regs->SDRAM_TIM_3   = 
		(5<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_PDLL_UL_SHIFT)| 	/*This field must always be programmed to 0x5.*/
		((5)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CSTA_SHIFT)| 	/*This field should be set according to PHY requirements as 0x5.*/
		(_max2(3, (unsigned int)(5.625*clock_MHz/1000.f+0.9999f))<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CKESR_SHIFT)|
		((64-1)<<CSL_EMIF4F_SDRAM_TIM_3_REG_ZQ_ZQCS_SHIFT)|
		((unsigned int)(160*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RFC_SHIFT)|
		(15<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RAS_MAX_SHIFT); 	/*This field must always be programmed to 0xF.*/

	DDR_Regs->DDR_PHY_CTRL_1  = 0x00100100|
		(12<<CSL_EMIF4F_DDR_PHY_CTRL_1_REG_READ_LATENCY_SHIFT); 	/*between CAS Latency + 1 and CAS Latency + 7*/

	DDR_Regs->ZQ_CONFIG = 
		((0)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS1EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS0EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_DUALCALEN_SHIFT)| 	/*This bit should always be set to 1.*/
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_SFEXITEN_SHIFT)|
		((512/256-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQINIT_MULT_SHIFT)| 	/*T_ZQ_ZQINIT_MULT = (tZQinit/tZQoper C 1)*/
		((256/64-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQCL_MULT_SHIFT)| 	/*T_ZQ_ZQCL_MULT = (tZQoper/tZQCS C 1)*/
		/*interval between ZQCS commands = 0.5%/((TSens x Tdriftrate) + (VSens x Vdriftrate))
		=0.5%/((max (dRTTdT, dRONdTM) x Tdriftrate in C/second) + (max(dRTTdV, dRONdVM) x Vdriftrate in mV/second))
		this time need be converted to refresh period number*/
		((151515151/(64000000/8192))<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_REFINTERVAL_SHIFT);

	/*map priority 0,1,2,3 to COS0,
	map priority 3,5,6,7 to COS1*/
	DDR_Regs->PRI_COS_MAP = 
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_COS_MAP_EN_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_7_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_6_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_5_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_4_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_3_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_2_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_1_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_0_COS_SHIFT);

	/*master based COS map is disabled*/
	DDR_Regs->MSTID_COS_1_MAP= 0;
	DDR_Regs->MSTID_COS_2_MAP= 0;

	/*LAT_CONFIG*/
	DDR_Regs->VBUSM_CONFIG= 
		(8<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_1_SHIFT)|
		(16<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_2_SHIFT)|
		(32<<CSL_EMIF4F_VBUSM_CONFIG_REG_PR_OLD_COUNT_SHIFT);

	DDR_Regs->ECC_CTRL = 
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_PROT_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_2_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_1_EN_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_1= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_STRT_ADDR_1_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_END_ADDR_1_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_2= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_STRT_ADDR_2_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_END_ADDR_2_SHIFT);

	/* enables DRAM configuration.  It still has the refresh interval 
	programmed to the longer number needed during DRAM initialization.*/
	DDR_Regs->SDRAM_REF_CTRL = (unsigned int)(500.f*clock_MHz/16.f); 

	DDR_Regs->SDRAM_CONFIG = 
		(3<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_TYPE_SHIFT)| 	/*Set to 3 for DDR3. All other values reserved.*/
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_POS_SHIFT)|
		(DDR_TERM_RZQ_OVER_6<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_TERM_SHIFT)|
		(DDR_DYN_ODT_DISABLED<<CSL_EMIF4F_SDRAM_CONFIG_REG_DYN_ODT_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_DISABLE_DLL_SHIFT)|
		(SDRAM_DRIVE_RZQ_OVER_7<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_DRIVE_SHIFT)|
		(DDR_CWL_7<<CSL_EMIF4F_SDRAM_CONFIG_REG_CWL_SHIFT)|
		(DDR_BUS_WIDTH_64<<CSL_EMIF4F_SDRAM_CONFIG_REG_NARROW_MODE_SHIFT)|
		(DDR_CL_9<<CSL_EMIF4F_SDRAM_CONFIG_REG_CL_SHIFT)|
		(DDR_ROW_SIZE_14_BIT<<CSL_EMIF4F_SDRAM_CONFIG_REG_ROWSIZE_SHIFT)|
		(DDR_BANK_NUM_8<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_EBANK_SHIFT)|
		(DDR_PAGE_SIZE_10_BIT_1024_WORD<<CSL_EMIF4F_SDRAM_CONFIG_REG_PAGESIZE_SHIFT);

	for(i=0;i<100000;i++)
		asm(" nop"); 	//Wait 600us for HW init to complete

//	DDR_Regs->SDRAM_REF_CTRL    = 64000000/8192/(1000/clock_MHz);
	DDR_Regs->SDRAM_REF_CTRL    = (unsigned int)64000.f*clock_MHz/8192.f;

#if 1
	//enable full leveling, no incremental leveling
	DDR_Regs->RDWR_LVL_RMP_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVL_EN_SHIFT);
	//start full leveling
	DDR_Regs->RDWR_LVL_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLFULL_START_SHIFT);

	TSC_delay_ms(3); 	//Wait 3ms for leveling to complete
#else
	/*Typically program a higher rate of incremental leveling in the ramp window*/
	DDR_Regs->RDWR_LVL_RMP_WIN = 50000/7.8; 	/*ramp window size= 50000us*/
	/*enable leveling, incremental leveling inteval = 100us during power ramp*/
	DDR_Regs->RDWR_LVL_RMP_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVL_EN_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVLINC_RMP_PRE_SHIFT)|
		(13<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLINC_RMP_INT_SHIFT)|
		(13<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLGATEINC_RMP_INT_SHIFT)|
		(13<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_WRLVLINC_RMP_INT_SHIFT);
	/*start leveling, incremental leveling inteval = 1ms*/
	DDR_Regs->RDWR_LVL_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLFULL_START_SHIFT)|
		(0x7F<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLINC_PRE_SHIFT)|
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLINC_INT_SHIFT)|
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLGATEINC_INT_SHIFT)|
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_WRLVLINC_INT_SHIFT);

	/*Incremental leveling of at least the read eye sample point must be 
	executed at least 64 times after full automatic leveling to converge 
	it to an initial optimum value.*/
	TSC_delay_ms(64); 	//Wait 64ms for leveling to complete

	/*reset incremental leveling inteval = 100ms to reduce overhead*/
	DDR_Regs->RDWR_LVL_CTRL = 
		(0x7F<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLINC_PRE_SHIFT)|
		(100<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLINC_INT_SHIFT)|
		(100<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLGATEINC_INT_SHIFT)|
		(100<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_WRLVLINC_INT_SHIFT);

#endif
	if(DDR_Regs->STATUS&(CSL_EMIF4F_STATUS_REG_RDLVLGATETO_MASK
		|CSL_EMIF4F_STATUS_REG_RDLVLTO_MASK
		|CSL_EMIF4F_STATUS_REG_WRLVLTO_MASK))
	{
		printf("DDR3 leveling has failed, STATUS = 0x%x\n", DDR_Regs->STATUS);
		return;
	}
}

/*configure DDR according to the clock speed
please note, clock_MHz is the clock speed in MHz, not data rate. 
For example, clock speed for 1333.333M data rate is 666.667MHz*/
void C6670_TCI6618_EVM_DDR_Init(float clock_MHz)
{
	int i;
	
	CSL_BootCfgUnlockKicker();

	//initial vale for leveling
	/*WRLVL_INIT_RATIO*/
	boot_cfg_regs->DDR3_CONFIG_REG[2]  = 0x5E;
	boot_cfg_regs->DDR3_CONFIG_REG[3]  = 0x5E;
	boot_cfg_regs->DDR3_CONFIG_REG[4]  = 0x5E;
	boot_cfg_regs->DDR3_CONFIG_REG[5]  = 0x51;
	boot_cfg_regs->DDR3_CONFIG_REG[6]  = 0x38;
	boot_cfg_regs->DDR3_CONFIG_REG[7]  = 0x3A;
	boot_cfg_regs->DDR3_CONFIG_REG[8]  = 0x24;
	boot_cfg_regs->DDR3_CONFIG_REG[9]  = 0x20;
	boot_cfg_regs->DDR3_CONFIG_REG[10] = 0x44;

	/*GTLVL_INIT_RATIO*/
	boot_cfg_regs->DDR3_CONFIG_REG[14] = 0xDD;
	boot_cfg_regs->DDR3_CONFIG_REG[15] = 0xDD;
	boot_cfg_regs->DDR3_CONFIG_REG[16] = 0xBE;
	boot_cfg_regs->DDR3_CONFIG_REG[17] = 0xCA;
	boot_cfg_regs->DDR3_CONFIG_REG[18] = 0xA9;
	boot_cfg_regs->DDR3_CONFIG_REG[19] = 0xA7;
	boot_cfg_regs->DDR3_CONFIG_REG[20] = 0x9E;
	boot_cfg_regs->DDR3_CONFIG_REG[21] = 0xA1;
	boot_cfg_regs->DDR3_CONFIG_REG[22] = 0xBA;
	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0xF; 	// set dll_lock_diff to 15

	/*Invert Clock Out*/
	boot_cfg_regs->DDR3_CONFIG_REG[0] &= ~(0x007FE000);  // clear ctrl_slave_ratio field
	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0x00200000;     // set ctrl_slave_ratio to 0x100
	boot_cfg_regs->DDR3_CONFIG_REG[12] |= 0x08000000;    // Set invert_clkout = 1
	
	boot_cfg_regs->DDR3_CONFIG_REG[23] |= 0x00000200; //Set bit 9 = 1 to use forced ratio leveling for read DQS

	/*the PHY_RESET is pulsed (0 -> 1 -> 0) to latch these 
	leveling configuration values into the PHY logic.*/
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 |= (0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);


	/*Drives CKE low.
	This is a JEDEC requirement that we have 500us delay between reset de-assert 
	and cke assert and then program the correct refresh rate
	The DDR internal clock is divide by 16 before SDCFG write*/
	DDR_Regs->SDRAM_REF_CTRL = 0x80000000|(unsigned int)(500.f*clock_MHz/16.f);

	DDR_Regs->SDRAM_TIM_1 =
		((unsigned int)(13.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RP_SHIFT)|
		((unsigned int)(13.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RCD_SHIFT)|
		((unsigned int)(15*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WR_SHIFT)|
		((unsigned int)(36*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RAS_SHIFT)|
		((unsigned int)(49.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RC_SHIFT)|
		((unsigned int)(45*clock_MHz/4000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RRD_SHIFT)| 	/*T_RRD = (tFAW/(4*tCK)) C 1*/
		((unsigned int)(7.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WTR_SHIFT);
	DDR_Regs->SDRAM_TIM_2   = 
		((unsigned int)(6*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XP_SHIFT)|
		((unsigned int)(120*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSNR_SHIFT)| 	/*T_XSNR = (tXS /tCK)C 1*/
		((512-1)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSRD_SHIFT)| 	/*T_XSRD =tXSDLLC 1*/
		((unsigned int)(7.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_RTP_SHIFT)|
		((unsigned int)(5.625*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_CKE_SHIFT);
	DDR_Regs->SDRAM_TIM_3   = 
		(5<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_PDLL_UL_SHIFT)| 	/*This field must always be programmed to 0x5.*/
		((5)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CSTA_SHIFT)| 	/*This field should be set according to PHY requirements as 0x5.*/
		((unsigned int)(5.625*clock_MHz/1000.f+0.9999f)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CKESR_SHIFT)|
		((64-1)<<CSL_EMIF4F_SDRAM_TIM_3_REG_ZQ_ZQCS_SHIFT)|
		((unsigned int)(110*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RFC_SHIFT)|
		(15<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RAS_MAX_SHIFT); 	/*This field must always be programmed to 0xF.*/

	DDR_Regs->DDR_PHY_CTRL_1  = 0x00100100|
		(13<<CSL_EMIF4F_DDR_PHY_CTRL_1_REG_READ_LATENCY_SHIFT); 	/*between CAS Latency + 1 and CAS Latency + 7*/

	DDR_Regs->ZQ_CONFIG = 
		((0)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS1EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS0EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_DUALCALEN_SHIFT)| 	/*This bit should always be set to 1.*/
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_SFEXITEN_SHIFT)|
		((512/256-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQINIT_MULT_SHIFT)| 	/*T_ZQ_ZQINIT_MULT = (tZQinit/tZQoper C 1)*/
		((256/64-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQCL_MULT_SHIFT)| 	/*T_ZQ_ZQCL_MULT = (tZQoper/tZQCS C 1)*/
		/*interval between ZQCS commands = 0.5%/((TSens x Tdriftrate) + (VSens x Vdriftrate))
		=0.5%/((max (dRTTdT, dRONdTM) x Tdriftrate in C/second) + (max(dRTTdV, dRONdVM) x Vdriftrate in mV/second))
		this time need be converted to refresh period number*/
		((151515151/(64000000/8192))<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_REFINTERVAL_SHIFT);

	/*map priority 0,1,2,3 to COS0,
	map priority 3,5,6,7 to COS1*/
	DDR_Regs->PRI_COS_MAP = 
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_COS_MAP_EN_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_7_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_6_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_5_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_4_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_3_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_2_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_1_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_0_COS_SHIFT);

	/*master based COS map is disabled*/
	DDR_Regs->MSTID_COS_1_MAP= 0;
	DDR_Regs->MSTID_COS_2_MAP= 0;

	/*LAT_CONFIG*/
	DDR_Regs->VBUSM_CONFIG= 
		(8<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_1_SHIFT)|
		(16<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_2_SHIFT)|
		(32<<CSL_EMIF4F_VBUSM_CONFIG_REG_PR_OLD_COUNT_SHIFT);

	DDR_Regs->ECC_CTRL = 
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_PROT_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_2_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_1_EN_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_1= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_STRT_ADDR_1_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_END_ADDR_1_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_2= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_STRT_ADDR_2_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_END_ADDR_2_SHIFT);

	/* enables DRAM configuration.  It still has the refresh interval 
	programmed to the longer number needed during DRAM initialization.*/
	DDR_Regs->SDRAM_REF_CTRL = (unsigned int)(500.f*clock_MHz/16.f); 

	DDR_Regs->SDRAM_CONFIG = 
		(3<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_TYPE_SHIFT)| 	/*Set to 3 for DDR3. All other values reserved.*/
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_POS_SHIFT)|
		(DDR_TERM_RZQ_OVER_6<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_TERM_SHIFT)|
		(DDR_DYN_ODT_DISABLED<<CSL_EMIF4F_SDRAM_CONFIG_REG_DYN_ODT_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_DISABLE_DLL_SHIFT)|
		(SDRAM_DRIVE_RZQ_OVER_7<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_DRIVE_SHIFT)|
		(DDR_CWL_7<<CSL_EMIF4F_SDRAM_CONFIG_REG_CWL_SHIFT)|
		(DDR_BUS_WIDTH_64<<CSL_EMIF4F_SDRAM_CONFIG_REG_NARROW_MODE_SHIFT)|
		(DDR_CL_9<<CSL_EMIF4F_SDRAM_CONFIG_REG_CL_SHIFT)|
		(DDR_ROW_SIZE_13_BIT<<CSL_EMIF4F_SDRAM_CONFIG_REG_ROWSIZE_SHIFT)|
		(DDR_BANK_NUM_8<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_EBANK_SHIFT)|
		(DDR_PAGE_SIZE_10_BIT_1024_WORD<<CSL_EMIF4F_SDRAM_CONFIG_REG_PAGESIZE_SHIFT);

	for(i=0;i<100000;i++)
		asm(" nop"); 	//Wait 600us for HW init to complete

//	DDR_Regs->SDRAM_REF_CTRL    = 64000000/8192/(1000/clock_MHz);
	DDR_Regs->SDRAM_REF_CTRL    = (unsigned int)64000.f*clock_MHz/8192.f;
#if 0
	//enable full leveling, no incremental leveling
	/*Typically program a higher rate of incremental leveling in the ramp window*/
	DDR_Regs->RDWR_LVL_RMP_WIN = 10000/7.8; 	/*10000us*/
	DDR_Regs->RDWR_LVL_RMP_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVL_EN_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVLINC_RMP_PRE_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLINC_RMP_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLGATEINC_RMP_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_WRLVLINC_RMP_INT_SHIFT);
	DDR_Regs->RDWR_LVL_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLFULL_START_SHIFT)|
		(0x7F<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLINC_PRE_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLINC_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLGATEINC_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_WRLVLINC_INT_SHIFT);

	TSC_delay_ms(3); 	//Wait 3ms for leveling to complete

	if(DDR_Regs->STATUS&(CSL_EMIF4F_STATUS_REG_RDLVLGATETO_MASK
		|CSL_EMIF4F_STATUS_REG_RDLVLTO_MASK
		|CSL_EMIF4F_STATUS_REG_WRLVLTO_MASK))
	{
		printf("DDR3 leveling has failed, STATUS = 0x%x\n", DDR_Regs->STATUS);
		return;
	}
#else
	Keystone_DDR_full_leveling();
#endif
}

/*configure DDR according to the clock speed
please note, clock_MHz is the clock speed in MHz, not data rate. 
For example, clock speed for 1333.333M data rate is 666.667MHz*/
void TCI6614_EVM_DDR_Init(float clock_MHz)
{
	int i;
	
	CSL_BootCfgUnlockKicker();

	//initial vale for leveling
	/*WRLVL_INIT_RATIO*/                                                      
	boot_cfg_regs->DDR3_CONFIG_REG[2] = 0x20;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[3] = 0x24;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[4] = 0x3A;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[5] = 0x38;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[6] = 0x51;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[7] = 0x5E;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[8] = 0x5E;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[9] = 0x5E;                                 
	boot_cfg_regs->DDR3_CONFIG_REG[10]= 0x44;                                
                                                                                  
	/*GTLVL_INIT_RATIO*/                                                      
	boot_cfg_regs->DDR3_CONFIG_REG[14] = 0xA1;                                
	boot_cfg_regs->DDR3_CONFIG_REG[15] = 0xA0;                                
	boot_cfg_regs->DDR3_CONFIG_REG[16] = 0xA7;                                
	boot_cfg_regs->DDR3_CONFIG_REG[17] = 0xA9;                                
	boot_cfg_regs->DDR3_CONFIG_REG[18] = 0xCA;                                
	boot_cfg_regs->DDR3_CONFIG_REG[19] = 0xBE;                                
	boot_cfg_regs->DDR3_CONFIG_REG[20] = 0xDD;                                
	boot_cfg_regs->DDR3_CONFIG_REG[21] = 0xDD;                                
	boot_cfg_regs->DDR3_CONFIG_REG[22] = 0xBA;                                

	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0xF; 	// set dll_lock_diff to 15

	/*Invert Clock Out*/
	boot_cfg_regs->DDR3_CONFIG_REG[0] &= ~(0x007FE000);  // clear ctrl_slave_ratio field
	boot_cfg_regs->DDR3_CONFIG_REG[0] |= 0x00200000;     // set ctrl_slave_ratio to 0x100
	boot_cfg_regs->DDR3_CONFIG_REG[12] |= 0x08000000;    // Set invert_clkout = 1

	boot_cfg_regs->DDR3_CONFIG_REG[23] |= 0x00000200; //Set bit 9 = 1 to use forced ratio leveling for read DQS

	/*the PHY_RESET is pulsed (0 -> 1 -> 0) to latch these 
	leveling configuration values into the PHY logic.*/
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 |= (0x00008000);
	DDR_Regs->DDR_PHY_CTRL_1 &= ~(0x00008000);


	/*Drives CKE low.
	This is a JEDEC requirement that we have 500us delay between reset de-assert 
	and cke assert and then program the correct refresh rate
	The DDR internal clock is divide by 16 before SDCFG write*/
	DDR_Regs->SDRAM_REF_CTRL = 0x80000000|(unsigned int)(500.f*clock_MHz/16.f);

	DDR_Regs->SDRAM_TIM_1 =
		((unsigned int)(13.75*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RP_SHIFT)|
		((unsigned int)(13.75*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RCD_SHIFT)|
		((unsigned int)(15*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WR_SHIFT)|
		((unsigned int)(36*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RAS_SHIFT)|
		((unsigned int)(49.5*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RC_SHIFT)|
		((unsigned int)(45*clock_MHz/4000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_RRD_SHIFT)| 	/*T_RRD = (tFAW/(4*tCK)) C 1*/
		(_max2(4-1, (unsigned int)(7.5*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_1_REG_T_WTR_SHIFT);
	DDR_Regs->SDRAM_TIM_2   = 
		(_max2(3-1, (unsigned int)(6*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XP_SHIFT)|
		(_max2(5-1, (unsigned int)((128+10)*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSNR_SHIFT)| 	/*T_XSNR = (tXS /tCK)C 1*/
		((512-1)<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_XSRD_SHIFT)| 	/*T_XSRD =tXSDLLC 1*/
		(_max2(4-1, (unsigned int)(7.5*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_RTP_SHIFT)|
		(_max2(3-1, (unsigned int)(5.625*clock_MHz/1000.f-0.0001f))<<CSL_EMIF4F_SDRAM_TIM_2_REG_T_CKE_SHIFT);
	DDR_Regs->SDRAM_TIM_3   = 
		(5<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_PDLL_UL_SHIFT)| 	/*This field must always be programmed to 0x5.*/
		((5)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CSTA_SHIFT)| 	/*This field should be set according to PHY requirements as 0x5.*/
		((unsigned int)(5.625*clock_MHz/1000.f+0.9999f)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_CKESR_SHIFT)|
		((64-1)<<CSL_EMIF4F_SDRAM_TIM_3_REG_ZQ_ZQCS_SHIFT)|
		((unsigned int)(128*clock_MHz/1000.f-0.0001f)<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RFC_SHIFT)|
		(15<<CSL_EMIF4F_SDRAM_TIM_3_REG_T_RAS_MAX_SHIFT); 	/*This field must always be programmed to 0xF.*/

	DDR_Regs->DDR_PHY_CTRL_1  = 0x00100100|
		(15<<CSL_EMIF4F_DDR_PHY_CTRL_1_REG_READ_LATENCY_SHIFT); 	/*between CAS Latency + 1 and CAS Latency + 7*/

	DDR_Regs->ZQ_CONFIG = 
		((0)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS1EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_CS0EN_SHIFT)|
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_DUALCALEN_SHIFT)| 	/*This bit should always be set to 1.*/
		((1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_SFEXITEN_SHIFT)|
		((512/256-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQINIT_MULT_SHIFT)| 	/*T_ZQ_ZQINIT_MULT = (tZQinit/tZQoper C 1)*/
		((256/64-1)<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_ZQCL_MULT_SHIFT)| 	/*T_ZQ_ZQCL_MULT = (tZQoper/tZQCS C 1)*/
		/*interval between ZQCS commands = 0.5%/((TSens x Tdriftrate) + (VSens x Vdriftrate))
		=0.5%/((max (dRTTdT, dRONdTM) x Tdriftrate in C/second) + (max(dRTTdV, dRONdVM) x Vdriftrate in mV/second))
		this time need be converted to refresh period number*/
		(((unsigned int)(1000000000*0.5/(1.5*1.2+0.15*15))/(64000000/8192))<<CSL_EMIF4F_ZQ_CONFIG_REG_ZQ_REFINTERVAL_SHIFT);

	/*map priority 0,1,2,3 to COS0,
	map priority 3,5,6,7 to COS1*/
	DDR_Regs->PRI_COS_MAP = 
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_COS_MAP_EN_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_7_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_6_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_5_COS_SHIFT)|
		((1)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_4_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_3_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_2_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_1_COS_SHIFT)|
		((0)<<CSL_EMIF4F_PRI_COS_MAP_REG_PRI_0_COS_SHIFT);

	/*master based COS map is disabled*/
	DDR_Regs->MSTID_COS_1_MAP= 0;
	DDR_Regs->MSTID_COS_2_MAP= 0;

	/*LAT_CONFIG*/
	DDR_Regs->VBUSM_CONFIG= 
		(8<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_1_SHIFT)|
		(16<<CSL_EMIF4F_VBUSM_CONFIG_REG_COS_COUNT_2_SHIFT)|
		(32<<CSL_EMIF4F_VBUSM_CONFIG_REG_PR_OLD_COUNT_SHIFT);

	DDR_Regs->ECC_CTRL = 
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_PROT_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_2_EN_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_CTRL_REG_ECC_ADDR_RNG_1_EN_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_1= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_STRT_ADDR_1_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_1_REG_ECC_END_ADDR_1_SHIFT);

	DDR_Regs->ECC_ADDR_RNG_2= 
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_STRT_ADDR_2_SHIFT)|
		((0)<<CSL_EMIF4F_ECC_ADDR_RNG_2_REG_ECC_END_ADDR_2_SHIFT);

	/* enables DRAM configuration.  It still has the refresh interval 
	programmed to the longer number needed during DRAM initialization.*/
	DDR_Regs->SDRAM_REF_CTRL = (unsigned int)(500.f*clock_MHz/16.f); 

	DDR_Regs->SDRAM_CONFIG = 
		(3<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_TYPE_SHIFT)| 	/*Set to 3 for DDR3. All other values reserved.*/
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_POS_SHIFT)|
		(DDR_TERM_RZQ_OVER_6<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_TERM_SHIFT)|
		(DDR_DYN_ODT_DISABLED<<CSL_EMIF4F_SDRAM_CONFIG_REG_DYN_ODT_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_DDR_DISABLE_DLL_SHIFT)|
		(SDRAM_DRIVE_RZQ_OVER_7<<CSL_EMIF4F_SDRAM_CONFIG_REG_SDRAM_DRIVE_SHIFT)|
		(DDR_CWL_8<<CSL_EMIF4F_SDRAM_CONFIG_REG_CWL_SHIFT)|
		(DDR_BUS_WIDTH_64<<CSL_EMIF4F_SDRAM_CONFIG_REG_NARROW_MODE_SHIFT)|
		(DDR_CL_11<<CSL_EMIF4F_SDRAM_CONFIG_REG_CL_SHIFT)|
		(DDR_ROW_SIZE_14_BIT<<CSL_EMIF4F_SDRAM_CONFIG_REG_ROWSIZE_SHIFT)|
		(DDR_BANK_NUM_8<<CSL_EMIF4F_SDRAM_CONFIG_REG_IBANK_SHIFT)|
		(0<<CSL_EMIF4F_SDRAM_CONFIG_REG_EBANK_SHIFT)|
		(DDR_PAGE_SIZE_10_BIT_1024_WORD<<CSL_EMIF4F_SDRAM_CONFIG_REG_PAGESIZE_SHIFT);

	for(i=0;i<100000;i++)
		asm(" nop"); 	//Wait 600us for HW init to complete

//	DDR_Regs->SDRAM_REF_CTRL    = 64000000/8192/(1000/clock_MHz);
	DDR_Regs->SDRAM_REF_CTRL    = (unsigned int)64000.f*clock_MHz/8192.f;

#if 0
	//enable full leveling, no incremental leveling
	/*Typically program a higher rate of incremental leveling in the ramp window*/
	DDR_Regs->RDWR_LVL_RMP_WIN = 10000/7.8; 	/*10000us*/
	DDR_Regs->RDWR_LVL_RMP_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVL_EN_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDWRLVLINC_RMP_PRE_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLINC_RMP_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_RDLVLGATEINC_RMP_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_RMP_CTRL_REG_WRLVLINC_RMP_INT_SHIFT);
	DDR_Regs->RDWR_LVL_CTRL = 
		(1<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLFULL_START_SHIFT)|
		(0x7F<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDWRLVLINC_PRE_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLINC_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_RDLVLGATEINC_INT_SHIFT)|
		(0<<CSL_EMIF4F_RDWR_LVL_CTRL_REG_WRLVLINC_INT_SHIFT);

	TSC_delay_ms(3); 	//Wait 3ms for leveling to complete

	//Read MMR to ensure full leveling is complete
	i= DDR_Regs->RDWR_LVL_RMP_CTRL;
	
	if(DDR_Regs->STATUS&(CSL_EMIF4F_STATUS_REG_RDLVLGATETO_MASK
		|CSL_EMIF4F_STATUS_REG_RDLVLTO_MASK
		|CSL_EMIF4F_STATUS_REG_WRLVLTO_MASK))
	{
		printf("DDR3 leveling has failed, STATUS = 0x%x\n", DDR_Regs->STATUS);
		return;
	}

	/*simple test*/
	*((Uint32 *)0x80000000)= 0xaaaaaaaa;
	if(0xaaaaaaaa!=(*((Uint32 *)0x80000000)))
		puts("DDR3 access test failed.");
#else
	Keystone_DDR_full_leveling();
#endif
}


/*
this function write the address to corresponding memory unit and readback for verification
*/
unsigned int DDR_Address_Test(unsigned int uiStartAddress, 
	unsigned int uiByteCount)
{
    unsigned int i;
    volatile unsigned long long *ulpAddressPointer;
    volatile unsigned long long ulReadBack;

	ulpAddressPointer = (unsigned long long *)uiStartAddress;
	for(i=0; i<uiByteCount/8; i++)
	{
		/* fill with address value */
        *ulpAddressPointer = _itoll(((unsigned int)ulpAddressPointer)+4, 
        	(unsigned int)ulpAddressPointer);	  
        ulpAddressPointer++;
    }

	WritebackInvalidCache((void *)uiStartAddress, uiByteCount);

	ulpAddressPointer = (unsigned long long *)uiStartAddress;
	for(i=0; i<uiByteCount/8; i++)
	{
        ulReadBack = *ulpAddressPointer;
        if ( ulReadBack != _itoll(((unsigned int)ulpAddressPointer)+4, 
        	(unsigned int)ulpAddressPointer)) /* verify data */
        {
			printf("Memory Test fails at 0x%8x, Write 0x%16llx, Readback 0x%16llx\n", 
				ulpAddressPointer, _itoll(((unsigned int)ulpAddressPointer)+4, (unsigned int)ulpAddressPointer), ulReadBack);
			return 1;
        }
        ulpAddressPointer++;
    }

    return 0;              /* show no error */
}

/*
this function write the address to corresponding memory unit and readback for verification
*/
unsigned int DDR_Fill_Test(unsigned int uiStartAddress, 
	unsigned int uiByteCount, unsigned int uiPattern)
{
    unsigned int i;
    volatile unsigned long long *ulpAddressPointer;
    volatile unsigned long long ulReadBack;

	ulpAddressPointer = (unsigned long long *)uiStartAddress;
	for(i=0; i<uiByteCount/8; i++)
	{
		/* fill with address value */
        *ulpAddressPointer = _itoll(uiPattern, uiPattern);	  
        ulpAddressPointer++;
    }

	WritebackInvalidCache((void *)uiStartAddress, uiByteCount);

	ulpAddressPointer = (unsigned long long *)uiStartAddress;
	for(i=0; i<uiByteCount/8; i++)
	{
        ulReadBack = *ulpAddressPointer;
        if ( ulReadBack != _itoll(uiPattern, uiPattern)) /* verify data */
        {
			printf("Memory Test fails at 0x%8x, Write 0x%16llx, Readback 0x%16llx\n", 
				ulpAddressPointer, _itoll(uiPattern, uiPattern), ulReadBack);
			return 1;
        }
        ulpAddressPointer++;
    }

    return 0;              /* show no error */
}

void KeyStone_DDR_init(float ref_clock_MHz, 
	unsigned int DDR_PLLM, unsigned int DDR_PLLD)
{
	Uint32 uiRetryCount=0;
	TDSP_Board_Type DSP_Board_Type;
	float DDR_Speed_MHz, DDR_Clock_MHz;

	DSP_Board_Type= Get_dsp_board_type();

	DDR_Speed_MHz= ref_clock_MHz*DDR_PLLM/DDR_PLLD;
	DDR_Clock_MHz= DDR_Speed_MHz/2; 	//data speed is double of clock speed

	printf("Initialize DDR speed = %.3fx%d/%d= %.1f\n", 
		ref_clock_MHz, DDR_PLLM, DDR_PLLD, DDR_Speed_MHz);

	while(100>uiRetryCount)
	{
		KeyStone_DDR_PLL_init(DDR_PLLM, DDR_PLLD);
		
		if(DUAL_TCI6616_TCI6618_EVM==DSP_Board_Type)
			Dual_TCI6618_EVM_DDR_Init(DDR_Clock_MHz);
		else if(C6678_TCI6608_EVM==DSP_Board_Type)
			C6678_TCI6608_EVM_DDR_Init(DDR_Clock_MHz);
		else if(C6670_TCI6618_EVM==DSP_Board_Type)
			C6670_TCI6618_EVM_DDR_Init(DDR_Clock_MHz);
		else if(TCI6614_EVM==DSP_Board_Type)
			TCI6614_EVM_DDR_Init(DDR_Clock_MHz);
		else
		{
			puts("unknown DSP board type!");
			break;
		}

		//simple test, return if it pass, retry if it fail
		if((0==DDR_Fill_Test(0x80000000,256, 0))
			&&(0==DDR_Fill_Test(0x80000000,256, 0xFFFFFFFF))
			&&(0==DDR_Address_Test(0x80000000,256)))
			break;
			
		uiRetryCount++;
		printf("DDR reinitialization %d\n", uiRetryCount);
	}
}
