/******************************************************************************\
*           Copyright (C) 2000 Texas Instruments Incorporated.
*                           All Rights Reserved
*------------------------------------------------------------------------------
* FILENAME...... main.c   
* PROJECT NAME.. VcpProcessing
* DATE CREATED.. 04/09/2001 
* LAST MODIFIED. 08/07/2003 - Chad Courtney
* Updated Code for Released Version of CSL 
* Changed vcpDone Global Int32 to a Volatile as optimizations was causing while(!vcpDone)
* to only check on initial pass
*
*		Modified 07/01/2005  by Harry Thompson                          
*                                                                          
*                           csl_timerhal.h in CCS 3.0 was modified and a   
*                           parameter was added to control CTL bit. Added  
*                           parameter TIMER_CTL_SPND_EMURUN to             
*                           TimerControl. The lines:                       
*                                                                           
*   static Uint32 TimerControl = TIMER_CTL_RMK(                            
*   TIMER_CTL_INVINP_NO,                                                   
*                                                                          
*                           were changed to:                               
*                                                                          
*  static Uint32 TimerControl = TIMER_CTL_RMK(                             
*   TIMER_CTL_SPND_EMURUN,                                                 
*   TIMER_CTL_INVINP_NO,                                                   
*                                                                          
*
* The purpose of this project is to test out the VCP additions to the Chip
* support library. This program will submit a variety of data to the VCP
* for processing and verify that the correct results are obtained.
\******************************************************************************/


#include <csl.h>
#include <csl_vcp.h>
#include <csl_edma.h>
#include <csl_emifa.h>
#include <csl_timer.h>
#include <stdio.h>
#include "vcp_parameters.h"
#include "ceil.h"

/* Constant definitions */
#define TB_CE0_CTL    0xB00F0008

/* Prototype declarations */
void intConfig(void);
void initExtMem(void);
Int32 checkResults(Uint32 *actual, Uint32 *reference, Uint32 size);
interrupt void edmaIsr(void);
void submitEdma(VCP_UserData *userData, VCP_Params *vcpParameters,
                Uint32 **decisions, Uint32 **outParms, Uint32 *numDec);

/* Global variable declarations */
volatile Int32 vcpDone;
int tcc = -1;
EDMA_Handle hEdmaRevt, hEdmaXevt, hEdmaBrMet, hEdmaInter,
            hEdmaOutPar, hEdmaNull;

static TIMER_Handle hTimer;

static Uint32 TimerControl = TIMER_CTL_RMK(
  TIMER_CTL_SPND_EMURUN,
  TIMER_CTL_INVINP_NO,
  TIMER_CTL_CLKSRC_CPUOVR8,
  TIMER_CTL_CP_PULSE,
  TIMER_CTL_HLD_YES,
  TIMER_CTL_GO_NO,
  TIMER_CTL_PWID_ONE,
  TIMER_CTL_DATOUT_0,
  TIMER_CTL_INVOUT_NO,
  TIMER_CTL_FUNC_GPIO
); 

int cnts[NUM_USERS];

/****************************************************************************\
 * main function                                                            *
\****************************************************************************/
void
main(void)
{
volatile Uint32 error = 0;
Uint32 *decisions, *outParms, numDec;
VCP_Params vcpParms;
int i,cnt;

    CSL_init();
    
    initExtMem();
    
	printf("running VCP test \n");
	printf("Cycles provided in CPU cycles, VCP cycles would be 1/4 CPU cycles \n");
	
      /* TIMER Put in for Profiling of TCP Proc */
      hTimer = TIMER_open(TIMER_DEV1, TIMER_OPEN_RESET);
      TIMER_configArgs(hTimer,
    	TimerControl, /* use predefined control value  */
    	0xffffffff,   /* set period                    */
    	0x00000000    /* start count value at zero     */
  	  );
  	  
    for(i = 0; i < NUM_USERS; i++){
      vcpDone = 0;    
      vcpParms = vcpParameters[i];
      submitEdma(userData[i], &vcpParameters[i], &decisions, &outParms, &numDec);

		/* Setup Timer and Start Timer */
      TIMER_setCount(hTimer,0x00000000);        
      TIMER_start(hTimer);
      
      /* Start VCP */
      VCP_start();
      while(!vcpDone);

		/* Stop Timer and get cycle count, which includes EDMA xfers time as well */
      cnt = TIMER_getCount(hTimer);
      cnts[i] = cnt;
      
      printf("Timer for FL = %d, CPU cycles = %d (8 cycle accuracy due to timer) \n", vcpParms.frameLen, cnt*8);                            


      error = checkResults(decisions, referenceDec[i],
                           vcpParameters[i].frameLen);
      if (error) break;
      free(decisions);
    } /* end for i */
    
    while(1);
    
} /* end main */ 


/****************************************************************************\
 * intConfig: Configure the CPU interrupt controller to receive interrupts  *
 * from the EDMA.                                                           *
\****************************************************************************/
void
intConfig(void)
{
    IRQ_resetAll();                       /* Reset all maskable interrupts  */
      IRQ_enable(IRQ_EVT_EDMAINT);          /* Enable EDMA -> CPU interrupt   */
    IRQ_nmiEnable();                      /* Enable non-maskable interrupt  */
    IRQ_globalEnable();                   /* Globally enable all interrupts */
} /* end intConfig() */


/****************************************************************************\
 * initExtMem: Configure the EMIF to have SDRAM in CE0.                     *
\****************************************************************************/
void
initExtMem(void)
{
EMIFA_Config config;

    config.gblctl = EMIFA_GBLCTL_DEFAULT;
    config.cectl0 = EMIFA_CECTL_RMK(0,0,0,0,0,0,EMIFA_CECTL_MTYPE_SDRAM32,0,0);
    config.cectl1 = EMIFA_CECTL_DEFAULT;
    config.cectl2 = EMIFA_CECTL_DEFAULT;
    config.cectl3 = EMIFA_CECTL_DEFAULT;
    config.sdctl  = EMIFA_SDCTL_DEFAULT;
    config.sdext  = EMIFA_SDEXT_DEFAULT;
    config.sdtim  = EMIFA_SDEXT_DEFAULT;
    
    EMIFA_config(&config);

    *(volatile Uint32 *)TB_CE0_CTL = config.cectl0;
} /* end initExtMem */


/****************************************************************************\
 * checkResults: Verify that the hard decisions and output parameters sent  *
 * by the VCP are correct. If the hard decisions are not correct, the index *
 * of the first incorrect value is returned. If the output parameters (the  *
 * number of iterations) is not correct, then a -1 is returned. If all      *
 * values are correct then a 0 is returned.                                 *
\****************************************************************************/
Int32
checkResults(Uint32 *actual ,Uint32 *reference, Uint32 size) 
{
Uint32 i;
Uint32 mismatch=0;
Uint32 numWords = size>>5;
    
    for (i=0; i<numWords; i++) {
      if (actual[i]!=reference[i]) {
        mismatch++;
        break;
      } /* end if actual != reference */
    } /* end for i */
    
    if ((!mismatch) && (size & 1)) {    
      if ((short)actual[i] != (short)reference[i]) mismatch++;
    } /* end if */

    if (mismatch) return(i+1); /* return index for 1st error that occured. */
                               /* Index will range from 1 to framelength   */
    else return(0);            /* no error */
    
} /* end checkResults() */


/****************************************************************************\
 * edmaIsr: EDMA interrupt service routine. The vcpDone flag is set to one  *
 * and all EDMA handles are closed, freeing them for use on the next        *
 * iteration.                                                               *
\****************************************************************************/
interrupt void
edmaIsr(void)  
{ 
    vcpDone = EDMA_intTest(tcc);
    EDMA_intClear(tcc);
    EDMA_intFree(tcc);
    EDMA_close(hEdmaRevt);
    EDMA_close(hEdmaXevt);
    EDMA_freeTable(hEdmaBrMet);
    EDMA_freeTable(hEdmaOutPar);
    EDMA_freeTable(hEdmaNull);

} /* end edmaIsr */


/****************************************************************************\
 * submitEdma: The EDMA is programmed to service the user channel. The IC   *
 * values are configured and all EDMA parameters are calculated here.       *
\****************************************************************************/
void
submitEdma(VCP_UserData *userData, VCP_Params *vcpParameters,
           Uint32 **decisions, Uint32 **outParms, Uint32 *numDec)
{
EDMA_Config edmaConfig;
VCP_ConfigIc configIc = {0,0,0,0,0,0};
Uint32 frameLen  = vcpParameters->frameLen;
Uint32 constLen  = vcpParameters->constLen;
Uint32 rate      = vcpParameters->rate;
Uint32 symx      = vcpParameters->bmBuffLen;
Uint32 symr      = vcpParameters->decBuffLen;
Uint32 decision  = vcpParameters->decision;
Uint32 traceBack = vcpParameters->traceBack;
Uint32 numBm, length, elCount, numFrames;
Uint32 *decAddr;

    /* Completely reset the EDMA */
    EDMA_resetAll();
    
    /* Configure CPU interrupts */
    intConfig();
    
    /* Allocate transfer completion code */
    tcc = EDMA_intAlloc(-1);

    /* Calculate the VCP IC values */
    VCP_genIc(vcpParameters, &configIc);
      
    /* Open handles to the EDMA channels */
    hEdmaRevt    = EDMA_open(EDMA_CHA_VCPREVT, EDMA_OPEN_RESET);
    hEdmaXevt    = EDMA_open(EDMA_CHA_VCPXEVT, EDMA_OPEN_RESET);
    EDMA_allocTableEx(1, &hEdmaBrMet);
    EDMA_allocTableEx(1, &hEdmaOutPar);
    EDMA_allocTableEx(1, &hEdmaNull);
    EDMA_reset(hEdmaNull);

    /* Configure channel parameters for IC registers */
    edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                     EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_INC,
                     EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_INC,
                     EDMA_OPT_TCINT_NO,             EDMA_OPT_TCC_DEFAULT,
                     EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                     EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                     EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                     EDMA_OPT_FS_YES);

    edmaConfig.src = (Uint32)&configIc;
    edmaConfig.cnt = (Uint32)NB_32BITWORD_INPUT_PARAMS;
    edmaConfig.dst = (Uint32)VCP_ICMEM_ADDR;
    edmaConfig.idx = 0;
    edmaConfig.rld = 0;
    EDMA_config(hEdmaXevt, &edmaConfig);
    EDMA_link(hEdmaXevt, hEdmaBrMet);
        
    /* Configure channel parameters for branch metrics data */
    numBm = 1 << (rate - 1);                              /* number of Branch Metrics       */
    length = frameLen + constLen - 1;                     /* calculate the frame length     */
    numFrames = _normalceil(length, 4 * symx);            /* find the number of sub frames  */
    
    length = _ceil(length, 2)<<2;                         /* Find frame length used for xfr */
    if (symx < length) elCount = symx * numBm;            /* Select the smaller count value */
    else elCount = length * numBm;                        /*  #symbols per evt or frame len */
    
    edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                     EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_INC,
                     EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_NONE,
                     EDMA_OPT_TCINT_NO,             EDMA_OPT_TCC_DEFAULT,
                     EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                     EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                     EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                     EDMA_OPT_FS_YES);

    edmaConfig.src = (Uint32)userData;
    edmaConfig.cnt = EDMA_CNT_RMK(numFrames - 1, elCount);
    edmaConfig.dst = (Uint32)VCP_BMMEM_ADDR;
    EDMA_config(hEdmaBrMet, &edmaConfig);
    EDMA_link(hEdmaBrMet, hEdmaNull);

    /* Configure channel parameters for decision data */
    if ((decision == VCP_DECISION_HARD) && (traceBack == VCP_TRACEBACK_TAILED)){
      edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                       EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_NONE,
                       EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_DEC,
                       EDMA_OPT_TCINT_NO,             EDMA_OPT_TCC_DEFAULT,
                       EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                       EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                       EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                       EDMA_OPT_FS_YES);
      if (64 * symr < frameLen) elCount = 64 * symr;
      else elCount = frameLen;
      elCount = 2 * _ceil(elCount, 6);
      numFrames = _normalceil(frameLen, 64 * symr);
    } /* end if Hard decisions */
    else if ((decision == VCP_DECISION_HARD) && !(traceBack == VCP_TRACEBACK_TAILED)){
      edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                       EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_NONE,
                       EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_INC,
                       EDMA_OPT_TCINT_NO,             EDMA_OPT_TCC_DEFAULT,
                       EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                       EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                       EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                       EDMA_OPT_FS_YES);
      if (64 * symr < frameLen) elCount = 64 * symr;
      else elCount = frameLen;
      elCount = 2 * _ceil(elCount, 6);
      numFrames = _normalceil(frameLen, 64 * symr);
    } /* end if Hard decisions */
    else if (traceBack == VCP_TRACEBACK_TAILED) { /* Soft Dec and Tailed TB */
      edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                       EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_NONE,
                       EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_DEC,
                       EDMA_OPT_TCINT_NO,             EDMA_OPT_TCC_DEFAULT,
                       EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                       EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                       EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                       EDMA_OPT_FS_YES);
      if (symr < frameLen) elCount = 4 * symr;
      else elCount = frameLen;
      elCount = 2 * _ceil(elCount, 2);
      numFrames = _normalceil(frameLen, 4 * symr);
    } /* end if Soft decisions and tailed traceback mode */
    else { /* Soft Dec and either convergent or mixed traceback */
      edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                       EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_NONE,
                       EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_INC,
                       EDMA_OPT_TCINT_NO,             EDMA_OPT_TCC_DEFAULT,
                       EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                       EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                       EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                       EDMA_OPT_FS_YES);
      elCount = 2 * symr;
      numFrames = _normalceil(frameLen, 4 * symr);
    } /* end else if Soft decisions and either convergent or mixed traceback */
    *numDec = numFrames * elCount;
    *decisions = (Uint32 *)memalign(2 * sizeof(Uint32), *numDec * sizeof(Uint32));
    decAddr = *decisions;
    if (traceBack == VCP_TRACEBACK_TAILED){
      decAddr = &decAddr[*numDec - 1];
      if((decision == VCP_DECISION_HARD) && (frameLen > 1024)) { 
        *decisions = decAddr - ((_ceil(frameLen, 6)) << 1) + 1;
      } /* end if VCP_DECISION_HARD */
      else if ((decision == VCP_DECISION_SOFT) && (frameLen > 64)) {
        *decisions = decAddr - ((_ceil(frameLen, 2)) << 1) + 1;
      } /* end if VCP_DECISION_SOFT */
    } /* end if VCP_TRACEBACK_TAILED) */
    
    edmaConfig.src = (Uint32)VCP_HDMEM_ADDR;
    edmaConfig.cnt = EDMA_CNT_RMK(numFrames - 1, elCount);
    edmaConfig.dst = (Uint32)decAddr;
    EDMA_config(hEdmaRevt, &edmaConfig);
    EDMA_link(hEdmaRevt, hEdmaOutPar);

    /* Configure channel parameters for VCP output parameters */
    *outParms = (Uint32 *)memalign(2 * sizeof(Uint32),
                          NB_32BITWORD_OUTPUT_PARAMS * sizeof(Uint32));
    edmaConfig.opt = EDMA_OPT_RMK(EDMA_OPT_PRI_LOW, EDMA_OPT_ESIZE_32BIT,
                     EDMA_OPT_2DS_NO,               EDMA_OPT_SUM_INC,
                     EDMA_OPT_2DD_NO,               EDMA_OPT_DUM_INC,
                     EDMA_OPT_TCINT_YES,            EDMA_OPT_TCC_OF(tcc),
                     EDMA_OPT_TCCM_DEFAULT,         EDMA_OPT_ATCINT_NO,
                     EDMA_OPT_ATCC_DEFAULT,         EDMA_OPT_PDTS_DISABLE,
                     EDMA_OPT_PDTD_DISABLE,         EDMA_OPT_LINK_YES,
                     EDMA_OPT_FS_YES);
    edmaConfig.src = (Uint32)VCP_OPMEM_ADDR;
    edmaConfig.cnt = (Uint32)NB_32BITWORD_OUTPUT_PARAMS;
    edmaConfig.dst = (Uint32)*outParms;
    EDMA_config(hEdmaOutPar, &edmaConfig);
    EDMA_link(hEdmaOutPar, hEdmaNull);

    /* Enable the TCC to generate a CPU interrupt */
    EDMA_intEnable(tcc);
    
    /* Enable the transmit and receive channels */
    EDMA_enableChannel(hEdmaRevt);
    EDMA_enableChannel(hEdmaXevt);

} /* end submitEdma() */
