/*
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/*
 *    Two stage boot example.
 *
 *    This code demonstrates a two stage boot procedure using the I2C as the
 *    pin strapped boot mode. The code reconfigures the ROM boot loader
 *    to execute a SRIO boot, performs PLL initializations that are required
 *    but not present in the boot ROM, then branches back into the ROM
 *    which performs the standard SRIO boot.
 */

#include "tiboot.h"


/* Nyquist/Shannon boot modes */

#define BOOT_MODE_PIN_SLEEP_EMIF25_SUBMODE_SLEEP   0
#define BOOT_MODE_PIN_SLEEP_EMIF25_SUBMODE_EMIF25  1

#define BOOT_MODE_ETH               10
#define BOOT_MODE_RAPIDIO           20
#define BOOT_MODE_PCIE              30
#define BOOT_MODE_I2C               40
#define BOOT_MODE_I2C_PASSIVE       41
#define BOOT_MODE_I2C_MASTER_WRITE  42
#define BOOT_MODE_SPI               50
#define BOOT_MODE_VUSR              60
#define BOOT_MODE_EMIF25            70
#define BOOT_MODE_SLEEP             100

/*
 *    ROM reboot configuration.
 *
 *    Adjust these parameters to match the platform
 */
const BOOT_PARAMS_ETHERNET_T boot_params =
{
    sizeof(BOOT_PARAMS_ETHERNET_T),      /*  length */
    0,                              /*  checksum */
    BOOT_MODE_ETH,                  /*  boot mode */
    0,                              /*  port num */
    16400,							/* SW PLL Cfg_MSW */
	258,							/* SW PLL Cfg_LSW */
    BOOT_PARAMS_ETH_OPTIONS_RMII_10,     /* options */
	37079,										/* Mac Addr Hi */
    60173,                                   /* Mac Addr Med */
	4681,                                  /* Mac Addr Lo */
	65535,										/* Multi/Broad cast Mac Addr Hi */
	65535,										/* Multi/Broad cast Mac Addr Med */
	65535,										/* Multi/Broad cast Mac Addr Lo */
	0,										/* Source Port 0 */
	9,									/* Destination port */
	12336,									/* Device ID 1_2*/
	12336,										/* DeviceID 3_4 */
	65535,										/* Dest Mac Addr Hi */
	65535,										/* Dest Mac Addr Med */
	65535,									/* Dest Mac Addr Lo */
	16,										/* SGMII Config */
	1,									/* SGMII Control */
	1,										/* SGMII MR_ADV_Ability */
	1,									/* SGMII TX Config Hi */
	2209,									/* SGMII TX Config Lo */
	112,									/* SGMII RX Config Hi */
	1569,									/* SGMII RX Config Lo */
	0,									/* SGMII AUX Config Hi */
	65,									/* SGMII AUX Config Lo */
	32789,									/* Pkt PLL Cfg MSW */
	258,									/* Pkt PLL Cfg LSW */
};


#define CYCLES_PER_US   123     /* Assumes a 122.88 MHz ref clock */

extern volatile cregister unsigned int CSR;

/*
 * Assembly function
 */
void chipDelay32 (unsigned int cycleCount);

/*
 *  The following values are based on the ROM memory mapping, and defined for
 *  this program in the linker command file
 */
void romtBootReentry();




#pragma DATA_SECTION(romCfgTblNyquist, ".romCfgNyquist")
BOOT_PARAMS_ETHERNET_T romCfgTblNyquist;

#pragma DATA_SECTION(romCfgTblShannon, ".romCfgShannon")
BOOT_PARAMS_ETHERNET_T romCfgTblShannon;

/*
 *   The main function performs the following initializatons that are not done
 *   by the boot ROM:
 *
 *          -  resets the main PLL, waits 5us (min), releases main PLL reset
 *          -  sets the ensat bit for all PLLs
 *          -  enables the divide by 2 in the main PLL
 */
void main (void)
{
     unsigned int reg;
    
    /* Perform a reset on the main PLL. The delay is just a cycle delay. The PLL
     * is in bypass, so the reference clock of 122.88 is used for the counter.
     * Twenty times the minimum delay is used */

    /* Put the pll into bypass */
    reg = *((unsigned int *)0x2310100);
    reg = reg & ~1;
    *((unsigned int *)0x2310100) = reg;
    chipDelay32 (1000);

    /* Apply reset to the PLL, wait, release reset */
    reg = reg | 2;
    *((unsigned int *)0x2310100) = reg;
    chipDelay32 (CYCLES_PER_US* 5 * 20);
    reg = reg & ~2;
    *((unsigned int *)0x2310100) = reg;


    /* Enable the divide by 2 */
    reg = *((unsigned int *)0x2310108);
    reg = reg | (1 << 16) | (1 << 19);
    *((unsigned int *)0x2310108) = reg; 

    /* Unlock the registers */
    *((unsigned int *)0x2620038) = 0x83e70b13;
    *((unsigned int *)0x262003c) = 0x95a4f1e0;

    /* Set the ensat bits for all the PLLs */
    reg = *((unsigned int *)0x262032c);
    reg = reg | (1 << 6);
    *((unsigned int *)0x262032c) = reg;

    reg = *((unsigned int *)0x2620334);
    reg = reg | (1 << 6);
    *((unsigned int *)0x2620334) = reg;

    reg = *((unsigned int *)0x262033c);
    reg = reg | (1 << 6);
    *((unsigned int *)0x262033c) = reg;

#if 0
    /* This code is compiled little endian. If running in big endian the
     * table must be converted. Every entry is a 16 bit value */
    if ((CSR & 0x100) == 0)  {

        int    i;
        UINT16 swap;

        BOOT_PARAMS_T *pb = (BOOT_PARAMS_T *)&boot_params;

        for (i = 0; i < sizeof(BOOT_PARAMS_ETHERNET_T)/2; i += 2)  {
            swap               = pb->parameter[i+0];
            pb->parameter[i+0] = pb->parameter[i+1];
            pb->parameter[i+1] = swap;
        }

    }
#endif

    /* Now copy the ROM configuration table to its destination */
    reg = *((unsigned int *)0x2620018);

    if (reg == 0x9d02f)
        memcpy (&romCfgTblNyquist, &boot_params, sizeof(boot_params));
    else
        memcpy (&romCfgTblShannon, &boot_params, sizeof(boot_params));

    /* Re-execute the ROM */
    romtBootReentry ();

}





