#include <cslr.h>
#include <cslr_syscfg0_OMAPL138.h>
#include <soc_OMAPL138.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <soc_OMAPL138_AINTC.h>
#include <cslr_aintc.h>
#include <processor_mode.h>
#include "Coff.h"

#define SYS_BASE            0x01C14000
#define KICK0Ra             *(unsigned int*)(SYS_BASE + 0x038)
#define KICK1Ra             *(unsigned int*)(SYS_BASE + 0x03c)

unsigned int LoadDSP(const char *filename);
void setup_ARM_INTC (void);
void SYSCFG_CHIPINT1_isr(void);

CSL_SyscfgRegsOvly SYS_REGS = (CSL_SyscfgRegsOvly)CSL_SYSCFG_0_REGS;
CSL_PscRegsOvly psc0Regs = (CSL_PscRegsOvly) CSL_PSC_0_REGS;
CSL_AintcRegsOvly aintcRegs = (CSL_AintcRegsOvly)(CSL_AINTC_0_REGS);		/* ARM Interrupt Controller */

ISRpointer sysISRtbl[MAX_ISR_ASSIGNMENTS] = {NULL};    /* Table that maps user ISRs to AINTC system interrupts */


void main (void)
{
	unsigned int entry_point = 0;	                         //Entry point of DSP .out
	unsigned int delay = 10000;
	const char *filename = "..//OMAPL138_INT_DSP.out";       //Filename of DSP .out


	/************************* Setting ARM_INTC *************************/
	printf("======================================================== \n");
	printf("Setting ARM INTC! \n");
	printf("... \n");
	switch_to_privileged_mode();	//Switch to privileged mode
	setup_ARM_INTC();				//Setup ARM interrupt controller
	_enable_IRQ();	 				//Enable IRQ of ARM


	/************************* Load DSP .out file ************************/
	printf("======================================================== \n");
	printf("Loading DSP .out \n");
	printf("... \n");
	entry_point = LoadDSP(filename);

	KICK0Ra = 0x83e70b13;	//Open Permissions to SYSCFG Registers
	KICK1Ra = 0x95A4F1E0;	//(Not required for PG2.0 silicon and above)

	//Set DSP boot address vector to entry point of DSP program. This must be aligned to 1KB boundaries
	SYS_REGS->HOST1CFG = entry_point;

	//Wake up the DSP
	CSL_FINST(psc0Regs->MDCTL[CSL_PSC_DSP], PSC_MDCTL_NEXT, ENABLE);
	CSL_FINST(psc0Regs->PTCMD, PSC_PTCMD_GO1, SET);
	while(CSL_FEXT(psc0Regs->PTSTAT, PSC_PTSTAT_GOSTAT1)==CSL_PSC_PTSTAT_GOSTAT1_IN_TRANSITION);
	CSL_FINST(psc0Regs->MDCTL[CSL_PSC_DSP], PSC_MDCTL_LRST, DEASSERT);

	//Delay for DSP to do some initialization
	while(delay > 0)
	{
		delay--;
	}

	/***************** Assert CHIPINT2 interrupt to DSP *****************/
	printf("======================================================== \n");
	printf("Assert CHIPINT2 interrupt to DSP. \n");
	printf("In this example,DSP will start blinking led! \n");
	printf("... \n");
	KICK0Ra = 0x83e70b13;				//Open Permissions to SYSCFG Registers
	KICK1Ra = 0x95A4F1E0;				//Note: DSP may close the Permissions
	SYS_REGS->CHIPSIG = 0X00000004;		//Assert CHIPINT2

	while(1);
}


/**=====================================================**/
/** Function name : LoadDSP
@brief Load DSP executive file (Coff format) to RAM
@returnEntry point of DSP executive file
**=====================================================**/
unsigned int LoadDSP(const char *filename)
{
	unsigned int i;
	unsigned int entry_point;
	unsigned int section_size, load_addr, run_addr;
	FILE *DSPOutFile;

	//Structures to store the file information
	TCoffHeader tCoffHeader;
	TCoffOptionalHeader tCoffOptionalHeader;
	TSectionHeader tSectionHeader;

	memset(&tCoffHeader,0,HEADER_SIZE);
	memset(&tCoffOptionalHeader,0,OPTIONAL_HEADER_SIZE);
	memset(&tSectionHeader,0,SECTION_HEADER_SIZE);


	//Parse .out file, get statistics information
	if((DSPOutFile = fopen(filename,"rb"))==NULL)
	{
		printf("\n Cannot open the file!");
		exit(0);
	}
	fread((char *)&tCoffHeader,1,HEADER_SIZE,DSPOutFile);//Get file header


	if(tCoffHeader.uiOptionalHeaderBytes)   //Have a optional header
		{
		    //Get file optional header
		    fread((char *)&tCoffOptionalHeader,1,tCoffHeader.uiOptionalHeaderBytes,DSPOutFile);

		    entry_point = tCoffOptionalHeader.uiEntryPoint;
		}

	//Load sections
	for(i=0; i<tCoffHeader.usSectionNumber; i++)
	        {
					//Move the file pointer to the new section headers.
					fseek(DSPOutFile,HEADER_SIZE+tCoffHeader.uiOptionalHeaderBytes+i*SECTION_HEADER_SIZE,0);

					//Get section header
					fread((char *)&tSectionHeader,1,SECTION_HEADER_SIZE,DSPOutFile);

	                if(tSectionHeader.uiSectionSize==0)
	                    continue;
	                if((tSectionHeader.uiFlags&(SECTION_TYPE_DSECT|SECTION_TYPE_NOLOAD|SECTION_TYPE_COPY)))
	                    continue;

	                //Get section information
               		section_size = tSectionHeader.uiSectionSize; //Section size
               		load_addr = tSectionHeader.uiVirtalAddress;  //Load address
               		run_addr = tSectionHeader.uiPysicalAddress;  //Run address

               		//Move the file pointer to the raw section data.
               		fseek(DSPOutFile,tSectionHeader.uiRawDataPointer,0);

               		//Load section data
               		fread((char *)load_addr,1,section_size,DSPOutFile);
	        }
	fclose(DSPOutFile);

	return entry_point;
}


/**=====================================================**/
/**Function name:setup_ARM_INTC
@brief: Setup ARM Interrupt Controller.Map the system interrupt SYSCFG_CHIPINT1 to channel 2 and enabled the interrupt.
@return: none
**======================================================**/
void setup_ARM_INTC (void)
{
	//Disabled system interrupts 0:100
	aintcRegs->ECR1 = 0xFFFFFFFF;
	aintcRegs->ECR2 = 0xFFFFFFFF;
	aintcRegs->ECR3 = 0xFFFFFFFF;
	aintcRegs->ECR4 = 0xFFFFFFFF;

	//Clear system interrupts status 0:100
	aintcRegs->SECR1 = 0xFFFFFFFF;
	aintcRegs->SECR2 = 0xFFFFFFFF;
	aintcRegs->SECR3 = 0xFFFFFFFF;
	aintcRegs->SECR4 = 0xFFFFFFFF;

	sysISRtbl[AINTC_SYSCFG_CHIPINT1] = SYSCFG_CHIPINT1_isr;					/* Assign SYSCFG_CHIPINT1 ISR to SYSCFG_CHIPINT1 system interrupt */

	aintcRegs->VBR = (unsigned int) sysISRtbl;								/* Assign address of ISR table to VBR */
	aintcRegs->VSR = 0;														/* Function pointers size = 4 bytes */

	CSL_FINS(aintcRegs->CMR7, AINTC_CMR7_CHNL_NPLUS1, 2);					/* Map SYSCFG_CHIPINT1 Interrupts to Channel 2*/

	CSL_FINS(aintcRegs->EISR, AINTC_EISR_INDEX, AINTC_SYSCFG_CHIPINT1);		/* Enable SYSCFG_CHIPINT1 Interrupt */

	CSL_FINS(aintcRegs->HIEISR, AINTC_HIEISR_INDEX, 1);						/* Enable IRQ Interrupts */
	CSL_FINS(aintcRegs->GER, AINTC_GER_ENABLE, 1);							/* Enable Interrupts */
}


/**=====================================================**/
/**Function name: SYSCFG_CHIPINT1_isr
@brief: Interrupt service routine of SYSCFG_CHIPINT1
@return: none
**======================================================**/
void SYSCFG_CHIPINT1_isr(void)
{
	printf("======================================================== \n");
	printf("Entering SYSCFG_CHIPINT1_isr! \n");
	printf("SYSCFG_CHIPINT1 was asserted by DSP after blinking led. \n");
	printf("... \n");

	//Acknowledge SYSCFG_CHIPINT1 Interrupt
	CSL_FINS(aintcRegs->SICR, AINTC_SICR_INDEX, AINTC_SYSCFG_CHIPINT1);

	//Clear system interrupts status of SYSCFG_CHIPINT1
	aintcRegs->SECR1 = 0x20000000;

	//Disabled system interrupts SYSCFG_CHIPINT1
	aintcRegs->ECR1 = 0x20000000;

	printf("Leaving SYSCFG_CHIPINT1_isr! \n");
}




