#include <csl.h>
#include <cslr.h>
#include <cslr_gpio.h>
#include <cslr_syscfg0_OMAPL138.h>
#include <soc_OMAPL138.h>
#include <csl_types.h>
#include <cslr_i2c.h>
#include <cslr_dspintc.h>
#include <stdio.h>
#include <c6x.h>
#include "OMAP_L138_DSP.H"
#include "types.h"

#define NUM_CHARS       8
#define I2C_ADDR		0x21
#define CONFIG_PORT0    0x6
#define CONFIG_PORT1    0x7
#define OUTPUT_PORT0    0x2
#define OUTPUT_PORT1    0x3
#define WAIT_LOOP 		0x50000
#define SYS_BASE           0x01C14000
#define KICK0Ra             *(unsigned int*)(SYS_BASE + 0x038)
#define KICK1Ra             *(unsigned int*)(SYS_BASE + 0x03c)


extern void intcVectorTable(void);
extern void SYSCFG_CHIPINT2_Interrupt(void);
extern void Empty_Interrupt(void);

static void init_i2c(void);
static int writeI2C(unsigned, unsigned, unsigned);
static void InitDspIntc();

CSL_I2cRegsOvly i2cRegs = (CSL_I2cRegsOvly)CSL_I2C_0_REGS;
CSL_SyscfgRegsOvly SYS_REGS = (CSL_SyscfgRegsOvly)CSL_SYSCFG_0_REGS;
CSL_DspintcRegsOvly intcRegs = (CSL_DspintcRegsOvly)CSL_INTC_0_REGS;

void main (void)
{
	//Initialize i2c registers and start i2c running
	init_i2c();

	//Initialize DSP INTC
	InitDspIntc();

	while(1);
}


/**========================================================**/
/** Function nameinit_i2c
@briefSetup i2c registers and start i2c running
@returnnone
**========================================================**/
void init_i2c(void)
{
  // Open Permissions to SYSCFG Registers (Not required for PG2.0 silicon and above)
  KICK0Ra = 0x83e70b13;
  KICK1Ra = 0x95A4F1E0;

  //Setup Pinmux for I2C0 Pins
  SYS_REGS->PINMUX4 = (CSL_SYSCFG_PINMUX4_PINMUX4_15_12_I2C0_SDA <<  CSL_SYSCFG_PINMUX4_PINMUX4_15_12_SHIFT) |
    				  (CSL_SYSCFG_PINMUX4_PINMUX4_11_8_I2C0_SCL <<  CSL_SYSCFG_PINMUX4_PINMUX4_11_8_SHIFT);

  // Close permissions
  SYS_REGS->KICK0R = 0x00001111;

  //i2c mode register
  i2cRegs->ICMDR = CSL_FMKT(I2C_ICMDR_MST,MASTER_MODE) //master mode
                            | CSL_FMKT(I2C_ICMDR_TRX,TX_MODE) //transmitter mode
                            | CSL_FMKT(I2C_ICMDR_RM,ENABLE) //repeat mode
                            | CSL_FMKT(I2C_ICMDR_MST,MASTER_MODE) //Master mode
                            | CSL_FMKT(I2C_ICMDR_DLB,DISABLE); //digital loopback

  //set i2c clock low time divide down value, 0-FFFFh
  i2cRegs->ICCLKL = CSL_FMK(I2C_ICCLKL_ICCL,0x10);

  //set i2c clock high time divide down value, 0-FFFFh
  i2cRegs->ICCLKH = CSL_FMK(I2C_ICCLKH_ICCH,0x10);

  //set i2c prescalar divide down value, 0-FFh
  i2cRegs->ICPSC = CSL_FMK(I2C_ICPSC_IPSC,0x2);

  //bring i2c module out of reset
  CSL_FINST(i2cRegs->ICMDR,I2C_ICMDR_IRS,ENABLE);

}


/**========================================================**/
/** Function nameInitDspIntc
@briefInitialize DSP Interrupt Controller
@return  void
**========================================================**/
void InitDspIntc()
{
	//Disable all maskable interrupts
	DSP_disable_interrupts();

    //Clear all Event Flag
	intcRegs->EVTCLR[0] = 0XFFFFFFFF;
	intcRegs->EVTCLR[1] = 0XFFFFFFFF;
	intcRegs->EVTCLR[2] = 0XFFFFFFFF;
	intcRegs->EVTCLR[3] = 0XFFFFFFFF;

    //Disabled aLL event from being combined
	intcRegs->EVTMASK[0] = 0XFFFFFFFF;
	intcRegs->EVTMASK[1] = 0XFFFFFFFF;
	intcRegs->EVTMASK[2] = 0XFFFFFFFF;
	intcRegs->EVTMASK[3] = 0XFFFFFFFF;

    //Event 5 (SYSCFG_CHIPINT2) maps to CPU Interrupt 4 (CPUINT4)
	intcRegs->INTMUX1 = 5;
	intcRegs->INTMUX2 = 0;
	intcRegs->INTMUX3 = 0;

	//Clear interrupt exception status.
	intcRegs->INTXCLR = 0X01;

    //Disabled ALL Exception from being combined
	intcRegs->EXPMASK[0] = 0XFFFFFFFF;
	intcRegs->EXPMASK[1] = 0XFFFFFFFF;
	intcRegs->EXPMASK[2] = 0XFFFFFFFF;
	intcRegs->EXPMASK[3] = 0XFFFFFFFF;


    //Set ISTP to point to the vector table address
    ISTP = (unsigned int)intcVectorTable;

    //Clear all interrupts, interrupt 4 ~ interrupt 15
    ICR = 0xFFF0;

    // Enable the bits for non maskable interrupt(NMI) and  interrupt 4 (CPUINT4)   */
    IER = 0x13;

    //Enabled all maskable interrupts
    DSP_enable_interrupts();
}


/**========================================================**/
/** Function nameSYSCFG_CHIPINT2_Interrupt
@briefSYSCFG_CHIPINT2 interrupt service routine. Blink led.
@return  none
**========================================================**/
void SYSCFG_CHIPINT2_Interrupt(void)
{
  int j = 0;
  int k = 0;

  //Toggle LEDS on and off
  while(k<10)
  {
	writeI2C(I2C_ADDR, OUTPUT_PORT0, 0x7F);
    for(j=0;j<WAIT_LOOP;j++);

    writeI2C(I2C_ADDR, CONFIG_PORT0, 0x7F);
    for(j=0;j<WAIT_LOOP;j++);

	writeI2C(I2C_ADDR, OUTPUT_PORT0, 0xBF);
    for(j=0;j<WAIT_LOOP;j++);

    writeI2C(I2C_ADDR, CONFIG_PORT0, 0xBF);
    for(j=0;j<WAIT_LOOP;j++);

    k+=1;
  }

  SYS_REGS->KICK0R = 0x83E70B13;		/* Gain access to 	*/
  SYS_REGS->KICK1R = 0x95A4F1E0;		/* SYSCFG module 	*/

  SYS_REGS->CHIPSIG_CLR = 0X00000004;   /* Clear CHIPSIG[2] bit */
  SYS_REGS->CHIPSIG = 0X00000002;		/* Asserts SYSCFG_CHIPINT1 interrupt to ARM. */
  SYS_REGS->KICK0R = 0x00001111;		/* Close permissions to  SYSCFG module */

  intcRegs->EVTCLR[0] = 0X00000020;		//Clear Event Flag 5
  intcRegs->INTXCLR = 0X01;				//Clear Interrupt Exception Status.
  return;
}


int writeI2C(unsigned addr, unsigned reg, unsigned data)
{

    /* set slave address to be the I2C expander */
	i2cRegs->ICSAR = addr;

    /* set transmit byte count (ICCNT) to two bytes (register ID + value) */
	i2cRegs->ICCNT = 2;

    /* put the register ID into the transmit data register */
    i2cRegs->ICDXR = reg;

    /* start the transaction */
    i2cRegs->ICMDR |= CSL_FMKT(I2C_ICMDR_STT,SET);

    asm(" .global _PMI_waitT1");
    asm("_PMI_waitT1:");

    /* wait for TX ready status */
    while(CSL_FEXT(i2cRegs->ICSTR,I2C_ICSTR_ICXRDY)!= 1);

    /* put the register data value into the transmit data register */
    i2cRegs->ICDXR = data;

    asm(" .global _PMI_waitT2");
    asm("_PMI_waitT2:");

    /* wait for TX ready status */
    while(CSL_FEXT(i2cRegs->ICSTR,I2C_ICSTR_ICXRDY)!= 1);

   return 1;
}


/**========================================================**/
/** Function nameDSP_disable_interrupts
@briefGlobally disable interrupt.
@return  none
**========================================================**/
void DSP_disable_interrupts(void)
{
   M_CLRBIT(CSR, CSR_GIE);
}


/**========================================================**/
/** Function nameDSP_enable_interrupts
@briefopen interrupt
@return  none
**========================================================**/
void DSP_enable_interrupts(void)
{
   M_SETBIT(CSR, CSR_GIE);
}


/**========================================================**/
/** Function nameEmpty_Interrupt
@brief
@return
**========================================================**/
void Empty_Interrupt(void)
{
	return;
}
