/****************************************************************************\
 *           Copyright (C) 2000 Texas Instruments Incorporated.             *
 *                           All Rights Reserved                            *
 *                                                                          *
 * GENERAL DISCLAIMER                                                       *
 * ------------------                                                       *
 * All software and related documentation is provided "AS IS" and without   *
 * warranty or support of any kind and Texas Instruments expressly disclaims*
 * all other warranties, express or implied, including, but not limited to, *
 * the implied warranties of merchantability and fitness for a particular   *
 * purpose.  Under no circumstances shall Texas Instruments be liable for   *
 * any incidental, special or consequential damages that result from the    *
 * use or inability to use the software or related documentation, even if   *
 * Texas Instruments has been advised of the liability.                     *
 ****************************************************************************
 *                                                                          *
 * Written by :                                                             *
 *            Sebastien Tomas  (VCP2/TCP2)                                  *
 *            Texas Instruments                                             * 
 *            12 May, 2005                                                  *
 *                                                                          *
 * The purpose of this test is to evaluate Bit-Error Rate performance for   *
 * both VCP2 and TCP2 co-processors in the TCI6482 (Himalaya)               *
 *									                                        *
 * This test should not be redistributed for any reason without             *
 * permission.                                                              *
 *                                                                          *
\***************************************************************************/



/*--------------------------------------------------------------------------*/
/* CSL/RTS includes                                                         */
/*--------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <c6x.h>

#include <csl.h>
#include <csl_edma3.h>
#include <csl_tcp2.h>
#include <csl_vcp2.h>
#include <csl_tcp2Aux.h>
#include <csl_vcp2Aux.h>

/*--------------------------------------------------------------------------*/
/* header files                                                             */
/*--------------------------------------------------------------------------*/
#include "generateBits.h"
#include "convolutionalEncode.h"
#include "channelModel.h"
#include "calculateBM.h"
#include "generateTurboEncodedChannelData.h"
#include "tcpvcp2_3gpp.h"
#include "tcp2_frame_types.h"
#include "vcp2_frame_types.h"

/*------------------------------------------------------------------------------*/
/* defines constants                                                            */
/*------------------------------------------------------------------------------*/
#define TAILS_3GPP_RATE3 12
#define TAILS_3GPP2_RATE2 12
#define TAILS_3GPP2_RATE3 18
#define TAILS_3GPP2_RATE4 24

#define  NUM_IT_PER_POINT 1 	/* The same ber point is calculated several times */
								/* to ensure better statistical results		*/						


/*------------------------------------------------------------------------------*/
/* types                                                                        */
/*------------------------------------------------------------------------------*/
typedef struct
{
	U32 nBit;
	U32 nFrame;
	U32 nErroneousBit;
	U32 nErroneousFrame;
	U32 nErroneousBitThreshold;
	U32 nErroneousFrameThreshold;
	U32 nBitThreshold;
	U32 nCmodelHwdiff;
	U32 nErroneousBitCmodel;
	U32 nErroneousFrameCmodel;
	double nErroneousBitAverage;
	double nErroneousBitAverageCmodel;
	U32 nErroneousFrameTotal;	
} BERData;

/*------------------------------------------------------------------------------*/
/* look-up table                                                                */
/*------------------------------------------------------------------------------*/
unsigned char scale_table[448] = {
	253,249,246,242,238,235,232,229,225,222,219,217,214,211,208,206,203,201,198,196,
	194,192,189,187,185,183,181,179,177,175,173,172,170,168,166,165,163,162,160,159,
	157,156,154,153,151,150,149,147,146,145,143,142,141,140,139,138,136,135,134,133,
	132,131,130,129,128,127,126,125,124,123,122,121,121,120,119,118,117,116,116,115,
	114,113,112,112,111,110,110,109,108,107,107,106,105,105,104,103,103,102,
	102,101,100,100,99,99,98,97,97,96,96,95,95,94,94,93,93,92,92,91,91,90,90,89,
	89,88,88,87,87,86,86,85,85,85,84,84,83,83,83,82,82,81,81,81,80,80,79,79,79,78,78,
	78,77,77,77,76,76,75,75,75,74,74,74,73,73,73,73,72,72,72,71,71,71,70,70,70,70,69,
	69,69,68,68,68,68,67,67,67,67,66,66,66,66,65,65,65,65,64,64,64,64,63,63,63,63,62,
	62,62,62,61,61,61,61,61,60,60,60,60,60,59,59,59,59,58,58,58,58,58,57,57,57,57,57,
	57,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,54,53,53,53,53,53,53,52,52,52,
	52,52,52,51,51,51,51,51,51,50,50,50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,
	48,48,48,48,47,47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,45,45,45,45,45,45,45,
	45,44,44,44,44,44,44,44,44,44,43,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,
	42,42,41,41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,40,40,40,39,39,39,39,
	39,39,39,39,39,39,39,38,38,38,38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,
	37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,
	35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,
	33,33
	};

/*------------------------------------------------------------------------------*/
/* function declarations                                                        */
/*------------------------------------------------------------------------------*/
S32 		TestVCP2(VCP2_BaseParams *vcpParameters, unsigned char sigma_fixpt,
					 BERData *ber);
S32 		TestTCP2(TCP2_BaseParams *tcpParameters, unsigned char sigma_fixpt,
					 unsigned char scale_fixpt, BERData *ber);
extern int  VCP2(int big_endian, int sd_endian, int *configuration_parameters,
                     char *branch_metrics, int *sd_or_hd, int *Output_Param,
                     int *block_size );



/*------------------------------------------------------------------------------*/
/* main testbench - VCP/TCP BER                                                 */
/*------------------------------------------------------------------------------*/
int main(void)
{
Uint32 			i,Nc;
double 			sigma, scale, sigma_min, sigma_max;
unsigned char 	sigma_fixpt, scale_fixpt;
BERData 		TCPber,VCPber;		
FILE 			*fileTcp,*fileVcp,*filetraceTcp;
char 			filename[1000];
TCP2_BaseParams 	tcpParameters;
VCP2_BaseParams 	vcpParameters;
int 			AverageCount;

	AverageCount = (U32)NUM_IT_PER_POINT;
	

#ifdef RUN_TEST_TCP

	strcpy(filename,"tcp2results.txt");
	if ((fileTcp=fopen(filename,"w"))==NULL)
	{
		printf("\nCannot open %s\n",filename);
		return 1;
	}

	strcpy(filename,"tcp2traceresults.txt");
	if ((filetraceTcp=fopen(filename,"w"))==NULL)
	{
		printf("\nCannot open %s\n",filename);
		return 1;
	}

	for( Nc = 0 ; Nc < TCP2_NUM_CURVES ; Nc++ ){	
		tcpParameters.frameLen      = TcpCurveParameters[Nc].frameLen;
		tcpParameters.rate          = TcpCurveParameters[Nc].rate;       
		tcpParameters.intFlag       = TcpCurveParameters[Nc].intFlag;    
		tcpParameters.outParmFlag   = TcpCurveParameters[Nc].outParmFlag;
		tcpParameters.standard      = TcpCurveParameters[Nc].standard;   
		tcpParameters.snr           = TcpCurveParameters[Nc].snr;        
		tcpParameters.prologSize    = TcpCurveParameters[Nc].prologSize; 
		tcpParameters.maxIter       = TcpCurveParameters[Nc].maxIter;    
		tcpParameters.mode          = 0;
		tcpParameters.map           = 0;
		tcpParameters.maxStarEn     = 1;
		tcpParameters.prologRedEn   = 0;
		tcpParameters.minIter       = tcpParameters.maxIter;
		tcpParameters.inputSign     = 0;
		tcpParameters.outputOrder   = 0;
		tcpParameters.numCrcPass    = 0;
		tcpParameters.crcLen        = 0;
		tcpParameters.crcPoly       = 0;
        for (i=0;i<16;i++) tcpParameters.extrScaling[i] = 0;

		TCPber.nBitThreshold=TcpCurveParameters[Nc].nBitThreshold;
		TCPber.nErroneousFrameThreshold=TcpCurveParameters[Nc].nErroneousFrameThreshold;
		TCPber.nErroneousBitThreshold=TCPber.nErroneousFrameThreshold*tcpParameters.frameLen;
	
		sigma_min=TcpCurveParameters[Nc].sigmaMin;
		sigma_max=TcpCurveParameters[Nc].sigmaMax;
		
		fprintf(fileTcp,"\nFrame Type f:%u r:%u p:%u i:%u s:%u ;\n", \
	                tcpParameters.frameLen, tcpParameters.rate,  \
	                tcpParameters.prologSize, tcpParameters.maxIter, \
	                tcpParameters.snr);
	                
		fprintf(filetraceTcp,"\nFrame Type f:%u r:%u p:%u i:%u s:%u ;\n", \
	                tcpParameters.frameLen, tcpParameters.rate,  \
	                tcpParameters.prologSize, tcpParameters.maxIter, \
	                tcpParameters.snr);
	                
	
		printf("\n");
			
		for(sigma = sigma_max ; sigma >= sigma_min ; sigma -= 0.01 ){
		 	printf("TCP2 test Sigma=%5.5f\n",sigma);
		 	/*	w.c. sigma = 1.41 -> mysigma = 0xB5,
		 		in uns Q7: I.FFF FFFF
		 	*/	
		 	sigma_fixpt  = (unsigned char) (sigma * 128.0 + .5);  
		 
			/* The systematics and parities must be scaled by a factor -2a/sigma^2. 
			    The received symbols r can be written as r = ax + n = +/-a + n assuming
			    BPSK modulation and as we have a BPSK modulation for x and n is AWGN random
		   		variable with 0 mean and variance sigma^2. 
		   		In all simulation, a is constant and equal to 1.
		   		In real life, a is typically equal to the square root of the energy per symbol.
		   		Both a and the noise variance must be estimated.
		 	*/
		 
		 	scale = 2/(sigma*sigma);
		 
		 	/* value range for scale is from 1 -> 4.75
		 		w.c. scale = 4.08 -> myscale = 0x83,
		 		in uns Q5: III.F FFFF
		 	*/
		 	scale_fixpt = (unsigned char) (scale * 32.0 + .5);
		 
	
    	 	TCPber.nErroneousBitAverage = 0;
    	 	TCPber.nErroneousBitAverageCmodel = 0;
    	 	TCPber.nErroneousFrameTotal = 0;
    	 	fprintf(fileTcp,"%e ;\t",(double) sigma);    	 
    	
    		for( i = 0 ; i<AverageCount ; i++)
  			{				
				printf("TCP2 iteration =%d\n",i);
				TCPber.nBit=0;
				TCPber.nFrame=0;
				TCPber.nErroneousBit=0;
				TCPber.nErroneousFrame=0;
				TCPber.nErroneousBitCmodel=0;
				TCPber.nErroneousFrameCmodel=0;
 				TCPber.nCmodelHwdiff=0;

				TestTCP2(&tcpParameters,sigma_fixpt,scale_fixpt,&TCPber);
					
				fprintf(fileTcp,"%e ;\t",(double)TCPber.nErroneousBit/(double)TCPber.nBit);
				printf("\nBER : %e ;\n",(double)TCPber.nErroneousBit/(double)TCPber.nBit);
				fprintf(fileTcp,"%u ;\t%u ;\t%u ;\t%u ;\t",TCPber.nErroneousBit,TCPber.nBit,TCPber.nFrame,TCPber.nErroneousFrame);
				fprintf(fileTcp,"%e ;\t",(double)TCPber.nErroneousBitCmodel/(double)TCPber.nBit);
				TCPber.nErroneousBitAverage += (double)TCPber.nErroneousBit/(double)TCPber.nBit;
				TCPber.nErroneousFrameTotal += TCPber.nErroneousFrame;
								
			} /* end for i */
			fprintf(fileTcp,"%e ;\n",(double)TCPber.nErroneousBitAverage/AverageCount);
			fprintf(filetraceTcp,"TCP2  sigma:\t %5.5f \t Hardware BER:\t%e ;\n", sigma,(double)TCPber.nErroneousBitAverage/AverageCount);
			printf("TCP2 Hardware BER: %e ;\t",(double)TCPber.nErroneousBitAverage/AverageCount);
			printf("\n");						

			/* stop if no erroneous frames out of 10*nbitthreshold */
			if (TCPber.nErroneousFrameTotal == 0) sigma = 0;

		} /*end for sigma */
	
	} /* end for Nc*/		
	fclose(filetraceTcp);
	fclose(fileTcp);	

#endif


#ifdef RUN_TEST_VCP

	strcpy(filename,"vcp2results.txt");
	if ((fileVcp=fopen(filename,"w"))==NULL)
	{
		printf("\nCannot open %s\n",filename);
		return 1;
	}
	   		
	for( Nc = 0 ; Nc < VCP2_NUM_CURVES ; Nc++ )
	{	
    	vcpParameters.frameLen    		= VcpCurveParameters[Nc].frameLen;
		vcpParameters.rate        		= VcpCurveParameters[Nc].rate;
		vcpParameters.constLen    		= VcpCurveParameters[Nc].constLen;
		vcpParameters.yamTh       		= VcpCurveParameters[Nc].yamTh;
		vcpParameters.stateNum   		= VcpCurveParameters[Nc].stateNum;
		vcpParameters.decision   		= VcpCurveParameters[Nc].decision;
		vcpParameters.readFlag    		= VcpCurveParameters[Nc].readFlag;
		vcpParameters.tbConvrgMode      = 0;
		vcpParameters.tailBitEnable     = 0;
		vcpParameters.traceBackIndex    = 0;
		vcpParameters.outOrder          = 0;
		vcpParameters.perf              = VCP2_PERF_MOST_CRITICAL;
		
		VCPber.nBitThreshold	  		= VcpCurveParameters[Nc].nBitThreshold;
		VCPber.nErroneousFrameThreshold	= VcpCurveParameters[Nc].nErroneousFrameThreshold;
		VCPber.nErroneousBitThreshold	= VCPber.nErroneousFrameThreshold*vcpParameters.frameLen;

		sigma_min						= VcpCurveParameters[Nc].sigmaMin;
		sigma_max						= VcpCurveParameters[Nc].sigmaMax;

		fprintf(fileVcp,"\nFrame Type f:%u r:%u k:%u ;\n",vcpParameters.frameLen,vcpParameters.rate,vcpParameters.constLen);
	                
		printf("\n");
		
		for(sigma = sigma_max ; sigma >= sigma_min ; sigma -= 0.05 ){   	
		 	printf("VCP2 test Sigma=%5.5f\n",sigma);
		 	//w.c. sigma = 1.41 -> mysigma = 0xB5,
		 	//in uns Q7: I.FFF FFFF
		 	sigma_fixpt  = (unsigned char) (sigma * 128.0 + .5);   
    										
    	 	VCPber.nErroneousBitAverage = 0;
    	 	VCPber.nErroneousBitAverageCmodel = 0;
    	 	VCPber.nErroneousFrameTotal=0;

    		for( i = 0 ; i<AverageCount ; i++)
  			{				
				VCPber.nBit=0;
				VCPber.nFrame=0;
				VCPber.nErroneousBit=0;
				VCPber.nErroneousFrame=0;
				VCPber.nErroneousBitCmodel=0;
				VCPber.nErroneousFrameCmodel=0;
				VCPber.nCmodelHwdiff=0;				

				TestVCP2(&vcpParameters,sigma_fixpt,&VCPber);
					
				fprintf(fileVcp,"%e ;\t",(double)VCPber.nErroneousBit/(double)VCPber.nBit);
				printf("BER : %e \n",(double)VCPber.nErroneousBit/(double)VCPber.nBit);
				printf("BER (C Model) : %e \n",(double)VCPber.nErroneousBitCmodel/(double)VCPber.nBit);
				fprintf(fileVcp,"%u ;\t%u ;\t%u ;\t%u ;\t",VCPber.nErroneousBit,VCPber.nBit,VCPber.nFrame,VCPber.nErroneousFrame);
				fprintf(fileVcp,"%e ;\t",(double)VCPber.nErroneousBitCmodel/(double)VCPber.nBit);
				fprintf(fileVcp,"%u ;\t%u ;\t%u ;\t%u ;\t",VCPber.nErroneousBitCmodel,VCPber.nBit,VCPber.nFrame,VCPber.nErroneousFrameCmodel);	
				fprintf(fileVcp,"%u ;\t",VCPber.nCmodelHwdiff);
				
				VCPber.nErroneousBitAverage += (double)VCPber.nErroneousBit/(double)VCPber.nBit;
				VCPber.nErroneousBitAverageCmodel += (double)VCPber.nErroneousBitCmodel/(double)VCPber.nBit;
				VCPber.nErroneousFrameTotal += VCPber.nErroneousFrame;
					
			} /* end for i */
			fprintf(fileVcp,"%e ;\t",(double)VCPber.nErroneousBitAverage/AverageCount);
			fprintf(fileVcp,"%e ;\n",(double)VCPber.nErroneousBitAverageCmodel/AverageCount);
			printf("VCP2 Hardware BER: %e ;\t",(double)VCPber.nErroneousBitAverage/AverageCount);
			printf("VCP C-Model  BER: %e ;\n",(double)VCPber.nErroneousBitAverageCmodel/AverageCount);
			
			/* stop if no erroneous frames out of 10*nbitthreshold */
			if (VCPber.nErroneousFrameTotal == 0) sigma = 0;

		} /* end for sigma */
	
	} /* end for Nc*/		

	fclose(fileVcp);
#endif 

//-------------------------------------------------------//

		
	return 0;		

}


unsigned char varianceEstim(const unsigned int * restrict output, const unsigned int length)
{
 int i=0,j=0;  
 unsigned char ScaleFixpt; 
 int sigma_2;
 unsigned int e, f, EX2=0, E2X=1;     	 	

#pragma UNROLL(2);
   for (i=0,j=0;i<length;i+=4,j++){
		e=output[j];		
		f = _minu4(e, _sub4(0x00000000,e));
		EX2 += _dotpu4(f, f);
	}	   	 
 
 EX2=EX2/(length-1); //Q8
 sigma_2 = (EX2-(E2X<<8))-65;
  
 /* Should call look up table to get unsigned char Q5 scale_fixpt var */ 
 if (sigma_2>0) { 	
 	ScaleFixpt = scale_table[sigma_2];
 }else{
 	ScaleFixpt = (unsigned char)255;
 }
 // beyond look-up table
 if (sigma_2>447) ScaleFixpt = (unsigned char)255; 	
 	 
 return (ScaleFixpt);
}


int TestTCP2(TCP2_BaseParams *tcpParameters, U8 sigma_fixpt, U8 scale_fixpt, BERData *ber)
{
TCP2_ConfigIc    TcpConfigIc;
TCP2_Params      TcpConfigParms;		
Uint16 			index;
TCP2_UserData     hXabData;
U32 *in;
U32 *encoded;
U32 *modulated;
U32 *corrupted;
U16 *interleaverTable;
U32 *UserData;
U32 *UserDataRate5;
U32 *packedHD;
U32 nTails;
U32 i,j;
U8  scale_fixpt_est;
int hTcpOutParms[3]={0,0,0};
Uint32 mask=0xFFFFFFFF;
Uint32 decisionPerWord;
Uint32 errorInFrame;

	if (tcpParameters->standard==0) /* 3GPP */
	{
		nTails=TAILS_3GPP_RATE3;
		tcpParameters->rate=3;
	}
	else	/* 3GPP2 */
	{
		switch (tcpParameters->rate)
		{
			case 2:
				nTails=TAILS_3GPP2_RATE2;
				break;
			case 3:
				nTails=TAILS_3GPP2_RATE3;
				break;
			case 4:
				nTails=TAILS_3GPP2_RATE4;
				break;
			default:
				printf("\nRate not allowed\n");
				return 1;
		}
	}

	edma3Init();

    /* Generate all TCP parameters */
	TCP2_genParams(tcpParameters, &TcpConfigParms);


	if ((in=(U32*)malloc(sizeof(U32)*TCP2_normalCeil(tcpParameters->frameLen,32)))==NULL)	{
		printf("\nMemory allocation error for variable in\n");
		return 1;
	}		
	if ((interleaverTable=(U16*)malloc(sizeof(U16)*tcpParameters->frameLen+10))==NULL)	{//pm +10
		printf("\nMemory allocation error for variable interleaverTable\n");
		return 1;
	}
	if ((encoded=(U32*)malloc(sizeof(U32)*TCP2_normalCeil((tcpParameters->frameLen*tcpParameters->rate+nTails),32)))==NULL){
		printf("\nMemory allocation error for variable encoded\n");
		return 1;
	}
	if ((modulated=(U32*)malloc(sizeof(U32)*(((tcpParameters->frameLen*tcpParameters->rate+nTails)>>2)+10)))==NULL){
		printf("\nMemory allocation error for variable modulated\n");
		return 1;
	}	
	if ((corrupted=(U32*)malloc(sizeof(U32)*(((tcpParameters->frameLen*tcpParameters->rate+nTails)>>2)+10)))==NULL)	{
		printf("\nMemory allocation error for variable scaled\n");
		return 1;
	}	
	if ((packedHD=(U32*)malloc(sizeof(U32)*TCP2_normalCeil(tcpParameters->frameLen,32)))==NULL)	{
		printf("\nMemory allocation error for variable packedHD\n");
		return 1;
	}	
	if ((UserData = (U32*)malloc(sizeof(U32)*(((tcpParameters->frameLen*tcpParameters->rate+nTails)>>2)+24)))==NULL)	{
		printf("\nMemory allocation error for variable UserData\n");
		return 1;
	}
	if ((UserDataRate5 = (U32*)malloc(sizeof(U32)*(((tcpParameters->frameLen*5)>>2)+24)))==NULL)	{
		printf("\nMemory allocation error for variable UserData aligned on rate 5\n");
		return 1;
	}

	while (ber->nBit<ber->nBitThreshold && ber->nErroneousFrame<ber->nErroneousFrameThreshold && ber->nErroneousBit<ber->nErroneousBitThreshold)
	{
	
		ber->nBit+=tcpParameters->frameLen;
		ber->nFrame+=1;	
	 	
		if (GenerateBits(in,tcpParameters->frameLen))
		return 1;		
		
		if (TurboEncode3GPP(in,encoded,interleaverTable,tcpParameters->frameLen,tcpParameters->rate))
		return 1;

		MapOnBaseband(encoded,tcpParameters->frameLen*tcpParameters->rate+nTails,modulated);
				
		AddAWGN_fixpt(modulated,corrupted,sigma_fixpt,tcpParameters->frameLen*tcpParameters->rate+nTails);
		
		scale_fixpt_est = varianceEstim(corrupted, tcpParameters->frameLen*tcpParameters->rate+nTails);
				
		TCP2_quantizeSoft(corrupted,UserData,scale_fixpt_est,tcpParameters->frameLen*tcpParameters->rate+nTails);	

		TCP2_deintUnpunctSoft3((S8*)UserData,UserDataRate5,interleaverTable,tcpParameters->frameLen);

        /* Pointer to the X, A, and B data  */	    
        index = TcpConfigParms.frameLen * TcpConfigParms.rate;
        hXabData = &((U8*)UserData)[index];
		/* Calculate the TCP IC values */
        TCP2_genIc(&TcpConfigParms, (TCP2_TailData*)hXabData, &TcpConfigIc);
#ifdef RUN_TEST_TCP
#ifdef _LITTLE_ENDIAN
        //TCP2_setInterEndian(CSL_TCP2_TCPEND_ENDIAN_I0_32BITS);
#else
        TCP_setInterEndian(CSL_TCP2_TCPEND_ENDIAN_I0_NATIVE16);
#endif
    	if (Tcp2Test(&TcpConfigParms,&TcpConfigIc,(U8 *)UserDataRate5,interleaverTable,packedHD,(U32*)hTcpOutParms))
			return 1;
#endif

    	/* Compute # bit errors */
    	errorInFrame=0; 
        decisionPerWord=32;
        
			for (j=0;j<(tcpParameters->frameLen/decisionPerWord);j++)	{

				if (in[j]!=packedHD[j])	{
					if (!errorInFrame)	{
						errorInFrame++;
					}
					for (i=0;i<32/*tcpParameters.frameLen%decisionPerWord*/;i++)			{
						if (((in[j]>>i) & 0x1) != ((packedHD[j]>>i) & 0x1)){
							ber->nErroneousBit++;
						}
					}
				}
			}
			
			if (tcpParameters->frameLen%decisionPerWord) 		{
				mask=0xFFFFFFFF;
				mask>>=(32-(tcpParameters->frameLen%decisionPerWord));
				if ((in[tcpParameters->frameLen/decisionPerWord]&mask)!=(packedHD[tcpParameters->frameLen/decisionPerWord]&mask)) 	{
					if (!errorInFrame) 	{
						errorInFrame++;
					}
					for (i=0;i<tcpParameters->frameLen%decisionPerWord;i++) {
						if (((in[j]>>i) & 0x1) != ((packedHD[j]>>i) & 0x1)) {
							ber->nErroneousBit++;
						}
					}
				}
			}

		if (errorInFrame) {
			errorInFrame=0;
			ber->nErroneousFrame++;
		}
	}
	
	free(in);	
	free(corrupted);
	free(modulated);
	free(encoded);		
	free(UserData);
	free(UserDataRate5);
	free(interleaverTable);
	free(packedHD);		

	edma3Release();

	return 0;
}


S32 TestVCP2(VCP2_BaseParams *vcpParameters, unsigned char sigma_fixpt, BERData *ber)
{

	U32 j,i;
	U32 *in;
	U32 *encoded;
	S8  *branch_metrics;
	U32 *softInput;
	U32 *modulated;
	U32 *corrupted;
	U32 output[3];
	U32 *decisions,*decisions_soft,nDecisionWords;
	U32 mask=0xFFFFFFFF;
	U32 softInputResolution;
	U32 length, decisionPerWord;
	S16 sd; 	
	U32 shift;	
    VCP2_Params     		VcpConfigParms; 
    VCP2_ConfigIc    VcpConfigIc;
    U16          	 poly[4] = {0,0,0,0};    
 	U32 errorInFrame;
	U32 errorInFrame_cmodel;
	S32 block_size;
	
	edma3Init();

	/* Fill out VCP2_ parameters */
	VCP2_genParams(vcpParameters, &VcpConfigParms);  
    
	/* Calculate the VCP2_ IC values */
	VCP2_genIc(&VcpConfigParms, &VcpConfigIc);
			
	if (VcpConfigParms.poly0) poly[0] = VcpConfigParms.poly0|0x100;
	if (VcpConfigParms.poly1) poly[1] = VcpConfigParms.poly1|0x100;
	if (VcpConfigParms.poly2) poly[2] = VcpConfigParms.poly2|0x100;
	if (VcpConfigParms.poly3) poly[3] = VcpConfigParms.poly3|0x100; 

		/* if convergent mode, frame size must be incremented by the convergence distance */
		if (VcpConfigParms.traceBack==2)
		{
			length=VcpConfigParms.frameLen+VcpConfigParms.convDist;
		}
		else
		{
			length=VcpConfigParms.frameLen;
		}

		/* allocate buffer for information data */
		if ((in=(U32*)calloc(VCP2_normalCeil(length,32),sizeof(U32)))==NULL)
		{
			return 1;
		}
		/* allocate buffer for encoded data */
		if ((encoded=(U32*)calloc(VCP2_normalCeil(((length+vcpParameters->constLen-1)*vcpParameters->rate),32),sizeof(U32)))==NULL)
		{
			return 1;
		}
		/* allocate buffer for modulated data 0->1, 1->-1 (+10 PAD) */
		if ((modulated=(U32*)calloc((((length+vcpParameters->constLen-1)*vcpParameters->rate)>>2)+10,sizeof(U32)))==NULL)
		{
			return 1;
		}
		/* allocate buffer for corrupted data (+10 PAD)*/
		if ((corrupted=(U32*)calloc((((length+vcpParameters->constLen-1)*vcpParameters->rate)>>2)+10,sizeof(U32)))==NULL)
		{
			return 1;
		}
		/* allocate buffer for scaled soft inputs metrics */
		if ((softInput=(U32*)calloc((((length+vcpParameters->constLen-1)*vcpParameters->rate)>>2)+10,sizeof(U32)))==NULL)
		{
			return 1;
		}
		/* allocate buffer for branch metrics */
		if ((branch_metrics=(S8*)calloc((length+vcpParameters->constLen-1)*(0x1<<(vcpParameters->rate-1)),sizeof(S8)))==NULL) 
		{
			return 1;
		}
		/* allocate memory for decisions */
		if (vcpParameters->decision)
		{
/*
 * 			 the VCP2 function will write 1 SD per 32-bit word 
 * 			 but the pointer passed to the VCP2() function will point
 * 			 to SD packed data (2 per 32-bit word)
 */
			 decisionPerWord=1;
		}
		else
		{
			decisionPerWord=32;
		}
		if (VcpConfigParms.traceBack==2)
		{
			nDecisionWords=VCP2_normalCeil(length,decisionPerWord);
		}
		else
		{
			nDecisionWords=VCP2_normalCeil(length+vcpParameters->constLen-1,decisionPerWord);
		}
		if ((decisions=(U32*)calloc(nDecisionWords+10,sizeof(U32)))==NULL) 
		{
			return 1;
		}	

		if ((decisions_soft=(U32*)calloc(nDecisionWords+10,sizeof(U32)))==NULL) 
		{
			return 1;
		}



//	while (ber->nBit<ber->nBitThreshold && ber->nErroneousFrame<ber->nErroneousFrameThreshold && ber->nErroneousBit<ber->nErroneousBitThreshold)
//	{
		
		if (GenerateBits(in,length))
		{
			return 1;
		}    
		
		ber->nBit+=vcpParameters->frameLen;
		ber->nFrame+=1;
		
	
		/* Convolutional encoding */
		if (ConvolutionalEncode(in,encoded,length,poly))
		{
			return 1;
		}
	
		MapOnBaseband(encoded,(length+vcpParameters->constLen-1)*vcpParameters->rate,modulated);

		AddAWGN_fixpt(modulated,corrupted,sigma_fixpt,(length+vcpParameters->constLen-1)*vcpParameters->rate);

		/* make sure input soft decision (corrupted data) will not create overflow */
		
		if ((softInputResolution=VCP2_getSoftRes(vcpParameters->rate,vcpParameters->constLen))==0)
		{
			return 1;
		}
		
		VCP2_quantizeSoft(corrupted,softInput,softInputResolution,(length+vcpParameters->constLen-1)*vcpParameters->rate);						
		
    	/* compute branch metrics - not optimized in this framework */
		if (ComputeBranchMetrics((S8 *)softInput,branch_metrics,vcpParameters->rate,(length+vcpParameters->constLen-1)))
		{
			return 1;
		}
		
        #ifdef RUN_TEST_VCP		
		if (Vcp2Test(&VcpConfigParms,&VcpConfigIc,(U8*)branch_metrics,decisions,output))
		{
			return 1;
		}
        #endif

		
		#ifdef VCP2_C_MODEL
		if (VCP2(1/*big_endian*/, 0/*sd_endian*/, (S32*)&VcpConfigIc,branch_metrics,
		         (S32*)decisions_soft,(S32*)output,&block_size ))
		{
			return 1;
		}
		#endif //VCP2_C_MODEL
		
			
		/* compare */
		/* need to recalculate decision per word for the actual comparision */
		if (vcpParameters->decision)
		{
			 decisionPerWord=2;
		}
		else
		{
			decisionPerWord=32;
		}

		errorInFrame=0;
		errorInFrame_cmodel=0;
		if (!vcpParameters->decision) /* hard decisions */
		{
			for (j=0;j<(vcpParameters->frameLen/decisionPerWord);j++)
			{
				#ifdef VCP2_C_MODEL
				if (decisions[j]!=decisions_soft[j]){
					ber->nCmodelHwdiff++;
				}
				#endif //VCP2_C_MODEL
				if (in[j]!=decisions[j]){
					if (!errorInFrame)	{
						errorInFrame++;
					}
					for (i=0;i<32/*config->f%decisionPerWord*/;i++) {
						if (((in[j]>>i) & 0x1) != ((decisions[j]>>i) & 0x1)) {
							ber->nErroneousBit++;
						}
					}					
				}
				#ifdef VCP2_C_MODEL
				if (in[j]!=decisions_soft[j]){
					if (!errorInFrame_cmodel) {
						errorInFrame_cmodel++;
					}
					for (i=0;i<32/*config->f%decisionPerWord*/;i++) {
						if (((in[j]>>i) & 0x1) != ((decisions_soft[j]>>i) & 0x1)) {
							ber->nErroneousBitCmodel++;
						}
					}
				}
				#endif //VCP2_C_MODEL	
			}
			if (vcpParameters->frameLen%decisionPerWord)
			{
				mask=0xFFFFFFFF;
				mask>>=(32-(vcpParameters->frameLen%decisionPerWord));
				
				if ((decisions[vcpParameters->frameLen/decisionPerWord]&mask)!=(decisions_soft[vcpParameters->frameLen/decisionPerWord]&mask))
				{
				ber->nCmodelHwdiff++;
				}
				if ((in[vcpParameters->frameLen/decisionPerWord]&mask)!=(decisions[vcpParameters->frameLen/decisionPerWord]&mask))
				{
					if (!errorInFrame)
					{
						errorInFrame++;
					}
					for (i=0;i<vcpParameters->frameLen%decisionPerWord;i++)
					{
						if (((in[j]>>i) & 0x1) != ((decisions[j]>>i) & 0x1))
						{
							ber->nErroneousBit++;
						}
					}
				}
				#ifdef VCP2_C_MODEL
				if ((in[vcpParameters->frameLen/decisionPerWord]&mask)!=(decisions_soft[vcpParameters->frameLen/decisionPerWord]&mask)) {
					if (!errorInFrame_cmodel) 	{
						errorInFrame_cmodel++;
					}
					for (i=0;i<vcpParameters->frameLen%decisionPerWord;i++) {
						if (((in[j]>>i) & 0x1) != ((decisions_soft[j]>>i) & 0x1)) {
							ber->nErroneousBitCmodel++;
						}
					}
				}
				#endif //VCP2_C_MODEL	
			}
		}
		else /* soft decisions */
		{
			shift=0;
			i=0;
			for (j=0;j<(vcpParameters->frameLen/decisionPerWord);j++)	{
				if (shift==32) 	{
					shift=0;
					i++;
				}
				/* 1st DECISION in word */
				sd=(S16)(decisions[j]&0xFFFF);
				if (sd<0) /* negative number -> bit=1 */ 	{
					if (!((in[i]>>shift)&0x1)) {
						if (!errorInFrame) {
							errorInFrame++;
						}
						ber->nErroneousBit++;
					}
					
				}
				else /* positive number -> bit=0 */ {
					if ((in[i]>>shift)&0x1) {
						if (!errorInFrame) 	{
							errorInFrame++;
						}
						ber->nErroneousBit++;
					}
				}
				shift++;
				
				if (shift==32) {
					shift=0;
					i++;
				}
				/* second decision */
				sd=(S16)(decisions[j]>>16);
				if (sd<0) /* negative number -> bit=1 */ {
					if (!((in[i]>>shift)&0x1)) 	{
						if (!errorInFrame) 	{
							errorInFrame++;
						}
						ber->nErroneousBit++;
					}
					
				}
				else /* positive number -> bit=0 */ {
					if ((in[i]>>shift)&0x1) {						
						if (!errorInFrame) 	{
							errorInFrame++;
						}
						ber->nErroneousBit++;						
					}
				}
				shift++;
				
			}
			
			if (vcpParameters->frameLen%decisionPerWord) {
				mask=0xFFFF;
				if (shift==32) 	{
					shift=0;
					i++;					
				}
				/* 1st DECISION in word */
				sd=(S16)decisions[vcpParameters->frameLen/decisionPerWord]&0xFFFF;
				if (sd<0) /* negative number -> bit=1 */ {
					if (!((in[i]>>shift)&0x1)) 	{						
						if (!errorInFrame) {
							errorInFrame++;
						}
						ber->nErroneousBit++;						
					}
					
				}
				else /* positive number -> bit=0 */ {
					if ((in[i]>>shift)&0x1) {
						if (!errorInFrame) 	{
							errorInFrame++;
						}
						ber->nErroneousBit++;
					}
				}
				shift++;
			}
		}
		 		
		if (errorInFrame) {
			errorInFrame=0;
			ber->nErroneousFrame++;
		}
		#ifdef VCP2_C_MODEL		
		if (errorInFrame_cmodel) {
			errorInFrame_cmodel=0;
			ber->nErroneousFrameCmodel++;
		}
		#endif //VCP2_C_MODEL
//	}

		free(in);
		free(encoded);
		free(modulated);
		free(corrupted);
		free(softInput);
		free(branch_metrics);
		free(decisions);
		free(decisions_soft);

		edma3Release();

	return 0;

}

