/****************************************************************************\
 *           Copyright (C) 2000 Texas Instruments Incorporated.             *
 *                           All Rights Reserved                            *
 *                                                                          *
 * GENERAL DISCLAIMER                                                       *
 * ------------------                                                       *
 * All software and related documentation is provided "AS IS" and without   *
 * warranty or support of any kind and Texas Instruments expressly disclaims*
 * all other warranties, express or implied, including, but not limited to, *
 * the implied warranties of merchantability and fitness for a particular   *
 * purpose.  Under no circumstances shall Texas Instruments be liable for   *
 * any incidental, special or consequential damages that result from the    *
 * use or inability to use the software or related documentation, even if   *
 * Texas Instruments has been advised of the liability.                     *
 ****************************************************************************
 *                                                                          *
 * Written by :                                                             *
 *            Sebastien Tomas  (VCP2/TCP2)                                  *
 *            Texas Instruments                                             * 
 *            12 May, 2005                                                  *
 *                                                                          *
 * The purpose of this test is to evaluate Bit-Error Rate performance for   *
 * both VCP2 and TCP2 co-processors in the TCI6482 (Himalaya)               *
 *									                                        *
 * This test should not be redistributed for any reason without             *
 * permission.                                                              *
 *                                                                          *
\***************************************************************************/


#include "channelModel.h"

/*
This function generates gaussian distributed random values
in the range [-128, 127] with standard deviation equal to
sqrt(steps). The gaussian distribution is approximated 
with a binomial distribution and this distribution is 
calculated using "random walk". Since number of steps is 
large this makes a very accurate approximation gaussian 
distribution.
*/

char wgn_fixpt(unsigned char sigma)
{
  short value = 0;
  unsigned int number_of_steps;
  unsigned int number_of_iter;
  int i;
  unsigned int mask, remaining_steps;
  unsigned int positive_steps;
  
  number_of_steps = (short) sigma * (short) sigma;
  
  /*
  if sigma is even (odd) we always get an even (odd) number
  of steps and hence even (odd) output value. To correct that
  we make one more step (with probability=25%) or one less 
  step (p=25%) or leave it as is (p=50%).
  */
  mask = rand()&0x3;
  number_of_steps += (mask>>1)-(mask&1);
  
  /* in the loop below, 30 steps are calculated at once.
  Get # of iteration in the loop */
  
  number_of_iter = number_of_steps/30;
  remaining_steps = number_of_steps-number_of_iter*30; 

 /* rand() in rts lib is returning a value in [0, 32767], 
    i.e. the rightmost bit is always zero. Hence we get 
    15 random bits from each call to rand()
 */
    
  for(i=0; i<number_of_iter; i++)
  {
  	mask = ((rand() & 0x7FFF)<<15) | (rand() & 0x7FFF);
    positive_steps = _dotpu4(_bitc4(mask), 0x01010101);
  	value += (positive_steps<<1)-30; //i.e. bitc(mask) - (30 - bitc(mask))
  }
  
 /* do the remaining steps (if sigma^2 was not divisible by 30) */
  mask = ((rand() & 0x7FFF)<<15) | (rand() & 0x7FFF);;
  mask >>= (30-remaining_steps);
  positive_steps = _dotpu4(_bitc4(mask), 0x01010101);
  value += (positive_steps<<1)-remaining_steps;
 
 /* saturate and cast to char */
 if (value>127) value = 127;
 if (value<-128) value =-128;
 return ((char) value);
}


void AddAWGN_fixpt(U32 *in,U32 *out, const U8 sigma_fixpt, const U32 length)
{
U32 wgn_dat,sigma4_fixpt,noise,i;
S32 hi_temp1,lo_temp1,hi_temp2,lo_temp2;
double temp, temp1, temp2;
U32 j=0;

#ifdef SATURATE_ON_HIGH_SNR
S8 *in_char, *out_char;

	in_char = (S8 *) in;
	out_char = (S8 *) out;
    if (sigma_fixpt<20)
    {
    	for(i=0; i < length; i++){
    		if (in_char[i]>0) out_char[i]=127;
    		else out_char[i]=-128;
    	}	
    }
#else
    if (sigma_fixpt<20)
    {
    	for(i=0; i < length; i+=4, j++){
    		out[j]=in[j];
    	}	
    }
#endif    
    else{
      for(i=0; i < length; i+=4, j++){
    	/* Generate Noise - 4 noisy bytes */
    	wgn_dat = _packl4(_pack2(wgn_fixpt(32),wgn_fixpt(32)),
    						_pack2(wgn_fixpt(32),wgn_fixpt(32)));
        
        /* duplicate sigma in 4 bytes */
        sigma4_fixpt = _packl4(_pack2(sigma_fixpt,sigma_fixpt),_pack2(sigma_fixpt,sigma_fixpt));
        
        /* scale 4 gaussian values by sigma */
        // if wgn_dat is in Q5 with stdev 1 in Q5
        // and sigma4_fixpt holds unsigned Q7
        // then temp is in signed Q12
        temp=_mpysu4(wgn_dat,sigma4_fixpt);
                
        /* saturate into upper byte */
         temp1=_smpy2(_hi(temp),0x40004000);
         temp2=_smpy2(_lo(temp),0x40004000);
        // here, upper 16 bits of temp1 and temp2 will be in 
        // signed Q11, i.e. upper 8 bits are in Q3.
        // we need Q4 , hence scale one position left.
         
         hi_temp1=_sshl(_hi(temp1),1);
         lo_temp1=_sshl(_lo(temp1),1);
         hi_temp2=_sshl(_hi(temp2),1);
         lo_temp2=_sshl(_lo(temp2),1);
        
         /*hi_temp1=_hi(temp1);
         lo_temp1=_lo(temp1);
         hi_temp2=_hi(temp2);
         lo_temp2=_lo(temp2);*/
        
        
        /* pack 4 upper bytes of sshl's */
         noise=_packh4(_packh2(hi_temp1,lo_temp1),
                         _packh2(hi_temp2,lo_temp2));
      
         out[j]=_add4(in[j],noise);
                
    }        /* end for */
   }		/* end else*/ 

}


void MapOnBaseband(unsigned int* in,int len, unsigned int* out)
{
  unsigned int words,i;
  unsigned int inputWord;

  words=(len+31)>>5; /* round up */  

  for(i=0;i<words;i++)
  {
#ifdef _LITTLE_ENDIAN
//     inputWord=_bitr(~*in++);
	 inputWord=~*in++;
     *out++=_sub4(_xpnd4(inputWord>>0 ) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>4 ) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>8 ) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>12) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>16) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>20) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>24) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>28) & 0x20202020 , 0x10101010);    
#else
     inputWord=_bitr(~*in++);
     *out++=_sub4(_xpnd4(inputWord>>28) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>24) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>20) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>16) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>12) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>8 ) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>4 ) & 0x20202020 , 0x10101010);    
     *out++=_sub4(_xpnd4(inputWord>>0 ) & 0x20202020 , 0x10101010);    
#endif
  }
    
}

