/****************************************************************************\
 *           Copyright (C) 2000 Texas Instruments Incorporated.             *
 *                           All Rights Reserved                            *
 *                                                                          *
 * GENERAL DISCLAIMER                                                       *
 * ------------------                                                       *
 * All software and related documentation is provided "AS IS" and without   *
 * warranty or support of any kind and Texas Instruments expressly disclaims*
 * all other warranties, express or implied, including, but not limited to, *
 * the implied warranties of merchantability and fitness for a particular   *
 * purpose.  Under no circumstances shall Texas Instruments be liable for   *
 * any incidental, special or consequential damages that result from the    *
 * use or inability to use the software or related documentation, even if   *
 * Texas Instruments has been advised of the liability.                     *
 ****************************************************************************
 *                                                                          *
 * Written by :                                                             *
 *            Sebastien Tomas  (VCP2/TCP2)                                  *
 *            Texas Instruments                                             * 
 *            12 May, 2005                                                  *
 *                                                                          *
 * The purpose of this test is to evaluate Bit-Error Rate performance for   *
 * both VCP2 and TCP2 co-processors in the TCI6482 (Himalaya)               *
 *									                                        *
 * This test should not be redistributed for any reason without             *
 * permission.                                                              *
 *                                                                          *
\***************************************************************************/

#include "convolutionalEncode.h"
#include <stdio.h>
#include <stdlib.h>

#define VERBOSE_CC 0

#ifdef __cplusplus
extern "C" {
#endif
U8 getFeedforwardBit(const U8 state, const U16 poly, const U8 bitVal, const U32 K);
#ifdef __cplusplus
}
#endif

S32 ConvolutionalEncode(const U32 *in, U32 *out, const U32 size, const U16 *feedforward)
{
	U8 state=0;
	U32 i,j,k,m,n,bitPosition;
	U32 K=0,temp;
	U32 rate;
	U32 nextBit;
	U8 *rawOut;
	
	U32 nbFullWords;
	U32 nbRemainingBits;

	/* get inverse rate */
	i=0;
	while ((i<8) && feedforward[i])
	{
	 	i++;
	};
	rate=i;
	
	/* get K */
	for (i=0; i<rate; i++)
	{
		j=0;
		while (!(feedforward[i]&(temp=(0x8000>>j)))) 
		{
			j++;
		};
		
		temp=16-j;
		if (temp>K)
			K=temp;
	}
	
	/* printf("\nK=%d,rate=%d",K,rate); */

	if ((rawOut=(U8*)malloc(sizeof(U8)*(size+K-1)))==NULL)
	{
		return 1; 
	}
	
	nbFullWords=size/32;
	nbRemainingBits=size & 0x1F;

	n=0;
	for (i=0; i<nbFullWords; i++)
	{
		for (j=0; j<32; j++)
		{
			rawOut[n]=0;
			nextBit = ((in[i]>>j) & 0x1);

			/* feedforward */
			for (m=0;m<rate;m++)
			{
				temp=getFeedforwardBit(state,feedforward[m],nextBit,K);
				rawOut[n]|= (temp&0x1) << m;
			}
			
#if VERBOSE_CC
				printf("\nin=%d rawOut=%d%d%d oldState=%02X",((in[i]>>j) & 0x1),\
														rawOut[n]&0x1,\
														(rawOut[n]>>1)&0x1,\
														(rawOut[n]>>2)&0x1,\
														state);
#endif
			state>>=1;
			state|=nextBit<<(K-2);
#if VERBOSE_CC
			printf(" newState=%02X",state);
#endif
			n++;
		}
	}
	
	for (j=0; j<nbRemainingBits; j++)
	{
		nextBit = ((in[i]>>j) & 0x1);
		rawOut[n]=0;

		/* feedforward */
		for (m=0;m<rate;m++)
		{
			temp=getFeedforwardBit(state,feedforward[m],nextBit,K);
			rawOut[n]|= (temp&0x1) << m;
		}
	
#if VERBOSE_CC
		printf("\nin=%d rawOut=%d%d%d oldState=%02X",((in[i]>>j) & 0x1),\
													rawOut[n]&0x1,\
													(rawOut[n]>>1)&0x1,\
													(rawOut[n]>>2)&0x1,\
													state);
#endif

		state>>=1;
		state|=nextBit<<(K-2);
#if VERBOSE_CC
			printf(" newState=%02X",state);
#endif
		n++;
	}
	
	/* add tail bits */
	for (i=0; i<K-1; i++)
	{
		rawOut[n]=0;
		nextBit=0; /* feedback term */

		/* feedforward */
		for (m=0;m<rate;m++)
		{
			temp=getFeedforwardBit(state,feedforward[m],nextBit,K);
			rawOut[n]|= (temp&0x1) << m;
		}
	
#if VERBOSE_CC
		printf("\nin=%d rawOut=%d%d%d oldState=%02X",nextBit,\
													rawOut[n]&0x1,\
													(rawOut[n]>>1)&0x1,\
													(rawOut[n]>>2)&0x1,\
													state);
#endif

		state>>=1;
		state|=nextBit<<(K-2);
#if VERBOSE_CC
		printf(" newState=%02X",state);
#endif
		n++;
		
	}

	bitPosition=0;
	k=0;
	out[0]=0;
	for (i=0;i<(size+K-1);i++)
	{
		for (j=0;j<rate;j++)
		{
			if (bitPosition==32)
			{
				bitPosition=0;
				k++;
				out[k]=0;
			}
			out[k]|=((rawOut[i]>>j)&0x1)<<bitPosition;
			/* printf("\nwordIndex=%d, BitPosition=%d",k,bitPosition); */
			bitPosition++;
		}
	}

	free(rawOut);
	return 0;
}

U8 getFeedforwardBit(const U8 state, const U16 poly, const U8 bitVal, const U32 K)
{
	U32 i;
	U32 val=1<<(K-1);
	U32 temp1,temp2;
	U32 result=0;
	
	temp1=(poly&val);
	temp1>>= (K-1);
	temp1&=0x1;
	temp2=(bitVal<<(K-1))&val;
	temp2 >>= (K-1);
	temp2 &= 0x1;
	result=temp1&temp2;
	val>>=1;

	for (i=1;i<K;i++)
	{
		temp1 = (poly & val);
		temp1 >>= (K-1-i);
		temp1 &= 0x1;
		temp2 = (state & val);
		temp2 >>= (K-1-i);
		temp2 &= 0x1;
		result ^= temp1 & temp2;
		val>>=1;
	}
	return result;
}

