/****************************************************************************\
 *           Copyright (C) 2000 Texas Instruments Incorporated.             *
 *                           All Rights Reserved                            *
 *                                                                          *
 * GENERAL DISCLAIMER                                                       *
 * ------------------                                                       *
 * All software and related documentation is provided "AS IS" and without   *
 * warranty or support of any kind and Texas Instruments expressly disclaims*
 * all other warranties, express or implied, including, but not limited to, *
 * the implied warranties of merchantability and fitness for a particular   *
 * purpose.  Under no circumstances shall Texas Instruments be liable for   *
 * any incidental, special or consequential damages that result from the    *
 * use or inability to use the software or related documentation, even if   *
 * Texas Instruments has been advised of the liability.                     *
 ****************************************************************************
 *                                                                          *
 * Written by :                                                             *
 *            Sebastien Tomas  (VCP2/TCP2)                                  *
 *            Texas Instruments                                             * 
 *            12 May, 2005                                                  *
 *                                                                          *
 * The purpose of this test is to evaluate Bit-Error Rate performance for   *
 * both VCP2 and TCP2 co-processors in the TCI6482 (Himalaya)               *
 *									                                        *
 * This test should not be redistributed for any reason without             *
 * permission.                                                              *
 *                                                                          *
\***************************************************************************/

#include "lrsc.h"

#include <stdio.h>

#define VERBOSE_RSC 0

unsigned char lRscGetFeedforwardBit(const unsigned char state, const unsigned short poly, const unsigned char bitVal, const unsigned int K)
{
	unsigned int i;
	unsigned int val=1<<(K-1);
	unsigned int temp1,temp2;
	unsigned int result=0;
	
	temp1=(poly&val);
	temp1>>= (K-1);
	temp1&=0x1;
	temp2=(bitVal<<(K-1))&val;
	temp2 >>= (K-1);
	temp2 &= 0x1;
	result=temp1&temp2;
	val>>=1;

	for (i=1;i<K;i++)
	{
		temp1 = (poly & val);
		temp1 >>= (K-1-i);
		temp1 &= 0x1;
		temp2 = (state & val);
		temp2 >>= (K-1-i);
		temp2 &= 0x1;
		result ^= temp1 & temp2;
		val>>=1;
	}
	return result;
}



int lrsc(const unsigned int *in, unsigned char *out, const unsigned int size, const unsigned short *feedforward, const unsigned short feedback)
{
	unsigned char state=0;
	unsigned int i,j,m,n;
	unsigned int K=0,temp;
	unsigned int rate;
	unsigned int nextBit;
	
	unsigned int nbFullWords;
	unsigned int nbRemainingBits;

	/* get inverse rate */
	i=0;
	while ((i<8) && feedforward[i])
	{
	 	i++;
	};
	rate=i+1;
	
	/* get K */
	for (i=0; i<rate-1; i++)
	{
		j=0;
		while (!(feedforward[i]&(temp=(0x8000>>j)))) 
		{
			j++;
		};
		
		temp=16-j;
		if (temp>K)
			K=temp;
	}
	
	nbFullWords=size/32;
	nbRemainingBits=size & 0x1F;

	n=0;
	for (i=0; i<nbFullWords; i++)
	{
		for (j=0; j<32; j++)
		{
			nextBit = ((in[i]>>j) & 0x1);

			out[n]=nextBit;
			temp=lRscGetFeedforwardBit(state,feedback,nextBit,K);
			nextBit ^= temp; /* feedback term */
		
			/* feedforward */
			for (m=0;m<rate-1;m++)
			{
				temp=lRscGetFeedforwardBit(state,feedforward[m],nextBit,K);
				out[n]|= (temp&0x1) << (m+1);
			}
			
#if VERBOSE_RSC
				printf("\nin=%d out=%d%d%d oldState=%01X",((in[i]>>j) & 0x1),\
														out[n]&0x1,\
														(out[n]>>1)&0x1,\
														(out[n]>>2)&0x1,\
														state);
#endif
			state>>=1;
			state|=nextBit<<(K-2);
#if VERBOSE_RSC
			printf(" newState=%01X",state);
#endif
			n++;
		}
	}
	
	for (j=0; j<nbRemainingBits; j++)
	{
		nextBit = ((in[i]>>j) & 0x1);
		out[n]=nextBit;
		temp=lRscGetFeedforwardBit(state,feedback,nextBit,K);
		nextBit ^= temp; /* feedback term */

		/* feedforward */
		for (m=0;m<rate-1;m++)
		{
			temp=lRscGetFeedforwardBit(state,feedforward[m],nextBit,K);
			out[n]|= (temp&0x1) << (m+1);
		}
	
#if VERBOSE_RSC
		printf("\nin=%d out=%d%d%d oldState=%01X",((in[i]>>j) & 0x1),\
													out[n]&0x1,\
													(out[n]>>1)&0x1,\
													(out[n]>>2)&0x1,\
													state);
#endif

		state>>=1;
		state|=nextBit<<(K-2);
#if VERBOSE_RSC
			printf(" newState=%01X",state);
#endif
		n++;
	}
	
	/* add tail bits */
	for (i=0; i<K-1; i++)
	{
		temp=lRscGetFeedforwardBit(state,feedback,nextBit,K);
		out[n]=temp;
		nextBit=0; /* feedback term */

		/* feedforward */
		for (m=0;m<rate-1;m++)
		{
			temp=lRscGetFeedforwardBit(state,feedforward[m],nextBit,K);
			out[n]|= (temp&0x1) << (m+1);
		}
	
#if VERBOSE_RSC
		printf("\nin=%d out=%d%d%d oldState=%01X",out[n]&0x1,\
													out[n]&0x1,\
													(out[n]>>1)&0x1,\
													(out[n]>>2)&0x1,\
													state);
#endif

		state>>=1;
		state|=nextBit<<(K-2);
#if VERBOSE_RSC
		printf(" newState=%01X",state);
#endif
		n++;
		
	}

	
	return 0;
}


