/****************************************************************************\
 *           Copyright (C) 2000 Texas Instruments Incorporated.             *
 *                           All Rights Reserved                            *
 *                                                                          *
 * GENERAL DISCLAIMER                                                       *
 * ------------------                                                       *
 * All software and related documentation is provided "AS IS" and without   *
 * warranty or support of any kind and Texas Instruments expressly disclaims*
 * all other warranties, express or implied, including, but not limited to, *
 * the implied warranties of merchantability and fitness for a particular   *
 * purpose.  Under no circumstances shall Texas Instruments be liable for   *
 * any incidental, special or consequential damages that result from the    *
 * use or inability to use the software or related documentation, even if   *
 * Texas Instruments has been advised of the liability.                     *
 ****************************************************************************
 *                                                                          *
 * Written by :                                                             *
 *            Sebastien Tomas  (VCP2/TCP2)                                  *
 *            Texas Instruments                                             * 
 *            12 May, 2005                                                  *
 *                                                                          *
 * The purpose of this test is to evaluate Bit-Error Rate performance for   *
 * both VCP2 and TCP2 co-processors in the TCI6482 (Himalaya)               *
 *									                                        *
 * This test should not be redistributed for any reason without             *
 * permission.                                                              *
 *                                                                          *
\***************************************************************************/
#include <c6x.h>

/* Include files for CSL */
#include <csl.h>
#include <csl_intc.h>
#include <csl_edma3.h>
#include <csl_edma3Aux.h>
#include <cslr_tcp2.h>
#include <csl_tcp2.h>
#include <csl_tcp2Aux.h>
#include <csl_vcp2.h>
#include <csl_vcp2Aux.h>
#include <stdio.h>
#include <csl_tsc.h>

#include "soc.h"
#include "edmaCommon.h"
#include "tcpvcp2_3gpp.h"

/* Static Prototype declarations */
#ifdef EDMA_SHADOW_REGION
void edmaIsr();
#else
interrupt void edmaIsr(void);
#endif

Uint32 edma3Init();
Uint32 edma3Release();
Uint32 tcpvcp2_init();

Uint32 vcp2SubmitEdma(VCP2_UserData hUserData, VCP2_Params *vcpParms,
                VCP2_ConfigIc *configIc, Uint32 *decisions, Uint32 *outParms);

Uint32 tcp2SubmitEdma(TCP2_UserData userData, TCP2_ConfigIc *configIc,
                TCP2_Params *tcpParameters, Uint16 *interleaverTable,
                Uint32 *hardDecisions, Uint32 *outParms);

/* Global variable declarations */
Int32 tcpTcc = CSL_EDMA3_CHA_TCP2REVT;    /* Transfer completion code used by tcp   */
Int32 vcpTcc = CSL_EDMA3_CHA_VCP2REVT;    /* Transfer completion code used by vcp   */

CSL_Edma3ChannelHandle
            hEdmaVcpRecv = NULL,  /* EDMA channel VCP2 Receive               */
            hEdmaVcpXmit = NULL,  /* EDMA channel VCP2 Transmit              */
			hEdmaTcpRecv = NULL,  /* EDMA channel TCP2 Receive               */
            hEdmaTcpXmit = NULL;  /* EDMA channel TCP2 Transmit              */

CSL_Edma3ParamHandle 
            hParamVcpIc,          /* EDMA channel used for IC Values         */
            hParamVcpBrMet,       /* EDMA channel used for Branch Metrics    */
            hParamVcpDec,         /* EDMA channel used for Hard Decisions    */                                 
            hParamVcpOutPar,      /* EDMA channel used for Output Parameters */
            
            hParamTcpIc,          /* EDMA channel used for IC Values         */
            hParamTcpSysPar,      /* EDMA channel used for Systematics and   */
                                  /*      Parities data                      */
            hParamTcpInter,       /* EDMA channel used for Interleaver data  */
            hParamTcpApriori,     /* EDMA channel used for Apriori data      */
            hParamTcpDec,         /* EDMA channel used for Hard Decisions    */                                 
            hParamTcpOutPar,      /* EDMA channel used for Output Parameters */
			hParamDummy;          /* EDMA dummy set of parameters */

CSL_Edma3ChannelObj ChObjVcpXmit, ChObjVcpRecv;
CSL_Edma3ChannelObj ChObjTcpXmit, ChObjTcpRecv;

CSL_Edma3Handle       hEdma; 
CSL_Edma3Obj          edmaObj;
 
#ifdef EDMA_SHADOW_REGION  
CSL_IntcEventHandlerRecord  EventRecord;
#endif
CSL_IntcHandle       hIntcEdma;
CSL_IntcEventHandlerRecord Record[30];
CSL_IntcObj          intcObjEdma;	
CSL_IntcParam        vectId; 
CSL_IntcGlobalEnableState state;

CSL_Tcp2CfgRegs *tcp2RegHandle = (CSL_Tcp2CfgRegs*) CSL_TCP2_CFG_REGS;
Uint32 vcp2DecAddress = (Uint32)&(hVcp2Vbus->VCPRDECS);

/****************************************************************************\
 * Vcp2Test: VCP2 driver example                                            *
\****************************************************************************/
Uint32 Vcp2Test(VCP2_Params *hVcpConfigParms, VCP2_ConfigIc *hVcpConfigIc, VCP2_UserData hVcpUserData,
				Uint32* hVcpDecisions ,Uint32* hVcpOutParms)
{    

	CSL_Uint64  CycleStart,CycleCost,overhead;
  	volatile Uint32 error = 0;			   
    
    /* Program EDMA3.0 to service User Channel */
	vcp2SubmitEdma(hVcpUserData, hVcpConfigParms, hVcpConfigIc, hVcpDecisions,
		          hVcpOutParms);
    CSL_tscEnable(); 	//Enable time stamp counter for profiling
	CycleStart= CSL_tscRead();
    overhead = CSL_tscRead() - CycleStart;
    CycleStart= CSL_tscRead();

	/* Start the VCP to begin the EDMA transfers */    
	VCP2_start();
		      	
    asm("		IDLE");
    CycleCost = CSL_tscRead() - CycleStart - overhead;
 	printf("the cost cycle is %d",CycleCost);	    
	return(error);	    	    	    
        
} /* Vcp2Test */ 


/****************************************************************************\
 * vcp2SubmitEdma: The EDMA is programmed to service the user channel.The IC*
 * values are configured and all EDMA parameters are calculated here.       *
\****************************************************************************/
Uint32
vcp2SubmitEdma(VCP2_UserData hUserData, VCP2_Params *vcpParameters, VCP2_ConfigIc *configIc,
           Uint32 *decisions, Uint32 *outParms)
{
CSL_Edma3ParamSetup     edmaConfig;
VCP2_Rate rate       = vcpParameters->rate;
Uint16 symx         = vcpParameters->bmBuffLen;
Uint16 numBmFrames  = vcpParameters->numBmFrames;
Uint16 numDecFrames = vcpParameters->numDecFrames;
Uint16 symr         = vcpParameters->decBuffLen;
Uint8 traceBack     = vcpParameters->traceBack;
Uint8 readFlag		= vcpParameters->readFlag;
Uint32 numBm;

    hParamVcpIc            = CSL_edma3GetParamHandle(hEdmaVcpXmit,CSL_EDMA3_CHA_VCP2XEVT,NULL);
	hParamVcpBrMet         = CSL_edma3GetParamHandle(hEdmaVcpXmit,64,NULL);
	hParamVcpDec           = CSL_edma3GetParamHandle(hEdmaVcpRecv,CSL_EDMA3_CHA_VCP2REVT,NULL);
	hParamVcpOutPar        = CSL_edma3GetParamHandle(hEdmaVcpRecv,65,NULL);
	hParamDummy            = CSL_edma3GetParamHandle(hEdmaVcpXmit,66,NULL);

	/* Configure dummy channel - used to avoid event missed situations */
	edmaConfig.option      = 0; edmaConfig.srcAddr     = 0;
    edmaConfig.dstAddr     = 0; edmaConfig.aCntbCnt    = 0;
    edmaConfig.cCnt        = 1; edmaConfig.srcDstBidx  = 0;
	edmaConfig.srcDstCidx  = 0; edmaConfig.linkBcntrld = 0x0000FFFF;

	CSL_edma3ParamSetup(hParamDummy,&edmaConfig);


    /* Configure channel parameters for IC registers */
    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
					                       0,                        CSL_EDMA3_TCC_NORMAL,
                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
                                           CSL_EDMA3_SYNC_AB,        CSL_EDMA3_ADDRMODE_INCR,
					                       CSL_EDMA3_ADDRMODE_INCR);

    edmaConfig.srcAddr     = (Uint32)(((Uint32)configIc)|GLOBAL_L2_MASK);
    edmaConfig.dstAddr     = (Uint32)&(hVcp2Vbus->VCPIC0);
    edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE((Uint32)(VCP2_NUM_IC<<2),1);
    edmaConfig.cCnt        = 1;
    edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,0);
	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
    edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamVcpBrMet,1);
	CSL_edma3ParamSetup(hParamVcpIc,&edmaConfig);

    /* Configure channel parameters for branch metrics data */
	numBm = 4 << (rate - 1); 

    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
					                       0,                        CSL_EDMA3_TCC_NORMAL,
                                           CSL_EDMA3_FIFOWIDTH_64BIT,CSL_EDMA3_STATIC_DIS,
                                           CSL_EDMA3_SYNC_A,         CSL_EDMA3_ADDRMODE_FIFO,
					                       CSL_EDMA3_ADDRMODE_INCR);


    edmaConfig.srcAddr     = (Uint32)(((Uint32)hUserData)|GLOBAL_L2_MASK);
    edmaConfig.dstAddr     = (Uint32)&(hVcp2Vbus->VCPWBM);
    edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE(numBm * (symx+1),numBmFrames);
	edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(numBm * (symx+1),0);
	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
    edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamDummy,1);
	CSL_edma3ParamSetup(hParamVcpBrMet,&edmaConfig);
    
    if (readFlag){
    	/* Configure channel parameters for decision data */
		if (traceBack == VCP2_TRACEBACK_TAILED)
		    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
		                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
							                       0,                        CSL_EDMA3_TCC_EARLY,
		                                           CSL_EDMA3_FIFOWIDTH_64BIT,CSL_EDMA3_STATIC_DIS,
		                                           CSL_EDMA3_SYNC_AB,         CSL_EDMA3_ADDRMODE_INCR,
							                       CSL_EDMA3_ADDRMODE_FIFO);

		else
		    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
		                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
							                       0,                        CSL_EDMA3_TCC_EARLY,
		                                           CSL_EDMA3_FIFOWIDTH_64BIT,CSL_EDMA3_STATIC_DIS,
		                                           CSL_EDMA3_SYNC_AB,        CSL_EDMA3_ADDRMODE_INCR,
							                       CSL_EDMA3_ADDRMODE_FIFO);

		
    }else{
    	/* Configure channel parameters for decision data */
		if (traceBack == VCP2_TRACEBACK_TAILED)
		    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,    CSL_EDMA3_TCCH_DIS,
		                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_EN,
							                       vcpTcc,                   CSL_EDMA3_TCC_NORMAL,
		                                           CSL_EDMA3_FIFOWIDTH_64BIT,CSL_EDMA3_STATIC_DIS,
		                                           CSL_EDMA3_SYNC_AB,         CSL_EDMA3_ADDRMODE_INCR,
							                       CSL_EDMA3_ADDRMODE_FIFO);

		else
		    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,    CSL_EDMA3_TCCH_DIS,
		                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_EN,
							                       vcpTcc,                   CSL_EDMA3_TCC_NORMAL,
		                                           CSL_EDMA3_FIFOWIDTH_64BIT,CSL_EDMA3_STATIC_DIS,
		                                           CSL_EDMA3_SYNC_AB,        CSL_EDMA3_ADDRMODE_INCR,
							                       CSL_EDMA3_ADDRMODE_FIFO);

    }	

    edmaConfig.srcAddr     = vcp2DecAddress;
    if (traceBack == VCP2_TRACEBACK_TAILED){
    	edmaConfig.dstAddr     = (Uint32)(((Uint32)(decisions + (2 * (symr) * (numDecFrames))))|GLOBAL_L2_MASK);
    	edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE(8, (symr+1) * numDecFrames);
	    edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,(Uint32)0xFFF8);
	}else{
    	edmaConfig.dstAddr     = (Uint32)((Uint32)decisions|GLOBAL_L2_MASK);
    	edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE(8 * (symr+1),numDecFrames);
	    edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,8 * (symr+1));
	}
    edmaConfig.cCnt        = 1;
	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
    if (readFlag) edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamVcpOutPar,1);
    else          edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(CSL_EDMA3_LINK_NULL,1);
	CSL_edma3ParamSetup(hParamVcpDec,&edmaConfig);	
    
    if (readFlag){    	

	    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,    CSL_EDMA3_TCCH_DIS,
	                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_EN,
						                       vcpTcc,                   CSL_EDMA3_TCC_NORMAL,
	                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
	                                           CSL_EDMA3_SYNC_AB,        CSL_EDMA3_ADDRMODE_INCR,
						                       CSL_EDMA3_ADDRMODE_INCR);


        edmaConfig.srcAddr     = (Uint32)&(hVcp2Vbus->VCPOUT0);
        edmaConfig.dstAddr     = (Uint32)(((Uint32)outParms)|GLOBAL_L2_MASK);
        edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE((Uint32)(VCP2_NUM_OP<<2),1);
        edmaConfig.cCnt        = 1;
        edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,0);
    	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
        edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(CSL_EDMA3_LINK_NULL,1);
	    CSL_edma3ParamSetup(hParamVcpOutPar,&edmaConfig);
    }
	    
    /* Enable the transmit and receive channels */
	CSL_edma3ChannelEnable(hEdmaVcpXmit);
	CSL_edma3ChannelEnable(hEdmaVcpRecv);

    return(0);

} /* end vcp2SubmitEdma() */

/****************************************************************************\
 * Tcp2Test: TCP2 driver example                                            *
\****************************************************************************/
Uint32 Tcp2Test(TCP2_Params *hTcpParameters, TCP2_ConfigIc *hTcpConfigIc, TCP2_UserData hTcpUserDataRate5,
                Uint16 *hTcpInterleaverTable, Uint32* hTcpDecisions,Uint32* hTcpOutParms)
{				
    volatile Uint32 error = 0;

    /* Program EDMA to service User Channel */
    tcp2SubmitEdma(hTcpUserDataRate5, hTcpConfigIc, hTcpParameters,
    		   hTcpInterleaverTable, hTcpDecisions, hTcpOutParms);    

	/* Start the VCP to begin the EDMA transfers */    
	TCP2_start();
    //(*(volatile Uint32*)(0x02ba004c)) = ((Uint32)(0x00000001)); // CSL misses that function
		      	
    asm("		IDLE");
		    
	return(error);	    	    	           

}

/****************************************************************************\
 * tcp2SubmitEdma: The EDMA is programmed to service the user channel.The IC*
 * values are configured and all EDMA parameters are calculated here.       *
\****************************************************************************/
Uint32
tcp2SubmitEdma(TCP2_UserData userData, TCP2_ConfigIc *configIc, TCP2_Params *tcpParameters,
               Uint16 *interleaverTable, Uint32 *hardDecisions, Uint32 *outParms)
{
CSL_Edma3ParamSetup     edmaConfig;
Uint32 numSysPar = tcpParameters->frameLen * 4; 
Uint32 numInter  = tcpParameters->frameLen << 1;
Uint32 intFlag   = tcpParameters->intFlag;
Uint32 outParmFlag  = tcpParameters->outParmFlag;
Uint32 numHd     = TCP2_ceil(tcpParameters->frameLen, 3);

    hParamTcpIc            = CSL_edma3GetParamHandle(hEdmaTcpXmit,CSL_EDMA3_CHA_TCP2XEVT,NULL);
	hParamTcpSysPar        = CSL_edma3GetParamHandle(hEdmaTcpXmit,64,NULL);
	hParamTcpInter         = CSL_edma3GetParamHandle(hEdmaTcpXmit,65,NULL);
	hParamTcpDec           = CSL_edma3GetParamHandle(hEdmaTcpRecv,CSL_EDMA3_CHA_TCP2REVT,NULL);
	hParamTcpOutPar        = CSL_edma3GetParamHandle(hEdmaTcpRecv,66,NULL);
	hParamDummy            = CSL_edma3GetParamHandle(hEdmaTcpXmit,67,NULL);

	/* Configure dummy channel - used to avoid event missed situations */
	edmaConfig.option      = 0; edmaConfig.srcAddr     = 0;
    edmaConfig.dstAddr     = 0; edmaConfig.aCntbCnt    = 0;
    edmaConfig.cCnt        = 1; edmaConfig.srcDstBidx  = 0;
	edmaConfig.srcDstCidx  = 0; edmaConfig.linkBcntrld = 0x0000FFFF;

	CSL_edma3ParamSetup(hParamDummy,&edmaConfig);

    /* Configure channel parameters for IC registers */    
    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
					                       0,                        CSL_EDMA3_TCC_NORMAL,
                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
                                           CSL_EDMA3_SYNC_A,         CSL_EDMA3_ADDRMODE_INCR,
					                       CSL_EDMA3_ADDRMODE_INCR);

    edmaConfig.srcAddr     = (Uint32)(((Uint32)configIc)|GLOBAL_L2_MASK);
    edmaConfig.dstAddr     = (Uint32)CSL_TCP2_CFG_REGS;
    edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE((Uint32)(TCP2_NUM_IC<<2),1);
    edmaConfig.cCnt        = 1;
    edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,0);
	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
    edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamTcpSysPar,1);
	CSL_edma3ParamSetup(hParamTcpIc,&edmaConfig);

    /* Configure channel parameters for systematics and parities */
    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
					                       0,                        CSL_EDMA3_TCC_NORMAL,
                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
                                           CSL_EDMA3_SYNC_A,         CSL_EDMA3_ADDRMODE_INCR,
					                       CSL_EDMA3_ADDRMODE_INCR);

    edmaConfig.srcAddr     = (Uint32)(((Uint32)userData)|GLOBAL_L2_MASK);
    edmaConfig.dstAddr     = (Uint32)CSL_TCP2_X0_MEM;
    edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE(numSysPar,1);
	edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,0);
	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
   	if (intFlag) edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamTcpInter,1);
	else         edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamDummy,1);
	CSL_edma3ParamSetup(hParamTcpSysPar,&edmaConfig);

	if (intFlag){
        /* Configure channel parameters for interleaver data */
	    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
	                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
						                       0,                        CSL_EDMA3_TCC_NORMAL,
	                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
	                                           CSL_EDMA3_SYNC_A,         CSL_EDMA3_ADDRMODE_INCR,
						                       CSL_EDMA3_ADDRMODE_INCR);

       edmaConfig.srcAddr     = (Uint32)(((Uint32)interleaverTable)|GLOBAL_L2_MASK);
       edmaConfig.dstAddr     = (Uint32)CSL_TCP2_I0_MEM;
       edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE(numInter+8,1);
       edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,0);
	   edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
       edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamDummy,1);
	   CSL_edma3ParamSetup(hParamTcpInter,&edmaConfig);   
    }

    if (outParmFlag){
    	/* Configure channel parameters for decision data */
	    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
	                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_DIS,
						                       0,                        CSL_EDMA3_TCC_NORMAL,
	                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
	                                           CSL_EDMA3_SYNC_A,         CSL_EDMA3_ADDRMODE_INCR,
						                       CSL_EDMA3_ADDRMODE_INCR);

    }else{
    	/* Configure channel parameters for decision data */
	    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
	                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_EN,
						                       tcpTcc,                   CSL_EDMA3_TCC_NORMAL,
	                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
	                                           CSL_EDMA3_SYNC_A,         CSL_EDMA3_ADDRMODE_INCR,
						                       CSL_EDMA3_ADDRMODE_INCR);


    }
	edmaConfig.srcAddr     = (Uint32)CSL_TCP2_O0_MEM;
    edmaConfig.dstAddr     = (Uint32)(((Uint32)hardDecisions)|GLOBAL_L2_MASK);
    edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE(numHd,1);
    edmaConfig.cCnt        = 1;
    edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,0);
	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
    if (outParmFlag) edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamTcpOutPar,1);
    else             edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(CSL_EDMA3_LINK_NULL,1);
	CSL_edma3ParamSetup(hParamTcpDec,&edmaConfig);	
    
    if (outParmFlag){    	

	    edmaConfig.option = CSL_EDMA3_OPT_MAKE(CSL_EDMA3_ITCCH_DIS,      CSL_EDMA3_TCCH_DIS,
	                                           CSL_EDMA3_ITCINT_DIS,     CSL_EDMA3_TCINT_EN,
						                       tcpTcc,                   CSL_EDMA3_TCC_NORMAL,
	                                           CSL_EDMA3_FIFOWIDTH_NONE, CSL_EDMA3_STATIC_DIS,
	                                           CSL_EDMA3_SYNC_A,         CSL_EDMA3_ADDRMODE_INCR,
						                       CSL_EDMA3_ADDRMODE_INCR);

        edmaConfig.srcAddr     = (Uint32)(CSL_TCP2_CFG_REGS + 0x40);
        edmaConfig.dstAddr     = (Uint32)(((Uint32)outParms)|GLOBAL_L2_MASK);
        edmaConfig.aCntbCnt    = CSL_EDMA3_CNT_MAKE((Uint32)(TCP2_NUM_OP<<2),1);
        edmaConfig.cCnt        = 1;
        edmaConfig.srcDstBidx  = CSL_EDMA3_BIDX_MAKE(0,0);
    	edmaConfig.srcDstCidx  = CSL_EDMA3_CIDX_MAKE(0,0);
        edmaConfig.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(CSL_EDMA3_LINK_NULL,1);
	    CSL_edma3ParamSetup(hParamTcpOutPar,&edmaConfig);
    }

    /* Enable the transmit and receive channels */
	CSL_edma3ChannelEnable(hEdmaTcpXmit);
	CSL_edma3ChannelEnable(hEdmaTcpRecv);

    return(0);

} /* end tcp2SubmitEdma() */

/****************************************************************************\
 * edmaIsr: EDMA interrupt service routine.                                 *
 *          Checks EDMA transfer completion codes for TCP2/VCP2             *
 *          Closes EDMA channels for TCP2/VCP2                              *
\****************************************************************************/
#ifndef EDMA_SHADOW_REGION  
interrupt void edmaIsr(void)
#else
void edmaIsr()  
#endif
{         
#ifndef EDMA_SHADOW_REGION
    CSL_BitMask32 		 	intrQuery[2]; 
    CSL_BitMask32        	intrClear[2]; 
	Uint32					temp;

    CSL_edma3GetIntrPendStatus(hEdma,CSL_EDMA3_REGION_GLOBAL,&intrQuery[0],&intrQuery[1]);
	temp=0;
    if (intrQuery[0] & (1<<vcpTcc))
    {
   	  temp |= 1<<vcpTcc;
	}
	if (intrQuery[0] & (1<<tcpTcc))
    {
   	  temp |= 1<<tcpTcc;
	}
   	intrClear[0]=temp;
   	intrClear[1]=0;
	CSL_edma3InterruptClear(hEdma,CSL_EDMA3_REGION_GLOBAL,intrClear[0],intrClear[1]);
#endif    

} /* end edmaIsr */

/****************************************************************************\
 * VCP2_getSoftRes: returns the soft input resolution                        *
\****************************************************************************/
Uint32 VCP2_getSoftRes(const Uint32 rate, const Uint32 K)
{
	Uint32 res;
	
	switch (rate) 	{

		case 2:
			if (K>9 || K<5) {
				res=0;
			}
			else if (K>6) {
				res=6;
			}
			else 	{
				res=7;
			}
			break;
		case 3:
			if (K>9 || K<5) {
				res=0;
			}			
			else 	{
				res=6;
			}
			break;
		case 4:
			if (K>9 || K<5) 	{
				res=0;
			}
			else if (K>7) 	{
				res=5;
			}
			else 	{
				res=6;
			}
			break;
		default:
			res=0;
			break;

	}

	return res;
}

/****************************************************************************\
 * VCP2_quantizeSoft: performs soft input scaling and saturation             *
\****************************************************************************/
void VCP2_quantizeSoft(Uint32 * restrict in, Uint32 * restrict out, const Uint32 softInputResolution,const Uint32 length)
{
unsigned int i, j = 0;
char  maxpos;
char  maxneg;
unsigned int maxpos4;
unsigned int maxneg4;
unsigned int rangemax4 = 0x1f1f1f1f;  // +31, max positive that won't be saturated
unsigned int rangemin4 = 0xE0E0E0E0;  //-32, most negative that won't be saturated
double temp;	
unsigned int outtemp, temp1, temp2, shr_amnt;
unsigned int mask_repl_pos, mask_repl_neg, mask_not_repl, mask_neg_or_pos;
unsigned int repl_neg, repl_pos, repl;

	maxpos = _set(0x00000000,0,softInputResolution-2);	
	maxneg = _sshvr((-128), (8-softInputResolution));

    maxpos4 = _packl4(_pack2(maxpos,maxpos),_pack2(maxpos,maxpos));
    maxneg4 = _packl4(_pack2(maxneg,maxneg),_pack2(maxneg,maxneg));
	
	shr_amnt = 6-softInputResolution;    
    
    
	for (i=0;i<length;i+=4, j++){			
        // pack outtemp with "normal" output, some may be saturated later
        	temp    = _mpysu4(in[j],0x01010101);
         	temp1   = _shr2(_hi(temp),shr_amnt);
         	temp2   = _shr2(_lo(temp),shr_amnt);
          	outtemp = _packl4(temp1, temp2);

        // determine which bytes to saturate and which to keep 
			mask_neg_or_pos	= _cmpgtu4(in[j], 0x7F7F7F7F);
			mask_repl_neg 	= _xpnd4(_cmpgtu4(rangemin4, in[j])&mask_neg_or_pos);			
			mask_repl_pos 	= _xpnd4(_cmpgtu4(in[j], rangemax4)&~mask_neg_or_pos);
			mask_not_repl = ~(mask_repl_pos | mask_repl_neg);

        // clear outtemp from the bytes that are going to be saturated 
            outtemp  = outtemp & mask_not_repl;
            repl_neg = mask_repl_neg & maxneg4;
            repl_pos = mask_repl_pos & maxpos4;

        // repl holds saturated bytes (saturated to positive and negative) 
            repl = repl_neg | repl_pos;

        //  merge "normal" and saturated bytes 
            out[j] = outtemp | repl;
		}
	  
	 // At least 14 bytes (6+8) must be zero'd for branch metrics optimized reads.	
	 for (i=0;i<8;i++) out[j++]=0;	
}

/****************************************************************************\
 * TCP2_quantizeSoft: performs soft input scaling and saturation             *
\****************************************************************************/
void TCP2_quantizeSoft(Uint32 *restrict in,Uint32 *restrict out, const Uint8 scale_fixpt, const Uint32 length)
{     
double 	temp,temp1,temp2;
Int32 	hi_temp1,lo_temp1,hi_temp2,lo_temp2;
Uint32	   	scale, i, j=0;

unsigned int rangemax4 = 0x1f1f1f1f;  // +31, max positive that won't be saturated
unsigned int rangemin4 = 0xE0E0E0E0;  //-32, most negative that won't be saturated
unsigned int outtemp;
unsigned int mask_repl_pos, mask_repl_neg, mask_not_repl, mask_neg_or_pos;
unsigned int repl_neg, repl_pos, repl;
	

/* scale_fixpt = (2/sigma2)*32 + 0.5 */
/* the function will scale if sigma is bigger than 0.8 */
/* otherwise the values will be saturated */

	/* duplicate scale in 4 bytes */
    scale = _packl4(_pack2(scale_fixpt,scale_fixpt),_pack2(scale_fixpt,scale_fixpt));
    
       for(i=0; i < length; i+=4, j++)
    	  {
          	
          	/* scale 4 input values by scaling factor */          	
        	// in[j] is in signed Q4 
        	// and scale holds unsigned Q5
        	// then temp is in signed Q9
          	temp=_mpysu4(in[j],scale);
          	
    	    /* saturate into upper byte and negate */
			temp1=_smpy2(_hi(temp),0xC000C000);
         	temp2=_smpy2(_lo(temp),0xC000C000);
         	
         	// here, upper 16 bits of temp1 and temp2 will be in 
        	// signed Q8, i.e. upper 8 bits are in Q0.
        	// we need Q2 (?), hence scale 2 positions left.
         	hi_temp1=_sshl(_hi(temp1),2);
         	lo_temp1=_sshl(_lo(temp1),2);
         	hi_temp2=_sshl(_hi(temp2),2);
         	lo_temp2=_sshl(_lo(temp2),2);

          	/* pack 4 upper bytes of sshl's */
          	outtemp=_packh4(_packh2(hi_temp1,lo_temp1),
                         _packh2(hi_temp2,lo_temp2));

            // determine which bytes to saturate and which to keep 
			mask_neg_or_pos	= _cmpgtu4(outtemp, 0x7F7F7F7F);
			mask_repl_neg 	= _xpnd4(_cmpgtu4(rangemin4, outtemp)&mask_neg_or_pos);			
			mask_repl_pos 	= _xpnd4(_cmpgtu4(outtemp, rangemax4)&~mask_neg_or_pos);
			mask_not_repl = ~(mask_repl_pos | mask_repl_neg);

            // clear outtemp from the bytes that are going to be saturated 
            outtemp  = outtemp & mask_not_repl;
            repl_neg = mask_repl_neg & rangemin4;
            repl_pos = mask_repl_pos & rangemax4;

            // repl holds saturated bytes (saturated to positive and negative) 
            repl = repl_neg | repl_pos;

        //  merge "normal" and saturated bytes 
            out[j] = outtemp | repl;

          	
      	 }
}      	 


/****************************************************************************\
 * TCP2_deintUnpunctSoft3:deinterleave interleaved parities                  *
 *                       unpuncture a rate 1/3 data stream into rate 1/5    *
 *                       those are TCP2 requirements                        *
\****************************************************************************/
/*static Uint16 deintTable[5114];
void TCP2_deintUnpunctSoft3(Int8 *UserData, Int8 *UserDataRate5, Uint16 *intTable,
                          const Uint32 length)
{
Uint32 i;
Uint32 length_m3 = (length*3+2)/3;
Uint32 *out_ptr = (Uint32*)UserDataRate5;

    // deinterleave interleaved parities
	for (i=0;i<length;i++) deintTable[intTable[i]]=i;

	// unpuncture 
	for (i=0;i<length_m3;i++)
	{			
		*out_ptr++ =  (_extu(UserData[3*i],26,26)<<0)
		             |(_extu(UserData[(3*i)+1],26,26)<<6)
					 |(_extu(UserData[(3*deintTable[i])+2],26,26)<<18);
	}

}*/

static Uint16 deintTable[5114+4];

void TCP2_deintUnpunctSoft3(Int8   *restrict	UserData,	//rate 1/3 data 2nd interleaved
 	                       Uint32 *restrict	out_ptr,	//rate 1/5 data deinterleaved
 	                       Uint16 *restrict intTable,	//NULL to reuse deintTable
                           const Uint32 length)
{
	Uint32 i;

    // deinterleave interleaved parities
	if (intTable)
	{
	  for (i=0;i<length;i++) deintTable[intTable[i]]=i;
	  deintTable[length+0]=length+0;
	  deintTable[length+1]=length+1;
	  deintTable[length+2]=length+2;
	  deintTable[length+3]=length+3;
	}

	for (i=0;i<length+4;i+=3)
	{			
		unsigned long long	sys1par1;
		unsigned long long	deintCBA;
		double				deint_lo,deint_hi;
		Uint32				par2a,par2b,par2c;

		sys1par1   = _mem8(&(UserData[3*i]));			//LE p1C:sC:x:p1B;sB:x:p1A:sA
														//BE sA:p1A:x:sB:p1b:x:sC:p1C
		deintCBA   = _mem8(&(deintTable[i]));
		deint_lo   = _mpy2(deintCBA    ,0x00030003);
		deint_hi   = _mpy2(deintCBA>>32,0x00030003);

#ifdef _LITTLE_ENDIAN
		par2a	   = UserData[_lo(deint_lo)+2];	//p2A
		par2b	   = UserData[_hi(deint_lo)+2];	//p2B
		par2c	   = UserData[_lo(deint_hi)+2];	//p2C
#else
		par2a	   = UserData[_hi(deint_hi)+2];	//p2A
		par2b	   = UserData[_lo(deint_hi)+2];	//p2B
		par2c	   = UserData[_hi(deint_lo)+2];	//p2C
#endif
		//
		//..pppppp:pppppppp:pppppppp:ppssssss
		//..rrrrrr:rrrrrrrr:rrrrrrrr:rryyyyyy
		//..444444:22222233:33331111:11ssssss
		//
#ifdef _LITTLE_ENDIAN
		*out_ptr++ = _pack2(_lo(_mpyu4(par2a,4)),				//par2a
		                    _rotl(_extu(sys1par1    ,18,26),6)	//par1a
		                       | (_extu(sys1par1    ,26,26)));	//sys_a
//		*out_ptr++ =      (_extu(sys1par1    ,26,26))			//sys_a
//		           | _rotl(_extu(sys1par1    ,18,26),6)			//par1a
//				   |      (_extu(par2a       ,26, 8));			//par2a
		*out_ptr++ =      (_extu(sys1par1    , 2,26))			//sys_b
		           | _rotl(_extu(sys1par1>>32,26,26),6)			//par1b
				   |      (_extu(par2b       ,26, 8));			//par2b
		*out_ptr++ =      (_extu(sys1par1>>32,10,26))			//sys_c
		           | _rotl(_extu(sys1par1>>32, 2,26),6)			//par1c
				   |      (_extu(par2c       ,26, 8));			//par2c
#else
		*out_ptr++ = _pack2(_lo(_mpyu4(par2a,4)),				//par2a
		                    _rotl(_extu(sys1par1>>32,10,26),6)	//par1a
		                       | (_extu(sys1par1>>32, 2,26)));	//sys_a
//		*out_ptr++ =      (_extu(sys1par1    ,02,26))			//sys_a
//		           | _rotl(_extu(sys1par1    ,10,26),6)			//par1a
//				   |      (_extu(par2a       ,26, 8));			//par2a
		*out_ptr++ =      (_extu(sys1par1>>32,26,26))			//sys_b
		           | _rotl(_extu(sys1par1    ,02,26),6)			//par1b
				   |      (_extu(par2b       ,26, 8));			//par2b
		*out_ptr++ =      (_extu(sys1par1    ,18,26))			//sys_c
		           | _rotl(_extu(sys1par1    ,26,26),6)			//par1c
				   |      (_extu(par2c       ,26, 8));			//par2c
#endif
	}
}



Uint32
edma3Release()
{
#ifdef EDMA_SHADOW_REGION
  CSL_Edma3CmdDrae   draeAttr;
#endif
   if (hEdmaVcpXmit) {CSL_edma3ChannelClose(hEdmaVcpXmit);hEdmaVcpXmit=NULL;}
   if (hEdmaVcpRecv) {CSL_edma3ChannelClose(hEdmaVcpRecv);hEdmaVcpRecv=NULL;}
   if (hEdmaTcpXmit) {CSL_edma3ChannelClose(hEdmaTcpXmit);hEdmaTcpXmit=NULL;}
   if (hEdmaTcpRecv) {CSL_edma3ChannelClose(hEdmaTcpRecv);hEdmaTcpRecv=NULL;}

#ifdef EDMA_SHADOW_REGION
	// Setup the DRAE Masks
    draeAttr.region = REGION_EDMA;
    draeAttr.drae   = 0xffffffff;
    draeAttr.draeh  = 0x0;
	CSL_edma3HwControl(hEdma,CSL_EDMA3_CMD_DMAREGION_DISABLE,&draeAttr);
#endif

    CSL_edma3Close(hEdma);   
	CSL_intcClose(hIntcEdma);

	CSL_intcGlobalDisable(&state);

   return(0);
}

CSL_IntcContext        context;
CSL_Edma3Context       edmaContext;

Uint32
edma3Init()
{
    CSL_Status             modStatus;
#ifdef EDMA_SHADOW_REGION
	CSL_Edma3CmdDrae   draeAttr;
#endif

	CSL_Edma3ChannelAttr   channelParam;
	CSL_IntcParam          vectId; 

    CSL_BitMask32          intrEn[2]; 
	CSL_Edma3HwSetup       hwSetup;
    CSL_Edma3HwDmaChannelSetup  dmahwSetup[CSL_EDMA3_NUM_DMACH]   = CSL_EDMA3_DMACHANNELSETUP_DEFAULT;
    CSL_Edma3HwQdmaChannelSetup qdmahwSetup[CSL_EDMA3_NUM_QDMACH] = CSL_EDMA3_QDMACHANNELSETUP_DEFAULT;


    context.numEvtEntries = 30;
    context.eventhandlerRecord = Record;

	CSL_intcInit(&context);
	/* Global APIs */
	/* Enable NMIs */
	CSL_intcGlobalNmiEnable();
	/* Enable Global Interrupts */
	modStatus = CSL_intcGlobalEnable(&state);

	/* Opening a handle for the Event edma */
	vectId = CSL_INTC_VECTID_4;
#ifdef EDMA_SHADOW_REGION
	hIntcEdma = CSL_intcOpen (&intcObjEdma, REGION_EDMA_EVT, &vectId , NULL);
#else
	hIntcEdma = CSL_intcOpen (&intcObjEdma, GLOBAL_EDMA_EVT, &vectId , NULL);
#endif

    // Initialize and Open Module
    modStatus = CSL_edma3Init(&edmaContext);
	hEdma     = CSL_edma3Open(&edmaObj,0,NULL,&modStatus);

    // EDMA Setup
    hwSetup.dmaChaSetup  = &dmahwSetup[0];
    hwSetup.qdmaChaSetup = &qdmahwSetup[0];
    CSL_edma3HwSetup(hEdma,&hwSetup);


#ifdef EDMA_SHADOW_REGION
	// Setup the DRAE Masks
    draeAttr.region = REGION_EDMA;
    draeAttr.drae   = 0xffffffff;
    draeAttr.draeh  = 0x0;
	modStatus = CSL_edma3HwControl(hEdma,CSL_EDMA3_CMD_DMAREGION_ENABLE,&draeAttr);
#endif

  	// VCP Channel Open
#ifdef EDMA_SHADOW_REGION
	channelParam.regionNum = REGION_EDMA;
#else
	channelParam.regionNum = CSL_EDMA3_REGION_GLOBAL;
#endif
	channelParam.chaNum    = CSL_EDMA3_CHA_VCP2XEVT; // VCP Xmit
    hEdmaVcpXmit           = CSL_edma3ChannelOpen(&ChObjVcpXmit,0,&channelParam,&modStatus);
    channelParam.chaNum    = CSL_EDMA3_CHA_VCP2REVT; // VCP Recv 
    hEdmaVcpRecv           = CSL_edma3ChannelOpen(&ChObjVcpRecv,0,&channelParam,&modStatus);
  	// TCP Channel Open
#ifdef EDMA_SHADOW_REGION
	channelParam.regionNum = REGION_EDMA;
#else
	channelParam.regionNum = CSL_EDMA3_REGION_GLOBAL;
#endif
	channelParam.chaNum    = CSL_EDMA3_CHA_TCP2XEVT; // TCP Xmit
    hEdmaTcpXmit           = CSL_edma3ChannelOpen(&ChObjTcpXmit,0,&channelParam,&modStatus);
    channelParam.chaNum    = CSL_EDMA3_CHA_TCP2REVT; // TCP Recv 
    hEdmaTcpRecv           = CSL_edma3ChannelOpen(&ChObjTcpRecv,0,&channelParam,&modStatus);

	// VCP Channel Setup
	CSL_edma3HwChannelSetupQue(hEdmaVcpXmit,CSL_EDMA3_QUE_0);
	CSL_edma3HwChannelSetupQue(hEdmaVcpRecv,CSL_EDMA3_QUE_0);

	// TCP Channel Setup
	CSL_edma3HwChannelSetupQue(hEdmaTcpXmit,CSL_EDMA3_QUE_0);
	CSL_edma3HwChannelSetupQue(hEdmaTcpRecv,CSL_EDMA3_QUE_0);

#ifdef EDMA_SHADOW_REGION
	EventRecord.handler = &eventEdmaHandler;
	EventRecord.arg     = hEdma;
	CSL_intcPlugEventHandler(hIntcEdma,&EventRecord);	
#else
    /**EventRecord.handler = (CSL_IntcEventHandler)edmaIsr;
	EventRecord.arg = NULL;*/
	CSL_intcHookIsr(vectId,&edmaIsr);
#endif 

	/* Enabling edma to interrupt CPU  */
	CSL_intcHwControl(hIntcEdma,CSL_INTC_CMD_EVTENABLE,NULL);

	/* Enable the TCCs to generate a CPU interrupt */
	intrEn[0] = ((1<<vcpTcc)|(1<<tcpTcc));
	intrEn[1] = 0;
#ifdef EDMA_SHADOW_REGION
    //Hook up the EDMA Event with an ISR
	EdmaEventHook(vcpTcc, edmaIsr);	
	EdmaEventHook(tcpTcc, edmaIsr);
	CSL_edma3InterruptEnable(hEdma,REGION_EDMA,intrEn[0],intrEn[1]); 
#else
	CSL_edma3InterruptEnable(hEdma,CSL_EDMA3_REGION_GLOBAL,intrEn[0],intrEn[1]); 
#endif

  return(0);
}

#define  PSVR_BASE                 0x02AC0000 
#define  PSVR_LCKREG               PSVR_BASE + 0x4
#define  PSVR_MDCTL0               PSVR_BASE + 0x8
#define  PSVR_MDSTAT0              PSVR_BASE + 0x14
#define  PSVR_TCP2_EN               0x5
#define  PSVR_TCP2_LCKVAL           0x0F0A0B00
#define  PSVR_TCP2_MDSTAT_MASK      0x7
#define  PSVR_TCP2_MDSTAT_EN        0x1

Uint32
tcpvcp2_init() {
CSL_Status status = 1;

    (*(volatile Uint32*)(PSVR_LCKREG)) = PSVR_TCP2_LCKVAL;   

     asm(" NOP 5 ");

     (*(volatile Uint32*)(PSVR_MDCTL0))|= PSVR_TCP2_EN;   

     asm(" NOP 5 ");

     do {
             asm(" NOP 5 ");
             asm(" NOP 5 ");

         } while(((*(volatile Uint32*)(PSVR_MDSTAT0)) & PSVR_TCP2_MDSTAT_MASK)
                      != PSVR_TCP2_MDSTAT_EN);

    return status;    
}


