/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *  ============================================================================
 */
 
/** ============================================================================
 *   @file  csl_dev.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  Header file for functional layer of DEV CSL
 *
 */

/** @mainpage DEVICE Configuration
 *
 * @section Introduction
 *
 * @subsection xxx Purpose and Scope
 * The purpose of this document is to detail the  CSL APIs for the
 * DEV Module.
 *
 * @subsection aaa Terms and Abbreviations
 *   -# CSL     : Chip Support Library
 *   -# API     : Application Programmer Interface
 *   -# DEV     : Device configuration
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  06-Apr-2005 Sudeep File Created.
 *  27-Jan-2006 sd     Modifications for the module name and spec changes   
 * =============================================================================
 */
 
#ifndef _CSL_DEV_H_
#define _CSL_DEV_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <csl.h>
#include <soc.h>
#include <cslr_dev.h>


/** Macros to get specific device reset configuration information */

/** Get the boot mode value from the reset configuration value passed         */
#define CSL_DEV_BOOT_MODE_GET(val)      CSL_FEXT(val, DEV_DEVSTAT_BOOTMODE)

/** Get the EMIFA input clock selected from the reset configuration  
 * value passed 
 */
#define CSL_DEV_EMIFA_IN_CLK_GET(val)   CSL_FEXT(val, DEV_DEVSTAT_AECLKINSEL)

/** Get the HPI width configuration from the reset configuration value passed */
#define CSL_DEV_HPI_WIDTH_GET(val)      CSL_FEXT(val, DEV_DEVSTAT_HPI_WIDTH)

/** Get the device endian mode from the reset configuration value passed      */
#define CSL_DEV_ENDIAN_MODE_GET(val)    CSL_FEXT(val, DEV_DEVSTAT_LENDIAN)

/** Get the UTOPIA enable status from the reset configuration value passed
  * @n 1 - UTOPIA enabled
  * @n 0 - EMAC/MDIO enabled
  */
#define CSL_DEV_UTOPIA_EN_STAT(val)     CSL_FEXT(val, DEV_DEVSTAT_UTOPIA_EN)

/** Get the EMAC interface mode from the reset configuration value passed     */
#define CSL_DEV_EMAC_MODE_GET(val)      CSL_FEXT(val, DEV_DEVSTAT_MACSEL)

/** Get the PCI I2C EEPROM auto initialization status from the reset  
  * configuration value passed
  * @n 1 - Enabled
  * @n 0 - Disabled
  */
#define CSL_DEV_PCI_EEAI_STAT(val)      CSL_FEXT(val, DEV_DEVSTAT_PCI_EEAI)

/** Get the VLYNQ enable status from the reset configuration value passed
  * @n 1 - VLYNQ enabled
  * @n 0 - McBSP1/GPIO and McBSP0 pin functions are enabled
  */
#define CSL_DEV_VLYNQ_EN_STAT(val)      CSL_FEXT(val, DEV_DEVSTAT_VLYNQ_EN)

/** Get the PCI frequency selected from the reset configuration value passed  */
#define CSL_DEV_PCI_FREQ_GET(val)       CSL_FEXT(val, DEV_DEVSTAT_PCI66)

/** Get the MCBSP1 enable status from the reset configuration value passed
  * @n 1 - MCBSP1 enabled
  * @n 0 - GPIO pin functions enabled on the muxed GPIO/MCBSP1 pins
  */
#define CSL_DEV_MCBSP1_EN_STAT(val)     CSL_FEXT(val, DEV_DEVSTAT_MCBSP1_EN)

/** Get the SYSCLKOUT enable status from the reset configuration value passed
  * @n 1 - SYSCLK4 pin function enabled on the muxed SYSCLK4/GP[1] pin
  * @n 0 - GP[1] pin function enabled on the muxed SYSCLK4/GP[1] pin
  */
#define CSL_DEV_SYSCLKOUT_EN_STAT(val)  CSL_FEXT(val, DEV_DEVSTAT_SYSCLKOUT_EN)
                                        
/** Get the CFGGP configuration from the reset configuration value passed     */
#define CSL_DEV_CFGGP_VAL_GET(val)      CSL_FEXT(val, DEV_DEVSTAT_CFGGP)

/** Get the status of HPI/PCI and PCI/UTOPIA multiplexed pins from the reset 
  * configuration value passed
  * @n 1 - PCI pin functions are enabled
  * @n 0 - HPI and UTOPIA pin functions are enabled
  */
#define CSL_DEV_PCI_EN_STAT(val)        CSL_FEXT(val, DEV_DEVSTAT_PCI_EN)

/** Get the status of DDR2 Memory Controller peripheral pins from the reset 
  * configuration value passed
  * @n 1 - DDR2 Memory Controller peripheral pins are enabled
  * @n 0 - DDR2 Memory Controller peripheral pins are disabled
  */
#define CSL_DEV_DDR2_EN_STAT(val)       CSL_FEXT(val, DEV_DEVSTAT_DDR2_EN)

/** Get the status of EMIFA peripheral pins from the reset 
  * configuration value passed
  * @n 1 - EMIFA peripheral pins are enabled
  * @n 0 - EMIFA peripheral pins are disabled
  */
#define CSL_DEV_EMIFA_EN_STAT(val)      CSL_FEXT(val, DEV_DEVSTAT_EMIFA_EN)

/** Enum to describe the PCI frequency selection */
typedef enum {
    /** PCI frequency 33MHz */
    CSL_DEV_PCI_33MHZ = 0,
    
    /** PCI frequency 66MHz */
    CSL_DEV_PCI_66MHZ = 1
} CSL_DevPciFreqSel;

/** Enum to describe the EMAC interface modes */
typedef enum {
    /** EMAC MII interface mode */
    CSL_DEV_EMAC_MII = 0,
    /** EMAC RMII interface mode */
    CSL_DEV_EMAC_RMII = 1,
    /** EMAC GMII interface mode */
    CSL_DEV_EMAC_GMII = 2,
    /** EMAC RGMII interface mode */
    CSL_DEV_EMAC_RGMII = 3
} CSL_DevEmacMode;

/** Enum to describe the device endian mode */
typedef enum {
    /** Big endian mode */
    CSL_DEV_BIG_ENDIAN = 0,
    
    /**Little endian mode */
    CSL_DEV_LITTLE_ENDIAN = 1
} CSL_DevEndianMode;

/** Enum to describe the HPI width */
typedef enum {
    /** HPI width 16 bit */
    CSL_DEV_HPI_WIDTH_16_BIT = 0,
    
    /** HPI width 32 bit */
    CSL_DEV_HPI_WIDTH_32_BIT = 1
} CSL_DevHpiWidth;

/** Enum to describe the EMIFA input clock selection */
typedef enum {
    /** EMIFA input clock is AECLKIN */
    CSL_DEV_EMIFA_CLK_AECLKIN = 0,
    
    /** EMIFA input clock is SYSCLK4 */
    CSL_DEV_EMIFA_CLK_SYSCLK4 = 1
} CSL_DevEmifaInputClk;

/** @brief DEV Peripherals Enumeration
 * This provides the name of peripherals controlled by the DEV module.
 * 
 */
typedef enum {
    /** TCP peripheral*/
    CSL_DEV_PERI_TCP        =   0,
    /** VCP peripheral */
    CSL_DEV_PERI_VCP        =   1,
    /** EMAC peripheral */
    CSL_DEV_PERI_EMAC       =   2,
    /** TIMER0 peripheral */
    CSL_DEV_PERI_TIMER0     =   3,
    /** TIMER1 peripheral */
    CSL_DEV_PERI_TIMER1     =   4,
    /** GPIO peripheral */
    CSL_DEV_PERI_GPIO       =   5,
    /** I2C peripheral */
    CSL_DEV_PERI_I2C        =   6,
    /** MCBSP0 peripheral */
    CSL_DEV_PERI_MCBSP0     =   7,
    /** MCBSP1 peripheral */
    CSL_DEV_PERI_MCBSP1     =   8,
    /** HPI peripheral */
    CSL_DEV_PERI_HPI        =   9,
    /** PCI peripheral */
    CSL_DEV_PERI_PCI        =   10,
    /** UTOPIA peripheral */
    CSL_DEV_PERI_UTOPIA     =   11,
    /** VLYNQ peripheral */
    CSL_DEV_PERI_VLYNQ      =   12,
    /** RSA0 peripheral */
    CSL_DEV_PERI_RSA0       =   13,
    /** RSA1 peripheral */
    CSL_DEV_PERI_RSA1       =   14,
    /** DDR2 peripheral */
    CSL_DEV_PERI_DDR2       =   15,
    /** EMIFA peripheral */
    CSL_DEV_PERI_EMIFA      =   16,
    /** SRIO peripheral */
    CSL_DEV_PERI_SRIO       =   17
} CSL_DevPeripherals;

/**
 * This is the set of control commands that are passed to
 * CSL_devHwControl(), with an optional argument type-casted to void*
 * The arguments, if any, to be passed with each command are specified
 * next to that command.
 */
typedef enum {
    /**
     * @brief   Disable the module. This is not applicable to DDR2 and EMIFA.
     * @param   (CSL_DevPeripherals *)
     */
    CSL_DEV_CMD_PERI_DISABLE            = 0,
    /**
     * @brief   Enable the module
     * @param   (CSL_DevPeripherals *)
     */
    CSL_DEV_CMD_PERI_ENABLE             = 1,
    /**
     * @brief   Powerdown the EMU pin buffers
     * @param   None
     */
    CSL_DEV_CMD_EMUBUF_POWERDN      = 2,
    /**
     * @brief   Powerup the EMU pin buffers
     * @param   None
     */
    CSL_DEV_CMD_EMUBUF_POWERUP      = 3,
    /**
     * @brief   Assign proririty for the masters in the system
     * @param   (CSL_DevPriorityCfg *)
     */
    CSL_DEV_CMD_PRIORITY_SET = 4,
    /**
     * @brief   Assert the EAMC RMII logic reset
     * @param   None
     */
    CSL_DEV_CMD_RMII_RESET_ASSERT = 5,
    /**
     * @brief   Release the EMAC RMII logic reset
     * @param   None
     */
    CSL_DEV_CMD_RMII_RESET_RELEASE = 6

} CSL_DevHwControlCmd;

/** Enumeration to describe the state of the peripheral */
typedef enum {
    /** Peripheral is in disabled state */
    CSL_DEV_PERI_DISABLED   = 0,
    /** Peripheral is in enabled state */
    CSL_DEV_PERI_ENABLED    = 1,
    /** Peripheral is in static power down state */
    CSL_DEV_PERI_STATIC_PWRDOWN     = 3,
    /** Peripheral is in disable in progress state */
    CSL_DEV_PERI_DISABLE_IN_PROG    = 4,
    /** Peripheral is in enable in progress state */
    CSL_DEV_PERI_ENABLE_IN_PROG = 5
} CSL_DevPeriState;

/**
 * This is the set of query commands to get the status of various
 * operations controlled by the DEV
 * The arguments, if any, to be passed with each query are specified
 * next to that query command
 */
typedef enum {
    /**
     * @brief   Get the state of the peripheral identified by the argument.
     *          This is not valid for peripherals SRIO, EMIFA and DDR2
     * @param   (CSL_DevPeriStatQuery *)
     */
    CSL_DEV_QUERY_PERI_STATE    = 0,
    /**
     * @brief   Get the device ID
     * @param   (CSL_DevIdInfo *)
     */
    CSL_DEV_QUERY_DEV_ID    = 1,
    /**
     * @brief   Get the device reset configuration. Macros are provided which 
     *          can be used to extract specific information.
     * @param   (Uint32 *)
     */
    CSL_DEV_QUERY_DEV_RESET_CFG = 2
} CSL_DevHwStatusQuery;

/** @brief DEV Object Structure
 *  This structure contains the register details, called register overlay,
 *  of the DEV module.
 */
typedef struct CSL_DevObj {
    /** Pointer to the register overlay structure of the DEV module */
    CSL_DevRegsOvly    regs;
    /** Specifies a particular instance of the DEV module */
    CSL_InstNum        perNum;
} CSL_DevObj;

/** @brief DEV Base Address
 * This structure is used by CSL_devGetBaseAddress() to get the base address 
 * of the DEV configuration registers.
 * 
 */
typedef struct CSL_DevBaseAddress {
    /** Base-address of the configuration registers of the DEV module */
    CSL_DevRegsOvly    regs;
} CSL_DevBaseAddress;

/** @brief DEV Parameter Structure
 * 
 * This structure contains DEV module specific parameters. 
 * Present implementation of DEV CSL doesn't have any module specific 
 * parameters. 
 * The below declaration is just a place-holder for future implementation.
 */
typedef struct CSL_DevParam {
    /** Module specific parameters. */  
    CSL_BitMask16    flags;
} CSL_DevParam;

/** @brief DEV Context
 * This structure is used to represent the context information of DEV.
 * Present implementation of DEV CSL doesn't have any context information. 
 * The below declaration is just a place-holder for future implementation.
 */
typedef struct CSL_DevContext {
    /**Context information of DEV module. */
    Uint16    contextInfo;
} CSL_DevContext;

/** @brief DEV Hardware Configuration
 *         This structure is used to pass the values to configure the
 *         DEV registers to CSL_devHwSetupRaw() function.
 */
typedef struct CSL_DevConfig {
    /** Priority allocation register */
    Uint32    PRI_ALLOC;    

    /** This is used to change the state of the peripherals controlled by the 
      * PERCFG0 register 
      */
    Uint32    PERCFG0;
    /** This is used to change the state of the peripherals controlled by the 
      * PERCFG1 registere 
      */
    Uint32    PERCFG1;

    /** Control signals for powering down/up IO buffers for the EMU pins */
    Uint32    EMUBUFPDN;
} CSL_DevConfig;

/** Defaults for the CSL_DevConfig structure */
#define CSL_DEV_CONFIG_DEFAULTS { \
    (Uint32)CSL_DEV_PRI_ALLOC_RESETVAL, \
    (Uint32)CSL_DEV_PERCFG0_RESETVAL, \
    (Uint32)CSL_DEV_PERCFG1_RESETVAL, \
    (Uint32)CSL_DEV_EMUBUFPDN_RESETVAL \
}

/** @brief DEV Peripheral State Query Structure
 * This structure is used to assign peripheral and to get value for 
 * state query.
 */
typedef struct CSL_DevPeriStatQuery {
    /** Provides the name of the peripheral for state query */
    CSL_DevPeripherals    periName;
    /** Returns the peripheral state */
    CSL_DevPeriState      periState;
} CSL_DevPeriStatQuery;

/** Structure to assign priority to the various masters
 * in the system. The priority ranges from 0 to 7. A value of 0 
 * has the highest priority, while 7 has the lowest priority. 
 */
typedef struct CSL_DevPriorityCfg {
    /** Configure the EMAC priority */
    Uint8   emacPriority;
    /** Configure the SRIO priority */
    Uint8   srioPriority;
    /** Configure the priority for HPI/PCI peripheral*/
    Uint8   hostPriority;
    /** Configure the VLYNQ priority */
    Uint8   vlynqPriority;
} CSL_DevPriorityCfg;

/** @brief Device identification structure
 *  This structure is used to return the device identification details.
 */
typedef struct CSL_DevIdInfo {
    /** The value of this field depends on the silicon revision
      * being used. 
      */
    Uint8           variant;
    /** Part number  */
    Uint16          partNumber;
    /** Manufacturer ID */
    Uint16          manufacturerId;
} CSL_DevIdInfo;

/**
 * @brief   The DEV module handle
 *
 * This is a pointer to @a CSL_DevObj and is passed as the first
 *  parameter to all DEV CSL APIs
 */
typedef struct CSL_DevObj    *CSL_DevHandle;

/*
 * =============================================================================
 *   @func CSL_devInit
 *
 *   @desc
 *      This function is idempotent i.e. calling it many times is same as
 *      calling it once. This function is only for book-keeping purpose
 *      and it doesn't touch the hardware (read/write registers) in any manner.
 *
 *   @arg pContext
 *        Context information for DEV
 *
 *   @ret CSL_Status
 *        CSL_SOK - Always returns 
 *
 *   @eg
 *     CSL_devInit (NULL);
 * =============================================================================
 */
CSL_Status CSL_devInit (
    CSL_DevContext    *pContext
);

/*
 *============================================================================
 *   @func CSL_devOpen
 *
 *   @desc
 *      Opens the instance of DEV module. The device can be re-opened 
 *      anytime after it has been normally closed, if so required. The DEV  
 *      handle returned  by this call is input as an essential argument for
 *      the rest of the APIs in DEV module. 
 *
 *   @arg pDevObj
 *        Pointer to the object that holds reference to the instance of DEV 
 *        requested after the call
 *
 *   @arg devNum
 *        Instance of DEV CSL to be opened.
 *        
 *   @arg pDevParam
 *        Pointer to module specific parameters
 *
 *   @arg pStatus
 *        pointer for returning status of the function call
 *
 *   @ret CSL_DevHandle
 *        Valid DEV instance handle will be returned 
 *        if status value is equal to CSL_SOK.
 *
 *   @eg
 *          CSL_DevHandle     hDev;
 *          CSL_DevObj        devObj;
 *          CSL_Status          status;
 *          ...
 *          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);
 *          ...
 * ===========================================================================
 */
CSL_DevHandle CSL_devOpen (
    CSL_DevObj      *pDevObj,
    CSL_InstNum     devNum,
    CSL_DevParam    *pDevParam,
    CSL_Status      *pStatus
);

/*
 *============================================================================
 *   @func CSL_devClose
 *
 *   @desc
 *      Unreserves the DEV instance identified by the handle passed.
 *
 *   @arg hDev
 *        DEV module handle
 *
 *   @ret CSL_Status
 *        CSL_SOK            - Close successful
 *        CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   @eg
 *      CSL_DevHandle     hDev;
 *      CSL_DevObj        devObj;
 *      CSL_DevHwSetup    devSetup;
 *      CSL_Status          status;
 *      ...
 *      hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);
 *      ...
 *      CSL_devClose(hDev);
 * ===========================================================================
 */
CSL_Status  CSL_devClose (
    CSL_DevHandle    hDev
);

/*
 * =============================================================================
 *   @func CSL_devHwSetupRaw
 *
 *   @desc
 *      This function initializes the device registers with the register-values
 *   provided through the Config Data structure.  For information passed through
 *   the Config Data structure refer to @a CSL_DevConfig.
 *
 *   @arg hDev
 *        Handle to the DEV instance
 *
 *   @arg config
 *        Pointer to config structure
 *
 *   @ret CSL_Status
 *        CSL_SOK             - Configuration successful
 *        CSL_ESYS_BADHANDLE  - Invalid handle
 *        CSL_ESYS_INVPARAMS  - Config structure is not
 *                              properly initialized
 *
 *   @eg
 *          CSL_DevHandle hDev;
 *          CSL_DevConfig config;
 *
 *          config.PERCFG0      = 0xc0505445;
 *          config.PERCFG1      = 0x03;
 *          config.EMUBUFPDN    = 0x00;
 *
 *          ....
 *          CSL_devHwSetupRaw(hDev, &config);
 *
 * ===========================================================================
 */
CSL_Status  CSL_devHwSetupRaw (
    CSL_DevHandle    hDev,
    CSL_DevConfig    *pConfig
);

/*
 * ============================================================================
 *   @func CSL_devHwControl
 *
 *   @desc
 *      This function takes an input control command with an optional argument
 *      and accordingly controls the DEV module.
 *
 *   @arg hDev
 *        Handle to the DEV instance
 *
 *   @arg cmd
 *        Operation to be performed on the DEV instance
 *
 *   @arg cmdArg
 *        Arguement specific to the command 
 *
 *   @ret CSL_Status
 *        CSL_SOK            - Command execution successful.
 *        CSL_ESYS_BADHANDLE - Invalid handle
 *        CSL_ESYS_INVCMD    - Invalid command
 *
 *   @eg
 *        CSL_Status            status;
 *        CSL_DevHandle         hDev;
 *        CSL_DevPeripherals    peripheral = CSL_DEV_PERI_TCP;
 *        ...
 *        // DEV object defined
 *        ...
 *      
 *        // Init successfully done
 *        ...
 *        // Open successfully done
 *        ...
 *     
 *       status = CSL_devHwControl(hDev, CSL_DEV_CMD_PERI_ENABLE, &peripheral);
 *        ...
 * ===========================================================================
 */
CSL_Status  CSL_devHwControl (
    CSL_DevHandle               hDev,
    CSL_DevHwControlCmd         cmd,
    void                        *cmdArg
);

/** ============================================================================
 *   @func CSL_devGetHwStatus
 *
 *   @desc
 *      Gets the status of different operations or configuration of DEV.
 *      The status is returned through the third parameter.
 *      For details about the various status queries supported and the 
 *      associated data structure to record the response, refer to 
 *      @a CSL_DevHwStatusQuery 
 *
 *   @arg hDev
 *        Handle to the DEV instance
 *
 *   @arg myQuery
 *        Query to be performed
 *
 *   @arg response
 *        Pointer to buffer to return the data requested by the query passed
 *
 *   @ret CSL_Status
 *        CSL_SOK              - Query execution successful.
 *        CSL_ESYS_BADHANDLE   - Invalid handle
 *        CSL_ESYS_INVQUERY    - Invalid Query
 *
 *   @eg
 *          CSL_DevHandle       hDev;
 *          CSL_Status          status;
 *          CSL_DevPeriStatQuery    peripheral;
 *
 *          peripheral.periName     = CSL_DEV_PERI_TCP;
 *          peripheral.periStatus = 0;
 *
 *          ...
 *          status = CSL_devGetHwStatus(hDev, CSL_DEV_QUERY_PERI_STAT, 
 *                                    &peripheral);
 *
 * ===========================================================================
 */
CSL_Status  CSL_devGetHwStatus (
    CSL_DevHandle           hDev,
    CSL_DevHwStatusQuery    myQuery,
    void                    *response
);

/** ============================================================================
 *   @n@b CSL_devGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_devOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            devNum          Specifies the instance of the DEV to be opened.

            pDevParam       Module specific parameters.

            pBaseAddress    Pointer to baseaddress structure.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK            Open call is successful
 *   @li                    CSL_ESYS_FAIL     The instance number is invalid.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status          status;
        CSL_DevBaseAddress  baseAddress;

       ...
      status = CSL_devGetBaseAddress(CSL_DEV, NULL, &baseAddress);

    @endverbatim
 * ===========================================================================
 */
CSL_Status CSL_devGetBaseAddress (
    CSL_InstNum              devNum,
    CSL_DevParam          *pDevParam,
    CSL_DevBaseAddress    *pBaseAddress
);

#ifdef __cplusplus
}
#endif

#endif

