/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/*  ============================================================================
 *   @file  csl_devAux.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  Auxiliary API header file for DEV CSL
 *
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  06-Apr-2005 Sudeep File Created.
 *  27-Jan-2006 sd     Modifications for the module name and spec changes   
 *  08-Mar-2006 ds     Removed disable command for DDR2 and EMIFA from 
 *                     CSL_devPeripheralDisable () API.
 * =============================================================================
 */

#ifndef _CSL_DEVAUX_H_
#define _CSL_DEVAUX_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr_dev.h>
#include <csl_dev.h>

/*  ============================================================================
 *   @n@b CSL_devPeripheralDisable
 *
 *   @b Description
 *   @n This function disables the peripheral identified by the argument 
 *      of type CSL_DevPeripherals.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

            cmdArg          Identifies the peripheral to be
                            disabled

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  The specified peripheral is disabled.
 *
 *   @b Modifies
 *   @n  The PERCFG0/PERCFG1 register.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_SRIO;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
    
          CSL_devPeripheralDisable (hDev, &peripheral);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devPeripheralDisable (
    CSL_DevHandle         hDev,
    CSL_DevPeripherals    cmdArg
)
{
    /* Disable the peripheral*/
    switch (cmdArg) {
        case (CSL_DEV_PERI_SRIO):
            /* PERLOCK register setting to unlock the PERCFG0 */
            hDev->regs->PERLOCK = CSL_DEV_PERLOCK_LOCKVAL_UNLOCK;
            CSL_FINST(hDev->regs->PERCFG0, DEV_PERCFG0_SRIOCTL, DISABLE);
            break;
        default:
            /* PERLOCK register setting to unlock the PERCFG0 */
            hDev->regs->PERLOCK = CSL_DEV_PERLOCK_LOCKVAL_UNLOCK;
            CSL_FINSR(hDev->regs->PERCFG0, (((Uint32)cmdArg<<1)), \
                      ((Uint32)cmdArg<<1), 0);
            break;
    }
}

/*  ============================================================================
 *   @n@b CSL_devPeripheralEnable
 *
 *   @b Description
 *   @n This function enables the peripheral identified by the argument 
 *      of type CSL_DevPeripherals.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

            cmdArg          Identifies the peripheral to be
                            enabled

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  The specified peripheral is enabled.
 *
 *   @b Modifies
 *   @n  The PERCFG0/PERCFG1 register.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_SRIO;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
    
          CSL_devPeripheralEnable (hDev, &peripheral);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devPeripheralEnable (
    CSL_DevHandle         hDev,
    CSL_DevPeripherals    cmdArg
)
{
    /* Enable the peripheral*/
    switch (cmdArg) {
        case (CSL_DEV_PERI_SRIO):
            /* PERLOCK register setting to unlock the PERCFG0 */
            hDev->regs->PERLOCK = CSL_DEV_PERLOCK_LOCKVAL_UNLOCK;
            CSL_FINST(hDev->regs->PERCFG0, DEV_PERCFG0_SRIOCTL, ENABLE);
            break;

        case (CSL_DEV_PERI_DDR2):
            CSL_FINST(hDev->regs->PERCFG1, DEV_PERCFG1_DDR2CTL, ENABLE);
            break;

        case (CSL_DEV_PERI_EMIFA):
            CSL_FINST(hDev->regs->PERCFG1, DEV_PERCFG1_EMIFACTL, ENABLE);
            break;

        default:
            /* PERLOCK register setting to unlock the PERCFG0 */
            hDev->regs->PERLOCK = CSL_DEV_PERLOCK_LOCKVAL_UNLOCK;
            CSL_FINSR(hDev->regs->PERCFG0, (((Uint32)cmdArg<<1)), \
                      ((Uint32)cmdArg<<1), 1);
            break;
    }
}

/*  ============================================================================
 *   @n@b CSL_devEmubufPwrdn
 *
 *   @b Description
 *   @n This function disables the power to the pin buffers of the emulator 
 *      pins.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  Power to the EMU pin buffers is disbled.
 *
 *   @b Modifies
 *   @n  EMUBUFPDN register.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
    
          CSL_devEmubufPwrdn (hDev);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devEmubufPwrdn(
    CSL_DevHandle    hDev
)
{
    CSL_FINST(hDev->regs->EMUBUFPDN, DEV_EMUBUFPDN_EMUCTL, \
             DISABLE);
}

/*  ============================================================================
 *   @n@b CSL_devEmubufPwrup
 *
 *   @b Description
 *   @n This function enables the power to the pin buffers of the emulator 
 *      pins.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  Power to the EMU pin buffers is enabled.
 *
 *   @b Modifies
 *   @n  EMUBUFPDN register.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
    
          CSL_devEmubufPwrup (hDev);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devEmubufPwrup(
    CSL_DevHandle    hDev
)
{
    CSL_FINST(hDev->regs->EMUBUFPDN, DEV_EMUBUFPDN_EMUCTL, \
             ENABLE);
}

/*  ============================================================================
 *   @n@b CSL_devMasterPrioritySet
 *
 *   @b Description
 *   @n This function sets the priorities for the various masters.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

            cmdArg          Priorities for the various masters

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  The priorities are set in the PRI_ALLOC register.
 *
 *   @b Modifies
 *   @n  The PRI_ALLOC register.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPriorityCfg prioCfg;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
          prioCfg.emacPriority = 1;
          prioCfg.vlynqPriority = 2;
          prioCfg.hostPriority = 2;
          prioCfg.srioPriority = 4;
            
          CSL_devMasterPrioritySet (hDev, &prioCfg);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devMasterPrioritySet (
    CSL_DevHandle          hDev,
    CSL_DevPriorityCfg*    cmdArg
)
{
    /* set the priority allocation register */
    CSL_FINS (hDev->regs->PRI_ALLOC, DEV_PRI_ALLOC_EMAC, 
                                            cmdArg->emacPriority);
    CSL_FINS (hDev->regs->PRI_ALLOC, DEV_PRI_ALLOC_SRIO, 
                                            cmdArg->srioPriority);
    CSL_FINS (hDev->regs->PRI_ALLOC, DEV_PRI_ALLOC_HOST, 
                                            cmdArg->hostPriority);
    CSL_FINS (hDev->regs->PRI_ALLOC, DEV_PRI_ALLOC_VLYNQ, 
                                            cmdArg->vlynqPriority);
}

/*  ============================================================================
 *   @n@b CSL_devEmacRmiiResetAssert
 *
 *   @b Description
 *   @n This function resets the RMII logic of EMAC.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  RMII logic reset is asserted
 *
 *   @b Modifies
 *   @n EMACCFG register.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
    
          CSL_devEmacRmiiResetAssert (hDev);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devEmacRmiiResetAssert(
    CSL_DevHandle    hDev
)
{
    CSL_FINST(hDev->regs->EMACCFG, DEV_EMACCFG_RMII_RST, \
             ASSERT);
}

/*  ============================================================================
 *   @n@b CSL_devEmacRmiiResetRelease
 *
 *   @b Description
 *   @n This function releases the RMII logic of EMAC out of reset.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  RMII logic reset is released
 *
 *   @b Modifies
 *   @n  EMACCFG register.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
    
          CSL_devEmacRmiiResetRelease (hDev);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devEmacRmiiResetRelease(
    CSL_DevHandle    hDev
)
{
    CSL_FINST(hDev->regs->EMACCFG, DEV_EMACCFG_RMII_RST, \
             RELEASE);
}

/*  status queries */
/*  ============================================================================
 *   @n@b CSL_devGetTcpState
 *
 *   @b Description
 *   @n This function returns the state of the TCP peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the TCP peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_TCP;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetTcpState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetTcpState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_TCPSTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetVcpState
 *
 *   @b Description
 *   @n This function returns the state of the VCP peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the VCP peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_VCP;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetVcpState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetVcpState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_VCPSTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetEmacState
 *
 *   @b Description
 *   @n This function returns the state of the EMAC peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the EMAC peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_EMAC;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetEmacState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetEmacState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_EMACSTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetTimer0State
 *
 *   @b Description
 *   @n This function returns the state of the TIMER0 peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the TIMER0 peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_TIMER0;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetTimer0State (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetTimer0State (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_TIMER0STAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetTimer1State
 *
 *   @b Description
 *   @n This function returns the state of the TIMER1 peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the TIMER1 peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_TIMER1;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetTimer1State (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetTimer1State (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_TIMER1STAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetGpioState
 *
 *   @b Description
 *   @n This function returns the state of the GPIO peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the GPIO peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_GPIO;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetGpioState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetGpioState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_GPIOSTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetI2cState
 *
 *   @b Description
 *   @n This function returns the state of the I2C peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the I2C peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_I2C;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetI2cState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetI2cState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_I2CSTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetMcbsp0State
 *
 *   @b Description
 *   @n This function returns the state of the MCBSP0 peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the MCBSP0 peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_MCBSP0;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetMcbsp0State (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetMcbsp0State (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_MCBSP0STAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetMcbsp1State
 *
 *   @b Description
 *   @n This function returns the state of the MCBSP1 peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the MCBSP1 peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_MCBSP1;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetMcbsp1State (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetMcbsp1State (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_MCBSP1STAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetHpiState
 *
 *   @b Description
 *   @n This function returns the state of the HPI peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the HPI peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_HPI;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetHpiState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetHpiState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT0,
             DEV_PERSTAT0_HPISTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetPciState
 *
 *   @b Description
 *   @n This function returns the state of the PCI peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the PCI peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_PCI;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetPciState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetPciState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT1,
             DEV_PERSTAT1_PCISTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetUtopiaState
 *
 *   @b Description
 *   @n This function returns the state of the UTOPIA peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the UTOPIA peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_UTOPIA;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetUtopiaState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetUtopiaState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT1,
             DEV_PERSTAT1_UTOPIASTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetVlynqState
 *
 *   @b Description
 *   @n This function returns the state of the VLYNQ peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the VLYNQ peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_VLYNQ;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetVlynqState (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetVlynqState (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT1,
             DEV_PERSTAT1_VLYNQSTAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetRsa0State
 *
 *   @b Description
 *   @n This function returns the state of the RSA0 peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the RSA0 peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_RSA0;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetRsa0State (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetRsa0State (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT1,
             DEV_PERSTAT1_RSA0STAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetRsa1State
 *
 *   @b Description
 *   @n This function returns the state of the RSA1 peripheral. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_DevPeriState
 *   @li                    State of the RSA1 peripheral
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevPeripherals peripheral = CSL_DEV_PERI_RSA1;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devPeripheralEnable (hDev, &peripheral);

          while (CSL_DEV_PERI_ENABLED != CSL_devGetRsa1State (hDev));
     @endverbatim
 * ===========================================================================
 */
static inline
CSL_DevPeriState CSL_devGetRsa1State (
    CSL_DevHandle    hDev
)
{
    return ((CSL_DevPeriState) CSL_FEXT (hDev->regs->PERSTAT1,
             DEV_PERSTAT1_RSA1STAT));
}

/*  ============================================================================
 *   @n@b CSL_devGetDevID
 *
 *   @b Description
 *   @n This function returns the device identification information. 
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance

            devId           Structure to hold the device identification 
                            information
     @endverbatim
 *
 *   <b> Return Value </b>  
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          CSL_DevIdInfo         devId;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          CSL_devGetDevID (hDev, &devId);
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_devGetDevID (
    CSL_DevHandle       hDev,
    CSL_DevIdInfo       *devId
)
{
    devId->manufacturerId = CSL_FEXT (hDev->regs->JTAGID, DEV_JTAGID_MANUFACTURER);
    devId->partNumber     = CSL_FEXT (hDev->regs->JTAGID, DEV_JTAGID_PARTNUMBER);
    devId->variant        = CSL_FEXT (hDev->regs->JTAGID, DEV_JTAGID_VARIANT);
}

/*  ============================================================================
 *   @n@b CSL_devGetResetCfg
 *
 *   @b Description
 *   @n This function returns the device configuration at reset. This returns 
 *      the value of DEVSTAT register.
 *
 *   @b Arguments
 *   @verbatim
            hDev            Handle to DEV instance
     @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *   @li                    Reset device configuration
 *
 *   <b> Pre Condition </b>
 *   @n  None 
 *
 *   <b> Post Condition </b>
 *   @n  None.
 *
 *   @b Modifies
 *   @n  None.
 *
 *   @b Example
 *   @verbatim

          CSL_DevHandle     hDev;
          CSL_DevObj        devObj;
          CSL_Status        status;
          Uint32            devResetCfg;
          ...
          hDev = CSL_devOpen(&devObj, CSL_DEV, NULL, &status);

          ...
                
          devResetCfg = CSL_devGetResetCfg (hDev);
     @endverbatim
 * ===========================================================================
 */
static inline
Uint32 CSL_devGetResetCfg (
    CSL_DevHandle       hDev
)
{
    return hDev->regs->DEVSTAT;
}

#ifdef __cplusplus
}
#endif

#endif

