/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
 /** ============================================================================
 *   @file  csl_vlynq.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  Header file for functional layer of VLYNQ CSL
 *
 * ===========================================================================
 */

/** ============================================================================
 *  @mainpage VLYNQ CSL 3.x (Serial Interface Module)
 *
 *  @section Introduction
 *
 *  @subsection xxx Purpose and Scope
 *   The purpose of this document is to identify a set of common CSL APIs for
 *   the VLYNQ Serial Interface Module across various devices. The CSL developer
 *   is expected to refer to this document while designing APIs for these
 *   modules. Some of the listed APIs may not be applicable to a given VLYNQ
 *   Interface Module. While in other cases this list of APIs may not be
 *   sufficient to cover all the features of VLYNQ Serial Interface Module.
 *   The CSL developer should use his discretion in designing new APIs or
 *   extending the existing ones to cover these.
 * 
 *
 *  @subsection aaa Terms and Abbreviations
 *   -# CSL:  Chip Support Library
 *   -# API:  Application Programmer Interface
 *
 *  @subsection References
 *    -# CSL 3.x Technical Requirements Specifications Version 0.5, dated
 *       May 14th, 2003
 *    -# VLYNQ Module Specification Version 2.5.2, dated December 2, 2003
 *      http://dspdesign.india.ti.com/twiki/pub/Trinity/VLYNQInterface/
        vlynq_2p5p2.doc
 *       
 *    -# VLYNQ Module Specification Version 2.6.0, dated April 15, 2004
 *       http://dspdesign.india.ti.com/twiki/pub/Trinity/VLYNQInterface/
 *       vlynq_2p6p0.pdf
 *
 *  @subsection Assumptions
 *     The abbreviations VLYNQ, Vlynq and vlynq have been used throughout this
 *     document to refer to the VLYNQ Serial Interface Module
 * ============================================================================
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  29-Oct-2004 sd  File taken from TI code base.
 *
 *  13-Dec-2005 NG  - Removed instance number, 
 *                  - Removed REVID from enum CSL_VlynqHwStatusQuery, which is 
 *                    reserved
 *                  - Updated according to guidelines
 *  02-02-2006 ds   - Updated Hwsetup Default macro
 *                  - Removed CSL_VLYNQ_HW_CONFIG_SETUP_DEFAULTS Macro
 * =============================================================================
 */


#ifndef _CSL_VLYNQ_H_
#define _CSL_VLYNQ_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <csl.h>
#include <cslr_vlynq.h>
#include <soc.h>

/******************************************************************************\
*   VLYNQ global macro declarations
\******************************************************************************/
/** Peripheral any */
#define CSL_VLYNQ_PER_ANY                   -1
    
/** Interrput pointer default value */
#define CSL_VLYNQ_INTPTR_DEF_VAL            0x14
    
/** Receive sample default value */
#define CSL_VLYNQ_DEFAULT_RXSAMPLEVAL       0x03
    
/** RerRor status */ 
#define CSL_VLYNQ_STAT_RERROR_CLEAR         0x01
    
/** LerRor status */
#define CSL_VLYNQ_STAT_LERROR_CLEAR         0x01
    
/** RerRor Rstatus */
#define CSL_VLYNQ_RSTAT_RERROR_CLEAR        0x01
    
/** LerRor Rstatus */
#define CSL_VLYNQ_RSTAT_LERROR_CLEAR        0x01
    
/** interrupt enable */
#define CSL_VLYNQ_INT_ENA                   0x01
    
/** interrupt disable */
#define CSL_VLYNQ_INT_DIS                   0x00
    
/** Pls interrupt type  */
#define CSL_VLYNQ_INT_TYPE_PLS              0x01
    
/** Lvl type interrupt */
#define CSL_VLYNQ_INT_TYPE_LVL              0x00
    
/** Active low inerrupt */
#define CSL_VLYNQ_INT_POL_ACTIVELOW         0x01
    
/** Active High interrupt*/
#define CSL_VLYNQ_INT_POL_ACTIVEHIGH        0x00

/** INTVEC0's Interrupt3 */
#define CSL_VLYNQ_INTVEC0_INT3              0x80000000
    
/** INTVEC0's Interrupt2 */
#define CSL_VLYNQ_INTVEC0_INT2              0x00800000
    
/** INTVEC0's Interrupt1 */
#define CSL_VLYNQ_INTVEC0_INT1              0x00008000
    
/** INTVEC0's Interrupt0 */
#define CSL_VLYNQ_INTVEC0_INT0              0x00000080
    
/** INTVEC1's Interrupt7 */
#define CSL_VLYNQ_INTVEC1_INT7              0x80000000
    
/** INTVEC1's Interrupt6 */
#define CSL_VLYNQ_INTVEC1_INT6              0x00800000
    
/** INTVEC1's Interrupt5 */
#define CSL_VLYNQ_INTVEC1_INT5              0x00008000
    
/** INTVEC1's Interrupt4 */
#define CSL_VLYNQ_INTVEC1_INT4              0x00000080

/******************************************************************************\
* VLYNQ global typedef declarations
\******************************************************************************/

/** @brief Enumeration for control commands passed to @a CSL_vlynqHwControl()
 *
 * This is the set of commands that are passed to the @a CSL_vlynqHwControl()
 * with an optional argument type-casted to @a void* . The arguments to be
 * passed with each enumeration if any are specified next to the enumeration */
typedef enum {
    /** Commands for local VLYNQs' configuration register */    
        /** 
         * @brief Enable Power Management
         * @param  (None)  
         */
      CSL_VLYNQ_CMD_PMEN_ENA = 1,
        /** 
         * @brief  Disable Power Management
         * @param  (None)
         */
      CSL_VLYNQ_CMD_PMEN_DIS,
        /** 
         * @brief  Serial Clock Pull-up Disable
         * @param  (None)
         */
      CSL_VLYNQ_CMD_SCLKPUDIS_DIS,
        /** 
         * @brief  Serial Clock Pull-up Enable 
         * @param  (None)
         */
      CSL_VLYNQ_CMD_SCLKPUDIS_ENA,
        /** 
         * @brief  Set RTM sample value 
         * @param  (CSL_VlynqRtmSampleVal *)
         */
      CSL_VLYNQ_CMD_RXSAMPLEVAL_SET,
       /** 
        * @brief   Disable RTM (Receive Timing Manager) 
        * @param   (None)
        */
    CSL_VLYNQ_CMD_RTMENABLE_DIS,
        /** 
         * @brief   Enable RTM (Receive Timing Manager) 
         * @param   (None)
         */
    CSL_VLYNQ_CMD_RTMENABLE_ENA,
        /** 
         * @brief   Selects TX Slow path 
         * @param   (None)
         */
    CSL_VLYNQ_CMD_TXFASTPATH_DIS,
        /** 
         * @brief   Selects TX fast path
         * @param   (None)
         */
    CSL_VLYNQ_CMD_TXFASTPATH_ENA,
        /** 
         * @brief   Selects Serial Clock divider value
         * @param   (CSL_VlynqSclkDiv *)
         */
    CSL_VLYNQ_CMD_CLKDIV_SET,
        /** 
         * @brief   VLYNQ uses internal clock, divided down vesrion of 
         *          vlynk_clk_ref
         * @param   (None)
         */
    CSL_VLYNQ_CMD_CLKDIR_OUTPUT,
        /** 
         * @brief   VLYNQ clock is sourced externally
         * @param   (None)
         */
    CSL_VLYNQ_CMD_CLKDIR_INPUT,
        /**  
         * @brief   Post Interrupts locally
         * @param   (None)
         */
    CSL_VLYNQ_CMD_INTLOCAL_LOCAL,
        /** 
         * @brief   Forward/Transmit Interrupt packets over serial- interface
         * @param   (None)
         */
    CSL_VLYNQ_CMD_INTLOCAL_REMOTE,
        /**
         * @brief   Enables Status Interrupts
         * @param   (None)
         */
    CSL_VLYNQ_CMD_INTENABLE_ENA,
        /**
         * @brief   Disables Status Interrupts
         * @param   (None)
         */
    CSL_VLYNQ_CMD_INTENABLE_DIS,
    /* Razak: Added following 2 new command for writing to 'intvec' 
     *  field of CTRL reg 
     */
        /**
         * @brief    Sets intvec field
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC_SET,
        /** 
         * @brief    Clears intvec field
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC_CLR,
        /**
         * @brief    Use INTPTR register as a pointer to Configuration register
         * @param    (None)
         */
    CSL_VLYNQ_CMD_INT2CFG_ENA,
        /**
         * @brief    Use INTPTR register as a pointer to memory
         * @param    (None)
         */
    CSL_VLYNQ_CMD_INT2CFG_DIS,
        /**
         * @brief    Disable Address Optimization
         * @param    (None)
         */
    CSL_VLYNQ_CMD_AOPTDISABLE_DIS,
        /**
         * @brief    Enable Address Optimization
         * @param    (None)
         */
    CSL_VLYNQ_CMD_AOPTDISABLE_ENA,
        /**
         * @brief    Enable internal loop-back
         * @param    (None)
         */
    CSL_VLYNQ_CMD_ILOOP_ENA,
        /**
         * @brief    Disable internal loop-back
         * @param    (None)
         */
    CSL_VLYNQ_CMD_ILOOP_DIS,
        /**
         * @brief    Assert soft reset
         * @param    (None)
         */
    CSL_VLYNQ_CMD_SOFTRESET_ASSERT,
        /**
         * @brief    Deassert soft reset
         * @param    (None)
         */
    CSL_VLYNQ_CMD_SOFTRESET_DEASSERT,
        /**
         * @brief    Clear remote error status flag
         * @param    (None)
         */
    CSL_VLYNQ_CMD_RERROR_CLR,
        /**
         * @brief    Clear local error status flag
         * @param    (None)
         */
    CSL_VLYNQ_CMD_LERROR_CLR,
        /**
         * @brief  Clears the interrupt status (which has got highest priority)
         *         pending from the INTSTATCLR register
         * @param  (None)
         */
    CSL_VLYNQ_CMD_INTPRI_INTSTAT_CLR,
        /**
         * @brief    Clear interrupt(s) status bits (refers to INTSTATCLR)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_INTSTATCLR_INTCLR_CLR,
        /**
         * @brief    Sets/generates interrupt (refers to INTPENDSET register)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_INTPENDSET_INTSET_SET,
        /**
         * @brief    Clear pending interrupts from INTPENDSET register
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_INTPENDSET_INTSET_CLR,
        /**
         * @brief    Sets the INTPTR to point a configuration register
         * @param    (None)
         */
    CSL_VLYNQ_CMD_INTPTR_REGPTR,
        /**
         * @brief    Sets the INTPTR to point to memory
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_INTPTR_MEMPTR,
        /**
         * @brief   Sets the Tx Address Map (XAM) register with a given address/
         *          value used in translating the transmit packet address
         * @param   (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_XAM_TXADRMAP_SET,
        /**
         * @brief    Sets the Rx Address Map Size1 (RAMS1) register with a given
         *           address/value used in calculating the inbound packet 
         *           address
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_RAMS1_RXADRSIZE1_SET,
        /**
         * @brief    Sets the Rx Address Map Offset1 (RAMO1) register with a 
         *           given address/value used in calculating the inbound packet
         *           address 
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_RAMO1_RXADROFFSET1_SET,
        /**
         * @brief    Sets the Rx Address Map Size2 (RAMS2) register with a given
         *           address/value used in calculating the inbound packet
         *           address
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_RAMS2_RXADRSIZE2_SET,
        /**
         * @brief   Sets the Rx Address Map Offset2 (RAMO2) register with a 
         *          given address/value used in calculating the inbound packet 
         *          address
         * @param   ( CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_RAMO2_RXADROFFSET2_SET,
        /**
         * @brief  Sets the Rx Address Map Size3 (RAMS3) register with a 
         *         given address/value used in calculating the inbound packet 
         *         address
         * @param  (Uint32 *)
         */
    CSL_VLYNQ_CMD_RAMS3_RXADRSIZE3_SET,
        /**
         * @brief   Sets the Rx Address Map Offset3 (RAMO3) register with a 
         *          given address/value used in calculating the inbound packet 
         *          address
         * @param   (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_RAMO3_RXADROFFSET3_SET,
        /**
         * @brief   Sets the Rx Address Map Size4 (RAMS4) register with a given
         *          address/value used in calculating the inbound packet address
         * @param   (Uint32 *)
         */
    CSL_VLYNQ_CMD_RAMS4_RXADRSIZE4_SET,
        /**
         * @brief   Sets the Rx Address Map Offset4 (RAMO4) register with a 
         *          given address/value used in calculating the inbound packet 
         *          address
         * @param   (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_RAMO4_RXADROFFSET4_SET,
        /**
         * @brief    Enable INTVEC0 (3-0) interrupt(s)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_INTVEC0_INT_ENA,
        /**
         * @brief    Disable INTVEC0 (3-0) interrupt(s)
         * @param    (UInt32 *)
         */
    CSL_VLYNQ_CMD_INTVEC0_INT_DIS,
        /**
         * @brief    INTVEC0s' Interrupt-3 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC0_INT3_TYPE_SET,
        /**
         * @brief    INTVEC0s' Interrupt-3 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC0_INT3_POL_SET,
        /**
         * @brief    INTVEC0's Interrupt-3 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC0_INT3_INTVEC_SET,
        /**
         * @brief    INTVEC0s' Interrupt-2 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC0_INT2_TYPE_SET,
        /**
         * @brief    INTVEC0s' Interrupt-2 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC0_INT2_POL_SET,
        /**
         * @brief    INTVEC0's Interrupt-2 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC0_INT2_INTVEC_SET,
        /**
         * @brief    INTVEC0s' Interrupt-1 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC0_INT1_TYPE_SET,
        /**
         * @brief    INTVEC0s' Interrupt-1 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
        CSL_VLYNQ_CMD_INTVEC0_INT1_POL_SET,
        /**
         * @brief    INTVEC0's Interrupt-1 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC0_INT1_INTVEC_SET,
        /**
         * @brief    INTVEC0s' Interrupt-0 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC0_INT0_TYPE_SET,
        /**
         * @brief    INTVEC0s' Interrupt-0 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC0_INT0_POL_SET,
        /**
         * @brief    INTVEC0's Interrupt-0 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC0_INT0_INTVEC_SET,
        /**
         * @brief    Enable INTVEC1 (7-4) interrupt(s)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_INTVEC1_INT_ENA,
        /**
         * @brief    Disable INTVEC1 (7-4) interrupt(s)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_INTVEC1_INT_DIS,
        /**
         * @brief    INTVEC1s' Interrupt-7 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT7_TYPE_SET,
        /**
         * @brief    INTVEC1s' Interrupt-7 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT7_POL_SET,
        /**
         * @brief    INTVEC1's Interrupt-7 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC1_INT7_INTVEC_SET,
        /**
         * @brief    INTVEC1s' Interrupt-6 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT6_TYPE_SET,
        /**
         * @brief    INTVEC1s' Interrupt-6 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT6_POL_SET,
        /**
         * @brief    INTVEC1's Interrupt-6 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC1_INT6_INTVEC_SET,
        /**
         * @brief    INTVEC1s' Interrupt-5 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT5_TYPE_SET,
        /**
         * @brief    INTVEC1s' Interrupt-5 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT5_POL_SET,
        /**
         * @brief    INTVEC1's Interrupt-5 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC1_INT5_INTVEC_SET,
        /**
         * @brief    INTVEC1s' Interrupt-4 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT4_TYPE_SET,
        /**
         * @brief    INTVEC1s' Interrupt-4 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_INTVEC1_INT4_POL_SET,
        /**
         * @brief    INTVEC1's Interrupt-4 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_INTVEC1_INT4_INTVEC_SET,

    /* Commands for remote VLYNQs' configuration register */    

        /**
         * @brief    Disable remote VLYNQ Power Management
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_PMEN_DIS,
        /**
         * @brief    Enable remote VLYNQ Power Management
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_PMEN_ENA,
        /**
         * @brief    Enable remote VLYNQ Serial Clock Pull-up-disable
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_SCLKPUDIS_ENA,
        /**
         * @brief    Disable remote VLYNQ Serial Clock Pull-up-disable
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_SCLKPUDIS_DIS,
        /**
         * @brief    Set remote VLYNQ's RTM sample value
         * @param    (CSL_VlynqRtmSampleVal *)
         */
    CSL_VLYNQ_CMD_REMOTE_RXSAMPLEVAL_SET,
        /**
         * @brief    Enable remote VLYNQs' RTM (Receive Timing Manager)
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_RTMENABLE_ENA,
        /**
         * @brief    Disable remote VLYNQs' RTM (Receive Timing Manager)
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_RTMENABLE_DIS,
        /**
         * @brief    Selects remote VLYNQs' TX fast path
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_TXFASTPATH_ENA,
        /**
         * @brief    Selects remote VLYNQs' TX Slow path
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_TXFASTPATH_DIS,
        /**
         * @brief    Selects remote VLYNQs' Serial Clock divider value
         * @param    (CSL_VlynqSclkDiv *)
         */
    CSL_VLYNQ_CMD_REMOTE_CLKDIV_SET,
        /**
         * @brief    Remote VLYNQ clock is sourced externally
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_CLKDIR_INPUT,
        /**
         * @brief    Remote VLYNQ uses internal clock, divided down 
         *           vesrion of vlynk_clk_ref
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_CLKDIR_OUTPUT,
        /**
         * @brief   In remote VLYNQ, forward/transmit Interrupt packets 
         *          over serial-interface
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_INTLOCAL_REMOTE,
        /**
         * @brief    In remote VLYNQ, post Interrupts locally
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_INTLOCAL_LOCAL,
        /**
         * @brief    Disables remote VLYNQs' Status Interrupts
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_INTENABLE_DIS,
        /**
         * @brief    Enables remote VLYNQs' Status Interrupts
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_INTENABLE_ENA,
    /* Razak: Added following 2 new command for writing to 'intvec' 
     * field of CTRL reg 
     */
        /**
         * @brief    Sets remote VLYNQs' intvec field
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC_SET,
        /**
         * @brief    Clears remote VLYNQs' intvec field
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC_CLR,
        /**
         * @brief    Disable remote VLYNQs' intcfg: Use INTPTR register 
         *           as a pointer to memory
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_INT2CFG_DIS,
        /**
         * @brief    Enable remote VLYNQs' intcfg: Use INTPTR register as a 
         *           pointer to Configuration register
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_INT2CFG_ENA,
        /**
         * @brief    Enable remote VLYNQs' Address Optimization
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_AOPTDISABLE_ENA,
        /**
         * @brief    Disable remote VLYNQs' Address Optimization
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_AOPTDISABLE_DIS,
        /**
         * @brief    Disable remote VLYNQs' internal loop-back
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_ILOOP_DIS,
        /**
         * @brief    Enable remote VLYNQs' internal loop-back
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_ILOOP_ENA,
        /**
         * @brief    Deassert remote VLYNQ's Soft RESET
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_SOFTRESET_DEASSERT,
        /**
         * @brief    Assert remote VLYNQ's Soft RESET
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_SOFTRESET_ASSERT,
        /**
         * @brief    Clear remote error status flag at remote VLYNQ
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_RERROR_CLR,
        /**
         * @brief    Clear local error status flag at remote VLYNQ
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_LERROR_CLR,
        /**
         * @brief    Clear remote VLYNQ's interrupt status (which has got  
         *           highest priority) pending from the RINTSTATCLR register
         * @param    (None)
         */
    CSL_VLYNQ_CMD_REMOTE_INTPRI_INTSTAT_CLR,
        /**
         * @brief   Clear remote VLYNQ's interrupt(s) status bits 
         *          (refers to RINTSTATCLR)
         * @param   (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTSTATCLR_INTCLR_CLR,
        /**
         * @brief   Sets/generates remote VLYNQ's interrupt 
         *          (refers to RINTPENDSET register)
         * @param   (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTPENDSET_INTSET_SET,
        /**
         * @brief    Clear remote VLYNQ's pending interrupts from RINTPENDSET 
         *           register
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTPENDSET_INTSET_CLR,
        /**
         * @brief    Sets remote VLYNQ's RINTPTR to point a configuration 
         *           register
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTPTR_REGPTR,
        /**
         * @brief    Sets remote VLYNQ's RINTPTR to point to memory
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTPTR_MEMPTR,
        /**
         * @brief    Sets remote VLYNQ's  Tx Address Map (RXAM) register with a 
         *           given address/value used in translating the transmit packet 
         *           address
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_REMOTE_XAM_TXADRMAP_SET,
        /**
         * @brief    Sets remote VLYNQ's Rx Address Map Size1 (RRAMS1) register  
         *           with a given address/value used in calculating the inbound  
         *           packet address
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMS1_RXADRSIZE1_SET,
        /**
         * @brief    Sets remote VLYNQ's Rx Address Map Offset1 (RRAMO1) register 
         *           with a given address/value used in calculating the inbound 
         *           packet address
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMO1_RXADROFFSET1_SET,
        /**
         * @brief    Sets remote VLYNQ's Rx Address Map Size2 (RRAMS2) register
         *           with  a given address/value used in calculating the inbound 
         *           packet address
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMS2_RXADRSIZE2_SET,
        /**
         * @brief    Sets remote VLYNQ's Rx Address Map Offset2 (RRAMO2)  
         *           register with a given address/value used in calculating the  
         *           inbound packet address
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMO2_RXADROFFSET2_SET,
        /**
         * @brief    Sets remote VLYNQ's Rx Address Map Size3 (RRAMS3) register  
         *           with a given address/value used in calculating the inbound 
         *           packet address
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMS3_RXADRSIZE3_SET,
        /**
         * @brief    Sets remote VLYNQ's Rx Address Map Offset3 (RRAMO3)  
         *           register with a given address/value used in calculating the  
         *           inbound packet address
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMO3_RXADROFFSET3_SET,
        /**
         * @brief   Sets remote VLYNQ's Rx Address Map Size4 (RRAMS4) register  
         *          with a given address/value used in calculating the inbound 
         *          packet address
         * @param   (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMS4_RXADRSIZE4_SET,
        /**
         * @brief    Sets remote VLYNQ's Rx Address Map Offset4 (RRAMO4)  
         *           register with a given address/value used in calculating the  
         *           inbound packet address
         * @param    (CSL_VlynqAdrPtr *)
         */
    CSL_VLYNQ_CMD_REMOTE_RAMO4_RXADROFFSET4_SET,
        /**
         * @brief    Enable remote VLYNQ's RINTVEC0 (3-0) interrupt(s)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT_ENA,
        /**
         * @brief    Disable remote VLYNQ's RINTVEC0 (3-0) interrupt(s)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT_DIS,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-3 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT3_TYPE_SET,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-3 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT3_POL_SET,
        /**
         * @brief    RINTVEC0's (remote VLYNQ's) Interrupt-3 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT3_INTVEC_SET,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-2 type   
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT2_TYPE_SET,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-2 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT2_POL_SET,
        /**
         * @brief    RINTVEC0's (remote VLYNQ's) Interrupt-2 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT2_INTVEC_SET,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-1 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT1_TYPE_SET,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-1 polarity
         * @param    (CSL_VlynqIntPolType *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT1_POL_SET,
        /**
         * @brief    RINTVEC0's (remote VLYNQ's) Interrupt-1 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT1_INTVEC_SET,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-0 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT0_TYPE_SET,
        /**
         * @brief    RINTVEC0s' (remote VLYNQ's) Interrupt-0 polarity
         * @param    (CSL_VlynqIntPolType *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT0_POL_SET,
        /**
         * @brief    RINTVEC0's (remote VLYNQ's) Interrupt-0 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC0_INT0_INTVEC_SET,
        /**
         * @brief   Enable remote VLYNQ's RINTVEC1 (7-4) interrupt(s)
         * @param   (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT_ENA,
        /**
         * @brief   Disable remote VLYNQ's RINTVEC1 (7-4) interrupt(s)
         * @param   (Uint32 *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT_DIS,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-7 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT7_TYPE_SET,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-7 polarity
         * @param    (CSL_VlynqIntPolType *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT7_POL_SET,
        /**
         * @brief    RINTVEC1's (remote VLYNQ's) Interrupt-7 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT7_INTVEC_SET,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-6 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT6_TYPE_SET,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-6 polarity
         * @param    (CSL_VlynqIntPolType *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT6_POL_SET,
        /**
         * @brief    RINTVEC1's (remote VLYNQ's) Interrupt-6 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT6_INTVEC_SET,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-5 type
         * @param    (CSL_VlynqIntType *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT5_TYPE_SET,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-5 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT5_POL_SET,
        /**
         * @brief    RINTVEC1's (remote VLYNQ's) Interrupt-5 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT5_INTVEC_SET,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-4 type
         * @param    (CSL_VlynqIntType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT4_TYPE_SET,
        /**
         * @brief    RINTVEC1s' (remote VLYNQ's) Interrupt-4 polarity
         * @param    (CSL_VlynqIntPolType *)
         */ 
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT4_POL_SET,
        /**
         * @brief    RINTVEC1's (remote VLYNQ's) Interrupt-4 interrupt vector
         * @param    (CSL_VlynqIntVector *)
         */
    CSL_VLYNQ_CMD_REMOTE_INTVEC1_INT4_INTVEC_SET    
} CSL_VlynqHwControlCmd;


/** @brief Enumeration for queries passed to @a CSL_vlynqGetHwStatus()
 *
 * This is used to get the status of different operations or to get the
 * existing setup of VLYNQ. The arguments to be passed with each
 * enumeration if any are specified next to the enumeration */
typedef enum {
/** Queries related to local VLYNQ */
        /**
         * @brief    Power Management info
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_PMEN = 1, 
        /**
         * @brief    Value in rxsmapleval bit-field
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_RXSAMPLEVAL,
        /**
         * @brief    RTM Logic info
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_RTMENABLE,
        /**
         * @brief    Transmit Path
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_TXFASTPATH, 
        /**
         * @brief    Serial clock direction
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_CLKDIR,
        /**
         * @brief    Interrupt Postage Info (local/serial interface)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTLOCAL,
        /**
         * @brief    Status Interrupts Enanbled/Disabled
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTENABLE,
        /**
         * @brief    intvec: Interrupts Vector Status 
         *           (bit in INTPENDSET, set for VLYNQ module)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC,
        /**
         * @brief    INT2CFG Status: writing the status of Interrupt Packet 
         *           to (Config-reg or memory)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INT2CFG,
        /**
         * @brief    Address Optimization
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_AOPTDISABLE,
        /**
         * @brief    Internal Loop
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_ILOOP,
        /**
         * @brief    Size/Width of inbound serial data
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_SWIDTHIN,
        /**
         * @brief    Size/Width of outbound serial data
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_SWIDTHOUT,
        /**
         * @brief    Current clock sample value used by RTM
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_RXCURRENTSAMPLE, 
        /**
         * @brief    RTM logic inclusion in RTL
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_RTM, 
        /**
        * @brief   Inbound Flow Control
        * @param   (Uint32 *)
        */
      CSL_VLYNQ_QUERY_IFLOW,
        /**
         * @brief    Outbound Flow Control
         * @param    (Uint32 *)
         */
      CSL_VLYNQ_QUERY_OFLOW,
        /**
         * @brief   Remote Packet Error status
         * @param   (Uint32 *)
         */
      CSL_VLYNQ_QUERY_RERROR,
        /**
         * @brief   Local/ Inbound Packet Error status
         * @param   (Uint32 *)
         */
      CSL_VLYNQ_QUERY_LERROR,
        /**
         * @brief    FIFO-3 Slave Command FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_NFEMPTY3,
        /**
         * @brief    FIFO-2 Slave Data FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_NFEMPTY2,
        /**
         * @brief    FIFO-1 Master Command FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_NFEMPTY1,
        /**
         * @brief    FIFO-0 Master Data FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_NFEMPTY0,
        /**
         * @brief    Any Pending Requests at Slave Interface
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_SPEND,
        /**
         * @brief    Any Pending Requests at Master Interface
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_MPEND,
        /**
         * @brief    Status of Serial Link establishment
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_LINK,
        /**
         * @brief    Any pending interrupts from INTSTATCLR register 
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTPRI_NOINTPEND,
        /**
         * @brief    Status of the highest priority interrupt pending from
         *           INTSTATCLR register
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTPRI_INTSTAT,
        /**
         * @brief    Status flags of interrupt(s) from INTSTATCLR register
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTSTATCLR_INTCLR,
        /**
         * @brief    Status of interrupt/s pending from INTPENDSET register
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTPENDSET_INTSET,
        /**
         * @brief    Get address of Config-reg/mem-location from INTPTR
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTPTR,
        /**
         * @brief    Get Chip version (Version of VLYNQ device)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_CHIPVER_DEVREV,
        /**
         * @brief    Get Chip ID (ID of VLYNQ device)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_CHIPVER_DEVID,
        /**
         * @brief    Get the Auto Negotiation Protocol Info (2.x or 1.x)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_AUTONGO_2X,
        /**
         * @brief    INTVEC0s' Interrupt-3 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT3_TYPE,
        /**
         * @brief    INTVEC0s' Interrupt-3 polarity
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT3_POL,
        /**
         * @brief    INTVEC0's Interrupt-3 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC0_INT3_INTVEC,
        /**
         * @brief    INTVEC0s' Interrupt-2 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT2_TYPE,
        /**
         * @brief   INTVEC0s' Interrupt-2 polarity
         * @param   (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT2_POL,
        /**
         * @brief    INTVEC0's Interrupt-2 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC0_INT2_INTVEC,
        /**
         * @brief    INTVEC0s' Interrupt-1 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT1_TYPE,
        /**
         * @brief    INTVEC0s' Interrupt-1 polarity
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT1_POL,
        /**
         * @brief    INTVEC0's Interrupt-1 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC0_INT1_INTVEC,
        /**
         * @brief    INTVEC0s' Interrupt-0 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT0_TYPE,
        /**
         * @brief    INTVEC0s' Interrupt-0 polarity
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC0_INT0_POL,
        /**
         * @brief    INTVEC0's Interrupt-0 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC0_INT0_INTVEC,
        /**
         * @brief    INTVEC1s' Interrupt-7 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT7_TYPE,
        /**
         * @brief    INTVEC1s' Interrupt-7 polarity
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT7_POL,
        /**
         * @brief    INTVEC1's Interrupt-7 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC1_INT7_INTVEC,
        /**
         * @brief    INTVEC1s' Interrupt-6 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT6_TYPE,
        /**
         * @brief    INTVEC1s' Interrupt-6 polarity
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT6_POL,
        /**
         * @brief    INTVEC1's Interrupt-6 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC1_INT6_INTVEC,
        /**
         * @brief    INTVEC1s' Interrupt-5 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT5_TYPE,
        /**
         * @brief    INTVEC1s' Interrupt-5 polarity
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT5_POL,
        /**
         * @brief    INTVEC1's Interrupt-5 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC1_INT5_INTVEC,
        /**
         * @brief    INTVEC1s' Interrupt-4 type
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT4_TYPE,
        /**
         * @brief    INTVEC1s' Interrupt-4 polarity
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_INTVEC1_INT4_POL,
        /**
         * @brief    INTVEC1's Interrupt-4 interrupt vector
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_INTVEC1_INT4_INTVEC,
    
    /** * @brief    Queries related to remote VLYNQ */
        /**
         * @brief    major and minor revision of remote VLYNQ module
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_REVID, 
    /* Razak: added following 2 queries for major & minor revisions */
        /**
         * @brief    major revision of remote VLYNQ module
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_REVMAJ,
        /**
         * @brief    minor revision of VLYNQ module
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_REVMIN,
        /**
         * @brief    Power Management info of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_PMEN,
        /**
         * @brief    Value in rxsmapleval bit-field of remote VLYNQ     
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RXSAMPLEVAL, 
        /** 
         * @brief    RTM Logic info of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RTMENABLE,
        /**
         * @brief    Transmit Path of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_TXFASTPATH, 
        /**
         * @brief   Serial clock direction of remote VLYNQ
         * @param   (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_CLKDIR,
        /** 
         * @brief    Remote VLYNQs' Interrupt Postage Info 
         *           (local/serial interface)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTLOCAL,
        /**
         * @brief    Remote VLYNQs' Status Interrupts Enanbled/Disabled
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTENABLE,
        /** 
         * @brief    Remote VLYNQs'intvec: Interrupts Vector Status 
         *           (bit in INTPENDSET, set for VLYNQ module)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTVEC,
        /**
         * @brief    Remote VLYNQs' INT2CFG Status: writing the status of 
         *           Interrupt Packet to (Config-reg or memory)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INT2CFG,
        /**
         * @brief    Address Optimization of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_AOPTDISABLE,
        /**
         * @brief    Internal Loop info of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_ILOOP,
        /**
         * @brief    Remote VLYNQs' Size/Width of inbound serial data
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_SWIDTHIN,
        /**
         * @brief    Remote VLYNQs' Size/Width of outbound serial data
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_SWIDTHOUT,
        /**
         * @brief    Current clock sample value used by RTM at remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RXCURRENTSAMPLE, 
        /**
         * @brief    Remote VLYNQs' RTM logic inclusion in RTL
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RTM, 
        /**
         * @brief    Remote VLYNQs' inbound Flow Control
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_IFLOW,
        /**
         * @brief    Remote VLYNQs' outbound Flow Control
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_OFLOW,
        /**
         * @brief    Remote Packet Error status of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RERROR,
        /**
         * @brief    Local/Inbound Packet Error status of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_LERROR,
        /**
         * @brief    Remote VLYNQs' FIFO-3 Slave Command FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_NFEMPTY3,
        /**
         * @brief    Remote VLYNQs' FIFO-2 Slave Data FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_NFEMPTY2,
        /**
         * @brief    Remote VLYNQs' FIFO-1 Master Command FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_NFEMPTY1,
        /**
         * @brief    Remote VLYNQs' FIFO-0 Master Data FIFO not empty status
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_NFEMPTY0,
        /**
         * @brief    Any Pending Requests at Slave Interface in remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_SPEND,
        /**
         * @brief    Any Pending Requests at Master Interface in remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_MPEND,
        /**
         * @brief    Serial Link establishment status of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_LINK,
        /**
         * @brief    Any pending interrupts in INTSTATCLR register in remote 
         *           VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTPRI_NOINTPEND,
        /**
         * @brief    Status of the highest priority interrupt pending from
         *           INTSTATCLR register in remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTPRI_INTSTAT,
        /**
         * @brief    Status flags of interrupt(s) in INTSTATCLR register 
         *           in remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTSTATCLR_INTCLR,
        /**
         * @brief    Status of interrupt(s) pending from INTPENDSET register
         *           in remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTPENDSET_INTSET,
        /**
         * @brief    Get address of Config-reg/mem-location from INTPTR of 
         *           remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_INTPTR,
        /**
         * @brief    Get remote VLYNQs' Chip version (Version of VLYNQ device)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_CHIPVER_DEVREV,
        /**
         * @brief    Get remote VLYNQs' Chip ID (ID of VLYNQ device)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_CHIPVER_DEVID,
        /**
         * @brief    Get remote VLYNQs' Auto Negotiation Protocol 
         *           Info (2.x or 1.x)
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_AUTONGO_2X,
        /**
         * @brief    RINTVEC0s' Interrupt-3 type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT3_TYPE,
        /**
         * @brief    RINTVEC0s' Interrupt-3 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT3_POL,
        /**
         * @brief    RINTVEC0's Interrupt-3 interrupt vector of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT3_INTVEC,
        /**
         * @brief    RINTVEC0s' Interrupt-2 type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT2_TYPE,
        /**
         * @brief    RINTVEC0s' Interrupt-2 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT2_POL,
        /**
         * @brief    RINTVEC0's Interrupt-2 interrupt vector of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT2_INTVEC,
        /**
         * @brief    INTVEC0s' Interrupt-1 type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT1_TYPE,
        /**
         * @brief    RINTVEC0s' Interrupt-1 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT1_POL,
        /**
         * @brief    RINTVEC0's Interrupt-1 interrupt vector of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT1_INTVEC,
        /**
         * @brief    RINTVEC0s' Interrupt-0 type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT0_TYPE,
        /**
         * @brief    RINTVEC0s' Interrupt-0 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT0_POL,
        /**
         * @brief    RINTVEC0's Interrupt-0 interrupt vector of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC0_INT0_INTVEC,
        /**
         * @brief    RINTVEC1s' Interrupt-7 type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT7_TYPE,
        /**
         * @brief    RINTVEC1s' Interrupt-7 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT7_POL,
        /**
         * @brief    RINTVEC1's Interrupt-7 interrupt vector of remote VLYNQ 
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT7_INTVEC,
        /**
         * @brief    RINTVEC1s' Interrupt-6type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT6_TYPE,
        /**
         * @brief    RINTVEC1s' Interrupt-6 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT6_POL,
        /**
         * @brief    RINTVEC1's Interrupt-6 interrupt vector of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT6_INTVEC,
        /**
         * @brief    RINTVEC1s' Interrupt-5 type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT5_TYPE,
        /**
         * @brief    RINTVEC1s' Interrupt-5 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT5_POL,
        /**
         * @brief    RINTVEC1's Interrupt-5 interrupt vector of remote VLYNQ
         * @param    (Uint32 *)
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT5_INTVEC,
        /**
         * @brief    RINTVEC1s' Interrupt-4 type of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT4_TYPE,
        /**
         * @brief    RINTVEC1s' Interrupt-4 polarity of remote VLYNQ
         * @param    (Uint32 *)
         */ 
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT4_POL,
        /**
         * @brief    RINTVEC1's Interrupt-4 interrupt vector of remote VLYNQ
         * @param    (Uint32 *) 
         */
    CSL_VLYNQ_QUERY_REMOTE_RINTVEC1_INT4_INTVEC
} CSL_VlynqHwStatusQuery;

/** @brief Enumeration for VLYNQ serial clock output divider */
typedef enum {
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_0 + 1) */
    CSL_VLYNQ_CLKDIV_0 = 0,
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_1 + 1) */
    CSL_VLYNQ_CLKDIV_1 = 1,
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_2 + 1) */
    CSL_VLYNQ_CLKDIV_2 = 2,
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_3 + 1) */
    CSL_VLYNQ_CLKDIV_3 = 3,
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_4 + 1) */
    CSL_VLYNQ_CLKDIV_4 = 4,
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_5 + 1) */
    CSL_VLYNQ_CLKDIV_5 = 5,
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_6 + 1) */
    CSL_VLYNQ_CLKDIV_6 = 6,
    /** Serial Clock frequency is vlynq_clk_ref/(CSL_VLYNQ_CLKDIV_7 + 1) */
    CSL_VLYNQ_CLKDIV_7 = 7
} CSL_VlynqSclkDiv;

/** @brief Enumeration for VLYNQ serial clock direction */
typedef enum {
    /** Clock sourced externally */
    CSL_VLYNQ_CLKDIR_EXTERNAL = 0, /*CSL_VLYNQ_CTRL_CLKDIR_INPUT */
    /** Internal clock source from divded down version of vlynq_clk_ref
     *  (VBUS Clock) */
    CSL_VLYNQ_CLKDIR_INTERNAL = 1 /*CSL_VLYNQ_CTRL_CLKDIR_OUTPUT*/
} CSL_VlynqClkDir;


/** @brief Enumeration for VLYNQ Interrpt type */
typedef enum {
    /** Level sensitive type */
    CSL_VLYNQ_INT_TYPELVL = 0,
    /** Pulse tpye */
    CSL_VLYNQ_INT_TYPEPLS = 1
} CSL_VlynqIntType;

/** @brief Enumeration for VLYNQ Interrpt polarity */
typedef enum {
    /** Active Low */
    CSL_VLYNQ_INT_POL_ACTIVE_LOW = 0,
    /** Active High */
    CSL_VLYNQ_INT_POL_ACTIVE_HIGH = 1
} CSL_VlynqIntPolType;

/** @brief Enumeration for VLYNQ Interrpt vector */
typedef enum {
    /** Interrupt Vector 0 */
    CSL_VLYNQ_INT_VECTOR_0 = 0,
    /** Interrupt Vector 1 */
    CSL_VLYNQ_INT_VECTOR_1 = 1,
    /** Interrupt Vector 2 */
    CSL_VLYNQ_INT_VECTOR_2 = 2,
    /** Interrupt Vector 3 */
    CSL_VLYNQ_INT_VECTOR_3 = 3,
    /** Interrupt Vector 4 */
    CSL_VLYNQ_INT_VECTOR_4 = 4,
    /** Interrupt Vector 5 */
    CSL_VLYNQ_INT_VECTOR_5 = 5,
    /** Interrupt Vector 6 */
    CSL_VLYNQ_INT_VECTOR_6 = 6,
    /** Interrupt Vector 7 */
    CSL_VLYNQ_INT_VECTOR_7 = 7,
    /** Interrupt Vector 8 */
    CSL_VLYNQ_INT_VECTOR_8 = 8,
    /** Interrupt Vector 9 */
    CSL_VLYNQ_INT_VECTOR_9 = 9,
    /** Interrupt Vector 10 */
    CSL_VLYNQ_INT_VECTOR_10 = 10,
    /** Interrupt Vector 11 */
    CSL_VLYNQ_INT_VECTOR_11 = 11,
    /** Interrupt Vector 12 */
    CSL_VLYNQ_INT_VECTOR_12 = 12,
    /** Interrupt Vector 13 */
    CSL_VLYNQ_INT_VECTOR_13 = 13,
    /** Interrupt Vector 14 */
    CSL_VLYNQ_INT_VECTOR_14 = 14,
    /** Interrupt Vector 15 */
    CSL_VLYNQ_INT_VECTOR_15 = 15,
    /** Interrupt Vector 16 */
    CSL_VLYNQ_INT_VECTOR_16 = 16,
    /** Interrupt Vector 17 */
    CSL_VLYNQ_INT_VECTOR_17 = 17,
    /** Interrupt Vector 18 */
    CSL_VLYNQ_INT_VECTOR_18 = 18,
    /** Interrupt Vector 19 */
    CSL_VLYNQ_INT_VECTOR_19 = 19,
    /** Interrupt Vector 20 */
    CSL_VLYNQ_INT_VECTOR_20 = 20,
    /** Interrupt Vector 21 */
    CSL_VLYNQ_INT_VECTOR_21 = 21,
    /** Interrupt Vector 22 */
    CSL_VLYNQ_INT_VECTOR_22 = 22,
    /** Interrupt Vector 23 */
    CSL_VLYNQ_INT_VECTOR_23 = 23,
    /** Interrupt Vector 24 */
    CSL_VLYNQ_INT_VECTOR_24 = 24,
    /** Interrupt Vector 25 */
    CSL_VLYNQ_INT_VECTOR_25 = 25,
    /** Interrupt Vector 26 */
    CSL_VLYNQ_INT_VECTOR_26 = 26,
    /** Interrupt Vector 27 */
    CSL_VLYNQ_INT_VECTOR_27 = 27,
    /** Interrupt Vector 28 */
    CSL_VLYNQ_INT_VECTOR_28 = 28,
    /** Interrupt Vector 29 */
    CSL_VLYNQ_INT_VECTOR_29 = 29,
    /** Interrupt Vector 30 */
    CSL_VLYNQ_INT_VECTOR_30 = 30,
    /** Interrupt Vector 31 */
    CSL_VLYNQ_INT_VECTOR_31 = 31
} CSL_VlynqIntVector;


/** @brief Enumeration for VLYNQ RTM Sample Values */
typedef enum {
    /** RTM Sample Value = 0 */
    CSL_VLYNQ_RXSAMPLEVAL_0 = 0,
    /** RTM Sample Value = 1 */
    CSL_VLYNQ_RXSAMPLEVAL_1 = 1,
    /** RTM Sample Value = 2 */
    CSL_VLYNQ_RXSAMPLEVAL_2 = 2,
    /** RTM Sample Value = 3 */
    CSL_VLYNQ_RXSAMPLEVAL_3 = 3,
    /** RTM Sample Value = 4 */
    CSL_VLYNQ_RXSAMPLEVAL_4 = 4,
    /** RTM Sample Value = 5 */
    CSL_VLYNQ_RXSAMPLEVAL_5 = 5,
    /** RTM Sample Value = 6 */
    CSL_VLYNQ_RXSAMPLEVAL_6 = 6,
    /** RTM Sample Value = 7 */
    CSL_VLYNQ_RXSAMPLEVAL_7 = 7
} CSL_VlynqRtmSampleVal;


/** @brief this is a void pointer used to point register, memory-location */
typedef void *CSL_VlynqAdrPtr;

/** @brief main structure that is used to setup the VLYNQ's control reg
 *
 * This structure is used to setup or obtain the existing setup of
 * VLYNQ using @a CSL_vlynqHwSetup() & @a CSL_vlynqGetHwStatus() functions
 * respectively. If a particualr member pointer is null, then these
 * functions do not setup or get the setup of the corresponding part
 * of VLYNQ respectively */
typedef struct CSL_VlynqHwControlSetup {
    /** Power Management Enable */
    Bool                    pMen;
    /** Serial Clock pull-up Disable*/
    Bool                    sclkPuDis;
    /** RTM Sample Value (Defualt = 0x03)*/
    CSL_VlynqRtmSampleVal   rxSampleVal;
    /** RTM Enable (0-Disable) */
    Bool                    rtmEnable;
    /** TX transmit Path (0-Slow) */
    Bool                    txFastPath; 
    /** Serial Clock Divider for VLYNQ */
    CSL_VlynqSclkDiv        sclkDiv;
    /** Clock direction */
    CSL_VlynqClkDir         sclkDir;
    /** Interrupt Post (1-local INTSTATCLR reg)
     *  (0-packet transmission via setial interface) */
    Bool                    intLocal;
    /** VLYNQ Status interrupts enable/disable (0-Disable) */
    Bool                    intEnable;
    /** Interrupt to Configuration Register
     *  (1-Interrupt packet is written to local config register)
     *  (0-Interrupt packet is written to memory pointed by INTPTR reg) */
    Bool                    int2cfg;
    /** Address Optimization Disable (1-Disable) */
    Bool                    aOptDisable;
    /** Internal Loopback (0-Disable) */
    Bool                    iLoop;
    /** Soft reset (0-Reset Deassert) */
    Bool                    reset;
    /** Interrupt Pointer (default pointer to INTPENDSET-offset 0x14) */    
    CSL_VlynqAdrPtr         intPtr; 
    /** Tx Address Map */
    CSL_VlynqAdrPtr         txAdrMap;
    /** Rx Addres Map Size1 */
    Uint32                  rxAdrSize1;
    /** Rx Address Map Offset1 */
    CSL_VlynqAdrPtr         rxAdrOffset1;
    /** Rx Addres Map Size2 */
    Uint32                  rxAdrSize2;
    /** Rx Address Map Offset2 */
    CSL_VlynqAdrPtr         rxAdrOffset2;
    /** Rx Addres Map Size3 */
    Uint32                  rxAdrSize3;
    /** Rx Address Map Offset3 */
    CSL_VlynqAdrPtr         rxAdrOffset3;
    /** Rx Addres Map Size4 */
    Uint32                  rxAdrSize4;
    /** Rx Address Map Offset4 */
    CSL_VlynqAdrPtr         rxAdrOffset4;
} CSL_VlynqHwControlSetup;


/** @brief this is a pointer to @a CSL_VlynqHwControlSetup & 
 *  is passed as parameter to all VLYNQ CSL APIs */
typedef CSL_VlynqHwControlSetup *CSL_VlynqPtrHwSetup;


/** @brief main structure that is used to setup the VLYNQ module
 *
 * This structure is used to setup or obtain the existing setup of
 * VLYNQ using @a CSL_vlynqHwSetup() & @a CSL_vlynqGetHwStatus() functions
 * respectively. If a particualr member pointer is null, then these
 * functions do not setup or get the setup of the corresponding part
 * of VLYNQ respectively */
typedef struct CSL_VlynqHwSetup {
    /** Pointer to CSL_VlynqHwControlSetup used to hold the 
     *  information for local VLYNQ's control setup */
    CSL_VlynqPtrHwSetup     localHwSetup;
    /** Pointer to CSL_VlynqHwControlSetup used to hold the 
     *  information for remote VLYNQ's control setup */
    CSL_VlynqPtrHwSetup     remoteHwSetup;
} CSL_VlynqHwSetup;

/** @brief Config-structure used to configure the VLYNQ module using
 * CSL_vlynqHwSetupRaw()
 */
typedef struct CSL_VlynqConfig {
    /** Control Register */
    Uint32 CTRL;
    /** Status Register */
    Uint32 STAT;
    /** Interrupt Pending/Set Register */
    Uint32 INTPENDSET;
    /** Interrupt Pointer Register */
    Uint32 INTPTR;
    /** Tx Address Map Register */
    Uint32 XAM;
    /** Rx Address Map Size 1 Register */
    Uint32 RAMS1;
    /** Rx Address Map Offset 1 Register */
    Uint32 RAMO1;
    /** Rx Address Map Size 2 Register */
    Uint32 RAMS2;
    /** Rx Address Map Offset 2 Register */
    Uint32 RAMO2;
    /** Rx Address Map Size 3 Register */
    Uint32 RAMS3;
    /** Rx Address Map Offset 3 Register */
    Uint32 RAMO3;
    /** Rx Address Map Size 4 Register */
    Uint32 RAMS4;
    /** Rx Address Map Offset 4 Register */
    Uint32 RAMO4;
    /** Remote Control Register */
    Uint32 RCTRL;
    /** Remote Status Register */
    Uint32 RSTAT;
    /** Remote Interrupt Pending/Set Register */
    Uint32 RINTPENDSET;
    /** Remote Interrupt Pointer Register */
    Uint32 RINTPTR;
    /** Remote Tx Address Map Register */
    Uint32 RXAM;
    /** Remote Rx Address Map Size 1 Register */
    Uint32 RRAMS1;
    /** Remote Rx Address Map Offset 1 Register */
    Uint32 RRAMO1;
    /** Remote Rx Address Map Size 2 Register */
    Uint32 RRAMS2;
    /** Remote Rx Address Map Offset 2 Register */
    Uint32 RRAMO2;
    /** Remote Rx Address Map Size 3 Register */
    Uint32 RRAMS3;
    /** Remote Rx Address Map Offset 3 Register */
    Uint32 RRAMO3;
    /** Remote Rx Address Map Size 4 Register */
    Uint32 RRAMS4;
    /** Remote Rx Address Map Offset 4 Register */
    Uint32 RRAMO4;
} CSL_VlynqConfig;


/** @brief This structure contains Module specific context information. 
 * Present implementation of VLYNQ CSL doesn't have any context information.
 */
typedef struct {
    /**
     * Context information of VLYNQ port CSL.
     * The below declaration is just a place-holder for future implementation.
     */
    Uint16  contextInfo;
} CSL_VlynqContext;


/** @brief This structure contains Module specific parameters. 
 * Present implementation of VLYNQ CSL doesn't have any module specific
 * parameters.
 */
typedef struct {
    /**
     * Bit mask to be used for module specific parameters. The below
     * declaration is just a place-holder for future implementation.
     */
    CSL_BitMask16       flags;
} CSL_VlynqParam;


/** @brief This structure contains the base-address information for the
 * peripheral instance.
 */
typedef struct {
    /** Base-address of the configuration registers of the peripheral */
    CSL_VlynqRegsOvly       regs;
} CSL_VlynqBaseAddress;


/** @brief VLYNQ module object structure */
typedef struct CSL_VlynqObj {
    /** Pointer to the register overlay structure of VLYNQ module   */
    CSL_VlynqRegsOvly    regs;

    /** Instance of VLYNQ module being referred by this object          */
    CSL_InstNum          perNum;
} CSL_VlynqObj;


/** @brief this is a pointer to @a CSL_VlynqObj & is passed as the first
 * parameter to all VLYNQ CSL APIs */
typedef struct CSL_VlynqObj *CSL_VlynqHandle;


/** hardware control setup defaults */
#define CSL_VLYNQ_HW_CONTROL_SETUP_DEFAULTS  {\
    CSL_VLYNQ_CMD_PMEN_ENA,           \
    CSL_VLYNQ_CMD_SCLKPUDIS_DIS,      \
    (CSL_VlynqRtmSampleVal)3,\
    CSL_VLYNQ_CMD_RTMENABLE_DIS,      \
    CSL_VLYNQ_CMD_REMOTE_TXFASTPATH_DIS,  \
    CSL_VLYNQ_CLKDIV_0,              \
    CSL_VLYNQ_CLKDIR_EXTERNAL,          \
    CSL_VLYNQ_CTRL_INTLOCAL_RESETVAL,   \
    CSL_VLYNQ_CMD_INTENABLE_DIS,        \
    CSL_VLYNQ_CMD_INT2CFG_DIS,           \
    CSL_VLYNQ_CMD_AOPTDISABLE_DIS,       \
    CSL_VLYNQ_CMD_ILOOP_DIS,             \
    CSL_VLYNQ_CMD_REMOTE_SOFTRESET_ASSERT,  \
    0x0,                             \
    0x0,                             \
    0x0,                             \
    0x0,                             \
    0x0,                             \
    0x0,                             \
    0x0,                             \
    0x0,                             \
    0x0,                             \
    0x0                              \
}

/** Default hwsetup */
#define CSL_VLYNQ_HWSETUP_DEFAULTS { \
 &(CSL_VlynqPtrHwSetup)(CSL_VLYNQ_HW_CONTROL_SETUP_DEFAULTS), \
 &(CSL_VlynqPtrHwSetup)(CSL_VLYNQ_HW_CONTROL_SETUP_DEFAULTS) \
}


/** config defaults */
#define CSL_VLYNQ_CONFIG_DEFAULTS  {   \
    CSL_VLYNQ_CTRL_RESETVAL,           \
    CSL_VLYNQ_STAT_RESETVAL,           \
    CSL_VLYNQ_INTPENDSET_RESETVAL,     \
    CSL_VLYNQ_INTPTR_RESETVAL,         \
    CSL_VLYNQ_XAM_RESETVAL,            \
    CSL_VLYNQ_RAMS1_RESETVAL,          \
    CSL_VLYNQ_RAMO1_RESETVAL,          \
    CSL_VLYNQ_RAMS2_RESETVAL,          \
    CSL_VLYNQ_RAMO2_RESETVAL,          \
    CSL_VLYNQ_RAMS3_RESETVAL,          \
    CSL_VLYNQ_RAMO3_RESETVAL,          \
    CSL_VLYNQ_RAMS4_RESETVAL,          \
    CSL_VLYNQ_RAMO4_RESETVAL,          \
    CSL_VLYNQ_RCTRL_RESETVAL,          \
    CSL_VLYNQ_RSTAT_RESETVAL,          \
    CSL_VLYNQ_RINTPENDSET_RESETVAL,    \
    CSL_VLYNQ_RINTPTR_RESETVAL,        \
    CSL_VLYNQ_RXAM_RESETVAL,           \
    CSL_VLYNQ_RRAMS1_RESETVAL,         \
    CSL_VLYNQ_RRAMO1_RESETVAL,         \
    CSL_VLYNQ_RRAMS2_RESETVAL,         \
    CSL_VLYNQ_RRAMO2_RESETVAL,         \
    CSL_VLYNQ_RRAMS3_RESETVAL,         \
    CSL_VLYNQ_RRAMO3_RESETVAL,         \
    CSL_VLYNQ_RRAMS4_RESETVAL,         \
    CSL_VLYNQ_RRAMO4_RESETVAL,         \
}


/** ============================================================================
 *   @n@b CSL_vlynqInit
 *
 *   @b Description
 *   @n This is the initialization function for the vlynq CSL.
 *      The function must be called before calling any other API from this CSL.
 *      This function is idem-potent. Currently, the function just return
 *      status CSL_SOK, without doing anything.
 *
 *   @b Arguments
 *   @verbatim
 
        pContext   Pointer to module-context. As vlynq doesn't have any context
                   based information user is expected to pass NULL.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                        CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The CSL for vlynq is initialized
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
            CSL_vlynqInit();
     @endverbatim
 * =============================================================================
 */
CSL_Status  CSL_vlynqInit(
    CSL_VlynqContext*   pContext
);


/** ============================================================================
 *   @n@b CSL_vlynqOpen
 *
 *   @b Description
 *   @n This function populates the peripheral data object for the VLYNQ instance
 *      and returns a handle to the instance.
 *      The open call sets up the data structures for the particular instance
 *      of VLYNQ device. The device can be re-opened anytime after it has been
 *      normally closed if so required. The handle returned by this call is
 *      input as an essential argument for rest of the APIs described
 *      for this module.
 *
 *   @b Arguments
 *   @verbatim
            pVlynqObj       Pointer to VLYNQ object.

            vlynqNum        Instance of VLYNQ CSL to be opened.
                            So, the value for this parameter will be
                            based on the instance.

            pVlynqParam     Module specific parameters.

            status          Status of the function call
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_VlynqHandle
 *   @n                         Valid VLYNQ handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_vlynqInit(), CSL_vlynqOpen() must be called successfully in that
 *       order before this API can be invoked
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid VLYNQ handle is returned
 *   @li            CSL_ESYS_FAIL       The VLYNQ instance is invalid
 *
 *        2.    VLYNQ object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. VLYNQ object structure
 *
 *   @b Example
 *   @verbatim
            
       CSL_VlynqObj     vlynqObj;
       CSL_Status       status;
       ...
       hVlynq = CSL_vlynqOpen(&vlynqObj, CSL_VLYNQ, NULL, &status);
       ...
    @endverbatim
 * =============================================================================
 */
CSL_VlynqHandle CSL_vlynqOpen (
    /* pointer to the object that holds reference to the
     * instance of VLYNQ requested after the call */
    CSL_VlynqObj   *pVlynqObj,
    
    /* instance of VLYNQ to which a handle is requested */
    CSL_InstNum    vlynqNum,
    
    /* VLYNQ module specific parameters */
    CSL_VlynqParam *pVlynqParam,
    
    /* This returns the status (success/errors) of the call */
    CSL_Status     *status
);


/** ============================================================================
 *   @n@b CSL_vlynqClose
 *
 *   @b Description
 *   @n This function marks that CSL for the VLYNQ instance is closed.
 *      CSL for the VLYNQ instance need to be reopened before using any
 *      VLYNQ CSL API.
 *
 *   @b Arguments
 *   @verbatim
            hVlynq         Handle to the VLYNQ instance
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            -VLYNQ is closed
 *                                               successfully
 *
 *   @li                    CSL_ESYS_BADHANDLE - The handle passed is invalid
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_vlynqInit(), CSL_vlynqOpen() must be called successfully in that
 *       order before this API can be invoked
 *
 *   <b> Post Condition </b>
 *   @n  1. The VLYNQ CSL APIs can not be called until the VLYNQ
 *          CSL is reopened again using CSL_vlynqOpen()
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
            ...
            CSL_vlynqClose(hVlynq);
    @endverbatim
 * ===========================================================================
 */
CSL_Status  CSL_vlynqClose(
    /* pointer to the object that holds reference to the
     * instance of VLYNQ requested after the call */
    CSL_VlynqHandle hVlynq
);


/** =========================================================================== 
 *   @n@b CSL_vlynqHwSetup
 *
 *   @b Description
 *   @n It configures the  VLYNQ instance registers as per the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hVlynq          Handle to the VLYNQ instance

            hwSetup         Pointer to hardware setup structure

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup retrived
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
 *                                                properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_vlynqInit(), CSL_vlynqOpen() must be called successfully in that
 *       order before this API can be invoked
 *
 *   <b> Post Condition </b>
 *   @n  The specified instance will be setup according to value passed
 *
 *   @b Modifies
 *   @n None
 *   
 *   @b Example:
 *   @verbatim
        CSL_VlynqHandle             hVlynq;
        CSL_VlynqHwSetup            hwSetup;
        CSL_VlynqHwControlSetup     localSetup;
        CSL_VlynqHwControlSetup     remoteSetup;
        
        localSetup.pMen             = (Bool)0;
        localSetup.sclkPuDis        = (Bool)1;
        localSetup.rxSampleVal      = CSL_VLYNQ_RXSAMPLEVAL_3;
        localSetup.rtmEnable        = (Bool)0;
        localSetup.txFastPath       = (Bool)1;
        localSetup.sclkDiv          = CSL_VLYNQ_CLKDIV_1;
        localSetup.sclkDir          = CSL_VLYNQ_CLKDIR_INTERNAL;
        localSetup.intLocal         = (Bool)1;
        localSetup.intEnable        = (Bool)1;
        localSetup.int2cfg          = (Bool)1;
        localSetup.aOptDisable      = (Bool)1;
        localSetup.iLoop            = (Bool)0;
        localSetup.reset            = (Bool)0;
        localSetup.intPtr           = (CSL_VlynqAdrPtr)0x14;
        localSetup.txAdrMap         = (CSL_VlynqAdrPtr)VLYNQ_TXADR_MAP;
        localSetup.rxAdrSize1       = 50;
        localSetup.rxAdrOffset1     = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP1;
        localSetup.rxAdrSize2       = 50;
        localSetup.rxAdrOffset2     = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP2;
        localSetup.rxAdrSize3       = 50;
        localSetup.rxAdrOffset3     = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP3;
        localSetup.rxAdrSize4       = 50;
        localSetup.rxAdrOffset4     = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP4;

        remoteSetup.pMen            = (Bool)0;
        remoteSetup.sclkPuDis       = (Bool)1;
        remoteSetup.rxSampleVal     = CSL_VLYNQ_RXSAMPLEVAL_3;
        remoteSetup.rtmEnable       = (Bool)0;
        remoteSetup.txFastPath      = (Bool)1;
        remoteSetup.sclkDiv         = CSL_VLYNQ_CLKDIV_1;
        remoteSetup.sclkDir         = CSL_VLYNQ_CLKDIR_EXTERNAL;
        remoteSetup.intLocal        = (Bool)0;
        remoteSetup.intEnable       = (Bool)0;
        remoteSetup.int2cfg         = (Bool)0;
        remoteSetup.aOptDisable     = (Bool)0;
        remoteSetup.iLoop           = (Bool)0;
        remoteSetup.reset           = (Bool)0;
        remoteSetup.intPtr          = (CSL_VlynqAdrPtr)0x94;
        remoteSetup.txAdrMap        = (CSL_VlynqAdrPtr)VLYNQ_TXADR_MAP;
        remoteSetup.rxAdrSize1      = 50;
        remoteSetup.rxAdrOffset1    = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP1;
        remoteSetup.rxAdrSize2      = 50;
        remoteSetup.rxAdrOffset2    = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP2;
        remoteSetup.rxAdrSize3      = 50;
        remoteSetup.rxAdrOffset3    = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP3;
        remoteSetup.rxAdrSize4      = 50;
        remoteSetup.rxAdrOffset4    = (CSL_VlynqAdrPtr)VLYNQ_RXADR_MAP4;

        hwSetup.localHwSetup  = &localSetup;
        hwSetup.remoteHwSetup = &remoteSetup;
        

        CSL_vlynqHwSetup(hVlynq, &hwSetup);
   @endverbatim
 * ============================================================================
 */
CSL_Status  CSL_vlynqHwSetup(
    /* pointer to the object that holds reference to the
     * instance of VLYNQ requested after the call */
    CSL_VlynqHandle     hVlynq,
    /* pointer to setup structure which contains the
     * information to program VLYNQ to a useful state */
    CSL_VlynqHwSetup    *setup
);


/** ============================================================================
 *   @n@b CSL_vlynqGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters
 *
 *   @b Arguments
 *   @verbatim
            hVlynq          Handle to the VLYNQ instance

            hwSetup         Pointer to hardware setup structure

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup retrived
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_vlynqInit(), CSL_vlynqOpen() must be called successfully in that
 *       order before this API can be invoked
 *
 *   <b> Post Condition </b>
 *   @n  The hardware set up structure will be populated with values from
 *       the registers
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_status          status;
        CSL_VlynqHwSetup    hwSetup;
        CSL_VlynqHandle     hVlynq

        status = CSL_tmrGetHwsetup(hVlynq, &hwSetup);
     @endverbatim
 * ===========================================================================
 */
CSL_Status  CSL_vlynqGetHwSetup(
    /* pointer to the object that holds reference to the
     * instance of VLYNQ requested after the call */
    CSL_VlynqHandle     hVlynq,
    
    /* pointer to setup structure which contains the
     * information to program VLYNQ to a useful state */
    CSL_VlynqHwSetup    *setup
);


/** =========================================================================== 
 *   @n@b CSL_vlynqHwControl
 *
 *   @b Description
 *   @n This function performs various control operations on the VLYNQ instance,
 *      based on the command passed.
 *
 *   @b Arguments
 *   @verbatim
            hTmr         Handle to the VLYNQ instance

            cmd          Operation to be performed on the VLYNQ

            cmdArg       Arguement specific to the command

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Command execution successful.
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *   @li                    CSL_ESYS_INVCMD    - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_vlynqInit(), CSL_vlynqOpen() must be called successfully in that
 *       order before this API can be invoked
 *
 *   <b> Post Condition </b>
 *   @n  Registers of the VLYNQ instance are configured according to the command
 *       and the command arguments. The command determines which registers are
 *       modified.
 *
 *   @b Modifies
 *   @n Registers determined by the command
 *
 *   @b Example:
 *   @verbatim
         CSL_VlynqHandle hVlynq;
         CSL_Status status;
         Uint32 mask;
         ...
         // for enabling interrupts [0..3]
         mask = CSL_VLYNQ_INTVEC0_INT3 | CSL_VLYNQ_INTVEC0_INT2 |
                CSL_VLYNQ_INTVEC0_INT1 | CSL_VLYNQ_INTVEC0_INT0 ;
         CSL_vlynqHwControl(hVlynq, CSL_VLYNQ_CMD_INTVEC0_INT_ENA, &mask);
         ...
         ...
         // for disabling interrupts 1 and 3
         mask = CSL_VLYNQ_INTVEC0_INT3 | CSL_VLYNQ_INTVEC0_INT1;
         CSL_vlynqHwControl(hVlynq, CSL_VLYNQ_CMD_INTVEC0_INT_DIS, &mask);
         ...
   @endverbatim
 * ===========================================================================
 */
CSL_Status  CSL_vlynqHwControl(
    /* pointer to the object that holds reference to the
     * instance of VLYNQ requested after the call */
    CSL_VlynqHandle         hVlynq,
    
    /* the command to this API which indicates the action to be taken */
    CSL_VlynqHwControlCmd   cmd,
    
    /* an optional argument @a void* casted */
    void                    *arg
);


/** ===========================================================================
 *   @n@b CSL_vlynqGetHwStatus
 *
 *   @b Description
 *   @n This function is used to get the value of various parameters of the
 *      VLYNQ instance. The value returned depends on the query passed.
 *
 *   @b Arguments
 *   @verbatim
            hVlynq          Handle to the VLYNQ instance

            query           Query to be performed

            response        Pointer to buffer to return the data requested by
                            the query passed
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Successful completion of the
 *                                               query
 *
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   @li                    CSL_ESYS_INVQUERY  - Query command not supported
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_vlynqInit(), CSL_vlynqOpen() must be called successfully in that
 *       order before this API can be invoked
 *
 *   <b> Post Condition </b>
 *       Data requested by the query is returned through the variable "response"
 *
 *   @b Modifies
 *   @n  The input arguement "response" is modified
 *
 * @b Example:
 * @verbatim
      CSL_VlynqHandle hVlynq;
      CSL_Status status;
      Uint32 response = 0;
      ...
      // check for link establishment
      while (!response) {
          status = CSL_vlynqGetHwStatus(hVlynq,
                                        CSL_VLYNQ_QUERY_LINK,
                                        &response);
      }

      ...
      // check any pending requests at VBUSP slave interface
      CSL_vlynqGetHwStatus(hVlynq, CSL_VLYNQ_QUERY_SPEND, &response);
      if (response) {
          ...
      }
   @endverbatim
 * ============================================================================
 */
CSL_Status  CSL_vlynqGetHwStatus(
    /* pointer to the object that holds reference to the
     * instance of VLYNQ requested after the call */
    CSL_VlynqHandle         hVlynq,
    
    /* the query to this API which indicates the status/setup
     * to be returned */
    CSL_VlynqHwStatusQuery  query,

    /* placeholder to return the status; @a void* casted */
    void                    *response
);


/** ============================================================================
 *   @n@b CSL_vlynqHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the config data structure.  
 *
 *   @b Arguments
 *   @verbatim
            hVlynq          Handle to the VLYNQ instance
            
            pConfig         Pointer to the config structure containing the
                            device register values
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Configuration successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Configuration structure
 *                                                pointer is not properly
 *                                                initialized
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_vlynqInit(), CSL_vlynqOpen() must be called successfully in that
 *       order before this API can be invoked
 *
 *   <b> Post Condition </b>
 *   @n The registers of the specified VLYNQ instance will be setup 
 *      according to the values passed through the config structure.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified VLYNQ instance. 
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle       hVlynq;
        CSL_VlynqConfig       config = CSL_VLYNQ_CONFIG_DEFAULTS;
        CSL_Status            status;

        ...
        status = CSL_vlynqHwSetupRaw(hVlynq, &config);
        ...        
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_vlynqHwSetupRaw (
    CSL_VlynqHandle hVlynq,
    
    CSL_VlynqConfig *pHwConfig
);


/** @brief Returns the base address of the VLYNQ instance passed */
extern CSL_Status CSL_vlynqGetBaseAddress (
    CSL_InstNum             vlynqNum,
    CSL_VlynqParam*         pVlynqParam,
    CSL_VlynqBaseAddress*   pBaseAddress
);

#ifdef __cplusplus
}
#endif

#endif /* _CSL_VLYNQ_H_ */
