/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
 
/** ===========================================================================
 * @file  csl_vlynqAux.h
 *
 * @path  $(CSLPATH)\inc
 *
 * @desc API Auxilary header file for VLYNQ CSL.
 *       It gives the definitions of the status query & control functions.
 * ===========================================================================
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  29-Oct-2004 sd  File taken form TI codebase and modified for 5912.
 *
 *  13-Dec-2005 NG  - Removed in-line functions CSL_vlynqRevIdGet(), 
 *                    CSL_vlynqRevMajGet() and CSL_vlynqRevMinGet()
 *                  - Updated according to guidelines
 * =============================================================================
 */
#ifndef _CSL_VLYNQAUX_H_
#define _CSL_VLYNQAUX_H_

#include <csl_vlynq.h>

#ifdef __cplusplus
extern "C" {
#endif

/** Control commands */


/** ============================================================================
 *   @n@b CSL_vlynqPwrMgmtConfigure
 *
 *   @b Description
 *   @n Configure the value of the PMEN bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq            Handle to the VLYNQ instance

        pwrMgmtVal       value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The PMEM bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqPwrMgmtConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqPwrMgmtConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           pwrMgmtVal
)
{
    /* Configure the power Mangement bit */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_PMEN, pwrMgmtVal);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqSerialClkPullConfig
 *
 *   @b Description
 *   @n Configure the value of the SCLKXPUDIS bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq            Handle to the VLYNQ instance

        sclkPullUpVal       value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The SCLKXPUDIS bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqSerialClkPullConfig (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqSerialClkPullConfig (
    CSL_VlynqHandle  hVlynq,
    Uint32           sclkPullUpVal
)
{
    /* Configure the power Mangement bit */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_SCLKXPUDIS, sclkPullUpVal);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqSetRxSampleVal
 *
 *   @b Description
 *   @n Set the RTM sample value.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq         Handle to the VLYNQ instance

        rxSampleVal    RTM sample value to be configured.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RTM sample value is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqSetRxSampleVal (hVlynq, rxSampleVal );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqSetRxSampleVal (
    CSL_VlynqHandle          hVlynq,
    CSL_VlynqRtmSampleVal    rxSampleVal
)
{
    Uint32 temp;
    /* Set RTM sample value for simultaneous write to rxsampleval and rtmvaliwr
     * fields
     */
    temp = hVlynq->regs->CTRL;
    CSL_FINS(temp, VLYNQ_CTRL_RXSAMPELVAL, rxSampleVal);
    CSL_FINS(temp, VLYNQ_CTRL_RTMVALIDWR, 1);
    hVlynq->regs->CTRL = temp;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRtmDisable
 *
 *   @b Description
 *   @n Configure the value of the RTM Disable bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RTM Disable bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRtmDisable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRtmDisable (
    CSL_VlynqHandle  hVlynq
)
{
    /* Disable the RTM (Receive Timing Manager) */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_RTMENABLE,
                     CSL_VLYNQ_CTRL_RTMENABLE_DISABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRtmEnable
 *
 *   @b Description
 *   @n Configure the value of the RTM Enable bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RTM Enable bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRtmEnable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRtmEnable (
    CSL_VlynqHandle  hVlynq
)
{
    /* Disable the RTM (Receive Timing Manager) */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_RTMENABLE,
                     CSL_VLYNQ_CTRL_RTMENABLE_ENABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqTxSlowPathConfigure
 *
 *   @b Description
 *   @n Configure the value of the Transmit Fast Path bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Transmit Fast Path bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqTxSlowPathConfigure (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqTxSlowPathConfigure (
    CSL_VlynqHandle  hVlynq
)
{   
    /* Selects TX slow path */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_TXFASTPATH,
                    CSL_VLYNQ_CTRL_TXFASTPATH_DISABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqTxFastPathConfigure
 *
 *   @b Description
 *   @n Configure the value of the Transmit Fast Path bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Transmit Fast Path bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqTxFastPathConfigure (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqTxFastPathConfigure (
    CSL_VlynqHandle  hVlynq
)
{   
    /* Selects TX fast path */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_TXFASTPATH,
                    CSL_VLYNQ_CTRL_TXFASTPATH_ENABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqAddrOptDisable
 *
 *   @b Description
 *   @n Disable the Address Optimization bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Address Optimization bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqAddrOptDisable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqAddrOptDisable (
    CSL_VlynqHandle  hVlynq
)
{   
    /* Disable Address Optimization */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_AOPTDISABLE,
                     CSL_VLYNQ_CTRL_AOPTDISABLE_DISABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqAddrOptEnable
 *
 *   @b Description
 *   @n Enable the Address Optimization bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Address Optimization bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqAddrOptEnable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqAddrOptEnable (
    CSL_VlynqHandle  hVlynq
)
{   
    /* Enable Address Optimization */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_AOPTDISABLE,
                     CSL_VLYNQ_CTRL_AOPTDISABLE_ENABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqSoftResetAssert
 *
 *   @b Description
 *   @n Configure the value of the reset bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        reset        Value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The reset bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqSoftResetAssert (hVlynq, reset);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqSoftResetAssert (
    CSL_VlynqHandle  hVlynq,
    Uint32           reset
)
{
    /* Soft RESET:assert/Deassert soft reset */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_RESET, reset);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqPriIntrStatClear
 *
 *   @b Description
 *   @n Clear the interrupt status bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding interrupt status bits are cleared.
 *
 *   @b Modifies
 *   @n VLYNQ interrupt status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqPriIntrStatClear (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqPriIntrStatClear (
    CSL_VlynqHandle  hVlynq
)
{
    /* Clears the interrupt status (which has got highest priority)
     * pending from the INTSTATCLR register
     */
    hVlynq->regs->INTPRI |= CSL_FMK(VLYNQ_INTPRI_INSTAT,
                                    CSL_FEXT(hVlynq->regs->INTPRI,
                                             VLYNQ_INTPRI_INSTAT));
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntvec0Enable
 *
 *   @b Description
 *   @n Enables the INTVEC0 (3-0) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq         Handle to the VLYNQ instance

        intEnable      value for Enables the INTVEC0 (3-0) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the INTVEC0 (3-0) interrupt(s) Enables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntvec0Enable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntvec0Enable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intEnable
)
{
    /* Enable INTVEC0 (3-0) interrupt(s) */
    hVlynq->regs->INTVEC0 |= intEnable;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntvec0Disable
 *
 *   @b Description
 *   @n Disables the INTVEC0 (3-0) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        intDisable      value for Disables the INTVEC0 (3-0) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the INTVEC0 (3-0) interrupt(s) Disables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntvec0Disable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntvec0Disable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intDisable
)
{
    /* Disable INTVEC0 (3-0) interrupt(s) */
    hVlynq->regs->INTVEC0 &= ~(intDisable);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt3TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType3      Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt3TypeSet (hVlynq, intType3 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt3TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType3
)
{
    /* INTVEC0s' Interrupt-3 type */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE3, intType3);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt3PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol3      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt3PolSet (hVlynq, intPol3);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt3PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol3
)
{
    /* INTVEC0s' Interrupt-3 polarity */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL3,intPol3);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt3IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector3   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt3IntvecSet (hVlynq, intVector3);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt3IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector3
)
{
    /* INTVEC0's Interrupt-3 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC3, intVector3);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt2TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType2      Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt2TypeSet (hVlynq, intType2 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt2TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType2
)
{
    /* INTVEC0s' Interrupt-2 type */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE2, intType2);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt2PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol2      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt2PolSet (hVlynq, intPol2);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt2PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol2
)
{
    /* INTVEC0s' Interrupt-2 polarity */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL2,intPol2);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt2IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector2   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt2IntvecSet (hVlynq, intVector2);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt2IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector2
)
{
    /* INTVEC0's Interrupt-2 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC2, intVector2);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt1TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType1      Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt1TypeSet (hVlynq, intType1 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt1TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType1
)
{
    /* INTVEC0s' Interrupt-1 type */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE1, intType1);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt1PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol1      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt1PolSet (hVlynq, intPol1);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt1PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol1
)
{
    /* INTVEC0s' Interrupt-1 polarity */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL1,intPol1);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt1IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector1   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt1IntvecSet (hVlynq, intVector1);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt1IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector1
)
{
    /* INTVEC0's Interrupt-1 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC1, intVector1);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt0TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType0      Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt0TypeSet (hVlynq, intType0 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt0TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType0
)
{
    /* INTVEC0s' Interrupt-0 type */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE0, intType0);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt0PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol0      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt0PolSet (hVlynq, intPol0);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt0PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol0
)
{
    /* INTVEC0s' Interrupt-0 polarity */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL0,intPol0);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt0IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector0   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt0IntvecSet (hVlynq, intVector0);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt0IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector0
)
{
    /* INTVEC0's Interrupt-0 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC0, intVector0);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntvec1Enable
 *
 *   @b Description
 *   @n Enables the INTVEC1 (7-4) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq         Handle to the VLYNQ instance

        intEnable      value for Enables the INTVEC1 (7-4) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the INTVEC1 (7-4) interrupt(s) Enables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntvec1Enable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntvec1Enable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intEnable
)
{
    /* Enable INTVEC1 (7-4) interrupt(s) */
    hVlynq->regs->INTVEC1 |= intEnable;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntvec1Disable
 *
 *   @b Description
 *   @n Disables the INTVEC1 (7-4) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        intDisable      value for Disables the INTVEC1 (7-4) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the INTVEC1 (7-4) interrupt(s) Disables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntvec1Disable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntvec1Disable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intDisable
)
{
    /* Disable INTVEC1 (7-4) interrupt(s) */
    hVlynq->regs->INTVEC1 &= ~(intDisable);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt7TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType7     Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt7TypeSet (hVlynq, intType7 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt7TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType7
)
{
    /* INTVEC1s' Interrupt-7 type */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE7, intType7);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt7PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol7      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt7PolSet (hVlynq, intPol7);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt7PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol7
)
{
    /* INTVEC1s' Interrupt-7 polarity */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL7,intPol7);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt7IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector7   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt7IntvecSet (hVlynq, intVector7);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt7IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector7
)
{
    /* INTVEC1's Interrupt-7 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC7, intVector7);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt6TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType6     Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt6TypeSet (hVlynq, intType6 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt6TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType6
)
{
    /* INTVEC1s' Interrupt-6 type */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE6, intType6);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt6PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol6      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt6PolSet (hVlynq, intPol6);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt6PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol6
)
{
    /* INTVEC1s' Interrupt-6 polarity */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL6,intPol6);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt6IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector6   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt6IntvecSet (hVlynq, intVector6);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt6IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector6
)
{
    /* INTVEC1's Interrupt-6 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC6, intVector6);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt5TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType5     Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt5TypeSet (hVlynq, intType5 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt5TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType5
)
{
    /* INTVEC1s' Interrupt-5 type */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE5, intType5);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt5PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol5      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt5PolSet (hVlynq, intPol5);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt5PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol5
)
{
    /* INTVEC1s' Interrupt-5 polarity */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL5,intPol5);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt5IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector5   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt5IntvecSet (hVlynq, intVector5);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt5IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector5
)
{
    /* INTVEC1's Interrupt-5 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC5, intVector5);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt4TypeSet
 *
 *   @b Description
 *   @n Set the Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType4     Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt4TypeSet (hVlynq, intType4 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt4TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType4
)
{
    /* INTVEC1s' Interrupt-4 type */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE4, intType4);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt4PolSet
 *
 *   @b Description
 *   @n Set the Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol4      Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt4PolSet (hVlynq, intPol4);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt4PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol4
)
{
    /* INTVEC1s' Interrupt-4 polarity */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL4,intPol4);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqInt4IntvecSet
 *
 *   @b Description
 *   @n Set the Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector4   Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt4IntvecSet (hVlynq, intVector4);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt4IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector4
)
{
    /* INTVEC1's Interrupt-4 interrupt vector */
    CSL_FINS(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC4, intVector4);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqClkDivSet
 *
 *   @b Description
 *   @n Set the clock divisor.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        clkDiv       Clock divisor value to be configured.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The clock divisor is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqClkDivSet (hVlynq, CSL_VLYNQ_CLKDIV_1 );
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqClkDivSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqSclkDiv clkDiv
)
{
    /* Set the clock divisor */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_CLKDIV, clkDiv);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqClkDirSet
 *
 *   @b Description
 *   @n Set the clock direction.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        clkDir       Clock direction to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The clock direction is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqClkDirSet (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqClkDirSet (
    CSL_VlynqHandle  hVlynq,
    Uint32           clkDir
)
{
    /* Set the clock direction */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_CLKDIR, clkDir);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntLocalConfigure
 *
 *   @b Description
 *   @n Configure the value of the INTLOC bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq            Handle to the VLYNQ instance

        intLocalVal       value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The INTLOC bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntLocalConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqIntLocalConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           intLocalVal
)
{
    /* Configure the interrupt local bit */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_INTLOCAL, intLocalVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntEnConfigure
 *
 *   @b Description
 *   @n Configure the value of the INTENABLE bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intEnVal     value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The INTENABLE bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntEnConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqIntEnConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           intEnVal
)
{
    /* Configure the interrupt enable bit */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_INTENABLE, intEnVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVecConfigure
 *
 *   @b Description
 *   @n Configure the value of the INTVEC bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVecVal    value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The INTVEC bits is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntVecConfigure (hVlynq, CSL_VLYNQ_INT_VECTOR_2);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqIntVecConfigure (
    CSL_VlynqHandle        hVlynq,
    CSL_VlynqIntVector     intVecVal
)
{
    /* Configure the interrupt vector bits */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_INTVEC, intVecVal);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVecClear
 *
 *   @b Description
 *   @n Clears the corresponding bit in the interrupt pending register.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVecVal    value to be cleared
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Clears the corresponding bit in the interrupt pending register.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntVecClear (hVlynq, CSL_VLYNQ_INT_VECTOR_2);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqIntVecClear (
    CSL_VlynqHandle        hVlynq,
    CSL_VlynqIntVector     intVecVal
)
{
    /* Clear the interrupt vector bits */
    hVlynq->regs->CTRL &= ~(CSL_FMK(VLYNQ_CTRL_INTVEC, intVecVal));
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqInt2CfgConfigure
 *
 *   @b Description
 *   @n Configure the value of the INT2CFG bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        int2Cfg      Value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The INT2CFG bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt2CfgConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqInt2CfgConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           int2CfgVal
)
{
    /* Configure the INT2CFG bit */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_INT2CFG, int2CfgVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqILoopConfigure
 *
 *   @b Description
 *   @n Configure the value of the internal loopback bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        iLoopVal     Value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The ILOOP bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqILoopConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqILoopConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           iLoopVal
)
{
    /* Configure internal loopback */
    CSL_FINS(hVlynq->regs->CTRL, VLYNQ_CTRL_ILOOP, iLoopVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRErrorClear
 *
 *   @b Description
 *   @n Clear the remote error status bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote error status bit is cleared.
 *
 *   @b Modifies
 *   @n VLYNQ status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRErrorClear (hVlynq);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRErrorClear (
    CSL_VlynqHandle  hVlynq
)
{
    /* Clear the remote error status bit */
    CSL_FINS(hVlynq->regs->STAT, VLYNQ_STAT_RERROR,
             CSL_VLYNQ_STAT_RERROR_CLEAR);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqLErrorClear
 *
 *   @b Description
 *   @n Clear the local error status bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The local error status bit is cleared.
 *
 *   @b Modifies
 *   @n VLYNQ status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqLErrorClear (hVlynq);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqLErrorClear (
    CSL_VlynqHandle  hVlynq
)
{
    /* Clear the local error status bit */
    CSL_FINS(hVlynq->regs->STAT, VLYNQ_STAT_LERROR,
             CSL_VLYNQ_STAT_LERROR_CLEAR);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntrStatClear
 *
 *   @b Description
 *   @n Clear the interrupt status bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        intrStatClrVal  interrupt status bits to be cleared.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding interrupt status bits are cleared.
 *
 *   @b Modifies
 *   @n VLYNQ interrupt status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntrStatClear (hVlynq, 0xFF);
        ...
     @endverbatim
 *  ===========================================================================
 */

static inline
void CSL_vlynqIntrStatClear (
    CSL_VlynqHandle  hVlynq,
    Uint32           intrStatClrVal
)
{
    /* Clear the interrupt status register bits */
    CSL_FINS(hVlynq->regs->INTSTATCLR, VLYNQ_INTSTATCLR_INTCLR,
                                        intrStatClrVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntrPendSet
 *
 *   @b Description
 *   @n Set the interrupt pending bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intrPend     interrupt pend bits to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding interrupt pending bits are set.
 *
 *   @b Modifies
 *   @n VLYNQ interrupt pending register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntrPendSet (hVlynq, 0xFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqIntrPendSet (
    CSL_VlynqHandle  hVlynq,
    Uint32           intrPend
)
{
    /* Set the interrupt pend/set register bits */
    hVlynq->regs->INTPENDSET |= *(Uint32*)intrPend;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntrPendSetClr
 *
 *   @b Description
 *   @n Clear the interrupt pending bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intrPend     interrupt pend bits to be cleared.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding interrupt pending bits are cleared.
 *
 *   @b Modifies
 *   @n VLYNQ interrupt pending register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntrPendSetClr (hVlynq, 0xFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqIntrPendSetClr (
    CSL_VlynqHandle  hVlynq,
    Uint32           intrPendClr
)
{
    /* Clear the interrupt pend/set register bits */
    hVlynq->regs->INTPENDSET &= ~(*(Uint32*)intrPendClr);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqIntrPtrSet
 *
 *   @b Description
 *   @n Set the interrupt pointer register to the address passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intrPtr     interrupt address pointer.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The interrupt pointer is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ interrupt pointer register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqIntrPtrSet (hVlynq, 0xFF0000FF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqIntrPtrSet (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    intrPtr
)
{
    /* Set the interrupt pointer register */
    hVlynq->regs->INTPTR = (Uint32)intrPtr;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqTxAddrMapSet
 *
 *   @b Description
 *   @n Set the transmit address map to the address passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrMap     TX address map pointer.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The TXADDRMAP is set to the address passed.
 *
 *   @b Modifies
 *   @n VLYNQ transmit address map register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqTxAddrMapSet (hVlynq, 0xFF0000FF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqTxAddrMapSet (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrMap
)
{
    /* Set the TX address map */
    hVlynq->regs->XAM = (Uint32)((CSL_VlynqAdrPtr)addrMap);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrSize1Set
 *
 *   @b Description
 *   @n Set the receive address map size 1 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 1 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADRSIZE1 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map size 1 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrSize1Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrSize1Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the RX address map size */
    hVlynq->regs->RAMS1 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrOffset1Set
 *
 *   @b Description
 *   @n Set the receive address map offset to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADROFFSET1 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrOffset1Set (hVlynq, 0xFF0000FF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrOffset1Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset1 register */
    hVlynq->regs->RAMO1 = (Uint32)addrOffset;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrSize2Set
 *
 *   @b Description
 *   @n Set the receive address map size 2 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 2 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADRSIZE2 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map size 2 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrSize2Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrSize2Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the RX address map size */
    hVlynq->regs->RAMS2 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrOffset2Set
 *
 *   @b Description
 *   @n Set the receive address map offset 2 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map 2 offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADROFFSET2 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrOffset2Set (hVlynq, 0x00FFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrOffset2Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset2 register */
    hVlynq->regs->RAMO2 = (Uint32)addrOffset;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrSize3Set
 *
 *   @b Description
 *   @n Set the receive address map size 3 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 3 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADRSIZE3 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map size 2 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrSize3Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrSize3Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the RX address map size */
    hVlynq->regs->RAMS3 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrOffset3Set
 *
 *   @b Description
 *   @n Set the receive address map offset 3 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map 3 offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADROFFSET3 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrOffset3Set (hVlynq, 0x00FFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrOffset3Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset3 register */
    hVlynq->regs->RAMO3 = (Uint32)addrOffset;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrSize4Set
 *
 *   @b Description
 *   @n Set the receive address map size 4 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 4 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADRSIZE3 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map size 4 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrSize4Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrSize4Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the RX address map size */
    hVlynq->regs->RAMS4 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRxAddrOffset4Set
 *
 *   @b Description
 *   @n Set the receive address map offset 4 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map 4 offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RXADROFFSET4 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRxAddrOffset4Set (hVlynq, 0x00FFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRxAddrOffset4Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset4 register */
    hVlynq->regs->RAMO4 = (Uint32)addrOffset;

    return;
}


/*  remote control commands */

/** ============================================================================
 *   @n@b CSL_vlynqRemPmenDisable
 *
 *   @b Description
 *   @n Disable the Power Management Control bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Power Management Control bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemPmenDisable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemPmenDisable (
    CSL_VlynqHandle  hVlynq
)
{
    /* Configure the Power Management disable bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_PMEN, 
             CSL_VLYNQ_RCTRL_PMEN_DISABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemPmenEnable
 *
 *   @b Description
 *   @n Enable the Power Management Control bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Power Management Control bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemPmenEnable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemPmenEnable (
    CSL_VlynqHandle  hVlynq
)
{
    /* Configure the Power Management enable bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_PMEN,
             CSL_VLYNQ_RCTRL_PMEN_ENABLE);
                     
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemSclkpudisEnable
 *
 *   @b Description
 *   @n Enable the Serial Clock Pull-up.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        sclkpudis    value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Serial Clock Pull-up Control bit is configured with the value
 *       passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemSclkpudisEnable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemSclkpudisEnable (
    CSL_VlynqHandle  hVlynq
)
{
    /* Configure the remote Serial Clock Pull-up enable/disable bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_SCLKXPUDIS,
             CSL_VLYNQ_RCTRL_SCLKXPUDIS_ENABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemSclkpudisDisable
 *
 *   @b Description
 *   @n Disable the Serial Clock Pull-up.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        sclkpudis    value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Serial Clock Pull-up Control bit is configured with the value
 *       passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemSclkpudisDisable (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemSclkpudisDisable (
    CSL_VlynqHandle  hVlynq
)
{
    /* Configure the remote Serial Clock Pull-up enable/disable bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_SCLKXPUDIS,
             CSL_VLYNQ_RCTRL_SCLKXPUDIS_DISABLE);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemSetRxSampleVal
 *
 *   @b Description
 *   @n Set the RTM sample value.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq         Handle to the VLYNQ instance

        rxSampleVal    RTM sample value to be configured.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RTM sample value is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle         hVlynq;
        CSL_VlynqRtmSampleVal   rxSampleVal;
         ...
        CSL_vlynqRemSetRxSampleVal (hVlynq, rxSampleVal );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemSetRxSampleVal (
    CSL_VlynqHandle          hVlynq,
    CSL_VlynqRtmSampleVal    rxSampleVal
)
{
    Uint32 temp;
    /* Set remote RTM sample value for simultaneous write to rxsampleval and
     * rtmvaliwr fields
     */
    temp = hVlynq->regs->RCTRL;
    CSL_FINS(temp, VLYNQ_RCTRL_RXSAMPELVAL, rxSampleVal);
    CSL_FINS(temp, VLYNQ_RCTRL_RTMVALIDWR, 1);
    hVlynq->regs->RCTRL = temp;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRtmConfigure
 *
 *   @b Description
 *   @n Configure the value of the RTM Enable/Disable bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        rtmEnval     value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The RTM Enable/Disable bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
        Uint32              rtmEnval = 0x1;
         ...
        CSL_vlynqRemRtmConfigure (hVlynq, rtmEnval);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRtmConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           rtmEnval
)
{
    /* setup the remote Enable/Disable bit of  RTM (Receive Timing Manager) */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_RTMENABLE, rtmEnval);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemTxFastPathConfigure
 *
 *   @b Description
 *   @n Configure the value of the Transmit Fast Path bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        txFastPath   value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Transmit Fast Path bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
        Uint32              txFastPath = 0x1;
         ...
        CSL_vlynqRemTxFastPathConfigure (hVlynq, txFastPath);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemTxFastPathConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           txFastPath
)
{
    /* Selects remote TX fast path */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_TXFASTPATH, txFastPath);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemClkDivSet
 *
 *   @b Description
 *   @n Set the remote clock divisor.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        clkDiv       Clock divisor value to be configured.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote clock divisor is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemClkDivSet (hVlynq, CSL_VLYNQ_CLKDIV_1 );
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemClkDivSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqSclkDiv    clkDiv
)
{
    /* Set the remote clock divisor */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_CLKDIV, clkDiv);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemAoptDisableConfigure
 *
 *   @b Description
 *   @n Configure the value of the remote internal loopback bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        aOptDisable  Value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote Address optimization disable bit is configured with
 *       the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemAoptDisableConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemAoptDisableConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           aOptDisable
)
{
    /* Disable remote VLYNQs' Address Optimization */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_AOPTDISABLE, aOptDisable);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemClkDirSet
 *
 *   @b Description
 *   @n Set the remote clock direction.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        clkDir       Clock direction to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote clock direction is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemClkDirSet (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemClkDirSet (
    CSL_VlynqHandle  hVlynq,
    Uint32           clkDir
)
{
    /* Set the remote clock direction */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_CLKDIR, clkDir);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntLocalConfigure
 *
 *   @b Description
 *   @n Configure the value of the remote INTLOC bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq            Handle to the VLYNQ instance

        intLocalVal       value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote INTLOC bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntLocalConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntLocalConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           intLocalVal
)
{
    /* Configure the remote interrupt local bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_INTLOCAL, intLocalVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntEnConfigure
 *
 *   @b Description
 *   @n Configure the value of the remote INTENABLE bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intEnVal     value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote INTENABLE bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntEnConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntEnConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           intEnVal
)
{
    /* Configure the remote interrupt enable bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_INTENABLE, intEnVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVecConfigure
 *
 *   @b Description
 *   @n Configure the value of the remote INTVEC bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVecVal    value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote INTVEC bits is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntVecConfigure (hVlynq, CSL_VLYNQ_INT_VECTOR_2);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntVecConfigure (
    CSL_VlynqHandle        hVlynq,
    CSL_VlynqIntVector     intVecVal
)
{
    /* Configure the interrupt vector bits */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_INTVEC, intVecVal);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVecClear
 *
 *   @b Description
 *   @n Clears the corresponding bit in the remote interrupt pending register.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVecVal    value to be cleared
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Clears the corresponding bit in the remote interrupt pending register.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntVecClear (hVlynq, CSL_VLYNQ_INT_VECTOR_2);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntVecClear (
    CSL_VlynqHandle        hVlynq,
    CSL_VlynqIntVector     intVecVal
)
{
    /* Clear the remote interrupt vector bits */
    hVlynq->regs->RCTRL &= ~(CSL_FMK(VLYNQ_RCTRL_INTVEC, intVecVal));
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt2CfgConfigure
 *
 *   @b Description
 *   @n Configure the value of the remote INT2CFG bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        int2Cfg      Value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote INT2CFG bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt2CfgConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemInt2CfgConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           int2CfgVal
)
{
    /* Configure the remote INT2CFG bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_INT2CFG, int2CfgVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemILoopConfigure
 *
 *   @b Description
 *   @n Configure the value of the remote internal loopback bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        iLoopVal     Value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote ILOOP bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemILoopConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemILoopConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           iLoopVal
)
{
    /* Configure internal loopback */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_ILOOP, iLoopVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemSoftResetConfigure
 *
 *   @b Description
 *   @n Configure the value of the remote reset bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        assertVal    Value to be configured
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RESET bit is configured with the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote control register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemSoftResetConfigure (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemSoftResetConfigure (
    CSL_VlynqHandle  hVlynq,
    Uint32           assertVal
)
{
    /* configures the Remote VLYNQ's Soft RESET bit */
    CSL_FINS(hVlynq->regs->RCTRL, VLYNQ_RCTRL_RESET, assertVal);
    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRErrorClear
 *
 *   @b Description
 *   @n Clear the remote error status bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote error status bit is cleared.
 *
 *   @b Modifies
 *   @n VLYNQ remote status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRErrorClear (hVlynq);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRErrorClear (
    CSL_VlynqHandle  hVlynq
)
{
    /* Clear the remote error status bit */
    CSL_FINS(hVlynq->regs->RSTAT, VLYNQ_RSTAT_RERROR,
             CSL_VLYNQ_RSTAT_RERROR_CLEAR);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemLErrorClear
 *
 *   @b Description
 *   @n Clear the local error status bit.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The local error status bit is cleared.
 *
 *   @b Modifies
 *   @n VLYNQ remote status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemLErrorClear (hVlynq);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemLErrorClear (
    CSL_VlynqHandle  hVlynq
)
{
    /* Clear the local error status bit */
    CSL_FINS(hVlynq->regs->RSTAT, VLYNQ_RSTAT_LERROR,
             CSL_VLYNQ_RSTAT_LERROR_CLEAR);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntPriStatClear
 *
 *   @b Description
 *   @n Clear the remote interrupt status bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        intrStatClrVal  interrupt status bits to be cleared.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding interrupt status bits are cleared.
 *
 *   @b Modifies
 *   @n VLYNQ remote interrupt status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntPriStatClear (hVlynq);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntPriStatClear (
    CSL_VlynqHandle  hVlynq
)
{
    /* Clear remote VLYNQ's interrupt status
     * (which has got highest priority) pending from the
     * RINTSTATCLR register
     */
     hVlynq->regs->RINTPRI |= CSL_FMK(VLYNQ_RINTPRI_INSTAT,
                                      CSL_FEXT(hVlynq->regs->RINTPRI,
                                               VLYNQ_RINTPRI_INSTAT));

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntrStatClear
 *
 *   @b Description
 *   @n Clear the remote interrupt status bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        intrStatClrVal  interrupt status bits to be cleared.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding interrupt status bits are cleared.
 *
 *   @b Modifies
 *   @n VLYNQ remote interrupt status register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntrStatClear (hVlynq, 0xFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntrStatClear (
    CSL_VlynqHandle  hVlynq,
    Uint32           intrStatClrVal
)
{
    /* Clear the remote interrupt status register bits */
    CSL_FINS(hVlynq->regs->RINTSTATCLR, VLYNQ_RINTSTATCLR_INTCLR,
                                        intrStatClrVal);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRIntrPendSet
 *
 *   @b Description
 *   @n Set the remote interrupt pending bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intrPend     interrupt pend bits to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding remote interrupt pending bits are set.
 *
 *   @b Modifies
 *   @n VLYNQ remote interrupt pending register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntrPendSet (hVlynq, 0xFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntrPendSet (
    CSL_VlynqHandle  hVlynq,
    Uint32           intrPend
)
{
    /* Set the interrupt pend/set register bits */
    hVlynq->regs->RINTPENDSET |= *(Uint32*)intrPend;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntrPendSetClr
 *
 *   @b Description
 *   @n Clear the remote interrupt pending bits.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intrPend     interrupt pend bits to be cleared.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The corresponding remote interrupt pending bits are cleared.
 *
 *   @b Modifies
 *   @n VLYNQ remote interrupt pending register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntrPendSetClr (hVlynq, 0xFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntrPendSetClr (
    CSL_VlynqHandle  hVlynq,
    Uint32           intrPendClr
)
{
    /* Clear the remote interrupt pend/set register bits */
    hVlynq->regs->RINTPENDSET &= ~(*(Uint32*)intrPendClr);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntrPtrSet
 *
 *   @b Description
 *   @n Set the remote interrupt pointer register to the address passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intrPtr      interrupt address pointer.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote interrupt pointer is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote interrupt pointer register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntrPtrSet (hVlynq, 0xFF0000FF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemIntrPtrSet (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    intrPtr
)
{
    /* Set the interrupt pointer register */
    hVlynq->regs->RINTPTR = (Uint32)intrPtr;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemTxAddrMapSet
 *
 *   @b Description
 *   @n Set the remote transmit address map to the address passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrMap     TX address map pointer.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote TXADDRMAP is set to the address passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote transmit address map register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemTxAddrMapSet (hVlynq, 0xFF0000FF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemTxAddrMapSet (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrMap
)
{
    /* Set the TX address map */
    hVlynq->regs->RXAM = (Uint32)((CSL_VlynqAdrPtr)addrMap);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrSize1Set
 *
 *   @b Description
 *   @n Set the remote receive address map size 1 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 1 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADRSIZE1 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map size 1 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrSize1Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrSize1Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the remote RX address map size */
    hVlynq->regs->RRAMS1 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrOffset1Set
 *
 *   @b Description
 *   @n Set the remote receive address map offset to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADROFFSET1 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrOffset1Set (hVlynq, 0xFF0000FF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrOffset1Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset1 register */
    hVlynq->regs->RRAMO1 = (Uint32)addrOffset;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrSize2Set
 *
 *   @b Description
 *   @n Set the remote receive address map size 2 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 2 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADRSIZE2 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map size 2 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrSize2Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrSize2Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the RX address map size */
    hVlynq->regs->RRAMS2 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrOffset2Set
 *
 *   @b Description
 *   @n Set the remote receive address map offset 2 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map 2 offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADROFFSET2 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrOffset2Set (hVlynq, 0x00FFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrOffset2Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset2 register */
    hVlynq->regs->RRAMO2 = (Uint32)addrOffset;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrSize3Set
 *
 *   @b Description
 *   @n Set the remote receive address map size 3 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 3 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADRSIZE3 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map size 2 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrSize3Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrSize3Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the RX address map size */
    hVlynq->regs->RRAMS3 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrOffset3Set
 *
 *   @b Description
 *   @n Set the remote receive address map offset 3 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map 3 offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADROFFSET3 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrOffset3Set (hVlynq, 0x00FFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrOffset3Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset3 register */
    hVlynq->regs->RRAMO3 = (Uint32)addrOffset;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrSize4Set
 *
 *   @b Description
 *   @n Set the remote receive address map size 4 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        addrMapSize     RX address map 4 size.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADRSIZE3 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map size 4 register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrSize4Set (hVlynq, 0xFFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrSize4Set (
    CSL_VlynqHandle    hVlynq,
    Uint32             addrMapSize
)
{
    /* Set the RX address map size */
    hVlynq->regs->RRAMS4 = addrMapSize;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxAddrOffset4Set
 *
 *   @b Description
 *   @n Set the remote receive address map offset 4 to the value passed.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq      Handle to the VLYNQ instance

        addrOffset  RX address map 4 offset.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The remote RXADROFFSET4 is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ remote receive address map offset register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemRxAddrOffset4Set (hVlynq, 0x00FFFF);
        ...
     @endverbatim
 *  ============================================================================
 */

static inline
void CSL_vlynqRemRxAddrOffset4Set (
    CSL_VlynqHandle    hVlynq,
    CSL_VlynqAdrPtr    addrOffset
)
{
    /* Set the RX address offset4 register */
    hVlynq->regs->RRAMO4 = (Uint32)addrOffset;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntvec0Enable
 *
 *   @b Description
 *   @n Enables the RINTVEC0 (3-0) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq         Handle to the VLYNQ instance

        intEnable      value for Enables the INTVEC0 (3-0) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the RINTVEC0 (3-0) interrupt(s) Enables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntvec0Enable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntvec0Enable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intEnable
)
{
    /* Enable RINTVEC0 (3-0) interrupt(s) */
    hVlynq->regs->RINTVEC0 |= intEnable;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntvec0Disable
 *
 *   @b Description
 *   @n Disables the RINTVEC0 (3-0) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        intDisable      value for Disables the RINTVEC0 (3-0) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the INTVEC0 (3-0) interrupt(s) Disables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntvec0Disable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntvec0Disable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intDisable
)
{
    /* Disable RINTVEC0 (3-0) interrupt(s) */
    hVlynq->regs->RINTVEC0 &= ~(intDisable);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt3TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType3      Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt3TypeSet (hVlynq, intType3 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt3TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType3
)
{
    /* RINTVEC0s' Interrupt-3 type */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE3, intType3);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt3PolSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol3      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt3PolSet (hVlynq, intPol3);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt3PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol3
)
{
    /* RINTVEC0s' Interrupt-3 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL3,intPol3);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt3IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector3   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt3IntvecSet (hVlynq, intVector3);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt3IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector3
)
{
    /* RINTVEC0's Interrupt-3 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC3, intVector3);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt2TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType2     Remote Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt2TypeSet (hVlynq, intType2 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt2TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType2
)
{
    /* RINTVEC0s' Interrupt-2 type */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE2, intType2);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt2PolSet
 *
 *   @b Description
 *   @n Set the Remove Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol2      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqInt2PolSet (hVlynq, intPol2);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt2PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol2
)
{
    /* RINTVEC0s' Interrupt-2 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL2,intPol2);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt2IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector2   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt2IntvecSet (hVlynq, intVector2);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt2IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector2
)
{
    /* RINTVEC0's Interrupt-2 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC2, intVector2);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt1TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType1      Remote Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt1TypeSet (hVlynq, intType1 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt1TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType1
)
{
    /* RINTVEC0s' Interrupt-1 type */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE1, intType1);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt1PolSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol1      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt1PolSet (hVlynq, intPol1);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt1PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol1
)
{
    /* RINTVEC0s' Interrupt-1 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL1,intPol1);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt1IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector1   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt1IntvecSet (hVlynq, intVector1);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt1IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector1
)
{
    /* RINTVEC0's Interrupt-1 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC1, intVector1);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt0TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType0     Remote Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt0TypeSet (hVlynq, intType0 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt0TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType0
)
{
    /* RINTVEC0s' Interrupt-0 type */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE0, intType0);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt0PolSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol0      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt0PolSet (hVlynq, intPol0);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt0PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol0
)
{
    /* RINTVEC0s' Interrupt-0 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL0,intPol0);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt0IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector0   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 3-0 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt0IntvecSet (hVlynq, intVector0);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt0IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector0
)
{
    /* RINTVEC0's Interrupt-0 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC0, intVector0);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntvec1Enable
 *
 *   @b Description
 *   @n Enables the RINTVEC1 (7-4) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq         Handle to the VLYNQ instance

        intEnable      value for Enables the RINTVEC1 (7-4) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the RINTVEC1 (7-4) interrupt(s) Enables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntvec1Enable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntvec1Enable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intEnable
)
{
    /* Enable INTVEC1 (7-4) interrupt(s) */
    hVlynq->regs->RINTVEC1 |= intEnable;

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntvec1Disable
 *
 *   @b Description
 *   @n Disables the RINTVEC1 (7-4) interrupt(s).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq          Handle to the VLYNQ instance

        intDisable      value for Disables the RINTVEC1 (7-4) interrupt(s).
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  the RINTVEC1 (7-4) interrupt(s) Disables value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemIntvec1Disable (hVlynq, 0x01);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntvec1Disable (
    CSL_VlynqHandle  hVlynq,
    Uint32           intDisable
)
{
    /* Disable RINTVEC1 (7-4) interrupt(s) */
    hVlynq->regs->RINTVEC1 &= ~(intDisable);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt7TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType7     Remote Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt7TypeSet (hVlynq, intType7 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt7TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType7
)
{
    /* RemINTVEC1s' Interrupt-7 type */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE7, intType7);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt7PolSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol7      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt7PolSet (hVlynq, intPol7);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt7PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol7
)
{
    /* RINTVEC1s' Interrupt-7 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL7,intPol7);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt7IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector7   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt7IntvecSet (hVlynq, intVector7);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt7IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector7
)
{
    /* RINTVEC1's Interrupt-7 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC7, intVector7);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt6TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType6     Remote Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt6TypeSet (hVlynq, intType6 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt6TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType6
)
{
    /* RINTVEC1s' Interrupt-6 type */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE6, intType6);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt6PolSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol6      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt6PolSet (hVlynq, intPol6);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt6PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol6
)
{
    /* RINTVEC1s' Interrupt-6 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL6,intPol6);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt6IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector6   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt6IntvecSet (hVlynq, intVector6);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt6IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector6
)
{
    /* RINTVEC1's Interrupt-6 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC6, intVector6);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt5TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType5     Remote Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt5TypeSet (hVlynq, intType5 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt5TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType5
)
{
    /* RINTVEC1s' Interrupt-5 type */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE5, intType5);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt5PolSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol5      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt5PolSet (hVlynq, intPol5);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt5PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol5
)
{
    /* RINTVEC1s' Interrupt-5 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL5,intPol5);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt5IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector5   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt5IntvecSet (hVlynq, intVector5);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt5IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector5
)
{
    /* RINTVEC1's Interrupt-5 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC5, intVector5);

    return;
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt4TypeSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt type.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intType4     Remote Interrupt type to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt type is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt4TypeSet (hVlynq, intType4 );
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt4TypeSet (
    CSL_VlynqHandle  hVlynq,
    CSL_VlynqIntType intType4
)
{
    /* RINTVEC1s' Interrupt-4 type */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE4, intType4);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt4PolSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt polarity.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intPol4      Remote Interrupt polarity to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt polarity is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt4PolSet (hVlynq, intPol4);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt4PolSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntPolType intPol4
)
{
    /* RINTVEC1s' Interrupt-4 polarity */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL4,intPol4);

    return;
}


/** ============================================================================
 *   @n@b CSL_vlynqRemInt4IntvecSet
 *
 *   @b Description
 *   @n Set the Remote Interrupt vector.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq       Handle to the VLYNQ instance

        intVector4   Remote Interrupt vector to be set.
     @endverbatim
 *
 *   <b> Return Value </b>
 *       None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The Remote Interrupt vector is set to the value passed.
 *
 *   @b Modifies
 *   @n VLYNQ Remote Interrupt Vector 7-4 Register.
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle     hVlynq;
         ...
        CSL_vlynqRemInt4IntvecSet (hVlynq, intVector4);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt4IntvecSet (
    CSL_VlynqHandle     hVlynq,
    CSL_VlynqIntVector  intVector4
)
{
    /* RINTVEC1's Interrupt-4 interrupt vector */
    CSL_FINS(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC4, intVector4);

    return;
}

/**
 *  Status query functions of the VLYNQ port
 */

/** ============================================================================
 *   @n@b CSL_vlynqPowMgmtGet
 *
 *   @b Description
 *   @n Gets the Power Management iformation of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        ...
        CSL_vlynqPowMgmtGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqPowMgmtGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Power Management info */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->CTRL, VLYNQ_CTRL_PMEN);
}

/** ============================================================================
 *   @n@b CSL_vlynqRxSampleValGet
 *
 *   @b Description
 *   @n Gets the rxsampleval of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        ...
        CSL_vlynqRxSampleValGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRxSampleValGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* get value from rxsampleval field */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->CTRL, VLYNQ_CTRL_RXSAMPELVAL);
}

/** ============================================================================
 *   @n@b CSL_vlynqRtmEnableGet
 *
 *   @b Description
 *   @n Gets the RTM Logic iformation of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRtmEnableGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRtmEnableGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RTM Logic info */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->CTRL, VLYNQ_CTRL_RTMENABLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqTxFastPathGet
 *
 *   @b Description
 *   @n Gets the Transmit Path info of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        ...
        CSL_vlynqTxFastPathGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqTxFastPathGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Transmit Path */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->CTRL, VLYNQ_CTRL_TXFASTPATH);
}

/** ============================================================================
 *   @n@b CSL_vlynqClkDirGet
 *
 *   @b Description
 *   @n Gets the serial clock direction.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqClkDirGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqClkDirGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the clock direction of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->CTRL, VLYNQ_CTRL_CLKDIR);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntLocGet
 *
 *   @b Description
 *   @n Gets the interrupt postage info.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               intLoc;

        ...
        CSL_vlynqIntLocGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntLocGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the interrupt postage info of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->CTRL, VLYNQ_CTRL_INTLOCAL);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntEnableStatGet
 *
 *   @b Description
 *   @n This function gets if the interrupts are enabled/disabled.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               intEnableStatus;

        ...
        CSL_vlynqIntEnableStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntEnableStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the interrupt enable status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->CTRL, VLYNQ_CTRL_INTENABLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVecStatGet
 *
 *   @b Description
 *   @n Gets the interrupt vector status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        ...
        CSL_vlynqIntVecStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVecStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the interrupt vector status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->CTRL, VLYNQ_CTRL_INTVEC);
}

/** ============================================================================
 *   @n@b CSL_vlynqInt2CfgStatGet
 *
 *   @b Description
 *   @n Gets the status of the int2cfg bit of the VLYNQ control register.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqInt2CfgStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqInt2CfgStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the INT2CFG bit status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->CTRL, VLYNQ_CTRL_INT2CFG);
}

/** ============================================================================
 *   @n@b CSL_vlynqAoptDisableGet
 *
 *   @b Description
 *   @n Gets the internal loopback bit status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqAoptDisableGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqAoptDisableGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Address Optimization */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->CTRL, VLYNQ_CTRL_AOPTDISABLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqILoopStatGet
 *
 *   @b Description
 *   @n Gets the internal loopback bit status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqILoopStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqILoopStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the internal loopback bit status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->CTRL, VLYNQ_CTRL_ILOOP);
}

/** ============================================================================
 *   @n@b CSL_vlynqSerWidthInGet
 *
 *   @b Description
 *   @n Gets the serial data width.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqSerWidthInGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqSerWidthInGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Size/Width of inbound serial data */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->STAT, VLYNQ_STAT_SWIDTHIN);
}

/** ============================================================================
 *   @n@b CSL_vlynqSerWidthOutGet
 *
 *   @b Description
 *   @n Gets the serial data width.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqSerWidthOutGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqSerWidthOutGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Size/Width of inbound serial data */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->STAT, VLYNQ_STAT_SWIDTHOUT);
}

/** ============================================================================
 *   @n@b CSL_vlynqRxCurrentSampleGet
 *
 *   @b Description
 *   @n Gets the Current clock sample value used by RTM.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRxCurrentSampleGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRxCurrentSampleGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Current clock sample value used by RTM */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->STAT, VLYNQ_STAT_RXCURRENTSAMPLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqRtmGet
 *
 *   @b Description
 *   @n Gets the RTM logic inclusion in RTL.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRtmGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRtmGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RTM logic inclusion in RTL */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->STAT, VLYNQ_STAT_RTM);
}

/** ============================================================================
 *   @n@b CSL_vlynqIFlowStatGet
 *
 *   @b Description
 *   @n Gets the inbound flow control status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIFlowStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIFlowStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the inbound flow control status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_IFLOW);
}

/** ============================================================================
 *   @n@b CSL_vlynqOFlowStatGet
 *
 *   @b Description
 *   @n Gets the outbound flow control status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqOFlowStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqOFlowStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the outbound flow control status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_OFLOW);
}

/** ============================================================================
 *   @n@b CSL_vlynqRErrorStatGet
 *
 *   @b Description
 *   @n Gets the remote packet error status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRErrorStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRErrorStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the remote packet error status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_RERROR);
}

/** ============================================================================
 *   @n@b CSL_vlynqLErrorStatGet
 *
 *   @b Description
 *   @n Gets the local packet error status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqLErrorStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqLErrorStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the local packet error status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_LERROR);
}

/** ============================================================================
 *   @n@b CSL_vlynqNFEmpty3StatGet
 *
 *   @b Description
 *   @n Gets the fifo 3 status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqNFEmpty3StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqNFEmpty3StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 3 status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_NFEMPTY3);
}


/** ============================================================================
 *   @n@b CSL_vlynqNFEmpty2StatGet
 *
 *   @b Description
 *   @n Gets the fifo 2 status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqNFEmpty2StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqNFEmpty2StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 2 status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_NFEMPTY2);
}


/** ============================================================================
 *   @n@b CSL_vlynqNFEmpty1StatGet
 *
 *   @b Description
 *   @n Gets the fifo 1 status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqNFEmpty1StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqNFEmpty1StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 1 status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_NFEMPTY1);
}


/** ============================================================================
 *   @n@b CSL_vlynqNFEmpty0StatGet
 *
 *   @b Description
 *   @n Gets the fifo 0 status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqNFEmpty0StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqNFEmpty0StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 0 status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_NFEMPTY0);
}

/** ============================================================================
 *   @n@b CSL_vlynqSPendStatGet
 *
 *   @b Description
 *   @n Gets the pending requests status at slave interface.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqSPendStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqSPendStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of pending requests at the slave interface */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_SPEND);
}


/** ============================================================================
 *   @n@b CSL_vlynqMPendStatGet
 *
 *   @b Description
 *   @n Gets the status of the pending requests at master interface.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqMPendStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqMPendStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of pending requests at the master interface */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_MPEND);
}

/** ============================================================================
 *   @n@b CSL_vlynqLinkStatGet
 *
 *   @b Description
 *   @n Gets the status of the serial link establishment.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqLinkStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqLinkStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of the serial link establishment */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->STAT, VLYNQ_STAT_LINK);
}

/** ============================================================================
 *   @n@b CSL_vlynqNoIntPendGet
 *
 *   @b Description
 *   @n Gets the status of the pending requests at master interface.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqNoIntPendGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqNoIntPendGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Status of any pending interrupts from
     * INTSTATCLR register
     */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->INTPRI, VLYNQ_INTPRI_NOINTPEND);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntPriStatGet
 *
 *   @b Description
 *   @n Gets the Status of the highest priority interrupt pending from
 *      INTSTATCLR register
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntPriStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntPriStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Status of the highest priority interrupt pending from
     * INTSTATCLR register
     */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->INTPRI, VLYNQ_INTPRI_INSTAT);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntrStatGet
 *
 *   @b Description
 *   @n Gets the status flags of the interrupts in INTSTATCLR register.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntrStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntrStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of INTSTATCLR register*/
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->INTSTATCLR,\
                                   VLYNQ_INTSTATCLR_INTCLR);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntrPendStatGet
 *
 *   @b Description
 *   @n Gets the status flags of the interrupts in INTPENDSET register.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               intrPendStat;

        ...
        CSL_vlynqIntrPendStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntrPendStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of INTPENDSET register*/
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->INTPENDSET,\
                                   VLYNQ_INTPENDSET_INTSET);
}


/** ============================================================================
 *   @n@b CSL_vlynqIntPtrGet
 *
 *   @b Description
 *   @n Gets the address of config-reg/mem-location from INTPTR register.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               intrPtr;

        ...
        CSL_vlynqIntPtrGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntPtrGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Get address of Config-reg/mem-location from INTPTR */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->INTPTR, VLYNQ_INTPTR_INTPTR);
}

/** ============================================================================
 *   @n@b CSL_vlynqDeviceRevGet
 *
 *   @b Description
 *   @n Gets the chip version.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqDeviceRevGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqDeviceRevGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Get Chip version (Type and Version of VLYNQ device) */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->CHIPVER, VLYNQ_CHIPVER_DEVREV);
}

/** ============================================================================
 *   @n@b CSL_vlynqDeviceIdGet
 *
 *   @b Description
 *   @n Gets the chip Id.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqDeviceIdGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqDeviceIdGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Get Chip version (Type and Version of VLYNQ device) */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->CHIPVER, VLYNQ_CHIPVER_DEVID);
}

/** ============================================================================
 *   @n@b CSL_vlynqAutoNgo2xGet
 *
 *   @b Description
 *   @n Gets the Auto Negotiation Protocol Info (2.x or 1.x).
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqAutoNgo2xGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqAutoNgo2xGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Get the Auto Negotiation Protocol Info (2.x or 1.x) */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->AUTNGO, VLYNQ_AUTNGO_2X);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int3TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-3 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int3TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int3TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-3 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE3);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int3PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-3 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int3PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int3PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-3 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL3);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int3IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-3 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int3IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int3IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0's Interrupt-3 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC3);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int2TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-2 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int2TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int2TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-2 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE2);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int2PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-2 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int2PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int2PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-2 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL2);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int2IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-2 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int2IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int2IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0's Interrupt-2 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC2);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int1TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-1 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int1TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int1TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-1 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE1);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int1PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-1 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int1PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int1PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-1 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL1);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int1IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-1 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int1IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int1IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0's Interrupt-1 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC1);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int0TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-0 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int0TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int0TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-0 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTTYPE0);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int0PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-0 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int0PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int0PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0s' Interrupt-0 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTPOL0);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec0Int0IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC0s Interrupt-0 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec0Int0IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec0Int0IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC0's Interrupt-0 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC0, VLYNQ_INTVEC0_INTVEC0);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int7TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-7 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int7TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int7TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-7 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE7);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int7PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-7 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int7PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int7PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-7 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL7);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int7IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-7 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int7IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int7IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1's Interrupt-7 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC7);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int6TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-6 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int6TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int6TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-6 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE6);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int6PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-6 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int6PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int6PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-6 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL6);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int6IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-6 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int6IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int6IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1's Interrupt-6 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC6);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int5TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-5 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int5TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int5TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-5 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE5);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int5PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-5 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int5PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int5PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-5 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL5);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int5IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-5 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int5IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int5IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1's Interrupt-5 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC5);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int4TypeGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-4 type 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int4TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int4TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-4 type */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTTYPE4);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int4PolGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-4 polarity 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int4PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int4PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1s' Interrupt-4 polarity */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTPOL4);
}

/** ============================================================================
 *   @n@b CSL_vlynqIntVec1Int4IntVectGet
 *
 *   @b Description
 *   @n Gets the INTVEC1s Interrupt-4 interrupt vector 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqIntVec1Int4IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqIntVec1Int4IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* INTVEC1's Interrupt-4 interrupt vector */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->INTVEC1, VLYNQ_INTVEC1_INTVEC4);
}

/* Status of remote registers */

/** ============================================================================
 *   @n@b CSL_vlynqRemRevIdGet
 *
 *   @b Description
 *   @n Gets the unique ID of remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               revId;

        ...
        CSL_vlynqRemRevIdGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRevIdGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the unique ID of the remote VLYNQ module */
    *(Uint32*)response = (Uint32) CSL_FEXT (hVlynq->regs->RREVID,
                                            VLYNQ_RREVID_ID);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemRevMajGet
 *
 *   @b Description
 *   @n Gets the major revision of remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               revMaj;

        ...
        CSL_vlynqRemRevMajGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRevMajGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the major revision of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RREVID, \
                                   VLYNQ_RREVID_REVMAJ);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRevMinGet
 *
 *   @b Description
 *   @n Gets the minor revision of remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               revMin;

        ...
        CSL_vlynqRemRevMinGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRevMinGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the minor revision of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RREVID, \
                                   VLYNQ_RREVID_REVMIN);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemPowMgmtGet
 *
 *   @b Description
 *   @n Gets the Remote Power Management iformation of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        ...
        CSL_vlynqRemPowMgmtGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemPowMgmtGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Power Management info */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RCTRL, VLYNQ_RCTRL_PMEN);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxSampleValGet
 *
 *   @b Description
 *   @n Gets the Remote rxsampleval of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        ...
        CSL_vlynqRemRxSampleValGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRxSampleValGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* get value from rxsampleval field */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RCTRL,
                                   VLYNQ_RCTRL_RXSAMPELVAL);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRtmEnableGet
 *
 *   @b Description
 *   @n Gets the Remote RTM Logic iformation of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemRtmEnableGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRtmEnableGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RTM Logic info */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RCTRL, VLYNQ_RCTRL_RTMENABLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemTxFastPathGet
 *
 *   @b Description
 *   @n Gets the Remote Transmit Path info of VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        ...
        CSL_vlynqRemTxFastPathGet (hVlynq,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemTxFastPathGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Transmit Path */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RCTRL, VLYNQ_RCTRL_TXFASTPATH);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemClkDirGet
 *
 *   @b Description
 *   @n Gets the remote serial clock direction.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               clkDir;

        ...
        CSL_vlynqRemClkDirGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemClkDirGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the clock direction of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RCTRL, VLYNQ_RCTRL_CLKDIR);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntLocGet
 *
 *   @b Description
 *   @n Gets the interrupt postage info of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               intLoc;

        ...
        CSL_vlynqRemIntLocGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntLocGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the interrupt postage info of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RCTRL, VLYNQ_RCTRL_INTLOCAL);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemIntEnableStatGet
 *
 *   @b Description
 *   @n Gets if the interrupts are enabled/disabled on the remote VLYNQ
 *      module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *      module..
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               intEnableStatus;

        ...
        CSL_vlynqRemIntEnableStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntEnableStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the interrupt enable status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RCTRL, VLYNQ_RCTRL_INTENABLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVecStatGet
 *
 *   @b Description
 *   @n Gets the interrupts vector status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               intVec;

        ...
        CSL_vlynqRemIntVecStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVecStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the interrupt vector status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RCTRL, VLYNQ_RCTRL_INTVEC);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemInt2CfgStatGet
 *
 *   @b Description
 *   @n Gets the status of the int2cfg bit of the remote VLYNQ control
 *      register.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               int2Cfg;

        ...
        CSL_vlynqRemInt2CfgStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemInt2CfgStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the INT2CFG bit status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RCTRL, VLYNQ_RCTRL_INT2CFG);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemAoptDisableGet
 *
 *   @b Description
 *   @n Gets the Remote Address optimization disable bit status.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemAoptDisableGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemAoptDisableGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Address Optimization */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RCTRL,
                                   VLYNQ_RCTRL_AOPTDISABLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemILoopStatGet
 *
 *   @b Description
 *   @n Gets the internal loopback bit status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               loopback;

        ...
        CSL_vlynqRemILoopStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemILoopStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the internal loopback bit status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RCTRL, VLYNQ_RCTRL_ILOOP);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemSerWidthInGet
 *
 *   @b Description
 *   @n Gets the Remote serial data width.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemSerWidthInGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemSerWidthInGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Size/Width of Remote inbound serial data */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RSTAT, VLYNQ_RSTAT_SWIDTHIN);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemSerWidthOutGet
 *
 *   @b Description
 *   @n Gets the Remote serial data width.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemSerWidthOutGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemSerWidthOutGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Size/Width of Remote  inbound serial data */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RSTAT, VLYNQ_RSTAT_SWIDTHOUT);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRxCurrentSampleGet
 *
 *   @b Description
 *   @n Gets the Current Remote clock sample value used by RTM.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemRxCurrentSampleGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRxCurrentSampleGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Current Remote clock sample value used by RTM */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->RSTAT, VLYNQ_RSTAT_RXCURRENTSAMPLE);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRtmGet
 *
 *   @b Description
 *   @n Gets the Remote RTM logic inclusion in RTL.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemRtmGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRtmGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RTM logic inclusion in RTL */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RSTAT, VLYNQ_RSTAT_RTM);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIFlowStatGet
 *
 *   @b Description
 *   @n Gets the inbound flow control status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIFlowStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIFlowStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the inbound flow control status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_IFLOW);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemOFlowStatGet
 *
 *   @b Description
 *   @n Gets the outbound flow control status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemOFlowStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemOFlowStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the outbound flow control status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_OFLOW);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemRErrorStatGet
 *
 *   @b Description
 *   @n Gets the remote packet error status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemRErrorStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemRErrorStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the remote packet error status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_RERROR);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemLErrorStatGet
 *
 *   @b Description
 *   @n Gets the local packet error status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemLErrorStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemLErrorStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the local packet error status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_LERROR);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemNFEmpty3StatGet
 *
 *   @b Description
 *   @n Gets the fifo 3 status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemNFEmpty3StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemNFEmpty3StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 3 status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_NFEMPTY3);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemNFEmpty2StatGet
 *
 *   @b Description
 *   @n Gets the fifo 2 status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemNFEmpty2StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemNFEmpty2StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 2 status of the VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_NFEMPTY2);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemNFEmpty1StatGet
 *
 *   @b Description
 *   @n Gets the fifo 1 status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemNFEmpty1StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemNFEmpty1StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 1 status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_NFEMPTY1);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemNFEmpty0StatGet
 *
 *   @b Description
 *   @n Gets the fifo 0 status of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemNFEmpty0StatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemNFEmpty0StatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the fifo 0 status of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_NFEMPTY0);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemSPendStatGet
 *
 *   @b Description
 *   @n Gets the status pending requests at slave interface of the remote
 *      VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemSPendStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemSPendStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of pending requests at the slave interface */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_SPEND);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemMPendStatGet
 *
 *   @b Description
 *   @n Gets the status of the pending requests at master interface of the
 *      remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemMPendStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemMPendStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of pending requests at the master interface */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_MPEND);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemLinkStatGet
 *
 *   @b Description
 *   @n Gets the status of the serial link establishment of the remote
 *      VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemLinkStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemLinkStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of pending requests at the master interface */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RSTAT, VLYNQ_RSTAT_LINK);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemNoIntPendGet
 *
 *   @b Description
 *   @n Gets the status of the Remote pending requests at master interface.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemNoIntPendGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemNoIntPendGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Status of any pending interrupts from
     * INTSTATCLR register
     */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->RINTPRI, VLYNQ_RINTPRI_NOINTPEND);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntPriStatGet
 *
 *   @b Description
 *   @n Gets the Status of the Remote highest priority interrupt pending from
 *      INTSTATCLR register
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntPriStatGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntPriStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Status of the Remote highest priority interrupt pending from
     * INTSTATCLR register
     */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->RINTPRI, VLYNQ_RINTPRI_INSTAT);
}


/** ============================================================================
 *   @n@b CSL_vlynqRemIntrStatGet
 *
 *   @b Description
 *   @n Gets the status flags of the interrupts in INTSTATCLR register of
 *      the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        CSL_vlynqRemIntrStatGet (hVlynq ,response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntrStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of RINTSTATCLR register of the remote VLYNQ module */
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RINTSTATCLR,
                                   VLYNQ_RINTSTATCLR_INTCLR);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntrPendStatGet
 *
 *   @b Description
 *   @n Gets the status flags of the interrupts in INTPENDSET register of
 *      the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        CSL_vlynqRIntrPendStatGet (hVlynq ,response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntrPendStatGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Read the status of RINTPENDSSET register*/
    *(Uint32*)response = CSL_FEXT (hVlynq->regs->RINTPENDSET,
                                   VLYNQ_RINTPENDSET_INTSET);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntPtrGet
 *
 *   @b Description
 *   @n Gets the address of config-reg/mem-location from RINTPTR register of
 *      the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        CSL_vlynqRemIntPtrGet (hVlynq ,response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntPtrGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Get address of Config-reg/mem-location from INTPTR of
     * remote VLYNQ
     */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->RINTPTR, VLYNQ_RINTPTR_INTPTR);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemDeviceRevGet
 *
 *   @b Description
 *   @n Returns the chip version of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance
        
        response      Placeholder to return status.
     @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns the chip version of the remote VLYNQ module.
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               devRev;

        ...
        devRev = CSL_vlynqRemDeviceRevGet (hVlynq, response);
        ...
     @endverbatim
 *  ============================================================================
 */
static inline
void CSL_vlynqRemDeviceRevGet (
    CSL_VlynqHandle  hVlynq,
    void             *response
)
{
    /* Read the chip version */
    *(Uint32 *)response  = CSL_FEXT(hVlynq->regs->RCHIPVER, 
                                    VLYNQ_RCHIPVER_DEVREV);

}


/** ============================================================================
 *   @n@b CSL_vlynqRemDeviceIdGet
 *
 *   @b Description
 *   @n Returns the chip ID of the remote VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance
        
        response      Placeholder to return status.
     @endverbatim
 *
 *   <b> Return Value </b>  Uint32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n Returns the chip ID of the remote VLYNQ module.
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        Uint32               devId;

        ...
        devId = CSL_vlynqRemDeviceIDGet (hVlynq, response);
        ...
     @endverbatim
 *  ============================================================================
 */
static inline
void CSL_vlynqRemDeviceIdGet (
    CSL_VlynqHandle  hVlynq,
    void             *response
)
{
    /* Read the chip ID */
    *(Uint32 *)response = CSL_FEXT(hVlynq->regs->RCHIPVER, VLYNQ_RCHIPVER_DEVID);
   
    
}

/** ============================================================================
 *   @n@b CSL_vlynqRemAutongo2xGet
 *
 *   @b Description
 *   @n Gets the Auto Negotiation Protocol Info (2.x or 1.x) of the remote
 *      VLYNQ module.
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;

        CSL_vlynqRemAutongo2xGet (hVlynq ,response);

     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemAutongo2xGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* Get remote VLYNQs' Auto Negotiation Protocol Info (2.x or 1.x) */
    *(Uint32 *)response =
        CSL_FEXT(hVlynq->regs->RAUTNGO, VLYNQ_RAUTNGO_2X);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int3TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-3 type of remote VLYNQ 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int3TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int3TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0s' Interrupt-3 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE3);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int3PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-3 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int3PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int3PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{   
    /* RINTVEC0s' Interrupt-3 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL3);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int3IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0's Interrupt-3 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int3IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int3IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0's Interrupt-3 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC3);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int2TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-2 type of remote VLYNQ 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int2TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int2TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0s' Interrupt-2 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE2);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int2PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-2 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int2PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int2PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{   
    /* RINTVEC0s' Interrupt-2 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL2);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int2IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0's Interrupt-2 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int2IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int2IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0's Interrupt-2 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC2);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int1TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-1 type of remote VLYNQ 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int1TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int1TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0s' Interrupt-1 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE1);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int1PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-1 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int1PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int1PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{   
    /* RINTVEC0s' Interrupt-1 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL1);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int1IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0's Interrupt-1 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int1IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int1IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0's Interrupt-1 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC1);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int0TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-0 type of remote VLYNQ 
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int0TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int0TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0s' Interrupt-0 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTTYPE0);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int0PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0s' Interrupt-0 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int0PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int0PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{   
    /* RINTVEC0s' Interrupt-0 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTPOL0);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec0Int0IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC0's Interrupt-0 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec0Int0IntVectGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec0Int0IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC0's Interrupt-0 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC0, VLYNQ_RINTVEC0_INTVEC0);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int7TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-7 type of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int7TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int7TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-7 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE7);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int7PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-7 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int7PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int7PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-7 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL7);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int7IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1's Interrupt-7 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int7IntVectGet (hVlynq, response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int7IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1's Interrupt-7 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC7);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int6TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-6 type of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int6TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int6TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-6 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE6);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int6PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-6 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int6PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int6PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-6 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL6);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int6IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1's Interrupt-6 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int6IntVectGet (hVlynq, response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int6IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1's Interrupt-6 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC6);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int5TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-5 type of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int5TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int5TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-5 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE5);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int5PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-5 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int5PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int5PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-5 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL5);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int5IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1's Interrupt-5 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int5IntVectGet (hVlynq, response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int5IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1's Interrupt-5 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC5);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int4TypeGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-4 type of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int4TypeGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int4TypeGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-4 type of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTTYPE4);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int4PolGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1s' Interrupt-4 polarity of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int4PolGet (hVlynq ,response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int4PolGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1s' Interrupt-4 polarity of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTPOL4);
}

/** ============================================================================
 *   @n@b CSL_vlynqRemIntVec1Int4IntVectGet
 *
 *   @b Description
 *   @n Gets the RINTVEC1's Interrupt-4 interrupt vector of remote VLYNQ
 *
 *   @b Arguments
 *   @verbatim
        hVlynq        Handle to the VLYNQ port instance

        response      Placeholder to return status.

     @endverbatim
 *
 *   <b> Return Value </b>  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_VlynqHandle      hVlynq;
        ...
        CSL_vlynqRemIntVec1Int4IntVectGet (hVlynq, response);
        ...
     @endverbatim
 * ===========================================================================
 */
static inline
void CSL_vlynqRemIntVec1Int4IntVectGet (
    CSL_VlynqHandle      hVlynq,
    void                 *response
)
{
    /* RINTVEC1's Interrupt-4 interrupt vector of remote VLYNQ */
    *(Uint32 *)response =
                    CSL_FEXT(hVlynq->regs->RINTVEC1, VLYNQ_RINTVEC1_INTVEC4);
}

#ifdef __cplusplus
}
#endif

#endif /* _CSL_VLYNQAUX_H_ */
