//------------------------------------------------------------------------------
// File main.c
//------------------------------------------------------------------------------
// Copyright (c) 2010, Texas Instruments Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice,
//       this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of the Texas Instruments Inc. nor the names of its
//       contributors may be used to endorse or promote products derived from
//       this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//------------------------------------------------------------------------------

#include "app.h"
#include "edma3.h"
#include "ti/pspiom/cslr/cslr_psc_C6747.h"
#include "ti/pspiom/cslr/cslr_pllc.h"
#include "ti/pspiom/cslr/cslr_syscfg_C6747.h"
#include "ti/pspiom/cslr/cslr_intc.h"

extern cregister volatile unsigned int    ISTP;    // interrupt clear register
extern cregister volatile unsigned int    ICR;    // interrupt clear register
extern cregister volatile unsigned int    IER;    // interrupt enable register


#define PLLEN_MUX_SWITCH  4 
#define PLL_LOCK_TIME_CNT 2400

/* Pointer to register overlay structure */
CSL_PllcRegsOvly     pllcRegs = ((CSL_PllcRegsOvly)CSL_PLLC_0_REGS);

/* sys config registers overlay                                               */
CSL_SyscfgRegsOvly   sysRegs  = (CSL_SyscfgRegsOvly)(CSL_SYSCFG_0_REGS);
/* Psc register overlay                                                       */
CSL_PscRegsOvly      psc0Regs = ((CSL_PscRegsOvly)CSL_PSC_0_REGS);                
CSL_PscRegsOvly      psc1Regs = (CSL_PscRegsOvly)(CSL_PSC_1_REGS);

extern  CSL_Edma3ccRegsOvly edmaCcRegs = (CSL_Edma3ccRegsOvly)CSL_EDMA3CC_0_REGS;

extern void intcVectorTable(void);
void SystemTargetInit(void);

static void PSC_InitialCfg(void);
static void SYSCFG_InitialCfg(void);
static void PLL_IntialCfg(int pll_multiplier);

static void PSC0_lPSC_enable(Uint8 LPSC_num);
static void PSC1_lPSC_enable(Uint8 LPSC_num);
//-----------------------------------------------------------------------------
// Application Main Function
//-----------------------------------------------------------------------------

int main()
{
    CSL_IntcRegsOvly intcRegs = (CSL_IntcRegsOvly)CSL_INTC_0_REGS;

	//EVMC6747_init();
	SystemTargetInit();

	CSL_FINS(intcRegs->INTMUX1, INTC_INTMUX1_INTSEL4,CSL_INTC_EVENTID_TPCC0_ERRINT);
	//CSL_FINS(intcRegs->INTMUX1, INTC_INTMUX1_INTSEL5,CSL_INTC_EVENTID_AXRINT);

	/* set ISTP to point to the vector table address                            */
	ISTP = (unsigned int)intcVectorTable;

	/* clear all interrupts, bits 4 thru 15                                     */
	ICR = 0xFFF0;

	/* enable the bits for non maskable interrupt and CPUINT4                   */
	IER |= 0x12;

	/* enable interrupts, set GIE bit                                           */
	_enable_interrupts();

	edmaCcRegs->ICR = 0xffffffff;
	edmaCcRegs->IECR = 0xffffffff;

    // initialize EVM, EDMA
    
    edmaInit();
    

	audio_tsk_fxn();

    return 0;
}




/*
******************************************************************************
*                           Golable Functions Realize
******************************************************************************
*/
void SystemTargetInit(void)
{
 	PSC_InitialCfg();

	SYSCFG_InitialCfg();
      
    PLL_IntialCfg(24);

}



/*
******************************************************************************
*                           Local Functions Realize
 *   @func   PSC_InitialCfg();
 *
 *   @desc
 *      ʹܲλGPIO
 *
******************************************************************************
*/
void PSC_InitialCfg()
{
	//PSC0
	PSC0_lPSC_enable(0);
	PSC0_lPSC_enable(1);
	PSC0_lPSC_enable(2);
	PSC0_lPSC_enable(3);
	PSC0_lPSC_enable(4);
	PSC0_lPSC_enable(5);
	PSC0_lPSC_enable(6);
	PSC0_lPSC_enable(7);
	PSC0_lPSC_enable(8);
	PSC0_lPSC_enable(9);
	PSC0_lPSC_enable(10);
	PSC0_lPSC_enable(11);
	PSC0_lPSC_enable(12);
	PSC0_lPSC_enable(13);

	//PSC1
    PSC1_lPSC_enable(1);
    PSC1_lPSC_enable(2);
    PSC1_lPSC_enable(3);  //GPIO
	PSC1_lPSC_enable(4);
    PSC1_lPSC_enable(5);
    PSC1_lPSC_enable(6);  // EMIFB
    PSC1_lPSC_enable(7);  //McASP0
    PSC1_lPSC_enable(8);
    PSC1_lPSC_enable(9);
    PSC1_lPSC_enable(10);
    PSC1_lPSC_enable(11);
    PSC1_lPSC_enable(12);
    PSC1_lPSC_enable(13);
    PSC1_lPSC_enable(16);
    PSC1_lPSC_enable(17);
    PSC1_lPSC_enable(20);
    PSC1_lPSC_enable(21);
    PSC1_lPSC_enable(24);
    PSC1_lPSC_enable(25);
    PSC1_lPSC_enable(26);
    PSC1_lPSC_enable(31);
}

/*
******************************************************************************
*                           Local Functions Realize
 *   @func   SYSCFG_InitialCfg();
 *
 *   @desc
 *      ʹGPIO2_11ӦPin mux registers
 *
******************************************************************************
*/
void SYSCFG_InitialCfg(void)
{
	volatile Uint32 temp = 0;
	/* Key to be written to enable the pin mux registers to be written        */
    sysRegs->KICK0R = 0x83e70b13;
    sysRegs->KICK1R = 0x95A4F1E0;    

    //MCASP0
	sysRegs->PINMUX9  = 0x11111100;  // AFSR0,ACLKR0,AHCLKR0,AFSX0,ACLKX0,AHCLKX0
    sysRegs->PINMUX10 = 0x11111111;  // McASP0(AXR0[0]--AXR0[6])
    sysRegs->PINMUX11 = 0x00000011;  // McASP0(AXR0[7],AXR0[8])
    //sysRegs->PINMUX12 = 0x08 << 0x10;


    /* lock the pinmux registers                                              */
    sysRegs->KICK0R = 0x00000000;
    sysRegs->KICK1R = 0x00000000;

	//gpioRegs->BANK[2].DIR = 0x00000000;
}

/*
******************************************************************************
*                           Local Functions Realize
 *   @func   PLL_InitialCfg();
 *
 *   @desc
 *      ϵͳʱӣⲿΪ24MHzpll_multiplier=24
 *      ʣ˺Ϊ24*24+1=600MHz;
 *          SYSCLK1 600/2= 300MHz;
 *
******************************************************************************
*/
void PLL_IntialCfg(int pll_multiplier)
{
    int i = 0;
     /* Configure ARM, DSP at 300MHz, EMIFs at 133MHz */
    //Uint8 DIV45_EN = 1;
    //CLKMODE λ0
    Uint8 CLKMODE = 0;
    Uint8 PLLM = 24;
    Uint8 POSTDIV = 1;
    Uint8 PLLDIV3 = 2;
    Uint8 PLLDIV5 = 5;
    Uint8 PLLDIV7 = 7;
    Uint8 PREDIV = 0;

	// Moved step 2c and 2d to step 0
   /*Set PLLEN=0 and PLLRST=0, Reset the PLL*/
    pllcRegs->PLLCTL &=  0xFFFFFFFE; 	/*PLL BYPASS MODE*/
   
   /*wait for 4 cycles to allow PLLEN mux switches properly to bypass clock*/
   for(i=0; i<PLLEN_MUX_SWITCH; i++) {;}   /*Make PLLEN_MUX_SWITCH as bootpacket*/

   /*Select the Clock Mode bit 8 as External Clock or On Chip Oscilator*/
	pllcRegs->PLLCTL &= 0xFFFFFEFF;  
    pllcRegs->PLLCTL |= (CLKMODE<<8);  /* Make CLKSRC as BootPacket to pass the value*/

   /*Set PLLENSRC '0',bit 5, PLL Enable(PLLEN) selection is controlled through MMR*/
    pllcRegs->PLLCTL &=  0xFFFFFFDF; 
   
   /*PLLCTL.EXTCLKSRC bit 9 should be left at 0 for Primus*/
    pllcRegs->PLLCTL &=  0xFFFFFDFF;

   /* Clear PLLRST bit to 0 -Reset the PLL */
   pllcRegs->PLLCTL &= 0xFFFFFFF7; 	
  
   /*Disable the PLL output*/ 
   pllcRegs->PLLCTL |= 0x10; 		
   
   /*PLL initialization sequence*/
   
   /*Power up the PLL- PWRDN bit set to 0 to bring the PLL out of power down bit*/
   pllcRegs->PLLCTL &= 0xFFFFFFFD;
   
   /*Enable the PLL from Disable Mode PLLDIS bit to 0 - This is step is not required for Primus*/
   pllcRegs->PLLCTL &= 0xFFFFFFEF;
   
   /*PLL stabilisation time- take out this step , not required here when PLL in bypassmode*/
  // for(i=0; i<PLL_STABILIZATION_TIME; i++) {;}  /* Make PLL_STABILIZATION_TIME as bootpacket*/
   pllcRegs->PREDIV = 0x8000 | PREDIV;   
   /*Program the required multiplier value in PLLM*/
   pllcRegs->PLLM    = PLLM; /* Make PLLMULTIPLEIR as bootpacket*/

   /*If desired to scale all the SYSCLK frequencies of a given PLLC, program the POSTDIV ratio*/
 
   pllcRegs->POSTDIV = 0x8000 | POSTDIV; /* Make POSTDIV as bootpacket*/

   /*If Necessary program the PLLDIVx*/
   /*Check for the GOSTAT bit in PLLSTAT to clear to 0 to indicate that no GO operation is currently in progress*/
   while(pllcRegs->PLLSTAT & 0x1==1){}

   /*Program the RATIO field in PLLDIVx with the desired divide factors. In addition, make sure in this step you leave the PLLDIVx.DxEN bits set so clocks are still enabled (default).*/
    pllcRegs->PLLDIV3 = 0x8000 | PLLDIV3; /* Make PLLDIV3 as bootpacket, do it for other PLLDIVx to if required*/
    pllcRegs->PLLDIV5 = 0x8000 | PLLDIV5; /* Make PLLDIV5 as bootpacket, do it for other PLLDIVx to if required*/
    pllcRegs->PLLDIV7 = 0x8000 | PLLDIV7; /* Make PLLDIV7 as bootpacket, do it for other PLLDIVx to if required*/

    /*Set the GOSET bit in PLLCMD to 1 to initiate a new divider transition.*/
    pllcRegs->PLLCMD |= 0x1;

	/*Wait for the GOSTAT bit in PLLSTAT to clear to 0 (completion of phase alignment).*/
    while(pllcRegs->PLLSTAT & 0x1==1) { } 
   
 
   /*Wait for PLL to reset properly. See PLL spec for PLL reset time - This step is not required here -step11*/
  // for(i=0; i<PLL_RESET_TIME_CNT; i++) {;}   /*128 MXI Cycles*/ /*Make PLL_RESET_TIME_CNT as boot packet*/
      
   /*Set the PLLRST bit in PLLCTL to 1 to bring the PLL out of reset*/
   pllcRegs->PLLCTL |= 0x8;
   
   /*Wait for PLL to lock. See PLL spec for PLL lock time*/
   for(i=0; i<PLL_LOCK_TIME_CNT; i++) {;} /*Make PLL_LOCK_TIME_CNT as boot Packet*/ 
   
   /*Set the PLLEN bit in PLLCTL to 1 to remove the PLL from bypass mode*/
   pllcRegs->PLLCTL |=  0x1;

   sysRegs->KICK0R = 0x83e70b13;  // Kick0 register + data (unlock)
   sysRegs->KICK1R = 0x95a4f1e0;  // Kick1 register + data (unlock)
//   CFGCHIP1 |=0x44;     //GPIO3ж
   sysRegs->CFGCHIP3 |= 0x4;       // Enable 4.5 divider PLL
   sysRegs->CFGCHIP3 |= 0x1;       // Select 4.5 divider for EMIFB clock source only (not EMIFA)

}


/*Enable Function for PSC0*/
void PSC0_lPSC_enable(Uint8 LPSC_num)
 {
	volatile Uint32 temp = 0;
	volatile Uint32 pscTimeoutCount = 10240u;

      /* Bring the GPIO module out of sleep state                                 */
	  /* Configure the GPIO Module to Enable state */
	  psc0Regs->MDCTL[LPSC_num] =( (psc0Regs->MDCTL[LPSC_num] & 0xFFFFFFE0) | \
	                                 CSL_PSC_MDSTAT_STATE_ENABLE );
	  /* Kick start the Enable Command */
	  temp = psc0Regs->PTCMD;
	  temp = ( (temp & CSL_PSC_PTCMD_GO0_MASK) |
	           (CSL_PSC_PTCMD_GO0_SET << CSL_PSC_PTCMD_GO0_SHIFT) );
	  psc0Regs->PTCMD |= temp;

	  /*Wait for the power state transition to occur */
	  while ( ((psc0Regs->PTSTAT & (CSL_PSC_PTSTAT_GOSTAT0_IN_TRANSITION)) != 0)
	                     && (pscTimeoutCount>0) )
	  {
	      pscTimeoutCount--;
	  }

	  /* Check if PSC state transition timed out */
	  if(pscTimeoutCount == 0)
	  {
	      printf("%d PSC0 transition to ON state timed out\n",LPSC_num);
	      return;
	  }

	  /* Wait for MODSTAT = ENABLE/DISABLE from LPSC */
	  pscTimeoutCount = 10240u;
	  while( ((psc0Regs->MDSTAT[LPSC_num] & (CSL_PSC_MDSTAT_STATE_MASK))
	                != CSL_PSC_MDSTAT_STATE_ENABLE) && (pscTimeoutCount>0))
	  {
	      pscTimeoutCount--;
	  }

	  /* If timeout, the resource may not be functioning */
	  if (0 == pscTimeoutCount)
	  {
	     printf("%d PSC0 Module Enable timed out\n",LPSC_num);
	     return;
	  }
}

/*Enable Function for PSC1*/
void PSC1_lPSC_enable(Uint8 LPSC_num)
 {
	volatile Uint32 temp = 0;
	volatile Uint32 pscTimeoutCount = 10240u;

      /* Bring the GPIO module out of sleep state                                 */
	  /* Configure the GPIO Module to Enable state */
	  psc1Regs->MDCTL[LPSC_num] =
	                              ( (psc1Regs->MDCTL[LPSC_num] & 0xFFFFFFE0) | \
	                                 CSL_PSC_MDSTAT_STATE_ENABLE );
	  /* Kick start the Enable Command */
	  temp = psc1Regs->PTCMD;
	  temp = ( (temp & CSL_PSC_PTCMD_GO0_MASK) |
	           (CSL_PSC_PTCMD_GO0_SET << CSL_PSC_PTCMD_GO0_SHIFT) );
	  psc1Regs->PTCMD |= temp;

	  /*Wait for the power state transition to occur */
	  while ( ((psc1Regs->PTSTAT & (CSL_PSC_PTSTAT_GOSTAT0_IN_TRANSITION)) != 0)
	                     && (pscTimeoutCount>0) )
	  {
	      pscTimeoutCount--;
	  }

	  /* Check if PSC state transition timed out */
	  if(pscTimeoutCount == 0)
	  {
	      printf("%d PSC1 transition to ON state timed out\n",LPSC_num);
	      return;
	  }

	  /* Wait for MODSTAT = ENABLE/DISABLE from LPSC */
	  pscTimeoutCount = 10240u;
	  while( ((psc1Regs->MDSTAT[LPSC_num] & (CSL_PSC_MDSTAT_STATE_MASK))
	                != CSL_PSC_MDSTAT_STATE_ENABLE) && (pscTimeoutCount>0))
	  {
	      pscTimeoutCount--;
	  }

	  /* If timeout, the resource may not be functioning */
	  if (0 == pscTimeoutCount)
	  {
	     printf("%d PSC1 Module Enable timed out\n",LPSC_num);
	     return;
	  }

}
//------------------------------------------------------------------------------
// End of File main.c
//------------------------------------------------------------------------------
