/*
 * ETB_Profile_Func.c
 *
 *  Created on: Mar 19, 2014
 *      Author: Jane
 */
#include <stdio.h>
#include <stdlib.h>
#include "ETB_Profile.h"

AET_jobParams StartTraceParams;
AET_jobParams EndTraceParams;


void Init_Profile(uint32_t* pDmaMemory, TraceJOBConfig* ProfileConfig)
{
    int idx;

    /* Fill EDMA destination memory with pattern to test for writes */
    for(idx = 0; idx < (EDMA_BFR_WORDS); idx++)
    {
        pDmaMemory[idx] = 0xcccccccc;
    }

    ProfileConfig->endJob = 0xff;
    ProfileConfig->endProfileAddr = 0;
    ProfileConfig->startJob = 0xff;
    ProfileConfig->startProfileAddr = 0;
    ProfileConfig->tracetype = 0;
}

void Setup_ETB_Profile(TraceJOBConfig* ProfileConfig, ETBHandle** pETBHandle, DSPTraceHandle** pDSPHandle)
{
    eETB_Error  etbRet;
    ETB_errorCallback pETBErrCallBack =0;
    uint8_t coreID =0;
    uint32_t etbWidth;


    eDSPTrace_Error dspRet;
    DSPTrace_errorCallback pDSPErrorCallBack=0;
//    DMAStatus dmaStatus;

     /* Determine the specific CorePac this application is running on */
    coreID = DNUM;
    printf("\n CORE ID: %d\n\n", coreID);

   /*** Setup ETB receiver ***/
   /* Open ETB module */
    etbRet = ETB_open(pETBErrCallBack, eETB_TI_Mode, coreID, pETBHandle, &etbWidth);
    if(etbRet != eETB_Success)
   {
      printf("Error opening ETB\n");
      return;
   }

    //Setup EDMA
    Config_EDMA_for_DSPETB(*pETBHandle, EDMA_DEST_ADDRESS, EDMA_BFR_WORDS, coreID);

    /* Enable ETB receiver */
    etbRet = ETB_enable(*pETBHandle, 0);
    if(etbRet != eETB_Success)
    {
        printf("Error enabling ETB\n");
        return;
    }

    /* ** Setup Trace Export ** */
    /* Open DSP Trace export module */
    dspRet = DSPTrace_open( pDSPErrorCallBack, pDSPHandle);
    if(dspRet != eDSPTrace_Success)
    {
        printf("Error opening DSP Trace Export block\n");
        return;
    }

    /* Setup trace export clock to FCLK/3 */
    dspRet= DSPTrace_setClock(*pDSPHandle, 3);
    if(dspRet != eDSPTrace_Success)
    {
        printf("Error setting up DSP trace export clock\n");
        return;
    }

    dspRet= DSPTrace_enable(*pDSPHandle, 0, 0);
    if(dspRet != eDSPTrace_Success)
    {
        printf("Error enabling DSP trace export\n");
        return;
    }

    /* ******************************* IMP. NOTE ****************************/
    /* AETLIB should be used to setup Trace triggers before enabling DSP Trace.*/
    /* ******************************* IMP. NOTE ****************************/
    /* Set up trace source to generate trace */
    StartTraceParams = AET_JOBPARAMS; /* Initialize Job Parameter Structure */
    StartTraceParams.programAddress = ProfileConfig->startProfileAddr; //(uint32_t) &runFunction;
    StartTraceParams.traceTriggers = ProfileConfig->tracetype; // AET_TRACE_TIMING | AET_TRACE_PA;
    StartTraceParams.traceActive = AET_TRACE_ACTIVE;

    AET_init();

    /* Claim the AET resource */
    if (AET_claim())
    {
        printf("Error claiming AET resources\n");
        return;
    }

#if USE_TEND
    {
    /* ******************************* IMP. NOTE ****************************/
    /* DSPTrace_setState will not change the TEND state until after         */
    /* AET_claim is executed successfully. Must clear TEND if currently set.*/
    /* ******************************* IMP. NOTE ****************************/
        uint32_t traceState = 0;
        uint32_t new_traceState = DSPTRACE_CLR_TEND;
        DSPTrace_getState(*pDSPHandle, &traceState);
        if ( (traceState & DSPTRACE_STATE_TEND) == DSPTRACE_STATE_TEND ){

            DSPTrace_setState(*pDSPHandle, new_traceState);
        }

        DSPTrace_getState(*pDSPHandle, &traceState);
        if ( (traceState & DSPTRACE_STATE_TEND) == DSPTRACE_STATE_TEND )
            printf("TEND detected - BAD\n");
        else
            printf("TEND not detected - GOOD\n");

    }
#endif

    /* Set up the desired job */
    if (AET_setupJob(AET_JOB_START_STOP_TRACE_ON_PC, &StartTraceParams))
    {
        printf("Error setting up AET resources\n");
        return;
    }

    ProfileConfig->startJob = StartTraceParams.jobIndex;

    printf("AET start job index is %d\n", ProfileConfig->startJob);

#if !USE_TEND
    EndTraceParams = AET_JOBPARAMS;
    EndTraceParams.programAddress = ProfileConfig->endProfileAddr; //(uint32_t)&doneFunction;
    EndTraceParams.traceTriggers = ProfileConfig->tracetype;
    EndTraceParams.traceActive = AET_TRACE_INACTIVE;

    if (AET_setupJob(AET_JOB_START_STOP_TRACE_ON_PC, &EndTraceParams))
        return;

    ProfileConfig->endJob = EndTraceParams.jobIndex;

    printf("The AET stop job index is %d\n", ProfileConfig->endJob );
#endif

    /* Enable AET */
    if (AET_enable())
    {
        printf("Error enabling AET \n");
        return;
    }

}


void Stop_ETB_Profile(TraceJOBConfig* ProfileConfig, ETBHandle** pETBHandle, DSPTraceHandle** pDSPHandle)
{
    eETB_Error  etbRet;
    uint32_t  retSize=0;
    uint32_t *pBuffer=0;


    eDSPTrace_Error dspRet;
    DMAStatus dmaStatus;


#if USE_TEND
    {
    /* ******************************* IMP. NOTE ****************************/
    /* Setting TEND will terminate Trace gracefully if you don't have an    */
    /* End Trace Job or an End All Trace job.                               */
    /* ******************************* IMP. NOTE ****************************/
        uint32_t traceState = 0;
        uint32_t new_traceState = DSPTRACE_SET_TEND;
        DSPTrace_getState(*pDSPHandle, &traceState);
        if ( (traceState & DSPTRACE_STATE_TEND) != DSPTRACE_STATE_TEND ){

            DSPTrace_setState(*pDSPHandle, new_traceState);
        }

    }
#endif

    /* Flush the DTF (Trace Formatter) output and the ETB input to guarantee all
     *  of the data will get read out of the FIFO.
	 * Also stop the DTF formatter
     */

    etbRet= ETB_flush(*pETBHandle);

    if(etbRet != eETB_Success)
    {
        printf("Error flushing ETB %d\n", etbRet);
        return;
    }

    /* NOTE:
     *  ** THIS FLUSH API CALL IS REQUIRED TO GET THE DMA STATUS **
     *
     * Before reading from the ETB drain buffer, make sure all data has been
     *  read out of the ETB.
     */
    etbRet = ETB_flush_dma(*pETBHandle, &dmaStatus);
    if(etbRet == eETB_Success)
    {
        /* NOTE:
         * There are 2 methods for reading the data from the ETB drain buffer.
         * -The 1st method is manually reading from the buffer's known location
         *  using the information returned in the dmaStatus structure when
         *  ETB_flush_dma is called. In that case the pBuffer pointer is set to
         *  the start address of the buffer and the wrap case is handled below
         *  while writing to an output file.
         * -The 2nd method uses the ETB_read API function to read parts of the
         *  drain buffer into a smaller temporary buffer that is then used to
         *  copy to the output file in seperate packets. A 1kB buffer is
         *  allocated and assigned to pBuffer.
         */
#ifdef _USE_MANUAL_READ_
        pBuffer = (uint32_t *)dmaStatus.startAddr;
#else
        pBuffer = (uint32_t*)malloc(TEMP_BFR_READ_SIZE);
#endif
        retSize = dmaStatus.availableWords;
        if(dmaStatus.isWrapped)
        {
            printf("ETB DrainBfr wrapped, Start: 0x%x, Size: %d\n",
                    dmaStatus.startAddr, retSize);
        }
        else
        {
            printf("ETB DrainBfr not wrapped, Start: 0x%x, Size: %d\n",
                    dmaStatus.startAddr, retSize);
        }
    }
    else
    {
        pBuffer = 0;
        printf("ETB_flush_dma ERROR: %d\n", etbRet);
    }

    /* Transport ETB data */
    if(pBuffer)
    {
        /* This example uses JTAG debugger via CIO to transport data to host PC
         *  An app can deploy any other transport mechanism to move the ETB
         *  buffer to the PC
         *
         * This process is slow and anything over a 16Kbyte drain buffer will
         *  take a considerable amount of time to transfer. A 256Kbyte buffer
         *  will take close to 40 seconds.
         */
        char * pFileName = "C:\\temp\\DSP_etbdata.bin";
        FILE* fp = fopen(pFileName, "wb");
        if(fp)
        {
            int32_t totalSize = 0;
            int32_t  cnt;
            uint32_t sz = 0;
            uint32_t i = 1;
            char *le = (char *) &i;

#ifdef _USE_MANUAL_READ_
            /* For manual ETB drain buffer read mode, set the buffer ending
             *  address to know when the buffer has wrapped back to the
             *  beginning.
             */
            uint32_t endAddress = dmaConfig.dbufAddress + (dmaConfig.dbufWords*4) - 1;
            sz = 0;
            for(cnt = 0; cnt < retSize; cnt++)
            {
                uint32_t etbword = *(pBuffer+sz);
                if(le[0] != 1) //Big endian
                   etbword = BYTE_SWAP32(etbword);

                size_t fret = fwrite((void*) &etbword, 4, 1, fp);
                if ( fret < 1 ) {
                    printf("Error writing data to  - %s \n", pFileName);
                    return;
                }
                sz++;

                /* Check for wrap condition with circular buffer */
                if((uint32_t)(pBuffer+sz) > endAddress)
                {
                    pBuffer = (uint32_t *)dmaConfig.dbufAddress;
                    sz = 0;
                }
            }
            totalSize = retSize;
#else
            /* Loop through the necessary number of packets to read the data
             *  from the ETB drain buffer.
             * - 16 1kB packets.
             * After the packet has been successfully read, it is written to
             *  the specified output file.
             */
            int32_t pktCnt;
            uint32_t maxPackets = ((EDMA_BFR_WORDS * 4) / TEMP_BFR_READ_SIZE);
            for(pktCnt = 0; pktCnt < maxPackets; pktCnt++)
            {
                etbRet = ETB_read(*pETBHandle, pBuffer,
                                  (EDMA_BFR_WORDS/maxPackets),
                                  (pktCnt*(EDMA_BFR_WORDS/maxPackets)),
                                  (EDMA_BFR_WORDS/maxPackets), &retSize);
                totalSize += retSize;
                if(etbRet != eETB_Success)
                {
                    printf("Error reading ETB data\n");
                    break;
                }

                sz = 0;
                for(cnt = 0; cnt < retSize; cnt++)
                {
                    uint32_t etbword = *(pBuffer+sz);
                    if(le[0] != 1) //Big endian
                       etbword = BYTE_SWAP32(etbword);

                    size_t fret = fwrite((void*) &etbword, 4, 1, fp);
                    if ( fret < 1 ) {
                        printf("Error writing data to  - %s \n", pFileName);
                        return;
                    }
                    sz++;
                }
            }
#endif
            printf("Successfully transported ETB data - %s, %d words\n",
                    pFileName, totalSize);

            fclose(fp);
        }
        else {
            printf("Error opening file - %s\n", pFileName);
        }
#ifndef _USE_MANUAL_READ_
        free(pBuffer);
#endif
    }

    /*** Now we are done and close all the handles **/
    /* Now disable trace capture - ETB receiver */
    etbRet = ETB_disable(*pETBHandle);
    if(etbRet != eETB_Success)
    {
        printf("Error disabling ETB\n");
        return;
    }

    /* Disable DSP Trace Export */
    dspRet= DSPTrace_disable(*pDSPHandle);
    if(dspRet != eDSPTrace_Success)
    {
        printf("Error disabling DSP trace export\n");
        return;
    }

    etbRet  = ETB_close(*pETBHandle);
    if(etbRet != eETB_Success)
    {
        printf("Error closing ETB\n");
        return;
    }

    dspRet= DSPTrace_close(*pDSPHandle);
    if(dspRet != eDSPTrace_Success)
    {
        printf("Error closing DSP trace export module\n");
        return;
    }

    AET_releaseJob(ProfileConfig->startJob);
#if !USE_TEND
    AET_releaseJob(ProfileConfig->endJob);
#endif
    AET_release();
}
