/* --------------------------------------------------------------------------
  FILE        : util.c                                                   
  PURPOSE     : Utility and misc. file
  PROJECT     : Flashing Utilities
  AUTHOR      : Daniel Allred
  DESC        : Utility functions for flashing applications
 ----------------------------------------------------------------------------- */

// General type include
#include "tistdtypes.h"

// This module's header file
#include "util.h"

// Device specific CSL
#include "device.h"


/************************************************************
* Explicit External Declarations                            *
************************************************************/


/************************************************************
* Local Macro Declarations                                  *
************************************************************/


/************************************************************
* Local Typedef Declarations                                *
************************************************************/


/************************************************************
* Local Function Declarations                               *
************************************************************/

static Uint32 LOCAL_reflectNum(Uint32 inVal, Uint32 num);


/************************************************************
* Local Variable Definitions                                *
************************************************************/


/************************************************************
* Global Variable Definitions                               *
************************************************************/

// Global memory allocation pointer
static VUint32 currMemPtr;


/************************************************************
* Global Function Definitions                               *
************************************************************/

// DDR Memory allocation routines (for storing large data)
void *UTIL_getCurrMemPtr(void)
{
  return ((void *)currMemPtr);
}

// Setup for an adhoc heap
void UTIL_setCurrMemPtr(void *value)
{
  currMemPtr = (VUint32)value;
}

// Allocate memory from the ad-hoc heap
void *UTIL_allocMem(Uint32 size)
{
  void *cPtr;
  Uint32 size_temp;

  // Ensure word boundaries
  size_temp = ((size + 4) >> 2 ) << 2;
  
  if((currMemPtr + size_temp) > (DEVICE_DDR2_RAM_SIZE))
  {
    return NULL;
  }

  cPtr = (void *) (DEVICE_DDR2_START_ADDR + currMemPtr);
  currMemPtr += size_temp;

  return cPtr;
}

// Simple wait loop - comes in handy.
void UTIL_waitLoop(Uint32 loopcnt)
{
  Uint32 i;
  for (i = 0; i<loopcnt; i++)
  {
    asm("   NOP");
  }
}

// CRC-32 routine (relflected, init xor val = 0xFFFFFFFF, final xor val = 0xFFFFFFFF)
Uint32 UTIL_calcCRC32(Uint8* lutCRC, Uint8 *data, Uint32 size, Uint32 currCRC)
{
  unsigned int len = size;
  Uint32 crc = currCRC;
  int i;

  // Prepare input to get back into calculation state ( this means initial
  // input when starting fresh should be 0x00000000 )
  crc = currCRC ^ 0xFFFFFFFF;

  // Perform the algorithm on each byte
  for (i = 0; i < len; i++)
  {
    crc = (crc >> 8) ^ lutCRC[(crc & 0xFF) ^ data[i]];
  }

  // Exclusive OR the result with the specified value
  crc = crc ^ 0XFFFFFFFF;

  return crc;
}

// Function to build a reflected CRC-32 table (for standard CRC-32 algorithm)
// lutCRC must have 1024 bytes (256 32-bit words) available
void UTIL_buildCRC32Table(Uint32* lutCRC, Uint32 poly)
{
  Uint32 i,j,tableLen = 256;
  Uint32 crcAccum;
  
  for (i = 0; i < tableLen; i++)
  {
    crcAccum = LOCAL_reflectNum(i, 8) << (32 - 8);
    for (j = 0; j < 8; j++)
    {
      if ((crcAccum & 0x80000000) != 0x00000000)
      {
        crcAccum = (crcAccum << 1) ^ poly;
      }
      else
      {
        crcAccum = (crcAccum << 1);
      }
      lutCRC[i] = LOCAL_reflectNum(crcAccum, 32);
    }
  }
}

// CRC-16 routine (relflected, init xor val = 0xFFFF, final xor val = 0xFFFF)
Uint16 UTIL_calcCRC16(Uint8* lutCRC, Uint8 *data, Uint32 size, Uint16 currCRC)
{
  unsigned int len = size;
  Uint16 crc = currCRC;
  int i;

  // Prepare input to get back into calculation state ( this means initial
  // input when starting fresh should be 0x00000000 )
  crc = currCRC ^ 0xFFFF;

  // Perform the algorithm on each byte
  for (i = 0; i < len; i++)
  {
    crc = (crc >> 8) ^ lutCRC[(crc & 0xFF) ^ data[i]];
  }

  // Exclusive OR the result with the specified value
  crc = crc ^ 0xFFFF;

  return crc;
}

// Function to build a reflected CRC-16 table (for standard CRC-16 algorithm)
// lutCRC must have 512 bytes (256 16-bit words) available
void UTIL_buildCRC16Table(Uint16* lutCRC, Uint16 poly)
{
  Uint32 i,j,tableLen = 256;
  Uint16 crcAccum;
  
  for (i = 0; i < tableLen; i++)
  {
    crcAccum = (Uint16) LOCAL_reflectNum(i, 8) << (16 - 8);
    for (j = 0; j < 8; j++)
    {
      if ((crcAccum & 0x8000) != 0x0000)
      {
        crcAccum = (crcAccum << 1) ^ poly;
      }
      else
      {
        crcAccum = (crcAccum << 1);
      }
      lutCRC[i] = (Uint16) LOCAL_reflectNum(crcAccum, 16);
    }
  }
}


/***********************************************************
* Local Function Definitions                               *
***********************************************************/
static Uint32 LOCAL_reflectNum(Uint32 inVal, Uint32 num)
{
  Uint32 i,outVal = 0x0;

  for (i = 1; i < (num + 1); i++)
  {
    outVal |= (Uint32)( ((inVal & 0x1)) << (num - i));
    inVal >>= 1;
  }
  return outVal;
}


/***********************************************************
* End file                                                 *
***********************************************************/

/* --------------------------------------------------------------------------
    HISTORY
        v1.00  -  DJA  -  16-Aug-2007
           Initial release 
 ----------------------------------------------------------------------------- */



