/*sdq.c*/


/*******************************************************************************/
/*This file contains the functions necessary to implement SDQ communication.   */
/*The code is written specifically for STR912 microcontroller using Port 9.3   */
/*as a push pull port. A 5k pullup resistor was used at 3.3V pullup voltage.   */
/*                                                                             */
/*Written by: Michael Vega                                                     */
/*            Applications, Battery Monitor Solutions	                       */
/*            Texas Instruments, Inc.                                          */
/*******************************************************************************/

#include "sdq.h"

//These includes are necessary to use STR912 library files
#include "STR91x.h"
#include "91x_type.h"
#include "91x_map.h"
#include "91x_gpio.h"
#include "91x_scu.h"
#include "91x_conf.h"



/**********************************************************************/
/* 	void wait_us(int usec)										      */
/*																      */
/*	Description : 		Creates a delay of approximately (usec + 5us) */
/*				  		when usec is greater.   					  */
/* 	Arguments : 		usec - adds 5us to this number on delay		  */
/*	Global Variables:	None   										  */
/*  Returns: 			None								          */
/**********************************************************************/
void wait_us(int usec){
   while (usec--);
}

/**********************************************************************/
/* 	void SendReset(void)										      */
/*																      */
/*	Description : 		Creates the Reset signal to initiate SDQ 	  */
/*				  		communication.								  */
/* 	Arguments : 		None										  */
/*	Global Variables:	None   										  */
/*  Returns: 			None								          */
/**********************************************************************/
void SendReset(void){
	*GPIO9_DIR |= 0x08;							//Set GPIO P9.3 as Output
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_SET);	//Set High
	wait_us(20);								//Allow PWR cap to charge and power IC	~ 25us
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_RESET);	//Set Low
	wait_us(480);								//Reset time greater then 480us
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_SET);	//Set High
	*GPIO9_DIR &= 0xF7;							//Set GPIO P9.3 as Input
	
}

/**********************************************************************/
/* 	unsigned char TestPresence(void)							      */
/*																      */
/*	Description : 		Detects if a device responds to Reset signal  */
/* 	Arguments : 		PresenceTimer - Sets timeout if no device	  */
/*							present									  */
/*						InputData - Actual state of GPIO			  */
/*						GotPulse - States if a pulse was detected	  */
/*	Global Variables:	None   										  */
/*  Returns: 			GotPulse         							  */
/**********************************************************************/
unsigned char TestPresence(void){
	unsigned int PresenceTimer;
	static volatile unsigned char InputData;
	static volatile unsigned char GotPulse;

	*GPIO9_DIR &= 0xF7;		//Set GPIO P9.3 as Input
	PresenceTimer = 300;	//Set timeout, enough time to allow presence pulse
	GotPulse = 0;			//Initialize as no pulse detected
	while ((PresenceTimer > 0) && (GotPulse == 0)) {
		InputData = GPIO_ReadBit(GPIO9, GPIO_Pin_3);	//Monitor logic state of GPIO 
		if (InputData == 0) {  		//If GPIO is Low,
			GotPulse = 1;			//it means that device responded
		}
		else {						//If GPIO is high
			GotPulse = 0;			//it means that device has not responded
			--PresenceTimer;		//Decrease timeout until enough time has been allowed for response
		}
	}
	wait_us(200);					//Wait some time to continue SDQ communication
	return GotPulse;				//Return if device detected or not
}

/**********************************************************************/
/* 	void WriteOneBit(unsigned char OneZero)						      */
/*																      */
/*	Description : 		This procedure outputs a bit in SDQ to the 	  */
/*				  		slave.								  		  */
/* 	Arguments : 		OneZero - value of bit to be written		  */
/*	Global Variables:	None   										  */
/*  Returns: 			None								          */
/**********************************************************************/
void WriteOneBit(unsigned char OneZero){
	*GPIO9_DIR |= 0x08;								//Set GPIO P9.3 as Output
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_SET);		//Set High
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_RESET);	//Set Low
	if (OneZero != 0x00){
		wait_us(2);									//approximately 7us	for a Bit 1
		GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_SET);	//Set High
		wait_us(60);								//approximately 65us
	}
	else{ 
		wait_us(60);								//approximately 65us for a Bit 0
		GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_SET);	//Set High
		wait_us(2);					   				//approximately 7us
	}
	wait_us(1);	  									//approximately 5us
}

/**********************************************************************/
/* 	void WriteOneByte(unsigned char Data2Send)					      */
/*																      */
/*	Description : 		This procedure calls the WriteOneBit() 		  */
/*				  		function 8 times to send a byte in SDQ.		  */
/* 	Arguments : 		Data2Send - Value of byte to be sent in SDQ	  */
/*	Global Variables:	None   										  */
/*  Returns: 			None								          */
/**********************************************************************/
void WriteOneByte(unsigned char Data2Send){
	unsigned char i;
	unsigned char MaskByte;
	unsigned char Bit2Send;

	MaskByte = 0x01;
	
	for (i = 0; i < 8; i++) {
		Bit2Send = Data2Send & MaskByte;		//Selects the bit to be sent
		WriteOneBit(Bit2Send);					//Writes the selected bit
		MaskByte <<= 1;							//Moves the bit mask to the next most significant position
	}
}

/**********************************************************************/
/* 	unsigned char ReadOneBit(void)								      */
/*																      */
/*	Description : 		This procedure receives the bit value returned*/
/*				  		by the SDQ slave.							  */
/* 	Arguments : 		InBit - Bit value returned by slave			  */
/*	Global Variables:	None   										  */
/*  Returns: 			InBit								          */
/**********************************************************************/
unsigned char ReadOneBit(void){
 	static unsigned char InBit;
	
	*GPIO9_DIR |= 0x08;					//Set GPIO P9.3 as Output
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_SET);		//Set High
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_RESET);	//Set Low
	*GPIO9_DIR &= 0xF7;								//Set GPIO P9.3 as Input
	wait_us(7);		   								//Strobe window	~ 12us
	InBit = GPIO_ReadBit(GPIO9, GPIO_Pin_3);		//This function takes about 3us
													//Between the wait_us and GPIO_ReadBit functions
													//approximately 15us should occur to monitor the 
													//GPIO line and determine if bit read is one or zero
	wait_us(60);									//End of Bit
	*GPIO9_DIR |= 0x08;					//Set GPIO P9.3 as Output
	GPIO_WriteBit(GPIO9, GPIO_Pin_3, Bit_SET);		//Set High
	return InBit;									//Return bit value
}

/**********************************************************************/
/* 	unsigned char ReadOneByte(void)								      */
/*																      */
/*	Description : 		This procedure reads 8 bits on the SDQ line   */
/*				  		and returns the byte value.					  */
/* 	Arguments : 		Databyte - Byte value returned by SDQ slave	  */
/*						MaskByte - Used to seperate each bit	      */
/*						i - Used for 8 time loop					  */
/*	Global Variables:	None   										  */
/*  Returns: 			DataByte							          */
/**********************************************************************/
unsigned char ReadOneByte(void){
 	unsigned char i;
	unsigned char DataByte;
	unsigned char MaskByte;

	DataByte = 0x00;			 				//Initialize return value
	
	for (i = 0; i < 8; i++) {		  			//Select one bit at a time
		MaskByte = ReadOneBit();				//Read One Bit
		MaskByte <<= i;							//Determine Bit position within the byte
		DataByte = DataByte | MaskByte;			//Keep adding bits to form the byte
	}
	return DataByte;							//Return byte value read
}
