/* revert
 * omap4_panda_aic31xx.c  --  SoC audio for TI OMAP4 Panda Board
 *
 * Author: Santosh Sivaraj <santosh.s@mistralsolutions.com>
 *
 * Based on:
 * sdp3430.c by Misael Lopez Cruz <x0052729@ti.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/gpio.h>

#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/jack.h>

#include <asm/mach-types.h>
#include <mach/hardware.h>
#include <mach/gpio.h>
#include <plat/mcbsp.h>

#include "omap-mcbsp.h"
#include "omap-pcm.h"

#include "../codecs/tlv320aic31xx.h"

#define KC1_HEADSET_DETECT_GPIO_PIN 		49


/* The following flag is used to enable/disable the DAPM support
 * for the driver 
 */

//#define DRIVER_DAPM_SUPPORT
#undef DRIVER_DAPM_SUPPORT

/* Global I2c Client Structure used for registering with I2C Bus */
static struct i2c_client *tlv320aic31xx_client;

static struct i2c_board_info tlv320aic31xx_hwmon_info = {
	I2C_BOARD_INFO("tlv320aic3110", 0x18),
};

/* Forward Declaration */
static int kc1_headset_jack_status_check (void);

/*
 * omap4_hw_params
 * This function is to configure the Machine Driver
 */
static int omap4_hw_params(struct snd_pcm_substream *substream,
			   struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	void __iomem *phymux_base = NULL;
	int ret;
 
        printk(KERN_INFO "omap4_hw_params invoked...\n");

	/* Set codec DAI configuration */
	ret = snd_soc_dai_set_fmt(codec_dai,
				  SND_SOC_DAIFMT_I2S |
				  SND_SOC_DAIFMT_NB_NF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (ret < 0) {
		printk(KERN_ERR "can't set codec DAI configuration\n");
		return ret;
	}
        printk (KERN_INFO "snd_soc_dai_set_fmt passed..\n");

	/* Set cpu DAI configuration */
	ret = snd_soc_dai_set_fmt(cpu_dai,
				  SND_SOC_DAIFMT_I2S |
				  SND_SOC_DAIFMT_NB_NF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (ret < 0) {
		printk(KERN_ERR "can't set cpu DAI configuration\n");
		return ret;
	}
        printk(KERN_INFO "snd_soc_dai_set_fmt passed...\n");

	/* Enabling the 19.2 Mhz Master Clock Output from OMAP4 for KC1 Board */
	phymux_base = ioremap(0x4a30a000, 0x1000);
	__raw_writel(0x00010100, phymux_base + 0x318);
 

	/* Mistral added the test code to configure the McBSP4 CONTROL_MCBSP_LP 
	 * register. This register ensures that the FSX and FSR on McBSP4 are
	 * internally short and both of them see the same signal from the 
	 * External Audio Codec. 
	 */
	phymux_base = ioremap (0x4a100000, 0x1000);
	__raw_writel (0xC0000000, phymux_base + 0x61c);

	/* Set the codec system clock for DAC and ADC. The
         * third argument is specific to the board being used.
         */
	ret = snd_soc_dai_set_sysclk(codec_dai, 0, AIC31XX_FREQ_19200000,
				     SND_SOC_CLOCK_IN);
	if (ret < 0) {
		printk(KERN_ERR "can't set codec system clock\n");
		return ret;
	}

	printk (KERN_INFO "omap4_hw_params passed...\n");

    return 0;
}

/* 
 * @struct omap4_ops
 *
 * Structure for the Machine Driver Operations
 */
static struct snd_soc_ops omap4_ops = {
	.hw_params = omap4_hw_params,
};

/* Headset jack information structure */
static struct snd_soc_jack hs_jack;

/* @struct hs_jack_pins
 * 
 * Headset jack detection DAPM pins 
 *
 * @pin:    name of the pin to update
 * @mask:   bits to check for in reported jack status
 * @invert: if non-zero then pin is enabled when status is not reported
 */
static struct snd_soc_jack_pin hs_jack_pins[] = {
	{
		.pin = "Headset Mic",
		.mask = SND_JACK_MICROPHONE,
	},
	{
		.pin = "Headset Stereophone",
		.mask = SND_JACK_HEADPHONE,
	},
};

/* 
 * @struct hs_jack_gpios
 * 
 * Headset jack detection gpios  
 *
 * @gpio:         Pin 49 on the KC1 Rev 1 Board
 * @name:         String Name "hsdet-gpio"
 * @report:       value to report when jack detected
 * @invert:       report presence in low state
 * @debouce_time: debouce time in ms
 */
static struct snd_soc_jack_gpio hs_jack_gpios[] = {
	{
                .gpio = KC1_HEADSET_DETECT_GPIO_PIN,
		.name = "hsdet-gpio",
		.report = SND_JACK_HEADSET,
		.debounce_time = 200,
                .jack_status_check = kc1_headset_jack_status_check, 
	},
};

#ifdef DRIVER_DAPM_SUPPORT
/* OMAP4 machine DAPM */
static const struct snd_soc_dapm_widget omap4_aic31xx_dapm_widgets[] = {
	SND_SOC_DAPM_HP ("Headphone Jack", NULL),
        SND_SOC_DAPM_SPK ("Speaker Jack", NULL),
        SND_SOC_DAPM_MIC ("Ext Mic", NULL),
        SND_SOC_DAPM_MIC ("Onboard Mic", NULL),
};

static const struct snd_soc_dapm_route audio_map[] = {
	/* External Speakers: HFL, HFR */
	{"Speaker Jack", NULL, "SPL"},
	{"Speaker Jack", NULL, "SPR"},

	/* Headset Mic: HSMIC with bias */
	{"EXTMIC", NULL, "Ext Mic"},
	{"INTMIC", NULL, "Onboard Mic"},

	/* Headset Stereophone (Headphone): HSOL, HSOR */
	{"Headphone Jack", NULL, "HPL"},
	{"Headphone Jack", NULL, "HPR"},
};

#endif

/*
 * omap4_aic31xx_init
 * This function is to initialize the machine Driver.
 */
static int omap4_aic31xx_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_codec *codec = rtd->codec;
	int ret;
        int gpiostatus;

        printk(KERN_INFO "entered the omap4_aic31xx_init function....\n");

#ifdef DRIVER_DAPM_SUPPORT
	/* Add OMAP4 specific widgets */
	ret = snd_soc_dapm_new_controls(codec->dapm, omap4_aic31xx_dapm_widgets,
					ARRAY_SIZE(omap4_aic31xx_dapm_widgets));
	if (ret) {
		printk (KERN_INFO "snd_soc_dapm_new_controls call failed. returning...\n");
                return ret;
        } 
        printk(KERN_INFO "snd_soc_dapm_new_controls passed..\n");
        printk(KERN_INFO "Disabling the DAPM Routes for now...\n");



	/* Set up OMAP4 specific audio path audio_map */
	ret = snd_soc_dapm_add_routes(codec->dapm, audio_map, ARRAY_SIZE(audio_map));

        if (ret != 0) {
		printk (KERN_INFO "snd_soc_dapm_add_routes failed... %d\n", ret);
        }

	/* OMAP4 connected pins */
	snd_soc_dapm_enable_pin(codec->dapm, "Ext Mic");
	snd_soc_dapm_enable_pin(codec->dapm, "Ext Spk");
	snd_soc_dapm_disable_pin(codec->dapm, "Headset Mic");
	snd_soc_dapm_disable_pin(codec->dapm, "Headset Stereophone");


	/* AIC31XX not connected pins */
	snd_soc_dapm_nc_pin(codec->dapm, "AUXL");
	snd_soc_dapm_nc_pin(codec->dapm, "AUXR");
	snd_soc_dapm_nc_pin(codec->dapm, "CARKITMIC");
	snd_soc_dapm_nc_pin(codec->dapm, "DIGIMIC0");
	snd_soc_dapm_nc_pin(codec->dapm, "DIGIMIC1");

	snd_soc_dapm_nc_pin(codec->dapm, "OUTL");
	snd_soc_dapm_nc_pin(codec->dapm, "OUTR");
	snd_soc_dapm_nc_pin(codec->dapm, "EARPIECE");
	snd_soc_dapm_nc_pin(codec->dapm, "PREDRIVEL");
	snd_soc_dapm_nc_pin(codec->dapm, "PREDRIVER");
	snd_soc_dapm_nc_pin(codec->dapm, "CARKITL");
	snd_soc_dapm_nc_pin(codec->dapm, "CARKITR");

	ret = snd_soc_dapm_sync(codec->dapm);
	if (ret != 0) {
		printk (KERN_INFO "snd_soc_dapm_sync failed... %d\n", ret);
		return ret;
        }  
        printk (KERN_INFO "snd_soc_dapm_sync passed..\n");

	/* Headset jack detection */
	ret = snd_soc_jack_new(codec, "Headset Jack",
			       SND_JACK_HEADSET, &hs_jack);
	if (ret != 0) {
		printk (KERN_INFO "snd_soc_jack_new failed...\n");
                return ret;
        } 

	ret = snd_soc_jack_add_pins(&hs_jack, ARRAY_SIZE(hs_jack_pins),
				    hs_jack_pins);
	if (ret != 0) {
		printk (KERN_INFO "snd_soc_jack_add_pins failed... %d\n", ret);
                return ret;
        }

	ret = snd_soc_jack_add_gpios(&hs_jack, ARRAY_SIZE(hs_jack_gpios),
				     hs_jack_gpios);
#endif
	/* Headset jack detection */
	gpiostatus = snd_soc_jack_new(codec, "Headset Jack",
			       SND_JACK_HEADSET, &hs_jack);
	if (gpiostatus != 0) {
		printk (KERN_INFO "snd_soc_jack_new failed. %d..\n", gpiostatus);
        } 

	gpiostatus = snd_soc_jack_add_pins(&hs_jack, ARRAY_SIZE(hs_jack_pins),
				    hs_jack_pins);
	if (gpiostatus != 0) {
		printk (KERN_INFO "snd_soc_jack_add_pins failed... %d\n", gpiostatus);
        }

        gpiostatus = snd_soc_jack_add_gpios(&hs_jack, ARRAY_SIZE(hs_jack_gpios),
				     hs_jack_gpios);

        if (gpiostatus != 0)
		printk (KERN_INFO "snd_soc_jack_add_gpios failed..%d\n", gpiostatus);
	return ret;
}


/*
 * kc1_headset_jack_status_check
 * This function is to check the Headset Jack Status
 */
static int kc1_headset_jack_status_check (void)
{
    int gpio_status;

    printk (KERN_INFO "##kc1_headset_jack_status_check invoked......\n");

    gpio_status = gpio_get_value (KC1_HEADSET_DETECT_GPIO_PIN);
  
    /* If the js_jack codec Member is not empty, Invoke the headset_Speaker_Path routine
     * Codec Driver.
     */
    if (hs_jack.codec != NULL)
      aic31xx_headset_speaker_path (hs_jack.codec, gpio_status);

    return 0;
}




/* Digital audio interface glue - connects codec <--> CPU */
static struct snd_soc_dai_link omap4_dai[] = {
	{
		/* .name = "AIC31XX", */
                .name = "tlv320aic3110",
		.stream_name = "AIC31XX Audio",
		.cpu_dai_name = "omap-mcbsp-dai.3",
		.codec_dai_name = "tlv320aic3110", 
/*                .no_codec = 1, */
		.platform_name = "omap-pcm-audio",
		.codec_name = "tlv320aic3110-codec", 
		.init = omap4_aic31xx_init,
		.ops = &omap4_ops,
	},
};

/* Audio machine driver */
static struct snd_soc_card snd_soc_omap4_panda = {
	.name = "OMAP4_KC1",
        .long_name = "OMAP4_KC1_AIC3110",
	.dai_link = omap4_dai,
	.num_links = 1,
};

static struct platform_device *omap4_snd_device;

/*
 * omap4_panda_soc_init
 * This function is used to initialize the machine Driver.
 */
static int __init omap4_panda_soc_init(void)
{
	int ret;
    struct i2c_adapter *adapter;

	printk(KERN_INFO "Entered omap4_panda_soc_init>...\n");

/*        if (!machine_is_omap4_panda()) {
		pr_debug("Not OMAP4 Panda board\n");
		return -ENODEV;
	}
*/
	printk(KERN_INFO "OMAP4 Panda SoC init\n");

	omap4_snd_device = platform_device_alloc("soc-audio", -1);
	
       if (!omap4_snd_device) {
		printk(KERN_ERR "Platform device allocation failed\n");
		return -ENOMEM;
	}

	platform_set_drvdata(omap4_snd_device, &snd_soc_omap4_panda);

	ret = platform_device_add(omap4_snd_device);
	if (ret) {
                printk(KERN_INFO "platform device add failed...\n");   
		goto err1;
        }
	/* For deafult OMAP4 based boards having the TWL6030 and the TWL6040
         * there is a core I2C Driver available from TI which handles all the
         * various client I2C Interactions. 
         * On boards which do not use the TWL6040, we need to utilize the
         * below block of code either in the Machine or the Codec Driver to 
         * register the Audio COdec I2C Client Device with I2C Bus Layer. 
         * In our case, we will be doing it from the Codec Driver itself.
         */
#if 0
        printk(KERN_INFO "Platform Device add success. Checking I2C Adapter..\n");

	adapter = i2c_get_adapter(3);
	if (!adapter) {
		printk(KERN_INFO "can't get i2c adapter\n");
		ret = -ENODEV;
		goto err1;
	}
        printk(KERN_INFO "i2c_get_adapter success. Creating a new i2c client device..\n");

	tlv320aic31xx_client = i2c_new_device(adapter, &tlv320aic31xx_hwmon_info);
	if (!tlv320aic31xx_client) {
		printk(KERN_INFO "can't add i2c device\n");
		ret = -ENODEV;
		goto err1;
	}
        printk (KERN_INFO "i2c_device Pntr %x\n", tlv320aic31xx_client);

 
        /* Enable the GPIO related code-base on the KC1 Board for Headphone/MIC Detection */
        if (gpio_request(OMAP4_HEADSET_MIC_DETECT_GPIO, "Headset_Mic_Detec") < 0) {
                printk (KERN_INFO "Unable to get the GPIO for Headset Mic Detection...\n");
        }
	/* Configure the GPIO as Input Pin */
        gpio_direction_input(OMAP4_HEADSET_MIC_DETECT_GPIO);
	
	/* Please note that the kernel should have enabled the interrupt for this pin. */
#endif

        printk(KERN_INFO "OMAP4 Panda Soc Init success..\n");
 
	return 0;

err1:
	printk(KERN_ERR "Unable to add platform device\n");
	platform_device_put(omap4_snd_device);

	return ret;
}
module_init(omap4_panda_soc_init);

/*
 * omap4_soc_exit
 * This function is used to exit the machine Driver.
 */
static void __exit omap4_soc_exit(void)
{
        printk("omap4_soc_init invoked...\n");

	snd_soc_jack_free_gpios(&hs_jack, ARRAY_SIZE(hs_jack_gpios),
				hs_jack_gpios);

	platform_device_unregister(omap4_snd_device);
	i2c_unregister_device(tlv320aic31xx_client);
}
module_exit(omap4_soc_exit);

MODULE_AUTHOR("Santosh Sivaraj <santosh.s@mistralsolutions.com>");
MODULE_DESCRIPTION("ALSA SoC OMAP4 Panda");
MODULE_LICENSE("GPL");
