/******************************************************************************
  Filename:        chip_information.c
  
  Description:    This file holds the chipInfoApp function which gather and 
                  displays information about the EM connected to the TrxEB.
                  The following information is displayed:
                  - Device name
                  - PG version
                  - Crystal frequency
  
  Notes: 
  
******************************************************************************/


/*****************************************************************************
* INCLUDES
*/
#include "hal_lcd_trxeb.h"
#include "hal_button_trxeb.h"
#include "trx_rf_spi.h"
#include "chip_detect.h"
#include "freq_xosc_detect.h"
#include "hal_timer_32k.h"
#include "menu_system.h"
#include "menu_driver.h"
#include "chip_information.h"

/******************************************************************************
* LOCAL VARIABLES
*/
radioChipType_t pRadioChipType;

uint16 id;
uint8  ver;
uint8  xoscFreq;

/******************************************************************************
* STATIC FUNCTIONS
*/
menu_t chipInfoMenu;

// menu item for chipInfoMenu
static const menuItem_t chipInfoMenuItems[] =
{
  {0x00,"<-"," Return to menu",0,0,0,0,0}
};

// menu to display chip information
menu_t chipInfoMenu =
{
  (menuItem_t*)chipInfoMenuItems,    // pItems          
  0,                                 // pParentMenu     
  0,                                 // pMenuGraphics   
  "Chip Information",                // pTextHeader     
  0,                                 // pTextMenuItems  
  1,                                 // nMenuItems      
  0,                                 // nCurrentItem    
  -1,                                // nSelectedItem   
  0,                                 // nScreen         
  0x7E                               // reservedAreas   
};

/******************************************************************************
 * @fn          chipInfoApp
 *
 * @brief       Detects chip partnummer, version number and crystal frequency
 *              and displays infomation on LCD.
 *                
 * @input       *pVoid - pointer to void. Not used 
 *
 * @return      0 - success
 ******************************************************************************/
uint8 chipInfoApp(void **pVoid)
{
  // Detect if a supported radio is present
  trxDetectChipType(&pRadioChipType);
  
  // Detect frequency of crystal on connected EM
  trxDetectRfCrystalFrequency(&pRadioChipType);
 
  // Get chip inf from struct
  id = pRadioChipType.deviceName;
  ver = pRadioChipType.ver;
  xoscFreq = pRadioChipType.xoscFreq;
  
  // Clear reserved area of menu to display chip info
  menuClearReservedArea(&chipInfoMenu);
  
  if(!(id | ver))
  {
    // Issue message to user to insert radio EM
    halLcdClear(0);
    halLcdPrintString(0,"Test could not",18,2);
    halLcdPrintString(0,"detect a supported",6,3);
    halLcdPrintString(0,"radio",36,4);
    halLcdSendBuffer(0);
    // Make the message visible for max 2 seconds
    halTimer32kMcuSleepTicks(TIMER_32K_CLK_FREQ);
    halTimer32kMcuSleepTicks(TIMER_32K_CLK_FREQ);
    // clear potential button pushes
    halButtonsPushed();
    
    return 1;
  }
  else
  {
    halLcdPrintString(0,"Device:",1,2);
    halLcdPrintString(0,"CC",86,2);
    if(id&0xF000)
    {
      halLcdPrintInt(0,((id&0xF000)>>12),98,2);
      halLcdPrintInt(0,((id&0x0F00)>>8),104,2);
      halLcdPrintInt(0,((id&0x00F0)>>4),110,2);
      halLcdPrintInt(0,((id&0x000F)),116,2);
    }
    else
    {
      halLcdPrintInt(0,((id&0x0F00)>>8),98,2);
      halLcdPrintInt(0,((id&0x00F0)>>4),104,2);
      halLcdPrintInt(0,((id&0x000F)),110,2);
      halLcdPrintString(0,"L",116,2); 
    }
    halLcdPrintString(0,"Version:",1,3);
    // Set version if CC112x detected
    if((id == CHIP_TYPE_CC1120) ||(id == CHIP_TYPE_CC1121) 
       ||(id == CHIP_TYPE_CC1125) ||(id ==  CHIP_TYPE_CC1175))
    {
      halLcdPrintInt(0,((ver&0xF0)>>4),105,3);
      halLcdPrintString(0,".",111,3);
      halLcdPrintInt(0,((ver&0x0F))   ,116,3);
    }
    else
    {
      halLcdPrintString(0,"n/a",104,3);
    }
   
    halLcdPrintString(0,"XOSC freq:",1,4);
    halLcdPrintInt(0,xoscFreq,91,4);
    halLcdPrintString(0,"MHz",104,4);  
 
    return 0;
  }
}
/***********************************************************************************
  Copyright 2011 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
***********************************************************************************/