/******************************************************************************
    Filename:    chip_detect.c 
    
    Description: Implementation file for radio chip type detection 
        
*******************************************************************************/

/******************************************************************************
 * INCLUDES
 */

#include "hal_types.h"
#include "chip_detect.h"
#include "trx_rf_spi.h"

/******************************************************************************
 * CONSTANTS
 */

/* Minimum definitions for reading chip ID and chip VERSION */
#define CC1101_READ_BURST               0xC0
#define CC1101_PARTNUM_ADDR             0x30
#define CC1101_VERSION_ADDR             0x31

#define CC1120_READ_BURST               0xC0
#define CC1120_EXT_MEM_ACCESS           0x2F
#define CC1120_PARTNUMBER_ADDR          0x8F
#define CC1120_PARTVERSION_ADDR         0x90



/******************************************************************************
 * LOCAL FUNCTIONS
 */


/******************************************************************************   
 * @fn          chipDetectRadio()                                       
 *                                                                                
 * @brief       This function detects if a chip is present on the EM
 *              socket. SPI initialization must be done before this function  
 *              can be called.  
 *              Note: Currently able to detect CC1101, CC110L, CC113L, CC115L
 * 
 * input parameters
 *
 * @param       pRadioChipType - pointer to radioChipType_t struct
 *
 * output parameters
 *          
 * @return      2 byte chip type
 */
static uint16 trxChipDetectRadio(radioChipType_t *pRadioChipType)
{
  volatile uint8 id;
  volatile uint8 ver;
  volatile uint16 type;
  
  // Relies on automatic POR   
  // Pull CSn low and wait for MISO goes low => clock ready 
  TRXEM_SPI_BEGIN();
  TRXEM_SPI_WAIT_MISO_LOW(id);                             // Wait for MISO Low
  if(id == 0) return CHIP_TYPE_NONE;                       // Return if failed 
  
  TRXEM_SPI_TX(CC1101_READ_BURST | CC1101_PARTNUM_ADDR);   // [7:6] = READ_BURST, [5:0] = part number address 
  TRXEM_SPI_WAIT_DONE();
  TRXEM_SPI_TX(0x00);
  TRXEM_SPI_WAIT_DONE();
  id = TRXEM_SPI_RX();
  
  
  TRXEM_SPI_TX(CC1101_READ_BURST | CC1101_VERSION_ADDR);    // [7:0] = ADDR 
  TRXEM_SPI_WAIT_DONE();
  TRXEM_SPI_TX(0x00);
  TRXEM_SPI_WAIT_DONE();
  ver = TRXEM_SPI_RX();
  TRXEM_SPI_END();
   
  if(id == 0x00 )
  {  
    switch(ver)
    {
     case 0x04:
       type = CHIP_TYPE_CC1101;
       break; 
     case 0x07:
       type = CHIP_TYPE_CC110L;
       break;
     case 0x08:
       type = CHIP_TYPE_CC113L;
       break;
     case 0x09:
       type = CHIP_TYPE_CC115L;
       break;
     default:
       type = CHIP_TYPE_NONE;
    }      
  }
  else
  {
    type = CHIP_TYPE_NONE;
  }
  
  // Populating the global radio device struct if specific radio was detected 
  if(type != CHIP_TYPE_NONE)
  {
    pRadioChipType->id = id;
    pRadioChipType->ver = ver;
    pRadioChipType->deviceName = type;
  }
  
  return type;
}


/******************************************************************************   
 * @fn          trxChipDetectCC112x()                                       
 *                                                                                
 * @brief       This function detects if a CC112x chip is present in the EM 
 *              socket. SPI initialization must be ensured before this function 
 *              can be called.  
 *              
 * input parameters
 *
 * @param       pRadioChipType - pointer to radioChipType_t struct
 *          
 * output parameters
 *
 * @return      2 byte chip type
 */
static uint16 trxChipDetectCC112x(radioChipType_t *pRadioChipType)
{
  volatile uint8 id;
  volatile uint8 ver;
  volatile uint16 type;
  
  // Pin reset 
  RF_RESET_N_PORT_SEL &= ~RF_RESET_N_PIN;
  RF_RESET_N_PORT_DIR |= RF_RESET_N_PIN;
  RF_RESET_N_PORT_OUT &= ~RF_RESET_N_PIN; // keep reset pin low 
  __delay_cycles(1000000);
  RF_RESET_N_PORT_OUT |= RF_RESET_N_PIN; // Release reset 
    
  // Pull CSn low and wait for MISO goes low => clock ready 
  TRXEM_SPI_BEGIN();
  TRXEM_SPI_WAIT_MISO_LOW(id);                             // Wait for MISO Low
  if(id == 0) return CHIP_TYPE_NONE;                       // Return if failed 
  
  TRXEM_SPI_TX(CC1120_READ_BURST | CC1120_EXT_MEM_ACCESS); // [7:6] = READ_BURST, [5:0] = extended memory access address 
  TRXEM_SPI_WAIT_DONE();
  
  TRXEM_SPI_TX(CC1120_PARTNUMBER_ADDR);                    // [7:0] = Partnumber address 
  TRXEM_SPI_WAIT_DONE();
  
  TRXEM_SPI_TX(0x00);
  TRXEM_SPI_WAIT_DONE();
  id = TRXEM_SPI_RX();
  
  TRXEM_SPI_TX(0x00);
  TRXEM_SPI_WAIT_DONE();
  ver = TRXEM_SPI_RX();
  
  TRXEM_SPI_END();
  
  switch(id)
  {
    case 0x40: //CC1121
      if(ver == 0x10) type = CHIP_TYPE_CC1121; // PG 2.0
      if(ver == 0x21) type = CHIP_TYPE_CC1121; // PG 2.1 
      break;
    case 0x48: // CC1120
      if(ver == 0x10) type = CHIP_TYPE_CC1120; // PG 2.0 
      if(ver == 0x21) type = CHIP_TYPE_CC1120; // PG 2.1 
      break;
    case 0x58: //CC1125
      if(ver == 0x20) type = CHIP_TYPE_CC1125; // PG 2.0
      if(ver == 0x21) type = CHIP_TYPE_CC1125; // PG 2.1
      break;
    case 0x5A: //CC1175
      if(ver == 0x10) type = CHIP_TYPE_CC1175; // PG 2.0
      if(ver == 0x21) type = CHIP_TYPE_CC1175; // PG 2.1
      break;
      
    default:
      type  = CHIP_TYPE_NONE;
  }
  
  // Populating the global radio device struct if specific radio was detected 
  if(type != CHIP_TYPE_NONE)
  {
    pRadioChipType->id = id;
    pRadioChipType->ver = ver;
    pRadioChipType->deviceName = type;
  }
  
  return type;
}

/******************************************************************************   
 * @fn          detectChipType()                                       
 *                                                                                
 * @brief       This function detects if a supported radio chip is present in  
 *              the EM socket.SPI init must be applied before this function   
 *              can be called. 
 *           
 * input parameters
 *      
 * @param       pRadioChipType  - holds data about a detected radio chip
 *        
 * output parameters  
 *
 * @return      uint8  - 1: radio chip detected
 *                       0: radio chip not detected
 */         
uint8 trxDetectChipType(radioChipType_t *pRadioChipType)
{
  uint16 chipType = CHIP_TYPE_NONE;
  chipType = trxChipDetectRadio(pRadioChipType);
  
  // Detect if CC112x is present if CC1101 isn't
  if(chipType == CHIP_TYPE_NONE)
  {
    chipType = trxChipDetectCC112x(pRadioChipType);
  }
  if(chipType == CHIP_TYPE_NONE)
  {
  	// Defaulting the struct values if a radio chip is not detected 
  	pRadioChipType->id  = 0x00; 
    pRadioChipType->ver = 0x00;
  	pRadioChipType->deviceName =  CHIP_TYPE_NONE;
  	return 0;
  }
  else
  {
    return 1;
  }
}  

/******************************************************************************
  Copyright 2010 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
*******************************************************************************/