/***********************************************************************************
    Filename:     hal_create_random.c

    Description:  creates a random byte using the ADC connected to the light sensor
***********************************************************************************/

/******************************************************************************
 * INCLUDES
 */
#include <msp430.h>
#include "hal_types.h"

/******************************************************************************
 * CONSTANTS
 */

/**************** Functions ***************************************************/


/******************************************************************************
 * @fn     halCreateRandomByte
 *   
 * @brief  creates a "random" byte by xor-ing three reads of the lower ADC byte
 *         that is not 0xFF or 0x00.
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      random byte
 */
uint8 halCreateRandomByte(void) {
    uint16 savedADC12MEM0 = 0;
    
    /* Sequence of channels, once per ACLK */
    ADC12CTL0 = ADC12SHT02 + ADC12REF2_5V + ADC12REFON + ADC12ON + ADC12MSC;   /* enable 2.5 V internal reference, turn ADC12 on                                */
    ADC12CTL1 = ADC12SHP + ADC12SSEL_3 + ADC12CONSEQ_0;                        /* source clock = ACLK, single channel/single conv                               */
    ADC12CTL2 = ADC12TCOFF + ADC12RES_2;                                       /* turn off temp sensor, 12 bit resolution, data in unsigned format              */
                                                                                                                                                               
    ADC12MCTL0 =  ADC12INCH_2 + ADC12SREF_0 + ADC12EOS;                         /* temporary: use AVcc and AVss as references */
    
    ADC12IFG &= ~(BIT1+BIT0);               /* Clear any pending flags */                                                                 
    ADC12CTL0 |=  ADC12ENC | ADC12SC ;      /* start conversion        */
    /*ADC12IE |= BIT2; */                   /* enable interrupt        */
    ADC12CTL0 |= ADC12REFON;                /* Turn on ADC12 reference */

    /* Delay to stabilize ADC12 reference assuming the fastest MCLK of 16 MHz.*/
    /* 35 us = 1 / 16 MHz * 560                                               */
    __delay_cycles(560);
    asm(" nop");

    /*ADC12IE |= BIT0; */                       /* Enable interrupt */
    ADC12CTL0 |=  ADC12ENC | ADC12SC;

    /* Read the ADC value 3 times and xor the values*/
    __delay_cycles(1000000);
    asm(" nop");
    savedADC12MEM0 = ADC12MEM0;              /* Store the first sampled data */
    __delay_cycles(1000000);
    asm(" nop");
    savedADC12MEM0 ^= ADC12MEM0;             /* XOR last read value with new one */
    __delay_cycles(1000000);
    asm(" nop");
    savedADC12MEM0 ^= ADC12MEM0;             /* XOR last read value with new one */
    
    ADC12CTL0 &= ~(ADC12ON);                /* Turn off ADC12 */

    if(((savedADC12MEM0 & 0x00FF)==0x00FF) || ((savedADC12MEM0 & 0x00FF)==0x0000))
    {
    	/* Defaulting the "random" value */
    	return 0x9D;
    }
    else
    {
      return (uint8) (0x00FF&savedADC12MEM0);
    }
}

/******************************************************************************
  Copyright 2010 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
*******************************************************************************/