/******************************************************************************
  Filename:   main.c

  Description: This file implements the startup of the board and and a menu
               menu driver.

******************************************************************************/

/******************************************************************************
 * INCLUDES
 */

#include  <msp430.h>
#include "hal_board.h"
#include "hal_mcu.h"
#include "hal_led_trxeb.h"
#include "hal_button_trxeb.h"
#include "hal_lcd_trxeb.h"
#include "trx_rf_spi.h"
#include "chip_detect.h"
#include "freq_xosc_detect.h"
#include "per_test.h"
#include "main_graphics.c"
#include "menu_system.h"
#include "menu_driver.h"
#include "sniff_mode.h"
#include "chip_information.h"
#include "simpleLink.h"

/******************************************************************************
* GLOBAL VARIABLES
*/
extern uint8  mclkFrequency; // Defined in hal_board.c

char version[] = "v2.0";    // Version number

/******************************************************************************
* LOCAL FUNCTIONS
*/
static uint8 contrastApp(void **pVoid);

/******************************************************************************
* The Main Menu
*/

menu_t mainMenu;
menu_t contrastMenu;

menuItem_t mainMenuItems[] =
{
  {0x00,"1","PER Test"                   ,0,&perAbstractHeadMenu    ,0,&perChipSelectApp    ,0},
  {0x00,"2","SimpleLink Test"            ,0,&simpleLinkMainMenu     ,0,&initSimpleLink      ,0}, 
  {0x00,"3","RX Sniff Test"              ,0,&sniffModeFrequencyMenu ,0,&sniffInitApp        ,0},
  {0x00,"4","Chip Information"           ,0,&chipInfoMenu          ,0,&chipInfoApp          ,0},
  {0x00,"5","Set Contrast"               ,0,&contrastMenu,0,0,0},
  {M_DISABLED,0,0,0,0,0,0,0},
  {M_DISABLED|M_STRING|M_RIGHT,0,0,version,0,0,0,0}
};

static const menuItem_t contrastMenuItems[] =
{
  {0x00,"1","High",0,0,0   ,&contrastApp,(void**)20},
  {0x00,"2","Medium",0,0,0,&contrastApp,(void**)25},
  {0x00,"3","Low",0,0,0  ,&contrastApp,(void**)30}
};

menu_t contrastMenu =
{
  (menuItem_t*)contrastMenuItems,   /* pItems          */
  &mainMenu,                        /* pParentMenu     */
  0,                                /* pMenuGraphics   */
  "Select Contrast",                /* pTextHeader     */
  "3",                              /* pTextMenuItems  */
  3,                                /* nMenuItems      */
  0,                                /* nCurrentItem    */
  0,                                /* nSelectedItem   */
  0,                                /* nScreen         */
  0                                 /* reservedAreas   */
};


menu_t mainMenu =
{
  (menuItem_t*)mainMenuItems,   /* pItems          */
  0,                            /* pParentMenu     */
  0,                            /* pMenuGraphics   */
  "Main Menu",                  /* pTextHeader     */
  "5",                          /* pTextMenuItems  */
  7,                            /* nMenuItems      */
  0,                            /* nCurrentItem    */
  -1,                           /* nSelectedItem   */
  0,                            /* nScreen         */
  0                             /* reservedAreas   */
};

/******************************************************************************
 * @fn          main
 *
 * @brief       Main handles all applications attached to the menu system
 *
 * input parameters
 *
 * output parameters
 *
 *@return
 */
void main( void )
{
  /* Stop watchdog timer to prevent time out reset */
  WDTCTL = WDTPW + WDTHOLD;

  /* Settingcapacitor values for XT1, 32768 Hz */
  halMcuStartXT1();

  /* Clocks:
   * mclk  = mclkFrequency
   * smclk = mclkFrequency
   * aclk  = 32768 Hz
   */
  mclkFrequency = HAL_MCU_SYSCLK_16MHZ;
  halMcuSetSystemClock(mclkFrequency);

  /* Care must be taken when handling power modes
   * - Peripheral units can request clocks and have them granted even if
   *   the system is in a power mode. Peripheral clock request is enabled
   *   as default.
   * - Per test only needs ACLK to be enabled to timers
   *   during power mode operation
   */
  halMcuDisablePeripheralClockRequest((MCLKREQEN+SMCLKREQEN));


  /* SPI flash uses same SPI interface as LCD -- we'll disable the SPI flash */
 P8SEL &= BIT6; /*ioflash_csn = gp.      */
 P8DIR |= BIT6; /*tpflash_csn = ouut.    */
 P8OUT |= BIT6; /*flash_csn = 1.         */



  /* Init leds and turn them on */
  halLedInit();

  /* Init Buttons */
  halButtonsInit();
  halButtonsInterruptEnable();

  /* Init LCD and issue a welcome */
  halLcdInit();
  halLcdClear(0);

  /* Instantiate tranceiver RF spi interface to SCLK = 1 MHz */
  trxRfSpiInterfaceInit(0x10);

  
  /* Welcome Screen Part */
  halLcdWriteBufferDirectly(trxebWelcomeScreen, 1024);
  halLcdPrintString(0,"TEXAS",60,6);
  halLcdPrintString(0,"INSTRUMENTS",60,7);
  halLcdSendBuffer(0);
  /* MCU will in sleep until button is pressed */
  __low_power_mode_3();
  halButtonsPushed();
  //Clear screen
  halLcdClear(0);

  /* Menu Driver */
  menu_t *pCurrentMenu = &mainMenu;
  uint8 menuButtonsPressed;
  menuDisplay(pCurrentMenu);
  __low_power_mode_3();
  while(1)
  {
    menuButtonsPressed = halButtonsPushed();
    switch(menuButtonsPressed)
    {
      case HAL_BUTTON_LEFT:
        pCurrentMenu = menuBack(pCurrentMenu);
        break;
      case HAL_BUTTON_RIGHT:
        pCurrentMenu = menuEnter(pCurrentMenu);
        break;
      case HAL_BUTTON_DOWN:
        menuDown(pCurrentMenu);
        break;
      case HAL_BUTTON_UP:
        menuUp(pCurrentMenu);
        break;
      default:
        break;
    }
    menuDisplay(pCurrentMenu);
    /* Enter low power mode while menu driver waits on user input */
    __low_power_mode_3();
  }
}


static uint8 contrastApp(void **pVoid)
{
  uint16 c = (uint16)*pVoid;
  halLcdSetContrast(c&0x00FF);
  return 0;
}

/******************************************************************************
  Copyright 2010 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
*******************************************************************************/