/******************************************************************************
  Filename:      cc1120_per_test_api.c
  
  Description:   Implementation file for api-like functions that the per test
                 will call if a CC1120_CC1190 combo was selected.             
  
                 CC1120_CC1190 will support the PER test with the following data rates:
                 - 1.2 Kb/s  with 25 KHz bandwidth, almost tc_01, named 
                   cc112xLowDataRateRfSettings, hence more noise on RSSI samples and lower
                   sensitivity.
  
******************************************************************************/

/******************************************************************************
 * INCLUDES
 */
#include "hal_types.h"
#include "hal_timer_32k.h"
#include "per_test.h"
#include "trx_rf_spi.h"
#include "trx_rf_int.h"
#include "cc112x_spi.h"
#include "cc1120_cc1190_per_test_api.h"
#include "freq_xosc_detect.h"


/******************************************************************************
 * TYPEDEFS
 */


/******************************************************************************
 * LOCAL FUNCTIONS
 */

static void cc1120cc1190RxIdle(void);
static void cc1120cc1190IdleRx(void);   



/* cc1190 pa and lna functions */
void perCC1120CC1190PaEnable(void);
void perCC1120CC1190PaDisable(void);
void perCC1120CC1190LnaEnable(void);
void perCC1120CC1190LnaDisable(void);
void perCC1120CC1190HgmEnable(void);
void perCC1120CC1190HgmDisable(void);
/******************************************************************************
 * CONSTANTS
 */
 
/********************************* TestCases *********************************/
   // Base settings: Variable packet length, CRC calculation, no Address check, 
   // Append Mode, Always give Clear channel indication 
   //

// Modem settings for test case 1 adjusted to 4 Byte preamble(")*/ 
static const registerSetting_t cc1120cc1190LowDataRateRfSettings[] =  
{
  {CC112X_IOCFG0            ,0x06}, // Route sync signal to GPIO0
  {CC112X_FS_DIG1           ,0x00},
  {CC112X_FS_DIG0           ,0x5F},
  {CC112X_FS_CAL1           ,0X40},
  {CC112X_FS_CAL0           ,0x0E},
  {CC112X_FS_DIVTWO         ,0x03},
  {CC112X_FS_DSM0           ,0x33},
  {CC112X_FS_DVC0           ,0x17},  
  {CC112X_FS_PFD            ,0x50},  
  {CC112X_FS_PRE            ,0x6E},
  {CC112X_FS_REG_DIV_CML    ,0x14},
  {CC112X_FS_SPARE          ,0xAC},
  {CC112X_FS_VCO0           ,0xB4},
  {CC112X_XOSC5             ,0x0E},
  {CC112X_XOSC4             ,0xA0},
  {CC112X_XOSC1             ,0x03},
  {CC112X_ANALOG_SPARE      ,0x00},
  {CC112X_FIFO_CFG          ,0x00},
  {CC112X_DEV_ADDR          ,0x00},  
  {CC112X_SETTLING_CFG      ,0x0B},
  {CC112X_FS_CFG            ,0x12}, //////////////////////////////////////////////////////////////////
  {CC112X_PKT_CFG2          ,0x00},
  {CC112X_PKT_CFG1          ,0x05}, // Address check off and CRC check on
  {CC112X_PKT_CFG0          ,0x20},  
  {CC112X_RFEND_CFG1        ,0x0F}, // Stay in RX after RX, No timeout for sync word search  
  {CC112X_RFEND_CFG0        ,0x00}, // IDle after TX, no interferring from MARC  
  {CC112X_SYNC3             ,0x93},/////////////////////////////////////////////////////////////////////
  {CC112X_SYNC2             ,0x0B}, 
  {CC112X_SYNC1             ,0x51}, 
  {CC112X_SYNC0             ,0xDE}, 
  {CC112X_SYNC_CFG1         ,0x0B}, 
  {CC112X_SYNC_CFG0         ,0x17}, 
  {CC112X_DEVIATION_M       ,0x06}, // (4000 Hz)       
  {CC112X_MODCFG_DEV_E      ,0x03}, // (4000 Hz)       
  {CC112X_DCFILT_CFG        ,0x1C}, //                 
  {CC112X_PREAMBLE_CFG1     ,0x18}, // 4" byte preamble 
  {CC112X_PREAMBLE_CFG0     ,0x2A}, //                 
  {CC112X_FREQ_IF_CFG       ,0x40}, // (62500 Hz)      
  {CC112X_IQIC              ,0xC6}, //                 
  {CC112X_CHAN_BW           ,0x14}, // (10000" Hz)      
  {CC112X_MDMCFG1           ,0x46}, //                 
  {CC112X_MDMCFG0           ,0x05}, //                 
  {CC112X_DRATE2            ,0x43}, // (1200 bps)      
  {CC112X_DRATE1            ,0xA9}, // (1200 bps)      
  {CC112X_DRATE0            ,0x2A}, // (1200 bps)      
  {CC112X_AGC_REF           ,0x20}, 
  {CC112X_AGC_CS_THR        ,0x19}, 
  {CC112X_AGC_GAIN_ADJUST   ,0x00}, 
  {CC112X_AGC_CFG3          ,0x91}, 
  {CC112X_AGC_CFG2          ,0x20}, 
  {CC112X_AGC_CFG1          ,0xA9}, 
  {CC112X_AGC_CFG0          ,0xCF}, 
  {CC112X_PA_CFG2           ,0x7F}, 
  {CC112X_PA_CFG1           ,0x56}, 
  {CC112X_PA_CFG0           ,0x7C}, 
  {CC112X_IF_MIX_CFG        ,0x00}, 
  {CC112X_FREQOFF_CFG       ,0x22}, 
  {CC112X_TOC_CFG           ,0x0B},
  {CC112X_SOFT_TX_DATA_CFG  ,0x00}
};



//Band select settings for the LO divider.
//out of lock detector enabled
static uint8 cc112xFsCfgs[1] = 
{
  0x12, // 869 MHz 
};
//Frequency programming for CC1120_CC1190 Combo
// Currently only one frequency option due to
// output power restrictions in other bands.
// perSettings.frequencyBand = 0 => 869 MHz 

// For CC112x with 32 MHz XOSC
static uint8 freqConfiguration[1][3] =
{
  {0x6C,0xB0,0xCD}, // 869.525 MHz   
};


/* Sensitivity table - Note: It's only valid for 3 bytes packets and for shipped register settings */
/* Used in Link Margin calculation */  
static const int8 sensitivity869Mhz[1] = 
{
  -125
};

/* RSSI offset table */
static const int8 rssiOffset869Mhz[1] = 
{
  101 
};

/* Values in this table must be ored with the PA_CFG2 register to account for  
 * pa_shape_en
 * Formula: paPowerRamp[index] = dec2hex((wanted_dBm_level+18)*2-1)
 */
static const uint8 paPowerRamp[2] = 
{
  0x03, /* -00 dBm - lowest power       - index 0 */
  0x37  /*  26 dBm - highest power - index 1*/
};
/* Access by index gives GUI values for TX power corresponding to the table above */
static const int8 paPowerGuiValues[2]=
{
    0,  /* - index 0 */
   26,  /* - index 1 */
};

static const float testCaseDataRate[1]=
{
    1.20,  /* testcase 1   <=> SMARTRF_CONFIGURATION_0 */
};
 
/******************************************************************************
 * LOCAL VARIABLES
 */

/* Variable is CC112X_STATE_RX when in RX. If not in RX it is CC112X_STATE_TX or 
 * CC112X_STATE_IDLE. The use of this variable is only to avoid an RX interrupt
 * when sending a packet. This facilitates a reduction in power consumption.
 */
static rfStatus_t cc1120cc1190RadioTxRx; 
static int8 cc1120cc1190RssiOffset; 
 

/******************************************************************************
* @fn          perCC1120CC1190GetGuiTxPower
*
* @brief       Returns the TX power in [dBm] used by the menu system. 
*              Implemented by LUT.
*
* input parameters
*
* @param       index - index to GUI TX power LUT
*                  
* output parameters
*
* @return      TX power [dBm]
*/
int8 perCC1120CC1190GetGuiTxPower(uint8 index)
{
  return paPowerGuiValues[index];
}
/******************************************************************************
* @fn          perCC1120CC1190GetDataRate
*
* @brief       Returns the data rate corresponding to the selected
*              Smart RF configuration
*
* input parameters
*
* @param       index - index to data rate table 
*                  
* output parameters
*
* @return      data rate
*/
float perCC1120CC1190GetDataRate(uint8 index)
{
  return testCaseDataRate[index];
}


/******************************************************************************
* @fn          perCC1120CC1190SetOutputPower
*
* @brief       Configures the output power of CC112x according to the provided
*              index:
*              0 =  xx dBm
*              1 =  26 dBm

*              NOTE: for PG2.0 pa_shape_en and pa_power_ramp has swapped 
*                    position
*
*
*
* input parameters
*
* @param       index - index to table <=> wanted output level
*                  
* output parameters
*
* @return      void
*/
void perCC1120CC1190SetOutputPower(uint8 index)
{
  uint8 level; 
  
  /* Reading the PA_CFG2 value to account for pa_shape_en */
  cc112xSpiReadReg(CC112X_PA_CFG2,&level,1);
  /* Saving pa_shape_en */
  level &= 0x40;
  /* Oring in the PA power ramp value */
  level |= paPowerRamp[index];
  /* Updating PA_CFG2 register with its' new value */
  cc112xSpiWriteReg(CC112X_PA_CFG2,&level,1);
  return;
}

/******************************************************************************
 * @fn          perCC1120CC1190RegConfig
 *
 * @brief       Configures the CC1120 radio with the selected 
 *              paramteres and test properties or uses the base configuration 
 *              with no address check. Assumes that the radio is in IDLE.
 *
 * input parameters
 *
 * output parameters
 *
 * @return      void
 */
void perCC1120CC1190RegConfig(void)
{
  uint8 data;
  /* Log that radio is in IDLE state */
  cc1120cc1190RxIdle();
  cc1120cc1190RadioTxRx = CC112X_STATE_IDLE;
    
  /* Extract what radio configuration to use */
  /* Only one configuration implemented so far, but keeping structure to add
     more configurations later */
  if((perSettings.masterSlaveLinked==PER_DEVICE_LINKED) || 
      (perSettings.masterSlaveLinked == PER_DEVICE_LINK_BYPASS))
  {
    switch(perSettings.smartRfConfiguration)
    {
      default:
        for(uint16 i = 0; i < (sizeof cc1120cc1190LowDataRateRfSettings/sizeof(registerSetting_t));i++)
        {
          data = cc1120cc1190LowDataRateRfSettings[i].data;
          cc112xSpiWriteReg(cc1120cc1190LowDataRateRfSettings[i].addr,&data,1);
        }
        perSettings.sensitivity = sensitivity869Mhz[0];
        cc1120cc1190RssiOffset = rssiOffset869Mhz[0];
        break;
    }
  }
  else
  {
    /* Base settings for communication */
        for(uint16 i = 0; i < (sizeof cc1120cc1190LowDataRateRfSettings/sizeof(registerSetting_t));i++)
        {
          data = cc1120cc1190LowDataRateRfSettings[i].data;
          cc112xSpiWriteReg(cc1120cc1190LowDataRateRfSettings[i].addr,&data,1);
        }
        perSettings.sensitivity = sensitivity869Mhz[0];
        cc1120cc1190RssiOffset = rssiOffset869Mhz[0];
  }
  /* Correct for chosen frequency band */     
  switch(perSettings.frequencyBand)      
  {    
    default: // 869.525 MHz
      cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[0],1);
      cc112xSpiWriteReg(CC112X_FREQ2,freqConfiguration[0],3);
      break;
  }    


  if(perSettings.masterSlaveLinked==PER_DEVICE_LINKED)
  {
    /* PKT_LEN set to user specified packet length: HW length filtering */
    cc112xSpiWriteReg(CC112X_PKT_LEN, &(perSettings.payloadLength),1);
    /* Turn on HW Address filtering */
    uint8 pkt_cfg1 = 0x10|0x05;
    cc112xSpiWriteReg(CC112X_PKT_CFG1,&pkt_cfg1,1);
    /* Set address */
    cc112xSpiWriteReg(CC112X_DEV_ADDR,&perSettings.address,1);
    /* Note: The Two-way link for the CC1120 PG1.0 use a different implementation
     *       of the two-way link on the master side than what CC1101 and CC1120 PG2.0 
     *       will. RXOFF and TXOFF mode is in this case the same as for the one-way link.
     *       
     */
    if((perSettings.linkTopology == LINK_2_WAY))
    {
      if(perSettings.deviceMode == MASTER_DEVICE)
      {
        /* IDLE after RX, RX after TX */
        data = 0x0F;
        cc112xSpiWriteReg(CC112X_RFEND_CFG1,&data,1);
        data = 0x30;
        cc112xSpiWriteReg(CC112X_RFEND_CFG0,&data,1);
      }
    } 
  }
  else
  {
    /* length of configuration packet + filter byte */
    data = PER_SETTINGS_PACKET_LEN; 
    cc112xSpiWriteReg(CC112X_PKT_LEN, &data,1);
  }  
  return;  
}

/******************************************************************************
 * @fn          perCC1120CC1190SendPacket
 *
 * @brief       Sends the contents that pData points to which has the 
 *              following structure:
 *
 *              txArray[0] = length byte
 *              txArray[n] = payload[n]
 *              | n<[sizeOf(RXFIFO)-2], variable packet length is assumed.
 * 
 *              The radio state after completing TX is dependant on the 
 *              CC112X_RFEND_CFG0 register setting. For PG10 this register 
 *              dictates IDLE after TX. For PG.0 this function must be 
 *              re-implemented since the 2way PER test relies on RX after TX.
 *              This function enables SYNC interrupt. This means that 
 *              an interrupt will fire when a packet has been sent, i.e sync 
 *              signal transitions from high to low. 
 *             
 *              The One-Way PER test disables the sync pin interrupt when TX
 *              finishes, while the Two-Way PER test doesn't to enable quick
 *              reception of Slave ACK.
 *
 *              Note: Assumes chip is ready
 *
 * input parameters
 *              
 * @param       *pData - pointer to data array that starts with length byte
 *                       and followed by payload.
 * output parameters
 *
 * @return      void
 */
void perCC1120CC1190SendPacket(uint8 *pData)
{
  uint8 len = *pData;
  /* PG1.0 errate fix: Before entering TX, the frequency word must be altered from that of RX */
  /* This means in general that TX from Idle is the only option, not TX from RX */
  perCC1120CC1190EnterIdle();
  /* Will only try to transmit if the whole packet can fit i RXFIFO 
   * and we're not currently sending a packet.
   */
  if(!(len > (PER_MAX_DATA-2)) && (cc1120cc1190RadioTxRx != CC112X_STATE_TX) )
  {
    cc112xSpiWriteTxFifo(pData,(len+1));
    /* Indicate state to the ISR and issue the TX strobe */
    trxEnableInt();
    /* Enable PA on CC1190 and be sure LNA is off */
    //perCC1190HgmEnable();
    perCC1120CC1190LnaDisable();
    perCC1120CC1190PaEnable();
    cc1120cc1190RadioTxRx = CC112X_STATE_TX; 
    trxSpiCmdStrobe(CC112X_STX);
    /* Wait until packet is sent before doing anything else */
    __low_power_mode_3();

    /* This function will not return before the complete packet
     * is sent and the radio is back in IDLE. The MSP will be 
     * be sleeping while the packet is beeing sent unless waken
     * by button presses.
     */
    while(cc1120cc1190RadioTxRx == CC112X_STATE_TX);
    
    if(perSettings.linkTopology == LINK_1_WAY)
    {
      /* Back in Idle*/
      trxDisableInt();
      /* Turn off PA on CC1190 */
      perCC1120CC1190PaDisable();
    }
  }
  return;
}


/******************************************************************************
 * @fn          perCC1120CC1190EnterRx
 *
 * @brief       Enters RX from IDLE. Function is used to abstract the 
 *              cc1120cc1190RadioTxRx functionality away from the lower-level radio 
 *              interface.
 *              Note: assumes chip is ready and in IDLE
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC1120CC1190EnterRx(void)
{
  perCC1120CC1190EnterIdle();
  cc1120cc1190IdleRx();
  cc1120cc1190RadioTxRx = CC112X_STATE_RX;
  return;
}


/******************************************************************************
 * @fn          perCC1120CC1190EnterSleep
 *
 * @brief       Enters Sleep. Function is used to abstract the cc1120cc1190RadioTxRx
 *              functionality away from the lower-level radio interface.
 *              Note: assumes chip is ready
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC1120CC1190EnterSleep(void)
{
  cc1120cc1190RxIdle();
  trxSpiCmdStrobe(CC112X_SPWD);
  /* Only important to differ between RX/TX and IDLE */
  cc1120cc1190RadioTxRx = CC112X_STATE_IDLE;
  return;
}


/******************************************************************************
 * @fn          perCC1120CC1190EnterIdle
 *
 * @brief       Enters IDLE from ANY state. Function is used to abstract the 
 *              cc1120cc1190RadioTxRx functionality away from the lower-level radio 
 *              interface.
 *           
 * input parameters
 *   
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC1120CC1190EnterIdle(void)
{
  /* wait until chip is ready */
  TRXEM_SPI_BEGIN();
  while(TRXEM_PORT_IN & TRXEM_SPI_MISO_PIN);
  cc1120cc1190RxIdle();
  cc1120cc1190RadioTxRx = CC112X_STATE_IDLE;
  return;
}

/******************************************************************************
 * @fn          perCC1120CC1190RxTxISR
 *
 * @brief       ISR that's called when sync signal goes low. 
 *              In RX State: Filters incoming data. The global rxData pointer
 *              always points to this functions static rxData_tmp(struct of
 *              same kind). The validnes of rxData fields is indicated by the
 *              the global flag packetSemaphore.
 *              In TX State: Nothing is done except it facilitates power 
 *              consumption reduction when TX since the program doesn't need
 *              to wait until TX is done before re-enabling sync pin interrupt.
 *              cc1120cc1190RadioTxRx is also set to CC112X_STATE_IDLE to be consistent 
 *              with program.
 * 
 * input parameters
 *             
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC1120CC1190RxTxISR(void)
{
  uint8 rxBytes,rxLength,rssiIndex,lqiIndex;
  /* This variable stores the data locally. Access is given to per_test by 
   * assigning this instance to the global rxData pointer
   */
  static rxData_t rxData_tmp;
            
  rxData = &rxData_tmp;
  
  /* Checking if the chip is in RX state:  */
  if(cc1120cc1190RadioTxRx != CC112X_STATE_RX)
  {
    /* Transmission finished */
    if((perSettings.deviceMode == MASTER_DEVICE) && (perSettings.linkTopology == LINK_2_WAY) && (perSettings.masterSlaveLinked ==PER_DEVICE_LINKED))
    {
      /* Only applicable when master in 2-way test */
      cc1120cc1190RadioTxRx=CC112X_STATE_RX;
    }
    else
    {
      cc1120cc1190RadioTxRx  = CC112X_STATE_IDLE;
    }
    return;
  }
  
  packetSemaphore |= SYNC_FOUND;
  
  if((perSettings.masterSlaveLinked == PER_DEVICE_LINKED) && (perSettings.deviceMode == MASTER_DEVICE) && (perSettings.testRunning == PER_TRUE))
  {
  	if(perSettings.linkTopology == LINK_1_WAY)
  	{
  	  /* Read timer value and set the perSettings.packetRate valu(adjustment for temperature drift */
      halTimer32kSetIntFrequency(perSettings.packetRate);
      halTimer32kIntEnable();
    }
    else
    {
    	/* LINK_2_WAY */ 
    	
    	/* Timeout interrupt configuring is handled by the 2-way Per test */
      timer32kValue = halTimer32kReadTimerValue();
    	halTimer32kAbort();
    }
  }
    
  cc112xSpiReadReg(CC112X_NUM_RXBYTES,&rxBytes,1);
  /* Checking if the FIFO is empty */
  if(rxBytes == PER_FALSE)
  {
    /* The packet was removed by HW due to addr or length filtering -> Do nothing */
    /* Report that a sync was detected */ 
    rxData_tmp.rssi = perCC1120CC1190Read8BitRssi();
    return;
  }
  else
  {
    /* The RX FIFO is not empty, process contents */    
    cc112xSpiReadRxFifo(&rxLength, 1);  
    /* Check that the packet length just read + FCS(2B) + length byte match the RXBYTES */
    /* If these are not equal:
     * - RXFIFO overflow: Received packets not processed while beeing in RX. 
     */
    if(rxBytes != (rxLength+3))
    {
      /* This is a fault FIFO condition -> clean FIFO and register a sync detection */
      /* IDLE -> FLUSH RX FIFO -> RX */
      cc1120cc1190RxIdle();     
      perCC1120CC1190EnterRx(); 
      /* Report that a sync was detected */
      rxData_tmp.rssi = perCC1120CC1190Read8BitRssi();
      return;
    }
    else
    {
      /* We don't have a FIFO error condition -> get packet */
      
      /* Length Field */
      rxData_tmp.data[0] = rxLength;
      rssiIndex = rxLength+1;
      lqiIndex  = rssiIndex +1;
      
      /* Payload(ADDR + DATA + FCS) */
      cc112xSpiReadRxFifo(&rxData_tmp.data[1], lqiIndex);
      
      /* The whole packet has been read from the FIFO.
       * Check if the CRC is correct and that the packet length is as expected.
       * If not correct: report sync found and do not update RSSI or LQI.
       */
      if((!(rxData_tmp.data[lqiIndex] & CC112X_LQI_CRC_OK_BM)) || (perSettings.payloadLength != rxLength ))
      {
        rxData_tmp.rssi = perCC1120CC1190Read8BitRssi();
        return;
      }
      /* A complete error-free packet has arrived  */
      
      /* Measured data */
      rxData_tmp.length  = rxLength;
      rxData_tmp.lqi     = rxData_tmp.data[lqiIndex] & CC112X_LQI_EST_BM;
      rxData_tmp.addr    = rxData_tmp.data[1]; 
      
      /* Convert RSSI value from 2's complement to decimal value accounting for offset value */
      rxBytes = rxData_tmp.data[rssiIndex];        
      rxData_tmp.rssi = (int16)((int8)rxBytes) - cc1120cc1190RssiOffset;
      /* Signal a good packet is received */
      packetSemaphore |= PACKET_RECEIVED;
      return;
    } 
  }   
}     
      
/*******************************************************************************
 * @fn          perCC1120CC1190Read8BitRssi
 *    
 * @brief       Reads MSB RSSI value from register, converts the dBm value to
 *              decimal and adjusts it according to RSSI offset
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      decimal RSSI value corrected for RSSI offset
 */ 
int8 perCC1120CC1190Read8BitRssi(void)
{
  uint8 rssi2compl,rssiValid;
  int16 rssiConverted;
  
  cc112xSpiReadReg(CC112X_RSSI0, &rssiValid,1);
  if(rssiValid & 0x01)
  {
    /* Read RSSI from MSB register */
    cc112xSpiReadReg(CC112X_RSSI1, &rssi2compl,1);
    rssiConverted = (int16)((int8)rssi2compl) - cc1120cc1190RssiOffset;
    return rssiConverted;
  }
  /* keep last value since new value is not valid */
  return rxData->rssi;
}

/*******************************************************************************
 * @fn          cc1120cc1190RxIdle
 *
 * @brief       Radio state is switched from RX to IDLE
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */ 
static void cc1120cc1190RxIdle(void)
{
  /* Disable pin interrupt */
  trxDisableInt();
  /* Strobe IDLE */
  trxSpiCmdStrobe(CC112X_SIDLE); 
  /* Wait until chip is in IDLE */
  while(trxSpiCmdStrobe(CC112X_SNOP) & 0xF0);
  //Disable LNA
  perCC1120CC1190LnaDisable();
  /* Flush the Receive FIFO */
  trxSpiCmdStrobe(CC112X_SFRX);
  /* Clear pin interrupt flag */
  trxClearIntFlag();
  return;
}

/*******************************************************************************
 * @fn          cc1120cc1190IdleRx
 *
 * @brief       Radio state is switched from Idle to RX. Function assumes that
 *              radio is in IDLE when called. 
 * 
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */ 
static void cc1120cc1190IdleRx(void)
{
  trxClearIntFlag();
  //perCC1190HgmEnable();
  perCC1120CC1190PaDisable();
  perCC1120CC1190LnaEnable();
  trxSpiCmdStrobe(CC112X_SRX);
  trxEnableInt();
  return;
}
/*******************************************************************************
 * @fn          perCC1120cc1190WriteTxFifo
 *
 * @brief       Means for PER test to write the TX FIFO
 * 
 * input parameters
 *
 * @param       *pData  - pointer to data array that will be written to TX Fifo
 * @param       len     - number of bytes in that will be written to TX Fifo
 *
 * output parameters
 *
 * @return      void
 */ 
void perCC1120CC1190WriteTxFifo(uint8 *pData, uint8 len)
{
  cc112xSpiWriteTxFifo(pData,len);
  return;
}

/*******************************************************************************
 * @fn                  perCC1120CC1190PaEnable
 *
 * @brief               Enables PA on CC1190
 * 
 * input parameters
 *
 * @param       
 *
 * output parameters
 *
 * @return      none
 */ 
void perCC1120CC1190PaEnable(void)
{
    TRXEM_CC1190_PORT_SEL &= ~(TRXEM_CC1190_PA);  // Set pin to zero for I/O function
    TRXEM_CC1190_PORT_DIR |=  (TRXEM_CC1190_PA);  // Set pin direction to output
    TRXEM_CC1190_PORT_OUT |=  (TRXEM_CC1190_PA);  // Set output pin high
}
/*******************************************************************************
 * @fn                  perCC1190LnaEnable
 *
 * @brief               Enables LNA on CC1190
 * 
 * input parameters
 *
 * @param       
 *
 * output parameters
 *
 * @return      none
 */ 
void perCC1120CC1190LnaEnable(void)
{
    TRXEM_PORT_SEL &= ~(TRXEM_CC1190_LNA);
    TRXEM_PORT_DIR |=  (TRXEM_CC1190_LNA);
    TRXEM_PORT_OUT |=  (TRXEM_CC1190_LNA);
}
/*******************************************************************************
 * @fn                  perCC1190PaDisable
 *
 * @brief               Disables PA on CC1190
 * 
 * input parameters
 *
 * @param       
 *
 * output parameters
 *
 * @return      none
 */ 
void perCC1120CC1190PaDisable(void)
{
    TRXEM_CC1190_PORT_SEL &= ~(TRXEM_CC1190_PA);  // Set pin to zero for I/O function
    TRXEM_CC1190_PORT_DIR |=  (TRXEM_CC1190_PA);  // Set pin direction to output
    TRXEM_CC1190_PORT_OUT &= ~(TRXEM_CC1190_PA);  // Set output pin low
}
/*******************************************************************************
 * @fn                  perCC1190LnaDisable
 *
 * @brief               Disables LNA on CC1190
 * 
 * input parameters
 *
 * @param       
 *
 * output parameters
 *
 * @return      none
 */ 
void perCC1120CC1190LnaDisable(void)
{
    TRXEM_PORT_SEL &= ~(TRXEM_CC1190_LNA);
    TRXEM_PORT_DIR |=  (TRXEM_CC1190_LNA);
    TRXEM_PORT_OUT &= ~(TRXEM_CC1190_LNA);    
}
/*******************************************************************************
 * @fn                  perCC1190HgmEnable
 *
 * @brief               Enables HGM on CC1190
 * 
 * input parameters
 *
 * @param       
 *
 * output parameters
 *
 * @return      none
 */ 
void perCC1120CC1190HgmEnable(void)
{
    TRXEM_PORT_SEL &= ~(TRXEM_CC1190_HGM);
    TRXEM_PORT_DIR |=  (TRXEM_CC1190_HGM);
    TRXEM_PORT_OUT |=  (TRXEM_CC1190_HGM);
}
/*******************************************************************************
 * @fn                  perCC1190HgmDisable
 *
 * @brief               Disables HGM on CC1190
 * 
 * input parameters
 *
 * @param       
 *
 * output parameters
 *
 * @return      none
 */ 
void perCC1120CC1190HgmDisable(void)
{
    TRXEM_PORT_SEL &= ~(TRXEM_CC1190_HGM);
    TRXEM_PORT_DIR |=  (TRXEM_CC1190_HGM);
    TRXEM_PORT_OUT &= ~(TRXEM_CC1190_HGM);
}
/******************************************************************************
  Copyright 2011 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
*******************************************************************************/

