/******************************************************************************
  Filename:      cc1120_per_test_api.c
  
  Description:   Implementation file for api-like functions that the per test
                 will call if a CC1120 was detected.             
  
                 CC1120 will support the PER test with the following data rates:
                 - 50.00 Kb/s
                 - 1.2 Kb/s  with 25 KHz bandwidth, almost tc_01, named 
                   cc112xLowDataRateRfSettings, hence more noise on RSSI samples and lower
                   sensitivity.
                 - 150 kbps 4-FSK
  
******************************************************************************/

/******************************************************************************
 * INCLUDES
 */
#include "hal_types.h"
#include "hal_timer_32k.h"
#include "per_test.h"
#include "trx_rf_spi.h"
#include "trx_rf_int.h"
#include "cc112x_spi.h"
#include "cc112x_per_test_api.h"
#include "freq_xosc_detect.h"


/******************************************************************************
 * TYPEDEFS
 */


/******************************************************************************
 * LOCAL FUNCTIONS
 */

static void perCC112xRxIdle(void);
static void perCC112xIdleRx(void);   

void writeRegisterConfiguration(const registerSetting_t *pRegisterSettings);

/******************************************************************************
 * CONSTANTS
 */
 
/********************************* TestCases *********************************/
   // Base settings: Variable packet length, CRC calculation, no Address check, 
   // Append Mode, Always give Clear channel indication 
   //

// Modem settings for test case 1 adjusted to 25kHz RXBW and 4 Byte preamble(")*/ 
static const registerSetting_t cc112xLowDataRateRfSettings[] =  
{
  {CC112X_IOCFG0            ,0x06}, // Route sync signal to GPIO0
  {CC112X_FS_DIG1           ,0x00},
  {CC112X_FS_DIG0           ,0x5F},
  {CC112X_FS_CAL0           ,0x0E},
  {CC112X_FS_DIVTWO         ,0x03},
  {CC112X_FS_DSM0           ,0x33},
  {CC112X_FS_DVC0           ,0x17},  
  {CC112X_FS_PFD            ,0x50},  
  {CC112X_FS_PRE            ,0x6E},
  {CC112X_FS_REG_DIV_CML    ,0x14},
  {CC112X_FS_SPARE          ,0xAC},
  {CC112X_XOSC5             ,0x0E},
  {CC112X_XOSC4             ,0xA0},
  {CC112X_XOSC3             ,0xC7},  
  {CC112X_XOSC1             ,0x03},
  {CC112X_ANALOG_SPARE      ,0x00},
  {CC112X_FIFO_CFG          ,0x00},
  {CC112X_DEV_ADDR          ,0x00},  
  {CC112X_SETTLING_CFG      ,0x0B},
  {CC112X_FS_CFG            ,0x12}, //////////////////////////////////////////////////////////////////
  {CC112X_PKT_CFG2          ,0x00},
  {CC112X_PKT_CFG1          ,0x05}, // Address check off and CRC check on
  {CC112X_PKT_CFG0          ,0x20},  
  {CC112X_RFEND_CFG1        ,0x0F}, // Stay in RX after RX, No timeout for sync word search  
  {CC112X_RFEND_CFG0        ,0x00}, // IDle after TX, no interferring from MARC  
  {CC112X_SYNC3             ,0x93},/////////////////////////////////////////////////////////////////////
  {CC112X_SYNC2             ,0x0B}, 
  {CC112X_SYNC1             ,0x51}, 
  {CC112X_SYNC0             ,0xDE}, 
  {CC112X_SYNC_CFG1         ,0x0B}, 
  {CC112X_SYNC_CFG0         ,0x17}, 
  {CC112X_DEVIATION_M       ,0x06}, // (4000 Hz)       
  {CC112X_MODCFG_DEV_E      ,0x03}, // (4000 Hz)       
  {CC112X_DCFILT_CFG        ,0x1C}, //                 
  {CC112X_PREAMBLE_CFG1     ,0x18}, // 4" byte preamble 
  {CC112X_PREAMBLE_CFG0     ,0x2A}, //                 
  {CC112X_FREQ_IF_CFG       ,0x40}, // (62500 Hz)      
  {CC112X_IQIC              ,0xC6}, //                 
  {CC112X_CHAN_BW           ,0x08}, // (25000" Hz)      
  {CC112X_MDMCFG1           ,0x46}, //                 
  {CC112X_MDMCFG0           ,0x05}, //                 
  {CC112X_DRATE2            ,0x43}, // (1200 bps)      
  {CC112X_DRATE1            ,0xA9}, // (1200 bps)      
  {CC112X_DRATE0            ,0x2A}, // (1200 bps)      
  {CC112X_AGC_REF           ,0x20}, 
  {CC112X_AGC_CS_THR        ,0x19}, 
  {CC112X_AGC_GAIN_ADJUST   ,0x00}, 
  {CC112X_AGC_CFG3          ,0x91}, 
  {CC112X_AGC_CFG2          ,0x20}, 
  {CC112X_AGC_CFG1          ,0xA9}, 
  {CC112X_AGC_CFG0          ,0xCF}, 
  {CC112X_PA_CFG2           ,0x7F}, 
  {CC112X_PA_CFG1           ,0x56}, 
  {CC112X_PA_CFG0           ,0x7C}, 
  {CC112X_IF_MIX_CFG        ,0x00}, 
  {CC112X_FREQOFF_CFG       ,0x22}, 
  {CC112X_TOC_CFG           ,0x0B},
  {CC112X_SOFT_TX_DATA_CFG  ,0x00}
};

// Modem settings for medium: 50 kb/s, 25kHz deviation, 200 kHz RXBW, IF = 68359 kHz
static const registerSetting_t cc112xMediumDataRateRfSettings[] = 
{
  {CC112X_IOCFG0            ,0x06}, // Route sync signal to GPIO0
  {CC112X_FS_DIG1           ,0x00},
  {CC112X_FS_DIG0           ,0x5F},
  {CC112X_FS_CAL0           ,0x0E},
  {CC112X_FS_DIVTWO         ,0x03},
  {CC112X_FS_DSM0           ,0x33},
  {CC112X_FS_DVC0           ,0x17},  
  {CC112X_FS_PFD            ,0x50},  
  {CC112X_FS_PRE            ,0x6E},
  {CC112X_FS_REG_DIV_CML    ,0x14},
  {CC112X_FS_SPARE          ,0xAC},
  {CC112X_XOSC5             ,0x0E},
  {CC112X_XOSC4             ,0xA0},
  {CC112X_XOSC3             ,0xC7},  
  {CC112X_XOSC1             ,0x03},
  {CC112X_ANALOG_SPARE      ,0x00},
  {CC112X_FIFO_CFG          ,0x00},
  {CC112X_DEV_ADDR          ,0x00},  
  {CC112X_SETTLING_CFG      ,0x0B},
  {CC112X_FS_CFG            ,0x12}, //////////////////////////////////////////////////////////////////
  {CC112X_PKT_CFG2          ,0x00},
  {CC112X_PKT_CFG1          ,0x05},  // Address check off and CRC check on
  {CC112X_PKT_CFG0          ,0x20},  
  {CC112X_RFEND_CFG1        ,0x0F},  // Stay in RX after RX, No timeout for sync word search  
  {CC112X_RFEND_CFG0        ,0x00},  // IDle after TX, no interferring from MARC  
  {CC112X_SYNC3             ,0x93},////////////////////////////////////////////////////////////////////          
  {CC112X_SYNC2             ,0x0B},           
  {CC112X_SYNC1             ,0x51},          
  {CC112X_SYNC0             ,0xDE},          
  {CC112X_SYNC_CFG1         ,0x08},       
  {CC112X_SYNC_CFG0         ,0x17},      
  {CC112X_DEVIATION_M       ,0x99},    
  {CC112X_MODCFG_DEV_E      ,0x0D},   
  {CC112X_DCFILT_CFG        ,0x15},    
  {CC112X_PREAMBLE_CFG1     ,0x18},  
  {CC112X_PREAMBLE_CFG0     ,0x2A},  
  {CC112X_FREQ_IF_CFG       ,0x00},     
  {CC112X_IQIC              ,0x00},           
  {CC112X_CHAN_BW           ,0x01},         
  {CC112X_MDMCFG1           ,0x46},        
  {CC112X_MDMCFG0           ,0x05},         
  {CC112X_DRATE2            ,0x99},        
  {CC112X_DRATE1            ,0x99},        
  {CC112X_DRATE0            ,0x99},        
  {CC112X_AGC_REF           ,0x3C},        
  {CC112X_AGC_CS_THR        ,0xEF},    
  {CC112X_AGC_GAIN_ADJUST   ,0x00}, 
  {CC112X_AGC_CFG3          ,0x83},       
  {CC112X_AGC_CFG2          ,0x60},      
  {CC112X_AGC_CFG1          ,0xA9},      
  {CC112X_AGC_CFG0          ,0xC0},      
  {CC112X_PA_CFG2           ,0x7F},        
  {CC112X_PA_CFG1           ,0x56},        
  {CC112X_PA_CFG0           ,0x79},        
  {CC112X_IF_MIX_CFG        ,0x04},     
  {CC112X_FREQOFF_CFG       ,0x20},    
  {CC112X_TOC_CFG           ,0x0A},         
  {CC112X_SOFT_TX_DATA_CFG  ,0x00}
};

// Settings for test case 9: 150 kb/s, 200 kHz RXBW, 4-GFSK, no IF  
static const registerSetting_t cc112xHighDataRateRfSettings[] =  
{  
  {CC112X_IOCFG0            ,0x06}, // Route sync signal to GPIO0
  {CC112X_FS_DIG1           ,0x00},
  {CC112X_FS_DIG0           ,0x5F},
  {CC112X_FS_CAL0           ,0x0E},
  {CC112X_FS_DIVTWO         ,0x03},
  {CC112X_FS_DSM0           ,0x33},
  {CC112X_FS_DVC0           ,0x17},  
  {CC112X_FS_PFD            ,0x50},  
  {CC112X_FS_PRE            ,0x6E},
  {CC112X_FS_REG_DIV_CML    ,0x14},
  {CC112X_FS_SPARE          ,0xAC},
  {CC112X_XOSC5             ,0x0E},
  {CC112X_XOSC4             ,0xA0},
  {CC112X_XOSC3             ,0xC7},  
  {CC112X_XOSC1             ,0x03},
  {CC112X_ANALOG_SPARE      ,0x00},
  {CC112X_FIFO_CFG          ,0x00},
  {CC112X_DEV_ADDR          ,0x00},  
  {CC112X_SETTLING_CFG      ,0x0B},
  {CC112X_FS_CFG            ,0x12}, //////////////////////////////////////////////////////////////////
  {CC112X_PKT_CFG2          ,0x00},
  {CC112X_PKT_CFG1          ,0x05},  // Address check off and CRC check on
  {CC112X_PKT_CFG0          ,0x20},  
  {CC112X_RFEND_CFG1        ,0x0F},  // Stay in RX after RX, No timeout for sync word search  
  {CC112X_RFEND_CFG0        ,0x00},  // IDle after TX, no interferring from MARC  
  {CC112X_SYNC3             ,0x93},///////////////////////////////////////////////////////////////////           
  {CC112X_SYNC2             ,0x0B},            // TEST CASE CONFIGURATION
  {CC112X_SYNC1             ,0x51},           
  {CC112X_SYNC0             ,0xDE},           
  {CC112X_SYNC_CFG1         ,0x08},        
  {CC112X_SYNC_CFG0         ,0x17},       
  {CC112X_DEVIATION_M       ,0x53},  // (83000 Hz)        
  {CC112X_MODCFG_DEV_E      ,0x2F},  // (83000 Hz)       
  {CC112X_DCFILT_CFG        ,0x04},  //                    
  {CC112X_PREAMBLE_CFG1     ,0x18},  // 4 byte preamble  
  {CC112X_PREAMBLE_CFG0     ,0x2A},  //                  
  {CC112X_FREQ_IF_CFG       ,0x00},  // (0 Hz)             
  {CC112X_IQIC              ,0x00},  //                         
  {CC112X_CHAN_BW           ,0x01},  // (200000 Hz)            
  {CC112X_MDMCFG1           ,0x46},  //                       
  {CC112X_MDMCFG0           ,0x05},  //                        
  {CC112X_DRATE2            ,0xA3},  // (75000 bps)           
  {CC112X_DRATE1            ,0x33},  // (75000 bps)           
  {CC112X_DRATE0            ,0x33},  // (75000 bps)           
  {CC112X_AGC_REF           ,0x3C},         
  {CC112X_AGC_CS_THR        ,0xEC},     
  {CC112X_AGC_GAIN_ADJUST   ,0x00},  
  {CC112X_AGC_CFG3          ,0x83},        
  {CC112X_AGC_CFG2          ,0x60},       
  {CC112X_AGC_CFG1          ,0xA9},       
  {CC112X_AGC_CFG0          ,0xC0},       
  {CC112X_PA_CFG2           ,0x7F},         
  {CC112X_PA_CFG1           ,0x56},         
  {CC112X_PA_CFG0           ,0x01},          
  {CC112X_IF_MIX_CFG        ,0x00},      
  {CC112X_FREQOFF_CFG       ,0x20},     
  {CC112X_TOC_CFG           ,0x0A},          
  {CC112X_SOFT_TX_DATA_CFG  ,0x00}
};       

static const registerSetting_t cc112xLowDataRateRfSettings40Mhz[] =
{
  {CC112X_DEVIATION_M,      0xA3},
  {CC112X_MODCFG_DEV_E,     0x02},   
  {CC112X_CHAN_BW,          0x0A},  
  {CC112X_DRATE2,           0x3F},
  {CC112X_DRATE1,           0x75},
  {CC112X_DRATE0,           0x10}      
};

static const registerSetting_t cc112xMediumDataRateRfSettings40Mhz[] = 
{
  {CC112X_DEVIATION_M,      0x47},
  {CC112X_MODCFG_DEV_E,     0x0D},   
  {CC112X_CHAN_BW,          0x41},  
  {CC112X_DRATE2,           0x94}, // 50 kbps
  {CC112X_DRATE1,           0x7A},
  {CC112X_DRATE0,           0xE1}      
};

static const registerSetting_t cc112xHighDataRateRfSettings40Mhz[] =
{
  {CC112X_DEVIATION_M,      0x0F},
  {CC112X_MODCFG_DEV_E,     0x2F},   
  {CC112X_CHAN_BW,          0x41},  
  {CC112X_DRATE2,           0x9E}, // 150 kbps
  {CC112X_DRATE1,           0xB8},
  {CC112X_DRATE0,           0x52}      
};

// Register settings for link bypass mode that differ from the
// other link cofigurations.
static const registerSetting_t linkBypassSettings[] = 
{
  {CC112X_SYNC3             ,0xD3},          
  {CC112X_SYNC2             ,0x91},           
  {CC112X_SYNC1             ,0xD3}, //sync word compatible with CC1101         
  {CC112X_SYNC0             ,0x91}          
};

//Band select settings for the LO divider.
//out of lock detector enabled
static uint8 cc112xFsCfgs[5] = 
{
  0x1A, // 169 MHz
  0x14, // 434 MHz 
  0x12, // 868 MHz
  0x12, // 915 MHz
  0x12, // 955 MHz  
};
//Frequency programming for CC112X PG2.0 and above
// perSettings.frequencyBand = 0 => 169 MHz 
// perSettings.frequencyBand = 1 => 434 MHz 
// perSettings.frequencyBand = 2 => 868 MHz 
// perSettings.frequencyBand = 3 => 915 Mhz 
// perSettings.frequencyBand = 4 => 955 Mhz 

// For CC112x with 32 MHz XOSC
static uint8 freqConfiguration[5][3] =
{
  {0x69,0xF1,0xFF}, // 169.5125 MHz
  {0x6C,0x80,0x00}, // 434 MHz
  {0x6C,0x80,0x00}, // 868 MHz   
  {0x72,0x60,0x00}, // 915 MHz
  {0x77,0x60,0x00}  // 955 MHz
};
// For CC1125 with 40 MHz TSXO
static uint8 freqConfiguration40Mhz[5][3] = 
{ 
  {0x54, 0xC1, 0x89},  // 169.5125 MHz 
  {0x56, 0xCC, 0xCC},  // 433 MHz  
  {0x56, 0xCC, 0xCC},  // 868 MHz 
  {0x5B, 0x80, 0x00},  // 915 MHz
  {0x5F, 0x80, 0x00}   // 955 MHz
};
/* These structs shall be 2-dimensional when PG2 is present */
/* Sensitivity table - Note: It's only valid for 3 bytes packets and for shipped register settings */
/* Used in Link Margin calculation */  
static const int8 sensitivity868Mhz[3] = 
{
  -118, 
  -98,
  -100
};

/* RSSI offset table */
static const int8 rssiOffset868Mhz[3] = 
{
  96, 
  96, 
  103 
};

/* Values in this table must be ored with the PA_CFG2 register to account for  
 * pa_shape_en
 * Formula: paPowerRamp[index] = dec2hex((wanted_dBm_level+18)*2-1)
 */
static const uint8 paPowerRamp[7] = 
{
  0x03, /* -16 dBm - lowest power       - index 0 */
  0x3F, /*  14 dBm - highest power      - index 1 */
  0x0F, /* -10 dBm - next lowest power  - index 2 */
  0x19, /*  -5 dBm -                    - index 3 */
  0x23, /*   0 dBm -                    - index 4 */
  0x2D, /*   5 dBm -                    - index 5 */
  0x37  /*  10 dBm - next highest power - index 6 */
};
/* Access by index gives GUI values for TX power corresponding to the table above */
static const int8 paPowerGuiValues[7] =
{
  -16,  /* - index 0 */
   14,  /* - index 1 */
  -10,  /* - index 2 */
   -5,  /* - index 3 */
    0,  /* - index 4 */
    5,  /* - index 5 */
   10   /* - index 6 */ 
};

static const float testCaseDataRate[3] =
{
    1.20,  /* testcase 1   <=> SMARTRF_CONFIGURATION_0 */
   50.00,  /* testcase 38  <=> SMARTRF_CONFIGURATION_1 */
  150.00   /* testcase 9   <=> SMARTRF_CONFIGURATION_2 */
};

/******************************************************************************
 * LOCAL VARIABLES
 */

/* Variable is CC112X_STATE_RX when in RX. If not in RX it is CC112X_STATE_TX or 
 * CC112X_STATE_IDLE. The use of this variable is only to avoid an RX interrupt
 * when sending a packet. This facilitates a reduction in power consumption.
 */
static rfStatus_t cc112xRadioTxRx; 
static int8 cc112xRssiOffset; 
 
/******************************************************************************
* @fn          perCC112xGetGuiTxPower
*
* @brief       Returns the TX power in [dBm] used by the menu system. 
*              Implemented by LUT.
*
* input parameters
*
* @param       index - index to GUI TX power LUT
*                  
* output parameters
*
* @return      TX power [dBm]
*/
int8 perCC112xGetGuiTxPower(uint8 index)
{
  return paPowerGuiValues[index];
}

/******************************************************************************
* @fn          perCC112xGetDataRate
*
* @brief       Returns the data rate corresponding to the selected
*              Smart RF configuration
*
* input parameters
*
* @param       index - index to data rate table 
*                  
* output parameters
*
* @return      data rate
*/
float perCC112xGetDataRate(uint8 index)
{
  return testCaseDataRate[index];
}


/******************************************************************************
* @fn          perCC112xSetOutputPower
*
* @brief       Configures the output power of CC112x according to the provided
*              index:
*              0 = -16 dBm
*              1 =  14 dBm
*              2 = -10 dBm
*              3 = -5 dBm 
*              4 =  0 dBm 
*              5 =  5 dBm
*              6 = 10 dBm
*              NOTE: for PG2.0 pa_shape_en and pa_power_ramp has swapped 
*                    position
*
*
*
* input parameters
*
* @param       index - index to table <=> wanted output level
*                  
* output parameters
*
* @return      void
*/
void perCC112xSetOutputPower(uint8 index)
{
  uint8 level; 
  
  /* Reading the PA_CFG2 value to account for pa_shape_en */
  cc112xSpiReadReg(CC112X_PA_CFG2,&level,1);
  /* Saving pa_shape_en */
  level &= 0x40;
  /* Oring in the PA power ramp value */
  level |= paPowerRamp[index];
  /* Updating PA_CFG2 register with its' new value */
  cc112xSpiWriteReg(CC112X_PA_CFG2,&level,1);
  return;
}

/******************************************************************************
 * @fn          perCC112xRegConfig
 *
 * @brief       Configures the CC1120 radio with the selected 
 *              paramteres and test properties or uses the base configuration 
 *              with no address check. Assumes that the radio is in IDLE.
 *
 * input parameters
 *
 * output parameters
 *
 * @return      void
 */
void perCC112xRegConfig(void)
{
  uint8 data;
  uint16 arraySize;
  uint16 arraySize40Mhz;
  const registerSetting_t *pRegisterSettings;
  const registerSetting_t *pRegisterSettings40Mhz;
  uint8 *pFreqConfig[5];
  
  
  /* Log that radio is in IDLE state */
  perCC112xRxIdle();
  cc112xRadioTxRx = CC112X_STATE_IDLE;
    
  /* Extract what radio configuration to use */
  if((perSettings.masterSlaveLinked==PER_DEVICE_LINKED) || 
      (perSettings.masterSlaveLinked == PER_DEVICE_LINK_BYPASS))
  {
    switch(perSettings.smartRfConfiguration)
    {
      case SMARTRF_CONFIGURATION_0:
        pRegisterSettings      = cc112xLowDataRateRfSettings;
        pRegisterSettings40Mhz = cc112xLowDataRateRfSettings40Mhz;
        arraySize      = (sizeof cc112xLowDataRateRfSettings/sizeof(registerSetting_t));
        arraySize40Mhz = (sizeof cc112xLowDataRateRfSettings40Mhz/sizeof(registerSetting_t));
        perSettings.sensitivity = sensitivity868Mhz[0];
        cc112xRssiOffset = rssiOffset868Mhz[0];
        break;
      case SMARTRF_CONFIGURATION_2:
        pRegisterSettings      = cc112xHighDataRateRfSettings;
        pRegisterSettings40Mhz = cc112xHighDataRateRfSettings40Mhz;
        arraySize      = (sizeof cc112xHighDataRateRfSettings/sizeof(registerSetting_t));
        arraySize40Mhz = (sizeof cc112xHighDataRateRfSettings40Mhz/sizeof(registerSetting_t));
        perSettings.sensitivity = sensitivity868Mhz[1];
        cc112xRssiOffset = rssiOffset868Mhz[1];
        break;
      default:
        pRegisterSettings      = cc112xMediumDataRateRfSettings;
        pRegisterSettings40Mhz = cc112xMediumDataRateRfSettings40Mhz;
        arraySize      = (sizeof cc112xMediumDataRateRfSettings/sizeof(registerSetting_t));
        arraySize40Mhz = (sizeof cc112xMediumDataRateRfSettings40Mhz/sizeof(registerSetting_t));
        perSettings.sensitivity = sensitivity868Mhz[2];
        cc112xRssiOffset = rssiOffset868Mhz[2];
        break;
    }
  }
  else
  {
    /* Base settings for communication */
    pRegisterSettings      = cc112xMediumDataRateRfSettings;
    pRegisterSettings40Mhz = cc112xMediumDataRateRfSettings40Mhz;
    arraySize      = (sizeof cc112xMediumDataRateRfSettings/sizeof(registerSetting_t));
    arraySize40Mhz = (sizeof cc112xMediumDataRateRfSettings40Mhz/sizeof(registerSetting_t));
    /* In lack of numbers for other freq's than 868 MHz */
    perSettings.sensitivity = sensitivity868Mhz[2];
    cc112xRssiOffset = rssiOffset868Mhz[2];
  }
  
  /* Write register settings to radio */
  for(uint16 i = 0; i < arraySize;i++)
  {
    data = pRegisterSettings[i].data;
    cc112xSpiWriteReg(pRegisterSettings[i].addr,&data,1);
  }
  /* If 40Mhz crystal detected correct affected registers */
  if(perRadioChipType.xoscFreq == XOSC_FREQ_40MHZ)
  {
    for(uint16 i = 0; i < arraySize40Mhz;i++)
    {
      data = pRegisterSettings40Mhz[i].data;
      cc112xSpiWriteReg(pRegisterSettings40Mhz[i].addr,&data,1);
    }
  }
 
  /* Load freq word depending on crystal detected */
  if(perRadioChipType.xoscFreq == XOSC_FREQ_40MHZ)
  {
    pFreqConfig[0] = freqConfiguration40Mhz[0];
    pFreqConfig[1] = freqConfiguration40Mhz[1];
    pFreqConfig[2] = freqConfiguration40Mhz[2];
    pFreqConfig[3] = freqConfiguration40Mhz[3];
    pFreqConfig[4] = freqConfiguration40Mhz[4];
  }
  else
  {
    pFreqConfig[0] = freqConfiguration[0];
    pFreqConfig[1] = freqConfiguration[1];
    pFreqConfig[2] = freqConfiguration[2];
    pFreqConfig[3] = freqConfiguration[3];
    pFreqConfig[4] = freqConfiguration[4];
  }
    
    switch(perSettings.frequencyBand)      
    {
      case 0: //169 MHz
        cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[0],1);
        cc112xSpiWriteReg(CC112X_FREQ2,pFreqConfig[0],3);
        break;
      case 1: // 434 MHz
        cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[1],1);
        cc112xSpiWriteReg(CC112X_FREQ2,pFreqConfig[1],3);
        break;
      case 2: // 868 MHz
        cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[2],1);
        cc112xSpiWriteReg(CC112X_FREQ2,pFreqConfig[2],3);
        break;      
      case 3: // 915 MHz
        cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[3],1);
        cc112xSpiWriteReg(CC112X_FREQ2,pFreqConfig[3],3);
        break;
      case 4: // 955 MHz
        cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[4],1);
        cc112xSpiWriteReg(CC112X_FREQ2,pFreqConfig[4],3);
        break;      
      default: // 868 MHz
        cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[2],1);
        cc112xSpiWriteReg(CC112X_FREQ2,pFreqConfig[2],3);
        break;
    }    

  if(perSettings.masterSlaveLinked==PER_DEVICE_LINKED)
  {
    /* PKT_LEN set to user specified packet length: HW length filtering */
    cc112xSpiWriteReg(CC112X_PKT_LEN, &(perSettings.payloadLength),1);
    /* Turn on HW Address filtering */
    uint8 pkt_cfg1 = 0x10|0x05;
    cc112xSpiWriteReg(CC112X_PKT_CFG1,&pkt_cfg1,1);
    /* Set address */
    cc112xSpiWriteReg(CC112X_DEV_ADDR,&perSettings.address,1);

    if((perSettings.linkTopology == LINK_2_WAY))
    {
      if(perSettings.deviceMode == MASTER_DEVICE)
      {
        /* IDLE after RX, RX after TX */
        data = 0x0F;
        cc112xSpiWriteReg(CC112X_RFEND_CFG1,&data,1);
        data = 0x30;
        cc112xSpiWriteReg(CC112X_RFEND_CFG0,&data,1);
      }
    } 
  }
  else if(perSettings.masterSlaveLinked == PER_DEVICE_LINK_BYPASS)
  {
    //Change sync word to apply to link bypass configuration
    for(uint16 i = 0; i < (sizeof linkBypassSettings/sizeof(registerSetting_t));i++)
    {
      data = linkBypassSettings[i].data;
      cc112xSpiWriteReg(linkBypassSettings[i].addr,&data,1);
    }  
    
    /* PKT_LEN set to user specified packet length: HW length filtering */
    cc112xSpiWriteReg(CC112X_PKT_LEN, &(perSettings.payloadLength),1);  
  }
  else
  {
    /* length of configuration packet + filter byte */
    data = PER_SETTINGS_PACKET_LEN; 
    cc112xSpiWriteReg(CC112X_PKT_LEN, &data,1);
  }  
  return;  
}

/******************************************************************************
 * @fn          perCC112xSendPacket
 *
 * @brief       Sends the contents that pData points to which has the 
 *              following structure:
 *
 *              txArray[0] = length byte
 *              txArray[n] = payload[n]
 *              | n<[sizeOf(RXFIFO)-2], variable packet length is assumed.
 * 
 *              The radio state after completing TX is dependant on the 
 *              CC112X_RFEND_CFG0 register setting. For PG10 this register 
 *              dictates IDLE after TX. For PG.0 this function must be 
 *              re-implemented since the 2way PER test relies on RX after TX.
 *              This function enables SYNC interrupt. This means that 
 *              an interrupt will fire when a packet has been sent, i.e sync 
 *              signal transitions from high to low. 
 *             
 *              The One-Way PER test disables the sync pin interrupt when TX
 *              finishes, while the Two-Way PER test doesn't to enable quick
 *              reception of Slave ACK.
 *
 *              Note: Assumes chip is ready
 *
 * input parameters
 *              
 * @param       *pData - pointer to data array that starts with length byte
 *                       and followed by payload.
 * output parameters
 *
 * @return      void
 */
void perCC112xSendPacket(uint8 *pData)
{
  uint8 len = *pData;
  /* PG1.0 errate fix: Before entering TX, the frequency word must be altered from that of RX */
  /* This means in general that TX from Idle is the only option, not TX from RX */
  perCC112xEnterIdle();
  /* Will only try to transmit if the whole packet can fit i RXFIFO 
   * and we're not currently sending a packet.
   */
  if(!(len > (PER_MAX_DATA-2)) && (cc112xRadioTxRx != CC112X_STATE_TX) )
  {
    cc112xSpiWriteTxFifo(pData,(len+1));
    /* Indicate state to the ISR and issue the TX strobe */
    trxEnableInt();
    cc112xRadioTxRx = CC112X_STATE_TX; 
    trxSpiCmdStrobe(CC112X_STX);
    /* Wait until packet is sent before doing anything else */
    __low_power_mode_3();

    /* This function will not return before the complete packet
     * is sent and the radio is back in IDLE. The MSP will be 
     * be sleeping while the packet is beeing sent unless waken
     * by button presses.
     */
    while(cc112xRadioTxRx == CC112X_STATE_TX);
    if(perSettings.linkTopology == LINK_1_WAY)
    {
      /* Back in Idle*/
      trxDisableInt();
    }
  }
  return;
}


/******************************************************************************
 * @fn          perCC112xEnterRx
 *
 * @brief       Enters RX from IDLE. Function is used to abstract the 
 *              cc112xRadioTxRx functionality away from the lower-level radio 
 *              interface.
 *              Note: assumes chip is ready and in IDLE
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC112xEnterRx(void)
{
  perCC112xEnterIdle();
  perCC112xIdleRx();
  cc112xRadioTxRx = CC112X_STATE_RX;
  return;
}


/******************************************************************************
 * @fn          perCC112xEnterSleep
 *
 * @brief       Enters Sleep. Function is used to abstract the cc112xRadioTxRx
 *              functionality away from the lower-level radio interface.
 *              Note: assumes chip is ready
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC112xEnterSleep(void)
{
  perCC112xRxIdle();
  trxSpiCmdStrobe(CC112X_SPWD);
  /* Only important to differ between RX/TX and IDLE */
  cc112xRadioTxRx = CC112X_STATE_IDLE;
  return;
}


/******************************************************************************
 * @fn          perCC112xEnterIdle
 *
 * @brief       Enters IDLE from ANY state. Function is used to abstract the 
 *              cc112xRadioTxRx functionality away from the lower-level radio 
 *              interface.
 *           
 * input parameters
 *   
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC112xEnterIdle(void)
{
  /* wait until chip is ready */
  TRXEM_SPI_BEGIN();
  while(TRXEM_PORT_IN & TRXEM_SPI_MISO_PIN);
  perCC112xRxIdle();
  cc112xRadioTxRx = CC112X_STATE_IDLE;
  return;
}

/******************************************************************************
 * @fn          perCC112xRxTxISR
 *
 * @brief       ISR that's called when sync signal goes low. 
 *              In RX State: Filters incoming data. The global rxData pointer
 *              always points to this functions static rxData_tmp(struct of
 *              same kind). The validnes of rxData fields is indicated by the
 *              the global flag packetSemaphore.
 *              In TX State: Nothing is done except it facilitates power 
 *              consumption reduction when TX since the program doesn't need
 *              to wait until TX is done before re-enabling sync pin interrupt.
 *              cc112xRadioTxRx is also set to CC112X_STATE_IDLE to be consistent 
 *              with program.
 * 
 * input parameters
 *             
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */
void perCC112xRxTxISR(void)
{
  uint8 rxBytes,rxLength,rssiIndex,lqiIndex;
  /* This variable stores the data locally. Access is given to per_test by 
   * assigning this instance to the global rxData pointer
   */
  static rxData_t rxData_tmp;
            
  rxData = &rxData_tmp;
  
  /* Checking if the chip is in RX state:  */
  if(cc112xRadioTxRx != CC112X_STATE_RX)
  {
    /* Transmission finished */
    if((perSettings.deviceMode == MASTER_DEVICE) && (perSettings.linkTopology == LINK_2_WAY) && (perSettings.masterSlaveLinked ==PER_DEVICE_LINKED))
    {
      /* Only applicable when master in 2-way test */
      cc112xRadioTxRx=CC112X_STATE_RX;
    }
    else
    {
      cc112xRadioTxRx  = CC112X_STATE_IDLE;
    }
    return;
  }
  
  packetSemaphore |= SYNC_FOUND;
  
  if(((perSettings.masterSlaveLinked == PER_DEVICE_LINKED)||(perSettings.masterSlaveLinked == PER_DEVICE_LINK_BYPASS)) && (perSettings.deviceMode == MASTER_DEVICE) && (perSettings.testRunning == PER_TRUE))
  {
  	if(perSettings.linkTopology == LINK_1_WAY)
  	{
  	  /* Read timer value and set the perSettings.packetRate valu(adjustment for temperature drift */
      halTimer32kSetIntFrequency(perSettings.packetRate);
      halTimer32kIntEnable();
    }
    else
    {
    	/* LINK_2_WAY */ 
    	
    	/* Timeout interrupt configuring is handled by the 2-way Per test */
      timer32kValue = halTimer32kReadTimerValue();
    	halTimer32kAbort();
    }
  }
    
  cc112xSpiReadReg(CC112X_NUM_RXBYTES,&rxBytes,1);
  /* Checking if the FIFO is empty */
  if(rxBytes == 0)
  {
    /* The packet was removed by HW due to addr or length filtering -> Do nothing */
    /* Report that a sync was detected */ 
    rxData_tmp.rssi = perCC112xRead8BitRssi();
    return;
  }
  else
  {
    /* The RX FIFO is not empty, process contents */    
    cc112xSpiReadRxFifo(&rxLength, 1);  
    /* Check that the packet length just read + status bytes(2B) + length byte match the RXBYTES */
    /* If these are not equal:
     * - Packet is not like expected
     */
    if(rxBytes != (rxLength+3))
    {
      /* This is a fault FIFO condition -> clean FIFO and register a sync detection */
      /* IDLE -> FLUSH RX FIFO -> RX */
      perCC112xRxIdle();     
      perCC112xEnterRx(); 
      /* Report that a sync was detected */
      rxData_tmp.rssi = perCC112xRead8BitRssi();
      return;
    }
    else
    {
      /* We don't have a FIFO error condition -> get packet */
      
      /* Length Field */
      rxData_tmp.data[0] = rxLength;
      rssiIndex = rxLength+1;
      lqiIndex  = rssiIndex +1;
      
      /* Payload(ADDR + DATA + STATUS BYTES) */
      cc112xSpiReadRxFifo(&rxData_tmp.data[1], lqiIndex);
      
      /* The whole packet has been read from the FIFO.
       * Check if the CRC is correct and that the packet length is as expected.
       * If not correct: report sync found and do not update RSSI or LQI.
       */
      if((!(rxData_tmp.data[lqiIndex] & CC112X_LQI_CRC_OK_BM)) || (perSettings.payloadLength != rxLength ))
      {
        rxData_tmp.rssi = perCC112xRead8BitRssi();
        return;
      }
      /* A complete error-free packet has arrived  */
      
      /* Measured data */
      rxData_tmp.length  = rxLength;
      rxData_tmp.lqi     = rxData_tmp.data[lqiIndex] & CC112X_LQI_EST_BM;
      rxData_tmp.addr    = rxData_tmp.data[1]; 
      
      /* Convert RSSI value from 2's complement to decimal value accounting for offset value */
      rxBytes = rxData_tmp.data[rssiIndex];        
      rxData_tmp.rssi = (int16)((int8)rxBytes) - cc112xRssiOffset;
      /* Signal a good packet is received */
      packetSemaphore |= PACKET_RECEIVED;
      return;
    } 
  }   
}     
      
/*******************************************************************************
 * @fn          perCC112xRead8BitRssi
 *    
 * @brief       Reads MSB RSSI value from register, converts the dBm value to
 *              decimal and adjusts it according to RSSI offset
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      decimal RSSI value corrected for RSSI offset
 */ 
int8 perCC112xRead8BitRssi(void)
{
  uint8 rssi2compl,rssiValid;
  int16 rssiConverted;
  
  cc112xSpiReadReg(CC112X_RSSI0, &rssiValid,1);
  if(rssiValid & 0x01)
  {
    /* Read RSSI from MSB register */
    cc112xSpiReadReg(CC112X_RSSI1, &rssi2compl,1);
    rssiConverted = (int16)((int8)rssi2compl) - cc112xRssiOffset;
    return rssiConverted;
  }
  /* keep last value since new value is not valid */
  return rxData->rssi;
}

/*******************************************************************************
 * @fn          perCC112xRxIdle
 *
 * @brief       Radio state is switched from RX to IDLE
 *
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */ 
static void perCC112xRxIdle(void)
{
  /* Disable pin interrupt */
  trxDisableInt();
  /* Strobe IDLE */
  trxSpiCmdStrobe(CC112X_SIDLE); 
  /* Wait until chip is in IDLE */
  while(trxSpiCmdStrobe(CC112X_SNOP) & 0xF0);
  /* Clear pin interrupt flag */
  trxClearIntFlag();
  return;
}

/*******************************************************************************
 * @fn          perCC112xIdleRx
 *
 * @brief       Radio state is switched from Idle to RX. Function assumes that
 *              radio is in IDLE when called. 
 * 
 * input parameters
 *
 * @param       none
 *
 * output parameters
 *
 * @return      void
 */ 
static void perCC112xIdleRx(void)
{
  trxClearIntFlag();
  trxSpiCmdStrobe(CC112X_SRX);
  trxEnableInt();
  return;
}
/*******************************************************************************
 * @fn          perCC112xWriteTxFifo
 *
 * @brief       Means for PER test to write the TX FIFO
 * 
 * input parameters
 *
 * @param       *pData  - pointer to data array that will be written to TX Fifo
 * @param       len     - number of bytes in that will be written to TX Fifo
 *
 * output parameters
 *
 * @return      void
 */ 
void perCC112xWriteTxFifo(uint8 *pData, uint8 len)
{
  cc112xSpiWriteTxFifo(pData,len);
  return;
}
void writeRegisterConfiguration(const registerSetting_t *pRegisterSettings)
{
  uint8 data;
  
  for(uint16 i = 0; i < (sizeof cc112xLowDataRateRfSettings/sizeof(registerSetting_t));i++)
  {
    data = pRegisterSettings[i].data;
    cc112xSpiWriteReg(pRegisterSettings[i].addr,&data,1);
  }
}
/******************************************************************************
  Copyright 2011 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
*******************************************************************************/

