/******************************************************************************
  Filename:   sniff_mode.c
  
  Description: Implementation file for the trxeb sniff_mode app.

******************************************************************************/


/******************************************************************************
 * INCLUDES
 */
 
#include <msp430.h>
#include "hal_types.h"
#include "hal_board.h"
#include "hal_button_trxeb.h"
#include "hal_lcd_trxeb.h"
#include "chip_detect.h"
#include "hal_timer_32k.h"
#include "trx_rf_int.h"
#include "trx_rf_spi.h"
#include "menu_driver.h"
#include "sniff_mode.h"
#include "cc112x_spi.h"

/******************************************************************************
 * TYPEDEFS
 */

typedef struct
{
  uint16  addr;
  uint8   data;
} radioSetting_t;


/******************************************************************************
* LOCAL FUNCTIONS
*/
uint8 sniffInitApp(void** pDummy);
static void  sniffModeRegConfig(void);
static void  worRssiInit(void);
static void radioRXISR(void);
static void trxRxIdle(void);
static void trxIdleWor(void);

/******************************************************************************
* LOCAL VARIABLES
*/ 
#define PKTLEN              5
#define ISR_ACTION_REQUIRED 1
#define ISR_IDLE            0

static uint8 writeByte;
static uint8 buttonPressed;
static uint8 comArray[20];
static uint32 pkt;
static uint32 pktMiss;
static uint32 pktExpected;
static uint8 packetStatus;

static int8 rssi;
static uint8 rssi_readout;
static uint8 cc112xRssiOffset = 96;


/******************************************************************************
* GLOBAL VARIABLES
*/ 
/******************************************************************************
* CONSTANTS
*/ 


static const radioSetting_t simpleLinkTestSniff[] = 
{
  {CC112X_IOCFG0            ,0x06}, // Route sync signal to GPIO0
  {CC112X_FS_DIG1           ,0x00},
  {CC112X_FS_DIG0           ,0x5F},
  {CC112X_FS_CAL1           ,0X40},
  {CC112X_FS_CAL0           ,0x0E},
  {CC112X_FS_DIVTWO         ,0x03},
  {CC112X_FS_DSM0           ,0x33},
  {CC112X_FS_DVC0           ,0x17},  
  {CC112X_FS_PFD            ,0x50},  
  {CC112X_FS_PRE            ,0x6E},
  {CC112X_FS_REG_DIV_CML    ,0x14},
  {CC112X_FS_SPARE          ,0xAC},
  {CC112X_FS_VCO0           ,0xB4},
  {CC112X_XOSC5             ,0x0E},
  {CC112X_XOSC4             ,0xA0},
  {CC112X_XOSC3             ,0x03},
  {CC112X_XOSC1             ,0x03},
  {CC112X_ANALOG_SPARE      ,0x00},
  {CC112X_FIFO_CFG          ,0x00},
  {CC112X_DEV_ADDR          ,0x00},  
  {CC112X_SETTLING_CFG      ,0x0B},
  {CC112X_FS_CFG            ,0x12}, //////////////////////////////////////////////////////////////////
  {CC112X_PKT_CFG2          ,0x00},
  {CC112X_PKT_CFG1          ,0x05}, // Address check off and CRC check on
  {CC112X_PKT_CFG0          ,0x20},  
  {CC112X_RFEND_CFG1        ,0x0F}, // Stay in RX after RX, No timeout for sync word search  
  {CC112X_RFEND_CFG0        ,0x00}, // IDle after TX, no interferring from MARC 
  {CC112X_PKT_LEN           ,0xFF}, 
  {CC112X_SYNC3             ,0x93},/////////////////////////////////////////////////////////////////////
  {CC112X_SYNC2             ,0x0B}, 
  {CC112X_SYNC1             ,0x51}, 
  {CC112X_SYNC0             ,0xDE}, 
  {CC112X_SYNC_CFG1         ,0x0B}, 
  {CC112X_SYNC_CFG0         ,0x17}, 
  {CC112X_DEVIATION_M       ,0x06}, // (4000 Hz)       
  {CC112X_MODCFG_DEV_E      ,0x03}, // (4000 Hz)       
  {CC112X_DCFILT_CFG        ,0x1C}, //                 
  {CC112X_PREAMBLE_CFG1     ,0x18}, // 4" byte preamble 
  {CC112X_PREAMBLE_CFG0     ,0x2A}, //                 
  {CC112X_FREQ_IF_CFG       ,0x40}, // (62500 Hz)      
  {CC112X_IQIC              ,0xC6}, //                 
  {CC112X_CHAN_BW           ,0x08}, // (25000" Hz)      
  {CC112X_MDMCFG1           ,0x46}, //                 
  {CC112X_MDMCFG0           ,0x05}, //                 
  {CC112X_DRATE2            ,0x43}, // (1200 bps)      
  {CC112X_DRATE1            ,0xA9}, // (1200 bps)      
  {CC112X_DRATE0            ,0x2A}, // (1200 bps)      
  {CC112X_AGC_REF           ,0x20}, 
  {CC112X_AGC_CS_THR        ,0x19}, 
  {CC112X_AGC_GAIN_ADJUST   ,0x00}, 
  {CC112X_AGC_CFG3          ,0x91}, 
  {CC112X_AGC_CFG2          ,0x20}, 
  {CC112X_AGC_CFG1          ,0xA9}, 
  {CC112X_AGC_CFG0          ,0xCF}, 
  {CC112X_PA_CFG2           ,0x7F}, 
  {CC112X_PA_CFG1           ,0x56}, 
  {CC112X_PA_CFG0           ,0x7C}, 
  {CC112X_IF_MIX_CFG        ,0x00}, 
  {CC112X_FREQOFF_CFG       ,0x22}, 
  {CC112X_TOC_CFG           ,0x0B},
  {CC112X_SOFT_TX_DATA_CFG  ,0x00} 
};

//Register Settings for different frequency bands.
static uint8 freqSettings[5][3] = 
{ 
 {0x69,0xF1,0xFF}, // 169.5125 MHz
 {0x6C,0x80,0x00}, // 434 MHz
 {0x6C,0x80,0x00}, // 868 MHz   
 {0x72,0x60,0x00}, // 915 MHz
 {0x77,0x60,0x00}  // 955 MHz
};
//Band select setting for LO divider
static uint8 cc112xFsCfgs[5] = 
{
  0x0A, // 169 MHz 
  0x04, // 434 MHz  
  0x02, // 868 MHz 
  0x02, // 915 MHz 
  0x02, // 955 MHz   
};

// GPIO output for master and slave. For debug purpose
uint8 gpioConfigSlave[] = 
{
    0x38, // WOR_EVENT1
    0x37, // WOR_EVENT0
    0x10, // CARRIER_SENSE_VALID
    0x06, // PKT_SYNC_RXTX        This signal is needed for the radioRXISR(). Do not change.
};
uint8 gpioConfigMaster[] = 
{
    0x0C, // MARC_ 2PIN_STATUS[1]
    0x26, // MARC_2PIN_STATUS[0]
    0x0B, // PQT_REACHED
    0x06, // PKT_SYNC_RXTX
};

/******************************************************************************
* FUNCTIONS
*/
/******************************************************************************
 * @fn          sniffModeRegConfig
 *
 * @brief       Writes packet and modem settings to registers
 *
 * input parameters
 *              
 * output parameters
 *
 * @return      void
*/
void sniffModeRegConfig(void)
{
    //Write register settings based on TC01 to chip
   for(uint16 i = 0; i < (sizeof simpleLinkTestSniff/sizeof(radioSetting_t));i++)
   {
    writeByte  = (uint8)simpleLinkTestSniff[i].data;
    cc112xSpiWriteReg(simpleLinkTestSniff[i].addr,&writeByte,1);
   }   
    // Put radio in powerdown to save power
    trxSpiCmdStrobe(CC112X_SPWD);
   

}
/******************************************************************************
 * @fn          sniffModeFreqConfig
 *
 * @brief       Writes frequency word depending on user input
 *
 * input         pFreqIndex - frequency band index casted to void**
 *              
 * output       none
 *
 * @return      void
*/
uint8 sniffModeFreqConfig(void** pFreqIndex)
{
   uint16 index = (uint16)*pFreqIndex;
   // Set frequency
    cc112xSpiWriteReg(CC112X_FS_CFG,&cc112xFsCfgs[index],1);
    cc112xSpiWriteReg(CC112X_FREQ2,freqSettings[index],3);
    
    // Put radio in powerdown to save power
    trxSpiCmdStrobe(CC112X_SPWD);
    
    //Insert Carrier Sense threshold warning in Sniff Test Menu
    drawInfo();
    
    return SNIFF_RETURN_SUCCESS;
}

/******************************************************************************
 * @fn          sniffMasterStartApp
 *
 * @brief       
 *
 * input parameters
 *
 * @param       pDummy  - pointer to pointer to void. Not used
 *
 * output parameters
 *
 * @return      SNIFF_RETURN_SUCCESS
 */
uint8 sniffMasterStartApp(void **pDummy)
{
  
  // Set first packet number
  pkt = 1;
  
  // Set up GPIO pins. For debug
  cc112xSpiWriteReg(CC112X_IOCFG3,&gpioConfigMaster[0],4);
  
  //Display while configuring radios*
  halLcdClear(0);
  halLcdPrintString(0,"    RX Sniff Test    ",0,0);
  halLcdSetHLine(0,0,LCD_COLS-1,7);
  halLcdPrintString(0,"     Radio in TX     ",0,2);
  halLcdPrintString(0,"                 ",0,3);
  halLcdPrintString(0,"  Press right button  ",0,4);
  halLcdPrintString(0,"    to send packet  ",0,5);
  halLcdPrintString(0," 1 Abort Master Mode ",0,7);
  halLcdSetHLine(0,0,LCD_COLS-1,55);
  halLcdInvertPage(0,0,LCD_COLS,7);
  halLcdSendBuffer(0);
  
  // Put MCU to sleep
    __low_power_mode_3();
    while(1)
    {
        if(buttonPressed = halButtonsPushed())
        {
            if(buttonPressed == HAL_BUTTON_LEFT)
            {
               // Left button pressed. Abort function
               // Put radio in powerdown to save power
               trxSpiCmdStrobe(CC112X_SPWD);
               //Insert Carrier Sense threshold warning in Sniff Test Menu
               drawInfo();
    
               return SNIFF_RETURN_FAILURE;
            }
            else if (buttonPressed == HAL_BUTTON_RIGHT)
            {
                //Right button pressed, send packet
                halLcdClear(0);
                // build packet
                comArray[0] = PKTLEN;   // length field
                comArray[1] = 0x00;     // address field
                comArray[2] = pkt>>24;  // payload
                comArray[3] = pkt>>16;
                comArray[4] = pkt>>8;
                comArray[5] = pkt;
                // Update LCD
                halLcdPrintString(0,"    RX Sniff Test    ",0,0);
                halLcdSetHLine(0,0,LCD_COLS-1,7);
                halLcdPrintString(0,"Sent Pkt number:",0,3);
                halLcdPrintInt(0,pkt,70,4);
                halLcdPrintString(0," 1 Abort Master Mode ",0,7);
                halLcdSetHLine(0,0,LCD_COLS-1,55);
                halLcdInvertPage(0,0,LCD_COLS,7);
                halLcdSendBuffer(0);
                // Update packet counter
                pkt++;
                // Strobe IDLE and fill TX FIFO
                trxSpiCmdStrobe(CC112X_SIDLE);
                // wait for radio to enter IDLE state
                while((trxSpiCmdStrobe(CC112X_SNOP)& 0xF0) != 0x00);
                cc112xSpiWriteTxFifo(comArray,PKTLEN+1);
                // Send packet
                trxSpiCmdStrobe(CC112X_STX);
                // Wait for radio to finish sending packet
                while((trxSpiCmdStrobe(CC112X_SNOP)& 0xF0) != 0x00);
                // Put radio in powerdown to save power
                trxSpiCmdStrobe(CC112X_SPWD);
                //Put MCU to sleep
                __low_power_mode_3(); 
            }
        }
    }
  
  
  
}
/******************************************************************************
 * @fn          sniffSlaveStartApp
 *
 * @brief       
 *
 * input parameters
 *
 * @param       pDummy  - pointer to pointer to void. Not used
 *
 * output parameters
 *
 * @return      SNIFF_RETURN_SUCCESS
 */
uint8 sniffSlaveStartApp(void **pDummy)
{  
   
   //Be sure we are in IDLE
   trxSpiCmdStrobe(CC112X_SIDLE);
   
   // Reset packet status
   packetStatus = ISR_IDLE;
   //Intit packet counters
   pkt =0;
   pktMiss = 0;
   pktExpected = 1;
   
   //Init Wake On Radio function
   worRssiInit();
   
   // Set up GPIO pins. For debug 
   //IOCFG0 has to be 0x06 for radioRXISR interrupt. Other pins are free to change 
   cc112xSpiWriteReg(CC112X_IOCFG3,&gpioConfigSlave[0],4);
   // Connect ISR function to interrupt from GDO0 (P1.7)   
   trxIsrConnect(&radioRXISR);   
   
   // Update LCD with status
   halLcdClear(0);
   halLcdPrintString(0,"    RX Sniff Test    ",0,0);
   halLcdSetHLine(0,0,LCD_COLS-1,7);
   halLcdPrintString(0,"     Radio in RX      ",0,3);
   halLcdPrintString(0,"  Waiting for packet  ",0,4);
   halLcdPrintString(0," 1 Abort Slave Mode ",0,7);
   halLcdSetHLine(0,0,LCD_COLS-1,55);
   halLcdInvertPage(0,0,LCD_COLS,7);
   halLcdSendBuffer(0);

   //Do SCAL to be sure radio is calibrated befor test starts
  trxSpiCmdStrobe(CC112X_SCAL);
   
   while(1)
   {
     if(packetStatus == ISR_ACTION_REQUIRED)
     {
        packetStatus = ISR_IDLE;
        // Update display
        halLcdClear(0);
        halLcdPrintString(0,"    RX Sniff Test    ",0,0);
        halLcdSetHLine(0,0,LCD_COLS-1,7);
        halLcdPrintString(0,"Rcv'd Pkt nr:",0,2);
        halLcdPrintInt(0,pkt,90,2);
        halLcdPrintString(0,"Missed Pkt's:",0,3);
        halLcdPrintInt(0,pktMiss,90,3); 
        halLcdPrintString(0,"RSSI:",0,5);
        halLcdPrintInt(0,rssi,40,5); 
        halLcdPrintString(0," 1 Abort Slave Mode ",0,7);
        halLcdSetHLine(0,0,LCD_COLS-1,55);
        halLcdInvertPage(0,0,LCD_COLS,7);
        halLcdSendBuffer(0);
        
     }
     // Re-enter WOR cycle
     trxIdleWor();
     
     // Put MCU to sleep
     __low_power_mode_3();
     
     // If left button pushed by user, abort test and exit to menu
     if(HAL_BUTTON_LEFT == halButtonsPushed())
     {
         trxRxIdle();
         //Insert Carrier Sense threshold warning in Sniff Test Menu
         drawInfo();
         // Put radio in powerdown to save power
         trxSpiCmdStrobe(CC112X_SPWD);
         break;
     }          

   }
   
   return SNIFF_RETURN_SUCCESS;
}
/******************************************************************************
 * @fn          sniffInitApp
 *
 * @brief       
 *
 * input parameters
 *              
 * @param        pDummy - manual chip selection for cc1190, casted to void**
 *
 * output parameters
 *
 * @return      SNIFF_RETURN_SUCCESS/SNIFF_RETURN_FAILURE
 */
uint8 sniffInitApp(void** pDummy)
{
  radioChipType_t sniffRadioChipType;
  // Detect if a supported radio is present  
  trxDetectChipType(&sniffRadioChipType);
  //If detected radio is not cc1101 go directly initiation 
  if((sniffRadioChipType.deviceName == CHIP_TYPE_CC1125 && sniffRadioChipType.ver == 0x20)
           || (sniffRadioChipType.deviceName == CHIP_TYPE_CC1125 && sniffRadioChipType.ver == 0x21)  
           || (sniffRadioChipType.deviceName == CHIP_TYPE_CC1120 && sniffRadioChipType.ver == 0x10)
           || (sniffRadioChipType.deviceName == CHIP_TYPE_CC1120 && sniffRadioChipType.ver == 0x21)              
           || (sniffRadioChipType.deviceName == CHIP_TYPE_CC1121 && sniffRadioChipType.ver == 0x10)
           || (sniffRadioChipType.deviceName == CHIP_TYPE_CC1121 && sniffRadioChipType.ver == 0x21))
  {
    // Configure radio registers
    sniffModeRegConfig();
    

      
    return SNIFF_RETURN_SUCCESS; 
  }
  else
  {
     //Not supported radio
    // Issue message to user to insert radio EM 
    halLcdClear(0);
    halLcdPrintString(0,"Sniff test could not",6,1);
    halLcdPrintString(0,"detect a supported",6,2);
    halLcdPrintString(0,"radio.",36,3);
    halLcdPrintString(0," ",36,4);
    halLcdPrintString(0,"Use CC112x PG 2.x.",6,5);
    halLcdSendBuffer(0);
    
    // Make the message visible for max 3 seconds 
    halTimer32kMcuSleepTicks(TIMER_32K_CLK_FREQ);
    halTimer32kMcuSleepTicks(TIMER_32K_CLK_FREQ);
    halTimer32kMcuSleepTicks(TIMER_32K_CLK_FREQ);
    
    // clear potential button pushes 
    halButtonsPushed();
      
    return SNIFF_RETURN_FAILURE;
  }
}
/***********************************************************************************
* @fn          worRssiInit
*
* @brief       Set up radio for WOR operation terminating on RSSI
*
* @param       none
*
* @return      none
*/

static void worRssiInit(void)
{ 
  // Set carrier sense gate mode on recevier, and set AGC carrier sense threshold
  writeByte = 0x37; //approx -40 dBm
  cc112xSpiWriteReg(CC112X_AGC_CS_THR,&writeByte,1);
  
  // Set RSSI valid count to minimum: 2 COMPUTATIONS
  writeByte = 0xC3;
  cc112xSpiWriteReg(CC112X_AGC_CFG0,&writeByte,1);

  //Set AGC window size to 8 samples
  //Set AGC settle wait time to 24 SAMPLES
  writeByte = 0xA0;
  cc112xSpiWriteReg(CC112X_AGC_CFG1,&writeByte,1);
  
  // Turning off autocal    
  writeByte = 0x03;
  cc112xSpiWriteReg(CC112X_SETTLING_CFG,&writeByte,1);

  // Setting WOR mode to NORMAL_MODE & WOR timer resolution to HIGH_RESOLUTION
  writeByte = 0x08;
  cc112xSpiWriteReg(CC112X_WOR_CFG1,&writeByte,1);

  //Setting Event0 timeout
  writeByte = 0x02; //20 ms 22 ms 23 ms
  cc112xSpiWriteReg(CC112X_WOR_EVENT0_MSB,&writeByte,1);
  writeByte = 0x80; //20 ms
  cc112xSpiWriteReg(CC112X_WOR_EVENT0_LSB,&writeByte,1);

  //Event1 timeout set to 1 ms to allow XOSC anddigital regulator to startup and settle
  writeByte = 0x0E;
  cc112xSpiWriteReg(CC112X_WOR_CFG1,&writeByte,1);

  //Disable Event2 configuration. RC_OSC calibraton interval disabled.
  //Setting RC osc enable
  //Setting RC calibration mode to continuous calibration  
  writeByte = 0x24;
  cc112xSpiWriteReg(CC112X_WOR_CFG0,&writeByte,1);

  // Strobe idle to calibrate
  trxSpiCmdStrobe(CC112X_SIDLE);
       
  //disable RC osc callibration
  writeByte = 0x20;
  cc112xSpiWriteReg(CC112X_WOR_CFG0,&writeByte,1);

  //Setting RX timeout to never 
  //(in case of noise making the radio hang in RX this can be set
  // to force powerdown after a certain time in RX)
  writeByte = 0x0E;
  cc112xSpiWriteReg(CC112X_RFEND_CFG1,&writeByte,1);

  //Setting RX_TERMINATE_ON_RSSI
  writeByte = 0x09;
  cc112xSpiWriteReg(CC112X_RFEND_CFG0,&writeByte,1);

}

/***********************************************************************************
* @fn          radioRXISR
*
* @brief       ISR for packet handling in RX
*
* @param       none
*
* @return      none
*/
static void radioRXISR(void)
{
  uint8 rxBytes;
  packetStatus = ISR_ACTION_REQUIRED;
  // Read NUM_RXBYTES for bytes in FIFO  
  cc112xSpiReadReg(CC112X_NUM_RXBYTES, &rxBytes, 1);
  
  if(rxBytes == PKTLEN+3) // PAYLOAD + LENGHT BYTE + 2 STATUS BYTES
  { 
    // Read RX FIFO
    cc112xSpiReadRxFifo(comArray,rxBytes);
    // Check CRC ok and read packet if CRC ok. (CRC_OK: bit7 in second status byte)
    if(comArray[rxBytes-1] | 0x80)
    {
      pkt = (((uint32)comArray[2])<<24)|(((uint32)comArray[3])<<16)|(((uint32)comArray[4])<<8)|(comArray[5]);
    }
   }
  // Check if we missed any packets
  if(pkt != pktExpected )
  {
      if(pkt > pktExpected)
      {
        pktMiss += (pkt-pktExpected);
      }
      else
      {
        pktMiss =0;
      }          
  }
  pktExpected = pkt+1;
  // Read RSSI
  cc112xSpiReadReg(CC112X_RSSI1,&rssi_readout,1);
  // Convert to decimal value from 2's complement rssi_readout. 
  rssi = (int16)((int8)rssi_readout) - cc112xRssiOffset;
  
  trxRxIdle();
}
/***************************************************************************************************
 * @fn          trxRxIdle
 *
 * @brief       Radio state is switched from RX to IDLE
 *              Note: assumes chip is ready
 *
 * @param       none
 *
 * @return      none
 **************************************************************************************************/ 
void trxRxIdle(void)
{
  // Disable pin interrupt 
  trxDisableInt();
  // Strobe IDLE 
  trxSpiCmdStrobe(CC112X_SIDLE);
  // Wait until chip is in IDLE 
  while(trxSpiCmdStrobe(CC112X_SNOP)& 0xF0);
  // Flush the Receive FIFO 
  trxSpiCmdStrobe(CC112X_SFRX);
  // Clear pin interrupt flag 
  trxClearIntFlag();
}

/***************************************************************************************************
 * @fn          trxIdleWor
 *
 * @brief       Radio state is switched from Idle to WOR  
 *              Note: assumes chip is ready
 *
 * @param       none
 *
 * @return      none
 **************************************************************************************************/ 
void trxIdleWor(void)
{
  trxClearIntFlag();
  // Reset WOR otherwise we might get stuck in powerdown
  trxSpiCmdStrobe(CC112X_SWORRST);
  trxSpiCmdStrobe(CC112X_SWOR);
  
  // Maybe switch 
  trxEnableInt();
}

/******************************************************************************
  Copyright 2011 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
*******************************************************************************/