#include "string.h"
#include "comdef.h"
#include "ioCC1110.h"
#include "dma.h"
#include "radio.h"
#include "RingBuffer.h"

#define RADIO_RX_SIZE  64
#define RADIO_TX_SIZE  64
#define RADIO_FIFO_SIZE  128
#define RF_PACK_SIZE     20

#define RX_DMA_LEN  RADIO_RX_SIZE

#define MRFI_DMA_CHAN  0
#define ABORT          0x80

#define RADIO_SFSTXON  0
#define RADIO_SCAL    0x01
#define RADIO_SRX     0x02
#define RADIO_STX     0x03
#define RADIO_SIDLE   0x04
#define RADIO_SNOP    0x0A

static DMA_DESC dma_cfg;
static uint8_t rf_state;
static uint8_t radio_interrupt_flag;
enum {RF_UNDEF=1,RF_RX,RF_TX,RF_SCAL,RF_IDLE};

static BYTE_RING_BUFFER rx_loop,tx_loop;

static uint8_t Rxbuffer[RADIO_RX_SIZE];
static uint8_t Txbuffer[RADIO_TX_SIZE];
static uint8_t rf_fifo[RADIO_FIFO_SIZE];


void radio_rxOn(void);

#pragma vector = RF_VECTOR
__interrupt void rf_ISR(void)
{
    uint8_t i;
    S1CON = 0;              /* Clear MCU interrupt flag */
    i = RFIF;
    RFIF &= ~i;       /* Clear the interrupt at the source */
    RFST = RADIO_SIDLE;
    radio_interrupt_flag = 1;
}

void rf_isr_handler(void)
{
    uint8_t i;
    if(radio_interrupt_flag==0)
        return;

    if(rf_state==RF_RX){
        byteRingWrite(&rx_loop,&rf_fifo[1],rf_fifo[0]);
        RFST = RADIO_SRX;
        DMAIRQ = ~DEF_B8(MRFI_DMA_CHAN);
        DMAARM |= DEF_B8( MRFI_DMA_CHAN );
    }
    else if(rf_state==RF_TX){
        i = byteRingGetCount(&tx_loop);
        if(i>0){
            if(i<RF_PACK_SIZE){
                rf_fifo[0] = byteRingRead(&tx_loop,&rf_fifo[1],i);
            }
            else{
                rf_fifo[0] = byteRingRead(&tx_loop,&rf_fifo[1],(RF_PACK_SIZE-1));
            }
            DMAARM |= DEF_B8( MRFI_DMA_CHAN );

            DMAIRQ = ~DEF_B8(MRFI_DMA_CHAN);

            S1CON = 0; /* Clear MCU interrupt flag */
            /* send strobe to enter receive mode */
            RFST = RADIO_STX;
            /* enable "receive/transmit done" interrupts */
            RFIM |= (1<<4);
        }    
        else{
            radio_rxOn();
        }
    }
    radio_interrupt_flag = 0;
}

void radio_rxOn(void)
{
    dma_cfg.SRCADDRH= HIGH_BYTE_OF_WORD(&X_RFD);
    dma_cfg.SRCADDRL= LOW_BYTE_OF_WORD(&X_RFD);
    dma_cfg.DESTADDRH= HIGH_BYTE_OF_WORD(&rf_fifo[0]);
    dma_cfg.DESTADDRL= LOW_BYTE_OF_WORD(&rf_fifo[0]);
    dma_cfg.VLEN = DMA_VLEN_1_P_VALOFFIRST;
    dma_cfg.LENH = HIGH_BYTE_OF_WORD(RADIO_FIFO_SIZE)&0x1F;
    dma_cfg.LENL = LOW_BYTE_OF_WORD(RADIO_FIFO_SIZE);
    dma_cfg.WORDSIZE = DMA_WORDSIZE_BYTE;
    dma_cfg.TMODE = DMA_TMODE_SINGLE;
    dma_cfg.TRIG = DMA_TRIG_RADIO;
    dma_cfg.SRCINC = DMA_SRCINC_0;
    dma_cfg.DESTINC = DMA_DESTINC_1;
    dma_cfg.IRQMASK = DMA_IRQMASK_DISABLE;
    dma_cfg.M8 = DMA_M8_USE_8_BITS;
    dma_cfg.PRIORITY = DMA_PRI_GUARANTEED;

    /* Abort any ongoing DMA transfer */
    DMAARM = ABORT | DEF_B8( MRFI_DMA_CHAN );
    DMAARM |= DEF_B8( MRFI_DMA_CHAN );

    //MRFI_STROBE_IDLE_AND_WAIT();      // put the radio in a known state
    memset(rf_fifo, 0x00, RADIO_FIFO_SIZE);
    /* Clear any pending DMA interrupts */
    DMAIRQ = ~DEF_B8(MRFI_DMA_CHAN);

    /* Clear interrupts */
    //  S1CON &= ~(RFIF_1 | RFIF_0); /* Clear MCU interrupt flag */
    S1CON = 0; /* Clear MCU interrupt flag */
    RFIF &= ~(1<<4);           /* Clear the interrupt at the source */

    /* send strobe to enter receive mode */
    RFST = RADIO_SRX;

    /* enable "receive/transmit done" interrupts */
    RFIM |= (1<<4)|(1<<6);
    rf_state = RF_RX;
}



void radio_txOn(uint8_t *tsrc,uint8_t len)
{
    
    rf_fifo[0] = len;
    memcpy(&rf_fifo[1],tsrc,len);
    dma_cfg.SRCADDRH= HIGH_BYTE_OF_WORD(&rf_fifo[0]);
    dma_cfg.SRCADDRL= LOW_BYTE_OF_WORD(&rf_fifo[0]);
    dma_cfg.DESTADDRH= HIGH_BYTE_OF_WORD(&X_RFD);
    dma_cfg.DESTADDRL= LOW_BYTE_OF_WORD(&X_RFD);
    dma_cfg.VLEN = DMA_VLEN_1_P_VALOFFIRST;
    dma_cfg.LENH = HIGH_BYTE_OF_WORD(RADIO_FIFO_SIZE)&0x1F;
    dma_cfg.LENL = LOW_BYTE_OF_WORD(RADIO_FIFO_SIZE);
    dma_cfg.WORDSIZE = DMA_WORDSIZE_BYTE;
    dma_cfg.TMODE = DMA_TMODE_SINGLE;
    dma_cfg.TRIG = DMA_TRIG_RADIO;
    dma_cfg.SRCINC = DMA_DESTINC_1;
    dma_cfg.DESTINC = DMA_SRCINC_0;
    dma_cfg.IRQMASK = DMA_IRQMASK_DISABLE;
    dma_cfg.M8 = DMA_M8_USE_8_BITS;
    dma_cfg.PRIORITY = DMA_PRI_GUARANTEED;

    /* Abort any ongoing DMA transfer */
    DMAARM = ABORT | DEF_B8( MRFI_DMA_CHAN );
    DMAARM |= DEF_B8( MRFI_DMA_CHAN );

    RFST = RADIO_SIDLE;
    while(MARCSTATE!=0x01);
    //MRFI_STROBE_IDLE_AND_WAIT();      // put the radio in a known state
    /* Clear any pending DMA interrupts */
    DMAIRQ = ~DEF_B8(MRFI_DMA_CHAN);

    /* Clear interrupts */
    //  S1CON &= ~(RFIF_1 | RFIF_0); /* Clear MCU interrupt flag */
    S1CON = 0; /* Clear MCU interrupt flag */
    RFIF &= ~(1<<4);           /* Clear the interrupt at the source */

    /* send strobe to enter receive mode */
    RFST = RADIO_STX;
    /* enable "receive/transmit done" interrupts */
    RFIM |= (1<<4);
    rf_state = RF_TX;
}

void radio_send(uint8_t *tsrc,uint8_t len)
{
    if(rf_state==RF_TX){
        ENTER_CRITICAL;
        byteRingWrite(&tx_loop,tsrc,len); 
        EXIT_CRITICAL;
    }
    else{
        if(len<RF_PACK_SIZE){
            radio_txOn(tsrc,len); 
        }
        else{
            radio_txOn(tsrc,RF_PACK_SIZE-1);
            ENTER_CRITICAL;
            byteRingWrite(&tx_loop,&tsrc[RF_PACK_SIZE],len-(RF_PACK_SIZE-1)); 
            EXIT_CRITICAL;
        }
    }
}
uint8_t radio_read(uint8_t *out_src,uint8_t len)
{
    uint8_t i;
    ENTER_CRITICAL;
    i = (uint8_t)byteRingGetCount(&rx_loop);
    if(i>0){
       i = byteRingRead(&rx_loop,out_src,len);
    }
    EXIT_CRITICAL;
    return i;
}


void radio_pn9(void)
{
    PKTCTRL0  = 0x44;   // Packet automation control.
    RFST = RADIO_SIDLE;
    while((MARCSTATE&0x1f)!=0x01);
    RFST = RADIO_STX;
    while(1);
}

void radio_init(uint8_t ch)
{
    uint32_t loop;
    radio_interrupt_flag = 0;
    byteRingBufInit(&rx_loop,Rxbuffer,RADIO_RX_SIZE);
    byteRingBufInit(&tx_loop,Txbuffer,RADIO_TX_SIZE);


    DMA0CFGH = HIGH_BYTE_OF_WORD( &dma_cfg );
    DMA0CFGL = LOW_BYTE_OF_WORD ( &dma_cfg );

    /* Setup radio with settings from SmartRF?Studio. The default settings are
     * used, except that "unmodulated" is chosen in the "Simple RX tab". This
     * results in an umodulated carrier with a frequency of approx. 2.433 GHz.
     */
    FSCTRL1   = 0x06;   // Frequency synthesizer control.
    FSCTRL0   = 0x00;   // Frequency synthesizer control.
    FREQ2     = 0x10;   // Frequency control word, high byte.
    FREQ1     = 0x89;   // Frequency control word, middle byte.
    FREQ0     = 0xD8;   // Frequency control word, low byte.
    MDMCFG4   = 0xCA;   // Modem configuration.
    MDMCFG3   = 0x83;   // Modem configuration.
    MDMCFG2   = 0x03;   // Modem configuration.
    MDMCFG1   = 0xA2;   // Modem configuration.
    MDMCFG0   = 0xF8;   // Modem configuration.
    CHANNR    = ch;     // Channel number.
    DEVIATN   = 0x34;   // Modem deviation setting (when FSK modulation is enabled).
    FREND1    = 0x56;   // Front end RX configuration.
    FREND0    = 0x10;   // Front end RX configuration.
    MCSM2     = 0x07;
    MCSM1     = 0x30;
    MCSM0     = 0x18;   // Main Radio Control State Machine configuration.
    FOCCFG    = 0x16;   // Frequency Offset Compensation Configuration.
    BSCFG     = 0x6C;   // Bit synchronization Configuration.
    AGCCTRL2  = 0x43;   // AGC control.
    AGCCTRL1  = 0x40;   // AGC control.
    AGCCTRL0  = 0x91;   // AGC control.
    FSCAL3    = 0xE9;   // Frequency synthesizer calibration.
    FSCAL2    = 0x2A;   // Frequency synthesizer calibration.
    FSCAL1    = 0x00;   // Frequency synthesizer calibration.
    FSCAL0    = 0x1F;   // Frequency synthesizer calibration.
    TEST2     = 0x81;   // Various test settings.
    TEST1     = 0x35;   // Various test settings.
    TEST0     = 0x0B;   // Various test settings.
    PA_TABLE0 = 0xC0;   // PA output power setting.
    PKTCTRL1  = 0x04;   // Packet automation control.
    PKTCTRL0  = 0x44;   // Packet automation control.
    ADDR      = 0x00;   // Device address.
    PKTLEN    = 0x14;   // Packet length.

    /* Settings not from SmartRF?Studio. Setting both sync word registers to
     * 0xAA = 0b10101010, i.e., the same as the preamble pattern. Not necessary,
     * but gives control of what the radio attempts to transmit.
     */
    SYNC1     = 0xD3;
    SYNC0     = 0x2C;
    
    RFST = RADIO_SIDLE;
    RFST = RADIO_SCAL;
    loop = 5000;
    while(loop--);

    rf_state = RF_IDLE; 
    RFTXRXIE = 1;
//    IEN2     |= 1;
}

