
#define  TMR_GLOBALS  
#include "tmr.h"  
#include "ioCC1110.h"
/* 
 ********************************************************************************************************* 
 *                                      LOCAL FUNCTION PROTOTYPES 
 ********************************************************************************************************* 
 */  

/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                                   CONFIGURE TIMER TIMEOUT FUNCTION 
 * 
 * Description : Set the user defined function when the timer expires. 
 * Arguments   : n         is the timer number 0..TMR_MAX_TMR-1 
 *               fnct      is a pointer to a function that will be executed when the timer expires 
 *               arg       is a pointer to an argument that is passed to 'fnct' 
 * Returns     : None. 
 ********************************************************************************************************* 
 */  

void  TmrCfgFnct (uint8_t n, uint8_t mode, void (*fnct)(void *), void *arg)  
{  
    TMR *ptmr;  

    if (n < TMR_MAX_TMR) {  
        ptmr             = &TmrTbl[n];  
        ENTER_CRITICAL;  
        ptmr->Mode       = mode;
        ptmr->TmrFnct    = fnct;                 /* Store pointer to user function into timer          */  
        ptmr->TmrFnctArg = arg;                  /* Store user's function arguments pointer            */  
        EXIT_CRITICAL;  
    }  
}  

/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                                             CHECK TIMER 
 * 
 * Description : This function checks to see if a timer has expired 
 * Arguments   : n      is the timer to check 
 * Returns     : 0      if the timer has expired 
 *               TmrCtr the remaining time before the timer expires in 1/10 second 
 ********************************************************************************************************* 
 */  

uint16_t  TmrChk (uint8_t n)  
{  
    uint16_t val;  

    val = 0;  
    if (n < TMR_MAX_TMR) {  
        ENTER_CRITICAL;  
        val = TmrTbl[n].TmrCtr;  
        EXIT_CRITICAL;  
    }  
    return (val);  
}  
/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                                     TIMER MANAGER INITIALIZATION 
 * 
 * Description : This function initializes the timer manager module. 
 * Arguments   : None 
 * Returns     : None. 
 ********************************************************************************************************* 
 */  

void  TmrInit (void)  
{  
    uint8_t  i;  
    TMR   *ptmr;  

    ptmr = &TmrTbl[0];  
    for (i = 0; i < TMR_MAX_TMR; i++) {          /* Clear and disable all timers                       */  
        ptmr->TmrEn   = TMR_STOP;  
        ptmr->Mode    = S_MODE;
        ptmr->TmrCtr  = 0;  
        ptmr->TmrInit = 0;  
        ptmr->TmrFnct = (void (*)(void *))0;;  
        ptmr++;  
    }  
}  

/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                                             RESET TIMER 
 * 
 * Description : This function reloads a timer with its initial value 
 * Arguments   : n         is the timer to reset 
 * Returns     : None. 
 ********************************************************************************************************* 
 */  

void  TmrReset (uint8_t n)  
{  
    TMR *ptmr;  

    if (n < TMR_MAX_TMR) {  
        ptmr         = &TmrTbl[n];  
        ENTER_CRITICAL;  
        ptmr->TmrCtr = ptmr->TmrInit;       /* Reload the counter                                      */  
        EXIT_CRITICAL;  
    }  
}  

/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                               SET TIMER (SPECIFYING TENTHS OF SECOND) 
 * 
 * Description : Set the timer with the specified number of 1/10 seconds. 
 * Arguments   : n         is the timer number 0..TMR_MAX_TMR-1 
 *               tenths    is the number of 1/10 second to load into the timer 
 * Returns     : None. 
 ********************************************************************************************************* 
 */  

void  TmrSetTick (uint8_t n, uint16_t tick)  
{  
    TMR *ptmr;  

    if (n < TMR_MAX_TMR) {  
        ptmr          = &TmrTbl[n];  
        ENTER_CRITICAL;  
        ptmr->TmrInit = tick;  
        ptmr->TmrCtr  = tick;  
        EXIT_CRITICAL;  
    }  
}  

/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                                             START TIMER 
 * 
 * Description : This function start a timer 
 * Arguments   : n         is the timer to start 
 * Returns     : None. 
 ********************************************************************************************************* 
 */  

void  TmrStart (uint8_t n)  
{  
    if (n < TMR_MAX_TMR) {  
        ENTER_CRITICAL;  
        TmrTbl[n].TmrEn = TMR_RUN;  
        EXIT_CRITICAL;  
    }  
}  

/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                                             STOP TIMER 
 * 
 * Description : This function stops a timer 
 * Arguments   : n         is the timer to stop 
 * Returns     : None. 
 ********************************************************************************************************* 
 */  

void  TmrStop (uint8_t n)  
{  
    if (n < TMR_MAX_TMR) {  
        ENTER_CRITICAL;  
        TmrTbl[n].TmrEn = TMR_STOP;  
        EXIT_CRITICAL;  
    }  
}  

/*$PAGE*/  
/* 
 ********************************************************************************************************* 
 *                                          TIMER MANAGER TASK 
 * 
 * Description : This task is created by TmrInit() and is responsible for updating the timers. 
 *               TmrTask() executes every 1/10 of a second. 
 * Arguments   : None. 
 * Returns     : None. 
 * Note(s)     : 1) The function to execute when a timer times out is executed outside the critical 
 *                  section. 
 ********************************************************************************************************* 
 */  


void  TmrTask (void)  
{  

    uint8_t   i;  
    TMR    *ptmr;
    void  (*pfnct)(void *);                           /* Function to execute when timer times out      */  
    void   *parg;                                     /* Arguments to pass to above function           */  

    pfnct = (void (*)(void *))0;                      /* Start off with no function to execute         */  
    parg  = (void *)0;  
    ptmr = &TmrTbl[0];                            /* Point at beginning of timer table             */  
    for (i = 0; i < TMR_MAX_TMR; i++) {  
        ENTER_CRITICAL;  
        if (ptmr->TmrEn == TMR_RUN) {                /* Decrement timer only if it is enabled         */  
            if (ptmr->TmrCtr > 0) {  
                ptmr->TmrCtr--;  
                if(ptmr->TmrCtr == 0) {               /* See if timer expired                     */  
                    if(ptmr->Mode==F_MODE){
                        ptmr->TmrCtr = ptmr->TmrInit;
                    }
                    else{
                        ptmr->TmrEn = TMR_STOP;               /* Yes, stop timer                          */  
                    }
                    pfnct       = ptmr->TmrFnct;       /* Get pointer to function to execute ...   */  
                    parg        = ptmr->TmrFnctArg;    /* ... and its argument                     */ 
                }  
            }  
        }  
        EXIT_CRITICAL;  
        if (pfnct != (void (*)(void *))0) {       /* See if we need to execute function for ...    */  
            (*pfnct)(parg);                       /* ... timed out timer.                          */  
            pfnct = (void (*)(void *))0;  
        }  
        ptmr++;  
    }  
}  

