/*
 * autobaud.c
 *
 * description: This routine will determine the baud rate of the host
 *              and set the client baud rate to match, it is run on the
 *              client side.
 *
 *              The DCO must be setup to operate at 1,048,576 Hz for the
 *              follow numbers to be accurate.
 *              The first CR will determine if the host is operating in
 *              the range 115k<=baud<=14,400 and if so the client baud rate
 *              generator registers will be set according to the table below.
 *              If the host is not the range 115k<=baud<=14,400 then a second
 *              carriage return will be required to determine the host baud range,
 *              9,600<=baud<=1200. If the host baud rate is in this range then
 *              the baud rate generator registers are set appropriately according
 *              to the table below. If it is NOT in this range, then and error
 *              condition is returned.
 *
 *              The UBR00/UBR01 register contains the prescale/divider
 *              used for generating the correct baud rate, its
 *              value is determine by the formula:
 *
 *              UBR00/01 = (DCO clk frequency)1,048,576Hz/baud rate
 *
 *              Apply this formula to determine the following table:
 *
 *              Baud Rate    count  UBR00  UBR01    Modulation
 *              ----------------------------------------------
 *              115,200       9.10   0x09   0x00    0x10
 *               57,600      18.20   0x12   0x00    0x84
 *               38,400      27.31   0x1b   0x00    0x94
 *               19,200      54.61   0x36   0x00    0xb5
 *               14,400      72.82   0x48   0x00    0x7b
 *                9,600     109.23   0x6d   0x00    0x44
 *                4,800     218.45   0xda   0x00    0xaa
 *                2,400     436.91   0xb4   0x01    0xdf
 *                1,200     873.81   0x69   0x03    0x7b
 *
 *              The modulation value that goes in the UMCTL0 register is the
 *              fractional portion from the above equation. This has been
 *              calculated according to the procedure outlined in the
 *              MSP430F1xx Family User's Guide.
 *
 **************************************************************************
 *
 *  C. Farrow
 *  Texas Instruments, Inc
 *  September 2004
 *  Built with IAR Embedded Workbench Version: 3.0B
 *
 *******************************************************************************/

#include <msp430x12x2.h>

typedef enum  { BAUD115K, BAUD57600,BAUD38400, BAUD19200,
                BAUD14400, BAUD9600, BAUD4800,  BAUD2400, BAUD1200,
              } baudrates;

typedef struct {
  char ubr00;
  char ubr01;
  char umctl0;
  char *desc;
} baudreg;

// Baud Rate Generator Register values (UBR00, UBR01, UMCTL0, Descr)
const baudreg baudregs[] =
{
    0x09, 0x00, 0x10, "115k",      // BAUD115K
    0x12, 0x00, 0x84, "57600",     // BAUD57600
    0x1b, 0x00, 0x94, "38400",     // BAUD38400
    0x36, 0x00, 0x5b, "19200",     // BAUD19200
    0x48, 0x00, 0x7b, "14400",     // BAUD14400
    0x6d, 0x00, 0x44, "9600",      // BAUD9600
    0xda, 0x00, 0xaa, "4800",      // BAUD4800
    0xb4, 0x01, 0xdf, "2400",      // BAUD2400
    0x69, 0x03, 0x7b, "1200",      // BAUD1200
};

extern const baudreg baudregs[];

char getchar(void);
void setbaud(int baud);
void showbaud(int baud);
extern void put_message (char *s);
int putchar(int c);
void delay_8_33ms(void);

/*
 * autobaud()
 * description: Routine that waits for one or two <CR> from PC terminal program
 *              which is used to determine the baud rate of the host. The UART
 *              baud registers are then set to match the host baud rate.
 *
 */
int autobaud(void)
{
  baudrates rate;

  while (!(UTCTL0 & TXEPT));                    // Confirm no TXing before setup

  UCTL0 |= SWRST;                               // Place USART in reset
  URCTL0 = URXEIE;                              // Allow erroneous input to interrupt
  UCTL0 &= ~SWRST;                              // Take USART out of reset

  setbaud(BAUD115K);

  switch (getchar())                            // Get UART character
  {
    case 0x0d :
      rate = BAUD115K;                          // Perfect match!
      break;
    case 0xe6 :
      rate = BAUD57600;
      break;
    case 0x1c :
      rate = BAUD38400;
      break;
    case 0xe0 :
      rate = BAUD19200;
      break;
    case 0x80 :
      rate = BAUD14400;
      break;
    case 0x00 :
      delay_8_33ms();                           // Delay program execution to
                                                // make sure a byte-receiption
                                                // at 1200bps is over

      setbaud(BAUD9600);                        // Set new baud rate

      switch (getchar())                        // Get 2nd UART character
      {
        case 0x0d :                             // Perfect match!
          rate = BAUD9600;
          break;
        case 0xe6 :
          rate = BAUD4800;
          break;
        case 0x78 :
          rate = BAUD2400;
          break;
        case 0x80 :
          rate = BAUD1200;
          break;
        default :
          return -1;                            // Failure
      }
      break;
    default :
      return -1;                                // Failure
  }

  delay_8_33ms();                               // Delay program execution to
                                                // make sure a byte-receiption
                                                // at 1200bps is over

  setbaud(rate);
  showbaud(rate);
  return 0;                                     // success
}
/*
 * setbaud()
 * description: Set the baud rate generator registers
 *
 */
void setbaud(int baud)
{
  UCTL0 |= SWRST;                               // USART held in reset
  ME2 &= ~(UTXE0 + URXE0);                      // Disable USART operation

  UBR00 = baudregs[baud].ubr00;                 // Set configuration for
  UBR10 = baudregs[baud].ubr01;                 // new baud rate
  UMCTL0 = baudregs[baud].umctl0;

  ME2 |= UTXE0 + URXE0;                         // Re-enable USART operation
  UCTL0 &= ~SWRST;                              // Release USART

  RXBUF0;                                       // Empty RX buffer and clear flags
}

/*
 * showbaud()
 * description: This routine simply displays the selected baud rate chosen
 *
 */
void showbaud(int baud)
{
  put_message("Baud Rate generator set at: ");
  put_message(baudregs[baud].desc);
}
/*
 * getchar()
 * description: None interrupt base UART read
 *
 */
char getchar(void)
{
  while (!(IFG2 & URXIFG0));
  return RXBUF0;
}
/*
 * function: put_message()
 * description: send a line to terminal display
 *
 */
void put_message(char *s)
{
  while (*s)
  {
    putchar(*s);
    s++;
  }
}
/*
 * function: delay_8_33ms()
 * description: delays program execution by 8.33ms @ ACLK=32KHz
 *
 */
void delay_8_33ms(void)
{
  TACCR0 = 272;                                 // 273 Ticks @ 32Khz = 8.33ms
  TACTL = TASSEL_1 + MC_1 + TACLR;              // Clear and start timer
  while (!(TACTL & TAIFG));                     // Wait for timer overflow
  TACTL = 0;                                    // Stop timer
}
static void __low_level_put(int c)
{
  while ((IFG2 & UTXIFG0) != UTXIFG0);          // USART0 TX buffer ready?
  TXBUF0 = c;                                   // send char to TXBUF0
}

int putchar(int c)
{
  if (c == '\n') {                              // Convert EOL to CR/LF
    __low_level_put('\r');
  }
  __low_level_put(c);
  return c;
}
