//******************************************************************************
//  MSP430x1232 UART Demo - USART0 UART 1200-115k ULP Echo ISR, 32kHz ACLK & DCO
//
//  Description; Automatically detect baud rate of host and set USART0. This
//  is done by pressing <CR> once/twice within a terminal program running on the PC.
//  The autobaud() function will then set USART to the matching baud rate.
//  The program will then echo any received characters, USART0 RX ISR at high-speed
//  used with ultra-low power techniques. Normal operation in LPM3, Set_DCO
//  subroutine needs to be called to configure DCO which is used for UART baud
//  generation. On valid RX character, character echoed  back. Use start-bit
//  edge detect - URXSE - to automatically (re)enable DCO and trigger ISR. ISR
//  must make sure DCO clock source remains enabled for the UART to receive
//  full character. Software also needs to make sure a character has been
//  completely TX'ed before entering LPM3, which disables DCO.
//  ACLK = LFXT1 = 32768/8,  MCLK = SMCLK = UCLK0 = DCOCLK  = 1048576
//  //*An external 32kHz watch crystal on XIN,XOUT is required for ACLK*//
//
//
//               MSP430F1232
//             -----------------
//         /|\|              XIN|-
//          | |                 | 32768Hz
//          --|RST          XOUT|-
//            |                 |
//            |             P3.6|------------>
//            |                 | 1200-to-115k - 8N1
//            |             P3.7|<------------
//
//
//  C. Farrow
//  Texas Instruments, Inc
//  September 2004
//  Built with IAR Embedded Workbench Version: 3.20A
//******************************************************************************

#include <msp430x12x2.h>

#define DELTA 256                             // Target DCO = DELTA*(4096) = 1048576

void Set_DCO (void);
int autobaud(void);

void main(void)
{
  volatile unsigned int i;                    // 'volatile' to prevent opt.

  WDTCTL = WDTPW + WDTHOLD;                   // Stop WDT

  P3SEL |= 0x30;                              // P3.4,5 = USART0 TXD/RXD
  P3DIR = 0x10;                               // Init P3.x

  for (i = 0xFFFF; i > 0; i--);               // Delay for XTAL to settle

  Set_DCO();                                  // Set DCO (ACLK must be active!)

  ME2 |= UTXE0 + URXE0;                       // Enabled USART0 TXD/RXD
  UCTL0 |= CHAR;                              // 8-bit character, SWRST=1
  UTCTL0 |= SSEL1 + URXSE;                    // UCLK=SMCLK, start edge detect
  UCTL0 &= ~SWRST;                            // Initalize USART state machine

  while (autobaud() == -1);                   // Loop until baud rate detection
                                              // was successful

  IE2 |= URXIE0;                              // Enable USART1 RX interrupt
  _EINT();                                    // Enable global ints

  for (;;)
  {
    while (!(UTCTL0 & TXEPT));                // Confirm no TXing before -> LPM3
    _DINT();                                  // Disable ints for flag test
    _NOP();

    if (UTCTL0 & SSEL0)
      _BIS_SR(LPM3_bits + GIE);               // Enter LPM3 with ints
    else
      _BIS_SR(LPM0_bits + GIE);               // RX'ing char, LPM0, int's active
  }
}
/* usart0_rx()
 * Description: ISR for USART0 recieve interrupt
 *
 */
#pragma vector = UART0RX_VECTOR
__interrupt void usart0_rx (void)
{
  if (IFG2 & URXIFG0)                         // Test URXIFG0
  {
    while (!(IFG2 & UTXIFG0));                // USART0 TX buffer ready?
    TXBUF0 = RXBUF0;                          // RXBUF0 to TXBUF0
    _BIC_SR_IRQ(LPM3_bits);                   // Exit LPM3 on reti
    UTCTL0 |= SSEL0;                          // SSEL0=1, no RX activity
  }
  else                                        // Start edge
  {
    UTCTL0 &= ~URXSE;                         // Clear URXS signal
    UTCTL0 |= URXSE;                          // Re-enable edge detect
    _BIC_SR_IRQ(SCG1 + SCG0);                 // DCO enabled on reti
    UTCTL0 &= ~SSEL0;                         // SSEL0=0, RX activity
  }
}
/*
 * Set_DCO()
 * Description: Set the DCO to 1,048,576 Hz
 *
 */
void Set_DCO(void)                            // Subroutine to set DCO
{
  unsigned int Compare, Oldcapture = 0;

  BCSCTL1 |= DIVA_3;                          // ACLK = LFXT1CLK/8
  CCTL2 = CM_1 + CCIS_1 + CAP;                // CAP, ACLK
  TACTL = TASSEL_2 + MC_2 + TACLR;            // SMCLK, cont-mode, clear

  while (1)
  {
    while (!(CCIFG & CCTL2));                 // Wait until capture occurred
    CCTL2 &= ~CCIFG;                          // Capture occurred, clear flag
    Compare = CCR2;                           // Get current captured SMCLK
    Compare = Compare - Oldcapture;           // SMCLK difference
    Oldcapture = CCR2;                        // Save current captured SMCLK

    if (DELTA == Compare)
    {
      break;                                  // Break loop
    }
    else if (DELTA < Compare)
    {
      DCOCTL--;
      if (DCOCTL == 0xFF)                     // DCO is too fast, slow it down
        if (BCSCTL1 & 0x07)
          BCSCTL1--;                          // DCO role under?, Sel lower RSEL
        else
          break;                              // Error condition, break loop
    }
    else
    {
      DCOCTL++;                               // DCO is too slow, speed it up
      if (DCOCTL == 0x00)
        if ((BCSCTL1 & 0x07) != 0x07)
          BCSCTL1++;                          // DCO role over? Sel higher RSEL
        else
          break;                              // Error condition, break loop
    }
  }
  TACTL = 0;                                  // Stop Timer_A
  CCTL2 = 0;                                  // Stop CCR2
  BCSCTL1 &= ~DIVA_3;                         // ACLK = LFXT1CLK
}

