//*****************************************************************************
//  MSP430AFE25x Demo - SD24, Continuous Conversion on a Group of 3 Channels
//
//  Description: This program uses the SD24 module to perform continuous
//  conversions on a group of channels (0, 1 and 2). A SD24 interrupt occurs
//  whenever the conversions have completed. Test by applying voltages to the
//  3 input channels and setting a breakpoint at the line indicated
//  below. Run program until it reaches the breakpoint, then use
//  the debugger's watch window to view the conversion results. Results
//  (upper 16 bits only) are stored in three arrays, one for each channel.
//  ACLK = n/a, MCLK = SMCLK = DCO =  ~ 1.1MHz
//  //* For minimum Vcc required for SD24 module - see datasheet       *//
//  //* 100nF cap btw Vref and AVss is recommended when using 1.2V ref *//
//
//                MSP430AFE25x
//             -----------------
//         /|\|              XIN|-
//          | |                 | 
//          --|RST          XOUT|-
//            |                 |
//   Vin1+ -->|A0.0+            |
//   Vin1- -->|A0.0-            |
//   Vin2+ -->|A1.0+            |
//   Vin2- -->|A1.0-            |
//   Vin3+ -->|A2.0+            |
//   Vin3- -->|A2.0-            |
//            |            VREF |---+
//            |                 |   |
//            |                 |  -+- 100nF
//            |                 |  -+-
//            |                 |   |
//            |            AVss |---+
//            |                 |
//
//  Naveen Kala
//  Texas Instruments, Inc
//  March 2011
//  Built with IAR Embedded Workbench Version: 5.20.1
//*****************************************************************************
#include <msp430afe253.h>

#define   Num_of_Results   8

/* Arrays to store SD24 conversion results */
unsigned int Ch0results[Num_of_Results];
unsigned int Ch1results[Num_of_Results];
unsigned int Ch2results[Num_of_Results];

void main(void)
{
  volatile unsigned int i;                  // Use volatile to prevent removal
                                            // by compiler optimization

  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT

  P1DIR |= BIT0+BIT1;                       // Set P1.0, P1.1 to output
  SD24CTL = SD24REFON + SD24SSEL0;          // 1.2V ref, SMCLK
  SD24CCTL0 |= SD24GRP+SD24DF;              // Group with CH1
  SD24CCTL1 |= SD24GRP+SD24DF;              // Group with CH2
  SD24CCTL2 |= SD24IE+SD24DF;               // Enable interrupt
  for (i = 0; i < 0x3600; i++);             // Delay for 1.2V ref startup
  
  SD24CCTL2 |= SD24SC;                      // Set bit to start conversion
  __bis_SR_register(LPM0_bits + GIE);       // Enter LPM0 w/ interrupts
}

#pragma vector=SD24_VECTOR
__interrupt void SD24AISR(void)
{
      static unsigned int index = 0;
    
      switch (SD24IV)
      {
      case 2:                                   // SD24MEM Overflow
        break;
      case 4:                                   // SD24MEM0 IFG
        break;
      case 6:                                   // SD24MEM1 IFG
        break;
      case 8:                                   // SD24MEM2 IFG
        Ch0results[index] = SD24MEM0;           // Save CH0 results (clears IFG)
        Ch1results[index] = SD24MEM1;           // Save CH1 results (clears IFG)
        Ch2results[index] = SD24MEM2;           // Save CH2 results (clears IFG)
        P1OUT ^=BIT0;
        if (++index == Num_of_Results)
        {
          index = 0;                            // SET BREAKPOINT HERE
          P1OUT ^=BIT1;
        }
        break;
      }
}
