//*****************************************************************************
//  MSP430AFE25x Demo - SD24, Single Conversion on a Single Channel Using ISR
//
//  Description: This program uses the SD24 module to perform a single
//  conversion on a single channel. A SD24 interrupt occurs when a
//  conversion has completed. Test by applying a voltage to CH2
//  (A2.0+, A2.0-) and setting a breakpoint at the line indicated below.
//  Run program until it reaches the breakpoint, then use the debugger's
//  watch window to view the conversion result. Conversion result is
//  stored in variable "result".  NOTE: first conversion result won't
//  be available until after reaching breakpoint for the second time.
//  ACLK = n/a, MCLK = SMCLK = DCO =  ~ 1.1MHz
//  //* For minimum Vcc required for SD24 module - see datasheet        *//
//  //* 100nF cap btw Vref and AVss is recommended when using 1.2V ref *//
//
//                 MSP430AFE25x
//             -----------------
//         /|\|              XIN|-
//          | |                 | 
//          --|RST          XOUT|-
//            |                 |
//    Vin+ -->|A2.0+            |
//    Vin- -->|A2.0-            |
//            |                 |
//            |            VREF |---+
//            |                 |   |
//            |                 |  -+- 100nF
//            |                 |  -+-
//            |                 |   |
//            |            AVss |---+
//            |                 |
//
//  Naveen Kala
//  Texas Instruments, Inc
//  March 2011
//  Built with IAR Embedded Workbench Version: 5.20.1
//*****************************************************************************
#include <msp430afe253.h>

unsigned int result;

void main(void)
{
  volatile unsigned int i;                  // Use volatile to prevent removal
                                            // by compiler optimization

  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT

  SD24CTL = SD24REFON+SD24SSEL0;            // 1.2V ref, SMCLK
  SD24CCTL2 |= SD24SNGL+SD24IE ;            // Single conv, enable interrupt
  SD24INCTL2 |= SD24INTDLY0;                // Interrupt on 3rd sample
  for (i = 0; i < 0x3600; i++);             // Delay for 1.2V ref startup

  while (1)
  {
    _NOP();                                 // SET BREAKPOINT HERE    
    SD24CCTL2 |= SD24SC;                    // Set bit to start conversion                                      
    __bis_SR_register(LPM0_bits + GIE);     // Enter LPM0 w/ interrupts
  }
}

#pragma vector=SD24_VECTOR
__interrupt void SD24AISR(void)
{
  switch (SD24IV)
  {
  case 2:                                   // SD24MEM Overflow
    break;
  case 4:                                   // SD24MEM0 IFG
    break;
  case 6:                                   // SD24MEM1 IFG     
    break;
  case 8:                                   // SD24MEM2 IFG
    result = SD24MEM2;                      // Save CH2 results (clears IFG)
    break;
  }

  __bic_SR_register_on_exit(LPM0_bits);        // Exit LPM0
}

