//----------------------------------------------------------------------------------
//	FILE:			Renewable-Main.C
//
//	Description:	Renewable Energy Board: Controlling Boost stage, Single Phase Inverter
//					with Line Synchronization and Buck Battery Charging --> system Main file.  
//
//	Version: 		1.0
//
//  Target:  		TMS320F280x or F2833x
//
//----------------------------------------------------------------------------------
//  Copyright Texas Instruments  2008
//----------------------------------------------------------------------------------
//  Revision History:
//----------------------------------------------------------------------------------
//  Date	  | Description / Status
//----------------------------------------------------------------------------------
// 16-Apr-09 | Release 1.0			Renewable Release (HRN)
//----------------------------------------------------------------------------------
//
// PLEASE READ - Useful notes about this Project

// Although this project is made up of several files, the most important ones are:
//	 "Renewable-Main.C"	- this file
//		- Application Initialization, Peripheral config,
//		- Application Interrupt Service Routine, control tasks
//		- Application management
//		- Slower background code loops and Task scheduling
//	 "Renewable-DevInit_F28xxx.C
//		- Device Initialization, e.g. Clock, PLL, WD, GPIO mapping
//		- Peripheral clock enables
//		- DevInit file will differ per each F28xxx device series, e.g. F280x, F2833x,
//	 "ProjectSettings.h"
//		- Global defines (settings) project selections are found here

// Code is made up of sections, e.g. "FUNCTION PROTOTYPES", "VARIABLE DECLARATIONS" ,..etc
//	each section has FRAMEWORK and USER areas.
//  FRAMEWORK areas provide useful ready made "infrastructure" code which for the most part
//	does not need modification, e.g. Task scheduling, ISR call, GUI interface support,...etc
//  USER areas have functional example code which can be modified by USER to fit their appl.
//
// Variables or parameters used for Multiple Channels are stored in Arrays,
//	the array index is the channel number, note: element zero (index=0) is not used
//
// Code can be compiled with various build options (Incremental Builds IBx), these
//  options are selected in file "FlashingLeds-Settings.h".  Note: "Rebuild All" compile
//  tool bar button must be used if this file is modified.
//----------------------------------------------------------------------------------

#include "ProjectSettings.h"
#include "PeripheralHeaderIncludes.h"
#include "DSP280x_EPWM_defines.h"
#include "sgen.h"
#include "dlog4ch_Renewable.h"
#include "IQmathLib.h"																		 
#include "cntl2p2z.h"
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// FUNCTION PROTOTYPES
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

// -------------------------------- FRAMEWORK --------------------------------------
void DeviceInit(void);
void SCIA_Init();
void SerialHostComms();
void InitFlash();


// State Machine function prototypes
//------------------------------------
// Alpha states
void A0(void);	//state A0
void B0(void);	//state B0
void C0(void);	//state C0

// A branch states
void A1(void);	//state A1
void A2(void);	//state A2
void A3(void);	//state A3
void A4(void);	//state A4

// B branch states
void B1(void);	//state B1
void B2(void);	//state B2
void B3(void);	//state B3
void B4(void);	//state B4

// C branch states
void C1(void);	//state C1
void C2(void);	//state C2
void C3(void);	//state C3
void C4(void);	//state C4

// Variable declarations
void (*Alpha_State_Ptr)(void);	// Base States pointer
void (*A_Task_Ptr)(void);		// State pointer A branch
void (*B_Task_Ptr)(void);		// State pointer B branch
void (*C_Task_Ptr)(void);		// State pointer C branch


// ---------------------------------- USER -----------------------------------------

// Prototype statements for functions found within this file.
void InitEPwm1Example(void);
void InitEPwm2Example(void);
void InitEPwm3Example(void); 
void InitEPwm4Example(void);
void InitEPwm5Example(void);
int  sinewave(void);
interrupt void epwm1_isr(void); 
void AdcInit(void);


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// VARIABLE DECLARATIONS - GENERAL
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

// -------------------------------- FRAMEWORK --------------------------------------

int16	VTimer0[4];					// Virtual Timers slaved off CPU Timer 0
int16	VTimer1[4];					// Virtual Timers slaved off CPU Timer 1
int16	VTimer2[4];					// Virtual Timers slaved off CPU Timer 2
int16	SerialCommsTimer;
int16	CommsOKflg;

// Used for running BackGround in flash, and ISR in RAM
extern Uint16 *RamfuncsLoadStart, *RamfuncsLoadEnd, *RamfuncsRunStart;

extern int16 DLOG_4CH_buff1[];
extern int16 DLOG_4CH_buff2[]; 
extern int16 DLOG_4CH_buff3[];
extern int16 DLOG_4CH_buff4[];   

int ChSel[16];						//ADC Channel Select
extern int DlogCh1; 
extern int DlogCh2; 
extern int DlogCh3; 
extern int DlogCh4; 

int 	EPWM1_TIMER_TBPRD =  2000;  // Period register - 25 KHz, up-down count mode @ 100 MHz

int16   VdcBus;
int16 	VdcIN;
int16 	closeloop_boost = 0, closeloop_buck = 0;
float	Reference;
int16 	Duty_boost = 1999, Duty_buck = 1999;
int16 	Relay;
Uint16	Voltage_U, Voltage_V, Voltage_W, Inv_Out, Battery, LineVoltage; 
int16   Rect_Inv_Out; 
int16	start = 0, first = 1;
//int16		reference;
int16	dutyAB, dutymin = 0, dutymax = 0;
int16	Charging_EN = 0;
int16	boost_cmd = 1999, buck_cmd = 499;

#define tenth _IQ(0.1)
#define rms_scale _IQ(0.01427)  

int16	Phase_U = 0, Phase_V = 0;
int16	mean = 0;
int16	LLmean = 0;
int16 	num_of_samples = 417;
int16 	temp1 = 0, inc = 10, error;
Uint16	ref_duty, cnt3, temp3;
int16	cnt1 = 0, cnt2 = 417, temp2;
int16	Freq_Set = 0;
float32 Vrms = 2.0;


// Instance 2P2Z control loop compensators

CNTL2P2Z cntl2p2z_boost = CNTL2P2Z_DEFAULTS; 
CNTL2P2Z 	chrging_V 	= CNTL2P2Z_DEFAULTS;
CNTL2P2Z 	chrging_I 	= CNTL2P2Z_DEFAULTS;


// PID gain Initialization for the control loops

int16 	Dmax  = 600;
int16	Pgain = 1;							// "counts" (Q0) 
int16	Igain = 1;							// "counts" (Q0) 
int16	Dgain = 0;							// "counts" (Q0)  

int16 	Dmax_Vbatt  = 600;
int16	Pgain_Vbatt = 1;					// "counts" (Q0) 
int16	Igain_Vbatt = 1;					// "counts" (Q0) 
int16	Dgain_Vbatt = 0;					// "counts" (Q0)

int16 	Dmax_Ibatt  = 600;
int16	Pgain_Ibatt = 1;					// "counts" (Q0) 
int16	Igain_Ibatt = 1;					// "counts" (Q0) 
int16	Dgain_Ibatt = 0;					// "counts" (Q0) 

//float pgain = 0.2, igain = 0.1, max_out = 0.6, min_out = 0.0; 
int 	change_coeff = 0;
int 	Boost_Set = 0;
int16 	boost_slew_temp = 0;
int16 	boostSetSlewed = 0;
int16 	boostSlewRate = 100;

int 	BuckV_Set = 0;
int16 	buckV_slew_temp = 0;
int16 	buckVSetSlewed = 0;
int16 	buckVSlewRate = 10;

int 	BuckI_Set = 0;
int16 	buckI_slew_temp = 0;
int16 	buckISetSlewed = 23210;  // = offset @ 0 current
int16 	buckISlewRate = 10;

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// VARIABLE DECLARATIONS - CCS WatchWindow / GUI support
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

// -------------------------------- FRAMEWORK --------------------------------------

//GUI support variables
// sets a limit on the amount of external GUI controls - increase as necessary
int16 	*varSetTxtList[16];					//16 textbox controlled variables
int16 	*varSetBtnList[16];					//16 button controlled variables
int16 	*varSetSldrList[16];				//16 slider controlled variables
int16 	*varGetList[16];					//16 variables sendable to GUI
int16 	*arrayGetList[16];					//16 arrays sendable to GUI		 

// ---------------------------------- USER -----------------------------------------

// Monitor ("Get")							// Display as:
int16	Gui_Ibatt 	 = 0;					// Q10
int16	Gui_Vbatt 	 = 0;					// Q9
int16	Gui_VdcIN 	 = 0;					// Q9
int16	Gui_VdcBus 	 = 0;					// Q9
int16	Gui_VdcBusSet= 0;					// Q9
int16	Gui_VbattSet = 0;					// Q9
int16	Gui_IbattSet = 0;					// Q10

//Scaling Constants (values found via spreadsheet)
int16	K_Ibatt;							// Q15
int16	K_Ioffset;
int16	K_Vbatt;							// Q15
int16	K_VdcIN;							// Q15
int16	K_VdcBus;							// Q15

int16	iK_Ibatt;							// Q14
int16	iK_Vbatt;							// Q14
int16	iK_VdcBus;							// Q14

// Variables for background support only (no need to access)
int16	i;									// common use incrementer
int16	I_HistPtr, V_HistPtr, HistPtr, temp_Scratch;
int16	temp_ChNum, temp_Iout;

// History arrays are used for Running Average calculation (boxcar filter)
// Used for CCS display and GUI only, not part of control loop processing
int16	Hist_Vbatt[HistorySize];
int16	Hist_Ibatt[HistorySize];
int16	Hist_VdcIN[HistorySize];
int16	Hist_VdcBus[HistorySize];

int16 	flag = 0;
int16   batt_good = 1;

int16	CV_threshold = 0, chrg_batt = 0;

int16	CC_charging = 0, CV_charging = 0;

int16	dlog_cnt = 0;

extern DLOG_4CH dlog;

extern void dlog_init(void);
extern void sine_init(void);
extern void ADC_CascSeqCNF(int ChSel[], int ACQPS, int NumConvSEQ1, int mode);

//inline void cntl_2p2z_calc(CNTL2P2Z *v)
void cntl_2p2z_calc(CNTL2P2Z *v)
{	
    // Compute the error
   // Compute the error
    v->Errn = v->Ref - v->Fdb; //e(n) = Ref - Fdbk

    // PreSat = e(n-2)*B2 + e(n-1)*B1 + e(n)*B0 + u(n-2)*A2 + u(n-1)*A1
    v->OutPreSat = _IQmpy(v->Coeff_B2,v->Errn2) + _IQmpy(v->Coeff_B1,v->Errn1) + _IQmpy(v->Coeff_B0,v->Errn) + _IQmpy(v->Coeff_A2,v->Out2) + _IQmpy(v->Coeff_A1,v->Out1);

    // Compute the integral output
    v->Errn2 = v->Errn1;

    // Compute the derivative output
    v->Errn1 = v->Errn;

    // Saturate the output
	v->Out = _IQsat(v->OutPreSat, v->OutMax, v->OutMin);

    // Compute the pre-saturated output
    v->Out2 = v->Out1;

    v->Out1 = v->Out;
 }


#define CNTL2P2Z_MACRO(v)\
    v.Errn = v.Ref - v.Fdb;\
	v.temp1 = _IQ26mpy(v.Coeff_B0,_IQtoIQ26(v.Errn));\
    v.OutPreSat = _IQ26mpy(v.Coeff_B2,v.Errn2) + _IQ26mpy(v.Coeff_B1,v.Errn1) + v.temp1 + _IQ26mpy(v.Coeff_A2,v.Out2) + _IQ26mpy(v.Coeff_A1,v.Out1);\
    v.Out2 = v.Out1;\
    v.Errn2 = v.Errn1;\
	v.Out1 = _IQsat(v.OutPreSat, v.OutMax, v.OutMin);\
    v.Errn1 = _IQtoIQ26(v.Errn);\
	v.Out = _IQ26toIQ(v.Out1);


#define CNTL2P2Z_MACRO2(sec_v)\
    sec_v.Errn = sec_v.Ref - sec_v.Fdb;\
	sec_v.temp1 = _IQ26mpy(sec_v.Coeff_B0,_IQtoIQ26(sec_v.Errn));\
    sec_v.OutPreSat = _IQ26mpy(sec_v.Coeff_B2,sec_v.Errn2) + _IQ26mpy(sec_v.Coeff_B1,sec_v.Errn1) + sec_v.temp1 + _IQ26mpy(sec_v.Coeff_A2,sec_v.Out2) + _IQ26mpy(sec_v.Coeff_A1,sec_v.Out1);\
    sec_v.Out2 = sec_v.Out1;\
    sec_v.Errn2 = sec_v.Errn1;\
	sec_v.Out1 = _IQsat(sec_v.OutPreSat, sec_v.OutMax, sec_v.OutMin);\
    sec_v.Errn1 = _IQtoIQ26(sec_v.Errn);\
	sec_v.Out = _IQ26toIQ(sec_v.Out1);




//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// MAIN CODE - starts here
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
void main(void)
{
//=================================================================================
//	INITIALISATION - General
//=================================================================================

//-------------------------------- FRAMEWORK --------------------------------------

	DeviceInit();	// Device Life support & GPIO
	SCIA_Init();  	// Initalize the Serial Comms A peripheral

// Only used if running from FLASH
// Note that the variable FLASH is defined by the compiler with -d FLASH
// (see TwoChannelBuck.pjt file)
#ifdef FLASH		
// Copy time critical code and Flash setup code to RAM
// The  RamfuncsLoadStart, RamfuncsLoadEnd, and RamfuncsRunStart
// symbols are created by the linker. Refer to the linker files. 
	MemCopy(&RamfuncsLoadStart, &RamfuncsLoadEnd, &RamfuncsRunStart);

// Call Flash Initialization to setup flash waitstates
// This function must reside in RAM
	InitFlash();	// Call the flash wrapper init function
#endif //(FLASH)

// Timing sync for background loops
// Timer period definitions found in PeripheralHeaderIncludes.h
	CpuTimer0Regs.PRD.all =  mSec1;		// A tasks
	CpuTimer1Regs.PRD.all =  mSec50;	// B tasks
	CpuTimer2Regs.PRD.all =  mSec100;	// C tasks

// Tasks State-machine init
	Alpha_State_Ptr = &A0;
	A_Task_Ptr = &A1;
	B_Task_Ptr = &B1;
	C_Task_Ptr = &C1;

	VTimer0[0] = 0;	
	VTimer1[0] = 0;
	VTimer2[0] = 0;
	CommsOKflg = 0;
	SerialCommsTimer = 0;

	K_Ibatt 	= 19926;						// Q15
	K_Ioffset 	= 23210;						// Q15
	K_Vbatt 	= 16896;						// Q15
	K_VdcIN 	= 16865;						// Q15
	K_VdcBus 	= 32256;						// Q15

	iK_Vbatt 	= 31775;						// Q14
	iK_Ibatt 	= 26943;						// Q14
	iK_VdcBus 	= 16644;						// Q14

	I_HistPtr 	= 0;
	V_HistPtr 	= 0;
	HistPtr 	= 0;
	
//=================================================================================
//	INITIALISATION - GUI connections
//=================================================================================
// Use this section only if you plan to "Instrument" your application using the 
// Microsoft C# freeware GUI Template provided by TI

	//"Set" variables
	//---------------------------------------
	// assign GUI variable Textboxes to desired "setable" parameter addresses
		//varSetTxtList[0] = &Var0;
	varSetTxtList[0] = &Gui_VdcBusSet;
	varSetTxtList[1] = &Gui_VbattSet;
	varSetTxtList[2] = &Gui_IbattSet;

	// assign GUI Buttons to desired flag addresses
		//varSetBtnList[0] = &Var1;
	varSetBtnList[0] = &start;
	varSetBtnList[1] = &closeloop_boost;
	varSetBtnList[2] = &closeloop_buck;
	varSetBtnList[3] = &Charging_EN;
	varSetBtnList[4] = &CV_charging;
	varSetBtnList[5] = &CC_charging;

	// assign GUI Sliders to desired "setable" parameter addresses
	//varSetSldrList[0] = &Var2;

  	varSetSldrList[0] = &boost_cmd;
	varSetSldrList[1] = &buck_cmd;

	//"Get" variables
	//---------------------------------------
	// assign a GUI "getable" parameter address
	//varGetList[0] = &Var3;
	varGetList[0] = &Gui_VdcIN;
	varGetList[1] = &Gui_VdcBus;
	varGetList[2] = &Gui_Vbatt;
	varGetList[3] = &Gui_Ibatt;
	varGetList[4] = &boost_cmd;
	varGetList[5] = &buck_cmd;

	// assign a GUI "getable" parameter array address
	// 		only need to set initial position of array, program will run through it 
	//       based on the array length specified in the GUI
	arrayGetList[0] = DLOG_4CH_buff1;		// Sine Ref
	arrayGetList[1] = DLOG_4CH_buff2;		// ZCD
 	arrayGetList[2] = DLOG_4CH_buff4; 		// Inverter Output


//==================================================================================
//	INITIALISATION - Peripherals used for support
//==================================================================================

// ---------------------------------- USER -----------------------------------------
//  Put peripheral initialisation here
// ePWMs 1, 2 and 3 modules drive the three legs (six outputs) of the inverter.
// Only ePWMs 1 and 2 are used here (Single phase output)
// ePWM 4A drives the boost stage, whel ePWM 5A/5B drive the 
// DC-DC buck battery charging stage.

	InitEPwm1Example();   		
    InitEPwm2Example();
    InitEPwm3Example();
    InitEPwm4Example();
	InitEPwm5Example();

	EALLOW;
	SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 1;
	EDIS;

	dlog_init();
	sine_init(); 

// "Raw" (R) ADC measurement name defines, ADC result registers are mirrored
// in single cycle read memory for faster access..

	ChSel[0] =  2; 		// ADCINA2 - VdcIN
	ChSel[1] = 10; 		// ADCINB2 - VdcBus
	ChSel[2] =  3;		// ADCINA3 - Voltage phase U
	ChSel[3] =  4;		// ADCINA4 - Voltage phase V
	ChSel[4] =  1;		// ADCINA1 - Battery voltage
	ChSel[5] =  6;		// ADCINA6 - Input AC signal
	ChSel[6] =  9;		// ADCINB1 - Battery charging current 

	ADC_CascSeqCNF(ChSel, 2, 7, 0);	// ACQPS=2, #Conv=4, Mode= Start/Stop (0)

// Configure the Start of Conversion for the ADC.
    EPwm4Regs.ETSEL.bit.SOCAEN = 1;        		// Enable SOC on A group
    EPwm4Regs.ETSEL.bit.SOCASEL = ET_CTR_PRD;   // Select SOC from counter = PRD
    EPwm4Regs.ETPS.bit.SOCAPRD = ET_1ST;        // Generate pulse on 1st event 


//=================================================================================
//	INTERRUPT & ISR INITIALISATION (best to run this section after other initialisation)
//=================================================================================

// GPIO24 is an external input - coming from zero crossing detection circuit   
    EALLOW;
    GpioCtrlRegs.GPAMUX2.bit.GPIO24 	= 0;    	// Configured the Pin as GPIO24
    GpioCtrlRegs.GPADIR.bit.GPIO24 		= 0;      	// GPIO24 is an input
    GpioCtrlRegs.GPAQSEL2.bit.GPIO24 	= 2;      	// XINT1 Synch to SYSCLKOUT only
    GpioCtrlRegs.GPACTRL.bit.QUALPRD2 	= 0x0A;
//  GPIO24 is XINT1
    GpioIntRegs.GPIOXINT1SEL.bit.GPIOSEL = 24;    	// XINT1 is GPI20  
    EDIS;

// Configure XINT1
//    XIntruptRegs.XINT1CR.bit.POLARITY 	= 0;      	// Falling edge interrupt
    XIntruptRegs.XINT1CR.bit.POLARITY 	= 1;      	// Rising edge interrupt

// Enable XINT1 
    XIntruptRegs.XINT1CR.bit.ENABLE 	= 1;       	// Enable XINT1

	PieCtrlRegs.PIEIER1.bit.INTx4 = 1;				// PIE level enable, Grp1 / Int4

// ISR where the control loops are executed.
	EALLOW;
	PieVectTable.EPWM1_INT = &epwm1_isr;			// Map Interrupt
	EDIS;
	PieCtrlRegs.PIEIER3.bit.INTx1 = 1;				// PIE level enable, Grp3 / Int1
    EPwm1Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;      	// Select INT on Zero event
    EPwm1Regs.ETSEL.bit.INTEN = 1;                 	// Enable INT
    EPwm1Regs.ETPS.bit.INTPRD = ET_1ST;

// Enable Peripheral, global Ints and higher priority real-time debug events:
//  IER |= M_INT1; // This is intentionally turned off - the system will poll the flag
	IER |= M_INT3; 

	EINT;   		// Enable Global interrupt INTM
	ERTM;   		// Enable Global realtime interrupt DBGM

	Relay = 0; 		// Input Relay turned Off - NO.


//=================================================================================
//	BACKGROUND (BG) LOOP
//=================================================================================

//--------------------------------- FRAMEWORK -------------------------------------
	for(;;)  //infinite loop
	{

	  if (Relay == 1)
   		{
   			GpioDataRegs.GPASET.bit.GPIO17 = 1;
   		}
   	  else
   		{
   			GpioDataRegs.GPACLEAR.bit.GPIO17 = 1;
	    } 

		// State machine entry & exit point
		//===========================================================
		(*Alpha_State_Ptr)();	// jump to an Alpha state (A0,B0,...)
		//===========================================================
	}
} //END MAIN CODE



//=================================================================================
//	STATE-MACHINE SEQUENCING AND SYNCRONIZATION
//=================================================================================

//--------------------------------- FRAMEWORK -------------------------------------
void A0(void)
{
	// loop rate synchronizer for A-tasks
	if(CpuTimer0Regs.TCR.bit.TIF == 1)
	{
		CpuTimer0Regs.TCR.bit.TIF = 1;	// clear flag

		//-----------------------------------------------------------
		(*A_Task_Ptr)();		// jump to an A Task (A1,A2,A3,...)
		//-----------------------------------------------------------

		VTimer0[0]++;			// virtual timer 0, instance 0 (spare)
		SerialCommsTimer++;
	}

	Alpha_State_Ptr = &B0;		// Comment out to allow only A tasks
}

void B0(void)
{
	// loop rate synchronizer for B-tasks
	if(CpuTimer1Regs.TCR.bit.TIF == 1)
	{
		CpuTimer1Regs.TCR.bit.TIF = 1;				// clear flag

		//-----------------------------------------------------------
		(*B_Task_Ptr)();		// jump to a B Task (B1,B2,B3,...)
		//-----------------------------------------------------------
		VTimer1[0]++;			// virtual timer 1, instance 0 (spare)
	}

	Alpha_State_Ptr = &C0;		// Allow C state tasks
}

void C0(void)
{
	// loop rate synchronizer for C-tasks
	if(CpuTimer2Regs.TCR.bit.TIF == 1)
	{
		CpuTimer2Regs.TCR.bit.TIF = 1;				// clear flag

		//-----------------------------------------------------------
		(*C_Task_Ptr)();		// jump to a C Task (C1,C2,C3,...)
		//-----------------------------------------------------------
		VTimer2[0]++;			//virtual timer 2, instance 0 (spare)
	}

	Alpha_State_Ptr = &A0;	// Back to State A0
}



//%%%%%%%%%%%%%%%    A-Tasks:   %%%%%%%%%%%%%%%%%%%%%%%%%
//=====================================================================
void A1(void) // SCI-GUI
//=====================================================================
{
	SerialHostComms();
	
	//-------------------
	//the next time CpuTimer0 'counter' reaches Period value go to A2
	A_Task_Ptr = &A2;
	//-------------------
}

//=====================================================================
void A2(void) // Enable/Disable for different power stages - Buck, Boost and Inverter
//=====================================================================
{	

	if ((start == 1)&&(first == 1)) // Setting Start enables Boost and Inverter switches
	{
	EALLOW;
	EPwm1Regs.TZCLR.bit.OST = 1; 
	EPwm2Regs.TZCLR.bit.OST = 1;
//	EPwm3Regs.TZCLR.bit.OST = 1;	// 3rd Inverter leg not enabled - Single Phase Output
	EPwm4Regs.TZCLR.bit.OST = 1;  	// Boost switch enabled
	EDIS;
	Reference = 3000;
	first = 0;
	}
	if (start == 0)
	{
//	closeloop_boost = 0;
	EALLOW;
	EPwm4Regs.TZFRC.bit.OST = 1; 	// Boost switch disabled
	EPwm1Regs.TZFRC.bit.OST = 1;
	EPwm2Regs.TZFRC.bit.OST = 1;
	EPwm3Regs.TZFRC.bit.OST = 1;
	EDIS;
	first = 1;
	}

	if (Charging_EN == 1)			// This enables the DC-DC buck switch
	{
	EALLOW;
	EPwm5Regs.TZCLR.bit.OST = 1;
	EDIS;
	}

	if (Charging_EN == 0)			// Disable buck switch
	{
	EALLOW;
	EPwm5Regs.TZFRC.bit.OST = 1;
	EDIS;
	EPwm5Regs.CMPA.half.CMPA = 499;	// Reset to '0' duty
    }

	//-------------------
	//the next time CpuTimer0 'counter' reaches Period value go to A1
	A_Task_Ptr = &A1; 	// To make task A3 active, change &A1 to &A3
	//-------------------
}

//=====================================================================
void A3(void) // SPARE (not active)
//=====================================================================
{
	//-----------------
	//the next time CpuTimer0 'counter' reaches Period value go to A1

	A_Task_Ptr = &A1; 	// To make task A4 active, change &A1 to &A4
	//-----------------
}


//=====================================================================
void A4(void) // SPARE (not active)
//=====================================================================
{
	//-----------------
	//the next time CpuTimer0 'counter' reaches Period value go to A1
	A_Task_Ptr = &A1; 	// After Task A4, start over with task A1
	//-----------------
}


//%%%%%%%%%%%%%%%    B-Tasks:   %%%%%%%%%%%%%%%%%%%%%%%%%
//=====================================================================
void B1(void) // Current Dashboard measurements
//===================================================================== 
{
// Current measurement calculated by:
//	Gui_Ibatt = IbattAvg * K_Ibatt, where IbattAvg = sum of 8 Ibatt samples
//----------------------------------------------------------------
// view following variables in Watch Window as:
//		Gui_Ibatt = Q10

	I_HistPtr++;
	if (I_HistPtr >= HistorySize)	I_HistPtr = 0;
	Hist_Ibatt[I_HistPtr] = AdcMirror.ADCRESULT6; 		// Raw ADC result (Q12)
	temp_Scratch = 0;
	for(i=0; i<HistorySize; i++)	temp_Scratch = temp_Scratch + Hist_Ibatt[i]; // Q12 * 8
	temp_Scratch = ( ((long) temp_Scratch -  (long) K_Ioffset) * (long) K_Ibatt ) >> 15; // (Q15 * Q15)>>15 = Q15
	
	if (temp_Scratch < 0) Gui_Ibatt = 0;
	else Gui_Ibatt = temp_Scratch;


// Current setting calculated by: (For CC_Charging)
// BuckI_Set = Gui_IbattSet * iK_Ibatt + K_Ioffset, where iK_Ibatt = 1/K_Ibatt (i.e. inverse K_Ibatt)
// view and set following variable in Watch Window as:
//		Gui_IbattSet = Q10 (Used as Q15 below)

	BuckI_Set = (( (long) Gui_IbattSet * (long) iK_Ibatt ) >> 14) + (long) K_Ioffset; // (Q15 * Q14) >> 14 = Q15

	//-----------------
	//the next time CpuTimer1 'counter' reaches Period value go to B2
	B_Task_Ptr = &B2;	
	//-----------------
}

//=====================================================================
void B2(void) //  Voltage Dashboard measurements
//=====================================================================
{
// Voltage measurement calculated by:
//	Gui_Vbatt = VbattAvg * K_Vbatt, where VbattAvg = sum of 8 Vbatt samples
//	Gui_VdcIN = VdcINAvg * K_VdcIN, where VdcINAvg = sum of 8 VdcIN samples
//	Gui_VdcBus = VdcBusAvg * K_VdcBus, where VdcBusAvg = sum of 8 VdcBus samples
//----------------------------------------------------------------
// view following variables in Watch Window as:
//	Gui_Vbatt = Q9
//	Gui_VdcIN = Q9
//	Gui_VdcBus = Q9

	V_HistPtr++;
	if (V_HistPtr >= HistorySize)	V_HistPtr = 0;

	Hist_Vbatt[V_HistPtr] = AdcMirror.ADCRESULT4; 		// Raw ADC result (Q12)
	temp_Scratch = 0;
	for(i=0; i<HistorySize; i++)	temp_Scratch = temp_Scratch + Hist_Vbatt[i]; // Q12 * 8
	Gui_Vbatt = ( (long) temp_Scratch * (long) K_Vbatt ) >> 15; // (Q15 * Q15)>>15 = Q15
	

	Hist_VdcIN[V_HistPtr] = AdcMirror.ADCRESULT0; 		// Raw ADC result (Q12)
	temp_Scratch = 0;
	for(i=0; i<HistorySize; i++)	temp_Scratch = temp_Scratch + Hist_VdcIN[i]; // Q12 * 8
	Gui_VdcIN = ( (long) temp_Scratch * (long) K_VdcIN ) >> 15; // (Q15 * Q15)>>15 = Q15

	Hist_VdcBus[V_HistPtr] = AdcMirror.ADCRESULT1; 		// Raw ADC result (Q12)
	temp_Scratch = 0;
	for(i=0; i<HistorySize; i++)	temp_Scratch = temp_Scratch + Hist_VdcBus[i]; // Q12 * 8
	Gui_VdcBus = ( (long) temp_Scratch * (long) K_VdcBus ) >> 15; // (Q15 * Q15)>>15 = Q15


// Voltage setting calculated by: (For CV_Charging)
// BuckV_Set = Gui_VbattSet * iK_Vbatt, where iK_Vbatt = 1/K_Vbatt (i.e. inverse K_Vbatt)
// view and set following variable in Watch Window as:
//		Gui_VbattSet = Q9 (Used as Q15 below) 

	BuckV_Set = ( (long) Gui_VbattSet * (long) iK_Vbatt ) >> 14; // (Q15 * Q14) >> 14 = Q15

// Voltage setting calculated by: (For Closed Loop Boost)
// Boost_Set = Gui_VdcBusSet * iK_VdcBus, where iK_VdcBus = 1/K_VdcBus (i.e. inverse K_VdcBus)
// view and set following variable in Watch Window as:
//		Gui_VdcBusSet = Q9 (Used as Q15 below)  

	Boost_Set = ( (long) Gui_VdcBusSet * (long) iK_VdcBus ) >> 14; // (Q15 * Q14) >> 14 = Q15

	//-----------------
	//the next time CpuTimer1 'counter' reaches Period value go to B1
	B_Task_Ptr = &B1;	// To make task B3 active, change &B1 to &B3
	//-----------------
}

//=====================================================================
void B3(void) //  SPARE (not active)
//=====================================================================
{
	//-----------------
	//the next time CpuTimer1 'counter' reaches Period value go to B1
	B_Task_Ptr = &B1;	// To make task B4 active, change &B1 to &B4
	//-----------------
}

//=====================================================================
void B4(void) //  SPARE (not active)
//=====================================================================
{
	//-----------------
	//the next time CpuTimer1 'counter' reaches Period value go to B1
	B_Task_Ptr = &B1;	// After Task B4, start over with task B1
	//-----------------
}


//=================================================================================
//	C - TASKS
//=================================================================================

//--------------------------------- USER ------------------------------------------

//=====================================================================
void C1(void) 	// Toggle GPIO-34 (LD3); 
//=====================================================================
{
	// task runs every 400ms = 100ms (CpuTimer2 period) * 4 (# of C Tasks)
	GpioDataRegs.GPBTOGGLE.bit.GPIO34 = 1;
	#ifdef DSP2833x_DEVICE_H 
	GpioDataRegs.GPBTOGGLE.bit.GPIO39 = 1;
	#endif

	//-----------------
	//the next time CpuTimer2 'counter' reaches Period value go to C2
	C_Task_Ptr = &C2;	
	//-----------------

}

//=====================================================================
void C2(void) //  Change Control loop coefficients - PID gains mapped to 2P2Z coeeficients
//=====================================================================
{

//if (change_coeff == 1)
//{ 
	cntl2p2z_boost.Coeff_B2 = Dgain * 67108;									// B2
	cntl2p2z_boost.Coeff_B1 = (Igain - Pgain - Dgain - Dgain)*67108;			// B1
	cntl2p2z_boost.Coeff_B0 = (Pgain + Igain + Dgain)*67108;					// B0
	cntl2p2z_boost.Coeff_A2 = 0x00000000;										// A2
	cntl2p2z_boost.Coeff_A1 = 0x04000000;									// A1 Corresponds to 1 in Q26
	cntl2p2z_boost.OutMax = Dmax * 67108;										// Clamp Hi limit (Q26)
	cntl2p2z_boost.OutMin = 0x00000000;									// Clamp Lo

	chrging_V.Coeff_B2 = Dgain_Vbatt * 67108;									// B2
	chrging_V.Coeff_B1 = (Igain_Vbatt - Pgain_Vbatt - Dgain_Vbatt - Dgain_Vbatt)*67108;			// B1
	chrging_V.Coeff_B0 = (Pgain_Vbatt + Igain_Vbatt + Dgain_Vbatt)*67108;					// B0
	chrging_V.Coeff_A2 = 0x00000000;										// A2
	chrging_V.Coeff_A1 = 0x04000000;									// A1 Corresponds to 1 in Q26
	chrging_V.OutMax = Dmax_Vbatt * 67108;										// Clamp Hi limit (Q26)
	chrging_V.OutMin = 0x00000000;									// Clamp Lo

	chrging_I.Coeff_B2 = Dgain_Ibatt * 67108;									// B2
	chrging_I.Coeff_B1 = (Igain_Ibatt - Pgain_Ibatt - Dgain_Ibatt - Dgain_Ibatt)*67108;			// B1
	chrging_I.Coeff_B0 = (Pgain_Ibatt + Igain_Ibatt + Dgain_Ibatt)*67108;					// B0
	chrging_I.Coeff_A2 = 0x00000000;										// A2
	chrging_I.Coeff_A1 = 0x04000000;									// A1 Corresponds to 1 in Q26
	chrging_I.OutMax = Dmax_Ibatt * 67108;										// Clamp Hi limit (Q26)
	chrging_I.OutMin = 0x00000000;									// Clamp Lo

//}
	//-----------------
	//the next time CpuTimer2 'counter' reaches Period value go to C3
	C_Task_Ptr = &C3;	
	//-----------------
}


//=====================================================================
void C3(void) //  Boost stage Slew Rate
//=====================================================================
// This is an example code for implementing the slew rate control in
// a slower state machine instead of implementing it in the ISR.
{
// boostSlewRate has to be a positive value
boost_slew_temp = Boost_Set - boostSetSlewed;

if (boost_slew_temp >= boostSlewRate) // Positive Command
{
	boostSetSlewed = boostSetSlewed + boostSlewRate;
}
else
	{
	if ((-1)*(boost_slew_temp) >= boostSlewRate) // Negative Command
		{
		boostSetSlewed = boostSetSlewed - boostSlewRate;
		}
	else boostSetSlewed = Boost_Set;
	}

	//-----------------
	//the next time CpuTimer2 'counter' reaches Period value go to C4
	C_Task_Ptr = &C4;	
	//-----------------
}


//=====================================================================
void C4(void) //  Battery Charging stage Slew Rate
//=====================================================================
// This is an example code for implementing the slew rate control in
// a slower state machine instead of implementing it in the ISR.
{
// Constant Voltage (CV) Charging Slew Limit
// buckVSlewRate has to be a positive value
buckV_slew_temp = BuckV_Set - buckVSetSlewed;

if (buckV_slew_temp >= buckVSlewRate) // Positive Command
{
	buckVSetSlewed = buckVSetSlewed + buckVSlewRate;
}
else
	{
	if ((-1)*(buckV_slew_temp) >= buckVSlewRate) // Negative Command
		{
		buckVSetSlewed = buckVSetSlewed - buckVSlewRate;
		}
	else buckVSetSlewed = BuckV_Set;
	}


// Constant Current (CC) Charging Slew Limit
// buckISlewRate has to be a positive value 
buckI_slew_temp = BuckI_Set - buckISetSlewed;

if (buckI_slew_temp >= buckISlewRate) // Positive Command
{
	buckISetSlewed = buckISetSlewed + buckISlewRate;
}
else
	{
	if ((-1)*(buckI_slew_temp) >= buckISlewRate) // Negative Command
		{
		buckISetSlewed = buckISetSlewed - buckISlewRate;
		}
	else buckISetSlewed = BuckI_Set;
	}

	//-----------------
	//the next time CpuTimer2 'counter' reaches Period value go to C1
	C_Task_Ptr = &C1;	
	//-----------------
}


interrupt void epwm1_isr(void)
 {

//-------------------------------------------------------------------------------------------------
	Phase_U = (AdcRegs.ADCRESULT2>>1)&0x7FFF;	// measure phase U
	Phase_V = (AdcRegs.ADCRESULT3>>1)&0x7FFF;	// measure phase V

  	Inv_Out = Phase_V - Phase_U;
 	LineVoltage = AdcRegs.ADCRESULT5^0x8000;

	if(Phase_U-Phase_V<0)						// rectify the inverter output
	{Rect_Inv_Out=Phase_V-Phase_U;}
	else  	
	{Rect_Inv_Out=Phase_U-Phase_V;}
	
	cnt1++;										// count the number of samples in each period
	mean=mean+_IQdiv(Rect_Inv_Out,_IQ(num_of_samples));	// calculate the mean of rectified inveter output
	if (PieCtrlRegs.PIEIFR1.bit.INTx4 == 1)				// if zero crossing pulse is received
	{ 
	  /*LLmean=mean;						// update LLmean
	  mean=0;	
   	  num_of_samples=cnt1;				// update num_of_samples which is typically ~ 417 at 60Hz (=25000/60)
	  cnt1=0;*/
	}
	
	if (LLmean<_IQmpy(_IQ(Vrms),rms_scale))		// 'Vrms' is the reference rms value set by user from watch window
							// 'rms_scale' converts it to mean 
	{ if (((ref_duty+inc)<0x7FFF) && (cnt3==temp3)) ref_duty=ref_duty+inc;}	// ref_duty adjust the gain of reference sine waveform in sgen1.c and saturates at 0x7FFF
	else 
	{if (LLmean>_IQmpy(_IQ(Vrms),rms_scale))						// cnt3 slows down the response of hysteresis control 
		{ if (((ref_duty-inc)>0x0000) && (cnt3==temp3)) ref_duty=ref_duty-inc;}	// ref_duty lower limit saturation is 0x0000
	}

	error=(LLmean-_IQmpy(_IQ(Vrms),rms_scale));		// find error

	if (error<5 && error>-5)				// depending on how large the error is, adjust the speed of loop and incremental steps
		{inc=0; temp3=100;}
	else if (error<25 && error>-25)
		{inc=1; temp3=75;}
	else
		{inc=10; temp3=20;}

	 if(cnt3==temp3){ cnt3=0;}  cnt3++;			//reset cnt3

   	EPwm1Regs.ETCLR.bit.INT = 1;
	dutyAB = sinewave();

	EPwm1Regs.CMPA.half.CMPA = dutyAB;

	EPwm2Regs.CMPA.half.CMPA = dutyAB;

 	VdcIN = AdcRegs.ADCRESULT0>>1;
  	VdcBus = AdcRegs.ADCRESULT1>>1;
  	Battery = AdcRegs.ADCRESULT4>>1;

  	Voltage_U = AdcRegs.ADCRESULT2>>1;
  	Voltage_V = AdcRegs.ADCRESULT3>>1;

  	Inv_Out = Voltage_V - Voltage_U; 			

 	LineVoltage = AdcRegs.ADCRESULT5^0x8000;

 	DlogCh3 = LineVoltage;
 	DlogCh4 = Inv_Out;


	if(closeloop_boost == 1)
	{
      cntl2p2z_boost.Ref = boostSetSlewed;
	  cntl2p2z_boost.Fdb = VdcBus; 
	  
      CNTL2P2Z_MACRO(cntl2p2z_boost);
//	  cntl_2p2z_calc(&cntl2p2z_boost);

	  boost_cmd = EPWM1_TIMER_TBPRD - (cntl2p2z_boost.Out*EPWM1_TIMER_TBPRD)/32768;//changed from 2000 to 10

	}
	
	if (boost_cmd < 1400) boost_cmd = 1400;		// Clamping Max duty
	
    EPwm4Regs.CMPA.half.CMPA = boost_cmd;

	if(closeloop_buck == 1)
	{
	 if (CV_charging == 1)
	  {
      	chrging_V.Ref = buckVSetSlewed;
	  	chrging_V.Fdb = Battery; 
		CNTL2P2Z_MACRO(chrging_V);
		buck_cmd = EPWM1_TIMER_TBPRD/4 - (chrging_V.Out*EPWM1_TIMER_TBPRD/4)/32768;//changed from 2000 to 10
	  }
	 else
	  {
		  if (CC_charging == 1)
	  		{
      			chrging_I.Ref = buckISetSlewed;
	  			chrging_I.Fdb = AdcMirror.ADCRESULT6<<3; 
				CNTL2P2Z_MACRO(chrging_I);
			    buck_cmd = EPWM1_TIMER_TBPRD/4 - (chrging_I.Out*EPWM1_TIMER_TBPRD/4)/32768;//changed from 2000 to 10
			}
	   }
	  
//	}

	}

	if (dlog_cnt == 3)  
	{
		dlog.update(&dlog);
		dlog_cnt = 0;
	}
	else dlog_cnt++;

	if (buck_cmd < 200) buck_cmd = 200;	// Clamping Max duty

	EPwm5Regs.CMPA.half.CMPA = buck_cmd;


// Reinitialize for next ADC sequence
  AdcRegs.ADCTRL2.bit.RST_SEQ1 = 1;         // Reset SEQ1
//  AdcRegs.ADCST.bit.INT_SEQ1_CLR = 1;       // Clear INT SEQ1 bit

  PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
 }


void InitEPwm1Example()
{
// Setup TBCLK
   EPwm1Regs.TBPRD = EPWM1_TIMER_TBPRD;           // Set timer period 801 TBCLKs
   EPwm1Regs.TBPHS.half.TBPHS = 0x0000;           // Phase is 0
   EPwm1Regs.TBCTR = 0x0000;                      // Clear counter
   
// Setup counter mode
//   EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up
   EPwm1Regs.TBCTL.bit.CTRMODE = TB_FREEZE; // Count up
   EPwm1Regs.TBCTL.bit.PHSEN = TB_DISABLE;        // Disable phase loading
   EPwm1Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
   EPwm1Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm1Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO;


// Setup ADC operation
/*   EPwm1Regs.ETSEL.bit.SOCAEN = 1;        // Enable SOC on A group
   EPwm1Regs.ETSEL.bit.SOCASEL = 1;       // Select SOC from counter = 0
   EPwm1Regs.ETPS.bit.SOCAPRD = 1;        // Generate pulse on 1st event 
*/

// Setup shadowing
   EPwm1Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm1Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm1Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero
   EPwm1Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   

// Setup Dead Band
   EPwm1Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
   EPwm1Regs.DBCTL.bit.IN_MODE = DBA_ALL;
   EPwm1Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
   EPwm1Regs.DBRED = 200;
   EPwm1Regs.DBFED = 200;


// Set actions
   EPwm1Regs.AQCTLA.bit.CAU = AQ_SET;             // Set PWM1A on event A, up count
   EPwm1Regs.AQCTLA.bit.CAD = AQ_CLEAR;           // Clear PWM1A on event A, down count

   EPwm1Regs.AQCTLB.bit.CAU = AQ_CLEAR;             // Set PWM1B on event B, up count
   EPwm1Regs.AQCTLB.bit.CAD = AQ_SET;           // Clear PWM1B on event B, down count


// Setup Trip functions
   EALLOW;
   EPwm1Regs.TZSEL.bit.OSHT1 = 1;
   EPwm1Regs.TZCTL.bit.TZA = 2;
   EPwm1Regs.TZCTL.bit.TZB = 2;
   EPwm1Regs.TZFRC.bit.OST = 1;
   EDIS; 

// Start the timer but the PWM outputs are inactive
   EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up/down

     
}

void InitEPwm2Example()
{
// Setup TBCLK
   EPwm2Regs.TBPRD = EPWM1_TIMER_TBPRD;           // Set timer period 801 TBCLKs
   EPwm2Regs.TBPHS.half.TBPHS = 0x0000;           // Phase is 0
   EPwm2Regs.TBCTR = 0x0000;                      // Clear counter
   
// Setup counter mode
//   EPwm2Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up
   EPwm2Regs.TBCTL.bit.CTRMODE = TB_FREEZE; // Count up
   EPwm2Regs.TBCTL.bit.PHSEN = TB_ENABLE;        // Disable phase loading
   EPwm2Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
   EPwm2Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm2Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;

// Setup Dead Band
   EPwm2Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
   EPwm2Regs.DBCTL.bit.IN_MODE = DBA_ALL;
   EPwm2Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
   EPwm2Regs.DBRED = 200;
   EPwm2Regs.DBFED = 200;

// Setup shadowing
   EPwm2Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm2Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm2Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero
   EPwm2Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   


// Set actions
   EPwm2Regs.AQCTLA.bit.CAU = AQ_CLEAR;             // Set PWM1A on event A, up count
   EPwm2Regs.AQCTLA.bit.CAD = AQ_SET;           // Clear PWM1A on event A, down count

   EPwm2Regs.AQCTLB.bit.CAU = AQ_SET;             // Set PWM1B on event B, up count
   EPwm2Regs.AQCTLB.bit.CAD = AQ_CLEAR;           // Clear PWM1B on event B, down count


// Setup Trip functions
   EALLOW;
   EPwm2Regs.TZSEL.bit.OSHT1 = TZ_ENABLE;
   EPwm2Regs.TZCTL.bit.TZA = TZ_FORCE_LO;
   EPwm2Regs.TZCTL.bit.TZB = TZ_FORCE_LO;
   EPwm2Regs.TZFRC.bit.OST = 1; 
   EDIS;

// Start the counter but the PWM output is inactive
   EPwm2Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;
 
}

void InitEPwm3Example()
{
// Setup TBCLK
   EPwm3Regs.TBPRD = EPWM1_TIMER_TBPRD;           // Set timer period 801 TBCLKs
   EPwm3Regs.TBPHS.half.TBPHS = 0x0000;           // Phase is 0
   EPwm3Regs.TBCTR = 0x0000;                      // Clear counter
   
// Setup counter mode
//   EPwm3Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up
   EPwm3Regs.TBCTL.bit.CTRMODE = TB_FREEZE; // Count up
   EPwm3Regs.TBCTL.bit.PHSEN = TB_DISABLE;        // Disable phase loading
   EPwm3Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
   EPwm3Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm3Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;

// Setup Dead Band
   EPwm3Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
   EPwm3Regs.DBCTL.bit.IN_MODE = DBA_ALL;
   EPwm3Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
   EPwm3Regs.DBRED = 200;
   EPwm3Regs.DBFED = 200;

// Setup shadowing
   EPwm3Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm3Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm3Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero
   EPwm3Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   


// Set actions
   EPwm3Regs.AQCTLA.bit.CAU = AQ_CLEAR;             // Set PWM1A on event A, up count
   EPwm3Regs.AQCTLA.bit.CAD = AQ_SET;           // Clear PWM1A on event A, down count

   EPwm3Regs.AQCTLB.bit.CAU = AQ_SET;             // Set PWM1B on event B, up count
   EPwm3Regs.AQCTLB.bit.CAD = AQ_CLEAR;           // Clear PWM1B on event B, down count

// Setup Trip functions
   EALLOW;
   EPwm3Regs.TZSEL.bit.OSHT1 = TZ_ENABLE;
   EPwm3Regs.TZCTL.bit.TZA = TZ_FORCE_LO;
   EPwm3Regs.TZCTL.bit.TZB = TZ_FORCE_LO;
   EPwm3Regs.TZFRC.bit.OST = 1;
   EDIS;

// Start the timer but the output is off
   EPwm3Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;
 
}

void InitEPwm4Example()
{
// Setup TBCLK
   EPwm4Regs.TBPRD = EPWM1_TIMER_TBPRD;           // Set timer period 801 TBCLKs
   EPwm4Regs.TBPHS.half.TBPHS = 0x0000;           // Phase is 0
   EPwm4Regs.TBCTR = 0x0000;                      // Clear counter
   
//   Set Compare values
   EPwm4Regs.CMPA.half.CMPA = EPWM1_TIMER_TBPRD- 1;     // Set compare A value
//   EPwm1Regs.CMPB = EPWM1_MAX_CMPB;               // Set Compare B value
   
// Setup counter mode
//   EPwm4Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up
   EPwm4Regs.TBCTL.bit.CTRMODE = TB_FREEZE; // Count up

   EPwm4Regs.TBCTL.bit.PHSEN = TB_ENABLE;        // Disable phase loading
   EPwm4Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
   EPwm4Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm4Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO;

// Setup shadowing
   EPwm4Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm4Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm4Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero
   EPwm4Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   


// Set actions
   EPwm4Regs.AQCTLA.bit.CAU = AQ_SET;             // Set PWM4A on event A, up count
   EPwm4Regs.AQCTLA.bit.CAD = AQ_CLEAR;           // Clear PWM4A on event A, down count
//   EPwm4Regs.AQCTLB.bit.ZRO = AQ_CLEAR;           // Keep PWM4B off 

	// Setup Trip functions
   EALLOW;
   EPwm4Regs.TZSEL.bit.OSHT1 = TZ_ENABLE;
   EPwm4Regs.TZCTL.bit.TZA = TZ_FORCE_LO;
   EPwm4Regs.TZCTL.bit.TZB = TZ_FORCE_LO;
   EPwm4Regs.TZFRC.bit.OST = 1;
   EDIS;

// Start the timer but the PWM output is off
   EPwm4Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;

 
}



void InitEPwm5Example()
{
// Setup TBCLK
   EPwm5Regs.TBPRD = EPWM1_TIMER_TBPRD/4;           // Set timer period 801 TBCLKs
//   EPwm5Regs.TBPRD = EPWM1_TIMER_TBPRD;           // Set timer period 801 TBCLKs
   EPwm5Regs.TBPHS.half.TBPHS = 0x0000;           // Phase is 0
   EPwm5Regs.TBCTR = 0x0000;                      // Clear counter
   
//   Set Compare values
   EPwm5Regs.CMPA.half.CMPA = EPWM1_TIMER_TBPRD/4-1;     // Set compare A value
//   EPwm5Regs.CMPA.half.CMPA = EPWM1_TIMER_TBPRD-1;     // Set compare A value
//   EPwm1Regs.CMPB = EPWM1_MAX_CMPB;               // Set Compare B value
   
// Setup counter mode
//   EPwm5Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up
   EPwm5Regs.TBCTL.bit.CTRMODE = TB_FREEZE; // Count up
   EPwm5Regs.TBCTL.bit.PHSEN =  TB_DISABLE;        // Disable phase loading
   EPwm5Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
   EPwm5Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm5Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO;

// Setup shadowing
   EPwm5Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm5Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm5Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero
   EPwm5Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   


// Setup Dead Band
   EPwm5Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE; //DB_DISABLE; // 
   EPwm5Regs.DBCTL.bit.IN_MODE = DBA_ALL;

//   EPwm5Regs.DBCTL.bit.IN_MODE = DBB_ALL;
   EPwm5Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
   EPwm5Regs.DBRED = 10;
   EPwm5Regs.DBFED = 10;

// Set actions
   EPwm5Regs.AQCTLA.bit.CAU = AQ_SET;             // Set PWM5A on event A, up count
   EPwm5Regs.AQCTLA.bit.CAD = AQ_CLEAR;           // Clear PWM5A on event A, down count
   EPwm5Regs.AQCTLB.bit.CAU = AQ_CLEAR;
   EPwm5Regs.AQCTLB.bit.CAD = AQ_SET;

// TZ setups
   EALLOW;
   EPwm5Regs.TZSEL.bit.OSHT2 = 1;
   EPwm5Regs.TZCTL.bit.TZA = 2;
   EPwm5Regs.TZCTL.bit.TZB = 2;
   EPwm5Regs.TZFRC.bit.OST = 1;
   EDIS; 

// Start the timer but the PWM output is off
   EPwm5Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;

 
}
//===========================================================================
// No more.
//===========================================================================


