#include <windows.h>
#include <winioctl.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <tchar.h>
#include <io.h>
#include <fstream>
#include <iostream>

#include "crc16.h"
#include "TunTap.h"

using namespace std;

extern ECA_APPEMU_MGH ECA_AppEmu_mgh;
extern int eMeterSecurity;
extern UINT32 firmwareVersion;
void ECA_AppEmu_BuildPayload(HCT_MSG_BUFFER_t *msg, UINT16 len, UINT16 *ppkthdr, UINT16 hdrlen);
int HostAppEMU_SendSyncMsgToPLC(UINT16 msgtype, UINT8 *msg, int msglen, BOOL b_UseRPY);
void HostAppEMU_SendMsgToPLC(UINT16 msgtype, UINT8 *msg, int msglen, BOOL b_UseRPY);

WCHAR * ErrorMsg[] = 
	{_T("Succesful"), 
	_T("TunTap Not Installed"), 
	_T("Unable to open TunTap driver"), 
	_T("Error setting the media status of the tuntap device"),
	_T("Error setting the network info for Tun Tap device"),
	_T("Bad TunTap IPv4 Address"),
	_T("Unable to create TunTap read thread"),
	_T("Unable to allocate memory when sending message") };

TunTap::TunTap(char * tunTapDriverName, UINT32 tunTapIPv4Address, UINT32 mode)
{
  m_Mode = mode;
	m_Status = 0;
	m_DeviceHandle = INVALID_HANDLE_VALUE;
	m_TunTapListener_Thread = NULL;
	m_TunTapListener_ThreadID = 0;
	m_PayloadId = 0;
	strcpy(m_TunTapDriverName, tunTapDriverName);

	memset(m_DeviceGuid, 0, 512);
	memset(m_HumanName, 0, 128);

	m_TunTapIPv4Address = tunTapIPv4Address;

	m_TunTapWriteEvent = CreateEvent(NULL, FALSE, FALSE, NULL);

	if (m_TunTapIPv4Address == 0)
	{
		//
		// Bad TunTap IPv4 Address
		//
		m_Status = 5;
	}

	if (m_Status == 0)
	{
		HKEY hKey = HKEY_LOCAL_MACHINE;

		FindDeviceGuid(hKey, _T("SYSTEM\\CurrentControlSet\\Control\\Class\\{4D36E972-E325-11CE-BFC1-08002BE10318}"));
		if (wcslen(m_DeviceGuid) == 0)
		{
			//
			// TunTap not installed
			//
			m_Status = 1;
		}
	}
	WCHAR userModeDeviceSpace[512] = {0};
	if (m_Status == 0)
	{
		FindHumanName(m_DeviceGuid); 

		wcscpy(userModeDeviceSpace,_T("\\\\.\\Global\\"));
		wcscat(userModeDeviceSpace,m_DeviceGuid);
		wcscat(userModeDeviceSpace,_T(".tap"));

		m_DeviceHandle = CreateFile(userModeDeviceSpace, 
			GENERIC_READ | GENERIC_WRITE,
			0, 
			NULL,
			OPEN_EXISTING,
			FILE_ATTRIBUTE_SYSTEM | FILE_FLAG_OVERLAPPED,
			0);

		if (m_DeviceHandle == INVALID_HANDLE_VALUE)
		{
			//
			// Unable to open the TunTap device..
			//
			m_Status = 2;
  		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Unable to open the TunTap Device.\n"));	
		}
	}

	int pStatus = 1;
	DWORD len = 0;
	BOOL status = false;

	if (m_Status == 0)
	{
		BOOL status = DeviceIoControl(m_DeviceHandle, TAP_CONTROL_CODE(6, METHOD_BUFFERED) /* TAP_IOCTL_SET_MEDIA_STATUS */, &pStatus, 4, &pStatus, 4, &len, 0);
		if (status == FALSE)
		{
			//
			// Error setting the media status of the tuntap device.
			//
			m_Status = 3;
  		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Error setting the media status of the tuntap device.\n"));	
		}
	}

	if (m_Status == 0)
	{
		unsigned long address = tunTapIPv4Address;
		unsigned long network = tunTapIPv4Address & 0x00FFFFFF;
		unsigned long mask = 0x00FFFFFF;
	 
		unsigned long ipData[] = {address, network, mask};

		status = DeviceIoControl(m_DeviceHandle, TAP_CONTROL_CODE(10, METHOD_BUFFERED) /* TAP_IOCTL_CONFIG_TUN */, ipData, sizeof(unsigned long)*3,
					ipData, sizeof(unsigned long)*3,&len, 0);

		if (status == FALSE)
		{
			//
			// Error setting the network info for Tun Tap
			//
  		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Error setting the network info for Tun Tap.\n"));	
			m_Status = 4;
		}
	}

	if (m_Status == 0)
	{

		m_TunTapListener_Thread = CreateThread(NULL, 0, TunTapListener, this, 0, &m_TunTapListener_ThreadID);
		if (!m_TunTapListener_ThreadID)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Tun Tap Listener CreateThread() failed\n"), __FUNCTION__);
			HostAppEMU_DbgPrintLastError();
			//
			// Unable to create thread;
			//
  		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Error unable to create read thread.\n"));	
			m_Status = 6;
		}
	}
}

TunTap::~TunTap(void)
{
	CloseHandle(m_DeviceHandle);
}


unsigned int TunTap::TAP_CONTROL_CODE(unsigned int request, unsigned int method)
{
  return CTL_CODE(FILE_DEVICE_UNKNOWN, request, method, FILE_ANY_ACCESS);
}

DWORD WINAPI TunTap::TunTapListener(LPVOID lpParm)
{
	TunTap * driver = (TunTap *) lpParm;
	
	OVERLAPPED overLap;
	overLap.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
	overLap.Internal = 0;
	overLap.InternalHigh = 0;
	overLap.Offset = 0;
	overLap.OffsetHigh = 0;
	overLap.Pointer = NULL;

	unsigned char dataBuffer[0xFFFF];
	DWORD bytesRead = 0;

	while (true)
	{
		while(true)
		{
			BOOL readStatus = ReadFile(driver->m_DeviceHandle, dataBuffer, 0xFFFF, &bytesRead, &overLap );
			if (readStatus == FALSE)
			{
				DWORD lastError = GetLastError();
				if (lastError == ERROR_IO_PENDING)
				{
					GetOverlappedResult(driver->m_DeviceHandle, & overLap, & bytesRead, TRUE);
					break;
				}
				else if (lastError == ERROR_HANDLE_EOF)
				{
					Sleep(10);
				}
				else
				{
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Error reading TunTap device, quitting.\n"));
					goto end;
				}
			}
		}
//		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Received msg from TunTap Driver.\n"));
		//
		// Send out a IPv6 Data message here..
		//
		UINT32 destinationIPv4Address;
		UINT32 sourceIPv4Address;

		memcpy(&sourceIPv4Address, &dataBuffer[12], sizeof(UINT32));
		memcpy(&destinationIPv4Address, &dataBuffer[16], sizeof(UINT32));
		
		UINT16 shortAddress = (UINT16) _rotr(sourceIPv4Address, 16) & 0x0000FFFF;
		//
		// If the short address is 0x1FE then this is the base node, set it's ipv6 short address to zero
		//
		if (shortAddress == 0xFE01)
		{
			shortAddress = 0;
		}
		UINT16 temp = _rotr(shortAddress, 8);
		temp |= _rotl(shortAddress, 8);

		IPv6Address sourceIPv6Address(ECA_AppEmu_mgh.G3_Pan_ID, temp);
			
		shortAddress = (UINT16) _rotr(destinationIPv4Address, 16) & 0x0000FFFF;
		//
		// If the destination short address is 0x1FE then this is the base node, set it's ipv6 short address to zero
		//
		if (shortAddress == 0xFE01)
		{
			shortAddress = 0;
		}

		temp = _rotr(shortAddress, 8);
		temp |= _rotl(shortAddress, 8);

		IPv6Address destinationIPv6Address(ECA_AppEmu_mgh.G3_Pan_ID, temp);

		if (driver->m_Mode == 0)
		{
			//
			// Send the message over the serial port..
			//
			driver->SendNodeMessage(IPv4Message, (UINT8 *)dataBuffer, (UINT16)bytesRead, destinationIPv6Address, sourceIPv6Address);
		}
		else
		{
			driver->SendNodeMessage(IPv4Message, (UINT8 *)dataBuffer, (UINT16)bytesRead, temp);
		}
 	}
end:
	return 0;
}
void TunTap::SendNodeMessage(UINT16 payloadType, UINT8 * data, UINT16 payloadLength, UINT16 shortAddress)
{
	//
	// micro IP only supports UPD
	// The ipv4 header is 20 bytes
	// The udp packet is 8 bytes
	// Offset the copy by the address placed into the data buffer.
	// source port = uint16
	// destination port = uint16
	// address type = uint16 = 0 = short address
	// address = uint16 
	///
	int messageLength = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + 2+2+2+2 /* short address header */+ payloadLength;
	UINT8 * pup_msg = (UINT8 *) LocalAlloc(LPTR, messageLength);

	if (pup_msg == NULL)
	{
		ECA_OutOfMemory();
		m_Status = 7;
	}
	else
	{
		m_Status = 0;
		//
		// offset the payload by the message transport 
		//
		int payloadOffset = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE; 
		UINT8 * payload = & pup_msg[payloadOffset];

		UINT16 flags = ECA_AppEmu_mgh.tx_nsdu_handle++;
		flags |= HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_FLAGS_DISCOVERY_ROUTE_ENABLED;
		
		if (eMeterSecurity == 1)
		{
			flags |= HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_FLAGS_SEC_ENABLED;
		}
	  
		*(UINT16 *) &payload[0] = 0; // port = 0; 
		*(UINT16 *) &payload[2] = 0; // port = 0; 
		*(UINT16 *) &payload[4] = 0x0000; // address type = 0 (but in network order
		*(UINT16 *) &payload[6] = shortAddress; // short address 
		//
		// Copy the data data packet
		//
		memcpy(&payload[2+2+2+2], data, payloadLength);
		//
		// save the message id to confirm later
		//
		ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_SystemTimeRead]++;
		ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;

		HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, messageLength, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
	}
}
void TunTap::SendNodeMessage(UINT16 payloadType, UINT8 * data, UINT16 payloadLength, IPv6Address destination, IPv6Address source)
{
	IPv6MainHeader ipWrapper;
	//
	// the plus 6 is for the payload type (UINT16) and payload id (UINT32) and the extended address (8 bytes)
	//
	ipWrapper.SetPayloadLength(payloadLength);
	ipWrapper.SetSourceAddress((UINT8*)source.m_IpAddress);
	ipWrapper.SetDestinationAddress((UINT8*)destination.m_IpAddress);

	m_PayloadId++;
	//
	// the plus 6 is for the payload type (UINT16) and payload id (UINT32)
	//
	int messageLength = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + IPv6MainHeader::IP6vHeaderSize + payloadLength;
	UINT8 * pup_msg = (UINT8 *) LocalAlloc(LPTR, messageLength);

	if (pup_msg == NULL)
	{
		ECA_OutOfMemory();
		m_Status = 7;
	}
	else
	{
		m_Status = 0;

		int payloadOffset = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + IPv6MainHeader::IP6vHeaderSize;
		UINT8 * payload = & pup_msg[payloadOffset];

		UINT16 flags = ECA_AppEmu_mgh.tx_nsdu_handle++;
		flags |= HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_FLAGS_DISCOVERY_ROUTE_ENABLED;
		
		if (eMeterSecurity == 1)
		{
		  flags |= HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_FLAGS_SEC_ENABLED;
		}

		*(UINT16 *) &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags] = flags;
		
		ipWrapper.Encode(pup_msg, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE);
		//
		// Copy the data IPv4 data packet
		//
		memcpy(payload, data, payloadLength);
		//
		// save the message id to confirm later
		//
		ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_SystemTimeRead]++;
		ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;

		HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, messageLength, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
	}
}


void TunTap::FindHumanName(WCHAR * guid)
{
	WCHAR *connectionKey = _T("SYSTEM\\CurrentControlSet\\Control\\Network\\{4D36E972-E325-11CE-BFC1-08002BE10318}");
	if (guid != NULL && wcslen(guid) > 0)
  {
		HKEY key = NULL;
		WCHAR newKey[512]= {0};
		wcscpy(newKey, connectionKey);
		wcscat(newKey, _T("\\"));
		wcscat(newKey, guid);
		wcscat(newKey, _T("\\Connection"));

		LONG status = RegOpenKeyEx(HKEY_LOCAL_MACHINE, newKey, 0L, KEY_READ, &key);
		if (status == 0)
		{
			WCHAR name[256] = {0};
			DWORD bufferSize = 256;
			status  = RegQueryValueEx(key, _T("Name"), NULL, NULL, (LPBYTE)name, & bufferSize);
			if (status == 0)
			{
				wcscpy(m_HumanName, name);
			}
		}
  }
}

void TunTap::FindDeviceGuid(HKEY hKey, WCHAR * startingKey)
{
	WCHAR componentName[512] = {0};
	WCHAR * netCfgInstanceId = new WCHAR[512];
  memset(netCfgInstanceId,0, 512);

	HKEY key = NULL;

	LONG status = RegOpenKeyEx(hKey, startingKey, 0L, KEY_READ, &key);

	WCHAR name[100];
	status = 0;
	DWORD index = 0;
	DWORD bufferSize = 100;
  WCHAR componentId[100];
	
	memset(m_DeviceGuid, 0, 512);

	while (wcslen(m_DeviceGuid) == 0)
	{
		HKEY subKey;
		status = RegEnumKey(key, index, name, bufferSize);
		WCHAR newName[512] = {0};
		wcscpy(newName, startingKey);
		wcscat(newName, _T("\\"));
		wcscat(newName, name);

		if (status == 0)
		{
			bufferSize = 100;
			status = RegOpenKeyEx(hKey, newName, 0L, KEY_READ, &subKey);
			if (status == 0)
			{
				bufferSize = 100;
				status  = RegQueryValueEx(subKey, _T("ComponentId"), NULL, NULL, (LPBYTE)componentId, & bufferSize);
			
				if (status == 0)
				{
					bufferSize = 100;
					//NetCfgInstanceId
					wcscpy(componentName,componentId);
					if (wcscmp(componentName, _T("tap0901")) == 0)
					{
						status  = RegQueryValueEx(subKey, _T("NetCfgInstanceId"), NULL,NULL, (LPBYTE)componentId, & bufferSize);
						wcscpy(m_DeviceGuid, componentId);
					}
				}
				RegCloseKey(subKey);
			}
		}
		index++;
	}
	RegCloseKey(key);
}

WCHAR * TunTap::LastError()
{
	return ErrorMsg[m_Status];
}

void TunTap::WriteToTunTap(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY)
{
	if (msg_len != 4)
	{
		if (m_Mode == 0)
		{
			UINT8 * message= (UINT8*)pmsg;
			//
			// Get the ip header and skip the status info, 2 bytes
			//
			IPv6MainHeader ipWrapper(&message[2]);
			//
			// get a pointer to the message..
			//
			UINT8 * payload = ipWrapper.GetData(&message[2]);
			
			UINT8 * ipv4Data = &payload[0];
			//
			// The length of the IPv4 data is the message length - the IPv6 header size, less the payload type, id, and extended address size + the ipv4 header size
			//
			int messageLength = msg_len-ipWrapper.IP6vHeaderSize-2 + 20;

			DWORD bytesWritten= 0;

			OVERLAPPED overLap;
			memset(&overLap, 0, sizeof(OVERLAPPED));

			overLap.hEvent = m_TunTapWriteEvent;
			overLap.Internal = 0;
			overLap.InternalHigh = 0;
			overLap.Offset = 0;
			overLap.OffsetHigh = 0;
			overLap.Pointer = NULL;

			BOOL writeStatus = WriteFile(m_DeviceHandle, ipv4Data, messageLength, &bytesWritten, &overLap);
			if (writeStatus == FALSE)
			{
				DWORD lastError = GetLastError();
				if (lastError == ERROR_IO_PENDING)
				{
					WaitForSingleObject(overLap.hEvent, -1);
				}
				else
				{
					DWORD lastError = GetLastError();
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Write failed to Tuntap %d.\n"), lastError);
				}
			}
		}
		else
		{
			UINT8 * message= (UINT8*)pmsg;
			//
			// The PLC always sends the IPv6 address back..
			//
			UINT8 * ipv4Data = &message[2+2+2+2+16];
			//
			// The length of the IPv4 data is the message length - the address data size 
			//
			int messageLength = msg_len-2-2-2-16;

			DWORD bytesWritten= 0;

			OVERLAPPED overLap;
			memset(&overLap, 0, sizeof(OVERLAPPED));

			overLap.hEvent = m_TunTapWriteEvent;
			overLap.Internal = 0;
			overLap.InternalHigh = 0;
			overLap.Offset = 0;
			overLap.OffsetHigh = 0;
			overLap.Pointer = NULL;

			BOOL writeStatus = WriteFile(m_DeviceHandle, ipv4Data, messageLength, &bytesWritten, &overLap);
			if (writeStatus == FALSE)
			{
				DWORD lastError = GetLastError();
				if (lastError == ERROR_IO_PENDING)
				{
					WaitForSingleObject(overLap.hEvent, -1);
				}
				else
				{
					DWORD lastError = GetLastError();
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Write failed to Tuntap %d.\n"), lastError);
				}
			}
		}
	}
}

UINT16 TunTap::CheckSum(unsigned char * data, int length)
{
	UINT16 crc = 0;
	UINT16 * crcData = (UINT16 *)data;
	int wordLength = (length + 1) >> 1;
	for (int index = 0; index < wordLength; index++)
	{
		crc += crcData[index];
	}
	crc = ~crc;
	return crc;
}
void TunTap::EchoPingRequest(HCT_MSG_BUFFER_t * pmsg, UINT16 msg_len, BOOL bRPY)
{
	if (msg_len != 4)
	{
		UINT8 * message= (UINT8*)pmsg;
		//
		// Get the ip header and skip the status info, 2 bytes
		//
		IPv6MainHeader pingPacket(&message[2]);
		//
		// get a pointer to the message..
		//
		UINT8 * payload = pingPacket.GetData(&message[2]);
		
		UINT16 payloadType = *((UINT16 *) &payload[0]);
		UINT32 payloadId   = *((UINT32 *) &payload[2]);
		UINT8 * ipv4Data = &payload[14];

		for (int i = 0; i < 4; ++i)
    {
      byte tmp = ipv4Data[12 + i]; 
			ipv4Data[12 + i] = ipv4Data[16 + i]; 
			ipv4Data[16 + i] = tmp;
    }
		ipv4Data[20] = 0; // type
		ipv4Data[21] = ipv4Data[21]; // code

		
		unsigned short crc16 =  CheckSum( &ipv4Data[20], 0x24 );

		ipv4Data[22] = crc16 & 0x00FF;
		ipv4Data[23] = (crc16 & 0xFF00) >> 8;
		//
		// Send the message over the serial port..
		//
		int payloadLength = pingPacket.GetPayloadLength() - 14;
		//
		// flip the addresses in the ipv6 header..
		//
		SendNodeMessage(IPv4Message, ipv4Data, (UINT16)payloadLength, pingPacket.m_SourceAddress, pingPacket.m_DestinationAddress);
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Sent Echoed ping to PLC.\n"));
	}
}
