/******************************************************************************
* FILE PURPOSE: HOSTAPPEMU main.
*******************************************************************************
*
* FILE NAME: HOSTAPPEMU.cpp
*
* DESCRIPTION:
*
* HISTORY:
* 09/01/2007    stephen smith           initial revision
* 12/08/2009    stephen smith           port from TIWiMAXWrapper DLL
*
* LIST OF FUNCTIONS:
*
* Copyright (c) 2007-2009 Texas Instruments Inc.  All rights reserved.
******************************************************************************/
#include "stdafx.h"
#include <crtdbg.h>
#include <Winsock2.h>   // for ntoh, hton
#include <tchar.h>
#include <psapi.h>
#include <shlwapi.h>
#include <strsafe.h>
#include <intrin.h>
#include <winioctl.h>

#include "hostappemu.h"
#include "os_fifo.h"
#include "hct_msg_def.h"
#include "hostappemu_com.h"
#include "hostappemu_ip.h"
#include "hostappemu_data.h"
#include "hostappemu_sn.h"
#include "hostappemu_p2p.h"
#include "ECA_APPEMU_MGH.h"

#include "tuntap.h"

static int HostAppEMU_Instance;
static TCHAR DebugBuffer[0x1000];

static int HostAppEMU_ProcessAttachedCounter = 0;
static int HostAppEMU_DbgLevel = HostAppEMU_DBG_LEVEL_VERBOSE;//HostAppEMU_DBG_LEVEL_DUMP_BUFFER;//HostAppEMU_DBG_LEVEL_COMM;//HostAppEMU_DBG_LEVEL_VERBOSE;
unsigned long HostAppEMU_Flags = HostAppEMU_OUTPUT_DEBUG_FILE;
static HANDLE HostAppEMU_Timer = INVALID_HANDLE_VALUE;
static HANDLE hDebugLog = INVALID_HANDLE_VALUE;

int host_port = HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_TYPE_SCI_B;
int diag_port = HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_TYPE_SCI_A;

int RPY_delay = 0;
int g3_test_pktSize = 26;
int flashTimeout = -1;
int eMeterRetryCount = 3;
int eMeterSecurity = 1;
BOOL eMeterEmulation = false;
UINT16 g3LongAddress[] = {0,0,0,0};
BOOL skipEmeterMessages = false;
BOOL g3DataOnlyTesting = false;
UINT32 emeterMessageDelay = 0;
UINT32 emeterMessageTimeout = 5;
UINT32 emeterBlackList[25] = {0};
UINT16 emeterBlackListCount = 0;
UINT32 emeterDiscoveryTimeout = 30;
UINT32 emeterPathDiscoveryInterval = 0;
bool   emeterSyncServiceNodes = false;
UINT8 g3MACSegmentLength = 239;
UINT32 g3ToneMaskSelection = 0;
UINT32 g3Modulation = 0;
UINT16 panId = 0x7755;

char tunTapDriverName[256] = {};
UINT32 tunTapIPv4Address = 0;
TunTap * pTunTapDriver = NULL;
int microIP = 0;

UINT32 TMR = 0;

extern int SetTunTapAddress(UINT16 shortAddress);

extern ECA_APPEMU_MGH ECA_AppEmu_mgh;

bool requestMacStatePib = false;

UINT32 registrationDelay = ECA_APPEMU_REGISTRATION_DELAY;
UINT32 connectionDelay = ECA_APPEMU_CONNECTION_DELAY;

char *HostAppEMU_SendFileName = (char *) NULL;
char *HostAppEMU_RecvFileName = (char *) NULL;
char *HostAppEMU_SendTextMessage = (char *) NULL;
char *HostAppEMU_FirmwareFilename = (char *) NULL;

TCHAR sModuleVersion[] = TEXT("3.2.5.1");

static HANDLE HostAppEMU_Mutex = INVALID_HANDLE_VALUE;
static HANDLE mDebugLog = INVALID_HANDLE_VALUE;

OS_FIFO HostAppEMU_EC_Task_MsgFifo = { };
OS_FIFO_CONTROL HostAppEMU_EC_Task_MsgFifoControl = { };
ECF_TASK_s HostAppEMU_EC_Task_MsgFifoBuffer[HostAppEMU_MSG_FIFO_COUNT] = { };

// local functions
BOOL HostAppEMU_Init();
void HostAppEMU_Uninit();
DWORD WINAPI HostAppEMU_Timer_ThreadProc(LPVOID lpParam);
DWORD WINAPI HostAppEMU_Init_ThreadProc(LPVOID lpParam);
void HostAppEMU_DbgPrintFile(LPCTSTR buf);
void dump_os_fifo_stats(void);
HANDLE HostAppEMU_MAC_HeartBeat_Thread = NULL;
DWORD WINAPI HostAppEMU_MAC_HeartBeat(LPVOID lpParam);
DWORD HostAppEMU_MAC_HeartBeat_ThreadID = 0;

extern ECA_APPEMU_MGH pECA_APPEMU_mgh;

void invalid_cmdline_param(void)
{
  puts("Invalid command line parameter, use -? for help.\n");
  exit(1);
}

/******************************************************************************
* FUNCTION NAME: DllMain
*
* DESCRIPTION:
*
* Return Value:      none
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
int main(int argc, char *argv[])
{
  extern ECF_STATUS ECA_AppEmu_ProcessTask(ECF_TASK_s *ptask);
	extern void TestRpyMessages(int messageSubtype);
  extern int defaultComPort;
  extern int defaultBaudRate;
  extern char *psrcaddr;
  extern char *pdstaddr;
  extern int udp_data_port;
  extern int udp_host_port;
  
  DWORD dwHandle;
  UINT bufLen;
  DWORD size = GetFileVersionInfoSize(_T("hostappemu.exe"), &dwHandle);
  LPTSTR * pData = new LPTSTR[size];

  GetFileVersionInfo(_T("hostappemu.exe"), NULL, size, pData);
  VS_FIXEDFILEINFO *pFileInfo;
  
  VerQueryValue(pData,_T("\\"), (LPVOID*)&pFileInfo, &bufLen);
  UINT majorVersion = HIWORD(pFileInfo->dwFileVersionMS);
  UINT minorVersion = LOWORD(pFileInfo->dwFileVersionMS);
  UINT buildNumber = HIWORD(pFileInfo->dwFileVersionLS);
  UINT revisionNumber = LOWORD(pFileInfo->dwFileVersionLS);

  UINT productMajorVersion = HIWORD(pFileInfo->dwProductVersionMS);
  UINT productMinorVersion = LOWORD(pFileInfo->dwProductVersionMS);
  UINT productBuildVersion = HIWORD(pFileInfo->dwProductVersionLS);
  UINT productRevisonVersion = LOWORD(pFileInfo->dwProductVersionLS);

  delete pData;

  printf("HostAppEmu: Version: %d.%d.%d.%d - Product Version: %d.%d.%d.%d\n", 
    majorVersion, minorVersion, buildNumber, revisionNumber, 
    productMajorVersion, productMinorVersion, productBuildVersion, productRevisonVersion);

	if ( SetPriorityClass(GetCurrentProcess(), REALTIME_PRIORITY_CLASS) == 0)
	{
		puts("Failed to set process to realtime priority\n");
	}

  ECF_TASK_s ecf_task;
  char *pEnd;

  while (--argc)
  {
    argv++;
    if (**argv == '-' || **argv == '/')
    {
      switch (*++*argv)
      {
      case '?':
        puts("usage: hostappemu [options]");
        puts("   options:");
        printf("      -p [#]     COM port (default %d)\n", defaultComPort);
        printf("      -b [#]     baud rate (default %d)\n", defaultBaudRate);
        printf("      -h [#]     host port (0: SCI-A, 1:SCI-B (default %d)\n", host_port);
		printf("      -v [#]     diag port (0: SCI-A, 1:SCI-B (default %d)\n", diag_port);
        puts("      -x         P2P mode");
        puts("      -m         MAC mode");
		puts("      -M [###]   G3 MAC Segment Length 1-239");
		puts("      -a [0,1]   0 = TMR off, 1 = TMR on");
//				puts("      -Z         G3 micro IP Mode.");
#ifdef APPEMU_IEC432
        puts("      -4         CL IEC432 mode");
#endif // #ifdef APPEMU_IEC432
        puts("      -l [auto]  CL LLC mode");
        puts("      -i <4,6>   IP mode");
        puts("      -s [addr]  IP source address");
        puts("      -d [addr]  IP destination address");
        printf("      -u [#]     destination data UDP port (default %d)\n", udp_data_port);
        printf("      -o [#]     device host UDP port (default %d)\n", udp_host_port);
		puts("      -H         Request MAC State PIB every 30 seconds.");
        puts("      -G [##]    Registraion Delay in seconds. Default is 10 seconds");
        puts("      -C [##]    Connection Delay in seconds. Default is 5 seconds");

        puts("      -g <n,c>   G3 mode <'n'ode, 'c'oncentrator>");
        puts("      -L [####:####:####:####] G3 Node long address to use (include leading zeros)");
		puts("      -T [TunTap_Driver_Name] Name of the TunTap Network Adapter");
		printf("      -e [#] eMeter Emulation; retry # times (default is %d)\n", eMeterRetryCount);
		puts("      -E eMeter Emulation; Disconnect and connect test only.");
        puts("      -B [####] eMeter Emulation; Do not send data to this short address.");
		puts("      -D eMeter Emulation; Data only testing no disconnects.");
        puts("      -k [##] eMeter Emulation; Data message timeout in seconds.");
        puts("      -y [##] eMeter Emulation; Concentrator message delay.");
        puts("      -I [##]   eMeter Emulation: Discovery timeout in seconds. Default is 30 seconds");
        puts("      -P [####] eMeter Emulation: Path Discovery interval timer length (seconds).");
        puts("      -j [####] Set the G3 PAN Id.");
        puts("      -q        eMeter Emulation: Sync Service Node testing.");
        puts("      -S <0,1>   If set to 0 the G3 data security is disabled.  The default is enabled.");
        puts("      -t [text]  send text message");
        puts("      -f [file]  transmit file");
        puts("      -r <#>     use RPY (optional delay # seconds before sending reply)");
        puts("      -R         reset device before starting");
		puts("      -F [filename] Flash the device.");
		puts("                    If the file name is specified the file is compared to the file sent by the PLC and not flashed");
        puts("      -w [#]    G3 Tonemask Selection. 0 = CENELEC A 36, 1 = CENELEC A 25, 2 = CENELEC B, 3 = CENDLEC BC");
        puts("                                       4 = CENELEC BCD, 5 = FCC Low Band, 6 = FCC High Band, 7 = FCC Full band");
        puts("      -n [#]    G3 Modulation. 0 = ROBO, 1 = BPSK, 2 = QPSK, 3 = 8PSK");
// comment out for now.				puts("      -M #		   test reply message to PCL (0-4, 15.");
        puts("");
        puts("If a IPv4 destination address is specified, the INET interface will be used,");
        puts("  otherwise the HCT interface over the serial port will be used. To use");
        puts("  a specific INET interface, specify the IPv4 source address.");
        return 1;
      case 'a':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        TMR = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
			case 'z':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        g3_test_pktSize = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'L':
        {
          if (argc == 1)
          {
            invalid_cmdline_param();
          }
          char * temp= *++argv;
          char tempLongAddress[30];
          memset(tempLongAddress, 0, sizeof(tempLongAddress));
          int count = 0;
          for(unsigned int index = 0, i=0; index < strlen(temp); index++)
          {
            if (temp[index] != ':')
            {
              tempLongAddress[i++] = temp[index];
            }
            else
            {
              count++;
              tempLongAddress[i++] = 0x00;
            }
          }
          if (count != 3)
          {
            puts("Invalid format for the long address. The valid format is ####:####:####:####\n");
            invalid_cmdline_param();
            break;
          }

          for (int index = 0, i = 0; index < 20; index+=5)
          {
            unsigned long digit = strtoul(&tempLongAddress[index], NULL, 16);
            g3LongAddress[i++] = _rotr16((UINT16)digit,8);
          }
          argc--;
        }
        break;
			
			case 'T':
				{
          if (argc == 1)
          {
            invalid_cmdline_param();
          }
          char * driverName= *++argv;
					strcpy(tunTapDriverName,driverName);
					argc--;
				}
				break;
      case 'R':
        HostAppEMU_Flags |= HostAppEMU_ResetDevice;
        break;
			case 'p':
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          defaultComPort = strtol(*++argv, &pEnd, 0);
        }
        else
        {
          invalid_cmdline_param();
        }
        argc--;
        break;
			case 'M':
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          g3MACSegmentLength = (UINT8)  strtol(*++argv, &pEnd, 0);
					ECA_AppEmu_mgh.G3_MAC_Segment_Length = g3MACSegmentLength;
        }
        argc--;
        break;
			case 'w':
				
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          g3ToneMaskSelection = (int)  strtol(*++argv, &pEnd, 0);
        }
        argc--;
        break;
			case 'n':
				
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          g3Modulation = (int)  strtol(*++argv, &pEnd, 0);
        }
        argc--;
        break;
      case 'b':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        defaultBaudRate = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'C':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        connectionDelay = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'G':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        registrationDelay = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'h':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        host_port = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'v':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        diag_port = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
			case 'Z':
				microIP = 1;
				break;
      case 'x':
        HostAppEMU_Flags |= HostAppEMU_P2P;
        break;
			case 'H':
				requestMacStatePib = true;
				break;
			case 'm':
        HostAppEMU_Flags |= HostAppEMU_MAC;
        break;
#ifdef APPEMU_IEC432
      case '4':
        HostAppEMU_Flags |= HostAppEMU_CL_IEC432;
        break;
#endif // #ifdef APPEMU_IEC432
      case 'l':
        HostAppEMU_Flags |= HostAppEMU_CL_LLC;
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          if (stricmp(*++argv, "auto") == 0)
          {
            HostAppEMU_Flags |= HostAppEMU_LLC_AUTO;
          }
          argc--;
        }

        break;
      case 'i':
        HostAppEMU_Flags |= HostAppEMU_Use_IP;
        HostAppEMU_Flags |= HostAppEMU_IPv4;
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          switch (strtol(*++argv, &pEnd, 0))
          {
          case 4:
            break;
          case 6:
            HostAppEMU_Flags &= ~HostAppEMU_IPv4;
            HostAppEMU_Flags |= HostAppEMU_IPv6;
            puts("IPv6 is not supported!\n");
            return 1;
          default:
            invalid_cmdline_param();
            return 1;
          }
          argc--;
        }
        break;
      case 's':
        if (argc == 1)
        {
          invalid_cmdline_param();
          return 1;
        }
        psrcaddr = *++argv;
        argc--;
        break;
      case 'd':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        pdstaddr = *++argv;
        argc--;
        break;
      case 'u':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        udp_data_port = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'B':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        emeterBlackList[emeterBlackListCount++] = strtol(*++argv, &pEnd, 0); 
        argc--;
        break;
      case 'P':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        emeterPathDiscoveryInterval = strtol(*++argv, &pEnd, 0); 
        argc--;
        break;
      case 'j':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        panId = (UINT16)strtol(*++argv, &pEnd, 0); 
        argc--;
        break;
      case 'I':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        emeterDiscoveryTimeout = strtol(*++argv, &pEnd, 0); 
        argc--;
        break;
      case 'y':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        emeterMessageDelay = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'k':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        emeterMessageTimeout = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'o':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        udp_host_port = strtol(*++argv, &pEnd, 0);
        argc--;
        break;
      case 'g':
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          *++argv;
          if (**argv == 'n')
          {
            HostAppEMU_Flags |= HostAppEMU_G3_Node;
          }
          else if (**argv == 'c')
          {
            HostAppEMU_Flags |= HostAppEMU_G3_Conc;
          }
          else
          {
            invalid_cmdline_param();
          }
          argc--;
        }
        else
        {
          invalid_cmdline_param();
        }
        break;
      case 'e':
				eMeterEmulation = true;
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          *++argv;
	        eMeterRetryCount = strtol(*++argv, &pEnd, 0);
	        argc--;
				}
        break;
      case 'S':
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          eMeterSecurity = strtol(*++argv, &pEnd, 0);
          argc--;
        }
        break;
      case 'E':
				skipEmeterMessages = true;
        break;
      case 'q':
        emeterSyncServiceNodes = true;
        break;
      case 'D':
        g3DataOnlyTesting = true;
        break;

      case 't':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        HostAppEMU_Flags |= HostAppEMU_SendTextMsg;
        HostAppEMU_SendTextMessage = *++argv;
        argc--;
        break;
      case 'f':
        if (argc == 1)
        {
          invalid_cmdline_param();
        }
        HostAppEMU_Flags |= HostAppEMU_SendFile;
        HostAppEMU_SendFileName = *++argv;
        argc--;
        break;
      case 'r':
        HostAppEMU_Flags |= HostAppEMU_USE_RPY;
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
          RPY_delay = strtol(*++argv, &pEnd, 0);
          argc--;
        }
        break;
			case 'F':
				HostAppEMU_Flags |= HostAppEMU_Flash;
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
	        HostAppEMU_FirmwareFilename = *++argv;
          argc--;
        }
        break;
/*
				case 'M':
        if (argc > 1 && *argv[1] != '-' && *argv[1] != '/')
        {
					TestRpyMessages(atoi(*++argv));
          argc--;
        }
        break;
*/
      default:
        invalid_cmdline_param();
        return 1;
      }
    }
    else
    {
      invalid_cmdline_param();
      return 1;
    }
  }

  if ( (host_port != HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_TYPE_SCI_A)
    && (host_port != HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_TYPE_SCI_B)
  )
  {
    puts("ERROR: Invalid host port specified, use -? for help.\n");
  }
	
  if ( (diag_port != HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_TYPE_SCI_A)
    && (diag_port != HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_TYPE_SCI_B)
  )
  {
    puts("ERROR: Invalid diag port specified, use -? for help.\n");
  }

  switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK) 
  {
  case HostAppEMU_P2P:
  case HostAppEMU_MAC:
#ifdef APPEMU_IEC432
  case HostAppEMU_CL_IEC432:
#endif // #ifdef APPEMU_IEC432
  case HostAppEMU_CL_LLC:
  case HostAppEMU_IPv4:
  case HostAppEMU_G3_Node:
  case HostAppEMU_G3_Conc:
    break;
  default:
    puts("ERROR: multiple modes specified.\n");
    return 1;
  }

  if ( (!(HostAppEMU_Flags & HostAppEMU_CL_LLC)) && (HostAppEMU_Flags & HostAppEMU_LLC_AUTO) )
  {
    puts("ERROR: Auto mode is only valid for LLC mode.\n");
    return 1;
  }

  if (HostAppEMU_Flags & HostAppEMU_IPv4)
  {
    if (!pdstaddr)
    {
      puts("ERROR: IPv4 mode specified but not destination address given.\n");
      return 1;
    }
  }

  if (HostAppEMU_Flags & HostAppEMU_SendTextMsg)
  {
    if (!(HostAppEMU_SendTextMessage))
    {
      puts("ERROR: Message not specified, use -? for help.\n");
      return 1;
    }
    if (!(HostAppEMU_Flags & HostAppEMU_P2P))
    {
      puts("ERROR: P2P mode is required for send text message operation.\n");
      return 1;
    }
  }

  if (HostAppEMU_Flags & HostAppEMU_SendFile)
  {
    if (!(HostAppEMU_SendFileName))
    {
      puts("ERROR: Send file not specified, use -? for help.\n");
      return 1;
    }
    if (!(HostAppEMU_Flags & HostAppEMU_P2P))
    {
      puts("ERROR: P2P mode is required for send file operation.\n");
      return 1;
    }
  }

  HostAppEMU_Init();

  if (HostAppEMU_Flags & HostAppEMU_ResetDevice)
  {
    extern void ECA_ResetDevice(void);

    puts("Resetting device...\n");
    ECA_ResetDevice();
  }

  if (!ECA_initialize())
  {
    puts("ERROR: Unable to initialize, aborting...\n");
    return 2;
  }

  if (HostAppEMU_Flags & HostAppEMU_SendTextMsg)
  {
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
      hostappemu_ipv4_SendTextMessage((UINT8 *) HostAppEMU_SendTextMessage, strlen(HostAppEMU_SendTextMessage));
    }
    else
    {
      hostappemu_p2p_SendTextMessage((UINT8 *) HostAppEMU_SendTextMessage, strlen(HostAppEMU_SendTextMessage));
    }
  }

  if (HostAppEMU_Flags & HostAppEMU_SendFile)
  {
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
//      hostappemu_ipv4_SendFile(HostAppEMU_SendFileName);
    }
    else
    {
      hostappemu_p2p_SendFile(HostAppEMU_SendFileName);
    }
  }

	if (HostAppEMU_Flags & HostAppEMU_Flash)
	{
		// xraf - anything to do here? Probably not just wait for the 
		// message from the PLC to start the flash..
	}
	//
	// Create the MAC PIB Heartbeat thread
	//
	if (requestMacStatePib)
	{
		HostAppEMU_MAC_HeartBeat_Thread = CreateThread(NULL, 0, HostAppEMU_MAC_HeartBeat, NULL, 0, &HostAppEMU_MAC_HeartBeat_ThreadID);
		if (!HostAppEMU_MAC_HeartBeat_ThreadID)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! MAC HeartBeat CreateThread() failed\n"), __FUNCTION__);
			HostAppEMU_DbgPrintLastError();
			return HostAppEMU_STATUS_FAILURE;
		}
	}
	//
	// Create the tun tap listener / repeater if a tun tap ipv4 address is given
	//
	if (strlen(tunTapDriverName) > 0 && ((HostAppEMU_Flags & HostAppEMU_G3_Conc) == HostAppEMU_G3_Conc))
	{
		tunTapIPv4Address = inet_addr("192.168.1.254");
		SetTunTapAddress(0x1FE);
	}

  for (;;)
  {
    unsigned long flags;

    HostAppEMU_GetFlags(&flags);
    if (flags & HostAppEMU_Exit)
    {
      break;
    }

//    dump_os_fifo_stats();

		os_fifo_get(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
    ECA_AppEmu_ProcessTask(&ecf_task);
  }

//  HostAppEMU_Uninit();
}

DWORD WINAPI HostAppEMU_MAC_HeartBeat(LPVOID lpParam)
{
	while (true)
	{
		Sleep(30000);
		int messageLength = 2; // only one MAC PIB will be requested.
		UINT16 pib = 0x24;     // MAC State is the only pib requested.

		UINT8 * pup_msg = (UINT8 *) LocalAlloc(LPTR, messageLength);
		memcpy(pup_msg, &pib, 2);

		HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_GET_MAC_PIB, (UINT8 *) pup_msg, 2, FALSE);
	}
	return 0;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Init
*
* DESCRIPTION:
*       Initialization routine..
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void dump_os_fifo_stats(void)
{
  OS_FIFO_STATS fifostats;

  os_fifo_stats(&HostAppEMU_EC_Task_MsgFifo, &fifostats);
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("OS_FIFO: curr_items:%d max_items:%d max_wait_put_item:%d max_wait_put_fifo:%d max_wait_get_fifo:%d\n"),
    fifostats.curr_items,
    fifostats.max_items,
    fifostats.max_wait_put_item,
    fifostats.max_wait_put_fifo,
    fifostats.max_wait_get_fifo
  );
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Init
*
* DESCRIPTION:
*       Initialization routine..
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
BOOL HostAppEMU_Init()
{
  HANDLE HostAppEMU_Init_Thread = 0;
  DWORD HostAppEMU_Init_ThreadID = 0;
  BOOL bResult = TRUE;

  mDebugLog = CreateMutex(NULL, FALSE, 0 /* TEXT("HOSTAPPEMU_Mutex_DebugLog") */);

  if (HostAppEMU_Flags & HostAppEMU_OUTPUT_DEBUG_FILE)
  {
    HostAppEMU_OpenDebugFile();
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ALWAYS, TEXT(">>> LOADED <<<\n"), __FUNCTION__);

  HostAppEMU_Mutex = CreateMutex(NULL, FALSE, 0 /* TEXT("HOSTAPPEMU_Mutex") */);
  if (!HostAppEMU_Mutex)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateMutex() failed\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
  }

  HostAppEMU_Init_ThreadProc(0);

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("%S: STARTED\n"), __FUNCTION__);

  return TRUE;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Uninit
*
* DESCRIPTION:
*       Uninitialization routine..
*
* Return Value:      none
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void HostAppEMU_Uninit()
{
  HostAppEMU_TimerKill();
  CloseHandle(HostAppEMU_Timer);

  HostAppEMU_UninitData();

  if (HostAppEMU_Flags & HostAppEMU_Use_IP)
  {
    HostAppEMU_Uninit_IP();
  }
  else
  {
    HostAppEMU_UninitCom();
  }

  CloseHandle(HostAppEMU_Mutex);

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ALWAYS, TEXT(">>> UNLOADED <<<\n"), __FUNCTION__);
  HostAppEMU_CloseDebugFile();
  CloseHandle(mDebugLog);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Init_ThreadProc
*
* DESCRIPTION:
*       Initialization thread procedure.
*
* Return Value:      DWORD       return value of window
* Input Parameters:  LPVOID      instance handle of window
* Output Parameters: none
******************************************************************************/
DWORD WINAPI HostAppEMU_Init_ThreadProc(LPVOID lpParam) 
{
  int status;

  status = os_fifo_create(
      &HostAppEMU_EC_Task_MsgFifoControl,
      0,
      sizeof(HostAppEMU_EC_Task_MsgFifoBuffer[0]),
      sizeof(HostAppEMU_EC_Task_MsgFifoBuffer) / sizeof(HostAppEMU_EC_Task_MsgFifoBuffer[0])
  );
  if (status != OS_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! os_fifo_create() failed, error %#x\n"), __FUNCTION__, status);
  }
  if (status == HostAppEMU_STATUS_SUCCESS)
  {
    status = os_fifo_open(&HostAppEMU_EC_Task_MsgFifo, &HostAppEMU_EC_Task_MsgFifoControl, &HostAppEMU_EC_Task_MsgFifoBuffer[0]);
    if (status != OS_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! os_fifo_open() failed, error %#x\n"), __FUNCTION__, status);
    }
  }

  HostAppEMU_Timer = CreateWaitableTimer(NULL, TRUE, NULL);
  if (!HostAppEMU_Timer)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateWaitableTimer() failed\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
  }

  if (HostAppEMU_Flags & HostAppEMU_Use_IP)
  {
    status = HostAppEMU_Init_IP();
    if (status != HostAppEMU_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! HostAppEMU_Init_IP() failed\n"), __FUNCTION__);
    }
  }
  else
  {
    status = HostAppEMU_InitCom();
    if (status != HostAppEMU_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! HostAppEMU_InitCom() failed\n"), __FUNCTION__);
    }
  }

  status = HostAppEMU_InitData();
  if (status != HostAppEMU_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! HostAppEMU_InitData() failed\n"), __FUNCTION__);
  }

  return 0;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_GetModuleVersion
*
* DESCRIPTION:
*       Retrieves our module version.
*
* Return Value:      TIWW_STATUS TIWW_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  int         debug level
* Output Parameters: none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_GetModuleVersion(LPTSTR sVersion, size_t bufsize)
{
  HRESULT result;

  result = StringCbCopy(sVersion, bufsize, (LPCTSTR) &sModuleVersion[0]);
  return SUCCEEDED(result) ? HostAppEMU_STATUS_SUCCESS : HostAppEMU_STATUS_FAILURE;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_SetDebugLevel
*
* DESCRIPTION:
*       Sets the current debug level.
*
* Return Value:      HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  int         debug level
* Output Parameters: none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_SetDebugLevel(int dbglevel)
{
  int result;

  if (dbglevel >= HostAppEMU_DBG_LEVEL_MIN && dbglevel <= HostAppEMU_DBG_LEVEL_MAX)
  {
    HostAppEMU_DbgLevel = dbglevel;
    result = HostAppEMU_STATUS_SUCCESS;
  }
  else
  {
    result = HostAppEMU_STATUS_FAILURE;
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_GetDebugLevel
*
* DESCRIPTION:
*       Gets the current debug level.
*
* Return Value:      HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  int *       pointer to receive debug level
* Output Parameters: none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_GetDebugLevel(int *pdbglevel)
{
  *pdbglevel = HostAppEMU_DbgLevel;
  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_DbgPrintFile
*
* DESCRIPTION:
*       Prints a string to the debug log file.
*
* Return Value:      none
* Input Parameters:  int         debug level
*                    LPCTSTR     string format
*                    ...         string format parameters
* Output Parameters: none
******************************************************************************/
void HostAppEMU_DbgPrintFile(LPCTSTR buf)
{
  HRESULT result;
  size_t len;
  DWORD writelen;

  if (hDebugLog != INVALID_HANDLE_VALUE)
  {
    result = StringCbLength((LPCTSTR) &DebugBuffer, sizeof(DebugBuffer), &len);
    if (SUCCEEDED(result))
    {
      char *s;
      
      s = (char *) LocalAlloc(0, len + sizeof(char));
      if (s)
      {
        // convert Unicode to ANSI:
        WideCharToMultiByte(CP_ACP, 0, (LPCWSTR) &DebugBuffer, -1, s, (int) len + 1, NULL, NULL);
        result = StringCbLengthA(s, len, &len);
        if (SUCCEEDED(result))
        {
          WriteFile(hDebugLog, s, (DWORD) len, &writelen, 0);
        }

        LocalFree(s);
      }
    }
  }
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_DbgPrint
*
* DESCRIPTION:
*       Prints a formatted message to the debugger.
*
* Return Value:      none
* Input Parameters:  int         debug level
*                    LPCTSTR     string format
*                    ...         string format parameters
* Output Parameters: none
******************************************************************************/
void HostAppEMU_DbgPrint(BOOL bPrintTime, int dbglevel, LPCTSTR lpFormat, ...)
{
  va_list arglist;
  DWORD dwExceptionCode = 0;
  SYSTEMTIME st;
  DWORD dw;
  HRESULT result;
  size_t len;

  if (lpFormat == (LPCTSTR) NULL)
  {
    return;
  }

  if (dbglevel <= HostAppEMU_DbgLevel)
  {
    dw = WaitForSingleObject(mDebugLog, INFINITE);

    __try
    {
      if (bPrintTime)
      {
        GetLocalTime(&st);
        StringCbPrintf((LPTSTR) &DebugBuffer, sizeof(DebugBuffer), TEXT("[%02d:%02d:%02d.%03d]:"),
            st.wHour, st.wMinute, st.wSecond, st.wMilliseconds
        );
      }
      else
      {
        StringCbCopy((LPTSTR) &DebugBuffer, sizeof(DebugBuffer), TEXT(""));
      }
      result = StringCbLength((LPCTSTR) &DebugBuffer, sizeof(DebugBuffer), &len);
      if (SUCCEEDED(result))
      {
        va_start(arglist, lpFormat);

        StringCbVPrintf((LPTSTR) &DebugBuffer[len / sizeof(TCHAR)], sizeof(DebugBuffer) - len, lpFormat, arglist);
//        OutputDebugString((LPCTSTR) &DebugBuffer);
        wprintf((LPCTSTR) &DebugBuffer);
        fflush(stdout);
        HostAppEMU_DbgPrintFile((LPCTSTR) &DebugBuffer);

        va_end(arglist);
      }
    }
    __except(dwExceptionCode = GetExceptionCode(), EXCEPTION_EXECUTE_HANDLER)
    {
      wprintf((LPCTSTR) TEXT("DbgPrint Exception: Internal error in DbgPrint routine!"));
      fflush(stdout);
    }

    ReleaseMutex(mDebugLog);
  }
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_DbgDumpBuffer
*
* DESCRIPTION:
*       Dumps a given buffer.
*
* Return Value:      none
* Input Parameters:  int         debug level
*                    char *      buffer
*                    int         buffer length
* Output Parameters: none
******************************************************************************/
void HostAppEMU_DbgDumpBuffer(int dbglvl, char *pbuffer, int length)
{
  int i, offset;
  TCHAR s[83 * sizeof(TCHAR)], *cp;

  if (dbglvl > HostAppEMU_DbgLevel)
  {
    return;
  }

  HostAppEMU_DbgPrint(TRUE, dbglvl, TEXT("%p [%#x %d]\n"), pbuffer, length, length);

  for (offset = 0; offset < length; offset += 16)
  {
    StringCbPrintf(s, sizeof(s), TEXT("\t%06x  "), offset);
    cp = s + wcslen(s);
    for (i = 0; i < 16; i++)
    {
      if (offset + i < length)
      {
        StringCbPrintf(cp, cp - s, TEXT("%02x%s"), (*(pbuffer + offset + i) & 0xff), i == 7 ? TEXT("  ") : TEXT(" "));
      }
      else
      {
        StringCbPrintf(cp, cp - s, TEXT("  %s"), i == 7 ? TEXT("  ") : TEXT(" "));
      }
      cp += wcslen(cp);
    }
    for (i = 0; i < 16; i++)
    {
      if (offset + i < length)
      {
        unsigned char c;

        c = *(pbuffer + offset + i);
        StringCbPrintf(cp, cp - s, TEXT("%c"), c >= 0x20 && c <= 0x7f ? c : '.');
      }
      else
      {
        StringCbPrintf(cp, cp - s, TEXT("%c"), ' ');
      }
      cp += wcslen(cp);
    }

    HostAppEMU_DbgPrint(FALSE, dbglvl, TEXT("%s\n"), s);
  }
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_DbgPrintLastError
*
* DESCRIPTION:
*       Prints the last system error to the debugger.
*
* Return Value:      none
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void HostAppEMU_DbgPrintLastError(void)
{
  DWORD dw;
  LPTSTR buf;

  dw = FormatMessage(
      FORMAT_MESSAGE_ALLOCATE_BUFFER|FORMAT_MESSAGE_FROM_SYSTEM, 
      0, 
      GetLastError(), 
      0,
      (LPWSTR) &buf, 
      0,
      0
  );
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, buf);
  LocalFree(buf);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_GetInstance
*
* DESCRIPTION:
*       Gets the current DLL load instance count.  This is useful to know
*       if the DLL has already been running and another process has loaded it.
*
* Return Value:      HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  int *       pointer to receive instance
* Output Parameters: int *       instance
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_GetInstance(int *pinstance)
{
  *pinstance = HostAppEMU_Instance;
  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Lock
*
* DESCRIPTION:
*       Locks the global data protection mutex.
*
* Return Value:      HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_Lock(void)
{
  DWORD dw;

  dw = WaitForSingleObject(HostAppEMU_Mutex, INFINITE);
  if (dw != 0)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! WaitForSingleObject failed.\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
  }

  return (dw == 0 ? HostAppEMU_STATUS_SUCCESS : HostAppEMU_STATUS_FAILURE);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Unlock
*
* DESCRIPTION:
*       Unlocks the global data protection mutex.
*
* Return Value:      HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_Unlock(void)
{
  BOOL result;
  
  result = ReleaseMutex(HostAppEMU_Mutex);
  if (!result)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! ReleaseMutex failed.\n"));
    HostAppEMU_DbgPrintLastError();
  }

  return (result ? HostAppEMU_STATUS_SUCCESS : HostAppEMU_STATUS_FAILURE);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Timer_ThreadProc
*
* DESCRIPTION:
*       Timer routine thread.  This thread starts a timer, waits for it to expire
*       and calls the callback from the PHostAppEMU_TIMER_CTRL parameter.
*
* Return Value:      DWORD       thread return value.
* Input Parameters:  LPVOID      PHostAppEMU_TIMER_CTRL control structure.
* Output Parameters: none
******************************************************************************/
DWORD WINAPI HostAppEMU_Timer_ThreadProc(LPVOID lpParam) 
{
  PHostAppEMU_TIMER_CTRL pTimerCtrl = (PHostAppEMU_TIMER_CTRL) lpParam;
  LARGE_INTEGER liDueTime;
  int result;

  for (result = 0; result == 0; )
  {
    liDueTime.QuadPart = -(pTimerCtrl->ms) * 10000;

    if (!SetWaitableTimer(HostAppEMU_Timer, &liDueTime, 0, NULL, NULL, 0))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! SetWaitableTimer() failed\n"), __FUNCTION__);
      HostAppEMU_DbgPrintLastError();
      result = HostAppEMU_STATUS_FAILURE;
      break;
    }

    if (WaitForSingleObject(HostAppEMU_Timer, INFINITE) != WAIT_OBJECT_0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! WaitForSingleObject() failed\n"), __FUNCTION__);
      HostAppEMU_DbgPrintLastError();
      result = HostAppEMU_STATUS_FAILURE;
      break;
    }

    if (pTimerCtrl->TimerCallback)
    {
      result = (*pTimerCtrl->TimerCallback)(pTimerCtrl->ms);
    }

    if (pTimerCtrl->oneshot)
    {
      break;
    }
  }

  LocalFree(pTimerCtrl);
  return (DWORD) result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_TimerStart
*
* DESCRIPTION:
*       Routine that starts the HostAppEMU_Timer.
*
* Return Value:      HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  HostAppEMU_TIMER_CALLBACK routine to call when the timer expires.
*                    int         timer milliseconds
*                    BOOL        TRUE if one shot timer
* Output Parameters: none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_TimerStart(HostAppEMU_TIMER_CALLBACK TimerCallback, int ms, BOOL oneshot)
{
  HANDLE HostAppEMU_Timer_Thread;
  DWORD HostAppEMU_Timer_ThreadID;
  PHostAppEMU_TIMER_CTRL pTimerCtrl;

  if (HostAppEMU_TimerKill())
  {
    Sleep(50);
  }

  pTimerCtrl = (PHostAppEMU_TIMER_CTRL) LocalAlloc(0, sizeof(*pTimerCtrl));
  if (!pTimerCtrl)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! LocalAlloc(%d) failed.\n"), sizeof(*pTimerCtrl));
    HostAppEMU_DbgPrintLastError();
    return HostAppEMU_STATUS_NO_MEMORY;
  }
  pTimerCtrl->TimerCallback = TimerCallback;
  pTimerCtrl->ms = ms;
  pTimerCtrl->oneshot = oneshot;

  HostAppEMU_Timer_Thread = CreateThread(NULL, 0, HostAppEMU_Timer_ThreadProc, pTimerCtrl, 0, &HostAppEMU_Timer_ThreadID);
  if (!HostAppEMU_Timer_Thread)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateThread() failed\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
    LocalFree(pTimerCtrl);
    return HostAppEMU_STATUS_FAILURE;
  }

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_TimerKill
*
* DESCRIPTION:
*       Routine that kills the HostAppEMU_Timer.
*
* Return Value:      HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_TimerKill(void)
{
  BOOL b;

  b = CancelWaitableTimer(HostAppEMU_Timer);
  return (b ? HostAppEMU_STATUS_SUCCESS : HostAppEMU_STATUS_FAILURE);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_TempName
*
* DESCRIPTION:
*       Generates a temporary name.
*
* Return Value:       none
* Input Parameters:   LPTSTR  buffer to place tempname into
*                     size_t  size of buffer
* Output Parameters:  none
******************************************************************************/
void HostAppEMU_TempName(LPTSTR buf, size_t bufsize)
{
  static DWORD dw = 0;

  if (dw == 0)
  {
    srand(GetTickCount());
  }
  dw = ((rand() << 16) | rand());
  StringCbPrintf(buf, bufsize, TEXT("%x"), dw);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_OpenDebugFile
*
* DESCRIPTION:
*       Opens the debug log file.
*
* Return Value:       HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:   LPTSTR  buffer to place tempname into
*                     size_t  size of buffer
* Output Parameters:  none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_OpenDebugFile(void)
{
  if (hDebugLog == INVALID_HANDLE_VALUE)
  {
//    CreateDirectory(TEXT("c:\\windows\\system32\\logfiles"), NULL);

    hDebugLog = CreateFile(
          TEXT("HOSTAPPEMU.log"), //"c:\\windows\\system32\\logfiles\\HOSTAPPEMU.log"),
          GENERIC_READ|GENERIC_WRITE,
          FILE_SHARE_READ|FILE_SHARE_WRITE,
          NULL,
          OPEN_ALWAYS,
          0,
          NULL
    );

    if (hDebugLog != INVALID_HANDLE_VALUE)
    {
      TCHAR sver[40];
      SYSTEMTIME systime;

      HostAppEMU_GetModuleVersion((LPTSTR) &sver, sizeof(sver));

      HostAppEMU_Flags |= HostAppEMU_OUTPUT_DEBUG_FILE;
      SetFilePointer(hDebugLog, 0, NULL, FILE_END);
      GetLocalTime(&systime);
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ALWAYS, TEXT("Log Opened %02u/%02u/%04u %02u:%02u:%02u (module version %s)\n"), 
          systime.wMonth, systime.wDay, systime.wYear,
          systime.wHour, systime.wMinute, systime.wSecond,
          sver
      );
    }
  }

  return (hDebugLog != INVALID_HANDLE_VALUE ? HostAppEMU_STATUS_SUCCESS : HostAppEMU_STATUS_FAILURE);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_CloseDebugFile
*
* DESCRIPTION:
*       Closes the debug log file.
*
* Return Value:       HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:   LPTSTR  buffer to place tempname into
*                     size_t  size of buffer
* Output Parameters:  none
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_CloseDebugFile(void)
{
  SYSTEMTIME systime;

  if (hDebugLog != INVALID_HANDLE_VALUE)
  {
    SetFilePointer(hDebugLog, 0, NULL, FILE_END);
    GetLocalTime(&systime);
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ALWAYS, TEXT("Log Closed %02u/%02u/%04u %02u:%02u:%02u\n\n"), 
        systime.wMonth, systime.wDay, systime.wYear,
        systime.wHour, systime.wMinute, systime.wSecond, systime.wMilliseconds
    );
    CloseHandle(hDebugLog);
    hDebugLog = INVALID_HANDLE_VALUE;
  }

  HostAppEMU_Flags &= ~HostAppEMU_OUTPUT_DEBUG_FILE;

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_GetFlags
*
* DESCRIPTION:
*       Returns the flags value.
*
* Return Value:       HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:   unsigned long * pointer to store flags in
* Output Parameters:  unsigned long * filled with value fo flags
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_GetFlags(unsigned long *pflags)
{
  HostAppEMU_Lock();
  *pflags = HostAppEMU_Flags;
  HostAppEMU_Unlock();

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_SetFlags
*
* DESCRIPTION:
*       Sets the flags value.
*
* Return Value:       HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:   unsigned long * pointer to store flags in
* Output Parameters:  unsigned long * filled with value fo flags
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_SetFlags(unsigned long flags)
{
  HostAppEMU_Lock();
  HostAppEMU_Flags |= flags;
  HostAppEMU_Unlock();

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ClearFlags
*
* DESCRIPTION:
*       Clears the flags value.
*
* Return Value:       HostAppEMU_STATUS HostAppEMU_STATUS_SUCCESS, or errorlevel.
* Input Parameters:   unsigned long * pointer to store flags in
* Output Parameters:  unsigned long * filled with value fo flags
******************************************************************************/
HostAppEMU_STATUS HostAppEMU_ClearFlags(unsigned long flags)
{
  HostAppEMU_Lock();
  HostAppEMU_Flags &= ~flags;
  HostAppEMU_Unlock();

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_exitapp
*
* DESCRIPTION:
*       Exit application routine.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_exitapp(void)
{
  ECF_TASK_s ecf_task;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Exiting...\n"));

  ecf_task.task_type = ECA_TASK_EXIT;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}
