/******************************************************************************
* FILE PURPOSE: AppEMUC com port module.
*******************************************************************************
*
* FILE NAME: HostAppEMU_com.cpp
*
* DESCRIPTION:
*       Com port module.  We hook up to the serial port, where the WiMAX
*       hardware is, and read the messages as they are sent.
*
* HISTORY:
* 09/01/2007    stephen smith           initial revision
* 12/08/2009    stephen smith           port from TIWiMAXWrapper DLL
*
* Copyright (c) 2007-2009 Texas Instruments Inc.  All rights reserved.
******************************************************************************/
#include "stdafx.h"
#include <Winsock2.h>   // for ntoh, hton
#include <strsafe.h>

#include "hostappemu.h"
#include "hct_msg_def.h"
#include "hostappemu_com.h"
#include "hostappemu_data.h"
#include "crc16.h"

HostAppEMU_COM_CONFIG ComConfig = { };

int defaultComPort = 1;
int defaultBaudRate = 57600;

HANDLE HostAppEMU_Com = 0;
HANDLE HostAppEMU_ComReadThread = 0;
DWORD HostAppEMU_ComReadThreadID = 0;

DWORD WINAPI HostAppEMU_ComReadThreadProc(LPVOID lpParam);
int HostAppEMU_ComRead(HostAppEMU_COM_CONFIG *pComConfig, void *buffer, int count) ;

/******************************************************************************
* FUNCTION NAME: HostAppEMU_InitCom
*
* DESCRIPTION:
*       Initialization routine for comport module.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_InitCom(void)
{
  int result;

  HostAppEMU_ComGetSettings(&ComConfig);

  result = HostAppEMU_ComPortOpen(&ComConfig);
  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_UninitCom
*
* DESCRIPTION:
*       Uninitialization routine for comport module.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_UninitCom(void)
{
  int result;

  result = HostAppEMU_ComPortClose(&ComConfig);
  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComPortOpen
*
* DESCRIPTION:
*       Open the com port for use.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   HostAppEMU_COM_CONFIG * com port control
* Output Parameters:  HostAppEMU_COM_CONFIG * updated
******************************************************************************/
int HostAppEMU_ComPortOpen(HostAppEMU_COM_CONFIG *pComConfig)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_ComPortClose(pComConfig);

  HostAppEMU_Lock();

  StringCbPrintf(pComConfig->sComName, sizeof(pComConfig->sComName), TEXT("\\\\.\\COM%d"), pComConfig->iComPort);
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Opening %s.\n"), __FUNCTION__, pComConfig->sComName);

  pComConfig->hCom = CreateFile(
      pComConfig->sComName,
      GENERIC_READ|GENERIC_WRITE,
      0,
      NULL,
      OPEN_EXISTING,
      0,
      NULL
  );
  if (pComConfig->hCom == INVALID_HANDLE_VALUE)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateFile(%s) failed.\n"), __FUNCTION__, pComConfig->sComName);
    HostAppEMU_DbgPrintLastError();
    result = HostAppEMU_STATUS_FAILURE;
  }

  if (result == HostAppEMU_STATUS_SUCCESS)
  {
    HostAppEMU_ComReadThread = CreateThread(NULL, 0, HostAppEMU_ComReadThreadProc, &ComConfig, 0, &HostAppEMU_ComReadThreadID);
    if (!HostAppEMU_ComReadThread)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateThread() failed\n"), __FUNCTION__);
      HostAppEMU_DbgPrintLastError();
      result = HostAppEMU_STATUS_FAILURE;
    }
  }

  HostAppEMU_Unlock();

  if (result == HostAppEMU_STATUS_SUCCESS)
  {
    result = HostAppEMU_ComApplySettings();
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComPortClose
*
* DESCRIPTION:
*       Closes the com port.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   HostAppEMU_COM_CONFIG * com port control
* Output Parameters:  HostAppEMU_COM_CONFIG * updated
******************************************************************************/
int HostAppEMU_ComPortClose(HostAppEMU_COM_CONFIG *pComConfig)
{
  HostAppEMU_Lock();

  if (pComConfig->hCom != INVALID_HANDLE_VALUE)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Closing %s.\n"), __FUNCTION__, pComConfig->sComName);
    CloseHandle(pComConfig->hCom);
    pComConfig->hCom = INVALID_HANDLE_VALUE;
  }

  HostAppEMU_Unlock();

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComGetSettings
*
* DESCRIPTION:
*       Retrieves the comport configuration.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   HostAppEMU_COM_CONFIG * com port control
* Output Parameters:  HostAppEMU_COM_CONFIG * updated
******************************************************************************/
int HostAppEMU_ComGetSettings(HostAppEMU_COM_CONFIG *pComConfig)
{
#if 0
  HKEY hRegKey;
  DWORD regType;
  DWORD regLen;
#endif
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();

  // initialize to defaults
  ComConfig.iComPort = defaultComPort;
  memset(&pComConfig->dcb, 0, sizeof(pComConfig->dcb));
  pComConfig->dcb.DCBlength           = sizeof(pComConfig->dcb);
//  pComConfig->dcb.BaudRate            = CBR_57600;
  pComConfig->dcb.BaudRate            = defaultBaudRate;    // LOL!  baud rates _FINALLY_ are the real rates, not the old funky defines!
  pComConfig->dcb.fBinary             = TRUE;
  pComConfig->dcb.fParity             = FALSE;
  pComConfig->dcb.fOutxCtsFlow        = FALSE;
  pComConfig->dcb.fOutxDsrFlow        = FALSE;
  pComConfig->dcb.fDtrControl         = DTR_CONTROL_DISABLE;
  pComConfig->dcb.fDsrSensitivity     = FALSE;
  pComConfig->dcb.fRtsControl         = RTS_CONTROL_DISABLE;
  pComConfig->dcb.fAbortOnError       = FALSE;
  pComConfig->dcb.ByteSize            = 8;
  pComConfig->dcb.Parity              = NOPARITY;
  pComConfig->dcb.StopBits            = ONESTOPBIT;

//  memset(&pComConfig->CommTimeouts, 0, sizeof(pComConfig->CommTimeouts));
  // http://msdn.microsoft.com/en-us/library/aa363190
  pComConfig->CommTimeouts.ReadIntervalTimeout = MAXDWORD;
  pComConfig->CommTimeouts.ReadTotalTimeoutMultiplier = MAXDWORD;
  pComConfig->CommTimeouts.ReadTotalTimeoutConstant = 1;
  pComConfig->CommTimeouts.ReadTotalTimeoutConstant = 100; //xraf 100 - serial timeout
  pComConfig->CommTimeouts.WriteTotalTimeoutConstant = 100; //xraf 100;

#if 0
  if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, HostAppEMU_REGISTRY_KEY, 0, KEY_ALL_ACCESS, &hRegKey) == ERROR_SUCCESS)
  {
    regLen = sizeof(pComConfig->iComPort);
    RegQueryValueEx(hRegKey, TEXT("ComPort"), 0, &regType, (LPBYTE) &pComConfig->iComPort, &regLen);
    regLen = sizeof(pComConfig->dcb.BaudRate);
    RegQueryValueEx(hRegKey, TEXT("BaudRate"), 0, &regType, (LPBYTE) &pComConfig->dcb.BaudRate, &regLen);
    regLen = sizeof(pComConfig->dcb.ByteSize);
    RegQueryValueEx(hRegKey, TEXT("DataBits"), 0, &regType, (LPBYTE) &pComConfig->dcb.ByteSize, &regLen);
    regLen = sizeof(pComConfig->dcb.Parity);
    RegQueryValueEx(hRegKey, TEXT("Parity"), 0, &regType, (LPBYTE) &pComConfig->dcb.Parity, &regLen);
    regLen = sizeof(pComConfig->dcb.StopBits);
    RegQueryValueEx(hRegKey, TEXT("StopBits"), 0, &regType, (LPBYTE) &pComConfig->dcb.StopBits, &regLen);
    // TODO: need flow control settings here

    RegCloseKey(hRegKey);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: No com port settings found, using defaults.\n"), __FUNCTION__);
  }
#endif

  HostAppEMU_Unlock();

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComPutSettings
*
* DESCRIPTION:
*       Stores the comport configuration.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   HostAppEMU_COM_CONFIG * com port control
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComPutSettings(HostAppEMU_COM_CONFIG *pComConfig)
{
#if 0
  HKEY hRegKey;
#endif
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
#if 0
  if (RegCreateKeyEx(HKEY_LOCAL_MACHINE, HostAppEMU_REGISTRY_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hRegKey, NULL) == ERROR_SUCCESS)
  {
    RegSetValueEx(hRegKey, TEXT("ComPort"), 0, REG_BINARY, (const BYTE *) &pComConfig->iComPort, sizeof(pComConfig->iComPort));
    RegSetValueEx(hRegKey, TEXT("BaudRate"), 0, REG_BINARY, (const BYTE *) &pComConfig->dcb.BaudRate, sizeof(pComConfig->dcb.BaudRate));
    RegSetValueEx(hRegKey, TEXT("DataBits"), 0, REG_BINARY, (const BYTE *) &pComConfig->dcb.ByteSize, sizeof(pComConfig->dcb.ByteSize));
    RegSetValueEx(hRegKey, TEXT("Parity"), 0, REG_BINARY, (const BYTE *) &pComConfig->dcb.Parity, sizeof(pComConfig->dcb.Parity));
    RegSetValueEx(hRegKey, TEXT("StopBits"), 0, REG_BINARY, (const BYTE *) &pComConfig->dcb.StopBits, sizeof(pComConfig->dcb.StopBits));
    // TODO: need flow control settings here

    RegCloseKey(hRegKey);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! RegCreateKeyEx() failed.\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
    result = HostAppEMU_STATUS_FAILURE;
  }
#endif
  HostAppEMU_Unlock();
  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComApplySettings
*
* DESCRIPTION:
*       API to apply the current com port configuration.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComApplySettings(void)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  if (!SetCommState(ComConfig.hCom, &ComConfig.dcb))
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! SetCommState() failed.\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
    result = HostAppEMU_STATUS_FAILURE;
  }
  if (!SetCommTimeouts(ComConfig.hCom, &ComConfig.CommTimeouts))
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! SetCommTimeouts() failed.\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
    result = HostAppEMU_STATUS_FAILURE;
  }
  HostAppEMU_Unlock();
  if (result == HostAppEMU_STATUS_SUCCESS)
  {
    result = HostAppEMU_ComPutSettings(&ComConfig);
    if (result != HostAppEMU_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! HostAppEMU_ComPutSettings() failed.\n"), __FUNCTION__);
      HostAppEMU_DbgPrintLastError();
    }
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComSetPort
*
* DESCRIPTION:
*       API to set the desired com port, i.e., COM1, COM2, etc.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int        com port number to set [1, 2, ...]
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComSetPort(int iComport)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  if (ComConfig.iComPort == iComport)
  {
    HostAppEMU_Unlock();
  }
  else
  {
    ComConfig.iComPort = iComport;
    HostAppEMU_Unlock();

    HostAppEMU_ComPortClose(&ComConfig);
    result = HostAppEMU_ComPortOpen(&ComConfig);
    if (result == HostAppEMU_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: port set to %d.\n"), __FUNCTION__, iComport);
    }
    else
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Error setting port to %d.\n"), __FUNCTION__, iComport);
    }
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComGetPort
*
* DESCRIPTION:
*       API to get the current com port, i.e., COM1, COM2, etc.
*
* Return Value:       int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int *       com port number
* Output Parameters:  int *       com port number
******************************************************************************/
int HostAppEMU_ComGetPort(int *piComport)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  *piComport = ComConfig.iComPort;
  HostAppEMU_Unlock();

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComSetBaudRate
*
* DESCRIPTION:
*       API to set the com port baud rate value.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int        baud rate
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComSetBaudRate(int bps)
{
  DWORD val;
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  switch (bps)
  {
  case 110: val = CBR_110; break;
  case 300: val = CBR_300; break;
  case 600: val = CBR_600; break;
  case 1200: val = CBR_1200; break;
  case 2400: val = CBR_2400; break;
  case 4800: val = CBR_4800; break;
  case 9600: val = CBR_9600; break;
  case 14400: val = CBR_14400; break;
  case 19200: val = CBR_19200; break;
  case 38400: val = CBR_38400; break;
  case 57600: val = CBR_57600; break;
  case 115200: val = CBR_115200; break;
  default:
    result = HostAppEMU_STATUS_FAILURE;
    break;
  }
  HostAppEMU_Unlock();

  if (result == HostAppEMU_STATUS_SUCCESS)
  {
    if (ComConfig.dcb.BaudRate != val)
    {
      ComConfig.dcb.BaudRate = val;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: bps set to %d.\n"), __FUNCTION__, bps);
      HostAppEMU_ComPutSettings(&ComConfig);
    }
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Invalid bps %d.\n"), __FUNCTION__, bps);
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComGetBaudRate
*
* DESCRIPTION:
*       API to get the com port baud rate value.
*
* Return Value:       int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int *       baud rate
* Output Parameters:  int *       baud rate
******************************************************************************/
int HostAppEMU_ComGetBaudRate(int *pbps)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  switch (ComConfig.dcb.BaudRate)
  {
  case CBR_110: *pbps = 110; break;
  case CBR_300: *pbps = 300; break;
  case CBR_600: *pbps = 600; break;
  case CBR_1200: *pbps = 1200; break;
  case CBR_2400: *pbps = 2400; break;
  case CBR_4800: *pbps = 4800; break;
  case CBR_9600: *pbps = 9600; break;
  case CBR_14400: *pbps = 14400; break;
  case CBR_19200: *pbps = 19200; break;
  case CBR_38400: *pbps = 38400; break;
  case CBR_57600: *pbps = 57600; break;
  case CBR_115200: *pbps = 115200; break;
  default:
    *pbps = 0;
    result = HostAppEMU_STATUS_FAILURE;
    break;
  }
  HostAppEMU_Unlock();

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComSetDataBits
*
* DESCRIPTION:
*       API to set the com port data bits value.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int        data bits
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComSetDataBits(int databits)
{
  BYTE val;
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  val = (BYTE) databits;
  HostAppEMU_Unlock();

  if (result == HostAppEMU_STATUS_SUCCESS)
  {
    if (ComConfig.dcb.ByteSize != val)
    {
      ComConfig.dcb.ByteSize = val;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: data bits set to %d.\n"), __FUNCTION__, databits);
      HostAppEMU_ComPutSettings(&ComConfig);
    }
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Invalid data bits %d.\n"), __FUNCTION__, databits);
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComGetDataBits
*
* DESCRIPTION:
*       API to get the com port data bits value.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int *      data bits
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComGetDataBits(int *pdatabits)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  *pdatabits = ComConfig.dcb.ByteSize;
  HostAppEMU_Unlock();

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComSetParity
*
* DESCRIPTION:
*       API to set the com port parity value.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   TCHAR *    parity string
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComSetParity(TCHAR *sParity)
{
  BYTE val;
  int result = HostAppEMU_STATUS_FAILURE;

  HostAppEMU_Lock();
  if (CompareString(LOCALE_SYSTEM_DEFAULT, NORM_IGNORECASE, sParity, -1, TEXT("even"), -1) == CSTR_EQUAL)
  {
    val = EVENPARITY;
    result = HostAppEMU_STATUS_SUCCESS;
  }
  if (CompareString(LOCALE_SYSTEM_DEFAULT, NORM_IGNORECASE, sParity, -1, TEXT("odd"), -1) == CSTR_EQUAL)
  {
    val = ODDPARITY;
    result = HostAppEMU_STATUS_SUCCESS;
  }
  if (CompareString(LOCALE_SYSTEM_DEFAULT, NORM_IGNORECASE, sParity, -1, TEXT("none"), -1) == CSTR_EQUAL)
  {
    val = NOPARITY;
    result = HostAppEMU_STATUS_SUCCESS;
  }
  if (CompareString(LOCALE_SYSTEM_DEFAULT, NORM_IGNORECASE, sParity, -1, TEXT("mark"), -1) == CSTR_EQUAL)
  {
    val = MARKPARITY;
    result = HostAppEMU_STATUS_SUCCESS;
  }
  if (CompareString(LOCALE_SYSTEM_DEFAULT, NORM_IGNORECASE, sParity, -1, TEXT("space"), -1) == CSTR_EQUAL)
  {
    val = SPACEPARITY;
    result = HostAppEMU_STATUS_SUCCESS;
  }
  HostAppEMU_Unlock();

  if (result == HostAppEMU_STATUS_SUCCESS)
  {
    if (ComConfig.dcb.Parity != val)
    {
      ComConfig.dcb.Parity = val;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: parity set to %s.\n"), __FUNCTION__, sParity);
      HostAppEMU_ComPutSettings(&ComConfig);
    }
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Invalid parity %s.\n"), __FUNCTION__, sParity);
  }
  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComSetParity
*
* DESCRIPTION:
*       API to set the com port parity value.
*
* Return Value:       int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   TCHAR *     parity string
* Output Parameters:  TCHAR *     parity string
******************************************************************************/
int HostAppEMU_ComGetParity(TCHAR *sParity, int bufsize)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  switch (ComConfig.dcb.Parity)
  {
  case EVENPARITY: StringCbCopy(sParity, bufsize, TEXT("even")); break;
  case ODDPARITY: StringCbCopy(sParity, bufsize, TEXT("odd")); break;
  case NOPARITY: StringCbCopy(sParity, bufsize, TEXT("none")); break;
  case MARKPARITY: StringCbCopy(sParity, bufsize, TEXT("mark")); break;
  default:
    StringCbCopy(sParity, bufsize, TEXT(""));
    result = HostAppEMU_STATUS_FAILURE;
    break;
  }
  HostAppEMU_Unlock();

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComSetStopBits
*
* DESCRIPTION:
*       API to set the com port stop bits value.
*
* Return Value:       int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int         stop bits
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComSetStopBits(int stopbits)
{
  BYTE val;
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  switch (stopbits)
  {
  case 1: val = ONESTOPBIT; break;
  case 2: val = TWOSTOPBITS; break;
  case 15: val = ONE5STOPBITS; break;
  default:
    result = HostAppEMU_STATUS_FAILURE;
    break;
  }
  HostAppEMU_Unlock();

  if (result == HostAppEMU_STATUS_SUCCESS)
  {
    if (ComConfig.dcb.StopBits != val)
    {
      ComConfig.dcb.StopBits = val;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: stop bits set to %d.\n"), __FUNCTION__, stopbits);
      HostAppEMU_ComPutSettings(&ComConfig);
    }
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Invalid stop bits %d.\n"), __FUNCTION__, stopbits);
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComGetStopBits
*
* DESCRIPTION:
*       API to get the com port stop bits value.
*
* Return Value:       int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   int *       stop bits
* Output Parameters:  int *       stop bits
******************************************************************************/
int HostAppEMU_ComGetStopBits(int *pstopbits)
{
  int result = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();
  switch (ComConfig.dcb.StopBits)
  {
  case ONESTOPBIT: *pstopbits = 1; break;
  case TWOSTOPBITS: *pstopbits = 2; break;
  case ONE5STOPBITS: *pstopbits = 15; break;
  default:
    *pstopbits = 0;
    result = HostAppEMU_STATUS_FAILURE;
    break;
  }
  HostAppEMU_Unlock();

  return result;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComReadThreadProc
*
* DESCRIPTION:
*       Com Port read thread.  We read a message header, then read the message
*       body if necessary.
*
* Return Value:      DWORD       thread return value
* Input Parameters:  LPVOID      com port control
* Output Parameters: none
******************************************************************************/
DWORD WINAPI HostAppEMU_ComReadThreadProc(LPVOID lpParam)
{
  HostAppEMU_COM_CONFIG *pComConfig = (HostAppEMU_COM_CONFIG *) lpParam;
  UINT16 MsgHdr[2];
  HCT_MSG_EXT_HEADER_s ext_hdr_crcs;
  UINT16 crc16;
  unsigned short len;
  char *buf = (char *) NULL;
  HostAppEMU_STATUS status;
  int result;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: >>> Starting\n"), __FUNCTION__);
  Sleep(1000);

  for (;;)
  {
    // read message header from com port
    result = HostAppEMU_ComRead(pComConfig, &MsgHdr, sizeof(MsgHdr));
    if (result != HostAppEMU_STATUS_SUCCESS)
    {
      break;
    }

    if (HCT_MSG_is_EXT_TYPE(HCT_MSG_TYPE_GET(&MsgHdr)))
    {
      result = HostAppEMU_ComRead(pComConfig, &ext_hdr_crcs, sizeof(ext_hdr_crcs));
      if (result != HostAppEMU_STATUS_SUCCESS)
      {
        break;
      }
      if (ext_hdr_crcs.Header_CRC16 != 0)
      {
        crc16 = CRC16_BlockChecksum(&MsgHdr, HCT_MSG_HDR_SIZE);
        if (crc16 != ext_hdr_crcs.Header_CRC16)
        {
          /* TODO: note the error, but continue until we develop a recovery method */
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("CRC ERROR! message header (%#x <> %#x)\n"), crc16, ext_hdr_crcs.Header_CRC16);
        }
#if 0
        else
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("CRC %#x matches %#x\n"), crc16, ext_hdr_crcs.Header_CRC16);
        }
#endif
      }
      HCT_MSG_LEN_SET(&MsgHdr, HCT_MSG_LEN_GET(&MsgHdr) - sizeof(ext_hdr_crcs));
    }
    else
    {
      ext_hdr_crcs.Payload_CRC16 = ext_hdr_crcs.Header_CRC16 = 0;
    }

    // allocate a buffer for the message
    buf = (char *) LocalAlloc(0, (sizeof(MsgHdr) + HCT_MSG_LEN_GET(&MsgHdr) + 1) & ~1);
    if (! buf)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! LocalAlloc(%d) failed.\n"), HCT_MSG_LEN_GET(&MsgHdr));
      HostAppEMU_DbgPrintLastError();
      break;
    }

    // copy the message header into the buffer
    memcpy(buf, &MsgHdr, sizeof(MsgHdr));

    len = HCT_MSG_LEN_GET(&MsgHdr);
    if (len > 0)
    {
      // read the message into the buffer
      result = HostAppEMU_ComRead(pComConfig, buf + sizeof(MsgHdr), (len + 1) & ~1);
      if (result != HostAppEMU_STATUS_SUCCESS)
      {
        LocalFree(buf);
        buf = (char *) NULL;
        break;
      }

#if 0
      if (len & 1)
      {
        UINT8 c;

        result = HostAppEMU_ComRead(pComConfig, &c, 1);
        if (result != HostAppEMU_STATUS_SUCCESS)
        {
          LocalFree(buf);
          buf = (char *) NULL;
          break;
        }
      }
#endif

    }

    pComConfig->msgRead++;

    if (ext_hdr_crcs.Payload_CRC16 != 0)
    {
      crc16 = CRC16_BlockChecksum(buf + sizeof(MsgHdr), len);
      if (crc16 != ext_hdr_crcs.Payload_CRC16)
      {
        /* TODO: note the error, but continue until we develop a recovery method */
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("CRC ERROR! message payload (%#x <> %#x\n"), crc16, ext_hdr_crcs.Payload_CRC16);
      }
    }

    // queue the message to the processing queue
    status = HostAppEMU_ProcessDataMsg(buf);
    if (status != HostAppEMU_STATUS_SUCCESS)
    {
      LocalFree(buf);
      buf = (char *) NULL;
      if (status != HostAppEMU_STATUS_FIFO_FULL)
      {
        break;
      }
    }

    buf = (char *) NULL;
  }

  if (buf)
  {
    LocalFree(buf);
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: <<< Ending\n"), __FUNCTION__);
  return 0;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComRead
*
* DESCRIPTION:
*       Com Port read routine.  We read the given number of bytes from the com
*       port.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  HostAppEMU_COM_CONFIG *  com port control
*                    void *      buffer to read into
*                    int         count of bytes to read
* Output Parameters: void *      buffer filled with bytes read
******************************************************************************/
int HostAppEMU_ComRead(HostAppEMU_COM_CONFIG *pComConfig, void *buffer, int count)
{
  HANDLE hCom;
  DWORD maxread;
  DWORD dwNumberOfBytesRead = 0;
  int offset;

  HostAppEMU_Lock();
  hCom = pComConfig->hCom;
  HostAppEMU_Unlock();

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_TRACE, TEXT("%S: (%s, %p, %d)\n"), __FUNCTION__,
      pComConfig->sComName, buffer, count);

  for (offset = 0; offset < count; )
  {
    /* TODO:
    **  look into what appears to be if we're in a long read,
    **  other threads cannot get slice to run??
    **  for now, help out by reading for 5ms then sleep for 1
    */
    maxread = ((count - offset) > 36) ? 36 : count - offset; // ~ 5ms re5ad time

    if (!ReadFile(hCom, (char *) buffer + offset, maxread, &dwNumberOfBytesRead, 0))
    {
      unsigned long flags;

      HostAppEMU_GetFlags(&flags);
      if (!(flags & HostAppEMU_Exit))
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! ReadFile(%s, %d) failed.\n"), __FUNCTION__,
            pComConfig->sComName, count);
        HostAppEMU_DbgPrintLastError();
      }
      return HostAppEMU_STATUS_FAILURE;
    }

    if (dwNumberOfBytesRead > 0)
    {
      offset += (int) dwNumberOfBytesRead;
      if (offset < count)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: partial read of %d bytes, %d of %d completed\n"), __FUNCTION__,
            dwNumberOfBytesRead, offset, count);
        HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_COMM, (char *) buffer + (offset - dwNumberOfBytesRead), dwNumberOfBytesRead);
      }
    }

    Sleep(1);   // allow task switch
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: (%s, %p, %d).\n"), __FUNCTION__, pComConfig->sComName, buffer, count);
  HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_COMM, (char *) buffer, count);

  pComConfig->bytesRead += count;

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComWrite
*
* DESCRIPTION:
*       Com Port write routine.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   void *     buffer to write
*                     int        count of bytes to write
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_ComWrite(void *buffer, int count)
{
  DWORD dwNumberOfBytesWritten = 0;
  int status = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();

  if (!WriteFile(ComConfig.hCom, buffer, count, &dwNumberOfBytesWritten, 0))
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! WriteFile(%s, %d) failed.\n"), __FUNCTION__,
        ComConfig.sComName, count);
    HostAppEMU_DbgPrintLastError();
    status = HostAppEMU_STATUS_FAILURE;
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: (%s, %p, %d) == %d.\n"), __FUNCTION__, ComConfig.sComName, buffer, count, dwNumberOfBytesWritten);
  HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_COMM, (char *) buffer, dwNumberOfBytesWritten);

  HostAppEMU_Unlock();

  return status;
}
