/******************************************************************************
* FILE PURPOSE: AppEMUC data module.
*******************************************************************************
*
* FILE NAME: HostAppEMU_data.cpp
*
* DESCRIPTION:
*       Data module.
*
* HISTORY:
* 09/01/2007    stephen smith           initial revision
* 12/08/2009    stephen smith           port from TIWiMAXWrapper DLL
*
* LIST OF FUNCTIONS:
*
* Copyright (c) 2007-2009 Texas Instruments Inc.  All rights reserved.
******************************************************************************/
#include "stdafx.h"
#include <crtdbg.h>

#include "hostappemu.h"
#include "os_fifo.h"
#include "hostappemu_data.h"
#include "hostappemu_com.h"
#include "hostappemu_ip.h"
#include "hct_msg_def.h"
#include "crc16.h"
#include "hostappemu_sn.h"
#include "hostappemu_p2p.h"

HANDLE HostAppEMU_Data = 0;
HANDLE HostAppEMU_DataThread = 0;
DWORD HostAppEMU_DataThreadID = 0;

OS_FIFO HostAppEMU_MsgFifo = { };
OS_FIFO_CONTROL HostAppEMU_MsgFifoControl = { };
DWORD HostAppEMU_MsgFifoBuffer[HostAppEMU_MSG_FIFO_COUNT] = { };

HANDLE sema_sync_msg;
UINT16 sync_msg_type;

// local prototypes
DWORD WINAPI HostAppEMU_DataThreadProc(LPVOID lpParam);

/******************************************************************************
* FUNCTION NAME: HostAppEMU_InitData
*
* DESCRIPTION:
*       Initialization routine for data module.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
int HostAppEMU_InitData(void)
{
  int status = HostAppEMU_STATUS_SUCCESS;

  HostAppEMU_Lock();

  status = os_fifo_create(
      &HostAppEMU_MsgFifoControl,
      0,
      sizeof(HostAppEMU_MsgFifoBuffer[0]),
      sizeof(HostAppEMU_MsgFifoBuffer) / sizeof(HostAppEMU_MsgFifoBuffer[0])
  );
  if (status != OS_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! os_fifo_create() failed, error %#x\n"), __FUNCTION__, status);
  }
  if (status == HostAppEMU_STATUS_SUCCESS)
  {
    status = os_fifo_open(&HostAppEMU_MsgFifo, &HostAppEMU_MsgFifoControl, &HostAppEMU_MsgFifoBuffer[0]);
    if (status != OS_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! os_fifo_open() failed, error %#x\n"), __FUNCTION__, status);
    }
  }
  if (status == HostAppEMU_STATUS_SUCCESS)
  {
    HostAppEMU_DataThread = CreateThread(NULL, 0, HostAppEMU_DataThreadProc, 0, 0, &HostAppEMU_DataThreadID);
    if (!HostAppEMU_DataThread)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateThread() failed\n"), __FUNCTION__);
      HostAppEMU_DbgPrintLastError();
      status = HostAppEMU_STATUS_FAILURE;
    }
  }

  sema_sync_msg = CreateSemaphore(NULL, 0, 1, NULL);

  HostAppEMU_Unlock();

  return status;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_UninitData
*
* DESCRIPTION:
*       Uninitialization routine for data module.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
int HostAppEMU_UninitData(void)
{
  DWORD dw;
  OS_STATUS status = HostAppEMU_STATUS_SUCCESS;

  os_fifo_delete(&HostAppEMU_MsgFifo);
  for (;;)
  {
    if (!GetExitCodeThread(HostAppEMU_DataThread, &dw))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! GetExitCodeThread() failed\n"), __FUNCTION__);
      HostAppEMU_DbgPrintLastError();
      break;
    }
    if (dw != STILL_ACTIVE)
    {
      break;
    }
    Sleep(10);
  }
  CloseHandle(HostAppEMU_DataThread);

  return status;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ProcessDataMsg
*
* DESCRIPTION:
*       Process a data message.  We insert into the processing fifo and let
*       the process handle itself.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  char *      TLV data message to process
* Output Parameters: none
******************************************************************************/
int HostAppEMU_ProcessDataMsg(char *buf)
{
  OS_FIFO MsgFifo;
  int status;

  status = os_fifo_open(&MsgFifo, &HostAppEMU_MsgFifoControl, &HostAppEMU_MsgFifoBuffer[0]);
  if (status != OS_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! os_fifo_open() failed, error %#x\n"), __FUNCTION__, status);
    return status;
  }

  // queue the message to the processing queue
  status = os_fifo_put(&MsgFifo, &buf);
  if (status != OS_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! %#x inserting into MsgFifo, aborting.\n"), __FUNCTION__, status);
  }

  os_fifo_close(&MsgFifo);

  return status;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_DataThreadProc
*
* DESCRIPTION:
*       Data thread procedure.  Here we wait on a TLV message to be inserted
*       into the fifo, then call the process routine.
*
* Return Value:      DWORD       thread result
* Input Parameters:  LPVOID      pointer to TLV os_fifo object
* Output Parameters: none
******************************************************************************/
DWORD WINAPI HostAppEMU_DataThreadProc(LPVOID lpParam)
{
  OS_FIFO MsgFifo;
  char *buf;
  int status;

  status = os_fifo_open(&MsgFifo, &HostAppEMU_MsgFifoControl, &HostAppEMU_MsgFifoBuffer[0]);
  if (status != OS_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! os_fifo_open() failed, error %#x\n"), __FUNCTION__, status);
  }

  for (;;)
  {
    status = os_fifo_get(&MsgFifo, &buf);
    if (status != OS_STATUS_SUCCESS)
    {
      if (status != OS_STATUS_INTERRUPTED)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! %#x getting from MsgFifo, aborting.\n"), __FUNCTION__, status);
      }
      break;
    }

    HostAppEMU_Process_Message(buf);
  }

  os_fifo_close(&MsgFifo);

  return 0;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ReceivedSyncMsg
*
* DESCRIPTION:
*       Process receiving a Sync Message
*
* Return Value:      none
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void HostAppEMU_ReceivedSyncMsg(void)
{
//  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("%S: releasing semaphore.\n"), __FUNCTION__);
  ReleaseSemaphore(sema_sync_msg, 1, NULL);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Process_Message
*
* DESCRIPTION:
*       Routine that processes a TLV message.  We parse the message type, and
*       call the specific processing routine for that particular type.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  char *      TLV message
* Output Parameters: none
******************************************************************************/
int HostAppEMU_Process_Message(char *buf)
{
  UINT8 MsgHdr[HCT_MSG_HDR_SIZE];
  HCT_MSG_BUFFER_t *MsgPayload;
  int result = HostAppEMU_STATUS_SUCCESS;
  int msglen;
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 *pMsgHdr;

  memcpy(&MsgHdr[0], buf, HCT_MSG_HDR_SIZE);
  pMsgHdr = (UINT16 *) &MsgHdr[0];

  msglen = HCT_MSG_LEN_GET(&MsgHdr[0]);

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S\n"), __FUNCTION__);
  HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_COMM, buf, HCT_MSG_HDR_SIZE + msglen);

  memcpy(buf, buf + HCT_MSG_HDR_SIZE, msglen);
  MsgPayload = (HCT_MSG_BUFFER_t *) (buf);

  switch (HCT_MSG_TYPE_GET(&MsgHdr[0]))
  {
  case 0xa4:    // logger message
    break;

  case HCT_MSG_TYPE_DATA_TRANSFER:
    if (msglen > 0)
    {
      /* send DATA_TRANSFER reply if necessary */
      switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
      {
      case HostAppEMU_P2P:
        if (HostAppEMU_Flags & HostAppEMU_Use_IP)
        {
          hostappemu_ipv4_processPacket_DATA_TRANSFER(MsgPayload, msglen, HCT_MSG_RPY_GET(pMsgHdr) == HCT_MSG_TAG_MSG_RPY_ACK ? TRUE : FALSE);
        }
        else
        {
          hostappemu_p2p_processPacket_DATA_TRANSFER(MsgPayload, msglen, HCT_MSG_RPY_GET(pMsgHdr) == HCT_MSG_TAG_MSG_RPY_ACK ? TRUE : FALSE);
        }
        break;

      case HostAppEMU_MAC:
        if (msglen > HCT_MSG_RPY_DATA_TRANSFER_SIZE)
        {
          pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_RPY_DATA_TRANSFER_SIZE);
          if (pmsg)
          {
            *((UINT16 *) &pmsg[HCT_MSG_RPY_DATA_TRANSFER_OFFSET_Status >> 1]) = HCT_MSG_STATUS_OK;
            HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pmsg, HCT_MSG_RPY_DATA_TRANSFER_SIZE, TRUE);
          }
          else
          {
            ECA_OutOfMemory();
            HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to send DATA_TRANSFER reply.\n"), __FUNCTION__);
          }

          ECA_AppEmu_DATA_handler(MsgPayload, msglen, HCT_MSG_RPY_GET(pMsgHdr) == HCT_MSG_TAG_MSG_RPY_ACK ? TRUE : FALSE);
        }
        else
        {
          HostAppEMU_ReceivedSyncMsg();
        }
        break;

      case HostAppEMU_CL_LLC:
        /*
        ** check if DATA_TRANSFER.Indicate RPY,
        **  if set (we set it in LOAD_SYSTEM_CONFIG), we're going to tell the device
        **    when we can accept another packet by sending an empty DATA_TRANSFER message
        **  if not set
        **    the device can send us another packet anytime
        */
        ECA_AppEmu_DATA_handler(MsgPayload, msglen, HCT_MSG_RPY_GET(pMsgHdr) == HCT_MSG_TAG_MSG_RPY_ACK ? TRUE : FALSE);
        break;

      case HostAppEMU_IPv4:
        hostappemu_ipv4_processPacket_DATA_TRANSFER(MsgPayload, msglen, HCT_MSG_RPY_GET(pMsgHdr) == HCT_MSG_TAG_MSG_RPY_ACK ? TRUE : FALSE);
        break;

      case HostAppEMU_G3_Node:
      case HostAppEMU_G3_Conc:
        ECA_AppEmu_DATA_handler(MsgPayload, msglen, HCT_MSG_RPY_GET(pMsgHdr) == HCT_MSG_TAG_MSG_RPY_ACK ? TRUE : FALSE);
        break;

      }
    }
    else if (sync_msg_type != HCT_MSG_TYPE_DATA_TRANSFER)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: MSG SYNC ERROR! Expected reply for message type %#x, received %#x\n"),
          __FUNCTION__, sync_msg_type, HCT_MSG_TYPE_GET(pMsgHdr));
    }
    else
    {
      HostAppEMU_ReceivedSyncMsg();
    }
    break;

  case HCT_MSG_TYPE_SET_INFO:
    if (sync_msg_type != HCT_MSG_TYPE_SET_INFO)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: MSG SYNC ERROR! Expected reply for message type %#x, received %#x\n"),
          __FUNCTION__, sync_msg_type, HCT_MSG_TYPE_GET(pMsgHdr));
    }
    sync_msg_type = HCT_MSG_TYPE_INVALID_MSG;
    ECA_AppEmu_Set_Info_Handler(MsgPayload, msglen);
    HostAppEMU_ReceivedSyncMsg();
    break;

  case HCT_MSG_TYPE_SHUT_DOWN:

	
    if (sync_msg_type != HCT_MSG_TYPE_SHUT_DOWN)
    {
		//
		// Received a remote reset command
		//
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Received Remote Reset Request\n"), __FUNCTION__);
		ECA_AppEmu_RemoteRestart();
    }
	else
	{
		sync_msg_type = HCT_MSG_TYPE_INVALID_MSG;
		ECA_AppEmu_Shut_Down_Handler(MsgPayload, msglen);
		HostAppEMU_ReceivedSyncMsg();
	}
    break;

  case HCT_MSG_TYPE_GET_SYSTEM_INFO:
    if (sync_msg_type != HCT_MSG_TYPE_GET_SYSTEM_INFO)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: MSG SYNC ERROR! Expected reply for message type %#x, received %#x\n"),
          __FUNCTION__, sync_msg_type, HCT_MSG_TYPE_GET(pMsgHdr));
    }
    sync_msg_type = HCT_MSG_TYPE_INVALID_MSG;
    ECA_AppEmu_Get_System_Info_Handler(MsgPayload, msglen);
    HostAppEMU_ReceivedSyncMsg();
    break;

  case HCT_MSG_TYPE_SETUP_ALARM:
    if (sync_msg_type != HCT_MSG_TYPE_SETUP_ALARM)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: MSG SYNC ERROR! Expected reply for message type %#x, received %#x\n"),
          __FUNCTION__, sync_msg_type, HCT_MSG_TYPE_GET(pMsgHdr));
    }
    sync_msg_type = HCT_MSG_TYPE_INVALID_MSG;
    ECA_AppEmu_SETUP_ALARM_Handler(MsgPayload, msglen);
    HostAppEMU_ReceivedSyncMsg();
    break;

  case HCT_MSG_TYPE_ALARM:
    ECA_AppEmu_ALARM_Handler(pMsgHdr, MsgPayload, msglen);
    break;

  case HCT_MSG_TYPE_NETWORK_REGISTER:
//  case HCT_MSG_TYPE_NETWORK_START:
    ECA_AppEmu_NETWORK_REGISTER_or_NETWORK_START_handler(MsgPayload, msglen);
    break;

  case HCT_MSG_TYPE_NETWORK_UNREGISTER:
    ECA_AppEmu_NETWORK_UNREGISTER_handler(MsgPayload, msglen);
    break;

  case HCT_MSG_TYPE_CONNECT:
    ECA_AppEmu_CONNECT_handler(MsgPayload, msglen);
    break;

//  case HCT_MSG_TYPE_CL_ESTABLISH:
  case HCT_MSG_TYPE_ATTACH:
    ECA_AppEmu_CL_ESTABLISH_or_ATTACH_handler(MsgPayload, msglen);
    break;

//  case HCT_MSG_TYPE_CL_RELEASE:
  case HCT_MSG_TYPE_DETACH:
    ECA_AppEmu_CL_RELEASE_or_DETACH_handler(MsgPayload, msglen);
    break;

  case HCT_MSG_TYPE_LOAD_SYSTEM_CONFIG:
    ECA_AppEmu_Load_System_Config_Handler(MsgPayload, msglen);
    HostAppEMU_ReceivedSyncMsg();
    break;

  case HCT_MSG_TYPE_DISCOVER:
    ECA_AppEmu_DISCOVER_handler(MsgPayload, msglen);
    break;

	case HCT_MSG_TYPE_FIRMWARE_FLASH:
		ECA_AppEmu_FirmwareFlash(MsgPayload, msglen);
		break;

	case HCT_MSG_TYPE_FIRMWARE_UPGRADE:
		ECA_AppEmu_FirmwareUpgrade(MsgPayload, msglen);
		break;

	case HCT_MGG_TYPE_FIRMWARE_READ_BLOB:
		ECA_AppEmu_ReadBlob(MsgPayload, msglen);
		break;

	case HCT_MGG_TYPE_FIRMWARE_WRITE_BLOB:
		ECA_AppEmu_WriteBlob(MsgPayload, msglen);
		break;

	case HCT_MSG_TYPE_GET_MAC_PIB:
		ECA_AppEMU_Get_MAC_PIB(MsgPayload, msglen);
		break;
  default:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unknown message type %#x, len %d discarding.\n"), 
      __FUNCTION__, HCT_MSG_TYPE_GET(pMsgHdr), (int)(msglen));
    //remove for g3
    //HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_ERROR, buf, (int) (HCT_MSG_HDR_SIZE + msglen));
    result = HostAppEMU_STATUS_SUCCESS;
    LocalFree(buf);
    break;
  }

  if (result != HostAppEMU_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! %#x encountered processing message type %#x\n"),
        __FUNCTION__, result, HCT_MSG_TYPE_GET(pMsgHdr));
  }

  return result;
}

/******************************************************************************
* FUNCTION NAME: HCT_MSG_is_EXT_TYPE
*
* DESCRIPTION:
*       Returns TRUE if the message type is an extended HCT message.
*
* Return Value:       BOOL          TRUE if extended message type, otherwise FALSE
* Input Parameters:   UINT16        message type
* Output Parameters:  none
******************************************************************************/
BOOL HCT_MSG_is_EXT_TYPE(UINT16 msgtype)
{
  static UINT16 ext_msg_type[] = {
    HCT_MSG_TYPE_DATA_TRANSFER,
    HCT_MSG_TYPE_GET_SYSTEM_INFO,
    HCT_MSG_TYPE_GET_PHY_PIB,
    HCT_MSG_TYPE_GET_MAC_PIB,
    HCT_MSG_TYPE_SET_INFO,
    HCT_MSG_TYPE_SHUT_DOWN,
    HCT_MSG_TYPE_SETUP_ALARM,
    HCT_MSG_TYPE_ALARM,
    HCT_MSG_TYPE_NETWORK_REGISTER,
    HCT_MSG_TYPE_NETWORK_START,
    HCT_MSG_TYPE_NETWORK_UNREGISTER,
    HCT_MSG_TYPE_CONNECT,
    HCT_MSG_TYPE_DISCONNECT,
    HCT_MSG_TYPE_LOAD_SYSTEM_CONFIG,
    HCT_MSG_TYPE_CL_ESTABLISH,
    HCT_MSG_TYPE_CL_RELEASE,
    HCT_MSG_TYPE_ATTACH,
    HCT_MSG_TYPE_DETACH,
    HCT_MSG_TYPE_DISCOVER,
	HCT_MSG_TYPE_FIRMWARE_UPGRADE,
	HCT_MSG_TYPE_FIRMWARE_FLASH,
  };

  int i;

  for (i = 0; i < sizeof(ext_msg_type); i++)
  {
    if ((msgtype & HCT_MSG_TYPE_MASK) == ext_msg_type[i])
    {
      return TRUE;
    }
  }
  return FALSE;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_SendMsgToPLC
*
* DESCRIPTION:
*       Routine that sends a message to the PLC.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  char *      TLV message
* Output Parameters: none
******************************************************************************/
void HostAppEMU_SendMsgToPLC(UINT16 msgtype, UINT8 *msg, int msglen, BOOL b_UseRPY)
{
  int msgseq;
  UINT16 msghdr[HCT_MSG_HDR_SIZE >> 1];
  HCT_MSG_EXT_HEADER_s ext_hdr_crcs;
  BOOL b_ext_msg;

//  WaitForSingleObject(sema_SendMsgToPLC, INFINITE);

  msgseq = 0;
  b_ext_msg = HCT_MSG_is_EXT_TYPE(msgtype);

  msghdr[HCT_MSG_OFFSET_MSG_TYPE_TAG] = (msgtype & HCT_MSG_TYPE_MASK) | (HCT_MSG_TAG_ORG_HOST);

  HCT_MSG_TYPE_SET(&msghdr, msgtype);
  HCT_MSG_ORG_SET(&msghdr, HCT_MSG_TAG_ORG_HOST);


  /* RPY definition has changed.  It is _very_ specific to a certain message */
  if ((b_UseRPY)
        && (msgtype == HCT_MSG_TYPE_DATA_TRANSFER)
        && (
             ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_CL_LLC)
          || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_P2P)
        )
  )
  {
    HCT_MSG_RPY_SET(&msghdr, HCT_MSG_TAG_MSG_RPY_ACK);
  }
  else
  {
    HCT_MSG_RPY_SET(&msghdr, HCT_MSG_TAG_MSG_RPY_NO_ACK);
  }
  if (b_ext_msg)
  {
    /* extend the length to include CRCs */
    HCT_MSG_LEN_SET(&msghdr, msglen + sizeof(ext_hdr_crcs));
  }
  else
  {
    HCT_MSG_LEN_SET(&msghdr, msglen);
  }

  if (b_ext_msg)
  {
    ext_hdr_crcs.Header_CRC16 = CRC16_BlockChecksum(&msghdr[0], HCT_MSG_HDR_SIZE);
    if (msgtype != HCT_MSG_TYPE_DATA_TRANSFER)
    {
      ext_hdr_crcs.Payload_CRC16 = CRC16_BlockChecksum(msg, msglen);
    }
    else
    {
      ext_hdr_crcs.Payload_CRC16 = 0;
    }
  }

  if (HostAppEMU_Flags & HostAppEMU_Use_IP)
  {
    int len;
    UINT8 *pbuf;

    if (msgtype != HCT_MSG_TYPE_DATA_TRANSFER)
    {
      /* TODO:
      **  rework logic so that the buffer coming in has space for headers
      **  so we do not have to double buffer.
      */
      len = msglen + HCT_MSG_HDR_SIZE;
      if (b_ext_msg)
      {
        len += HCT_MSG_EXT_HDR_SIZE;
      }

      pbuf = (UINT8 *) LocalAlloc(LPTR, len);
      if (pbuf)
      {
        memcpy(pbuf, &msghdr, HCT_MSG_HDR_SIZE);
        if (b_ext_msg)
        {
          memcpy(&pbuf[HCT_MSG_HDR_SIZE], &ext_hdr_crcs, HCT_MSG_EXT_HDR_SIZE);
          if (msg)
          {
            memcpy(&pbuf[HCT_MSG_HDR_SIZE + HCT_MSG_EXT_HDR_SIZE], msg, msglen);
          }
          msglen += HCT_MSG_HDR_SIZE + HCT_MSG_EXT_HDR_SIZE;
        }
        else
        {
          if (msg)
          {
            memcpy(&pbuf[HCT_MSG_HDR_SIZE], msg, msglen);
          }
          msglen += HCT_MSG_HDR_SIZE;
        }
      }
      else
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Out of memory.\n"), __FUNCTION__);
        LocalFree(msg);
        return;
      }

      LocalFree(msg);
      msg = pbuf;
    }

    HostAppEMU_IP_SendPacket(msg, msglen, (msgtype != HCT_MSG_TYPE_DATA_TRANSFER) ? TRUE : FALSE);
    LocalFree(msg);
  }
  else
  {
    HostAppEMU_Lock();

    /* send the message header to the host */
    HostAppEMU_ComWrite(&msghdr[0], HCT_MSG_HDR_SIZE);

    if (b_ext_msg)
    {
      HostAppEMU_ComWrite(&ext_hdr_crcs, HCT_MSG_EXT_HDR_SIZE);
    }

    if (msglen > 0)
    {
      HostAppEMU_ComWrite(msg, msglen);

      if (msglen & 1)
      {
        UINT8 c;

        c = 0;
        HostAppEMU_ComWrite(&c, 1);
      }
      LocalFree(msg);
    }
    HostAppEMU_Unlock();
  }

//  ReleaseSemaphore(sema_SendMsgToPLC, 1, NULL);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_SendSyncMsgToPLC
*
* DESCRIPTION:
*       Routine that sends a syncronous message to the PLC and waits
*       for the reply.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  char *      TLV message
* Output Parameters: none
******************************************************************************/
int HostAppEMU_SendSyncMsgToPLC(UINT16 msgtype, UINT8 *msg, int msglen, BOOL b_UseRPY)
{
  DWORD dwWaitResult;

  sync_msg_type = msgtype;

  HostAppEMU_SendMsgToPLC(msgtype, msg, msglen, b_UseRPY);

  dwWaitResult = WaitForSingleObject(sema_sync_msg, 1000 * 5 /* INFINITE */);
  if (dwWaitResult != WAIT_OBJECT_0)
  {
    if (dwWaitResult == WAIT_TIMEOUT)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! Timeout send sync message 0x%04x, length %d (did not receive the response).\n"),
          msgtype, msglen);
      return -1;
    }
  }

  return 0;
}
