/******************************************************************************
* FILE PURPOSE: AppEMUC IP module.
*******************************************************************************
*
* FILE NAME: HostAppEMU_ip.cpp
*
* DESCRIPTION:
*       IP module.
*
* HISTORY:
* 04/26/2010    stephen smith           initial revision
*
* Copyright (c) 2010 Texas Instruments Inc.  All rights reserved.
******************************************************************************/
#include "stdafx.h"

#include <stdio.h>
#include <winsock2.h>
#include <ws2tcpip.h>

#include "hostappemu.h"
#include "hct_msg_def.h"
#include "hostappemu_ip.h"
#include "hostappemu_data.h"
#include "hostappemu_sn.h"
#include "hostappemu_p2p.h"
#include "crc16.h"

char *psrcaddr = (char *) NULL;
char *pdstaddr = (char *) NULL;
int udp_host_port = 8099;
int udp_data_port = 1234;   // port the base node is listening on

WSADATA wsaData;
SOCKET SendSocket = INVALID_SOCKET;
SOCKET DataSocket = INVALID_SOCKET;
SOCKET MsgSocket = INVALID_SOCKET;

HANDLE HostAppEMU_IP_Data_ReadThread = 0;
DWORD HostAppEMU_IP_Data_ReadThreadID = 0;
DWORD WINAPI HostAppEMU_IP_Data_ReadThreadProc(LPVOID lpParam);
HANDLE HostAppEMU_IP_Msg_ReadThread = 0;
DWORD HostAppEMU_IP_Msg_ReadThreadID = 0;
DWORD WINAPI HostAppEMU_IP_Msg_ReadThreadProc(LPVOID lpParam);

int HostAppEMU_IP_Recv(SOCKET RecvSocket, sockaddr_in *pRecvAddr, char *buffer, int *pcount);
static void HostAppEMU_IP_Print_WSA_Error(int WSA_Error_Code);

UINT32 HostAppEMU_IPv4Addr_to_UINT32(char *paddr)
{
  return (UINT32) htonl(inet_addr(paddr));
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Init_IP
*
* DESCRIPTION:
*       Initialization routine for IP module.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_Init_IP(void)
{
  sockaddr_in RecvAddr;
  int WSA_Error_Code;

  WSAStartup(MAKEWORD(2,2), &wsaData);

  SendSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (SendSocket == INVALID_SOCKET)
  {
    WSA_Error_Code = WSAGetLastError();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unable to open send socket, error %d.\n"), __FUNCTION__, WSA_Error_Code);
    HostAppEMU_IP_Print_WSA_Error(WSA_Error_Code);
    HostAppEMU_Uninit_IP();
    return HostAppEMU_STATUS_FAILURE;
  }

  DataSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (DataSocket == INVALID_SOCKET)
  {
    WSA_Error_Code = WSAGetLastError();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unable to open receive socket, error %d.\n"), __FUNCTION__, WSA_Error_Code);
    HostAppEMU_IP_Print_WSA_Error(WSA_Error_Code);
    HostAppEMU_Uninit_IP();
    return HostAppEMU_STATUS_FAILURE;
  }

  /* bind socket to any address on the port */
  RecvAddr.sin_family = AF_INET;
  RecvAddr.sin_port = htons(udp_data_port);
  RecvAddr.sin_addr.s_addr = htonl(INADDR_ANY);
  if (bind(DataSocket, (SOCKADDR *) &RecvAddr, sizeof(RecvAddr)) == SOCKET_ERROR)
  {
    WSA_Error_Code = WSAGetLastError();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unable to bind receive socket, error %d.\n"), __FUNCTION__, WSA_Error_Code);
    HostAppEMU_IP_Print_WSA_Error(WSA_Error_Code);
    HostAppEMU_Uninit_IP();
    return HostAppEMU_STATUS_FAILURE;
  }

  HostAppEMU_IP_Data_ReadThread = CreateThread(NULL, 0, HostAppEMU_IP_Data_ReadThreadProc, (LPVOID) DataSocket, 0, &HostAppEMU_IP_Data_ReadThreadID);
  if (!HostAppEMU_IP_Data_ReadThread)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateThread() failed\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
    return HostAppEMU_STATUS_FAILURE;
  }

  MsgSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (MsgSocket == INVALID_SOCKET)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unable to open receive socket, error %d.\n"), __FUNCTION__, WSAGetLastError());
    HostAppEMU_Uninit_IP();
    return HostAppEMU_STATUS_FAILURE;
  }

  /* bind socket to any address on the port */
  RecvAddr.sin_family = AF_INET;
  RecvAddr.sin_port = htons(udp_host_port);
  RecvAddr.sin_addr.s_addr = htonl(INADDR_ANY);
  if (bind(MsgSocket, (SOCKADDR *) &RecvAddr, sizeof(RecvAddr)) == SOCKET_ERROR)
  {
    WSA_Error_Code = WSAGetLastError();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unable to bind receive socket, error %d.\n"), __FUNCTION__, WSA_Error_Code);
    HostAppEMU_IP_Print_WSA_Error(WSA_Error_Code);
    HostAppEMU_Uninit_IP();
    return HostAppEMU_STATUS_FAILURE;
  }

  HostAppEMU_IP_Msg_ReadThread = CreateThread(NULL, 0, HostAppEMU_IP_Msg_ReadThreadProc, (LPVOID) MsgSocket, 0, &HostAppEMU_IP_Msg_ReadThreadID);
  if (!HostAppEMU_IP_Msg_ReadThread)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateThread() failed\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
    return HostAppEMU_STATUS_FAILURE;
  }

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_Uninit_IP
*
* DESCRIPTION:
*       Uninitialization routine for IP module.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_Uninit_IP(void)
{
  closesocket(DataSocket);
  closesocket(MsgSocket);
  closesocket(SendSocket);

  WSACleanup();

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_IP_ReadThreadProc
*
* DESCRIPTION:
*       IP read thread.
*
* Return Value:      DWORD       thread return value
* Input Parameters:  LPVOID      com port control
* Output Parameters: none
******************************************************************************/
DWORD WINAPI HostAppEMU_IP_Data_ReadThreadProc(LPVOID lpParam) 
{
  SOCKET Socket;
  char *pippacket;
  sockaddr_in RecvAddr;
  UINT16 *pMsgHdr;
  int count;
  HostAppEMU_STATUS status;
  int result;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: >>> Starting\n"), __FUNCTION__);
  Sleep(1000);

  pippacket = (char *) NULL;
  Socket = (SOCKET) lpParam;

  for (;;)
  {
    // allocate a buffer for the message
    count = 0x1000;
    pippacket = (char *) LocalAlloc(0, count + HCT_MSG_HDR_SIZE);
    if (! pippacket)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! LocalAlloc(%d) failed.\n"), 1500);
      HostAppEMU_DbgPrintLastError();
      break;
    }

    result = HostAppEMU_IP_Recv(Socket, &RecvAddr, &pippacket[HCT_MSG_HDR_SIZE], &count);
    if (result != HostAppEMU_STATUS_SUCCESS)
    {
      break;
    }

    /* create a pseudo message header so we can insert into the regular message queue */
    pMsgHdr = (UINT16 *) pippacket;
    HCT_MSG_TYPE_SET(pMsgHdr, HCT_MSG_TYPE_DATA_TRANSFER);
    HCT_MSG_TAG_SET(pMsgHdr, HCT_MSG_TAG_ORG_MAC);
    HCT_MSG_LEN_SET(pMsgHdr, count);

    // queue the message to the processing queue
    status = HostAppEMU_ProcessDataMsg(pippacket);
    if (status != HostAppEMU_STATUS_SUCCESS)
    {
      pippacket = (char *) NULL;
      if (status != HostAppEMU_STATUS_FIFO_FULL)
      {
        break;
      }
    }
  }

  if (pippacket)
  {
    LocalFree(pippacket);
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: <<< Ending\n"), __FUNCTION__);
  return 0;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_IP_ReadThreadProc
*
* DESCRIPTION:
*       IP read thread.
*
* Return Value:      DWORD       thread return value
* Input Parameters:  LPVOID      com port control
* Output Parameters: none
******************************************************************************/
DWORD WINAPI HostAppEMU_IP_Msg_ReadThreadProc(LPVOID lpParam) 
{
  SOCKET Socket;
  char *pippacket;
  sockaddr_in RecvAddr;
  UINT16 *pMsgHdr;
  HCT_MSG_EXT_HEADER_s *pext_hdr_crcs;
  UINT16 crc16;
  int count;
  BOOL b_ExtType;
  int msglen;
  HostAppEMU_STATUS status;
  int result;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: >>> Starting\n"), __FUNCTION__);
  Sleep(1000);

  pippacket = (char *) NULL;
  Socket = (SOCKET) lpParam;

  for (;;)
  {
    // allocate a buffer for the message
    count = 1500;
    pippacket = (char *) LocalAlloc(0, count);
    if (! pippacket)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! LocalAlloc(%d) failed.\n"), 1500);
      HostAppEMU_DbgPrintLastError();
      break;
    }

    result = HostAppEMU_IP_Recv(Socket, &RecvAddr, pippacket, &count);
    if (result != HostAppEMU_STATUS_SUCCESS)
    {
      break;
    }

    if (count < HCT_MSG_HDR_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Invalid packet size %d, expected at least %d!\n"), count, HCT_MSG_HDR_SIZE);
      LocalFree(pippacket);
      continue;
    }

    pMsgHdr = (UINT16 *) pippacket;

    msglen = HCT_MSG_HDR_SIZE + HCT_MSG_LEN_GET(pMsgHdr);
    if (count < msglen)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Invalid packet size %d, expected at least %d!\n"), count, msglen);
      LocalFree(pippacket);
      continue;
    }

    b_ExtType = HCT_MSG_is_EXT_TYPE(HCT_MSG_TYPE_GET(pMsgHdr));
    if (b_ExtType)
    {
      pext_hdr_crcs = (HCT_MSG_EXT_HEADER_s *) &pippacket[HCT_MSG_HDR_SIZE];
      if (pext_hdr_crcs->Header_CRC16 != 0)
      {
        crc16 = CRC16_BlockChecksum(pMsgHdr, HCT_MSG_HDR_SIZE);
        if (crc16 != pext_hdr_crcs->Header_CRC16)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("CRC ERROR! message header (%#x <> %#x)\n"), crc16, pext_hdr_crcs->Header_CRC16);
          LocalFree(pippacket);
          continue;
        }
      }
      if (pext_hdr_crcs->Payload_CRC16 != 0)
      {
        crc16 = CRC16_BlockChecksum(&pippacket[HCT_MSG_HDR_SIZE + HCT_MSG_EXT_HDR_SIZE], msglen - (HCT_MSG_HDR_SIZE + HCT_MSG_EXT_HDR_SIZE));
        if (crc16 != pext_hdr_crcs->Payload_CRC16)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("CRC ERROR! message payload (%#x <> %#x\n"), crc16, pext_hdr_crcs->Payload_CRC16);
          LocalFree(pippacket);
          continue;
        }
      }

      memcpy(pext_hdr_crcs, &pext_hdr_crcs[1], count - (HCT_MSG_HDR_SIZE + HCT_MSG_EXT_HDR_SIZE));
      msglen -= HCT_MSG_EXT_HDR_SIZE;
      HCT_MSG_LEN_SET(pMsgHdr, HCT_MSG_LEN_GET(pMsgHdr) - HCT_MSG_EXT_HDR_SIZE);
    }

    // queue the message to the processing queue
    status = HostAppEMU_ProcessDataMsg(pippacket);
    if (status != HostAppEMU_STATUS_SUCCESS)
    {
      pippacket = (char *) NULL;
      if (status != HostAppEMU_STATUS_FIFO_FULL)
      {
        break;
      }
    }
  }

  if (pippacket)
  {
    LocalFree(pippacket);
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: <<< Ending\n"), __FUNCTION__);
  return 0;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_ComRead
*
* DESCRIPTION:
*       Com Port read routine.  We read the given number of bytes from the com
*       port.
*
* Return Value:      int         HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:  HostAppEMU_COM_CONFIG *  com port control
*                    void *      buffer to read into
*                    int         count of bytes to read
* Output Parameters: void *      buffer filled with bytes read
******************************************************************************/
int HostAppEMU_IP_Recv(SOCKET RecvSocket, sockaddr_in *pRecvAddr, char *buffer, int *pcount) 
{
  int RecvAddrSize;
  int WSA_Error_Code;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_TRACE, TEXT("%S: (%p, %d)\n"), __FUNCTION__,
      buffer, *pcount);

  memset(pRecvAddr, 0, sizeof(*pRecvAddr));

  RecvAddrSize = sizeof(*pRecvAddr);
  *pcount = recvfrom(RecvSocket, buffer, *pcount, 0, (SOCKADDR *) pRecvAddr, &RecvAddrSize);
  if (*pcount == SOCKET_ERROR)
  {
    WSA_Error_Code = WSAGetLastError();
    if (WSA_Error_Code != WSAEINTR)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! recvfrom(%S, %d) failed, error %d.\n"), __FUNCTION__,
        pdstaddr, pRecvAddr->sin_port, WSA_Error_Code);
      HostAppEMU_IP_Print_WSA_Error(WSA_Error_Code);
    }
    return HostAppEMU_STATUS_FAILURE;
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_COMM, TEXT("%S: (%p, %d, %S, %d).\n"), __FUNCTION__,
    buffer, *pcount, inet_ntoa(pRecvAddr->sin_addr), pRecvAddr->sin_port);
  HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_COMM, (char *) buffer, *pcount);

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_IP_SendPacket
*
* DESCRIPTION:
*       IP send routine.
*
* Return Value:       int        HostAppEMU_STATUS_SUCCESS or errorlevel
* Input Parameters:   void *     buffer to write
*                     int        count of bytes to write
*                     BOOL       HCT message
* Output Parameters:  none
******************************************************************************/
int HostAppEMU_IP_SendPacket(void *buffer, int count, BOOL bMsg)
{
  sockaddr_in RecvAddr;
  int WSA_Error_Code;

  RecvAddr.sin_family = AF_INET;
  if (bMsg)
  {
    RecvAddr.sin_addr.S_un.S_addr = inet_addr(psrcaddr);
    RecvAddr.sin_addr.S_un.S_addr |= 0xff000000;   // convert to #.#.#.255
    RecvAddr.sin_port = htons(udp_host_port);
  }
  else
  {
    RecvAddr.sin_addr.S_un.S_addr = inet_addr(pdstaddr);
    RecvAddr.sin_port = htons(udp_data_port);
  }

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ALWAYS, TEXT("Send packet to %S::%d (len:%d).\n"),
    inet_ntoa(RecvAddr.sin_addr), ntohs(RecvAddr.sin_port), count);
  HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_COMM, (char *) buffer, count);

  if (sendto(SendSocket, (char *) buffer, count, MSG_DONTROUTE, (SOCKADDR *) &RecvAddr, sizeof(RecvAddr)) == SOCKET_ERROR)
  {
    WSA_Error_Code = WSAGetLastError();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! sendto(%S, %d) failed, error %d.\n"), __FUNCTION__,
        pdstaddr, RecvAddr.sin_port, WSA_Error_Code);
    HostAppEMU_IP_Print_WSA_Error(WSA_Error_Code);

    return HostAppEMU_STATUS_FAILURE;
  }

  return HostAppEMU_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_ipv4_SendTextMessage
*
* DESCRIPTION:
*       Send text message.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_ipv4_SendTextMessage(UINT8 *pui8, UINT16 len)
{
  UINT8 *pmsg;
  int msglen;

  /*
  ** in IPv4 mode we will just wrap in the MAC DATA_TRANSFER message
  */
  msglen = HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + sizeof(CPacketDataMessage_Header) + len;

  /* allocate the packet */
  pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    CPacketDataMessage_Header *pdmh;

    /* set LLC DATA_TRANSFER header */
    *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Conn_Handle]) = 0;

    /* set text message header */
    pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Data];
    pdmh->m_Signature = pdmh->Signature;
    pdmh->m_SubType = pdmh->SubType_Message;

    /* set text message payload */
    memcpy(&pdmh[1], pui8, len);

    /* send packet */
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Sending Text Message %*.*S.\n"), len, len, pui8);

    HostAppEMU_SendMsgToPLC(
        HCT_MSG_TYPE_DATA_TRANSFER,
        (UINT8 *) pmsg,
        msglen,
        FALSE
    );

    hostappemu_exitapp();
  }
  else
  {
    ECA_OutOfMemory();
  }
}

/******************************************************************************
* FUNCTION NAME: hostappemu_ipv4_processPacket_DATA_TRANSFER
*
* DESCRIPTION:
*       Process DATA_TRANSFER.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_ipv4_processPacket_DATA_TRANSFER(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY)
{
  UINT8 *pui8;

  pui8 = (UINT8 *) pmsg;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_TRACE, TEXT("%S: (%#x, %d, %d\n"), __FUNCTION__, pmsg, msg_len, bRPY);
  HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_COMM, (char *) pmsg, (int) msg_len);

  const CPacketDataMessage_Header *pPacket = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Data >> 1];
  short nLength = (short) msg_len - HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE;

  CXferInformation XferInfo;

  memset(&XferInfo, 0, sizeof(XferInfo));

  if (pPacket->m_Signature == pPacket->Signature)
  {
    switch (pPacket->m_SubType)
    {
    case pPacket->SubType_Message:
      {
        /* update transfer information */
        XferInfo.m_Xfer_TTL_RX_packets = 1;
        XferInfo.m_Xfer_TTL_RX_bytes = msg_len;
        XferInfo.m_Xfer_TTL_RX_messages = 1;

        pui8 = ((UINT8 *) pPacket) + sizeof(*pPacket);
        nLength -= sizeof(*pPacket);

//        OnUpdate_XferRxInfo(&XferInfo);

        /* display the received message */
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ALWAYS, TEXT("IPv4 Text Message Received (len:%d): %*.*S\n"), nLength, nLength, nLength, pui8);
      }
      break;
#if 0
#endif

    default:
      break;
    }
  }

  LocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: HostAppEMU_IP_Print_WSA_Error
*
* DESCRIPTION:
*       Prints the WSA Error code and description.
*
* Return Value:       none
* Input Parameters:   int        WSA Error Code
* Output Parameters:  none
******************************************************************************/
static void HostAppEMU_IP_Print_WSA_Error(int WSA_Error_Code)
{
  static const WSA_Error_Desc_s WSA_Error_Desc[] = {
    { /* 6 */
      WSA_INVALID_HANDLE,
      "WSA_INVALID_HANDLE",
      "Specified event object handle is invalid.",
      "An application attempts to use an event object, but the specified handle is not valid. Note that this error is returned by the operating system, so the error number may change in future releases of Windows."
    },

    { /* 8 */
      WSA_NOT_ENOUGH_MEMORY, 
      "WSA_NOT_ENOUGH_MEMORY," 
      "Insufficient memory available.",
      "An application used a Windows Sockets function that directly maps to a Windows function. The Windows function is indicating a lack of required memory resources. Note that this error is returned by the operating system, so the error number may change in future releases of Windows."
    },

    { /* 87 */
      WSA_INVALID_PARAMETER,
      "WSA_INVALID_PARAMETER",
      "One or more parameters are invalid.",
      "An application used a Windows Sockets function which directly maps to a Windows function. The Windows function is indicating a problem with one or more parameters. Note that this error is returned by the operating system, so the error number may change in future releases of Windows."
    },

    { /* 995 */
      WSA_OPERATION_ABORTED,
      "WSA_OPERATION_ABORTED",
      "Overlapped operation aborted.",
      "An overlapped operation was canceled due to the closure of the socket, or the execution of the SIO_FLUSH command in WSAIoctl. Note that this error is returned by the operating system, so the error number may change in future releases of Windows."
    },

    { /* 996 */
      WSA_IO_INCOMPLETE,
      "WSA_IO_INCOMPLETE",
      "Overlapped I/O event object not in signaled state.",
      "The application has tried to determine the status of an overlapped operation which is not yet completed. Applications that use WSAGetOverlappedResult (with the fWait flag set to FALSE) in a polling mode to determine when an overlapped operation has completed, get this error code until the operation is complete. Note that this error is returned by the operating system, so the error number may change in future releases of Windows."
    },

    { /* 997 */
      WSA_IO_PENDING,
      "WSA_IO_PENDING",
      "Overlapped operations will complete later. ",
      "The application has initiated an overlapped operation that cannot be completed immediately. A completion indication will be given later when the operation has been completed. Note that this error is returned by the operating system, so the error number may change in future releases of Windows."
    },

    { /* 10004  */
      WSAEINTR,
      "WSAEINTR",
      "Interrupted function call.",
      "A blocking operation was interrupted by a call to WSACancelBlockingCall."
    },

    { /* 10009  */
      WSAEBADF,
      "WSAEBADF",
      "File handle is not valid.",
      "The file handle supplied is not valid. "
    },

    { /* 10013  */
      WSAEACCES,
      "WSAEACCES",
      "Permission denied.",
      "An attempt was made to access a socket in a way forbidden by its access permissions. An example is using a broadcast address for sendto without broadcast permission being set using setsockopt(SO_BROADCAST). "
      "Another possible reason for the WSAEACCES error is that when the bind function is called (on Windows NT 4.0 with SP4 and later), another application, service, or kernel mode driver is bound to the same address with exclusive access. Such exclusive access is a new feature of Windows NT 4.0 with SP4 and later, and is implemented by using the SO_EXCLUSIVEADDRUSE option."
    },

    { /* 10014  */
      WSAEFAULT,
      "WSAEFAULT",
      "Bad address.",
      "The system detected an invalid pointer address in attempting to use a pointer argument of a call. This error occurs if an application passes an invalid pointer value, or if the length of the buffer is too small. For instance, if the length of an argument, which is a sockaddr structure, is smaller than the sizeof(sockaddr)."
    },

    { /* 10022  */
      WSAEINVAL,
      "WSAEINVAL",
      "Invalid argument.",
      "Some invalid argument was supplied (for example, specifying an invalid level to the setsockopt function). In some instances, it also refers to the current state of the socketfor instance, calling accept on a socket that is not listening."
    },

    { /* 10024  */
      WSAEMFILE,
      "WSAEMFILE",
      "Too many open files.",
      "Too many open sockets. Each implementation may have a maximum number of socket handles available, either globally, per process, or per thread."
    },

    { /* 10035  */
      WSAEWOULDBLOCK,
      "WSAEWOULDBLOCK",
      "Resource temporarily unavailable.",
      "This error is returned from operations on nonblocking sockets that cannot be completed immediately, for example recv when no data is queued to be read from the socket. It is a nonfatal error, and the operation should be retried later. It is normal for WSAEWOULDBLOCK to be reported as the result from calling connect on a nonblocking SOCK_STREAM socket, since some time must elapse for the connection to be established."
    },

    { /* 10036  */
      WSAEINPROGRESS,
      "WSAEINPROGRESS",
      "Operation now in progress.",
      "A blocking operation is currently executing. Windows Sockets only allows a single blocking operationper- task or threadto be outstanding, and if any other function call is made (whether or not it references that or any other socket) the function fails with the WSAEINPROGRESS error."
    },

    { /* 10037  */
      WSAEALREADY,
      "WSAEALREADY",
      "Operation already in progress.",
      "An operation was attempted on a nonblocking socket with an operation already in progressthat is, calling connect a second time on a nonblocking socket that is already connecting, or canceling an asynchronous request (WSAAsyncGetXbyY) that has already been canceled or completed."
    },

    { /* 10038  */
      WSAENOTSOCK,
      "WSAENOTSOCK",
      "Socket operation on nonsocket.",
      "An operation was attempted on something that is not a socket. Either the socket handle parameter did not reference a valid socket, or for select, a member of an fd_set was not valid."
    },

    { /* 10039  */
      WSAEDESTADDRREQ,
      "WSAEDESTADDRREQ",
      "Destination address required.",
      "A required address was omitted from an operation on a socket. For example, this error is returned if sendto is called with the remote address of ADDR_ANY."
    },

    { /* 10040  */
      WSAEMSGSIZE,
      "WSAEMSGSIZE",
      "Message too long.",
      "A message sent on a datagram socket was larger than the internal message buffer or some other network limit, or the buffer used to receive a datagram was smaller than the datagram itself."
    },

    { /* 10041  */
      WSAEPROTOTYPE,
      "WSAEPROTOTYPE",
      "Protocol wrong type for socket.",
      "A protocol was specified in the socket function call that does not support the semantics of the socket type requested. For example, the ARPA Internet UDP protocol cannot be specified with a socket type of SOCK_STREAM."
    },

    { /* 10042  */
      WSAENOPROTOOPT,
      "WSAENOPROTOOPT",
      "Bad protocol option.",
      "An unknown, invalid or unsupported option or level was specified in a getsockopt or setsockopt call."
    },

    { /* 10043  */
      WSAEPROTONOSUPPORT,
      "WSAEPROTONOSUPPORT",
      "Protocol not supported.",
      "The requested protocol has not been configured into the system, or no implementation for it exists. For example, a socket call requests a SOCK_DGRAM socket, but specifies a stream protocol."
    },

    { /* 10044  */
      WSAESOCKTNOSUPPORT,
      "WSAESOCKTNOSUPPORT",
      "Socket type not supported.",
      "The support for the specified socket type does not exist in this address family. For example, the optional type SOCK_RAW might be selected in a socket call, and the implementation does not support SOCK_RAW sockets at all."
    },

    { /* 10045  */
      WSAEOPNOTSUPP,
      "WSAEOPNOTSUPP",
      "Operation not supported.",
      "The attempted operation is not supported for the type of object referenced. Usually this occurs when a socket descriptor to a socket that cannot support this operation is trying to accept a connection on a datagram socket."
    },

    { /* 10046  */
      WSAEPFNOSUPPORT,
      "WSAEPFNOSUPPORT",
      "Protocol family not supported.",
      "The protocol family has not been configured into the system or no implementation for it exists. This message has a slightly different meaning from WSAEAFNOSUPPORT. However, it is interchangeable in most cases, and all Windows Sockets functions that return one of these messages also specify WSAEAFNOSUPPORT."
    },

    { /* 10047  */
      WSAEAFNOSUPPORT,
      "WSAEAFNOSUPPORT",
      "Address family not supported by protocol family.",
      "An address incompatible with the requested protocol was used. All sockets are created with an associated address family (that is, AF_INET for Internet Protocols) and a generic protocol type (that is, SOCK_STREAM). This error is returned if an incorrect protocol is explicitly requested in the socket call, or if an address of the wrong family is used for a socket, for example, in sendto."
    },

    { /* 10048  */
      WSAEADDRINUSE,
      "WSAEADDRINUSE",
      "Address already in use.",
      "Typically, only one usage of each socket address (protocol/IP address/port) is permitted. This error occurs if an application attempts to bind a socket to an IP address/port that has already been used for an existing socket, or a socket that was not closed properly, or one that is still in the process of closing. For server applications that need to bind multiple sockets to the same port number, consider using setsockopt (SO_REUSEADDR). Client applications usually need not call bind at allconnect chooses an unused port automatically. When bind is called with a wildcard address (involving ADDR_ANY), a WSAEADDRINUSE error could be delayed until the specific address is committed. This could happen with a call to another function later, including connect, listen, WSAConnect, or WSAJoinLeaf."
    },

    { /* 10049  */
      WSAEADDRNOTAVAIL,
      "WSAEADDRNOTAVAIL",
      "Cannot assign requested address.",
      "The requested address is not valid in its context. This normally results from an attempt to bind to an address that is not valid for the local computer. This can also result from connect, sendto, WSAConnect, WSAJoinLeaf, or WSASendTo when the remote address or port is not valid for a remote computer (for example, address or port 0)."
    },

    { /* 10050  */
      WSAENETDOWN,
      "WSAENETDOWN",
      "Network is down.",
      "A socket operation encountered a dead network. This could indicate a serious failure of the network system (that is, the protocol stack that the Windows Sockets DLL runs over), the network interface, or the local network itself."
    },

    { /* 10051  */
      WSAENETUNREACH,
      "WSAENETUNREACH",
      "Network is unreachable.",
      "A socket operation was attempted to an unreachable network. This usually means the local software knows no route to reach the remote host."
    },

    { /* 10052  */
      WSAENETRESET,
      "WSAENETRESET",
      "Network dropped connection on reset.",
      "The connection has been broken due to keep-alive activity detecting a failure while the operation was in progress. It can also be returned by setsockopt if an attempt is made to set SO_KEEPALIVE on a connection that has already failed."
    },

    { /* 10053  */
      WSAECONNABORTED,
      "WSAECONNABORTED",
      "Software caused connection abort.",
      "An established connection was aborted by the software in your host computer, possibly due to a data transmission time-out or protocol error."
    },

    { /* 10054  */
      WSAECONNRESET,
      "WSAECONNRESET",
      "Connection reset by peer.",
      "An existing connection was forcibly closed by the remote host. This normally results if the peer application on the remote host is suddenly stopped, the host is rebooted, the host or remote network interface is disabled, or the remote host uses a hard close (see setsockopt for more information on the SO_LINGER option on the remote socket). This error may also result if a connection was broken due to keep-alive activity detecting a failure while one or more operations are in progress. Operations that were in progress fail with WSAENETRESET. Subsequent operations fail with WSAECONNRESET."
    },

    { /* 10055  */
      WSAENOBUFS,
      "WSAENOBUFS",
      "No buffer space available.",
      "An operation on a socket could not be performed because the system lacked sufficient buffer space or because a queue was full."
    },

    { /* 10056  */
      WSAEISCONN,
      "WSAEISCONN",
      "Socket is already connected.",
      "A connect request was made on an already-connected socket. Some implementations also return this error if sendto is called on a connected SOCK_DGRAM socket (for SOCK_STREAM sockets, the to parameter in sendto is ignored) although other implementations treat this as a legal occurrence."
    },

    { /* 10057  */
      WSAENOTCONN,
      "WSAENOTCONN",
      "Socket is not connected.",
      "A request to send or receive data was disallowed because the socket is not connected and (when sending on a datagram socket using sendto) no address was supplied. Any other type of operation might also return this errorfor example, setsockopt setting SO_KEEPALIVE if the connection has been reset."
    },

    { /* 10058  */
      WSAESHUTDOWN,
      "WSAESHUTDOWN",
      "Cannot send after socket shutdown.",
      "A request to send or receive data was disallowed because the socket had already been shut down in that direction with a previous shutdown call. By calling shutdown a partial close of a socket is requested, which is a signal that sending or receiving, or both have been discontinued."
    },

    { /* 10059  */
      WSAETOOMANYREFS,
      "WSAETOOMANYREFS",
      "Too many references.",
      "Too many references to some kernel object."
    },

    { /* 10060  */
      WSAETIMEDOUT,
      "WSAETIMEDOUT",
      "Connection timed out.",
      "A connection attempt failed because the connected party did not properly respond after a period of time, or the established connection failed because the connected host has failed to respond."
    },

    { /* 10061  */
      WSAECONNREFUSED,
      "WSAECONNREFUSED",
      "Connection refused.",
      "No connection could be made because the target computer actively refused it. This usually results from trying to connect to a service that is inactive on the foreign hostthat is, one with no server application running."
    },

    { /* 10062  */
      WSAELOOP,
      "WSAELOOP",
      "Cannot translate name.",
      "Cannot translate a name."
    },

    { /* 10063  */
      WSAENAMETOOLONG,
      "WSAENAMETOOLONG",
      "Name too long.",
      "A name component or a name was too long."
    },

    { /* 10064  */
      WSAEHOSTDOWN,
      "WSAEHOSTDOWN",
      "Host is down.",
      "A socket operation failed because the destination host is down. A socket operation encountered a dead host. Networking activity on the local host has not been initiated. These conditions are more likely to be indicated by the error WSAETIMEDOUT."
    },

    { /* 10065  */
      WSAEHOSTUNREACH,
      "WSAEHOSTUNREACH",
      "No route to host.",
      "A socket operation was attempted to an unreachable host. See WSAENETUNREACH."
    },

    { /* 10066  */
      WSAENOTEMPTY,
      "WSAENOTEMPTY",
      "Directory not empty.",
      "Cannot remove a directory that is not empty."
    },

    { /* 10067  */
      WSAEPROCLIM,
      "WSAEPROCLIM",
      "Too many processes.",
      "A Windows Sockets implementation may have a limit on the number of applications that can use it simultaneously. WSAStartup may fail with this error if the limit has been reached."
    },

    { /* 10068  */
      WSAEUSERS,
      "WSAEUSERS",
      "User quota exceeded.",
      "Ran out of user quota. "
    },

    { /* 10069  */
      WSAEDQUOT,
      "WSAEDQUOT",
      "Disk quota exceeded.",
      "Ran out of disk quota. "
    },

    { /* 10070  */
      WSAESTALE,
      "WSAESTALE",
      "Stale file handle reference.",
      "The file handle reference is no longer available. "
    },

    { /* 10071  */
      WSAEREMOTE,
      "WSAEREMOTE",
      "Item is remote.",
      "The item is not available locally. "
    },

    { /* 10091  */
      WSASYSNOTREADY,
      "WSASYSNOTREADY",
      "Network subsystem is unavailable.",
      "This error is returned by WSAStartup if the Windows Sockets implementation cannot function at this time because the underlying system it uses to provide network services is currently unavailable. Users should check:"
      "  That the appropriate Windows Sockets DLL file is in the current path."
      "  That they are not trying to use more than one Windows Sockets implementation simultaneously. If there is more than one Winsock DLL on your system, be sure the first one in the path is appropriate for the network subsystem currently loaded."
      "  The Windows Sockets implementation documentation to be sure all necessary components are currently installed and configured correctly."
    },

    { /* 10092  */
      WSAVERNOTSUPPORTED,
      "WSAVERNOTSUPPORTED",
      "Winsock.dll version out of range.",
      "The current Windows Sockets implementation does not support the Windows Sockets specification version requested by the application. Check that no old Windows Sockets DLL files are being accessed."
    },

    { /* 10093  */
      WSANOTINITIALISED,
      "WSANOTINITIALISED",
      "Successful WSAStartup not yet performed.",
      "Either the application has not called WSAStartup or WSAStartup failed. The application may be accessing a socket that the current active task does not own (that is, trying to share a socket between tasks), or WSACleanup has been called too many times."
    },

    { /* 10101  */
      WSAEDISCON,
      "WSAEDISCON",
      "Graceful shutdown in progress.",
      "Returned by WSARecv and WSARecvFrom to indicate that the remote party has initiated a graceful shutdown sequence."
    },

    { /* 10102  */
      WSAENOMORE,
      "WSAENOMORE",
      "No more results.",
      "No more results can be returned by the WSALookupServiceNext function."
    },

    { /* 10103  */
      WSAECANCELLED,
      "WSAECANCELLED",
      "Call has been canceled.",
      "A call to the WSALookupServiceEnd function was made while this call was still processing. The call has been canceled."
    },

    { /* 10104  */
      WSAEINVALIDPROCTABLE,
      "WSAEINVALIDPROCTABLE",
      "Procedure call table is invalid.",
      "The service provider procedure call table is invalid. A service provider returned a bogus procedure table to Ws2_32.dll. This is usually caused by one or more of the function pointers being NULL."
    },

    { /* 10105  */
      WSAEINVALIDPROVIDER,
      "WSAEINVALIDPROVIDER",
      "Service provider is invalid.",
      "The requested service provider is invalid. This error is returned by the WSCGetProviderInfo and WSCGetProviderInfo32 functions if the protocol entry specified could not be found. This error is also returned if the service provider returned a version number other than 2.0."
    },

    { /* 10106  */
      WSAEPROVIDERFAILEDINIT,
      "WSAEPROVIDERFAILEDINIT",
      "Service provider failed to initialize.",
      "The requested service provider could not be loaded or initialized. This error is returned if either a service provider's DLL could not be loaded (LoadLibrary failed) or the provider's WSPStartup or NSPStartup function failed."
    },

    { /* 10107  */
      WSASYSCALLFAILURE,
      "WSASYSCALLFAILURE",
      "System call failure.",
      "A system call that should never fail has failed. This is a generic error code, returned under various conditions. "
      "  Returned when a system call that should never fail does fail. For example, if a call to WaitForMultipleEvents fails or one of the registry functions fails trying to manipulate the protocol/namespace catalogs."
      "  Returned when a provider does not return SUCCESS and does not provide an extended error code. Can indicate a service provider implementation error."
    },


    { /* 10108  */
      WSASERVICE_NOT_FOUND,
      "WSASERVICE_NOT_FOUND",
      "Service not found.",
      "No such service is known. The service cannot be found in the specified name space."
    },

    { /* 10109  */
      WSATYPE_NOT_FOUND,
      "WSATYPE_NOT_FOUND",
      "Class type not found.",
      "The specified class was not found."
    },

    { /* 10110  */
      WSA_E_NO_MORE,
      "WSA_E_NO_MORE",
      "No more results.",
      "No more results can be returned by the WSALookupServiceNext function."
    },

    { /* 10111  */
      WSA_E_CANCELLED,
      "WSA_E_CANCELLED",
      "Call was canceled.",
      "A call to the WSALookupServiceEnd function was made while this call was still processing. The call has been canceled."
    },

    { /* 10112  */
      WSAEREFUSED,
      "WSAEREFUSED",
      "Database query was refused.",
      "A database query failed because it was actively refused."
    },

    { /* 11001  */
      WSAHOST_NOT_FOUND,
      "WSAHOST_NOT_FOUND",
      "Host not found.",
      "No such host is known. The name is not an official host name or alias, or it cannot be found in the database(s) being queried. This error may also be returned for protocol and service queries, and means that the specified name could not be found in the relevant database."
    },

    { /* 11002  */
      WSATRY_AGAIN,
      "WSATRY_AGAIN",
      "Nonauthoritative host not found.",
      "This is usually a temporary error during host name resolution and means that the local server did not receive a response from an authoritative server. A retry at some time later may be successful."
    },

    { /* 11003  */
      WSANO_RECOVERY,
      "WSANO_RECOVERY",
      "This is a nonrecoverable error.",
      "This indicates that some sort of nonrecoverable error occurred during a database lookup. This may be because the database files (for example, BSD-compatible HOSTS, SERVICES, or PROTOCOLS files) could not be found, or a DNS request was returned by the server with a severe error."
    },

    { /* 11004  */
      WSANO_DATA,
      "WSANO_DATA",
      "Valid name, no data record of requested type.",
      "The requested name is valid and was found in the database, but it does not have the correct associated data being resolved for. The usual example for this is a host name-to-address translation attempt (using gethostbyname or WSAAsyncGetHostByName) which uses the DNS (Domain Name Server). An MX record is returned but no A recordindicating the host itself exists, but is not directly reachable."
    },

    { /* 11005  */
      WSA_QOS_RECEIVERS,
      "WSA_QOS_RECEIVERS",
      "QOS receivers.",
      "At least one QOS reserve has arrived."
    },

    { /* 11006  */
      WSA_QOS_SENDERS,
      "WSA_QOS_SENDERS",
      "QOS senders.",
      "At least one QOS send path has arrived."
    },

    { /* 11007  */
      WSA_QOS_NO_SENDERS,
      "WSA_QOS_NO_SENDERS",
      "No QOS senders.",
      "There are no QOS senders."
    },

    { /* 11008  */
      WSA_QOS_NO_RECEIVERS,
      "WSA_QOS_NO_RECEIVERS",
      "QOS no receivers.",
      "There are no QOS receivers."
    },

    { /* 11009  */
      WSA_QOS_REQUEST_CONFIRMED,
      "WSA_QOS_REQUEST_CONFIRMED",
      "QOS request confirmed.",
      "The QOS reserve request has been confirmed."
    },

    { /* 11010  */
      WSA_QOS_ADMISSION_FAILURE,
      "WSA_QOS_ADMISSION_FAILURE",
      "QOS admission error.",
      "A QOS error occurred due to lack of resources."
    },

    { /* 11011  */
      WSA_QOS_POLICY_FAILURE,
      "WSA_QOS_POLICY_FAILURE",
      "QOS policy failure.",
      "The QOS request was rejected because the policy system couldn't allocate the requested resource within the existing policy. "
    },

    { /* 11012  */
      WSA_QOS_BAD_STYLE,
      "WSA_QOS_BAD_STYLE",
      "QOS bad style.",
      "An unknown or conflicting QOS style was encountered."
    },

    { /* 11013  */
      WSA_QOS_BAD_OBJECT,
      "WSA_QOS_BAD_OBJECT",
      "QOS bad object.",
      "A problem was encountered with some part of the filterspec or the provider-specific buffer in general."
    },

    { /* 11014  */
      WSA_QOS_TRAFFIC_CTRL_ERROR,
      "WSA_QOS_TRAFFIC_CTRL_ERROR",
      "QOS traffic control error.",
      "An error with the underlying traffic control (TC) API as the generic QOS request was converted for local enforcement by the TC API. This could be due to an out of memory error or to an internal QOS provider error. "
    },

    { /* 11015  */
      WSA_QOS_GENERIC_ERROR,
      "WSA_QOS_GENERIC_ERROR",
      "QOS generic error.",
      "A general QOS error."
    },

    { /* 11016  */
      WSA_QOS_ESERVICETYPE,
      "WSA_QOS_ESERVICETYPE",
      "QOS service type error.",
      "An invalid or unrecognized service type was found in the QOS flowspec."
    },

    { /* 11017  */
      WSA_QOS_EFLOWSPEC,
      "WSA_QOS_EFLOWSPEC",
      "QOS flowspec error.",
      "An invalid or inconsistent flowspec was found in the QOS structure."
    },

    { /* 11018  */
      WSA_QOS_EPROVSPECBUF,
      "WSA_QOS_EPROVSPECBUF",
      "Invalid QOS provider buffer.",
      "An invalid QOS provider-specific buffer."
    },

    { /* 11019  */
      WSA_QOS_EFILTERSTYLE,
      "WSA_QOS_EFILTERSTYLE",
      "Invalid QOS filter style.",
      "An invalid QOS filter style was used."
    },

    { /* 11020  */
      WSA_QOS_EFILTERTYPE,
      "WSA_QOS_EFILTERTYPE",
      "Invalid QOS filter type.",
      "An invalid QOS filter type was used."
    },

    { /* 11021  */
      WSA_QOS_EFILTERCOUNT,
      "WSA_QOS_EFILTERCOUNT",
      "Incorrect QOS filter count.",
      "An incorrect number of QOS FILTERSPECs were specified in the FLOWDESCRIPTOR."
    },

    { /* 11022  */
      WSA_QOS_EOBJLENGTH,
      "WSA_QOS_EOBJLENGTH",
      "Invalid QOS object length.",
      "An object with an invalid ObjectLength field was specified in the QOS provider-specific buffer."
    },

    { /* 11023  */
      WSA_QOS_EFLOWCOUNT,
      "WSA_QOS_EFLOWCOUNT",
      "Incorrect QOS flow count.",
      "An incorrect number of flow descriptors was specified in the QOS structure."
    },

    { /* 11024  */
      WSA_QOS_EUNKOWNPSOBJ,
      "WSA_QOS_EUNKOWNPSOBJ",
      "Unrecognized QOS object.",
      "An unrecognized object was found in the QOS provider-specific buffer."
    },

    { /* 11025  */
      WSA_QOS_EPOLICYOBJ,
      "WSA_QOS_EPOLICYOBJ",
      "Invalid QOS policy object.",
      "An invalid policy object was found in the QOS provider-specific buffer."
    },

    { /* 11026  */
      WSA_QOS_EFLOWDESC,
      "WSA_QOS_EFLOWDESC",
      "Invalid QOS flow descriptor.",
      "An invalid QOS flow descriptor was found in the flow descriptor list."
    },

    { /* 11027  */
      WSA_QOS_EPSFLOWSPEC,
      "WSA_QOS_EPSFLOWSPEC",
      "Invalid QOS provider-specific flowspec.",
      "An invalid or inconsistent flowspec was found in the QOS provider-specific buffer."
    },

    { /* 11028  */
      WSA_QOS_EPSFILTERSPEC,
      "WSA_QOS_EPSFILTERSPEC",
      "Invalid QOS provider-specific filterspec.",
      "An invalid FILTERSPEC was found in the QOS provider-specific buffer."
    },

    { /* 11029  */
      WSA_QOS_ESDMODEOBJ,
      "WSA_QOS_ESDMODEOBJ",
      "Invalid QOS shape discard mode object.",
      "An invalid shape discard mode object was found in the QOS provider-specific buffer."
    },

    { /* 11030  */
      WSA_QOS_ESHAPERATEOBJ,
      "WSA_QOS_ESHAPERATEOBJ",
      "Invalid QOS shaping rate object.",
      "An invalid shaping rate object was found in the QOS provider-specific buffer."
    },

    { /* 11031  */
      WSA_QOS_RESERVED_PETYPE,
      "WSA_QOS_RESERVED_PETYPE",
      "Reserved policy QOS element type.",
      "A reserved policy element was found in the QOS provider-specific buffer."
    }
  };

  int i;

  for (i = 0; i < sizeof(WSA_Error_Desc) / sizeof(WSA_Error_Desc[1]); i++)
  {
    if (WSA_Error_Desc[i].WSA_Error_Code == WSA_Error_Code)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("WSA Error %d\n  %s\n  %s\n  %s\n\n"),
          WSA_Error_Desc[i].WSA_Error_Code,
          WSA_Error_Desc[i].WSA_Error_Code_Name,
          WSA_Error_Desc[i].short_desc,
          WSA_Error_Desc[i].long_desc
      );
    }
  }
}
