/******************************************************************************
* FILE PURPOSE: HOSTAPPEMU main.
*******************************************************************************
*
* FILE NAME: hostappemu_p2p.cpp
*
* DESCRIPTION:
*
* HISTORY:
* 05/14/2007    stephen smith           initial revision
*
* LIST OF FUNCTIONS:
*
* Copyright (c) 2010 Texas Instruments Inc.  All rights reserved.
******************************************************************************/
#include "stdafx.h"
#include <crtdbg.h>
#include <Winsock2.h>   // for ntoh, hton
#include <tchar.h>
#include <psapi.h>
#include <shlwapi.h>
#include <strsafe.h>

#include "os_fifo.h"
#include "hct_msg_def.h"
#include "crc16.h"

#include "hostappemu.h"
#include "hostappemu_data.h"
#include "hostappemu_sn.h"
#include "hostappemu_p2p.h"

CXferInformation m_XferInfo;

extern OS_FIFO HostAppEMU_EC_Task_MsgFifo;
extern int RPY_delay;

static UINT8 G3_NSDU_Handle = 0;

HANDLE hostappemu_p2p_SendFileThread = 0;
DWORD hostappemu_p2p_SendFileThreadID = 0;
DWORD WINAPI hostappemu_p2p_SendFileThreadProc(LPVOID lpParam);

static void OnUpdate_XferRxInfo(const CXferInformation *pXferInfo);
void XferUpdateRXPacketStat(CXferInformation *pXferInfo, short nLength);
void OnUpdate_XferRxTransfer(const CXferInformation *pXferInfo);
static void hostappemu_p2p_SendRPYAck(void);

static BOOL m_bUse_RPY;
HANDLE sema_sync_llc;

// file transfer support
static bool m_Xfer_bReceiving;
static char m_Xfer_Filename[_MAX_PATH];
static FILE *m_Xfer_File = (FILE *) NULL;
static DWORD m_Xfer_FileLength;
static DWORD m_Xfer_FileLength_Current;
static UINT m_Xfer_PacketNum;

static UINT m_FilePacketNum;
static UINT m_AckPacketNum;
static UINT16 m_nBytesRead;

//static CTime m_Xfer_RX_start_time;
void hostappemu_p2p_XferSendAbort(void);
void hostappemu_p2p_XferSendAbort_LLC(void);
void hostappemu_p2p_XferSendAbort_G3(void);
void hostappemu_p2p_XferSendAckStatus(DWORD PacketNum);
void hostappemu_p2p_XferSendAckStatus_LLC(DWORD PacketNum);
void hostappemu_p2p_XferSendAckStatus_G3(DWORD PacketNum);
void hostappemu_p2p_XferSendBadCRCStatus(DWORD PacketNum);
void hostappemu_p2p_XferSendBadCRCStatus_LLC(DWORD PacketNum);
void hostappemu_p2p_XferSendBadCRCStatus_G3(DWORD PacketNum);
void hostappemu_p2p_XferSendMissingPacketsStatus(DWORD PacketNum, UINT cntMissingPackets);
void hostappemu_p2p_XferSendMissingPacketsStatus_LLC(DWORD PacketNum, UINT cntMissingPackets);
void hostappemu_p2p_XferSendMissingPacketsStatus_G3(DWORD PacketNum, UINT cntMissingPackets);
static bool hostappemu_p2p_IncRename(char *filename);
bool hostappemu_p2p_SendFileStart(char *filename, FILE *fp);
bool hostappemu_p2p_SendFileEnd(void);
bool hostappemu_p2p_SendFileAbort(void);
bool hostappemu_p2p_SendFilePacket(FILE *fp);

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendRPYAck
*
* DESCRIPTION:
*       Send RPY ACK.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
static void hostappemu_p2p_SendRPYAck(void)
{
  if (m_bUse_RPY)
  {
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) NULL, 0, TRUE );
  }
}

/******************************************************************************
* FUNCTION NAME: OnUpdate_XferRxInfo
*
* DESCRIPTION:
*       Updates and displays RX transfer information.
*
* Return Value:      none
* Input Parameters:  CXferInformation *   pointer to xfer info block
* Output Parameters: none
******************************************************************************/
void OnUpdate_XferRxInfo(const CXferInformation *pXferInfo)
{
#if 0
  m_XferInfo.m_Xfer_TTL_RX_packets += pXferInfo->m_Xfer_TTL_RX_packets;
  m_XferInfo.m_Xfer_TTL_RX_bytes += pXferInfo->m_Xfer_TTL_RX_bytes;
  m_XferInfo.m_Xfer_TTL_RX_files += pXferInfo->m_Xfer_TTL_RX_files;
  m_XferInfo.m_Xfer_TTL_RX_messages += pXferInfo->m_Xfer_TTL_RX_messages;
  m_XferInfo.m_Xfer_TTL_RX_aborts += pXferInfo->m_Xfer_TTL_RX_aborts;
  m_XferInfo.m_Xfer_TTL_RX_crc_errors += pXferInfo->m_Xfer_TTL_RX_crc_errors;
  m_XferInfo.m_Xfer_TTL_RX_missing_packets += pXferInfo->m_Xfer_TTL_RX_missing_packets;

  printf("RX Transfer Totals\n");
  printf("==================\n");
  printf("Packets received...... %d\n", m_XferInfo.m_Xfer_TTL_RX_packets);
  printf("Bytes received........ %d\n", m_XferInfo.m_Xfer_TTL_RX_bytes);
  printf("Files received........ %d\n", m_XferInfo.m_Xfer_TTL_RX_files);
  printf("Messages received..... %d\n", m_XferInfo.m_Xfer_TTL_RX_messages);
  printf("Aborts................ %d\n", m_XferInfo.m_Xfer_TTL_RX_aborts);
  printf("CRC Errors............ %d\n", m_XferInfo.m_Xfer_TTL_RX_crc_errors);
  printf("Missing Packets....... %d\n", m_XferInfo.m_Xfer_TTL_RX_missing_packets);
#endif
}

/******************************************************************************
* FUNCTION NAME: XferUpdateRXPacketStat
*
* DESCRIPTION:
*       Updates and displays RX transfer information.
*
* Return Value:      none
* Input Parameters:  CXferInformation *   pointer to xfer info block
* Output Parameters: none
******************************************************************************/
void XferUpdateRXPacketStat(CXferInformation *pXferInfo, short nLength)
{
  CXferInformation XferInfo;

  if (pXferInfo)
  {
    pXferInfo->m_Xfer_RX_filename = m_Xfer_Filename;
    pXferInfo->m_Xfer_RX_packets = 1;
    pXferInfo->m_Xfer_RX_bytes = nLength;
//    pXferInfo->m_Xfer_RX_start_time = m_Xfer_RX_start_time;
    OnUpdate_XferRxTransfer(pXferInfo);
  }
  else
  {
    XferInfo.m_Xfer_RX_filename = m_Xfer_Filename;
    XferInfo.m_Xfer_RX_packets = 1;
    XferInfo.m_Xfer_RX_bytes = nLength;
//    XferInfo.m_Xfer_RX_start_time = m_Xfer_RX_start_time;
    OnUpdate_XferRxTransfer(&XferInfo);
  }
}

void OnUpdate_XferRxTransfer(const CXferInformation *pXferInfo)
{
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendTextMessage
*
* DESCRIPTION:
*       Initialization routine.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_SendTextMessage(UINT8 *pui8, UINT16 len)
{
  UINT8 *pmsg;
  int msglen;
  unsigned long flags;
  UINT16 ui16;

  sema_sync_llc = CreateSemaphore(NULL, 0, 1, NULL);

  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + sizeof(CPacketDataMessage_Header) + len;
  }
  else
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE + sizeof(CPacketDataMessage_Header) + len;
  }

  /* allocate the packet */
  pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    CPacketDataMessage_Header *pdmh;

    if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
    {
      /* set G3 DATA_TRANSFER header */
      ui16 = G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK;

      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags]) = ui16;

      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE];
    }
    else
    {
      /* set LLC DATA_TRANSFER header */
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Mode]) = HCT_MSG_DATA_TRANSFER_MODE_REQUEST;
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dst_LSAP_Src_LSAP]) = 0;
      memset(&pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dest_Addr], 0, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE_Dest_Addr);

      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE];
    }

    /* set text message header */
    pdmh->m_Signature = pdmh->Signature;
    pdmh->m_SubType = pdmh->SubType_Message;

    /* set text message payload */
    memcpy(&pdmh[1], pui8, len);

    /* send packet */
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Sending Text Message %*.*S.\n"), len, len, pui8);

    HostAppEMU_GetFlags(&flags);
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
      HostAppEMU_SendMsgToPLC(
          HCT_MSG_TYPE_DATA_TRANSFER,
          (UINT8 *) pmsg,
          msglen,
          FALSE
      );
    }
    else
    {
      HostAppEMU_SendSyncMsgToPLC(
          HCT_MSG_TYPE_DATA_TRANSFER,
          (UINT8 *) pmsg,
          msglen,
          (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE
      );

      DWORD dwWaitResult = WaitForSingleObject(sema_sync_llc, 1000 * 5 /*INFINITE*/);
      if (dwWaitResult != WAIT_OBJECT_0)
      {
        if (dwWaitResult == WAIT_TIMEOUT)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Waiting for LLC DATA_TRANSFER.Confirm!\n"), __FUNCTION__);
        }
      }
    }
  }
  else
  {
    ECA_OutOfMemory();
  }

  CloseHandle(sema_sync_llc);

  hostappemu_exitapp();
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_processPacket_DATA_TRANSFER
*
* DESCRIPTION:
*       Initialization routine..
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_processPacket_DATA_TRANSFER(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY)
{
  unsigned long flags;
  UINT8 *pui8;
  ECF_TASK_s ecf_task;
  const CPacketDataMessage_Header *pPacket;
  short nLength;

  pui8 = (UINT8 *) pmsg;

  HostAppEMU_GetFlags(&flags);
  if ( ((flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    UINT16 ui16;
    UINT16 lqi;
    UINT16 g3_flags;

    if (msg_len == HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Received G3 DATA_TRANSFER.Confirm.\n"), __FUNCTION__);
      HostAppEMU_ReceivedSyncMsg();

      ReleaseSemaphore(sema_sync_llc, 1, NULL);

      LocalFree(pmsg);
      return;
    }

    ui16 = *(UINT16 *) &pui8[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_OFFSET_LinkQualityIndicator_Flags];
    lqi = ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_LinkQualityIndicator_MASK;
    g3_flags = ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_FLAGS_MASK;

    pPacket = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_OFFSET_Data >> 1];
    nLength = (short) msg_len - HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_SIZE;
  }
  else
  {
    UINT8 Dst_LSAP;
    UINT8 Src_LSAP;
    UINT16 Dest_Addr[3];
    UINT16 Src_Addr[3];
    UINT16 Mode;

    Mode = *(UINT16 *) &pui8[HCT_MSG_DATA_TRANSFER_MODE_OFFSET];
    switch (Mode)
    {
    case HCT_MSG_DATA_TRANSFER_MODE_CONFIRM:
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Received LLC DATA_TRANSFER.Confirm.\n"), __FUNCTION__);
      HostAppEMU_ReceivedSyncMsg();

      ReleaseSemaphore(sema_sync_llc, 1, NULL);

      LocalFree(pmsg);
      return;

    case HCT_MSG_DATA_TRANSFER_MODE_INDICATION:
      break;

    default:
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unexpected LLC DATA_TRANSFER mode, discarding packet%d.\n"), __FUNCTION__, Mode);
      HostAppEMU_ReceivedSyncMsg();

      LocalFree(pmsg);
      return;
    }

    Dst_LSAP = pui8[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Dst_LSAP];
    Src_LSAP = pui8[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Src_LSAP];

    memcpy(&Dest_Addr, &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Dest_Addr >> 1], HCT_MSG_REQ_DATA_TRANSFER_INDICATION_SIZE_Dest_Addr);
    memcpy(&Src_Addr, &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Src_Addr >> 1], HCT_MSG_REQ_DATA_TRANSFER_INDICATION_SIZE_Src_Addr);

    pPacket = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Data >> 1];
    nLength = (short) msg_len - HCT_MSG_REQ_DATA_TRANSFER_INDICATION_SIZE;
  }

  CXferInformation XferInfo;

  memset(&XferInfo, 0, sizeof(XferInfo));

  if (pPacket->m_Signature == pPacket->Signature)
  {
    switch (pPacket->m_SubType)
    {
    case pPacket->SubType_Message:
      {
        /* update transfer information */
        XferInfo.m_Xfer_TTL_RX_packets = 1;
        XferInfo.m_Xfer_TTL_RX_bytes = msg_len;
        XferInfo.m_Xfer_TTL_RX_messages = 1;

        pui8 = ((UINT8 *) pPacket) + sizeof(*pPacket);
        nLength -= sizeof(*pPacket);

        OnUpdate_XferRxInfo(&XferInfo);

        /* display the received message */
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ALWAYS, TEXT("Text Message Received (len:%d): %*.*S\n"), nLength, nLength, nLength, pui8);
      }
      break;

    case pPacket->SubType_SendFileAck:
    case pPacket->SubType_SendFileStream:
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! Unsupported file transfer mode!\n"));
      break;

    case pPacket->SubType_SendFileStreamNoNACK:
      {
        CPacketSendFileMessage_Payload *pmsg = (CPacketSendFileMessage_Payload *) pPacket;

        if ((WORD) nLength < (sizeof(*pmsg) + pmsg->m_PacketLen))
        {
          if (pPacket->m_SubType != pPacket->SubType_SendFileStreamNoNACK)
          {
            hostappemu_p2p_XferSendBadCRCStatus(m_Xfer_PacketNum);
          }
          XferInfo.m_Xfer_RX_crc_errors = 1;
          sprintf(XferInfo.m_Xfer_RX_Last_Error, "Pkt #%d: Bad length %d, expected %d.",
            nLength, (sizeof(*pmsg) + pmsg->m_PacketLen));
          XferUpdateRXPacketStat(&XferInfo, 0);
          if (pPacket->m_SubType == pPacket->SubType_SendFileAck)
          {
            XferUpdateRXPacketStat((CXferInformation *) NULL, nLength + sizeof(CPacketAPEHeaderEx));
            break;
          }
        }
        else
        {
          WORD crc = pmsg->m_CRC;
          if (crc != 0)
          {
            pmsg->m_CRC = 0;
            WORD calccrc = (WORD) CRC16_BlockChecksum(pmsg, (int) nLength);
            if (crc != calccrc)
            {
              if (pPacket->m_SubType != pPacket->SubType_SendFileStreamNoNACK)
              {
                hostappemu_p2p_XferSendBadCRCStatus(pmsg->PacketNum_Min);
              }
              sprintf(XferInfo.m_Xfer_RX_Last_Error, "Pkt #%d: Bad CRC 0x%04x, expected 0x%04x (len: %d).",
                m_Xfer_PacketNum, calccrc, crc, nLength);
              XferInfo.m_Xfer_RX_crc_errors = 1;
              XferUpdateRXPacketStat(&XferInfo, 0);
              if (pPacket->m_SubType == pPacket->SubType_SendFileAck)
              {
                XferUpdateRXPacketStat((CXferInformation *) NULL, nLength + sizeof(CPacketAPEHeaderEx));
                break;
              }
            }
          }
        }

        switch (pmsg->m_PacketNum)
        {
        case pmsg->PacketNum_BOF:
          {
            if (m_Xfer_bReceiving)
            {
              fclose(m_Xfer_File);
              m_Xfer_File = (FILE *) NULL;

              hostappemu_p2p_IncRename(m_Xfer_Filename);

              XferInfo.m_Xfer_TTL_RX_aborts = 1;
              OnUpdate_XferRxInfo(&XferInfo);
            }
            else
            {
              m_Xfer_bReceiving = true;
            }

            char *pbuf = (char *) (pmsg + 1);

            m_Xfer_FileLength = *((DWORD *) pbuf);
            pbuf += sizeof(DWORD);

            CreateDirectory(TEXT(".\\Files"), NULL);
            sprintf(m_Xfer_Filename, ".\\Files\\%s", (LPCTSTR) pbuf);

            m_Xfer_File = fopen(m_Xfer_Filename, "w+b");
            HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("File Transfer started, receiving file %S\n"), m_Xfer_Filename);
            if (!m_Xfer_File)
            {
              sprintf(XferInfo.m_Xfer_RX_Last_Error, "Error creating file %S.", m_Xfer_Filename);
              HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
              hostappemu_p2p_XferSendAbort();

              HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
              break;
            }
            else if (pPacket->m_SubType == pPacket->SubType_SendFileAck)
            {
              hostappemu_p2p_XferSendAckStatus(pmsg->m_PacketNum);
            }
            m_Xfer_PacketNum = pmsg->PacketNum_Min;
            m_Xfer_FileLength_Current = 0;
//            m_Xfer_RX_start_time = CTime::GetCurrentTime();

            XferUpdateRXPacketStat(&XferInfo, nLength + sizeof(CPacketAPEHeaderEx));
          }
          break;

        case pmsg->PacketNum_EOF:
          if (m_Xfer_bReceiving)
          {
            HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("File Transfer ended.\n"));

            m_Xfer_bReceiving = false;

            fclose(m_Xfer_File);
            m_Xfer_File = (FILE *) NULL;

            if (m_Xfer_FileLength_Current != m_Xfer_FileLength)
            {
              hostappemu_p2p_IncRename(m_Xfer_Filename);
              XferInfo.m_Xfer_TTL_RX_aborts = 1;
              sprintf(XferInfo.m_Xfer_RX_Last_Error, "Bad file length %d, expected %d.",
                m_Xfer_FileLength_Current, m_Xfer_FileLength);
              HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
            }

            XferInfo.m_Xfer_TTL_RX_files = 1;
            XferInfo.m_Xfer_TTL_RX_bytes = nLength + sizeof(CPacketAPEHeaderEx);
            OnUpdate_XferRxInfo(&XferInfo);
          }

          if (pPacket->m_SubType == pPacket->SubType_SendFileAck)
          {
            hostappemu_p2p_XferSendAckStatus(pmsg->m_PacketNum);
          }

          m_Xfer_Filename[0] = (char) NULL;
          XferUpdateRXPacketStat(&XferInfo, 0);

          break;

        case pmsg->PacketNum_Abort:
          if (m_Xfer_bReceiving)
          {
            m_Xfer_bReceiving = false;

            fclose(m_Xfer_File);

            hostappemu_p2p_IncRename(m_Xfer_Filename);

            m_Xfer_Filename[0] = (char) NULL;
            XferUpdateRXPacketStat(&XferInfo, 0);
          }

          if (pPacket->m_SubType == pPacket->SubType_SendFileAck)
          {
            hostappemu_p2p_XferSendAckStatus(pmsg->m_PacketNum);
          }

          XferInfo.m_Xfer_TTL_RX_aborts = 1;
          XferInfo.m_Xfer_TTL_RX_bytes = nLength + sizeof(CPacketAPEHeaderEx);
          sprintf(XferInfo.m_Xfer_RX_Last_Error, "Abort from TX received.");
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
          OnUpdate_XferRxInfo(&XferInfo);

          break;

        default:
          {
            if (m_Xfer_bReceiving)
            {
              if (m_Xfer_PacketNum != pmsg->m_PacketNum)
              {
                if (m_Xfer_PacketNum < pmsg->m_PacketNum)
                {
                  XferInfo.m_Xfer_RX_missing_packets = pmsg->m_PacketNum - m_Xfer_PacketNum;
                }
                else if (pPacket->m_SubType == pPacket->SubType_SendFileAck)
                {
                  /* received same packet again, discard (our ACK was lost) */
                  hostappemu_p2p_XferSendAckStatus(pmsg->m_PacketNum);

                  sprintf(XferInfo.m_Xfer_RX_Last_Error, "Duplicate packet, expected %d, received %d.",
                      m_Xfer_PacketNum, pmsg->m_PacketNum);
                  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
                  XferInfo.m_Xfer_RX_missing_packets = m_Xfer_PacketNum - pmsg->m_PacketNum;
                  XferUpdateRXPacketStat(&XferInfo, 0);
                  break;
                }
                else
                {
                  XferInfo.m_Xfer_RX_missing_packets = m_Xfer_PacketNum - pmsg->m_PacketNum;
                }
                if (pPacket->m_SubType != pPacket->SubType_SendFileStreamNoNACK)
                {
                  hostappemu_p2p_XferSendMissingPacketsStatus(m_Xfer_PacketNum, XferInfo.m_Xfer_RX_missing_packets);
                }
                sprintf(XferInfo.m_Xfer_RX_Last_Error, "Missing packets, expected pkt #%d, received pkt #%d (ttl: %d).",
                    m_Xfer_PacketNum, pmsg->m_PacketNum, XferInfo.m_Xfer_RX_missing_packets);
                HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
                m_Xfer_PacketNum = pmsg->m_PacketNum;
              }

              int i = fwrite((char *) (pmsg + 1), sizeof(char), pmsg->m_PacketLen, m_Xfer_File);
              m_Xfer_FileLength_Current += pmsg->m_PacketLen;

              if (pPacket->m_SubType == pPacket->SubType_SendFileAck)
              {
                hostappemu_p2p_XferSendAckStatus(pmsg->m_PacketNum);
              }

              if (i != pmsg->m_PacketLen)
              {
                sprintf(XferInfo.m_Xfer_RX_Last_Error, "Error writing file.");
                HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
                hostappemu_p2p_XferSendAbort();
              }
            }
            else
            {
              sprintf(XferInfo.m_Xfer_RX_Last_Error, "Pkt but not receiving.");
              HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S\n"), XferInfo.m_Xfer_RX_Last_Error);
              hostappemu_p2p_XferSendAbort();
            }

            m_Xfer_PacketNum++;
            XferUpdateRXPacketStat(&XferInfo, nLength + sizeof(CPacketAPEHeaderEx));
          }
          break;
        }
      }
      break;

    case pPacket->SubType_SendFileStatus:
//      pMainFrame->SendMessage(WM_XFERFILESENDRCVPKT, nLength, (LPARAM) pPacket);
      break;

    default:
      break;
    }
  }

  LocalFree(pmsg);

  if (bRPY)
  {
    if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_P2P)
    {
      if (RPY_delay == 0)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Sending RPY to device.\n"), RPY_delay);
        HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) NULL, 0, FALSE);
      }
      else
      {
        ecf_task.task_type = ECA_TASK_SEND_RPY;
        ecf_task.u.task_param = 0;
        os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
      }
    }
  }

}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendFile
* DESCRIPTION:
*       Send file.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
bool hostappemu_p2p_SendFile(char *filename)
{
  hostappemu_p2p_SendFileThread = CreateThread(NULL, 0, hostappemu_p2p_SendFileThreadProc, filename, 0, &hostappemu_p2p_SendFileThreadID);
  if (!hostappemu_p2p_SendFileThread)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! CreateThread() failed\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
    return false;
  }
  return true;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendFileThreadProc
* DESCRIPTION:
*       Send file.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
DWORD WINAPI hostappemu_p2p_SendFileThreadProc(LPVOID lpParam)
{
  char *filename;
  FILE *fp;
  bool bStatus;

  filename = (char *) lpParam;
  sema_sync_llc = CreateSemaphore(NULL, 0, 1, NULL);

  fp = fopen(filename, "rb");
  if (!fp)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unable to open file %s\n"), filename);
    return false;
  }

  bStatus = hostappemu_p2p_SendFileStart(filename, fp);
  if (bStatus)
  {
    while (hostappemu_p2p_SendFilePacket(fp))
      ;
  }

  bStatus = hostappemu_p2p_SendFileEnd();

  fclose(fp);

  CloseHandle(sema_sync_llc);

  hostappemu_exitapp();
  return bStatus;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendAbort
*
* DESCRIPTION:
*       File transfer send abort.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendAbort(void)
{
  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    hostappemu_p2p_XferSendAbort_LLC();
  }
  else
  {
    hostappemu_p2p_XferSendAbort_G3();
  }
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendAbort_LLC
*
* DESCRIPTION:
*       File transfer send abort.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendAbort_LLC(void)
{
  CXferInformation XferInfo;
  CPacketSendFileMessageReply SendFileReplyMsg;
  unsigned long flags;

  SendFileReplyMsg.payload.payload_hdr.m_PacketNum = SendFileReplyMsg.payload.payload_hdr.PacketNum_Abort;
  SendFileReplyMsg.payload.payload_hdr.m_PacketLen = 0;
  SendFileReplyMsg.payload.payload_hdr.m_CRC = 0;
  SendFileReplyMsg.payload.Status = SendFileReplyMsg.payload.Status_Abort;

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) &SendFileReplyMsg, sizeof(SendFileReplyMsg),(flags & HostAppEMU_USE_RPY) ? TRUE : FALSE);

  if (m_Xfer_bReceiving)
  {
    m_Xfer_bReceiving = false;

    fclose(m_Xfer_File);
    m_Xfer_File = (FILE *) NULL;

    hostappemu_p2p_IncRename(m_Xfer_Filename);
  }

  m_Xfer_Filename[0] = (char) NULL;

  XferInfo.m_Xfer_RX_filename = "";
  XferInfo.m_Xfer_RX_packets = 1;
  XferInfo.m_Xfer_RX_bytes = sizeof(SendFileReplyMsg);
//  OnUpdate_XferRxTransfer(&XferInfo);

  XferInfo.m_Xfer_TTL_RX_aborts = 1;
  OnUpdate_XferRxInfo(&XferInfo);
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendAbort_G3
*
* DESCRIPTION:
*       File transfer send abort.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendAbort_G3(void)
{
  CXferInformation XferInfo;
  CPacketSendFileMessageReply_G3 SendFileReplyMsg;
  unsigned long flags;

  SendFileReplyMsg.g3_header.NSDU_handle = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

  SendFileReplyMsg.payload.payload_hdr.m_PacketNum = SendFileReplyMsg.payload.payload_hdr.PacketNum_Abort;
  SendFileReplyMsg.payload.payload_hdr.m_PacketLen = 0;
  SendFileReplyMsg.payload.payload_hdr.m_CRC = 0;
  SendFileReplyMsg.payload.Status = SendFileReplyMsg.payload.Status_Abort;

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) &SendFileReplyMsg, sizeof(SendFileReplyMsg),(flags & HostAppEMU_USE_RPY) ? TRUE : FALSE);

  if (m_Xfer_bReceiving)
  {
    m_Xfer_bReceiving = false;

    fclose(m_Xfer_File);
    m_Xfer_File = (FILE *) NULL;

    hostappemu_p2p_IncRename(m_Xfer_Filename);
  }

  m_Xfer_Filename[0] = (char) NULL;

  XferInfo.m_Xfer_RX_filename = "";
  XferInfo.m_Xfer_RX_packets = 1;
  XferInfo.m_Xfer_RX_bytes = sizeof(SendFileReplyMsg);
//  OnUpdate_XferRxTransfer(&XferInfo);

  XferInfo.m_Xfer_TTL_RX_aborts = 1;
  OnUpdate_XferRxInfo(&XferInfo);
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendAckStatus
*
* DESCRIPTION:
*       File transfer send ack.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendAckStatus(DWORD PacketNum)
{
  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    hostappemu_p2p_XferSendAckStatus_LLC(PacketNum);
  }
  else
  {
    hostappemu_p2p_XferSendAckStatus_G3(PacketNum);
  }
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendAckStatus_LLC
*
* DESCRIPTION:
*       File transfer send ack.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendAckStatus_LLC(DWORD PacketNum)
{
  unsigned long flags;
  CPacketSendFileMessageReply msg;

  msg.payload.payload_hdr.m_PacketNum = PacketNum;
  msg.payload.payload_hdr.m_PacketLen = 0;
  msg.payload.payload_hdr.m_CRC = 0;
  msg.payload.Status = msg.payload.Status_Success;

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(
      HCT_MSG_TYPE_DATA_TRANSFER,
      (UINT8 *) &msg,
      sizeof(msg),
      (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE
  );
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendAckStatus_G3
*
* DESCRIPTION:
*       File transfer send ack.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendAckStatus_G3(DWORD PacketNum)
{
  unsigned long flags;
  CPacketSendFileMessageReply_G3 msg;

  msg.g3_header.NSDU_handle = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

  msg.payload.payload_hdr.m_PacketNum = PacketNum;
  msg.payload.payload_hdr.m_PacketLen = 0;
  msg.payload.payload_hdr.m_CRC = 0;
  msg.payload.Status = msg.payload.Status_Success;

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(
      HCT_MSG_TYPE_DATA_TRANSFER,
      (UINT8 *) &msg,
      sizeof(msg),
      (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE
  );
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendBadCRCStatus
*
* DESCRIPTION:
*       File transfer send bad crc.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendBadCRCStatus(DWORD PacketNum)
{
  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    hostappemu_p2p_XferSendBadCRCStatus_LLC(PacketNum);
  }
  else
  {
    hostappemu_p2p_XferSendBadCRCStatus_G3(PacketNum);
  }
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendBadCRCStatus_LLC
*
* DESCRIPTION:
*       File transfer send bad crc.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendBadCRCStatus_LLC(DWORD PacketNum)
{
  unsigned long flags;
  CPacketSendFileMessageReply msg;

  msg.payload.payload_hdr.m_PacketNum = PacketNum;
  msg.payload.payload_hdr.m_PacketLen = 0;
  msg.payload.payload_hdr.m_CRC = 0;
  msg.payload.Status = msg.payload.Status_BadCrc;

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(
      HCT_MSG_TYPE_DATA_TRANSFER,
      (UINT8 *) &msg,
      sizeof(msg),
      (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE
  );
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendBadCRCStatus_G3
*
* DESCRIPTION:
*       File transfer send bad crc.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendBadCRCStatus_G3(DWORD PacketNum)
{
  unsigned long flags;
  CPacketSendFileMessageReply_G3 msg;

  msg.g3_header.NSDU_handle = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

  msg.payload.payload_hdr.m_PacketNum = PacketNum;
  msg.payload.payload_hdr.m_PacketLen = 0;
  msg.payload.payload_hdr.m_CRC = 0;
  msg.payload.Status = msg.payload.Status_BadCrc;

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(
      HCT_MSG_TYPE_DATA_TRANSFER,
      (UINT8 *) &msg,
      sizeof(msg),
      (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE
  );
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendMissingPacketsStatus
*
* DESCRIPTION:
*       File transfer send missing packet status.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendMissingPacketsStatus(DWORD PacketNum, UINT cntMissingPackets)
{
  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    hostappemu_p2p_XferSendMissingPacketsStatus_G3(PacketNum, cntMissingPackets);
  }
  else
  {
    hostappemu_p2p_XferSendMissingPacketsStatus_LLC(PacketNum, cntMissingPackets);
  }
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendMissingPacketsStatus_LLC
*
* DESCRIPTION:
*       File transfer send missing packet status.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendMissingPacketsStatus_LLC(DWORD PacketNum, UINT cntMissingPackets)
{
  unsigned long flags;
  CPacketSendFileMessageReply *pmsg;
  UINT msglen;
  char *pbuf;

  msglen = sizeof(*pmsg) + sizeof(WORD);
  pbuf = (char *) LocalAlloc(LPTR, ALIGN16(msglen));
  if (!pbuf)
  {
    ECA_OutOfMemory();
    return;
  }

  /* initialize the message header */
  pmsg = (CPacketSendFileMessageReply *) pbuf;
  memset(&pmsg->llc_header, 0, sizeof(pmsg->llc_header));

  pmsg->payload.payload_hdr.m_PacketNum = PacketNum;
  pmsg->payload.payload_hdr.m_PacketLen = 0;
  pmsg->payload.payload_hdr.m_CRC = 0;
  pmsg->payload.Status = pmsg->payload.Status_Missing;

  char *p = ((char *) pmsg) + sizeof(*pmsg);
  WORD w = (WORD) cntMissingPackets;
  memcpy(p, &w, sizeof(w));

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(
      HCT_MSG_TYPE_DATA_TRANSFER,
      (UINT8 *) pmsg,
      msglen,
      (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE
  );

  LocalFree(pbuf);
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_XferSendMissingPacketsStatus_G3
*
* DESCRIPTION:
*       File transfer send missing packet status.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if initialization fails.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
void hostappemu_p2p_XferSendMissingPacketsStatus_G3(DWORD PacketNum, UINT cntMissingPackets)
{
  unsigned long flags;
  CPacketSendFileMessageReply_G3 *pmsg;
  UINT msglen;
  char *pbuf;

  msglen = sizeof(*pmsg) + sizeof(WORD);
  pbuf = (char *) LocalAlloc(LPTR, ALIGN16(msglen));
  if (!pbuf)
  {
    ECA_OutOfMemory();
    return;
  }

  /* initialize the message header */
  pmsg = (CPacketSendFileMessageReply_G3 *) pbuf;
  pmsg->g3_header.NSDU_handle = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

  pmsg->payload.payload_hdr.m_PacketNum = PacketNum;
  pmsg->payload.payload_hdr.m_PacketLen = 0;
  pmsg->payload.payload_hdr.m_CRC = 0;
  pmsg->payload.Status = pmsg->payload.Status_Missing;

  char *p = ((char *) pmsg) + sizeof(*pmsg);
  WORD w = (WORD) cntMissingPackets;
  memcpy(p, &w, sizeof(w));

  HostAppEMU_GetFlags(&flags);
  HostAppEMU_SendSyncMsgToPLC(
      HCT_MSG_TYPE_DATA_TRANSFER,
      (UINT8 *) pmsg,
      msglen,
      (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE
  );

  LocalFree(pbuf);
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_IncRename
*
* DESCRIPTION:
*       Incremental rename file.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if error.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
bool hostappemu_p2p_IncRename(char *filename)
{
#if 0
  CString s;

  for (int i = 0; i <= 99; i++)
  {
    try
    {
      s.Format("%s.%d", filename, i);
      CFile::Rename(filename, s);
      return true;
    }
    catch (CFileException* pEx)
    {
      pEx->Delete();
    }
  }
#endif

#if 1
  remove(filename);
#endif
  return true;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendFileStart
*
* DESCRIPTION:
*       Send file transfer start.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if error.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
bool hostappemu_p2p_SendFileStart(char *filename, FILE *fp)
{
  extern char *HostAppEMU_SendFileName;
  UINT8 *pmsg;
  int msglen;
  unsigned long flags;
  ULONGLONG FileLength;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Sending file start...\n"));

  fseek(fp, 0, SEEK_END);
  FileLength = (ULONGLONG) ftell(fp);
  fseek(fp, 0, SEEK_SET);
  if (FileLength > ((ULONGLONG) ((CPacketSendFileMessage_Payload *) NULL)->PacketNum_Max) * ((ULONGLONG) HOSTAPPEMU_P2P_FILE_TRANSFER_PACKET_SIZE))
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("File length exceeds maximum length based on packet size, aborting!\n"));
    return false;
  }
  if (FileLength > (ULONGLONG) 0xffffffff)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("File length exceeds maximum, aborting!\n"));
    return false;
  }

  char sdrive[_MAX_DRIVE];
  char sdir[_MAX_DIR];
  char sfname[_MAX_FNAME];
  char sext[_MAX_EXT];

  _splitpath(filename, sdrive, sdir, sfname, sext);

  char fn[_MAX_FNAME + _MAX_EXT];
  sprintf(fn, "%s%s", sfname, sext);

  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + sizeof(CPacketSendFileMessage_Payload);
  }
  else
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE + sizeof(CPacketSendFileMessage_Payload);
  }

  msglen += sizeof(DWORD) + strlen(fn) + 1;

  /* allocate the packet */
  pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    CPacketDataMessage_Header *pdmh;

    if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
    {
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags]) = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

      /* set header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE];
    }
    else
    {
      /* set LLC DATA_TRANSFER header */
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Mode]) = HCT_MSG_DATA_TRANSFER_MODE_REQUEST;
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dst_LSAP_Src_LSAP]) = 0;
      memset(&pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dest_Addr], 0, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE_Dest_Addr);

      /* set file transfer protocol header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE];
    }
    pdmh->m_Signature = pdmh->Signature;
    pdmh->m_SubType = pdmh->SubType_SendFileStreamNoNACK;

    CPacketSendFileMessage_Payload *psfmp = (CPacketSendFileMessage_Payload *) pdmh;
    psfmp->m_PacketNum = ((CPacketSendFileMessage_Payload *) NULL)->PacketNum_BOF;
    psfmp->m_PacketLen = (WORD) msglen;

    /* set file transfer payload */
    UINT8 *ppayload = ((UINT8 *) psfmp) + sizeof(CPacketSendFileMessage_Payload);
    DWORD dw = (DWORD) FileLength;
    memcpy(ppayload, &dw, sizeof(DWORD));
    memcpy(ppayload + sizeof(DWORD), fn, strlen(fn) + 1);

    /* set file transfer protocol CRC */
    psfmp->m_CRC = 0;
    psfmp->m_CRC = (WORD) CRC16_BlockChecksum(pdmh, sizeof(CPacketSendFileMessage_Payload) + psfmp->m_PacketLen);

    /* send packet */
    HostAppEMU_GetFlags(&flags);
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
      HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg,msglen,FALSE);
    }
    else
    {
      HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg,msglen,(flags & HostAppEMU_USE_RPY) ? TRUE : FALSE);

      DWORD dwWaitResult = WaitForSingleObject(sema_sync_llc, 1000 * 5 /*INFINITE*/);
      if (dwWaitResult != WAIT_OBJECT_0)
      {
        if (dwWaitResult == WAIT_TIMEOUT)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Waiting for LLC DATA_TRANSFER.Confirm!\n"), __FUNCTION__);
        }
      }

    }
  }
  else
  {
    ECA_OutOfMemory();
  }

  m_FilePacketNum = ((CPacketSendFileMessage_Payload *) NULL)->PacketNum_Min;

  return true;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendFileEnd
*
* DESCRIPTION:
*       Send file transfer end.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if error.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
bool hostappemu_p2p_SendFileEnd(void)
{
  UINT8 *pmsg;
  int msglen;
  unsigned long flags;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Sending file end...\n"));

  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + sizeof(CPacketSendFileMessage_Payload);
  }
  else
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE + sizeof(CPacketSendFileMessage_Payload);
  }

  /* allocate the packet */
  pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    CPacketDataMessage_Header *pdmh;

    if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
    {
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags]) = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

      /* set header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE];
    }
    else
    {
      /* set LLC DATA_TRANSFER header */
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Mode]) = HCT_MSG_DATA_TRANSFER_MODE_REQUEST;
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dst_LSAP_Src_LSAP]) = 0;
      memset(&pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dest_Addr], 0, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE_Dest_Addr);

      /* set header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE];
    }
    pdmh->m_Signature = pdmh->Signature;
    pdmh->m_SubType = pdmh->SubType_SendFileStreamNoNACK;

    /* set file transfer protocol header */
    CPacketSendFileMessage_Payload *psfmp = (CPacketSendFileMessage_Payload *) pdmh;
    psfmp->m_PacketNum = ((CPacketSendFileMessage_Payload *) NULL)->PacketNum_EOF;
    psfmp->m_PacketLen = 0;

    /* set file transfer protocol CRC */
    psfmp->m_CRC = 0;
    psfmp->m_CRC = (WORD) CRC16_BlockChecksum(pdmh, sizeof(CPacketSendFileMessage_Payload) + psfmp->m_PacketLen);

    /* send packet */
    HostAppEMU_GetFlags(&flags);
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
      HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg, msglen,FALSE);
    }
    else
    {
      HostAppEMU_SendSyncMsgToPLC( HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg,msglen,(flags & HostAppEMU_USE_RPY) ? TRUE : FALSE);

      DWORD dwWaitResult = WaitForSingleObject(sema_sync_llc, 1000 * 5 /*INFINITE*/);
      if (dwWaitResult != WAIT_OBJECT_0)
      {
        if (dwWaitResult == WAIT_TIMEOUT)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Waiting for LLC DATA_TRANSFER.Confirm!\n"), __FUNCTION__);
        }
      }
    }
  }
  else
  {
    ECA_OutOfMemory();
  }

  return true;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendFileAbort
*
* DESCRIPTION:
*       Send file transfer end.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if error.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
bool hostappemu_p2p_SendFileAbort(void)
{
  UINT8 *pmsg;
  int msglen;
  unsigned long flags;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Sending file abort...\n"));

  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + sizeof(CPacketSendFileMessage_Payload);
  }
  else
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE + sizeof(CPacketSendFileMessage_Payload);
  }

  /* allocate the packet */
  pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    CPacketDataMessage_Header *pdmh;

    if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
    {
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags]) = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

      /* set header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE];
    }
    else
    {
      /* set LLC DATA_TRANSFER header */
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Mode]) = HCT_MSG_DATA_TRANSFER_MODE_REQUEST;
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dst_LSAP_Src_LSAP]) = 0;
      memset(&pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dest_Addr], 0, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE_Dest_Addr);

      /* set header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE];
    }
    pdmh->m_Signature = pdmh->Signature;
    pdmh->m_SubType = pdmh->SubType_SendFileStreamNoNACK;

    /* set file transfer protocol header */
    CPacketSendFileMessage_Payload *psfmp = (CPacketSendFileMessage_Payload *) pdmh;
    psfmp->m_PacketNum = ((CPacketSendFileMessage_Payload *) NULL)->PacketNum_Abort;
    psfmp->m_PacketLen = 0;

    /* set file transfer protocol CRC */
    psfmp->m_CRC = 0;
    psfmp->m_CRC = (WORD) CRC16_BlockChecksum(pdmh, sizeof(CPacketSendFileMessage_Payload) + psfmp->m_PacketLen);

    /* send packet */
    HostAppEMU_GetFlags(&flags);
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
			HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg,msglen,FALSE);
    }
    else
    {
      HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg, msglen, (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE);

      DWORD dwWaitResult = WaitForSingleObject(sema_sync_llc, 1000 * 5 /*INFINITE*/);
      if (dwWaitResult != WAIT_OBJECT_0)
      {
        if (dwWaitResult == WAIT_TIMEOUT)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Waiting for LLC DATA_TRANSFER.Confirm!\n"), __FUNCTION__);
        }
      }
    }
  }
  else
  {
    ECA_OutOfMemory();
  }

  return true;
}

/******************************************************************************
* FUNCTION NAME: hostappemu_p2p_SendFilePacket
*
* DESCRIPTION:
*       Send file transfer packet.
*
* Return Value:      BOOL        TRUE if it succeeds or FALSE if error.
* Input Parameters:  none
* Output Parameters: none
******************************************************************************/
bool hostappemu_p2p_SendFilePacket(FILE *fp)
{
  UINT8 *pmsg;
  int msglen;
  unsigned long flags;
  DWORD m_PayloadSizeMax;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Sending packet #%d\n"), m_FilePacketNum);
  if (m_FilePacketNum > ((CPacketSendFileMessage_Payload *) NULL)->PacketNum_Max)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Maximum packet count exceeded, aborting!\n"));
    return false;
  }

  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + sizeof(CPacketSendFileMessage_Payload);
    m_PayloadSizeMax = HOSTAPPEMU_P2P_FILE_TRANSFER_PACKET_SIZE - msglen;
  }
  else
  {
    msglen = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE + sizeof(CPacketSendFileMessage_Payload);
    m_PayloadSizeMax = HOSTAPPEMU_P2P_FILE_TRANSFER_PACKET_SIZE - msglen;
  }

  /* allocate the packet */
  pmsg = (UINT8 *) LocalAlloc(LPTR, msglen + m_PayloadSizeMax);
  if (pmsg)
  {
    CPacketDataMessage_Header *pdmh;

    if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
    {
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags]) = (G3_NSDU_Handle++ & HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_NSDU_HANDLE_MASK);

      /* set header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE];
    }
    else
    {
      /* set LLC DATA_TRANSFER header */
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Mode]) = HCT_MSG_DATA_TRANSFER_MODE_REQUEST;
      *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dst_LSAP_Src_LSAP]) = 0;
      memset(&pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dest_Addr], 0, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE_Dest_Addr);

      /* set header */
      pdmh = (CPacketDataMessage_Header *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE];
    }
    pdmh->m_Signature = pdmh->Signature;
    pdmh->m_SubType = pdmh->SubType_SendFileStreamNoNACK;

    /* set file transfer protocol header */
    CPacketSendFileMessage_Payload *psfmp = (CPacketSendFileMessage_Payload *) pdmh;
    psfmp->m_PacketNum = m_FilePacketNum++;

    /* set file transfer payload */
    UINT8 *ppayload = ((UINT8 *) psfmp) + sizeof(CPacketSendFileMessage_Payload);
    psfmp->m_PacketLen = fread(ppayload, sizeof(UINT8), m_PayloadSizeMax, fp);
    if (psfmp->m_PacketLen == 0)
    {
      if (!feof(fp))
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Error reading from file, aborting!\n"));
      }
      LocalFree(pmsg);
      return false;
    }

    /* set file transfer protocol CRC */
    psfmp->m_CRC = 0;
    psfmp->m_CRC = (WORD) CRC16_BlockChecksum(pdmh, sizeof(CPacketSendFileMessage_Payload) + psfmp->m_PacketLen);

    /* send packet */
    HostAppEMU_GetFlags(&flags);
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
      HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg, msglen + psfmp->m_PacketLen, FALSE );
    }
    else
    {
      HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER,(UINT8 *) pmsg, msglen + psfmp->m_PacketLen, (flags & HostAppEMU_USE_RPY) ? TRUE : FALSE);

      DWORD dwWaitResult = WaitForSingleObject(sema_sync_llc, 1000 * 5 /*INFINITE*/);
      if (dwWaitResult != WAIT_OBJECT_0)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Waiting for LLC DATA_TRANSFER.Confirm!\n"), __FUNCTION__);
      }
    }
  }
  else
  {
    ECA_OutOfMemory();
  }

  return true;
}
