/******************************************************************************
* FILE PURPOSE: Embedded Component Application.
*******************************************************************************
*
* FILE NAME: eca_appemu.c
*
* DESCRIPTION:
*       ECA AppEMU application.
*
* HISTORY:
* 07/07/2009  stephen smith   initial revision
* 12/08/2009  stephen smith   ported from PRIME project
*
* LIST OF FUNCTIONS:
*
* Copyright (c) 2009 Texas Instruments Inc.  All rights reserved.
******************************************************************************/
#define ECA_APPEMU_ALWAYS_REPLY     // always reply even when BN packet validation fails
#define ECA_APPEMU_HOST_APP         // host application

#include "stdafx.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <crtdbg.h>
#include <direct.h>
//#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <time.h>
#include <winsock2.h>

#define proj_assert   _ASSERT
#include "g3network.h"
#include "hostappemu.h"
#include "os_fifo.h"

#define PRIME
#define PRIME_IEC432
#include "diag_log.h"
#include "hct_msg_def.h"
#include "hct_msg_if_def.h"

#include "hostappemu_data.h"
#include "hostappemu_sn.h"
#include "crc32.h"
#include "eca_appemu_mgh.h"
#include "IPv6MainHeader.h"
#include "eca_appemu_mgh.h"
#include "tuntap.h"

extern int RPY_delay;
extern int g3_test_pktSize;
extern char * HostAppEMU_FirmwareFilename;
extern BOOL eMeterEmulation;
extern UINT16 g3LongAddress[];
extern UINT32 emeterDiscoveryTimeout;
extern UINT32 emeterPathDiscoveryInterval;
extern UINT32 g3ToneMaskSelection;
extern UINT32 g3Modulation;
extern UINT32 registrationDelay;
extern UINT32 connectionDelay;
extern UINT16 panId;

extern UINT32 tunTapIPv4Address;
extern char tunTapDriverName[];
extern TunTap * pTunTapDriver;
extern int microIP;

UINT32 pathDiscoveryInterval = 0;
UINT32 lastDiscoveryIndex = 0;
UINT32 emeterDiscoverNetworkPathNode = 0;

ECA_APPEMU_MGH ECA_AppEmu_mgh;

SINT16 PPDU_LengthTestingTable[PPDU_LENGTH_TESTING_OFFSET_COUNT][PPDU_LENGTH_TESTING_VALUES] = {
  { 18, 30, 66, 138, 210, 282, 354 },   // PPDU_LENGTH_TESTING_OFFSET_DBPSK_OFF
  { 17, 29, 65, 137, 209, 281, 353 },   // PPDU_LENGTH_TESTING_OFFSET_DBPSK_ON
  { 18, -1, 66, 138, 210, 282, 354 },   // PPDU_LENGTH_TESTING_OFFSET_DQPSK_OFF
  { 17, 29, 65, 137, 209, 281, 353 },   // PPDU_LENGTH_TESTING_OFFSET_DQPSK_ON
  { -1, 30, 66, 138, 210, 282, 354 },   // PPDU_LENGTH_TESTING_OFFSET_D8PSK_OFF
  { -1, 29, 65, 137, 209, 281, 353 }    // PPDU_LENGTH_TESTING_OFFSET_D8PSK_ON
};

BYTE ToneMasks[8][12] = 
	{
		{0x17, 0x24, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x00,0x00, 0x00, 0x00 },
		{0x17, 0x24, 0xFF, 0xFF, 0x00, 0xF8, 0x0F, 0x00, 0x00,0x00, 0x00, 0x00 },
		{0x3F, 0x10, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,0x00, 0x00, 0x00 },
		{0x3F, 0x1A, 0xFF, 0xFF, 0xFF, 0x03, 0x00, 0x00, 0x00,0x00, 0x00, 0x00 },
		{0x3F, 0x20, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00,0x00, 0x00, 0x00 },
		{0x1F, 0xA4, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x00,0x00, 0x00, 0x00 },
		{0x43, 0xA4, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0x00,0x00, 0x00, 0x00 },
		{0x1F, 0xC8, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00}
	};


/* local functions */
extern char *psrcaddr;
extern int udp_host_port;
extern int host_port;
extern int diag_port;

UINT16 deviceSerialNumberLength = 0;

void ECA_SetIPv4Config(void);
void ECA_ResetDevice(void);
bool ECA_SetSystemConfig(void);
void ECA_SetPhyTxParams(void);
void ECA_SetPhyRxParams(void);
void ECA_SetPhyTxParams_G3(void);
void ECA_SetPhyRxParams_G3(void);
int ECA_AppEmu_timerCallback(DWORD dwTime);
void ECA_AppEmu_timerCallback_handler(void);
ECF_STATUS ECA_AppEmu_ProcessTask(ECF_TASK_s *ptask);
void ECA_MLME_REGISTER_confirm_handler(APPEMU_status_t mac_status);
void ECA_MLME_UNREGISTER_confirm_handler(APPEMU_status_t mac_status);
void ECA_MLME_UNREGISTER_indicate_handler(void);
void ECA_AppEmu_Register(void);
void ECA_AppEmu_NETWORK_REGISTER_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen);
void ECA_AppEmu_NETWORK_START_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen);
void ECA_AppEmu_Discover(void);
void ECA_AppEmu_Network_Start(void);
BOOL ECA_GetSerialNumber(void);
void ECA_AppEmu_Set_Test_Length(void);
void ECA_AppEmu_Connect(void);
void ECA_AppEmu_Connect_MAC(void);
void ECA_AppEmu_Connect_LLC(void);
void ECA_AppEmu_Connect_G3(void);
void ECA_AppEmu_establishConnection(ConnHandle_t connHandle);
void ECA_CONNECT_confirm_MAC(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen, UINT16 status);
void ECA_CL_ESTABLISH_confirm(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen, UINT16 status);
void ECA_AppEmu_releaseConfirm(APPEMU_status_t mac_status);
void ECA_AppEmu_setNextStep(void);
void ECA_AppEmu_saveSerialNumber(UINT16 *pSerial);
void ECA_AppEmu_buildSerialNumber(UINT16 *pbuf);
void ECA_AppEMU_buildDeviceIdentifier(UINT8 *pbuf);
BOOL ECA_AppEmu_CheckPayload(HCT_MSG_BUFFER_t *msg, UINT16 len, UINT16 hdrlen);
void ECA_AppEmu_BuildPayload(HCT_MSG_BUFFER_t *msg, UINT16 len, UINT16 *ppkthdr, UINT16 hdrlen);
void ECA_AppEmu_UINT32_to_ZeroAscii(UINT32 v, UINT16 *pbuf, int length_octet);
void ECA_AppEmu_processPacket(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY);
void ECA_AppEmu_processSystemTimeRead(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr);
void ECA_AppEmu_processEnergyRegisterRead(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr);
void ECA_AppEmu_processAccumulatedInformationRead(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr);
void ECA_AppEmu_processLengthTesting(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr);

void ECA_AppEmu_CL_ESTABLISH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen);
void ECA_AppEmu_ATTACH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen);
void ECA_AppEmu_CL_RELEASE_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen);
void ECA_AppEmu_DETACH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen);

void ECA_AppEmu_DISCONNECT_indicate_MAC(UINT16 connHandle, UINT16 reason);
void ECA_AppEmu_DISCONNECT_indicate_LLC(UINT16 connHandle);
void ECA_AppEmu_DISCONNECT_indicate_G3(UINT16 connHandle);
void ECA_AppEmu_DISCONNECT_acknowledge_MAC(UINT16 connHandle);
void ECA_AppEmu_UNREGISTER(void);

void ECA_AppEmu_processPacket_MAC(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len);
void ECA_AppEmu_processPacket_IPv4(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len);
void ECA_AppEmu_processPacket_LLC(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY);
void ECA_AppEmu_processPacket_LLC_INDICATION(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len);
void ECA_AppEmu_processPacket_LLC_CONFIRM(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len);
void ECA_AppEmu_processPacket_G3(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY);
void ECA_AppEmu_processPacket_G3_INDICATION(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len);
void ECA_AppEmu_processPacket_G3_CONFIRM(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len);
void ECA_AppEmu_G3_SendPacket(void);

void ECA_AppEmu_Get_System_Info_Dump(HCT_MSG_BUFFER_t *msg, UINT16 msglen);
void ECA_AppEmu_ALARM_Handler_GENERAL_ALARM(UINT16 *Alarm_Data, UINT16 Alarm_Length);
const char *ECA_AppEmu_ALARM_TYPE_to_String(UINT16 AlarmType);

void ECA_AppEMU_CL_Release(UINT16 DestAddr);
void ECA_AppEMU_DETACH(UINT16 *pExtended_Address);
void ECA_AppEmu_DETACH_Confirm_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen);
void ECA_AppEmu_DETACH_Indicate_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen);


void SendFlashStatusPacket( UINT16 status );
void SendFlashPacket(void);
void SendBinaryFlashPacket(void);
UINT16 CheckSegmentCRC(UINT8 * pImage);

#ifdef APPEMU_IEC432
void ECA_AppEmu_Connect_CL_IEC432(void);
void ECA_CONNECT_confirm_CL_IEC432(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen, UINT16 status);
#endif // #ifdef APPEMU_IEC432

LPCTSTR eca_appemu_logger_strings[] = 
{
/* DIAG_LOG_200 */  TEXT("AppEMU: started.\n"),
/* DIAG_LOG_201 */  TEXT("AppEMU: Connection #%d started, serial # %02x.%02x.%02x.%02x.%02x.%02x.\n"),
/* DIAG_LOG_202 */  TEXT("AppEMU: Connection #%d time out (ttl:%d).\n"),
/* DIAG_LOG_203 */  TEXT("AppEMU: Registration started.\n"),
/* DIAG_LOG_204 */  TEXT("AppEMU: MAC Connect (MAC_EVT_ESTABLISH_CONFIRM cbStatus=%d)\n"),
/* DIAG_LOG_205 */  TEXT("AppEMU: Unexpected MAC_EVT_ESTABLISH_CONFIRM (ttl:%d).\n"),
/* DIAG_LOG_206 */  TEXT("AppEMU: MAC Disconnect Indicate (MAC_EVT_RELEASE_INDICATE).\n"),
/* DIAG_LOG_207 */  TEXT("AppEMU: Unexpected MAC_EVT_RELEASE_INDICATE (ttl:%d).\n"),
/* DIAG_LOG_208 */  TEXT("AppEMU: ERROR! MAC_CONN_release_respond() returned 0x%x.\n"),
/* DIAG_LOG_209 */  TEXT("AppEMU: Unexpected MAC_EVT_RELEASE_CONFIRM (ttl:%d).\n"),
/* DIAG_LOG_210 */  TEXT("AppEMU: Disconnect reason 0x%x received.\n"),
/* DIAG_LOG_211 */  TEXT("AppEMU: Registration complete.\n"),
/* DIAG_LOG_212 */  TEXT("AppEMU: Unexpected MAC_EVT_DATA_CONFIRM (ttl:%d).\n"),
/* DIAG_LOG_213 */  TEXT("AppEMU: MAC_EVT_DATA_INDICATE handle mismatch, expected 0x%x, received 0x%x (ttl:%d).\n"),
/* DIAG_LOG_214 */  TEXT("AppEMU: Connection #%d step %d.\n"),
/* DIAG_LOG_215 */  TEXT("AppEMU: Connection #%d step %d, length %d table[%d][%d].\n"),
/* DIAG_LOG_216 */  TEXT("AppEMU: Bad packet signature received (ttl:%d).\n"),
/* DIAG_LOG_217 */  TEXT("AppEMU: Bad packet provider received (ttl:%d).\n"),
/* DIAG_LOG_218 */  TEXT("AppEMU: Out of sequence packet, expected %d received %d (ttl:%d).\n"),
/* DIAG_LOG_219 */  TEXT("AppEMU: Invalid step %d received (ttl:%d).\n"),
/* DIAG_LOG_220 */  TEXT("AppEMU: Received payload mismatch Connection #%d step %d payload length %d (ttl:%d).\n"),
/* DIAG_LOG_221 */  TEXT("AppEMU: ERROR! MAC TX packet returned error 0x%02x (ttl:%d).\n"),
/* DIAG_LOG_222 */  TEXT("AppEMU: Unexpected MAC_EVT_DATA_INDICATE (ttl:%d).\n"),
/* DIAG_LOG_223 */  TEXT("AppEMU: ERROR! Out of memory! (ttl:%d).\n"),
/* DIAG_LOG_224 */  TEXT("AppEMU: Connection #%d step %d, received length %d, sending length %d.\n"),
/* DIAG_LOG_225 */  TEXT("AppEMU: PHY mode changed to %d.\n"),
/* DIAG_LOG_226 */  TEXT("AppEMU: Connection #%d step %d, length %d received, expected table index %d, index is %d.\n"),
/* DIAG_LOG_227 */  TEXT("AppEMU: Connection #%d step %d, length %d received, expected %d.\n"),
/* DIAG_LOG_228 */  TEXT("AppEMU: Connection failed, error 0x%x (ttl:%d).\n"),
/* DIAG_LOG_229 */  TEXT("\n"),
/* DIAG_LOG_230 */  TEXT("\n"),
/* DIAG_LOG_231 */  TEXT("AppEMU: ERROR! Registration failed, error 0x%02x.\n"),
/* DIAG_LOG_232 */  TEXT("AppEMU: Registration time out, waited %d seconds.\n"),
/* DIAG_LOG_233 */  TEXT("AppEMU: Unregistered.\n"),
/* DIAG_LOG_234 */  TEXT("AppEMU: Unexpected unregistration notification.\n"),
/* DIAG_LOG_235 */  TEXT("AppEMU: number free bd:%d.\n"),
/* DIAG_LOG_236 */  TEXT("AppEMU: Disconnect due to inactivity timeout %d (ttl:%d).\n"),
/* DIAG_LOG_237 */  TEXT("AppEMU: Disconnect complete.\n"),
/* DIAG_LOG_238 */  TEXT("AppEMU: Disconnect error 0x%x.\n"),
/* DIAG_LOG_239 */  TEXT("AppEMU: Unexpected disconnect complete notification.\n"),
/* DIAG_LOG_240 */  TEXT("AppEMU: Unexpected MAC event 0x%x (ttl:%d).\n"),
/* DIAG_LOG_241 */  TEXT("AppEMU: MAC_EVT_DATA_INDICATE error 0x%x (ttl:%d).\n"),
/* DIAG_LOG_242 */  TEXT("AppEMU: Disconnect handle 0x%x does not match connection handle %x (ttl:%d).\n"),
/* DIAG_LOG_243 */  TEXT("AppEMU: MAC_CM_connRelease returned error 0x%x.\n"),
/* DIAG_LOG_244 */  TEXT("AppEMU: NWM_Get(MAC_ATTR_ID_SERIAL_NUM) failed.\n"),
/* DIAG_LOG_245 */  TEXT("AppEMU: MAC_CONN_establish() returned error 0x%x.\n"),
/* DIAG_LOG_246 */  TEXT("AppEMU: IEC432 send data callback error 0x%x.\n"),
/* DIAG_LOG_247 */  TEXT("AppEMU: LLC send data callback error 0x%x.\n"),
/* DIAG_LOG_248 */  TEXT("AppEMU: Connection #%d step %d, length %d received, expected table index %d, length not found in table.\n"),
/* DIAG_LOG_249 */  TEXT("AppEMU: LLC Src Addr mismatch on data receive.  Expected 0x%04x, received 0x%04x (ttl:%d).\n"),
/* DIAG_LOG_250 */  TEXT("AppEMU: LLC Dest Addr mismatch on data receive.  Expected 0x%04x, received 0x%04x (ttl:%d).\n"),
/* DIAG_LOG_251 */  TEXT("AppEMU: LLC Src LSAP mismatch on data receive.  Expected 0x%04x, received 0x%04x (ttl:%d).\n"),
/* DIAG_LOG_252 */  TEXT("AppEMU: LLC Dest LSAP mismatch on data receive.  Expected 0x%04x, received 0x%04x (ttl:%d).\n"),

                    TEXT("")
};

extern OS_FIFO HostAppEMU_EC_Task_MsgFifo;
extern OS_FIFO_CONTROL HostAppEMU_EC_Task_MsgFifoControl;
extern ECF_TASK_s HostAppEMU_EC_Task_MsgFifoBuffer[HostAppEMU_MSG_FIFO_COUNT];
extern UINT32 TMR;

bool flashComplete = false;
UINT8 flashState = 0;
UINT8 macAddress[8] = {0,0,0,0,0,0,0,0};

UINT32 g3MessageCounter = 0;
UINT32 previousG3MessageCounter = 0xFFFF;  // make sure they don't equal the first time.
UINT8  g3HeartBeatCounter = 0;

UINT32 firmwareVersion = 0;
#define POLYNOMIAL 0x04C11DB7
typedef unsigned int crc;
#define WIDTH  (8 * sizeof(crc))
#define TOPBIT (1 << (WIDTH - 1))
UINT32 crcSlow(unsigned char const message[], int nBytes)
{
	UINT32 remainder = 0;
	int byte;
	/*
	 * Perform modulo-2 division, a byte at a time.
	 */
	for (byte = 0; byte < nBytes; ++byte)
	{
		unsigned char bit;
		/*
		 * Bring the next byte into the remainder.
		 */
		remainder ^= (message[byte] << (WIDTH - 8));
		/*
		 * Perform modulo-2 division, a bit at a time.
		 */
		for (bit = 8; bit > 0; --bit)
		{
			/*
			 * Try to divide the current data bit.
			 */

			if (remainder & TOPBIT)
			{
				remainder = (remainder << 1) ^ POLYNOMIAL;
			}
			else
			{
				remainder = (remainder << 1);
			}
		}
	}
	/*
	 * The final remainder is the CRC result.
	 */
	return (remainder);
}   /* crcSlow() */



void MyLocalFree(void *p)
{
  LocalFree(p);
}
bool ECA_Reinitialize(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  LPCTSTR pmode;
#ifdef DEBUG_SHU  
  UINT16 timeOutValue = 1000;
#else  
  UINT16 timeOutValue = 2000;
#endif

	switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
  case HostAppEMU_P2P:
    pmode = TEXT("P2P");
    break;
  case HostAppEMU_MAC:
    pmode = TEXT("MAC");
    break;
  case HostAppEMU_CL_LLC:
    pmode = TEXT("LLC");
    break;
  case HostAppEMU_IPv4:
    pmode = TEXT("IP");
    break;
  case HostAppEMU_G3_Node:
    pmode = TEXT("G3 Node");
    break;
  case HostAppEMU_G3_Conc:
    pmode = TEXT("G3 Concentrator");
    break;
  default:
    pmode = TEXT("unknown");
    break;
  }
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Running in %s mode.\n"), __FUNCTION__, pmode);
  /* get initial system configuration */
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Retrieving System Information from device...\n"));
  if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_GET_SYSTEM_INFO, NULL, 0, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
  {
    return false;
  }
  /* PHY parameters */
  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
		//
    ECA_AppEmu_mgh.flags |=  ECA_APPEMU_FLAGS_ROBO; // Robert asked this to be changed to ROBO from ECA_APPEMU_FLAGS_DBPSK;

		ECA_AppEmu_mgh.bToneMask = TRUE;
		ECA_AppEmu_mgh.modulation = g3Modulation;
		
		memcpy(ECA_AppEmu_mgh.ToneMask, ToneMasks[g3ToneMaskSelection], 12);
    ECA_AppEmu_mgh.phy_tx_level = 16;
    //debug shu
    ECA_AppEmu_mgh.discover_time = 1;
    //ECA_AppEmu_mgh.discover_time = 5;
    //debug shu
    //ECA_AppEmu_mgh.G3_Pan_ID = 0x0000;
    ECA_AppEmu_mgh.G3_Pan_ID = panId;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_AGC;
  }
  else
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DBPSK;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_FEC;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_AGC;
    ECA_AppEmu_mgh.phy_tx_level = 3;
  }

  /* MAC parameters */
  ECA_AppEmu_mgh.mac_cfp_bytes = 0;

  ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_None;

  ECA_AppEmu_mgh.PPDU_LengthTesting_Offset = PPDU_LENGTH_TESTING_OFFSET_COUNT;
  ECA_AppEmu_mgh.PPDU_LengthTesting_Offset_Save = 0;

 ECA_SetSystemConfig();
 ECA_ResetDevice();

 if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_GET_SYSTEM_INFO, NULL, 0, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
 {
   return false;
 }

  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    /* TODO: I think tonemask is broken.  Uncomment when working */
    ECA_SetPhyTxParams_G3();
    ECA_SetPhyRxParams_G3();
  }
  else
  {
    ECA_SetPhyTxParams();
    ECA_SetPhyRxParams();
  }

  if (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
    const int numAlarms = 12;
    int len;
    int offset;

    if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) != HostAppEMU_G3_Node) && ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) != HostAppEMU_G3_Conc) )
    {
      len = numAlarms * (HCT_MSG_REQ_SETUP_ALARM_SIZE + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE_Flags);
      pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, len);
      if (pmsg)
      {
        UINT8 *pui8;

        pui8 = (UINT8 *) pmsg;
        offset = 0;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_CONNECT_REQUEST;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_STARTED;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_COMPLETE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_CONNECT_COMPLETE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_PHY_RX_PPDU_CRC_FAIL;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_PHY_RX_HDR_SYNTAX_ERR;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_MAC_PROMOTED_SWITCH_NODE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_MAC_DEMOTED_SERVICE_NODE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_CL;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_GENERAL_ERROR;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_SETUP_ALARM, (UINT8 *) pmsg, len, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to setup alarms.\n"), __FUNCTION__);
        }
      }
      else
      {
        ECA_OutOfMemory();
      }
    }
  }

  DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_200, 0);

  return true;
}
/******************************************************************************
* FUNCTION NAME: ECA_initialize
*
* DESCRIPTION:
*       Initialization routine.  We set up our callbacks.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
bool ECA_initialize(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  unsigned long flags;
  UINT16 status;
  LPCTSTR pmode;
#ifdef DEBUG_SHU  
  UINT16 timeOutValue = 1000;
#else  
  UINT16 timeOutValue = 2000;
#endif

	switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
  case HostAppEMU_P2P:
    pmode = TEXT("P2P");
    break;
  case HostAppEMU_MAC:
    pmode = TEXT("MAC");
    break;
  case HostAppEMU_CL_LLC:
    pmode = TEXT("LLC");
    break;
  case HostAppEMU_IPv4:
    pmode = TEXT("IP");
    break;
  case HostAppEMU_G3_Node:
    pmode = TEXT("G3 Node");
    break;
  case HostAppEMU_G3_Conc:
    pmode = TEXT("G3 Concentrator");
    break;
  default:
    pmode = TEXT("unknown");
    break;
  }
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Running in %s mode.\n"), __FUNCTION__, pmode);

  /*
  ** if we're using IPv4, we need to set SYS_CFG IPv4 settings so the device can talk back to us
  **  note that we really only need to do this on Pre REV.D hardware or if we have not previously set
  **  up SYS_CFG.  Since I don't know what the state of SYS_CFG is and I can't just blindly do a
  **  LOAD_SYSTEM_CONFIG because if the IP address doesn't match I'll never get the reply, I'll
  **  just always do this.
  */
  if (psrcaddr)
  {
    ECA_SetIPv4Config();
  }

  /* get initial system configuration */
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Retrieving System Information from device...\n"));
  if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_GET_SYSTEM_INFO, NULL, 0, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
  {
    return false;
  }

  ECA_AppEmu_mgh.connHandle = INVALID_CONNECTION_HANDLE;

  /* PHY parameters */
  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
		//
    ECA_AppEmu_mgh.flags |=  ECA_APPEMU_FLAGS_ROBO; // Robert asked this to be changed to ROBO from ECA_APPEMU_FLAGS_DBPSK;

		ECA_AppEmu_mgh.bToneMask = TRUE;
		ECA_AppEmu_mgh.modulation = g3Modulation;
		
		memcpy(ECA_AppEmu_mgh.ToneMask, ToneMasks[g3ToneMaskSelection], 12);
    ECA_AppEmu_mgh.phy_tx_level = 16;
    //debug shu
    ECA_AppEmu_mgh.discover_time = 1;
    //ECA_AppEmu_mgh.discover_time = 5;
    //debug shu
    //ECA_AppEmu_mgh.G3_Pan_ID = 0x0000;
    ECA_AppEmu_mgh.G3_Pan_ID = panId;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_AGC;
  }
  else
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DBPSK;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_FEC;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_AGC;
    ECA_AppEmu_mgh.phy_tx_level = 3;
  }

  /* MAC parameters */
  ECA_AppEmu_mgh.mac_cfp_bytes = 0;

  ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_None;

  ECA_AppEmu_mgh.PPDU_LengthTesting_Offset = PPDU_LENGTH_TESTING_OFFSET_COUNT;
  ECA_AppEmu_mgh.PPDU_LengthTesting_Offset_Save = 0;

  status = os_fifo_open(&HostAppEMU_EC_Task_MsgFifo, &HostAppEMU_EC_Task_MsgFifoControl, &HostAppEMU_EC_Task_MsgFifoBuffer[0]);
  if (status != OS_STATUS_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! os_fifo_open() failed, error %#x\n"), __FUNCTION__, status);
  }

  HostAppEMU_TimerStart((HostAppEMU_TIMER_CALLBACK) ECA_AppEmu_timerCallback, timeOutValue, FALSE);

  BOOL bSysCfgNeeded = TRUE;  // Changed to always send the config info.. RAF - 2012-08-22

  switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
  case HostAppEMU_MAC:
    if (ECA_AppEmu_mgh.save_Device_Mode != HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_MAC)
    {
      bSysCfgNeeded = TRUE;
    }
    break;
  case HostAppEMU_P2P:
    if (ECA_AppEmu_mgh.save_Device_Mode != HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_POINT_TO_POINT)
    {
      bSysCfgNeeded = TRUE;
    }
    break;
  case HostAppEMU_CL_LLC:
  case HostAppEMU_IPv4:
    if (ECA_AppEmu_mgh.save_Device_Mode != HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_NORMAL)
    {
      bSysCfgNeeded = TRUE;
    }
    break;
  case HostAppEMU_G3_Node:
  case HostAppEMU_G3_Conc:
    //force G3 to load new parameters
    bSysCfgNeeded = TRUE;
    break;

  default:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("INTERNAL ERROR! Unknown device mode, aborting!\n"));
    return false;
  }

  if (bSysCfgNeeded)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Device Mode needs to be changed.\n"));
  }

  if (ECA_AppEmu_mgh.save_HostPort != host_port)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Host port needs to be changed from %d to %d.\n"), ECA_AppEmu_mgh.save_HostPort, host_port);
    bSysCfgNeeded = TRUE;
  }
	if (ECA_AppEmu_mgh.save_DiagPort != diag_port)
	{
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Diag port needs to be changed from %d to %d.\n"), ECA_AppEmu_mgh.save_DiagPort, diag_port);
    bSysCfgNeeded = TRUE;
	}
  if (!(bSysCfgNeeded))
  {
    HostAppEMU_GetFlags(&flags);

    if ( (!(bSysCfgNeeded)) && (flags & HostAppEMU_USE_RPY) && (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY)))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Use RPY specified, but device needs to be configured to Use RPY.\n"));
      bSysCfgNeeded = TRUE;
    }
    if ( (!(bSysCfgNeeded)) &&  (!(flags & HostAppEMU_USE_RPY)) && (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Use RPY was not specified, but device is configured to Use RPY.\n"));
      bSysCfgNeeded = TRUE;
    }

    if ( (!(bSysCfgNeeded)) && (flags & HostAppEMU_IPv6) && (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_IPv6)))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("IPv6 specified, but device needs to be configured to IPv6.\n"));
      bSysCfgNeeded = TRUE;
    }
    if ( (!(bSysCfgNeeded)) &&  (!(flags & HostAppEMU_IPv6)) && (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_IPv6))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("IPv6 was not specified, but device is configured to IPv6.\n"));
      bSysCfgNeeded = TRUE;
    }

    if ( (!(bSysCfgNeeded)) && (flags & HostAppEMU_LLC_AUTO) && (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_LLC_AUTO)))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("LLC Auto specified, but device needs to be configured to LLC Auto.\n"));
      bSysCfgNeeded = TRUE;
    }
    if ( (!(bSysCfgNeeded)) &&  (!(flags & HostAppEMU_LLC_AUTO)) && (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_LLC_AUTO))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("LLC Auto was not specified, but device is configured to LLC Auto.\n"));
      bSysCfgNeeded = TRUE;
    }
  }

  if (bSysCfgNeeded)
  {
    ECA_SetSystemConfig();
    ECA_ResetDevice();

    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_GET_SYSTEM_INFO, NULL, 0, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
      return false;
    }
  }

  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    /* TODO: I think tonemask is broken.  Uncomment when working */
    ECA_SetPhyTxParams_G3();
    ECA_SetPhyRxParams_G3();
  }
  else
  {
    ECA_SetPhyTxParams();
    ECA_SetPhyRxParams();
  }

  if (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
    const int numAlarms = 12;
    int len;
    int offset;

    if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) != HostAppEMU_G3_Node) && ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) != HostAppEMU_G3_Conc) )
    {
      len = numAlarms * (HCT_MSG_REQ_SETUP_ALARM_SIZE + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE_Flags);
      pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, len);
      if (pmsg)
      {
        UINT8 *pui8;

        pui8 = (UINT8 *) pmsg;
        offset = 0;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_CONNECT_REQUEST;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_STARTED;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_COMPLETE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_CONNECT_COMPLETE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_PHY_RX_PPDU_CRC_FAIL;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_PHY_RX_HDR_SYNTAX_ERR;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_MAC_PROMOTED_SWITCH_NODE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_MAC_DEMOTED_SERVICE_NODE;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_CL;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Type)]) = HCT_MSG_ALARM_TYPE_GENERAL_ERROR;
        *((UINT16 *) &pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_OFFSET_Alarm_Length)]) = HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;
        offset += HCT_MSG_REQ_SETUP_ALARM_SIZE;
        pui8[(offset + HCT_MSG_REQ_SETUP_ALARM_DEFAULT_OFFSET_Flags)] = 0; // ~HCT_MSG_ALARM_DEFAULT_FLAG_CLEAR
        offset += HCT_MSG_REQ_SETUP_ALARM_DEFAULT_SIZE;

        if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_SETUP_ALARM, (UINT8 *) pmsg, len, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to setup alarms.\n"), __FUNCTION__);
        }
      }
      else
      {
        ECA_OutOfMemory();
      }
    }
  }

  DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_200, 0);

  return true;
}
/******************************************************************************
* FUNCTION NAME: HostAppEMU_RemoteStart
*
* DESCRIPTION:
*     Resets the device.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_RemoteRestart(void)
{
	//
	// Create thread to do the remote reset
	//
	CreateThread(NULL, 0, HostAppEMU_RemoteReset, 0, 0, NULL);
}
/******************************************************************************
* FUNCTION NAME: ECA_ResetDevice
*
* DESCRIPTION:
*     Resets the device.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_ResetDevice(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  int payload_len;

  payload_len = HCT_MSG_REQ_SHUT_DOWN_SIZE;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, payload_len);
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SHUT_DOWN_OFFSET_Reset_Type >> 1]) = HCT_MSG_REQ_SHUT_DOWN_RESET_TYPE_SOFT_RESET;

    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_SHUT_DOWN, (UINT8 *) pmsg, HCT_MSG_REQ_SHUT_DOWN_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to Reset Device.\n"), __FUNCTION__);
    }
    else
    {
      /* give the device enough time to reset */
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Pausing to allow device to reset.\n"), __FUNCTION__);
      Sleep(2000);
    }
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to Reset Device.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_SetIPv4Config
*
* DESCRIPTION:
*     Set system configuration (example on using LOAD_SYSTEM_CONFIG)
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_SetIPv4Config(void)
{
  extern UINT32 HostAppEMU_IPv4Addr_to_UINT32(char *paddr);
  HCT_MSG_BUFFER_t *pmsg;
  int payload_len;
  UINT32 ui32;
  UINT16 Flags;

  ui32 = HostAppEMU_IPv4Addr_to_UINT32(psrcaddr);

  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Setting SYS_CFG IPV4 Source Address to %d.%d.%d.%d.\n"),
    (ui32 >> 24) & 0xff,
    (ui32 >> 16) & 0xff,
    (ui32 >> 8) & 0xff,
    (ui32) & 0xff
  );

  payload_len = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_SIZE;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, payload_len);
  if (pmsg)
  {
    int offset = 0;

    /* HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG */
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Type) >> 1]) = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG;
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Length) >> 1]) = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_SIZE;

    offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE;

    *((UINT32 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_OFFSET_SourceAddress) >> 1]) = ui32;
    *((UINT32 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_OFFSET_NetMask) >> 1]) = 0xffffff00;
    *((UINT32 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_OFFSET_Gateway) >> 1]) = (ui32 & 0xffffff00) | 0x000000fe;
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_OFFSET_ControlPort) >> 1]) = udp_host_port;

    Flags = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_FLAGS_IP_FLAG;
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_OFFSET_Flags) >> 1]) = Flags;

    offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_SIZE;

    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_LOAD_SYSTEM_CONFIG, (UINT8 *) pmsg, offset, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to Set IPv4 Config.\n"), __FUNCTION__);
    }
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to Set IPv4 Config.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_SetSystemConfig
*
* DESCRIPTION:
*     Set system configuration (example on using LOAD_SYSTEM_CONFIG)
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
bool ECA_SetSystemConfig(void)
{
  bool status = true;
  UINT8 *pmsg;
  UINT8 ui8;
  UINT8 mode;
  int payload_len;
  unsigned long flags;

  HostAppEMU_GetFlags(&flags);

  payload_len = 0
      + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_SIZE
      + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_SIZE
#if 0
      + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PHY_CONFIG_SIZE
      + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_MAC_CONFIG_SIZE
      + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_CL_IP_CONFIG_SIZE
#endif
  ;
	if (eMeterEmulation || tunTapIPv4Address != 0) // xraf -may want to add -g option also...
	{
		payload_len += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_SIZE;
	}

  pmsg = (UINT8 *) LocalAlloc(LPTR, payload_len);
  if (pmsg)
  {
    /* TODO: create TLV helper functions */
    int offset = 0;

    /* HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION */
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Type)]) = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION;
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Length)]) = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_SIZE;

    offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE;

    // Port 0 Designation (HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_SIZE)
    ui8 = 0;
    ui8 |= HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_SET_DATA_PORT(host_port);

    //add for g3, force the diagPort to be different, using diag_b
    //if (ECA_AppEmu_mgh.save_DiagPort == ECA_AppEmu_mgh.save_HostPort)
    //  ECA_AppEmu_mgh.save_DiagPort = 1;


    ui8 |= HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_SET_DIAG_PORT(diag_port);
    memcpy(&pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_OFFSET_Ports)], &ui8, HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_SIZE);

    offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_PORT_DESIGNATION_SIZE;

    /* HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG */
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Type)]) = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG;
    *((UINT16 *) &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Length)]) = HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_SIZE;

    offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE;

    memcpy(
        &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_OFFSET_SerialNumberLen)],
        &ECA_AppEmu_mgh.save_Serial_Number_Len,
        HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_SIZE_SerialNumberLen
    );
    memcpy(
        &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_OFFSET_SerialNumber)],
        &ECA_AppEmu_mgh.save_Serial_Number[0],
        HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_SIZE_SerialNumber
    );
    memcpy(
        &pmsg[(offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_OFFSET_EUI)],
        &ECA_AppEmu_mgh.save_EUI,
        HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_SIZE_EUI
    );

    switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
    {
    case HostAppEMU_P2P:
      mode = HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_POINT_TO_POINT;
      break;
    case HostAppEMU_CL_LLC:
    case HostAppEMU_IPv4:
      mode = HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_NORMAL;
			break;
		case HostAppEMU_G3_Node:
    case HostAppEMU_G3_Conc:
      mode = HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_NORMAL;
			if (microIP == 1)
			{
				mode = HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_mIP;
			}
      break;
    case HostAppEMU_MAC:
    default:
      mode = HCT_MSG_LOAD_SYSTEM_CONFIG_DEVICE_MODE_MAC;
      break;
    }

    *((UINT16 *) &pmsg[offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_OFFSET_DeviceMode_Flags])
        = (mode & HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_DeviceMode_Flags_DeviceMode_MASK);
    *((UINT16 *) &pmsg[offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_OFFSET_DeviceMode_Flags])
        |= (flags & HostAppEMU_USE_RPY) ? HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_DeviceMode_Flags_USE_RPY : 0;
    *((UINT16 *) &pmsg[offset + HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_OFFSET_DeviceMode_Flags])
        |= (flags & HostAppEMU_LLC_AUTO) ? HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_DeviceMode_Flags_AUTO_MODE : 0;

    offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_SYSTEM_CONFIG_SIZE;
		//
		// Set up the concentrator..
		//
		if ((tunTapIPv4Address != 0 || eMeterEmulation) &&((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc))
		{
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Type])
				= (UINT16)HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG;
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Length])
				= (UINT16)HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_SIZE;
			
			offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE;
			pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Address_Type] = 0;

			pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Address_Type+1] = 0x10; // set bit 0 ...

			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Short_Address])
				= 0;
	
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Pan_ID])
				= (UINT16)ECA_AppEmu_mgh.G3_Pan_ID;
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_MAC_Segment_Length])
				= (UINT16)ECA_AppEmu_mgh.G3_MAC_Segment_Length;
			
			//if (tunTapIPv4Address != 0)
			//{
			//	//
			//	// Concentrator address is always the pan id and 0 = 0x77.0x55.0x00.0x00
			//	//
			//	UINT16 shortAddress = (UINT16) _rotr(tunTapIPv4Address, 16) & 0x0000FFFF;
			//	shortAddress = _rotl((shortAddress & 0x00FF), 8) | _rotr((shortAddress & 0xFF00), 8) ;
			//	IPv6Address ipAddress(ECA_AppEmu_mgh.G3_Pan_ID, shortAddress);
			//  memcpy(&pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Long_Address], &ipAddress.m_IpAddress[4], 8);
  	  //	memcpy(macAddress, &ipAddress.m_IpAddress[4], 8);
			//}
      if (g3LongAddress[0] != 0)
      {
        //
        // Use the user supplied address..
        //
        memcpy(&pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Long_Address], g3LongAddress, 8);
  			memcpy(macAddress, g3LongAddress, 8);
      }
      else
      {
			  //
			  // Build a unique MAC address = PAN id xxxxx Random number
			  //
			  IPv6Address ipAddress(ECA_AppEmu_mgh.G3_Pan_ID, (UINT16)rand());
			  memcpy(&pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Long_Address], &ipAddress.m_IpAddress[4], 8);
  			memcpy(macAddress, &ipAddress.m_IpAddress[4], 8);
      }

			offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_SIZE;
						
		}
		//
		// Setup the service node
		// If using tuntap the initialize the ipv6 Address
		//
		if ((tunTapIPv4Address != 0 || eMeterEmulation) && ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node))
		{
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Type])
				= (UINT16)HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG;
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_OFFSET_Config_Length])
				= (UINT16)HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_SIZE;
			
			offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_SIZE;
			pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Address_Type] = 0;

			pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Address_Type+1] = 0x00; // set bit 0 ...

			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Short_Address])
				= 0;
	
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Pan_ID])
				= (UINT16)0;
			*((UINT16 *) &pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_MAC_Segment_Length])
				= (UINT16)ECA_AppEmu_mgh.G3_MAC_Segment_Length;
			//
			// Build a unquie MAC address = PAN id xxxxx Random number
			//
			//if (tunTapIPv4Address != 0)
			//{
			//	//
			//	// Service node address is the short address and the pan id: 0x77.0x55.0x00.0x64
			//	//
			//	UINT16 shortAddress = (UINT16) _rotr(tunTapIPv4Address, 16) & 0x0000FFFF;
			//	shortAddress = _rotl((shortAddress & 0x00FF), 8) | _rotr((shortAddress & 0xFF00), 8) ;
			//	IPv6Address ipAddress(ECA_AppEmu_mgh.G3_Pan_ID, shortAddress);
			//  memcpy(&pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Long_Address], &ipAddress.m_IpAddress[4], 8);
  	  //	memcpy(macAddress, &ipAddress.m_IpAddress[4], 8);
			//}
      if (g3LongAddress[0] != 0)
      {
        //
        // Use the user supplied address..
        //
        memcpy(&pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Long_Address], g3LongAddress, 8);
  			memcpy(macAddress, g3LongAddress, 8);
			}
      else
      {
			  //
			  // Build a unquie MAC address = PAN id xxxxx Random number
			  //
			  IPv6Address ipAddress(ECA_AppEmu_mgh.G3_Pan_ID, (UINT16)rand());
			  memcpy(&pmsg[offset+HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_OFFSET_G3_Long_Address], &ipAddress.m_IpAddress[4], 8);
  			memcpy(macAddress, &ipAddress.m_IpAddress[4], 8);
      }

			offset += HCT_MSG_REQ_LOAD_SYSTEM_CONFIG_TYPE_G3_CONFIG_SIZE;
						
		}
    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_LOAD_SYSTEM_CONFIG, (UINT8 *) pmsg, offset, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
	  status = false;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to Set System Config.\n"), __FUNCTION__);
    }
  }
  else
  {
    ECA_OutOfMemory();
	status = false;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to Set System Config.\n"), __FUNCTION__);
  }
  return status;
}

/******************************************************************************
* FUNCTION NAME: ECA_SetPhyTxParams
*
* DESCRIPTION:
*     Set PHY TX Parameters.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_SetPhyTxParams(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 *pui16;

  /* set configuration PHY TX Parameters */
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, (HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_SIZE));
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Type >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS;
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Length >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_SIZE;

    pui16 = ((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Value >> 1]);

    *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_OFFSET_Flags >> 1]) = 0;
    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_FEC)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_OFFSET_Flags >> 1]) |= HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_FLAG_FEC;
    }

    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DBPSK)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_OFFSET_Modulation >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_MODULATION_DBPSK;
    }
    else if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DQPSK)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_OFFSET_Modulation >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_MODULATION_DQPSK;
    }
    else if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DBPSK)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_OFFSET_Modulation >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_MODULATION_D8PSK;
    }
    else
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("%S: Modulation not specified, setting to DBPSK.\n"), __FUNCTION__);
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_OFFSET_Modulation >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_MODULATION_DBPSK;
    }

    *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_OFFSET_Level >> 1]) = ECA_AppEmu_mgh.phy_tx_level;

    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_SET_INFO, (UINT8 *) pmsg, HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_TX_PARAMS_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to set PHY TX Parameters.\n"), __FUNCTION__);
    }
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to setup PHY TX Parameters.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_SetPhyRxParams
*
* DESCRIPTION:
*     Set PHY TX Parameters.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_SetPhyRxParams(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 *pui16;

  /* set configuration PHY RX Parameters */
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, (HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_SIZE));
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Type >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS;
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Length >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_SIZE;

    pui16 = ((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Value >> 1]);

    *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_OFFSET_Flags >> 1]) = 0;
    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_AGC)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_OFFSET_Flags >> 1]) |= HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_FLAG_AGC;
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_OFFSET_Gain_Value >> 1]) = 0;
    }
    else
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_OFFSET_Gain_Value >> 1]) = ECA_AppEmu_mgh.gain_value;
    }
    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_ROBO)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_OFFSET_Flags >> 1]) |= HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_FLAG_ROBO;
    }

    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_SET_INFO, (UINT8 *) pmsg, HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_RX_PARAMS_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to set PHY RX Parameters.\n"), __FUNCTION__);
    }
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to setup PHY RX Parameters.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_SetPhyTxParams_G3
*
* DESCRIPTION:
*     Set G3 PHY TX Parameters.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_SetPhyTxParams_G3(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 *pui16;

  /* set configuration PHY TX Parameters */
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, (HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_TX_PARAMS_SIZE));
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Type >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_G3_PHY_TX_PARAMS;
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Length >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_TX_PARAMS_SIZE;

    pui16 = ((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Value >> 1]);

		*((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_TX_PARAMS_OFFSET_Modulation >> 1]) = (ECA_AppEmu_mgh.modulation << 8) + TMR;

		*((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_TX_PARAMS_OFFSET_Level >> 1]) = ECA_AppEmu_mgh.phy_tx_level;
    //
    // Adjust the tonemask size based on the version of the firmware.
    //
    int toneMaskLength = HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_TX_PARAMS_SIZE_Tonemask;

    memcpy(
        &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_TX_PARAMS_OFFSET_Tonemask >> 1],
        &ECA_AppEmu_mgh.ToneMask[0],
        toneMaskLength
    );

    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_SET_INFO, (UINT8 *) pmsg, HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_TX_PARAMS_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to set PHY G3 TX Parameters.\n"), __FUNCTION__);
    }
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to setup PHY G3 TX Parameters.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_SetPhyRxParams_G3
*
* DESCRIPTION:
*     Set G3 PHY TX Parameters.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_SetPhyRxParams_G3(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 *pui16;

  /* set configuration PHY RX Parameters */
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, (HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_SIZE));
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Type >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_G3_PHY_RX_PARAMS;
    *((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Length >> 1]) = HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_SIZE;

    pui16 = ((UINT16 *) &pmsg[HCT_MSG_REQ_SET_INFO_OFFSET_INFO_Value >> 1]);

    *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_OFFSET_Flags >> 1]) = 0;
    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_AGC)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_OFFSET_Flags >> 1]) |= HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_FLAG_AGC;
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_OFFSET_Gain_Value >> 1]) = 0;
    }
    else
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_OFFSET_Gain_Value >> 1]) = ECA_AppEmu_mgh.gain_value;
    }

    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_ROBO)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_OFFSET_Flags >> 1]) |= HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_FLAG_ROBO;
    }
    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_BLI)
    {
      *((UINT16 *) &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_OFFSET_Flags >> 1]) |= HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_FLAG_BLI;
    }

    int toneMaskLength = HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_SIZE_Tonemask;

    memcpy(
        &pui16[HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_OFFSET_Tonemask >> 1],
        &ECA_AppEmu_mgh.ToneMask[0],
        toneMaskLength
    );

    if (HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_SET_INFO, (UINT8 *) pmsg, HCT_MSG_REQ_SET_INFO_SIZE + HCT_MSG_REQ_SET_INFO_TYPE_PHY_G3_RX_PARAMS_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE) != 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to set PHY G3 RX Parameters.\n"), __FUNCTION__);
    }
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to setup PHY G3 RX Parameters.\n"), __FUNCTION__);
  }
}
/******************************************************************************
* FUNCTION NAME: ECA_NETWORK_REGISTER_confirm
*
* DESCRIPTION:
*       MLME registration callback.
*
* Return Value:       none
* Input Parameters:   APPEMU_status_t     status
*                     UINT16 *      pointer to SNA
*                     UINT16 *      pointer to EUI
* Output Parameters:  none
******************************************************************************/
void ECA_NETWORK_REGISTER_confirm(APPEMU_status_t status, UINT16 *pSNA, UINT16 *pEUI)
{
  ECF_TASK_s ecf_task;

  if (status == MAC_STAT_SUCCESS)
  {
    if (pSNA)
    {
      memcpy(&ECA_AppEmu_mgh.SNA[0], pSNA, sizeof(ECA_AppEmu_mgh.SNA));
    }
    if (pEUI)
    {
      memcpy(&ECA_AppEmu_mgh.EUI[0], pEUI, sizeof(ECA_AppEmu_mgh.EUI));
    }
  }
  ecf_task.task_type = ECA_TASK_REGISTER_CONFIRM;
  ecf_task.u.task_param = (void *) status;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_MLME_UNREGISTER_indicate
*
* DESCRIPTION:
*       MLME unregister callback.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_NETWORK_UNREGISTER_indicate(void)
{
  ECF_TASK_s ecf_task;

  ecf_task.task_type = ECA_TASK_UNREGISTER_INDICATE;
  ecf_task.u.task_param = 0;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_OutOfMemory
*
* DESCRIPTION:
*       Out of memory handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_OutOfMemory(void)
{
  ECA_AppEmu_mgh.stats.err_out_of_memory++;
  DIAG_printf(DIAG_LOG_LEVEL_ERROR, DIAG_LOG_223, 1, (UINT32) ECA_AppEmu_mgh.stats.err_out_of_memory);
}

/******************************************************************************
* FUNCTION NAME: ECF_difftime
*
* DESCRIPTION:
*       Computes a time difference.
*
* Return Value:       ECF_TIME_t    time difference
* Input Parameters:   ECF_TIME_t    ending time
*                     ECF_TIME_t    starting time
* Output Parameters:  none
******************************************************************************/
ECF_TIME_t ECF_difftime(ECF_TIME_t tend, ECF_TIME_t tstart)
{
  if (tend >= tstart)
  {
    return tend - tstart;
  }
  else
  {
    return (tend + ECF_TIME_PER_DAY) - tstart;
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_timerCallback
*
* DESCRIPTION:
*       ECF one second timer callback.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
int ECA_AppEmu_timerCallback(DWORD dwTime)
{
  ECF_TASK_s ecf_task;

  ECA_AppEmu_mgh.time_of_day++;
  if (ECA_AppEmu_mgh.time_of_day >= ECF_TIME_PER_DAY)
  {
    ECA_AppEmu_mgh.time_of_day = 0;
  }

//  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: timer callback %d.\n"), __FUNCTION__, ECA_AppEmu_mgh.time_of_day);

  ecf_task.task_type = ECA_TASK_TIMER;
  ecf_task.u.task_param = 0;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);

  return 0;
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_timerCallback_handler
*
* DESCRIPTION:
*       ECF imer callback handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_timerCallback_handler(void)
{
  ECF_TASK_s ecf_task;

//  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: timer callback handler %d.\n"), __FUNCTION__, ECA_AppEmu_mgh.time_of_day);

  if (ECA_AppEmu_mgh.flags & HostAppEMU_Exit)
  {
    if (ECA_AppEmu_mgh.exit_start == 0)
    {
      ECA_AppEmu_mgh.exit_start = ECA_AppEmu_mgh.task_start;
    }

    if (ECF_difftime(ECA_AppEmu_mgh.time_of_day, ECA_AppEmu_mgh.exit_start) >= ECA_APPEMU_EXIT_DELAY)
    {
      ECA_AppEmu_mgh.exit_start = 0;
      HostAppEMU_STATUS HostAppEMU_SetFlags(HostAppEMU_Exit);

      /* queue back up another so we will break in the message loop */
      ecf_task.task_type = ECA_TASK_EXIT;
      os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
    }
  }

  if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_SEND_RPY)
  {
    if (ECF_difftime(ECA_AppEmu_mgh.time_of_day, ECA_AppEmu_mgh.RPY_task_start) >= (ECF_TIME_t) RPY_delay)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("RPY Timeout, sending RPY to device.\n"), RPY_delay);
      HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) NULL, 0, FALSE);
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_SEND_RPY;
    }
  }
	
  if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_MAC)
    || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
    || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
  )
  {
    if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_REGISTRATION_COMPLETE))
    {
      if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_REGISTRATION_IN_PROGRESS))
      {
        if (ECF_difftime(ECA_AppEmu_mgh.time_of_day, ECA_AppEmu_mgh.task_start) >= registrationDelay)
        {
          ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_REGISTRATION_IN_PROGRESS;
          ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;

          if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_MAC)
          {
            ECA_AppEmu_Register();
          }
          else if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
          {
            ECA_AppEmu_Discover();
            g3HeartBeatCounter = 0;
          }
          else if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
          {
            ECA_AppEmu_Network_Start();
          }
        }
      }
      return;
    }
  }

  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS))
    {
      if (ECF_difftime(ECA_AppEmu_mgh.time_of_day, ECA_AppEmu_mgh.task_start) >= connectionDelay)
      {
        ECA_AppEmu_mgh.flags |= (ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS);
        ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;

        ECA_AppEmu_Connect();
      }
    }
    else if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
    {
      if (ECF_difftime(ECA_AppEmu_mgh.time_of_day, ECA_AppEmu_mgh.task_start) >= ECA_APPEMU_CONNECTION_COMPLETE_DELAY)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("CONNECT Timeout, restarting...\n"));
        ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
      }
    }
    return;
  }
  else
	{
    //
    // Should be connected to the concentrator here.
    //
		if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
    {
      if (g3MessageCounter != previousG3MessageCounter)
      {
        previousG3MessageCounter = g3MessageCounter;
        g3HeartBeatCounter = 0;
      }
			//
			// Don't timeout if running tuntap..
			//
      else if (strlen(tunTapDriverName)== 0 && emeterDiscoveryTimeout != 0 && g3HeartBeatCounter++ >= emeterDiscoveryTimeout)
      {
        // this is a one second timer so count to sixty and see if messages have been received
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("G3 Connection Timeout: performing Discovery...\n"));
        ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_REGISTRATION_IN_PROGRESS;
        ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_REGISTRATION_COMPLETE;
        ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
        ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECTED;
        ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;
        g3HeartBeatCounter = 0;
        ECA_AppEmu_Discover();
      }
    }
    else if (strlen(tunTapDriverName)== 0 && (HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
    {
      pathDiscoveryInterval++;
      if (emeterPathDiscoveryInterval > 0 && pathDiscoveryInterval >= emeterPathDiscoveryInterval)
      {
        pathDiscoveryInterval = 0;
        //
        // If waiting for another discovery response wait until finished..
        //
        // xraf - lets don't do this for now.. if PCL doesn't respond we would quit..
        //if (emeterDiscoverNetworkPathNode == 0)
        {
          G3_Network * network = G3_Network::GetNetwork(lastDiscoveryIndex++);
          if (network != NULL)
          {
            HCT_MSG_BUFFER_t *pmsg;
            pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_G3_ROUTE_DISCOVER_SIZE);
            if (pmsg)
            {
              ((HCT_MSG_REQ_G3_ROUTE_DISCOVER_s *)pmsg)->DiscoverType = HCT_MSG_DISCOVER_G3_TYPE_ROUTE;
              ((HCT_MSG_REQ_G3_ROUTE_DISCOVER_s *)pmsg)->MaxHops = 7;
              ((HCT_MSG_REQ_G3_ROUTE_DISCOVER_s *)pmsg)->DstAddress = network->NetworkShortAddress();

              emeterDiscoverNetworkPathNode = network->NetworkShortAddress();
              HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Performing route discover for %X\n"),network->NetworkShortAddress());
          
              HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DISCOVER, (UINT8 *) pmsg, HCT_MSG_REQ_G3_ROUTE_DISCOVER_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
            }
            else
            {
              ECA_OutOfMemory();
              HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to DISCOVER.\n"), __FUNCTION__);
            }
          }
        }
      }
    }


/* xraf - removed for G3 concentrator..

		if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
    {
    //  static int itercount = 0;

      if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DETACHING))
      {
#ifdef DEBUG_SHU
        //debug shu - only send 1 packet each round
        if ((itercount++ & 0xffff) == 0xffff)
#else
        if ((itercount++ & 0x03) == 0x03)
#endif        
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Calling G3 DETACH to drop connection.\n"));
          ECA_AppEMU_DETACH((UINT16 *) NULL);
        }
        else if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DATA_SEND))
        {
#ifdef DEBUG_SHU
          //debug shu - only send 1 packet 
          int delayTime = 7;
          if ( ((itercount-1)/delayTime <=0) && ((itercount-1)%delayTime == 0) ) {
            HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("\n\n\n.....send packet %d to BN\n"), (itercount-1)/delayTime);
#endif
            // node initiates data transfer 
            ECA_AppEmu_G3_SendPacket();
#ifdef DEBUG_SHU
          }
#endif          
        }
      }
    }

xraf - removed for G3 concentrator.. */
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_ProcessTask
*
* DESCRIPTION:
*       Processes a queued task.
*
* Return Value:       ECF_STATUS    ECF_STATUS_SUCCESS or errorlevel
* Input Parameters:   ECF_TASK_s    task to process
* Output Parameters:  none
******************************************************************************/
ECF_STATUS ECA_AppEmu_ProcessTask(ECF_TASK_s *ptask)
{
  ECF_STATUS ecf_status;
  unsigned long flags;

  ecf_status = ECF_STATUS_SUCCESS;

  if (ptask->task_type != ECA_TASK_TIMER)
  {
    ECA_AppEmu_mgh.last_task = ECA_AppEmu_mgh.time_of_day;
  }

  switch (ptask->task_type)
  {
  case ECA_TASK_EXIT:
    HostAppEMU_Flags |= HostAppEMU_Exit;
    break;

  case ECA_TASK_TIMER:
    ECA_AppEmu_timerCallback_handler();
    break;

  case ECA_TASK_REGISTER_CONFIRM:
    ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;
    ECA_MLME_REGISTER_confirm_handler((APPEMU_status_t) (UINT32) ptask->u.task_param);
    break;

  case ECA_TASK_UNREGISTER_INDICATE:
    ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;
    ECA_MLME_UNREGISTER_indicate_handler();
    break;

  case ECA_TASK_CONNECTION_ESTABLISHED:
    ECA_AppEmu_establishConnection((ConnHandle_t) (UINT32) ptask->u.task_param);
    break;

  case ECA_TASK_PACKET_RECEIVED:
    if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
    {
      //remove for g3
      //HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unexpected packet received, not connected, discarding.\n"), __FUNCTION__);
      //MyLocalFree(ptask->u.s.msg);
    }
    else
    {
    }
    ECA_AppEmu_processPacket(ptask->u.s.msg, ptask->u.s.msglen, ptask->bRPY);

    break;

  case ECA_TASK_DISCONNECT:
    if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
    {
      ECA_AppEmu_mgh.stats.unexpected_disconnect_indicate++;
      DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_207, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_disconnect_indicate);
    }
    else
    {
      DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_206, 0);
    }

    switch (ptask->conn_type)
    {
    case HCT_MSG_CONNECT_CONN_TYPE_MAC:
      ECA_AppEmu_DISCONNECT_acknowledge_MAC((UINT16) ptask->u.task_param);

      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Disconnect]++;
      ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;
      ECA_AppEmu_mgh.flags &= ~(0
          | ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS
          | ECA_APPEMU_FLAGS_CONNECTED
          | ECA_APPEMU_FLAGS_DATA_SEND
          | ECA_APPEMU_FLAGS_INACTIVITY_TIMEOUT
      );

      ECA_AppEmu_mgh.connHandle = INVALID_CONNECTION_HANDLE;
      ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_None;

      break;
    case HCT_MSG_CONNECT_CONN_TYPE_NONE:
      ECA_AppEmu_DISCONNECT_acknowledge_MAC((UINT16) ptask->u.task_param); /* TODO: this is currently coming down as a MAC event */
      HostAppEMU_GetFlags(&flags);
      ECA_AppEMU_CL_Release(ECA_AppEmu_mgh.Destination_Address);
      break;
    default:
      break;
    }

    break;

  case ECA_TASK_UNREGISTER_CONFIRM:
    ECA_MLME_UNREGISTER_confirm_handler((APPEMU_status_t) (UINT32) ptask->u.task_param);
    break;

  case ECA_TASK_SEND_RPY:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Pausing %d seconds before sending RPY.\n"), RPY_delay);

    ECA_AppEmu_mgh.RPY_task_start = ECA_AppEmu_mgh.time_of_day;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_SEND_RPY;
    break;

  case ECA_TASK_SEND_MSG_TO_HOST:
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_ALARM, (UINT8 *) ptask->u.s.msg, ptask->u.s.msglen, ptask->bRPY);
    break;

  default:
    proj_assert(0);
    ecf_status = ECF_STATUS_UNSUPPORTED;
    break;
  }

  return ecf_status;
}

/******************************************************************************
* FUNCTION NAME: ECA_MLME_REGISTER_confirm_handler
*
* DESCRIPTION:
*       MLME registration callback.
*
* Return Value:       none
* Input Parameters:   APPEMU_status_t     status
* Output Parameters:  none
******************************************************************************/
void ECA_MLME_REGISTER_confirm_handler(APPEMU_status_t mac_status)
{
  ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_REGISTRATION_IN_PROGRESS;

  if (mac_status == MAC_STAT_SUCCESS)
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_REGISTRATION_COMPLETE;
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_211, 0);
  }
  else
  {
    if (mac_status == MAC_STAT_REG_REDUNDANT)
    {
      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_REGISTRATION_COMPLETE;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Already registered.\n"), __FUNCTION__, mac_status);
      ECA_AppEmu_UNREGISTER();
    }
    else
    {
      DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_231, 1, (UINT32) mac_status);
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_STATE_MASK;
    }
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_MLME_UNREGISTER_confirm_handler
*
* DESCRIPTION:
*       MLME unregistration callback.
*
* Return Value:       none
* Input Parameters:   APPEMU_status_t     status
* Output Parameters:  none
******************************************************************************/
void ECA_MLME_UNREGISTER_confirm_handler(APPEMU_status_t mac_status)
{
  ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_REGISTRATION_IN_PROGRESS;

  if (mac_status != MAC_STAT_SUCCESS)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR %#x return from NETWORK_UNREGISTER.\n"), __FUNCTION__, mac_status);
  }

  ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_STATE_MASK;
  DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_233, 0);
}

/******************************************************************************
* FUNCTION NAME: ECA_MLME_UNREGISTER_indicate_handler
*
* DESCRIPTION:
*       MLME unregister callback.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_MLME_UNREGISTER_indicate_handler(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 msglen;

  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) && (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_REGISTRATION_COMPLETE))
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_233, 0);
  }
  else
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_234, 0);
  }
  ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_STATE_MASK;

  /* send ALARM acknowledgement */
  msglen = HCT_MSG_RPY_ALARM_SIZE + HCT_MSG_RPY_ALARM_TYPE_NETWORK_DEREGISTERED_SIZE;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_RPY_ALARM_OFFSET_Alarm_Type >> 1]) = HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED;
    *((UINT16 *) &pmsg[HCT_MSG_RPY_ALARM_OFFSET_Data_Length >> 1]) = HCT_MSG_RPY_ALARM_TYPE_NETWORK_DEREGISTERED_SIZE;

    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_ALARM, (UINT8 *) pmsg, msglen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to acknowledge ALARM.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Register
*
* DESCRIPTION:
*       Register with the BN.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Register(void)
{
  BOOL bStatus;
  HCT_MSG_BUFFER_t *pmsg;

  bStatus = ECA_GetSerialNumber();
  if (!bStatus)
  {
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_244, 0);

    /* just report some error so we'll timeout and try again */
    ECA_NETWORK_REGISTER_confirm(MAC_STAT_ERR_FAILURE, (UINT16 *) NULL, (UINT16 *) NULL);
    return;
  }

  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_MAC)
  {
    pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_NETWORK_REGISTER_SIZE);
    if (pmsg)
    {
      *((UINT16 *) &pmsg[HCT_MSG_REQ_NETWORK_REGISTER_OFFSET_Registration_Type >> 1]) = HCT_MSG_NETWORK_REGISTRATION_TYPE_AUTO;
      *((UINT16 *) &pmsg[HCT_MSG_REQ_NETWORK_REGISTER_OFFSET_Registration_Data_Length >> 1]) = 0;

      HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_NETWORK_REGISTER, (UINT8 *) pmsg, HCT_MSG_REQ_NETWORK_REGISTER_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
    }
    else
    {
      ECA_OutOfMemory();
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to NETWORK_REGISTER.\n"), __FUNCTION__);
    }
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_203, 0);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Discover
*
* DESCRIPTION:
*       Discover G3 Node.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Discover(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 ui16;
  static UINT32 snRoundCount;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DISCOVER_G3_SIZE);
  if (pmsg)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("************* Round %d *************\n"), snRoundCount++);

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Running DISCOVER type %d for %d seconds.\n"), __FUNCTION__,
        HCT_MSG_DISCOVER_G3_TYPE_NETWORK, ECA_AppEmu_mgh.discover_time
    );
    ui16 = HCT_MSG_DISCOVER_G3_TYPE_NETWORK;
    ui16 |= (ECA_AppEmu_mgh.discover_time << 8);
    *((UINT16 *) &pmsg[HCT_MSG_REQ_DISCOVER_G3_OFFSET_Type_Duration >> 1]) = ui16;

    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DISCOVER, (UINT8 *) pmsg, HCT_MSG_REQ_DISCOVER_G3_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to DISCOVER.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Network_Start
*
* DESCRIPTION:
*       Discover G3 Concentrator.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Network_Start(void)
{
  HCT_MSG_BUFFER_t *pmsg;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_NETWORK_START_G3_SIZE);
  if (pmsg)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Network Start, PanID:%#x.\n"), __FUNCTION__,
        ECA_AppEmu_mgh.G3_Pan_ID);

		printf("PAN ID= %x\n", ECA_AppEmu_mgh.G3_Pan_ID); //xraf

    *((UINT16 *) &pmsg[HCT_MSG_REQ_NETWORK_START_G3_OFFSET_Pan_ID >> 1]) = ECA_AppEmu_mgh.G3_Pan_ID;

    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_NETWORK_START, (UINT8 *) pmsg, HCT_MSG_REQ_NETWORK_START_G3_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to Start Network.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_GetSerialNumber
*
* DESCRIPTION:
*       Retrieve the serial number.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
BOOL ECA_GetSerialNumber(void)
{
  ECA_AppEmu_buildSerialNumber(&ECA_AppEmu_mgh.serial_number[0]);
  ECA_AppEMU_buildDeviceIdentifier(&ECA_AppEmu_mgh.LLC_DeviceIdentifier[0]);
  return TRUE;
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Set_Test_Length
*
* DESCRIPTION:
*       Sets the test PDU length.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Set_Test_Length(void)
{
  if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DBPSK)
  {
    ECA_AppEmu_mgh.PPDU_LengthTesting_Offset = PPDU_LENGTH_TESTING_OFFSET_DBPSK_OFF;
  }
  else if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DBPSK)
  {
    ECA_AppEmu_mgh.PPDU_LengthTesting_Offset = PPDU_LENGTH_TESTING_OFFSET_DQPSK_OFF;
  }
  else if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_DBPSK)
  {
    ECA_AppEmu_mgh.PPDU_LengthTesting_Offset = PPDU_LENGTH_TESTING_OFFSET_D8PSK_OFF;
  }
	else if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_ROBO)
  {
    ECA_AppEmu_mgh.PPDU_LengthTesting_Offset = PPDU_LENGTH_TESTING_OFFSET_DBPSK_OFF;
  }
  else
  {
    proj_assert(0);
  }
  if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_FEC)
  {
    ECA_AppEmu_mgh.PPDU_LengthTesting_Offset++;
  }

  if (ECA_AppEmu_mgh.PPDU_LengthTesting_Offset != ECA_AppEmu_mgh.PPDU_LengthTesting_Offset_Save)
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_225, 1, (UINT32) ECA_AppEmu_mgh.PPDU_LengthTesting_Offset);
    ECA_AppEmu_mgh.PPDU_LengthTesting_Offset_Save = ECA_AppEmu_mgh.PPDU_LengthTesting_Offset;
  }

  ECA_AppEmu_mgh.PPDU_LengthTesting_Index = 0;
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Connect
*
* DESCRIPTION:
*       Obtain a connection to the BN.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Connect(void)
{
  ECF_TASK_s ecf_task;

  ECA_AppEmu_mgh.stats.ttl_connect_attempts++;

  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_MAC)
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_201, 7,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) (ECA_AppEmu_mgh.serial_number[0] & 0xff),(UINT32) (ECA_AppEmu_mgh.serial_number[0] >> 8), 
        (UINT32) (ECA_AppEmu_mgh.serial_number[1] & 0xff),(UINT32) (ECA_AppEmu_mgh.serial_number[1] >> 8),
        (UINT32) (ECA_AppEmu_mgh.serial_number[2] & 0xff), (UINT32) (ECA_AppEmu_mgh.serial_number[2] >> 8)
    );
  }

  ECA_AppEmu_Set_Test_Length();

  switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
  case HostAppEMU_MAC:
    ECA_AppEmu_Connect_MAC();
    break;
#ifdef APPEMU_IEC432
  case HostAppEMU_CL_IEC432:
    ECA_AppEmu_Connect_CL_IEC432();
    break;
#endif // #ifdef APPEMU_IEC432
  case HostAppEMU_CL_LLC:
    if (HostAppEMU_Flags & HostAppEMU_LLC_AUTO)
    {
      /* just advance to the CONNECTED state */
      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_CONNECTED;

      ecf_task.u.task_param = (void *) FAKE_CONNECTION_HANDLE;
      ecf_task.task_type = ECA_TASK_CONNECTION_ESTABLISHED;
      os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
    }
    else
    {
      ECA_AppEmu_Connect_LLC();
    }
    break;
  case HostAppEMU_P2P:
  case HostAppEMU_IPv4:
    /* just advance to the CONNECTED state */
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_CONNECTED;

    ecf_task.u.task_param = (void *) FAKE_CONNECTION_HANDLE;
    ecf_task.task_type = ECA_TASK_CONNECTION_ESTABLISHED;
    os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
    break;

  case HostAppEMU_G3_Node:
    ECA_AppEmu_Connect_G3();
    break;

  case HostAppEMU_G3_Conc:
    break;

  default:
    proj_assert(0);
    return;
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Connect_MAC
*
* DESCRIPTION:
*       Obtain a MAC connection to the BN.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Connect_MAC(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 *pui16;
  UINT16 flags_ConnType;
  UINT16 msglen;

  msglen = HCT_MSG_REQ_CONNECT_SIZE + HCT_MSG_REQ_CONNECT_CONN_TYPE_MAC_SIZE + ECA_APPEMU_SERIAL_LEN;
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_CONNECT_OFFSET_Conn_Type >> 1]) = HCT_MSG_CONNECT_CONN_TYPE_MAC;
    *((UINT16 *) &pmsg[HCT_MSG_REQ_CONNECT_OFFSET_Connection_Data_Length >> 1]) = msglen - HCT_MSG_REQ_CONNECT_SIZE;

    pui16 = ((UINT16 *) &pmsg[HCT_MSG_REQ_CONNECT_OFFSET_Connection_Data >> 1]);

    flags_ConnType = 0;
    if (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_MAC_ARQ)
    {
      flags_ConnType |= HCT_MSG_REQ_CONNECT_CONN_TYPE_MAC_FLAG_ARQ;
    }

    flags_ConnType |= ECA_APPEMU_CON_TYPE;

    pui16[HCT_MSG_REQ_CONNECT_CONN_TYPE_MAC_OFFSET_Flags_ConnType >> 1] = flags_ConnType;
    pui16[HCT_MSG_REQ_CONNECT_CONN_TYPE_MAC_OFFSET_CfBytes >> 1] = 0;
    memcpy(
          &pui16[HCT_MSG_REQ_CONNECT_CONN_TYPE_MAC_OFFSET_Dest_Addr >> 1],
          &ECA_AppEmu_mgh.SNA,
          HCT_MSG_REQ_CONNECT_CONN_TYPE_MAC_SIZE_Dest_Addr
    );

    pui16 += (HCT_MSG_REQ_CONNECT_CONN_TYPE_MAC_SIZE >> 1);

    memcpy(pui16, &ECA_AppEmu_mgh.sSerial[0], ECA_APPEMU_SERIAL_LEN);

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: CONNECT to EUI %02x:%02x:%02x:%02x:%02x:%02x.\n"), __FUNCTION__,
        (UINT8) (ECA_AppEmu_mgh.EUI[0] >> 8) & 0xff, (UINT8) (ECA_AppEmu_mgh.EUI[0]) & 0xff,
        (UINT8) (ECA_AppEmu_mgh.EUI[1] >> 8) & 0xff, (UINT8) (ECA_AppEmu_mgh.EUI[1]) & 0xff,
        (UINT8) (ECA_AppEmu_mgh.EUI[2] >> 8) & 0xff, (UINT8) (ECA_AppEmu_mgh.EUI[2]) & 0xff
    );
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_CONNECT, (UINT8 *) pmsg, msglen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to attempt CONNECT.\n"), __FUNCTION__);
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
  }
}

#ifdef APPEMU_IEC432
/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Connect_CL_IEC432
*
* DESCRIPTION:
*       Obtain a CL IEC432 connection to the BN.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Connect_CL_IEC432(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 *pui16;
  UINT16 msglen;

  msglen = HCT_MSG_REQ_CONNECT_SIZE + HCT_MSG_REQ_CONNECT_CONN_TYPE_CL_IEC432_SIZE;
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_CONNECT_OFFSET_Conn_Type >> 1]) = HCT_MSG_CONNECT_CONN_TYPE_CL_IEC432;
    *((UINT16 *) &pmsg[HCT_MSG_REQ_CONNECT_OFFSET_Connection_Data_Length >> 1]) = msglen - HCT_MSG_REQ_CONNECT_SIZE;

    pui16 = ((UINT16 *) &pmsg[HCT_MSG_REQ_CONNECT_OFFSET_Connection_Data >> 1]);

    memset(&pui16[HCT_MSG_REQ_CONNECT_CONN_TYPE_CL_IEC432_OFFSET_Serial_Number >> 1], 0, HCT_MSG_REQ_CONNECT_CONN_TYPE_CL_IEC432_SIZE);
    memcpy(&pui16[HCT_MSG_REQ_CONNECT_CONN_TYPE_CL_IEC432_OFFSET_Serial_Number >> 1], &ECA_AppEmu_mgh.sSerial[0], ECA_APPEMU_SERIAL_LEN);

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: CL IEC432 CONNECT.\n"), __FUNCTION__);
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_CONNECT, (UINT8 *) pmsg, msglen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to attempt CONNECT.\n"), __FUNCTION__);
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
  }
}
#endif // #ifdef APPEMU_IEC432

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Connect_LLC
*
* DESCRIPTION:
*       Obtain a LLC connection to the BN.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Connect_LLC(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 msglen;

  ECA_GetSerialNumber();

  msglen = HCT_MSG_REQ_CL_ESTABLISH_SIZE;
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
		char temp[100];
		char temp1[10];
		temp[0] = 0;
		//
		// Get all but the last hex digit, then get the last digit without the ':'
		//
		for (int index = 0; index < deviceSerialNumberLength-1; index++)
		{
			sprintf(temp1,"%02x:", ECA_AppEmu_mgh.LLC_DeviceIdentifier[index]);
			strcat(temp, temp1);
		}
		sprintf(temp1,"%02x", ECA_AppEmu_mgh.LLC_DeviceIdentifier[deviceSerialNumberLength-1]);
		strcat(temp,temp1);

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: CL_ESTABLISH to %S\n"),
        __FUNCTION__, temp);

		((HCT_MSG_REQ_CL_ESTABLISH_s *)pmsg)->IdentifierLength = deviceSerialNumberLength;
    memcpy(
      ((UINT16 *) &pmsg[HCT_MSG_REQ_CL_ESTABLISH_OFFSET_Device_Identifier >> 1]),
      &ECA_AppEmu_mgh.LLC_DeviceIdentifier,
      HCT_MSG_REQ_CL_ESTABLISH_SIZE_Max_Device_Identifier
    );
		//
		// the actual message length is the length of the serial number plus 2 bytes for the length work
		// the serial number can not in in x00 or it will not be counted.
		//
		msglen = deviceSerialNumberLength + HCT_MSG_REQ_CL_ESTABLISH_SIZE_ID_LENGTH;
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_CL_ESTABLISH, (UINT8 *) pmsg, msglen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to attempt CL_ESTABLISH.\n"), __FUNCTION__);
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Connect_G3
*
* DESCRIPTION:
*       Obtain a G3 connection to the BN.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Connect_G3(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 msglen;

  msglen = HCT_MSG_REQ_ATTACH_G3_SIZE;
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, msglen);
  if (pmsg)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ATTACH to PanID:%02x LBA Address:%02x\n"),
        __FUNCTION__, ECA_AppEmu_mgh.G3_Pan_ID, ECA_AppEmu_mgh.LBA_Address);

		*((UINT16 *) &pmsg[HCT_MSG_REQ_ATTACH_G3_OFFSET_Pan_ID >> 1]) = ECA_AppEmu_mgh.G3_Pan_ID;
    *((UINT16 *) &pmsg[HCT_MSG_REQ_ATTACH_G3_OFFSET_LBA_Address >> 1]) = ECA_AppEmu_mgh.LBA_Address;

    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_ATTACH, (UINT8 *) pmsg, msglen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to attempt ATTACH.\n"), __FUNCTION__);
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_establishConnection
*
* DESCRIPTION:
*       Connection established handler.
*
* Return Value:       none
* Input Parameters:   ConnHandle_t   connection handle
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_establishConnection(ConnHandle_t connHandle)
{
  if (connHandle != INVALID_CONNECTION_HANDLE)
  {
    ECA_AppEmu_mgh.connHandle = connHandle;
    ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Connect]++;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_CONNECTED;

    ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_Loop_Start;
  }

  ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
}

/******************************************************************************
* FUNCTION NAME: ECA_CONNECT_confirm_MAC
*
* DESCRIPTION:
*       MAC Connection confirmation handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_CONNECT_confirm_MAC(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen, UINT16 status)
{
  UINT16 ui16;
  ECF_TASK_s ecf_task;

  if (msglen < (HCT_MSG_RPY_CONNECT_SIZE + HCT_MSG_RPY_CONNECT_CONN_TYPE_MAC_SIZE))
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_CONNECT_SIZE + HCT_MSG_RPY_CONNECT_CONN_TYPE_MAC_SIZE, msglen);
    status = ECF_STATUS_ERROR;
  }

  if (status == HCT_MSG_STATUS_SUCCESS)
  {
    ui16 = *((UINT16 *) &pmsg[HCT_MSG_RPY_CONNECT_OFFSET_Conn_Type >> 1]);
    if (ui16 != HCT_MSG_CONNECT_CONN_TYPE_MAC)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Conn Type Mismatch, expected Type %#x, received type %#x!\n"),
            __FUNCTION__, HCT_MSG_CONNECT_CONN_TYPE_MAC, ui16);
      status = ECF_STATUS_ERROR;
    }
  }

  if (status == HCT_MSG_STATUS_SUCCESS)
  {
    ui16 = *((UINT16 *) &pmsg[HCT_MSG_RPY_CONNECT_OFFSET_Conn_Handle >> 1]);
    ecf_task.u.task_param = (void *) ui16;
  }
  else
  {
    ECA_AppEmu_mgh.stats.err_connect++;
    if (status == MAC_STAT_ESTABLISH_TIMEOUT)
    {
      DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_202, 2,
          (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
          (UINT32) ECA_AppEmu_mgh.stats.err_connect
      );
    }
    else
    {
      DIAG_printf(DIAG_LOG_LEVEL_ERROR, DIAG_LOG_228, 2,
          (UINT32) status,
          (UINT32) ECA_AppEmu_mgh.stats.err_connect
      );
    }
    ecf_task.u.task_param = (void *) INVALID_CONNECTION_HANDLE;
  }

  ecf_task.task_type = ECA_TASK_CONNECTION_ESTABLISHED;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_CL_ESTABLISH_confirm
*
* DESCRIPTION:
*       LLC CL_ESTABLISH confirmation handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_CL_ESTABLISH_confirm(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen, UINT16 status)
{
  UINT8 *pui8;
  ECF_TASK_s ecf_task;
  UINT16 rpy_status;

/* this is a variable length message, can't test this way..

if (msglen < HCT_MSG_RPY_CL_ESTABLISH_SIZE_Status)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_CL_ESTABLISH_SIZE_Status, msglen);
    status = ECF_STATUS_ERROR;
  }

	*/
	UINT16 deviceIdLength = (UINT16)pmsg[HCT_MSG_RPY_CL_ESTABLIST_OFFSET_Id_Length >> 1];

	if (status == HCT_MSG_STATUS_SUCCESS)
  {
    pui8 = (UINT8 *) &pmsg[HCT_MSG_RPY_CL_ESTABLISH_OFFSET_Device_Identifier >> 1];
    if (memcmp(&ECA_AppEmu_mgh.LLC_DeviceIdentifier, pui8, deviceIdLength) != 0)
    {

			char temp[100];
			char temp1[10];
			temp[0] = 0;
			//
			// Get all but the last hex digit, then get the last digit without the ':'
			//
			for (int index = 0; index < deviceIdLength-1; index++)
			{
				sprintf(temp1,"%02x:", pui8[index]);
				strcat(temp, temp1);
			}
			sprintf(temp1,"%02x", pui8[deviceIdLength-1]);
			strcat(temp,temp1);

      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
            TEXT("%S: ERROR! Device Identifier Mismatch, received %S!\n"),
            __FUNCTION__, temp);

			status = ECF_STATUS_ERROR;
    }
  }

  if (status == HCT_MSG_STATUS_SUCCESS)
  {
		pui8 = (UINT8 *)pmsg;

    rpy_status = *((UINT16 *) &pmsg[HCT_MSG_RPY_CL_ESTABLISH_OFFSET_Status >> 1]);
    if (rpy_status == HCT_MSG_STATUS_SUCCESS)
    {
			UINT16 destinationAddressOffest = HCT_MSG_RPY_CL_ESTABLISH_OFFSET_Device_Identifier 
				+ deviceIdLength;

			UINT16 baseAddressOffset = HCT_MSG_RPY_CL_ESTABLISH_OFFSET_Device_Identifier 
				+ deviceIdLength 
				+ HCT_MSG_RPY_CL_ESTABLISH_SIZE_Destination_Address;
      
			memcpy(
          &ECA_AppEmu_mgh.Destination_Address,
          ((UINT8 *) &pui8[destinationAddressOffest]),
          HCT_MSG_RPY_CL_ESTABLISH_SIZE_Destination_Address
      );

      memcpy(
          &ECA_AppEmu_mgh.Base_Address,
          ((UINT8 *) &pui8[baseAddressOffset]),
          HCT_MSG_RPY_CL_ESTABLISH_SIZE_Base_Address
      );

      ecf_task.u.task_param = (void *) FAKE_CONNECTION_HANDLE;

      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("CL_ESTABLISH Success, Dest Addr %#x, Base Addr %#x!\n"),
            ECA_AppEmu_mgh.Destination_Address, ECA_AppEmu_mgh.Base_Address);
    }
    else
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ERROR! CL_ESTABLISH returned status %#x!\n"), rpy_status);
    }
  }
  else
  {
    ECA_AppEmu_mgh.stats.err_connect++;
    DIAG_printf(DIAG_LOG_LEVEL_ERROR, DIAG_LOG_228, 2,
        (UINT32) status,
        (UINT32) ECA_AppEmu_mgh.stats.err_connect
    );
    ecf_task.u.task_param = (void *) INVALID_CONNECTION_HANDLE;
  }

  ecf_task.task_type = ECA_TASK_CONNECTION_ESTABLISHED;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

#ifdef APPEMU_IEC432
/******************************************************************************
* FUNCTION NAME: ECA_CONNECT_confirm_CL_IEC432
*
* DESCRIPTION:
*       CL IEC432 Connection confirmation handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_CONNECT_confirm_CL_IEC432(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen, UINT16 status)
{
  UINT16 ui16;
  UINT16 *pui16;
  ECF_TASK_s ecf_task;

  if (msglen != (HCT_MSG_RPY_CONNECT_SIZE + HCT_MSG_RPY_CONNECT_CONN_TYPE_CL_IEC432_SIZE))
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_CONNECT_SIZE + HCT_MSG_RPY_CONNECT_CONN_TYPE_CL_IEC432_SIZE, msglen);
    status = ECF_STATUS_ERROR;
  }

  if (status == HCT_MSG_STATUS_SUCCESS)
  {
    ui16 = *((UINT16 *) &pmsg[HCT_MSG_RPY_CONNECT_OFFSET_Conn_Type >> 1]);
    if (ui16 != HCT_MSG_CONNECT_CONN_TYPE_CL_IEC432)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Conn Type Mismatch, expected Type %#x, received type %#x!\n"),
            __FUNCTION__, HCT_MSG_CONNECT_CONN_TYPE_MAC, ui16);
      status = ECF_STATUS_ERROR;
    }
  }

  if (status == HCT_MSG_STATUS_SUCCESS)
  {
    pui16 = (UINT16 *) &pmsg[HCT_MSG_RPY_CONNECT_OFFSET_Conn_Data >> 1];

    ecf_task.u.task_param = (void *) pui16[HCT_MSG_RPY_CONNECT_CONN_TYPE_CL_IEC432_OFFSET_Dest_Addr];
  }
  else
  {
    ECA_AppEmu_mgh.stats.err_connect++;
    if (status == MAC_STAT_ESTABLISH_TIMEOUT)
    {
      DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_202, 2,
          (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
          (UINT32) ECA_AppEmu_mgh.stats.err_connect
      );
    }
    else
    {
      DIAG_printf(DIAG_LOG_LEVEL_ERROR, DIAG_LOG_228, 2,
          (UINT32) status,
          (UINT32) ECA_AppEmu_mgh.stats.err_connect
      );
    }
    ecf_task.u.task_param = (void *) INVALID_CONNECTION_HANDLE;
  }

  ecf_task.task_type = ECA_TASK_CONNECTION_ESTABLISHED;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}
#endif // #ifdef APPEMU_IEC432

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket
*
* DESCRIPTION:
*       Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY)
{
  switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
  case HostAppEMU_MAC:
    ECA_AppEmu_processPacket_MAC(pmsg, msg_len);
    break;
#ifdef APPEMU_IEC432
  case HostAppEMU_CL_IEC432:
    ECA_AppEmu_processPacket_CL_IEC432(pmsg, msg_len, bRPY);
    break;
#endif // #ifdef APPEMU_IEC432
  case HostAppEMU_P2P:
    if (HostAppEMU_Flags & HostAppEMU_Use_IP)
    {
      ECA_AppEmu_processPacket_IPv4(pmsg, msg_len);
    }
    else
    {
      ECA_AppEmu_processPacket_LLC(pmsg, msg_len, bRPY);
    }
    break;
  case HostAppEMU_CL_LLC:
    ECA_AppEmu_processPacket_LLC(pmsg, msg_len, bRPY);
    break;
  case HostAppEMU_IPv4:
    ECA_AppEmu_processPacket_IPv4(pmsg, msg_len);
    break;
  
	case HostAppEMU_G3_Node:
    ECA_AppEmu_processPacket_G3(pmsg, msg_len, bRPY);
    break;

	case HostAppEMU_G3_Conc:
    ECA_AppEmu_processPacket_G3(pmsg, msg_len, bRPY);
    break;
  default:
    proj_assert(0);
    return;
  }
}
/******************************************************************************
* FUNCTION NAME:
*
* DESCRIPTION:
*       G3 Concentrator Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ProcessEmeterEmulationMessage_Node(HCT_MSG_BUFFER_t * pmsg, UINT16 msg_len);
void SendNodeResponse(IPv6MainHeader ipWrapper, UINT16 shortAddress,UINT16 payloadType, UINT32 payloadId, int oldLength, int length);
void ProcessEmeterEmulationMessage_Concentrator(HCT_MSG_BUFFER_t * pmsg, UINT16 msg_len);

void ProcessEmeterEmulationMessage(HCT_MSG_BUFFER_t * pmsg, UINT16 msg_len)
{
  if (msg_len == HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_SIZE)
  {
    ECA_AppEmu_processPacket_G3_CONFIRM(pmsg, msg_len);
  }
	else if (HostAppEMU_Flags & HostAppEMU_G3_Node)
	{
		ProcessEmeterEmulationMessage_Node(pmsg, msg_len);
	}
	else if (HostAppEMU_Flags & HostAppEMU_G3_Conc)
	{
		ProcessEmeterEmulationMessage_Concentrator(pmsg, msg_len);
	}
}

void ProcessEmeterEmulationMessage_Node(HCT_MSG_BUFFER_t * pmsg, UINT16 msg_len)
{
	// Remove the IPv6 wrapper
	UINT16 ui16;
  UINT8 lqi;
  BOOL bSEC;

	ui16 = *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_OFFSET_LinkQualityIndicator_Flags >> 1]);
  lqi = (ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_LinkQualityIndicator_MASK);
  bSEC = ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_FLAGS_SEC ? TRUE : FALSE;

	if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    ECA_AppEmu_mgh.stats.unexpected_data_indicate++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_222, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_data_indicate);
  }
	else
	{
		UINT8 * pui8;
	  pui8 = (UINT8 *) pmsg;
		UINT8 * payload  = NULL;
		UINT32 payloadId  = 0; 
		UINT16 payloadType = 0;
		int payloadLength = 0;
		IPv6MainHeader ipWrapper;
		UINT32 shortAddress;

		if (microIP == 0)
		{
			//
			// Get the ip header and skip the status info (in bytes)
			//
			ipWrapper.Decode(pui8, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE);
			//
			// get a pointer to the message..
			//
			payload = ipWrapper.GetData(&pui8[2]);
			payloadType = *((UINT16 *) &payload[0]);
			payloadId   = *((UINT32 *) &payload[2]);
			payloadLength = msg_len - ipWrapper.IP6vHeaderSize - 2;
			shortAddress = 0xFFFF;
		}
		else
		{
			//
			// PLC data always sends address type 2 = IPv6 address
			//
			payload = &pui8[2+2+2+2+16];
		  shortAddress = *(UINT16 *) &pui8[2+2+2+2+14];

			payloadType = *(UINT16 *) &payload[0];
			payloadId = *(UINT32 *) &payload[2];
			payloadLength = msg_len - (2+2+2+16) - 2;
		}
		//
		// get a pointer to the message..
		//

		g3MessageCounter++;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Service Node messages received and sent: %d\n"), g3MessageCounter);

		if (payloadType == G3_Network::SystemReadMsg) // system time read
		{
			SendNodeResponse(ipWrapper, shortAddress, payloadType, payloadId, payloadLength, 36);
		}
		else if (payloadType == G3_Network::EngeryRegisterReadMsg) // energy register read
		{
			SendNodeResponse(ipWrapper, shortAddress, payloadType, payloadId, payloadLength, 31);
		}
		else if (payloadType == G3_Network::AccumulatedInfomationMsg1) // Accumulated Info read step 1
		{
			SendNodeResponse(ipWrapper, shortAddress, payloadType, payloadId, payloadLength, 138);
		}
		else if (payloadType == G3_Network::AccumulatedInfomationMsg2) // Accumulated Info read step 2
		{
			SendNodeResponse(ipWrapper, shortAddress, payloadType, payloadId, payloadLength, 52);
		}
		else if (payloadType == G3_Network::PPDUTestMsg) // ppdu length testing..
		{
			SendNodeResponse(ipWrapper, shortAddress, payloadType, payloadId, payloadLength, -1);
		}
		else
		{
			//
			// all other test msg are assumed to be PPDU testing.
			// the test id is bumped by one for each test.
			// the same size message is echoed back.
			//
			SendNodeResponse(ipWrapper, shortAddress, payloadType, payloadId, payloadLength, -1);
		}
	}
	MyLocalFree(pmsg);
}

void SendNodeResponse(IPv6MainHeader ipWrapper, UINT16 shortAddress, UINT16 payloadType, UINT32 payloadId, int oldLength, int length)
{
	ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_SystemTimeRead]++;

	/* allocate the UP_MSG packet */

	int payloadLength = length;
	if (length == -1)
	{
		payloadLength = oldLength;
	}
	int messageLength;
	int payloadOffset;
	UINT8 * payload = NULL;
	UINT8 * pup_msg = NULL;

	if (shortAddress == 0xFFFF)
	{
		ipWrapper.SwapAddresses();
		ipWrapper.SetPayloadLength(payloadLength);

		messageLength = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + IPv6MainHeader::IP6vHeaderSize + payloadLength;
		payloadOffset = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + IPv6MainHeader::IP6vHeaderSize;
		pup_msg = (UINT8 *) LocalAlloc(LPTR, messageLength);
		payload = & pup_msg[payloadOffset];

		ipWrapper.Encode(pup_msg, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE);
	}
	else
	{
		messageLength = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + 2+2+2+2 + payloadLength;
		payloadOffset = HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + 2+2+2+2;
		pup_msg = (UINT8 *) LocalAlloc(LPTR, messageLength);
		payload = & pup_msg[payloadOffset];
		*(UINT16 *)&pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE] = 0; //port
		*(UINT16 *)&pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE+2] = 0; //port
		*(UINT16 *)&pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE+4] = 0; //address type
		*(UINT16 *)&pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE+6] = shortAddress; // short address
	}

	if (pup_msg == NULL)
	{
		ECA_OutOfMemory();
	}
	else
	{
		UINT16 flags = ECA_AppEmu_mgh.rx_nsdu_handle++;
		flags |= HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_FLAGS_DISCOVERY_ROUTE_ENABLED;

		*(UINT16 *) &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags] = flags;

		/* build packet */
		ECA_AppEmu_BuildPayload((HCT_MSG_BUFFER_t *)payload, payloadLength, NULL, 0);
		//
		// Send back packet type and id
		//
		*(UINT16 *) &payload[0] = payloadType;
		*(UINT32 *) &payload[2] = payloadId;
		memcpy(&payload[6], macAddress, sizeof(macAddress));

		/* send packet */

		ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_SystemTimeRead]++;
		ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;

		HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, messageLength, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);

		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Sent Service Node Response Type:%d, Id: %d, Length:%d\n"), __FUNCTION__, payloadType, payloadId, payloadLength);
	}
}

void ProcessEmeterEmulationMessage_Concentrator(HCT_MSG_BUFFER_t * pmsg, UINT16 msg_len)
{
	// Remove the IPv6 wrapper
	UINT16 ui16;
  UINT8 lqi;
  BOOL bSEC;

	ui16 = *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_OFFSET_LinkQualityIndicator_Flags >> 1]);
  lqi = (ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_LinkQualityIndicator_MASK);
  bSEC = ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_FLAGS_SEC ? TRUE : FALSE;

	if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    ECA_AppEmu_mgh.stats.unexpected_data_indicate++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_222, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_data_indicate);
  }
	else
	{
		ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_Loop_Start]++;

		UINT8 * pui8;
	  pui8 = (UINT8 *) pmsg;
		//
		// Get the ip header and skip the status info (in bytes)
		//
		UINT8 msgMacAddress[8];
		if (microIP == 0)
		{
			IPv6MainHeader ipWrapper(&pui8[2]);
			memcpy(msgMacAddress, &(ipWrapper.GetData(&pui8[2])[6]), 8);
		}
		else
		{
			memcpy(msgMacAddress, &pui8[30], 8);
		}
		G3_Network * pNetwork = G3_Network::FindMacAddress(msgMacAddress, pui8);
		if (pNetwork == NULL)
		{
			// xraf add ip address...
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Service Node not found: \n"), __FUNCTION__);
			//
			// release the memory since the thread is not..
			//
			MyLocalFree(pmsg);
		}
	}
	//
	// The memory is leased by the thread;
	//	MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_MAC
*
* DESCRIPTION:
*       MAC Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_MAC(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len)
{
  UINT16 pkthdr[(ECA_APPEMU_DW_MSG_HEADER_LEN + 1) >> 1];
  UINT16 connHandle;
  UINT32 ui32;
  ECA_APPEMU_STEP_t step;
  UINT16 *pdata;

  if (msg_len == HCT_MSG_RPY_DATA_TRANSFER_SIZE)
  {
    HostAppEMU_ReceivedSyncMsg();
    return;
  }

  connHandle = *(WORD *) &pmsg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Conn_Handle >> 1)];
  if (connHandle != ECA_AppEmu_mgh.connHandle)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("%S: Discarding packet, expected Connection Handle %#x, received packet for handle %#x.\n"), __FUNCTION__,
          ECA_AppEmu_mgh.connHandle, connHandle);
    MyLocalFree(pmsg);
    return;
  }

  pdata = &pmsg[HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Data >> 1];
  msg_len -= HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE;

  /* copy header to local buffer */
  memcpy(&pkthdr[0], pdata, ECA_APPEMU_DW_MSG_HEADER_LEN);

  /*
  ** parse DW_MSG
  */

  /* verify prefix */
  if (pkthdr[0] != ECA_APPEMU_DW_MSG_SIGNATURE)
  {
    ECA_AppEmu_mgh.stats.bad_recv_pkt_signature++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_216, 1, (UINT32) ECA_AppEmu_mgh.stats.bad_recv_pkt_signature);
    MyLocalFree(pmsg);
    return;
  }

  /* verify provider */
  ui32 = ((UINT32) pkthdr[1] << 16);
  ui32 |= (UINT32) pkthdr[2];
  if ((ui32 & ECA_APPEMU_PROVIDER_ID_MASK) != ECA_APPEMU_PROVIDER_ID)
  {
    ECA_AppEmu_mgh.stats.bad_recv_pkt_provider++;
//    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_217, 1, (UINT32) ECA_AppEmu_mgh.stats.bad_recv_pkt_provider);
#ifndef ECA_APPEMU_ALWAYS_REPLY
//    MyLocalFree(pmsg);
//    return;
#endif // #ifndef ECA_APPEMU_ALWAYS_REPLY
  }

  /* step (ECA_APPEMU_STEP_t) */
  step = (ECA_APPEMU_STEP_t) (((pkthdr[2] & 0xff00) >> 8) - '0');

  /* prefill the UP_MSG header reusing the DW_MSG header */
  proj_assert(ECA_APPEMU_UP_MSG_HEADER_LEN == ECA_APPEMU_DW_MSG_HEADER_LEN);
  pkthdr[0] = ECA_APPEMU_UP_MSG_SIGNATURE;

  /* process the packet */
  if (step != ECA_AppEmu_mgh.curr_step)
  {
    ECA_AppEmu_mgh.stats.err_rx_out_of_seq++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_218, 3,
        (UINT32) ECA_AppEmu_mgh.curr_step,
        (UINT32) step,
        (UINT32) ECA_AppEmu_mgh.stats.err_rx_out_of_seq
    );
    if (step >= ECA_APPEMU_STEP_Loop_Start && step <= ECA_APPEMU_STEP_Loop_End)
    {
      ECA_AppEmu_mgh.curr_step = step;
    }
  }

  pkthdr[0] = ECA_APPEMU_UP_MSG_SIGNATURE;

  switch (step)
  {
  case ECA_APPEMU_STEP_SystemTimeRead:
    ECA_AppEmu_processSystemTimeRead(pdata, msg_len, &pkthdr[0]);
    break;
  case ECA_APPEMU_STEP_EnergyRegisterRead:
    ECA_AppEmu_processEnergyRegisterRead(pdata, msg_len, &pkthdr[0]);
    break;
  case ECA_APPEMU_STEP_AccumulatedInformationRead:
    ECA_AppEmu_processAccumulatedInformationRead(pdata, msg_len, &pkthdr[0]);
    break;
  case ECA_APPEMU_STEP_PPDU_LengthTesting:
    ECA_AppEmu_processLengthTesting(pdata, msg_len, &pkthdr[0]);
    break;
  default:
    ECA_AppEmu_mgh.stats.invalid_recv_pkt_step++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_219, 3,
        (UINT32) step,
        (UINT32) ECA_AppEmu_mgh.stats.invalid_recv_pkt_step
    );
    break;
  }

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_setNextStep
*
* DESCRIPTION:
*       Sets the next expected step.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_setNextStep(void)
{
  int step;

  step = (int) ECA_AppEmu_mgh.curr_step ;
  step += (int) ECA_APPEMU_STEP_Loop_Inc;
  ECA_AppEmu_mgh.curr_step = (ECA_APPEMU_STEP_t) step;
  if (ECA_AppEmu_mgh.curr_step > ECA_APPEMU_STEP_Loop_End)
  {
    ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_Loop_Start;
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_saveSerialNumber
*
* DESCRIPTION:
*       Saves serial number in our context.
*
* Return Value:       none
* Input Parameters:   UINT16 *    pointer to serial number, as UINT16[3]
* Output Parameters:  ECA_AppEmu_mgh.sSerial  contains ASCII serial number
******************************************************************************/
void ECA_AppEmu_saveSerialNumber(UINT16 *pSerial)
{
#if 0
  /* convert to little endian */
  ECA_AppEmu_mgh.serial_number[0] = (pSerial[0] >> 8) | (pSerial[0] << 8);
  ECA_AppEmu_mgh.serial_number[1] = (pSerial[1] >> 8) | (pSerial[1] << 8);
  ECA_AppEmu_mgh.serial_number[2] = (pSerial[2] >> 8) | (pSerial[2] << 8);
#else
  memcpy(&ECA_AppEmu_mgh.serial_number[0], pSerial, sizeof(ECA_AppEmu_mgh.serial_number));
#endif
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_buildSerialNumber
*
* DESCRIPTION:
*       Builds SERIAL, in the form of "TXN00000".
*
* Return Value:       none
* Input Parameters:   UINT16 *    pointer to serial number, as UINT16[3]
* Output Parameters:  ECA_AppEmu_mgh.sSerial  contains ASCII serial number
******************************************************************************/
void ECA_AppEmu_buildSerialNumber(UINT16 *pSerial)
{
#if 0
  int offset;
  UINT32 ui32Serial;
  UINT16 buf[(ECA_APPEMU_SERIAL_LEN - ECA_APPEMU_PROVIDER_ID_LEN) >> 1];
  int i;

  /* convert to little endian */
  ECA_AppEmu_mgh.serial_number[0] = (ECA_AppEmu_mgh.serial_number[0] >> 8) | (ECA_AppEmu_mgh.serial_number[0] << 8);
  ECA_AppEmu_mgh.serial_number[1] = (ECA_AppEmu_mgh.serial_number[1] >> 8) | (ECA_AppEmu_mgh.serial_number[1] << 8);
  ECA_AppEmu_mgh.serial_number[2] = (ECA_AppEmu_mgh.serial_number[2] >> 8) | (ECA_AppEmu_mgh.serial_number[2] << 8);

  ui32Serial = ( (UINT32)(pSerial[1]) << 16) + pSerial[2];   // comes to us as UINT16[3]

  ECA_AppEmu_UINT32_to_ZeroAscii(ui32Serial, &buf[0], ECA_APPEMU_SERIAL_LEN - ECA_APPEMU_PROVIDER_ID_LEN);

  ECA_AppEmu_mgh.sSerial[0] = (ECA_APPEMU_PROVIDER_ID >> 16);
  ECA_AppEmu_mgh.sSerial[1] = (ECA_APPEMU_PROVIDER_ID & 0x00ff);
  for (offset = ECA_APPEMU_PROVIDER_ID_LEN, i = 0; offset < ECA_APPEMU_SERIAL_LEN; offset++, i++)
  {
    ((UINT8 *) &ECA_AppEmu_mgh.sSerial)[offset] = ((UINT8 *) &buf)[i];
  }
#else
  /* new serial number is now for this, just copy to existing buffer for now, cleanup later */
  memcpy(&ECA_AppEmu_mgh.sSerial, &ECA_AppEmu_mgh.serial_number, ECA_APPEMU_PROVIDER_ID_LEN + ECA_APPEMU_SERIAL_LEN);
#endif
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEMU_buildDeviceIdentifier
*
* DESCRIPTION:
*       Builds Device Identifier.
*
* Return Value:       none
* Input Parameters:   UINT16 *    pointer to serial number, as UINT16[3]
* Output Parameters:  ECA_AppEmu_mgh.DeviceIdentifier contains Device Identifier
******************************************************************************/
void ECA_AppEMU_buildDeviceIdentifier(UINT8 *pDeviceIdentifier)
{
  memset(pDeviceIdentifier, 0, sizeof(ECA_AppEmu_mgh.LLC_DeviceIdentifier));
  memcpy(pDeviceIdentifier, &ECA_AppEmu_mgh.sSerial[0], sizeof(ECA_AppEmu_mgh.sSerial));
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_CheckPayload
*
* DESCRIPTION:
*       Checks payload.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
*                     UINT16      header length
* Output Parameters:  none
******************************************************************************/
BOOL ECA_AppEmu_CheckPayload(HCT_MSG_BUFFER_t *msg, UINT16 len, UINT16 hdrlen)
{
  int c;
  UINT8 *p;

  /* remove header */
  p = ((UINT8 *) msg + hdrlen);

  len -= hdrlen;

  for (c = 'A'; len > 0; len--, p++)
  {
    if (c != *p)
    {
      return FALSE;
    }
    c++;
    if (c > 'Z')
    {
      c = 'A';
    }
  }

  return TRUE;
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_BuildPayload
*
* DESCRIPTION:
*       Builds payload.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
*                     UINT16 *    packet header
*                     UINT16      header length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_BuildPayload(HCT_MSG_BUFFER_t *msg, UINT16 len, UINT16 *ppkthdr, UINT16 hdrlen)
{
  UINT8 *p;
  int c;

  p = (UINT8 *) msg;

  if (ppkthdr && hdrlen != 0)
  {
    memcpy(p, ppkthdr, hdrlen);   // copy packet header
  }

  p += hdrlen;

  len -= hdrlen;

  for (c = 'A'; len > 0; len--, p++)
  {
    *p = c;
    c++;
    if (c > 'Z')
    {
      c = 'A';
    }
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_UINT32_to_ZeroAscii
*
* DESCRIPTION:
*       Function UINT16 to zero filled ascii.
*
* Return Value:       none
* Input Parameters:   UINT32      value
*                     UINT16 *    buffer
*                     int         total length (of octets)
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_UINT32_to_ZeroAscii(UINT32 v, UINT16 *pbuf, int length_octet)
{
  UINT16 ui16;
  UINT16 asciiui16;

  length_octet >>= 1; // dealing with UINT16 buffer

  for (length_octet--; length_octet >= 0; length_octet--)
  {
    ui16 = v % 10;
    v /= 10;
    asciiui16 = ui16 + '0';

    pbuf[length_octet] = ((asciiui16 & 0xff) << 8);

    ui16 = v % 10;
    v /= 10;
    asciiui16 = ui16 + '0';

    pbuf[length_octet] |= ((asciiui16 & 0xff));
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processSystemTimeRead
*
* DESCRIPTION:
*       Process a ECA_APPEMU_STEP_SystemTimeRead packet.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
*                     UINT16 *    filled DW packet header
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processSystemTimeRead(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr)
{
  HCT_MSG_BUFFER_t *pup_msg;
  UINT16 uplen;
  BOOL bReply;
  SINT16 pktlen;

  uplen = 36;

  pktlen = (SINT16) msglen;
  if (pktlen != 24)
  {
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_227, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_SystemTimeRead,
        (UINT32) pktlen,
        (UINT32) 24
    );
  }

  /* validate packet */
  if (!ECA_AppEmu_CheckPayload(msg, 24, ECA_APPEMU_DW_MSG_HEADER_LEN))
  {
    ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_SystemTimeRead]++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_220, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_SystemTimeRead,
        (UINT32) 24,
        (UINT32) ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_SystemTimeRead]
    );
#ifndef ECA_APPEMU_ALWAYS_REPLY
    bReply = FALSE;
#else // #ifndef ECA_APPEMU_ALWAYS_REPLY
    bReply = TRUE;
#endif // #ifndef ECA_APPEMU_ALWAYS_REPLY
  }
  else
  {
    bReply = TRUE;
  }

  /* update state before we proceed in case we block and we get another packet */
  ECA_AppEmu_mgh.NextDWLen = 0;
  ECA_AppEmu_setNextStep();

  if (bReply)
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_224, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_SystemTimeRead,
        (UINT32) 24,
        (UINT32) uplen
    );

    ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_SystemTimeRead]++;

    /* allocate the UP_MSG packet */
    pup_msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + uplen);
    if (pup_msg)
    {
      *(UINT16 *) &pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Conn_Handle >> 1)] = ECA_AppEmu_mgh.connHandle;

      /* build packet */
      ECA_AppEmu_BuildPayload(&pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Data >> 1)], uplen, ppkthdr, ECA_APPEMU_UP_MSG_HEADER_LEN);

      /* send packet */
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_SystemTimeRead]++;
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_None]++;
      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;
      HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + uplen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
    }
    else
    {
      ECA_OutOfMemory();
    }
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processEnergyRegisterRead
*
* DESCRIPTION:
*       Process a ECA_APPEMU_STEP_processEnergyRegisterRead packet.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
*                     UINT16 *    filled DW packet header
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processEnergyRegisterRead(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr)
{
  HCT_MSG_BUFFER_t *pup_msg;
  UINT16 uplen;
  BOOL bReply;
  SINT16 pktlen;

  uplen = 31;

  pktlen = (SINT16) msglen;
  if (pktlen != 24)
  {
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_227, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_EnergyRegisterRead,
        (UINT32) pktlen,
        (UINT32) 24
    );
  }

  /* validate packet */
  if (!ECA_AppEmu_CheckPayload(msg, 24, ECA_APPEMU_DW_MSG_HEADER_LEN))
  {
    ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_EnergyRegisterRead]++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_220, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_EnergyRegisterRead,
        (UINT32) pktlen,
        (UINT32) ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_EnergyRegisterRead]
    );
#ifndef ECA_APPEMU_ALWAYS_REPLY
    bReply = FALSE;
#else // #ifndef ECA_APPEMU_ALWAYS_REPLY
    bReply = TRUE;
#endif // #ifndef ECA_APPEMU_ALWAYS_REPLY
  }
  else
  {
    bReply = TRUE;
  }

  /* update state before we proceed in case we block and we get another packet */
  ECA_AppEmu_mgh.NextDWLen = 0;
  ECA_AppEmu_setNextStep();

  if (bReply)
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_224, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_EnergyRegisterRead,
        (UINT32) pktlen,
        (UINT32) uplen
    );

    ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_EnergyRegisterRead]++;

    /* allocate the UP_MSG packet */
    pup_msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + uplen);
    if (pup_msg)
    {
      *(UINT16 *) &pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Conn_Handle >> 1)] = ECA_AppEmu_mgh.connHandle;

      /* build packet */
      ECA_AppEmu_BuildPayload(&pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Data >> 1)], uplen, ppkthdr, ECA_APPEMU_UP_MSG_HEADER_LEN);

      /* send packet */
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_EnergyRegisterRead]++;
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_None]++;
      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;
      HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + uplen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
    }
    else
    {
      ECA_OutOfMemory();
    }
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processAccumulatedInformationRead
*
* DESCRIPTION:
*       Process a ECA_APPEMU_STEP_processAccumulatedInformationRead packet.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
*                     UINT16 *    filled DW packet header
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processAccumulatedInformationRead(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr)
{
  HCT_MSG_BUFFER_t *pup_msg;
  UINT16 uplen;
  BOOL bReply;
  SINT16 pktlen;

  if (ECA_AppEmu_mgh.NextDWLen == 75)
  {
    ECA_AppEmu_mgh.NextDWLen = 16;
  }
  else
  {
    ECA_AppEmu_mgh.NextDWLen = 75;
  }

  pktlen = (SINT16) msglen;
  if (pktlen != ECA_AppEmu_mgh.NextDWLen)
  {
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_227, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_AccumulatedInformationRead,
        (UINT32) pktlen,
        (UINT32) ECA_AppEmu_mgh.NextDWLen
    );

    ECA_AppEmu_mgh.NextDWLen = pktlen;
  }

  if (ECA_AppEmu_mgh.NextDWLen == 75)
  {
    uplen = 138;
  }
  else
  {
    uplen = 52;
  }

  /* validate packet */
  if (!ECA_AppEmu_CheckPayload(msg, ECA_AppEmu_mgh.NextDWLen, ECA_APPEMU_DW_MSG_HEADER_LEN))
  {
    ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_AccumulatedInformationRead]++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_220, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_AccumulatedInformationRead,
        (UINT32) ECA_AppEmu_mgh.NextDWLen,
        (UINT32) ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_AccumulatedInformationRead]
    );
#ifndef ECA_APPEMU_ALWAYS_REPLY
    ECA_AppEmu_mgh.NextDWLen = 0;
    bReply = FALSE;
#else // #ifndef ECA_APPEMU_ALWAYS_REPLY
    bReply = TRUE;
#endif // #ifndef ECA_APPEMU_ALWAYS_REPLY
  }
  else
  {
    bReply = TRUE;
  }

  if (ECA_AppEmu_mgh.NextDWLen == 16)
  {
    /* update state before we proceed in case we block and we get another packet */
    ECA_AppEmu_mgh.NextDWLen = 0;
    ECA_AppEmu_setNextStep();
  }

  if (bReply)
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_224, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_AccumulatedInformationRead,
        (UINT32) pktlen,
        (UINT32) uplen
    );

    ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_AccumulatedInformationRead]++;

    /* allocate the UP_MSG packet */
    pup_msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + uplen);
    if (pup_msg)
    {
      *(UINT16 *) &pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Conn_Handle >> 1)] = ECA_AppEmu_mgh.connHandle;

      /* build packet */
      ECA_AppEmu_BuildPayload(&pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Data >> 1)], uplen, ppkthdr, ECA_APPEMU_UP_MSG_HEADER_LEN);

      /* send packet */
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_AccumulatedInformationRead]++;
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_None]++;
      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;
      HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + uplen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
    }
    else
    {
      ECA_OutOfMemory();
    }
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processLengthTesting
*
* DESCRIPTION:
*       Process a ECA_APPEMU_STEP_processLengthTesting packet.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
*                     UINT16 *    filled DW packet header
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processLengthTesting(HCT_MSG_BUFFER_t *msg, UINT16 msglen, UINT16 *ppkthdr)
{
  HCT_MSG_BUFFER_t *pup_msg;
  SINT16 pktlen;
  BOOL bReply;
  UINT16 ui16;

  pktlen = (SINT16) msglen;

  /* find the offset in the table */
  for (ui16 = 0; ui16 < PPDU_LENGTH_TESTING_VALUES; ui16++)
  {
    if (PPDU_LengthTestingTable[ECA_AppEmu_mgh.PPDU_LengthTesting_Offset][ui16] == pktlen)
    {
      break;
    }
  }
  if (ui16 != ECA_AppEmu_mgh.PPDU_LengthTesting_Index)
  {
    if (PPDU_LengthTestingTable[ECA_AppEmu_mgh.PPDU_LengthTesting_Offset][ECA_AppEmu_mgh.PPDU_LengthTesting_Index] != -1)
    {
      if (ui16 < PPDU_LENGTH_TESTING_VALUES)
      {
        DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_226, 5,
            (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
            (UINT32) ECA_APPEMU_STEP_PPDU_LengthTesting,
            (UINT32) pktlen,
            (UINT32) ECA_AppEmu_mgh.PPDU_LengthTesting_Index,
            (UINT32) ui16
        );
      }
      else
      {
        DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_248, 4,
            (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
            (UINT32) ECA_APPEMU_STEP_PPDU_LengthTesting,
            (UINT32) pktlen,
            (UINT32) ECA_AppEmu_mgh.PPDU_LengthTesting_Index
        );
      }
    }
    ECA_AppEmu_mgh.PPDU_LengthTesting_Index = ui16;
  }

  /* validate packet */
  if (!ECA_AppEmu_CheckPayload(msg, pktlen, ECA_APPEMU_DW_MSG_HEADER_LEN))
  {
    ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_PPDU_LengthTesting]++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_220, 4,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_PPDU_LengthTesting,
        (UINT32) pktlen,
        (UINT32) ECA_AppEmu_mgh.stats.err_recv_pkt[ECA_APPEMU_STEP_PPDU_LengthTesting]
    );
#ifndef ECA_APPEMU_ALWAYS_REPLY
    bReply = FALSE;
#else // #ifndef ECA_APPEMU_ALWAYS_REPLY
    bReply = TRUE;
#endif // #ifndef ECA_APPEMU_ALWAYS_REPLY
  }
  else
  {
    bReply = TRUE;
  }

  if (bReply)
  {
    DIAG_printf(DIAG_LOG_LEVEL_APPEMU, DIAG_LOG_215, 5,
        (UINT32) ECA_AppEmu_mgh.stats.ttl_connect_attempts,
        (UINT32) ECA_APPEMU_STEP_PPDU_LengthTesting,
        (UINT32) pktlen,
        (UINT32) ECA_AppEmu_mgh.PPDU_LengthTesting_Offset,
        (UINT32) ECA_AppEmu_mgh.PPDU_LengthTesting_Index
    );

    ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_PPDU_LengthTesting]++;

    /* allocate the UP_MSG packet */
    pup_msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + pktlen);

    if (pup_msg)
    {
      *(UINT16 *) &pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Conn_Handle >> 1)] = ECA_AppEmu_mgh.connHandle;

      /* build packet */
      ECA_AppEmu_BuildPayload(&pup_msg[(HCT_MSG_REQ_DATA_TRANSFER_MAC_OFFSET_Data >> 1)], pktlen, ppkthdr, ECA_APPEMU_UP_MSG_HEADER_LEN);

      /* send packet */
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_PPDU_LengthTesting]++;
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_None]++;
      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;
      HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, HCT_MSG_REQ_DATA_TRANSFER_MAC_SIZE + pktlen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
    }
    else
    {
      ECA_OutOfMemory();
      return;
    }
  }

  while (ECA_AppEmu_mgh.PPDU_LengthTesting_Index < PPDU_LENGTH_TESTING_VALUES)
  {
    ECA_AppEmu_mgh.PPDU_LengthTesting_Index++;
    if (PPDU_LengthTestingTable[ECA_AppEmu_mgh.PPDU_LengthTesting_Offset][ECA_AppEmu_mgh.PPDU_LengthTesting_Index] != -1)
    {
      break;
    }
  }

  if (ECA_AppEmu_mgh.PPDU_LengthTesting_Index >= PPDU_LENGTH_TESTING_VALUES)
  {
    /* update state before we proceed in case we block and we get another packet */
    ECA_AppEmu_mgh.PPDU_LengthTesting_Index = 0;
    ECA_AppEmu_setNextStep();
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DATA_handler
*
* DESCRIPTION:
*       DATA handler.
*
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_DATA_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen, BOOL bRPY)
{
  ECF_TASK_s ecf_task;

  ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_None]++;
  ecf_task.task_type = ECA_TASK_PACKET_RECEIVED;
  ecf_task.bRPY = bRPY;
  ecf_task.u.s.msg = msg;
  ecf_task.u.s.msglen = msglen;

  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Get_System_Info_Handler
*
* DESCRIPTION:
*       HCT_MSG_TYPE_GET_SYSTEM_INFO handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Get_System_Info_Handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT16 ui16;

  ECA_AppEmu_saveSerialNumber(&msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Serial_Number >> 1]);

  /* save the values that we need to preserve for LOAD_SYSTEM_CONFIG */
  memcpy(&ECA_AppEmu_mgh.save_Serial_Number_Len, &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Serial_Number_Len >> 1], HCT_MSG_RPY_GET_SYSTEM_INFO_SIZE_Serial_Number_Len);
  memcpy(&ECA_AppEmu_mgh.save_Serial_Number[0], &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Serial_Number >> 1], HCT_MSG_RPY_GET_SYSTEM_INFO_SIZE_Serial_Number);
  memcpy(&ECA_AppEmu_mgh.save_EUI[0], &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_EUI >> 1], HCT_MSG_RPY_GET_SYSTEM_INFO_SIZE_EUI);
  ui16 = (*(UINT16 *) &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Device_Mode_Type >> 1]);
  ui16 &= HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_MASK;
  ECA_AppEmu_mgh.save_Device_Mode = (UINT8) (((*(UINT16 *) &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Device_Mode_Type >> 1]) & HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_MASK) >> 8);

  ui16 = *((UINT16 *) &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Flags >> 1]);
  ECA_AppEmu_mgh.save_HostPort = HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_GET_DATA_PORT(ui16);
  ECA_AppEmu_mgh.save_DiagPort = HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_GET_DIAG_PORT(ui16);


  if ((*(UINT16 *) &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Flags >> 1]) & HCT_MSG_GET_SYSTEM_INFO_FLAGS_USE_RPY)
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_USE_RPY;
  }
  else
  {
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_USE_RPY;
  }
  if ((*(UINT16 *) &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Flags >> 1]) & HCT_MSG_GET_SYSTEM_INFO_FLAGS_AUTO_MODE)
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_LLC_AUTO;
  }
  else
  {
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_LLC_AUTO;
  }
  if ((*(UINT16 *) &msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Flags >> 1]) & HCT_MSG_GET_SYSTEM_INFO_FLAGS_IP_FLAG)
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_IPv6;
  }
  else
  {
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_IPv6;
  }

  ui16 = *(UINT16 *) (&msg[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_MAC_Flags_Dflt_Security_Profile >> 1]);
  if (ui16 & HCT_MSG_GET_SYSTEM_INFO_MAC_FLAGS_DFLT_ARQ)
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_MAC_ARQ;
  }
  else
  {
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_MAC_ARQ;
  }

  ECA_AppEmu_Get_System_Info_Handler_IP(&ECA_AppEmu_mgh, msg, msglen);

  ECA_AppEmu_Get_System_Info_Handler_LLC(&ECA_AppEmu_mgh, msg, msglen);

  ECA_AppEmu_Get_System_Info_Dump(msg, msglen);

  MyLocalFree(msg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Set_Info_Handler
*
* DESCRIPTION:
*       HCT_MSG_TYPE_SET_INFO handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Set_Info_Handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT16 mask;
  UINT16 status;

  if (msglen != HCT_MSG_RPY_SET_INFO_SIZE)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! HCT_MSG_TYPE_SET_INFO reply length wrong, expected %d received %d.\n"),
        __FUNCTION__, HCT_MSG_RPY_SET_INFO_SIZE, msglen);
  }
  else
  {
    status = *((UINT16 *) &msg[HCT_MSG_RPY_SET_INFO_OFFSET_Status >> 1]);
    mask = HCT_MSG_STATUS_PHY_ERROR_CODE_GROUP|HCT_MSG_STATUS_MAC_ERROR_CODE_GROUP|HCT_MSG_STATUS_CL_ERROR_CODE_GROUP;
    if ((status & mask) == 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: HCT_MSG_TYPE_SET_INFO returned success.\n"), __FUNCTION__);
    }
    else
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! HCT_MSG_TYPE_SET_INFO returned error %#x.\n"),
          __FUNCTION__, status);
    }
  }

  MyLocalFree(msg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_SETUP_ALARM_Handler
*
* DESCRIPTION:
*       HCT_MSG_TYPE_SETUP_ALARM handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_SETUP_ALARM_Handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT16 Status;

  Status = *((UINT16 *) &msg[HCT_MSG_RPY_SETUP_ALARM_OFFSET_Status >> 1]);
  if (Status == 0)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: SET_ALARM returned success.\n"), __FUNCTION__);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR %#x from Set Alarm!\n"), __FUNCTION__, Status);
  }
  MyLocalFree(msg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_ALARM_TYPE_to_String
*
* DESCRIPTION:
*       Returns a string description of the given Alarm Type.
*
* Return Value:       const char *  Alarm Type string
* Input Parameters:   UINT16        Alarm Type
* Output Parameters:  none
******************************************************************************/
const char *ECA_AppEmu_ALARM_TYPE_to_String(UINT16 AlarmType)
{
  const static UINT16 AlarmTypes[] = 
  {
    HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED,
    HCT_MSG_ALARM_TYPE_CONNECT_REQUEST,
    HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION,
    HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_STARTED,
    HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_COMPLETE,
    HCT_MSG_ALARM_TYPE_CONNECT_COMPLETE,
    HCT_MSG_ALARM_TYPE_PHY_RX_PPDU_CRC_FAIL,
    HCT_MSG_ALARM_TYPE_PHY_RX_HDR_SYNTAX_ERR,
    HCT_MSG_ALARM_TYPE_MAC_PROMOTED_SWITCH_NODE,
    HCT_MSG_ALARM_TYPE_MAC_DEMOTED_SERVICE_NODE,
    HCT_MSG_ALARM_TYPE_CL,
    HCT_MSG_ALARM_TYPE_GENERAL_ERROR
  };
  const static char *AlarmTypesStrings[] = 
  {
    "HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED",
    "HCT_MSG_ALARM_TYPE_CONNECT_REQUEST",
    "HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION",
    "HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_STARTED",
    "HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_COMPLETE",
    "HCT_MSG_ALARM_TYPE_CONNECT_COMPLETE",
    "HCT_MSG_ALARM_TYPE_PHY_RX_PPDU_CRC_FAIL",
    "HCT_MSG_ALARM_TYPE_PHY_RX_HDR_SYNTAX_ERR",
    "HCT_MSG_ALARM_TYPE_MAC_PROMOTED_SWITCH_NODE",
    "HCT_MSG_ALARM_TYPE_MAC_DEMOTED_SERVICE_NODE",
    "HCT_MSG_ALARM_TYPE_CL",
    "HCT_MSG_ALARM_TYPE_GENERAL_ERROR"
  };

  int i;

  for (i = 0; i < sizeof(AlarmTypes) / sizeof(AlarmTypes[0]); i++)
  {
    if (AlarmTypes[i] == AlarmType)
    {
      return AlarmTypesStrings[i];
    }
  }

  return (const char *) "unknown";
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_ALARM_Handler
*
* DESCRIPTION:
*       HCT_MSG_TYPE_ALARM handler.
*
* Return Value:       none
* Input Parameters:   UINT16 * message header
*                     HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_ALARM_Handler(UINT16 *pMsgHdr, HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT16 Alarm_Type;
  UINT16 Alarm_Length;
  UINT16 *Alarm_Data;
  ECF_TASK_s ecf_task;

  Alarm_Type = *((UINT16 *) &msg[HCT_MSG_ALARM_OFFSET_Alarm_Type >> 1]);
  Alarm_Length = *((UINT16 *) &msg[HCT_MSG_ALARM_OFFSET_Alarm_Length >> 1]);
  Alarm_Data = ((UINT16 *) &msg[HCT_MSG_ALARM_OFFSET_Alarm_Data >> 1]);

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received alarm %#x: %S.\n"), Alarm_Type, ECA_AppEmu_ALARM_TYPE_to_String(Alarm_Type));

  switch (Alarm_Type)
  {
  case HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED:

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: NETWORK_DEREGISTERED (%#x) len, expected %d, received %d.\n"),
            __FUNCTION__, Alarm_Type, HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED_SIZE, Alarm_Length);

    if (Alarm_Length != HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  NETWORK_DEREGISTERED (%#x) invalid len, expected %d, received %d.\n"),
            __FUNCTION__, Alarm_Type, HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED_SIZE, Alarm_Length);
    }
    ECA_NETWORK_UNREGISTER_indicate();

    msglen = HCT_MSG_RPY_ALARM_TYPE_NETWORK_DEREGISTERED_SIZE;
    break;

  case HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION:
    if (Alarm_Length < HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  DISCONNECT_INDICATION (%#x) invalid len, expected at least %d, received %d.\n"),
            __FUNCTION__, Alarm_Type, HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE, Alarm_Length);
    }

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: DISCONNECT_INDICATION for Conn Type %#x.\n"),
              __FUNCTION__, Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Type >> 1]);

//    switch (Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Type >> 1])
    switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
    {
#ifdef APPEMU_IEC432
    case HCT_MSG_CONNECT_CONN_TYPE_CL_IEC432:
      ECA_AppEmu_DISCONNECT_indicate(
        Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle >> 1],
        0
      );
      msglen = HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_SIZE;
      break;
#endif // #ifdef APPEMU_IEC432
//    case HCT_MSG_CONNECT_CONN_TYPE_MAC:
    case HostAppEMU_MAC:
      if (Alarm_Length != HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE + HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION_CONN_TYPE_MAC_SIZE)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  MAC DISCONNECT_INDICATION (%#x) invalid len, expected %d, received %d.\n"),
              __FUNCTION__, Alarm_Type, HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE + HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE, Alarm_Length);
      }

      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: MAC DISCONNECT_INDICATION, Conn Hdl %#x, reason %#x.\n"),
            __FUNCTION__,
            Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle >> 1],
            Alarm_Data[(HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Disconnect_Data + HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION_CONN_TYPE_MAC_OFFSET_reason) >> 1]
      );

      ECA_AppEmu_DISCONNECT_indicate_MAC(
        Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle >> 1],
        Alarm_Data[(HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Disconnect_Data
            + HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION_CONN_TYPE_MAC_OFFSET_reason) >> 1]
      );

      msglen = HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_SIZE + HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_CONN_TYPE_MAC_SIZE;
      break;
//    case HCT_MSG_CONNECT_CONN_TYPE_NONE:
    case HostAppEMU_CL_LLC:
      if (Alarm_Length != HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  DISCONNECT_INDICATION (%#x) invalid len, expected %d, received %d.\n"),
              __FUNCTION__, Alarm_Type, HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE, Alarm_Length);
      }

      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: DISCONNECT_INDICATION, Conn Hdl %#x.\n"),
            __FUNCTION__,
            Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle >> 1]
      );
      ECA_AppEmu_DISCONNECT_indicate_LLC(
        Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle >> 1]
      );
      break;
    default:
      msglen = HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_SIZE;
      break;
    }

    break;

  case HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_COMPLETE:
    HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("%S: ALARM: Network Registration Complete.\n"), __FUNCTION__);
    msglen = 0;
    break;
  case HCT_MSG_ALARM_TYPE_CONNECT_COMPLETE:
    HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("%S: ALARM: Promoted to Switch Node.\n"), __FUNCTION__);
    msglen = 0;
    break;
  case HCT_MSG_ALARM_TYPE_MAC_PROMOTED_SWITCH_NODE:
    HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("%S: ALARM: Promoted to Switch Node.\n"), __FUNCTION__);
    msglen = 0;
    break;
  case HCT_MSG_ALARM_TYPE_MAC_DEMOTED_SERVICE_NODE:
    HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("%S: ALARM: Demoted from Switch Node.\n"), __FUNCTION__);
    msglen = 0;
    break;
  case HCT_MSG_ALARM_TYPE_CONNECT_REQUEST:
    HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("%S: ALARM: Connection Request received, rejecting.\n"), __FUNCTION__);
    if (Alarm_Length < HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  CONNECT_REQUEST (%#x) invalid len, expected at least %d, received %d.\n"),
            __FUNCTION__, Alarm_Type, HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_SIZE, Alarm_Length);
    }

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Connection Request received for Conn Type %#x, rejecting.\n"),
              __FUNCTION__, Alarm_Data[HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Type >> 1]);

    switch (Alarm_Data[HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Type >> 1])
    {
    case HCT_MSG_CONNECT_CONN_TYPE_MAC:
      if (Alarm_Length != HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_SIZE + HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_CONN_TYPE_MAC_SIZE)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  MAC CONNECT_REQUEST (%#x) invalid len, expected %d, received %d.\n"),
              __FUNCTION__, Alarm_Type, HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_SIZE + HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_CONN_TYPE_MAC_SIZE, Alarm_Length);
      }

      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: MAC CONNECT_REQUEST, Conn Hdl %#x.\n"),
            __FUNCTION__,
            Alarm_Data[HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Handle >> 1]
      );

      msglen = HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_SIZE + HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION_CONN_TYPE_MAC_SIZE;
      break;
    default:
      msglen = HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_SIZE;
      break;
    }

    break;

  case HCT_MSG_ALARM_TYPE_NETWORK_REGISTRATION_STARTED:
  case HCT_MSG_ALARM_TYPE_PHY_RX_PPDU_CRC_FAIL:
  case HCT_MSG_ALARM_TYPE_PHY_RX_HDR_SYNTAX_ERR:

		//xraf won't this cause a leak????
//    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  Unhandled Alarm Type %#x.\n"), __FUNCTION__, Alarm_Type);
		return;
  
	case HCT_MSG_ALARM_TYPE_CL:
		{
			UINT16 clAlarmType = (UINT16)msg[HCT_MSG_ALARM_OFFSET_Alarm_Data >> 1];

			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Received CL Alarm: Type= %d\n"), __FUNCTION__, clAlarmType);

			if (clAlarmType == 1)
			{
				ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;

				UINT16 deviceIdLength = (UINT16)Alarm_Data[1 + (HCT_MSG_RPY_CL_ESTABLIST_OFFSET_Id_Length >> 1)];
		    UINT8 *	pui8 = (UINT8 *) &Alarm_Data[1 + (HCT_MSG_RPY_CL_ESTABLISH_OFFSET_Device_Identifier >> 1)];
				
				memcpy(&ECA_AppEmu_mgh.LLC_DeviceIdentifier, pui8, deviceIdLength);

				ECA_AppEmu_CL_ESTABLISH_handler(Alarm_Data+1, msglen-3);
			}
			else
			{
				ECA_AppEmu_CL_RELEASE_handler(Alarm_Data+1, msglen-3);
			}
		}
    break;

  case HCT_MSG_ALARM_TYPE_GENERAL_ERROR:
    ECA_AppEmu_ALARM_Handler_GENERAL_ALARM(Alarm_Data, Alarm_Length);
    msglen = 0;
    break;

  default:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR!  Unknown Alarm Type %#x.\n"), __FUNCTION__, Alarm_Type);
    return;
  }

  MyLocalFree(msg);

  msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, (msglen + HCT_MSG_RPY_ALARM_SIZE));
  if (msg)
  {
    *((UINT16 *) &msg[HCT_MSG_RPY_ALARM_OFFSET_Alarm_Type >> 1]) = Alarm_Type;
    *((UINT16 *) &msg[HCT_MSG_RPY_ALARM_OFFSET_Data_Length >> 1]) = msglen;

    switch (Alarm_Type)
    {
    case HCT_MSG_ALARM_TYPE_NETWORK_DEREGISTERED:
      break;
    case HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION:
      *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Type) >> 1])
            = Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Type >> 1];
      *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle) >> 1])
            = Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle >> 1];
      switch (Alarm_Data[HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Type >> 1])
      {
      case HCT_MSG_CONNECT_CONN_TYPE_MAC:
        *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Disconnect_Data_Length) >> 1])
              = HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION_CONN_TYPE_MAC_SIZE;
        break;
      default:
        *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Disconnect_Data_Length) >> 1])
              = 0;
        break;
      }
      break;
    case HCT_MSG_ALARM_TYPE_CONNECT_REQUEST:
        *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Flags) >> 1])
              = ~HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_FLAG_ACCEPT;
        *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Type) >> 1])
              = Alarm_Data[HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Type >> 1];
        *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Handle) >> 1])
              = Alarm_Data[HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Handle >> 1];
        switch (Alarm_Data[HCT_MSG_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Conn_Type >> 1])
        {
        case HCT_MSG_CONNECT_CONN_TYPE_MAC:
          *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Connection_Data_Length) >> 1])
                = HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_CONN_TYPE_MAC_SIZE;
          break;
        default:
          *((UINT16 *) &msg[(HCT_MSG_RPY_ALARM_OFFSET_Data + HCT_MSG_RPY_ALARM_TYPE_CONNECT_REQUEST_OFFSET_Connection_Data_Length) >> 1])
                = 0;
          break;
        }
        break;
    default:
      break;
    }

    ecf_task.task_type = ECA_TASK_SEND_MSG_TO_HOST;
    ecf_task.bRPY = (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE;
    ecf_task.u.s.msg = msg;
    ecf_task.u.s.msglen = msglen + HCT_MSG_RPY_ALARM_SIZE;

    os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to acknowledge DATA_TRANSFER.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_NETWORK_REGISTER_or_NETWORK_START_handler
*
* DESCRIPTION:
*       NETWORK_REGISTER handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_NETWORK_REGISTER_or_NETWORK_START_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
  {
    ECA_AppEmu_NETWORK_START_handler(msg, msglen);
  }
  else
  {
    ECA_AppEmu_NETWORK_REGISTER_handler(msg, msglen);
  }

  MyLocalFree(msg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_NETWORK_REGISTER_handler
*
* DESCRIPTION:
*       NETWORK_REGISTER handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_NETWORK_REGISTER_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  ECA_NETWORK_REGISTER_confirm(
      (APPEMU_status_t) *((UINT16 *) &msg[HCT_MSG_RPY_NETWORK_REGISTER_OFFSET_STATUS >> 1]),
      ((UINT16 *) &msg[HCT_MSG_RPY_NETWORK_REGISTER_OFFSET_SNA >> 1]),
      ((UINT16 *) &msg[HCT_MSG_RPY_NETWORK_REGISTER_SIZE_EUI48 >> 1])
  );
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_NETWORK_START_handler
*
* DESCRIPTION:
*       NETWORK_REGISTER handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_NETWORK_START_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT16 status;
  ECF_TASK_s ecf_task;

  if (msglen != HCT_MSG_RPY_NETWORK_START_G3_SIZE)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_NETWORK_START_G3_SIZE, msglen);
  }
  else
  {
    status = *((UINT16 *) &msg[HCT_MSG_RPY_NETWORK_START_G3_OFFSET_STATUS >> 1]);
    if (status == HCT_MSG_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("NETWORK START Success.\n"));
      //for g3 base node, connect all the times
      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_CONNECTED;
    }
    else
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("NETWORK START returned error %#x.\n"), status);
    }
  }

  ecf_task.task_type = ECA_TASK_REGISTER_CONFIRM;
  ecf_task.u.task_param = (void *) status;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DISCOVER_handler
*
* DESCRIPTION:
*       DISCOVER handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
*
* There are three different types of discover messages
* Unfortunitly there is no identifier in the reply message to id which 
* Message type is being replied to.
*
* The Route Discover reply message size is unique so it's easy to id.
* The Network Discovery and Path Discovery messages are variable length
* so that the length may not be used to indicate which message type it is.
* If the service node used in the path discovery message 
* "emeterDiscoverNetworkPathNode" is not zero then it is assumed that the 
* reply message is a path discovery replay.  
* This should work because the BN will not use the network discovery message.
******************************************************************************/
void ECA_AppEmu_DISCOVER_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  APPEMU_status_t status;
  UINT16 pan_count;
  UINT16 index;

  status = (APPEMU_status_t) *((UINT16 *) &msg[HCT_MSG_RPY_DISCOVER_G3_OFFSET_Status >> 1]);
  pan_count = *((UINT16 *) &msg[HCT_MSG_RPY_DISCOVER_G3_OFFSET_Pan_Count >> 1]);
  //
  // If this is a route discover reply, send a path discovery message next
  //
  if (msglen == HCT_MSG_RPY_G3_ROUTE_DISCOVER_SIZE)
  {
    if (status != HCT_MSG_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Route discovery failed: SN= 0x%X  Status= 0x%X\n"),emeterDiscoverNetworkPathNode, status);
    }
    else
    {
      HCT_MSG_BUFFER_t *pmsg;
      pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_G3_PATH_DISCOVER_SIZE);
      if (pmsg == NULL)
      {
        ECA_OutOfMemory();
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to DISCOVER.\n"), __FUNCTION__);
        emeterDiscoverNetworkPathNode = 0;
      }
      else
      {
        ((HCT_MSG_REQ_G3_PATH_DISCOVER_s *)pmsg)->DiscoverType = HCT_MSG_DISCOVER_G3_TYPE_PATH;
        ((HCT_MSG_REQ_G3_PATH_DISCOVER_s *)pmsg)->Reserved = 0;
        ((HCT_MSG_REQ_G3_PATH_DISCOVER_s *)pmsg)->DstAddress = emeterDiscoverNetworkPathNode;

        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Performing path discover for 0x%X\n"),emeterDiscoverNetworkPathNode);
    
        HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DISCOVER, (UINT8 *) pmsg, HCT_MSG_REQ_G3_PATH_DISCOVER_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
      }
    }
  }
  //
  // Not a route discovery so if there is a valid network path node it must be a reply to a path discovery message
  // There is no definitive way to verify this message other than to check for the node 
  //
  else if (emeterDiscoverNetworkPathNode != 0)
  {
    if (status != HCT_MSG_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Path discovery failed: SN= %X  Status= %X\n"),emeterDiscoverNetworkPathNode, status);
    }
    else if (pan_count != emeterDiscoverNetworkPathNode)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Device Path: Error Destination Address not found = %X\n"),emeterDiscoverNetworkPathNode);
    }
    else 
    {
      int numberOfDevices = *((UINT16 *) & msg[HCT_MSG_RPY_G3_PATH_DISCOVER_OFFSET_NumberOfDevices >> 1]);
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Device Path: Destination Address = %X  Number of Devices= %X\n"),emeterDiscoverNetworkPathNode, numberOfDevices);

      for (int index = 0; index < numberOfDevices; index++)
      {
        int deviceAddress = *((UINT16 *) & msg[(HCT_MSG_RPY_G3_PATH_DISCOVER_OFFSET_Devices >> 1) + index]);
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("........Device Path: %X\n"), deviceAddress);
      }
    }
    emeterDiscoverNetworkPathNode = 0;
  }
  //
  // Only rpy left is the network discovery message rpy
  //
  else
  {
    if (status != HCT_MSG_STATUS_SUCCESS)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: DISCOVER returned error %#x.\n"), __FUNCTION__, status);
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_STATE_MASK;
    }
    else
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("DISCOVER: %d Pan entries:\n"), pan_count);

      if (pan_count != 0)
      {
        ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_REGISTRATION_COMPLETE;
      }
      else
      {
        ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_STATE_MASK;
      }
			int offset = HCT_MSG_RPY_DISCOVER_G3_OFFSET_PanID_Data;
      for (index = 0; index < pan_count; index++)
      {
				UINT8 * pMsg = (UINT8*)msg;
				UINT8 addressMode = pMsg[offset++];
				UINT8 LQI     = pMsg[offset++];
				UINT16 PAN_Id = *((UINT16 *) &pMsg[offset]);
				offset += 2;

				char temp[256];
				if (addressMode ==2 )
				{
					UINT16 shortAddr = *((UINT16 *) &pMsg[offset]);
					offset += 2;
					//sprintf(temp, "0x%#x", PAN_Id);
					ECA_AppEmu_mgh.LBA_Address = shortAddr;
					ECA_AppEmu_mgh.G3_Pan_ID = PAN_Id;
                                                                                  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, 
                                                                                               TEXT("  [%d] mode=%d panId=0x%x LQI= %d shortAddr=0x%x\n"),
						index,
						addressMode, 
						PAN_Id,
						LQI, shortAddr);
				}
				else if (addressMode == 3)
				{
					UINT8 extendedAddress[8];
					memcpy(extendedAddress, &pMsg[offset], 8);
					offset += 8;

					sprintf(temp, "0x%#x.%#x.%#x.%#x.%#x.%#x.%#x.%#x",
						extendedAddress[0],
						extendedAddress[1],
						extendedAddress[2],
						extendedAddress[3],
						extendedAddress[4],
						extendedAddress[5],
						extendedAddress[6],
						extendedAddress[7]);
				
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("  [%d] 0x%#x %s LQI= %d\n"),
						index,
						PAN_Id, 
						temp,
						LQI);
				}
				else
				{
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Bad Network Discover Message: Address Mode should be 2 or 3 is: %d\n"), addressMode);
					break;
				}
				if (index == 0)
				{
					ECA_AppEmu_mgh.G3_Pan_ID = PAN_Id;
					printf("Discover Hander: PAN ID= %x\n", ECA_AppEmu_mgh.G3_Pan_ID);

			    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_REGISTRATION_IN_PROGRESS;
				}
      }
    }
  }

  MyLocalFree(msg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_CONNECT_handler
*
* DESCRIPTION:
*       CONNECT handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_CONNECT_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  UINT16 status;

  if (msglen >= HCT_MSG_RPY_CONNECT_SIZE)
  {
    status = *((UINT16 *) &pmsg[HCT_MSG_RPY_CONNECT_OFFSET_Status >> 1]);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected at least %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_CONNECT_SIZE, msglen);
    status = ECF_STATUS_ERROR;
  }

  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS))
  {
    ECA_AppEmu_mgh.stats.unexpected_connect_confirm++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_205, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_connect_confirm);
  }
  else
  {
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_204, 1, (UINT32) status);
  }

  switch (HostAppEMU_Flags & HostAppEMU_MODE_MASK)
  {
  case HostAppEMU_MAC:
    ECA_CONNECT_confirm_MAC(pmsg, msglen, status);
    break;
#ifdef APPEMU_IEC432
  case HostAppEMU_CL_IEC432:
    ECA_CONNECT_confirm_CL_IEC432(pmsg, msglen, status);
    break;
#endif // #ifdef APPEMU_IEC432
  case HostAppEMU_P2P:
  case HostAppEMU_CL_LLC:
  case HostAppEMU_IPv4:
    break;

  default:
    proj_assert(0);
    return;
  }

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_CL_ESTABLISH_or_ATTACH_handler
*
* DESCRIPTION:
*       CL_ESTABLISH or ATTACH handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_CL_ESTABLISH_or_ATTACH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_CL_LLC)
  {
    ECA_AppEmu_CL_ESTABLISH_handler(pmsg, msglen);
  }
  else if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    ECA_AppEmu_ATTACH_handler(pmsg, msglen);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Warning! Unexpected CL_ESTABLISH or ATTACH reply!\n"));
  }

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_CL_ESTABLISH_handler
*
* DESCRIPTION:
*       CL_ESTABLISH handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_CL_ESTABLISH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  UINT16 status;

/* This is a varible length message can't test the size like this.
	if (msglen >= HCT_MSG_RPY_CL_ESTABLISH_SIZE)
  {
    status = *((UINT16 *) &pmsg[HCT_MSG_RPY_CL_ESTABLISH_OFFSET_Status >> 1]);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected at least %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_CL_ESTABLISH_SIZE, msglen);
    status = ECF_STATUS_ERROR;
  }
*/

  status = *((UINT16 *) &pmsg[HCT_MSG_RPY_CL_ESTABLISH_OFFSET_Status >> 1]);
	
  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS))
  {
    ECA_AppEmu_mgh.stats.unexpected_connect_confirm++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_205, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_connect_confirm);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("CL_ESTABLISH callback received.\n"),
          __FUNCTION__, status);
  }

  ECA_CL_ESTABLISH_confirm(pmsg, msglen, status);
}


int SetTunTapAddress(UINT16 shortAddress)
{
	STARTUPINFO si;
	PROCESS_INFORMATION pi;

	ZeroMemory(&si, sizeof(si));
	ZeroMemory(&pi, sizeof(pi));

	si.cb = sizeof(si);

	int low = shortAddress & 0x00FF;
	int high = shortAddress >> 8;

  TCHAR commandLine[256];
 
	swprintf(commandLine, TEXT("netsh interface ip set address name=\"%S\" static addr=192.168.%d.%d  mask=255.255.255.0"), tunTapDriverName, high, low);
    
	if (!CreateProcess(NULL, /* no module name */
		commandLine,	/*command line*/
		NULL,					/*Process handle not inhertiable*/
		NULL,					/*Thread hand not inhertiable */
		FALSE,				/*Set Handl inheritance to false */
		0,						/* No creation flags */
		NULL,					/* User parents enviroment */
		NULL,					/* parents starting dir */
		&si, 
		&pi))
	{
		int i = 0;
	}


	WaitForSingleObject(pi.hProcess, INFINITE);
	CloseHandle(pi.hProcess);
	CloseHandle(pi.hThread);
	//
	// Sleep 3 seconds to be sure everything is set..
	//
	Sleep(3000);

	char temp[32];
	sprintf(temp, "192.168.%d.%d", high, low);
	tunTapIPv4Address = inet_addr(temp);

	if (pTunTapDriver != NULL)
	{
		delete pTunTapDriver;
	}
	pTunTapDriver = new TunTap(tunTapDriverName, tunTapIPv4Address, microIP);

  return 0;   // Program successfully completed.
	
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_ATTACH_handler
*
* DESCRIPTION:
*       CL_ESTABLISH handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_ATTACH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  UINT16 status;
  ECF_TASK_s ecf_task;

  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
  {
    if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS))
    {
      ECA_AppEmu_mgh.stats.unexpected_connect_confirm++;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unexpected ATTACH reply received (ttl:%d)!\n"),
            __FUNCTION__, ECA_AppEmu_mgh.stats.unexpected_connect_confirm);

      DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_205, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_connect_confirm);
    }

    if (msglen != HCT_MSG_RPY_ATTACH_G3_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
            __FUNCTION__, HCT_MSG_RPY_ATTACH_G3_SIZE, msglen);
      return;
    }

    status = *((UINT16 *) &pmsg[HCT_MSG_RPY_ATTACH_G3_OFFSET_Status >> 1]);
    if (status == HCT_MSG_STATUS_SUCCESS)
    {
      ECA_AppEmu_mgh.Network_Address = *((UINT16 *) &pmsg[HCT_MSG_RPY_ATTACH_G3_OFFSET_Network_Address >> 1]);
      ECA_AppEmu_mgh.G3_Pan_ID = *((UINT16 *) &pmsg[HCT_MSG_RPY_ATTACH_G3_OFFSET_Pan_ID >> 1]);
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ATTACH Success, Network Address:%#x Pan ID:%#x.\n\n"),
            __FUNCTION__, ECA_AppEmu_mgh.Network_Address, ECA_AppEmu_mgh.G3_Pan_ID);

			if (tunTapDriverName != NULL && strlen(tunTapDriverName) > 0)
			{
				SetTunTapAddress(ECA_AppEmu_mgh.Network_Address);
			}

      ecf_task.u.task_param = (void *) FAKE_CONNECTION_HANDLE;
    }
    else
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ATTACH returned error, performing discovery. %#x!\n"),
            __FUNCTION__, status);
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_REGISTRATION_IN_PROGRESS;
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_REGISTRATION_COMPLETE;
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS;
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECTED;
      ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;
      g3HeartBeatCounter = 0;
      ECA_AppEmu_Discover();
      
      return;
    }

    ecf_task.task_type = ECA_TASK_CONNECTION_ESTABLISHED;
    os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
  }
  else if ((tunTapIPv4Address == 0) && (HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
  {
    UINT8 *pna1;
    UINT8 *pna2;
    UINT8 *pea;
    UINT8 ci;
    static UINT32 bnRoundCount = 0;

    if (msglen != HCT_MSG_IND_ATTACH_G3_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
            __FUNCTION__, HCT_MSG_IND_ATTACH_G3_SIZE, msglen);
      return;
    }

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("************* Round %d *************\n"), bnRoundCount++);

    pna1 = (UINT8 *) &pmsg[HCT_MSG_IND_ATTACH_G3_OFFSET_Network_Address_1 >> 1];
    pna2 = (UINT8 *) &pmsg[HCT_MSG_IND_ATTACH_G3_OFFSET_Network_Address_2 >> 1];
    pea = (UINT8 *) &pmsg[HCT_MSG_IND_ATTACH_G3_OFFSET_Extended_Address >> 1];
    ci = (*((UINT8 *) &pmsg[HCT_MSG_IND_ATTACH_G3_OFFSET_Capability_Info_Resv >> 1]) & HCT_MSG_IND_ATTACH_G3_Capability_Info_MASK);
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ATTACH Indication:\n  NA1:%02x%02x%02x%02x%02x%02x%02x%02x \n  NA2:%02x%02x%02x%02x%02x%02x%02x%02x. \n  EA:%02x%02x%02x%02x%02x%02x%02x%02x \n  CI:%#x.\n"),  
			pna1[0], pna1[1], pna1[2], pna1[3], pna1[4], pna1[5], pna1[6], pna1[7],
      pna2[0], pna2[1], pna2[2], pna2[3], pna2[4], pna2[5], pna2[6], pna2[7],
      pea[0], pea[1], pea[2], pea[3], pea[4], pea[5], pea[6], pea[7],  ci  );

		G3_Network * network = G3_Network::Add(pna1, pna2, pea, ci, TRUE);
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Conc Stats: Short Address: 0x%X  Attach: %d  Detach: %d  Msg Sent: %d   Msg Received: %d\n"),
      network->NetworkShortAddress(), network->m_AttachCount, network->m_DetachCount, network->m_MessagesSent, network->m_MessagesReceived);

#ifdef DEBUG_SHU
#if 0     
//WARNING: debug shu testing for BN initiated leave    
    {
      UINT16 *pAddr = (UINT16 *) &pmsg[HCT_MSG_IND_ATTACH_G3_OFFSET_Extended_Address >> 1];

      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Detach immediately, Addr:%04x%04x%04x%04x\n\n"),
        pAddr[0], pAddr[1], pAddr[2], pAddr[3]);
      ECA_AppEMU_DETACH((UINT16 *)pAddr);
    }
#endif
#endif

    //remove for g3 bn
    //ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_CONNECTED;
  }
}


/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DISCONNECT_MAC_indicate
*
* DESCRIPTION:
*       DISCONNECT handler.
*
* Return Value:       none
* Input Parameters:   UINT16      connection handle
*                     UINT16      disconnect reason
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_DISCONNECT_indicate_MAC(UINT16 connHandle, UINT16 reason)
{
  ECF_TASK_s ecf_task;

  if (ECA_AppEmu_mgh.connHandle != connHandle)
  {
    ECA_AppEmu_mgh.stats.err_bad_disconnect_handle++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_242, 3,
        (UINT32) ECA_AppEmu_mgh.connHandle,
        (UINT32) connHandle,
        (UINT32) ECA_AppEmu_mgh.stats.err_bad_disconnect_handle
      );
  }

  ecf_task.task_type = ECA_TASK_DISCONNECT;
  ecf_task.conn_type = HCT_MSG_CONNECT_CONN_TYPE_MAC;
  ecf_task.u.task_param = (void *) connHandle;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DISCONNECT_LLC_indicate
*
* DESCRIPTION:
*       DISCONNECT handler.
*
* Return Value:       none
* Input Parameters:   UINT16      connection handle
*                     UINT16      disconnect reason
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_DISCONNECT_indicate_LLC(UINT16 connHandle)
{
  ECF_TASK_s ecf_task;

  ecf_task.task_type = ECA_TASK_DISCONNECT;
  ecf_task.conn_type = HCT_MSG_CONNECT_CONN_TYPE_NONE;
  ecf_task.u.task_param = (void *) connHandle;
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DISCONNECT_acknowledge
*
* DESCRIPTION:
*       DISCONNECT acknowledge.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_DISCONNECT_acknowledge_MAC(UINT16 connHandle)
{
  HCT_MSG_BUFFER_t *pmsg;
  HCT_MSG_BUFFER_t *ppayload;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, (HCT_MSG_RPY_ALARM_SIZE + HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_SIZE));
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_RPY_ALARM_OFFSET_Alarm_Type >> 1]) = HCT_MSG_ALARM_TYPE_DISCONNECT_INDICATION;
    *((UINT16 *) &pmsg[HCT_MSG_ALARM_OFFSET_Alarm_Length >> 1]) = HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_SIZE;

    ppayload = &pmsg[HCT_MSG_RPY_ALARM_OFFSET_Data >> 1];
    *((UINT16 *) &ppayload[HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Type >> 1]) = HCT_MSG_CONNECT_CONN_TYPE_MAC;
    *((UINT16 *) &ppayload[HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Conn_Handle >> 1]) = connHandle;
    *((UINT16 *) &ppayload[HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_OFFSET_Disconnect_Data_Length >> 1]) = 0;

    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("%S: Sending ALARM DISCONNECT Acknowledge for connHandle %d\n"), __FUNCTION__, connHandle);
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_ALARM, (UINT8 *) pmsg, HCT_MSG_RPY_ALARM_SIZE + HCT_MSG_RPY_ALARM_TYPE_DISCONNECT_INDICATE_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to acknowledge DISCONNECT.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_UNREGISTER
*
* DESCRIPTION:
*       NETWORK_UNREGISTER.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_UNREGISTER(void)
{
  UINT8 *pmsg;

  pmsg = (UINT8 *) LocalAlloc(LPTR, HCT_MSG_REQ_NETWORK_UNREGISTER_SIZE);
  if (pmsg)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unregistering...\n"), __FUNCTION__);
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_NETWORK_UNREGISTER, pmsg, HCT_MSG_REQ_NETWORK_UNREGISTER_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to NETWORK_UNREGISTER.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_NETWORK_UNREGISTER_handler
*
* DESCRIPTION:
*       NETWORK_REGISTER handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_NETWORK_UNREGISTER_handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  ECF_TASK_s ecf_task;

  ecf_task.task_type = ECA_TASK_UNREGISTER_CONFIRM;
  ecf_task.u.task_param = (void *) *((UINT16 *) &msg[HCT_MSG_RPY_NETWORK_UNREGISTER_OFFSET_STATUS >> 1]);
  os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Load_System_Config_Handler
*
* DESCRIPTION:
*       HCT_MSG_TYPE_LOAD_SYSTEM_CONFIG handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Load_System_Config_Handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT16 status;

  status = *(UINT16 *) (&msg[HCT_MSG_RPY_LOAD_SYSTEM_CONFIG_OFFSET_Status >> 1]);
  switch (status)
  {
  case HCT_MSG_STATUS_OK:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: LOAD_SYSTEM_CONFIG returned success.\n"), __FUNCTION__);
    break;
  case HCT_MSG_STATUS_NO_EEPROM:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: LOAD_SYSTEM_CONFIG returned success but NO_EEPROM (%#x).\n"), __FUNCTION__, status);
    break;
  default:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! LOAD_SYSTEM_CONFIG returned %#x.\n"), __FUNCTION__, status);
    break;
  }

  MyLocalFree(msg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Shutdown_Handler
*
* DESCRIPTION:
*       HCT_MSG_TYPE_SHUT_DOWN handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Shut_Down_Handler(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT16 status;

  status = *(UINT16 *) (&msg[HCT_MSG_RPY_SHUT_DOWN_OFFSET_Status >> 1]);
  switch (status)
  {
  case HCT_MSG_STATUS_OK:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: SHUT_DOWN returned success.\n"), __FUNCTION__);
    break;
  default:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! SHUT_DOWN returned %#x.\n"), __FUNCTION__, status);
    break;
  }

  MyLocalFree(msg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_LLC
*
* DESCRIPTION:
*       LLC Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_LLC(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY)
{
  ECF_TASK_s ecf_task;
  UINT16 Mode;

  Mode = *((UINT16 *) &pmsg[HCT_MSG_DATA_TRANSFER_MODE_OFFSET]);
  switch (Mode)
  {
  case HCT_MSG_DATA_TRANSFER_MODE_INDICATION:
    ECA_AppEmu_processPacket_LLC_INDICATION(pmsg, msg_len);

    if (bRPY)
    {
//      if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_P2P)
      {
        if (RPY_delay == 0)
        {
          HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Sending RPY to device.\n"), RPY_delay);
          HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) NULL, 0, FALSE);
        }
        else
        {
          ecf_task.task_type = ECA_TASK_SEND_RPY;
          ecf_task.u.task_param = 0;
          os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
        }
      }
    }

    break;
  case HCT_MSG_DATA_TRANSFER_MODE_CONFIRM:
    ECA_AppEmu_processPacket_LLC_CONFIRM(pmsg, msg_len);
    break;
  case HCT_MSG_DATA_TRANSFER_MODE_REQUEST:
  case HCT_MSG_DATA_TRANSFER_MODE_REPLY_REQUEST:
  case HCT_MSG_DATA_TRANSFER_MODE_REPLY_CONFIRM:
  case HCT_MSG_DATA_TRANSFER_MODE_REPLY_INDICATION:
  case HCT_MSG_DATA_TRANSFER_MODE_UPDATE_REPLY_REQUEST:
  case HCT_MSG_DATA_TRANSFER_MODE_UPDATE_REPLY_CONFIRM:
  case HCT_MSG_DATA_TRANSFER_MODE_UPDATE_REPLY_INDICATION:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("%S: Unexpected LLC DATA_TRANSFER mode %#x, discarding packet.\n"), __FUNCTION__, Mode);
    MyLocalFree(pmsg);
    return;
  default:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("%S: Unknown LLC DATA_TRANSFER mode %#x, discarding packet.\n"), __FUNCTION__, Mode);
    MyLocalFree(pmsg);
    return;
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_LLC_INDICATION
*
* DESCRIPTION:
*       REQUEST LLC Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_LLC_INDICATION(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len)
{
  HCT_MSG_BUFFER_t *pup_msg;
  UINT8 *pui8;
  UINT16 uplen;
  UINT8 Dst_LSAP;
  UINT8 Src_LSAP;
  UINT16 Dest_Addr;
  UINT16 Src_Addr;
  UINT16 *pdata;

  pui8 = (UINT8 *) pmsg;

  Dst_LSAP = pui8[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Dst_LSAP];
  Src_LSAP = pui8[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Src_LSAP];

  Dest_Addr = *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Dest_Addr >> 1]);
  Src_Addr = *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Src_Addr >> 1]);

  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    ECA_AppEmu_mgh.stats.unexpected_data_indicate++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_222, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_data_indicate);
  }

#if 0
  if (Src_LSAP != ECA_AppEmu_mgh.destSap)
  {
    ECA_AppEmu_mgh.stats.err_bad_LLC_Src_LSAP++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_251, 3,
        (UINT32) ECA_AppEmu_mgh.destSap, (UINT32) Src_LSAP, (UINT32) ECA_AppEmu_mgh.stats.err_bad_LLC_Src_LSAP);
  }
  if (Dst_LSAP != ECA_AppEmu_mgh.srcSap)
  {
    ECA_AppEmu_mgh.stats.err_bad_LLC_Dest_LSAP++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_252, 3,
        (UINT32) ECA_AppEmu_mgh.srcSap, (UINT32) Dst_LSAP, (UINT32) ECA_AppEmu_mgh.stats.err_bad_LLC_Dest_LSAP);
  }
  if (Src_Addr != ECA_AppEmu_mgh.Destination_Address)
  {
    ECA_AppEmu_mgh.stats.err_bad_LLC_Dest_Addr++;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: LLC Dest Addr mismatch on data RECIEVE %04x.\n"), __FUNCTION__, Src_Addr);
  }
  if (Dest_Addr[0], &ECA_AppEmu_mgh.EUI[0], sizeof(sizeof(Dest_Addr))) != 0)
  {
    ECA_AppEmu_mgh.stats.err_bad_LLC_Dest_Addr++;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: LLC Dest Addr mismatch on data RECIEVE (%02x:%02x:%02x:%02x:%02x:%02x).\n"), __FUNCTION__,
        (UINT8) (Dest_Addr[0] >> 8) & 0xff, (UINT8) (Dest_Addr[0]) & 0xff,
        (UINT8) (Dest_Addr[1] >> 8) & 0xff, (UINT8) (Dest_Addr[1]) & 0xff,
        (UINT8) (Dest_Addr[2] >> 8) & 0xff, (UINT8) (Dest_Addr[2]) & 0xff
    );
  }
#endif

  pdata = (UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_OFFSET_Data >> 1];
  uplen = msg_len - HCT_MSG_REQ_DATA_TRANSFER_INDICATION_SIZE;

  /* loop back the packet we received */
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Looping back, LLC DATA_TRANSFER.Request, length %d.\n"), __FUNCTION__, uplen);

  /* allocate the UP_MSG packet */
  pup_msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE + uplen);
  if (pup_msg)
  {
    pui8 = (UINT8 *) pup_msg;
    /* build packet */
    *((UINT16 *) &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Mode >> 1]) = HCT_MSG_DATA_TRANSFER_MODE_REQUEST;

    pui8[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dst_LSAP] = Src_LSAP;
    pui8[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Src_LSAP] = Dst_LSAP;
    memcpy(
      &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Dest_Addr >> 1],
      &Src_Addr,
      HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE_Dest_Addr
    );
    memset(
      &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_reserved >> 1],
      0,
      HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE_reserved
    );
    memcpy(
      &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_OFFSET_Data >> 1],
      pdata,
      uplen
    );

    ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_None]++;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;

    /* send packet */
    HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_SIZE + uplen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
  }

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_LLC_CONFIRM
*
* DESCRIPTION:
*       CONFIRM LLC Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_LLC_CONFIRM(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len)
{
  UINT8 *pui8;
  UINT8 Dst_LSAP;
  UINT8 Src_LSAP;
  UINT16 Dest_Addr;

  pui8 = (UINT8 *) pmsg;
  Dst_LSAP = pui8[HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_OFFSET_Dst_LSAP];
  Src_LSAP = pui8[HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_OFFSET_Src_LSAP];

  memcpy(&Dest_Addr, &pmsg[HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_OFFSET_Dest_Addr >> 1], HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_SIZE_Dest_Addr);

  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    ECA_AppEmu_mgh.stats.unexpected_data_indicate++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_222, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_data_indicate);
  }
#if 0
  if (Dest_Addr != ECA_AppEmu_mgh.Destination_Address)
  {
    ECA_AppEmu_mgh.stats.err_bad_LLC_Dest_Addr++;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: LLC Dest Addr mismatch on data CONFIRM %04x.\n"), __FUNCTION__, Dest_Addr);
  }
  if (Src_LSAP != ECA_AppEmu_mgh.destSap)
  {
    ECA_AppEmu_mgh.stats.err_bad_LLC_Src_LSAP++;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: LLC Src LSAP mismatch on data receive.  Expected 0x%02x, received 0x%02x (ttl:%d).\n"), __FUNCTION__,
        ECA_AppEmu_mgh.destSap, Src_LSAP, ECA_AppEmu_mgh.stats.err_bad_LLC_Src_LSAP
    );
  }
  if (Dst_LSAP != ECA_AppEmu_mgh.srcSap)
  {
    ECA_AppEmu_mgh.stats.err_bad_LLC_Dest_LSAP++;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: LLC Dest LSAP mismatch on data receive.  Expected 0x%02x, received 0x%02x (ttl:%d).\n"), __FUNCTION__,
        ECA_AppEmu_mgh.destSap, Dst_LSAP, ECA_AppEmu_mgh.stats.err_bad_LLC_Dest_LSAP
    );
  }
#endif

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Received LLC DATA_TRANSFER.Confirm.\n"), __FUNCTION__);
  HostAppEMU_ReceivedSyncMsg();

  MyLocalFree(pmsg);

	/* test code commented out

  HostAppEMU_GetFlags(&flags);
  if (!(flags & HostAppEMU_LLC_AUTO))
  {
    static int itercount = 0;

    if ((itercount++ & 0x03) == 0x03)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_VERBOSE, TEXT("Calling CL_RELEASE to drop connection.\n"));
      ECA_AppEMU_CL_Release(ECA_AppEmu_mgh.Destination_Address);
    }
  }
	
	test code commented out */

}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_IPv4
*
* DESCRIPTION:
*       IPv4 Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_IPv4(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len)
{
  HCT_MSG_BUFFER_t *pup_msg;
  UINT16 uplen;

  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    ECA_AppEmu_mgh.stats.unexpected_data_indicate++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_222, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_data_indicate);
  }

  uplen = msg_len;

  /* loop back the packet we received */
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Looping back IPv4 datagram, length %d.\n"), __FUNCTION__, uplen);

  /* allocate the UP_MSG packet */
  pup_msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, uplen);
  if (pup_msg)
  {
    memcpy(pup_msg, pmsg, uplen);

    ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_None]++;
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;

    /* send packet */
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, uplen, FALSE);
  }
  else
  {
    ECA_OutOfMemory();
  }

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_G3
*
* DESCRIPTION:
*       G3 Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_G3(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len, BOOL bRPY)
{
  ECF_TASK_s ecf_task;

	if (tunTapIPv4Address != 0 && pTunTapDriver != NULL && msg_len != HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_SIZE)
	{
		pTunTapDriver->WriteToTunTap(pmsg, msg_len, bRPY);
	}
	else if (eMeterEmulation == TRUE) 
	{
		ProcessEmeterEmulationMessage(pmsg, msg_len);
	}
  else if (msg_len == HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_SIZE)
  {
    ECA_AppEmu_processPacket_G3_CONFIRM(pmsg, msg_len);
  }
  else
  {
    ECA_AppEmu_processPacket_G3_INDICATION(pmsg, msg_len);

    if (bRPY)
    {
      if (RPY_delay == 0)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Sending RPY to device.\n"), RPY_delay);
        HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) NULL, 0, FALSE);
      }
      else
      {
        ecf_task.task_type = ECA_TASK_SEND_RPY;
        ecf_task.u.task_param = 0;
        os_fifo_put(&HostAppEMU_EC_Task_MsgFifo, &ecf_task);
      }
    }
  }
}                                                           

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_G3_INDICATION
*
* DESCRIPTION:
*       REQUEST G3 Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_G3_INDICATION(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len)
{
  HCT_MSG_BUFFER_t *pup_msg;
  UINT8 *pui8;
  UINT16 uplen;
  UINT16 ui16;
  UINT8 lqi;
  BOOL bSEC;
  UINT16 *pdata;

  pui8 = (UINT8 *) pmsg;

  ui16 = *((UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_OFFSET_LinkQualityIndicator_Flags >> 1]);
  lqi = (ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_LinkQualityIndicator_MASK);
  bSEC = ui16 & HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_FLAGS_SEC ? TRUE : FALSE;

  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    ECA_AppEmu_mgh.stats.unexpected_data_indicate++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_222, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_data_indicate);

    //add for g3
    MyLocalFree(pmsg);
    return;
  }

  pdata = (UINT16 *) &pmsg[HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_OFFSET_Data >> 1];
  uplen = msg_len - HCT_MSG_REQ_DATA_TRANSFER_INDICATION_G3_SIZE;

  /* loop back the packet we received */
  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: G3 DATA_RX.Ind, LinkQI %d, SEC %s.\n"), __FUNCTION__,
      lqi, bSEC ? TEXT("ON") : TEXT("off"));

  ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_Loop_Start]++;

#ifdef DEBUG_SHU
  num_rxPkt++;
#else  
  HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_VERBOSE, (char *) pdata, uplen);
#endif

#ifdef DEBUG_SHU
  //only base node will echo back
  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
#else  
  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
#endif
  {
    /* allocate the UP_MSG packet */
    pup_msg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + uplen);
    if (pup_msg)
    {
      pui8 = (UINT8 *) pup_msg;

      /* build packet */
      *((UINT16 *) &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags >> 1]) = (UINT16) ECA_AppEmu_mgh.rx_nsdu_handle++;

      memcpy(
        &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_Data >> 1],
        pdata,
        uplen
      );

      ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;

#ifdef DEBUG_SHU
    //debug shu send multiple packets in a row
    //sleep time should be delayTime(at TX side) - the transmission duration (around 2s for 1500 bytes)
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT(".....sleep for 5 seconds to synchronize with SN\n\n"));
    Sleep(12500);
    //HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("\n\n\n.....wake up to synchronize with SN\n"));
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("\n\n\n\n\n.....wake up, ROUND %d BN ---> SN tx %d txCnf %d rx%d \n"), 
        num_txPkt+1, num_txPkt, num_txPktCnf, num_rxPkt);
    for (int i = 0; i<1; i++) {
      if (i>0)
       *((UINT16 *) &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags >> 1]) = (UINT16) ECA_AppEmu_mgh.rx_nsdu_handle++;
      num_txPkt++;
#endif
      /* send packet */
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Loop_Start]++;
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Echoing packet %d back to sender. nsduHandle 0x%04x\n\n"), 
        __FUNCTION__, ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Loop_Start], ECA_AppEmu_mgh.rx_nsdu_handle-1);
#ifdef DEBUG_SHU        
#else
      HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_VERBOSE, (char *) &pup_msg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_Data >> 1], uplen);
#endif      
      HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pup_msg, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + uplen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
#ifdef DEBUG_SHU
    //debug shu send multiple packets in a row
    }
#endif
    }
    else
    {
      ECA_OutOfMemory();
    }
  }

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_processPacket_G3_CONFIRM
*
* DESCRIPTION:
*       CONFIRM G3 Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_processPacket_G3_CONFIRM(HCT_MSG_BUFFER_t *pmsg, UINT16 msg_len)
{
  UINT8 *pui8;
  UINT16 nsdu_handle;
  UINT16 status;

  pui8 = (UINT8 *) pmsg;

  memcpy(&status, &pmsg[HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_OFFSET_Status >> 1], HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_SIZE_Status);
  memcpy(&nsdu_handle, &pmsg[HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_OFFSET_NSDU_Handle_Resv >> 1], HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_SIZE_NSDU_Handle_Resv);
  nsdu_handle &= HCT_MSG_REQ_DATA_TRANSFER_CONFIRM_G3_NSDU_Handle_MASK;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Received G3 DATA_TRANSFER.Confirm. nsdu_handle 0x%02x status 0x%04x\n\n"), 
    __FUNCTION__, nsdu_handle, status);

#ifdef DEBUG_SHU    
  num_txPktCnf++;
#endif

  if (!(ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_CONNECTED))
  {
    ECA_AppEmu_mgh.stats.unexpected_data_indicate++;
    DIAG_printf(DIAG_LOG_LEVEL_WARNING, DIAG_LOG_222, 1, (UINT32) ECA_AppEmu_mgh.stats.unexpected_data_indicate);
  }


  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
  {
    // if (((nsdu_handle + 1)&0xff) != ECA_AppEmu_mgh.tx_nsdu_handle)
    // {
    //   ECA_AppEmu_mgh.stats.err_bad_G3_nsdu_handle++;
    //   HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: G3 NSDU Handle mismatch on data CONFIRM.  Expected 0x%02x, received 0x%02x.\n\n\n"), 
    //     __FUNCTION__, (ECA_AppEmu_mgh.tx_nsdu_handle-1)&0xff, nsdu_handle);
    // }
  }
  else
  {
    // if (((nsdu_handle + 1)&0xff) != ECA_AppEmu_mgh.rx_nsdu_handle)
    // {
    //   ECA_AppEmu_mgh.stats.err_bad_G3_nsdu_handle++;
    //   HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: G3 NSDU Handle mismatch on data CONFIRM.  Expected 0x%02x, received 0x%02x.\n\n\n"), 
    //     __FUNCTION__, (ECA_AppEmu_mgh.rx_nsdu_handle-1)&0xff, nsdu_handle);
    //   ECA_AppEmu_mgh.rx_nsdu_handle = (UINT8)nsdu_handle+2;
    // }
  }

  ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_DATA_SEND;

  HostAppEMU_ReceivedSyncMsg();

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_G3_SendPacket
*
* DESCRIPTION:
*       CONFIRM G3 Receive packet handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message
*                     UINT16      message length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_G3_SendPacket(void)
{
  HCT_MSG_BUFFER_t *pmsg;
  UINT16 uplen;

#ifdef DEBUG_SHU
  uplen = g3_test_pktSize;
#else  
  uplen = 26;
#endif  
  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + uplen);
  if (pmsg)
  {
    ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Loop_Start]++;

    *(UINT16 *) &pmsg[(HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_NSDU_Handle_Flags >> 1)] = (UINT16) ECA_AppEmu_mgh.tx_nsdu_handle++;

    /* build packet */
    ECA_AppEmu_BuildPayload(&pmsg[(HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_Data >> 1)], uplen, NULL, 0);

    //customize the data packet to include short and ext address
    memcpy(&pmsg[HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_Data >> 1], &ECA_AppEmu_mgh.Network_Address, 2);
    memcpy(&pmsg[(HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_Data >> 1)+1], &ECA_AppEmu_mgh.G3_Long_Address, 8);

    /* send packet */
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DATA_SEND;
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Sending G3 packet %d, NSDU Handle:%#x, len %d.\n"), 
      ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Loop_Start], ECA_AppEmu_mgh.tx_nsdu_handle-1, uplen);
#ifdef DEBUG_SHU    
    num_txPkt++;
#else
    HostAppEMU_DbgDumpBuffer(HostAppEMU_DBG_LEVEL_VERBOSE, (char *) &pmsg[(HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_OFFSET_Data >> 1)], uplen);
#endif    
    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DATA_TRANSFER, (UINT8 *) pmsg, HCT_MSG_REQ_DATA_TRANSFER_REQUEST_G3_SIZE + uplen, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_Get_System_Info_Dump
*
* DESCRIPTION:
*       HCT_MSG_TYPE_GET_SYSTEM_INFO dump.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_Get_System_Info_Dump(HCT_MSG_BUFFER_t *msg, UINT16 msglen)
{
  UINT8 *pui8;
  UINT16 ui16;
  TCHAR *s;
  static TCHAR str[160];
  static TCHAR s1[10];
  UINT16 flags;
  UINT16 device_type;
  UINT16 device_mode;
  int i;

  pui8 = (UINT8 *) msg;

  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("\n*** SYSTEM INFORMATION ***\n"));

  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Firmware Version.................... %d.%d.%d.%d\n"),
    (*((UINT32 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Firmware_Version]) >> 24) & 0xff,
    (*((UINT32 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Firmware_Version]) >> 16) & 0xff,
    (*((UINT32 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Firmware_Version]) >> 8) & 0xff,
    (*((UINT32 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Firmware_Version])) & 0xff
  );

  firmwareVersion = *((UINT32 *)&pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Firmware_Version]);

  memset(&str[0], 0, sizeof(str));
  for (i = 0; i < *((UINT8 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Serial_Number_Len]); i++)
  {
    swprintf(s1, TEXT("%02x."), *((UINT8 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Serial_Number + i]));
    wcsncat(str, s1, wcslen(s1));
  }
  if (i != 0)
  {
    str[wcslen(str) - 1] = 0;
  }
	deviceSerialNumberLength = *((UINT8 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Serial_Number_Len]);

  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU,
      TEXT("Serial Number....................... [%d] %s\n"), 
      *((UINT8 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Serial_Number_Len]),
      str
  );

  device_type = (*((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Device_Mode_Type])) & HCT_MSG_GET_SYSTEM_INFO_DEVICE_TYPE_MASK;
  switch (device_type)
  {
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_TYPE_IEC432: s = TEXT("IEC-432-LLC convergence"); break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_TYPE_IP: s = TEXT("IPv4 convergence"); break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_TYPE_G3: s = TEXT("G3"); break;
  default:  s = TEXT("unknown"); break;
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Device Type......................... %s\n"), s);

  device_mode = (*((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Device_Mode_Type])) & HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_MASK;
  switch (device_mode)
  {
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_NORMAL: s = TEXT("Normal"); break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_APPEMU_MAC: s = TEXT("eAppEMU MAC"); break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_APPEMU_IEC432_LLC: s = TEXT("eAppEMU IEC432/LLC"); break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_POINT_TO_POINT: s = TEXT("P2P"); break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_MAC: s = TEXT("MAC"); break;
	case HCT_MSG_GET_SYSTEM_INFO_DEVICE_MODE_mIP: s = TEXT("G3 micro IP"); break;
  default:  s = TEXT("unknown"); break;
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Device Mode......................... %s\n"), s);

  ui16 = *((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Hardware_Revision]);
  if (ui16 == 0)
  {
    swprintf(str, TEXT("pre Rev. D"));
  }
  else if ((ui16 & 0xff) >= 'A' && (ui16 & 0xff) <= 'Z')
  {
    if ((ui16 & 0xff00) != 0)
    {
      swprintf(str, TEXT("Rev. %c.%d"), (ui16 & 0xff), (ui16 & 0xff00) >> 8);
    }
    else
    {
      swprintf(str, TEXT("Rev. %c"), (ui16 & 0xff));
    }
  }
  else
  {
    swprintf(str, TEXT("Rev. %d.%d"), (ui16 & 0xff), (ui16 & 0xff00) >> 8);
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Hardware Revision................... %s\n"), str);

  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("EUI................................. %02x:%02x:%02x:%02x:%02x:%02x\n"),
    pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_EUI + 0],
    pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_EUI + 1],
    pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_EUI + 2],
    pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_EUI + 3],
    pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_EUI + 4],
    pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_EUI + 5]
  );

  ui16 = *((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Flags]);
  switch (HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_GET_DATA_PORT(ui16))
  {
  case 0: s = TEXT("SCI-A"); break;
  case 1: s = TEXT("SCI-B"); break;
  default: s = TEXT("unknown"); break;
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Host Port........................... %s\n"), s);
  switch (HCT_MSG_GET_SYSTEM_INFO_PORT_DESIGNATION_GET_DIAG_PORT(ui16))
  {
  case 0: s = TEXT("SCI-A"); break;
  case 1: s = TEXT("SCI-B"); break;
  default: s = TEXT("unknown"); break;
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Diag Port........................... %s\n"), s );

  flags = ((*((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_Flags])) & HCT_MSG_GET_SYSTEM_INFO_FLAGS_MASK);
  if (flags & HCT_MSG_GET_SYSTEM_INFO_FLAGS_USE_RPY)
  {
    s = TEXT("YES");
  }
  else
  {
    s = TEXT("no");
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Apply RPY........................... %s\n"), s);

  if (flags & HCT_MSG_GET_SYSTEM_INFO_FLAGS_AUTO_MODE)
  {
    s = TEXT("YES");
  }
  else
  {
    s = TEXT("no");
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Auto Mode........................... %s\n"), s);

  if ((flags & HCT_MSG_GET_SYSTEM_INFO_PHY_MODE_MASK) & HCT_MSG_GET_SYSTEM_INFO_PHY_ROBO)
  {
    s = TEXT("ROBO");
  }
  else
  {
    s = TEXT("PRIME");
  }
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("PHY Mode............................ %s\n"), s );

  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("MAC_RX_Max_Hold_Ppdu................ %d\n"),
      *((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_MAC_RX_Max_Hold_Ppdu]) );
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("MAC_Max_Conn........................ %d\n"),
      *((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_MAC_Max_Conn]) );
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("MAC_Max_Queue_Len_Per_Conn.......... %d\n"),
      *((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_MAC_Max_Queue_Len_Per_Conn]) );

  ui16 = *((UINT16 *) &pui8[HCT_MSG_RPY_GET_SYSTEM_INFO_OFFSET_MAC_Flags_Dflt_Security_Profile]);
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Dflt. ARQ........................... %s\n"),
      ui16 & HCT_MSG_GET_SYSTEM_INFO_MAC_FLAGS_DFLT_ARQ ? TEXT("ON") : TEXT("off"));
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("Dflt. PAC........................... %s\n"),
      ui16 & HCT_MSG_GET_SYSTEM_INFO_MAC_FLAGS_DFLT_PAC ? TEXT("ON") : TEXT("off"));
  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("MAC_Default_Security_Profile........ %d\n"),
      ui16 & HCT_MSG_GET_SYSTEM_INFO_MAC_DFLT_SECURITY_PROFILE_MASK);

  switch (device_type)
  {
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_TYPE_IEC432:
    ECA_AppEmu_Get_System_Info_Dump_LLC(msg, msglen);
    break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_TYPE_IP:
    ECA_AppEmu_Get_System_Info_Dump_IP(msg, msglen);
    break;
  case HCT_MSG_GET_SYSTEM_INFO_DEVICE_TYPE_G3:
    ECA_AppEmu_Get_System_Info_Dump_G3(msg, msglen);
//    ECA_AppEmu_Get_System_Info_Handler_G3(msg, msglen);//xraf
		break;

	default:
    break;
  }

  HostAppEMU_DbgPrint(TRUE, DIAG_LOG_LEVEL_APPEMU, TEXT("\n"));
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_ALARM_Handler_GENERAL_ALARM
*
* DESCRIPTION:
*       GENERAL_ALARM handler.
*
* Return Value:       none
* Input Parameters:   UINT16 *      alarm data
*                     UINT16        alarm data length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_ALARM_Handler_GENERAL_ALARM(UINT16 *Alarm_Data, UINT16 Alarm_Length)
{
  UINT16 *pui16;

  if (Alarm_Length < HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Invalid message length %d, expected at least %d.\n"),
        __FUNCTION__, Alarm_Length, HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE);
  }

  pui16 = &Alarm_Data[HCT_MSG_ALARM_TYPE_GENERAL_ERROR_OFFSET_Error_Data >> 1];

  switch (Alarm_Data[HCT_MSG_ALARM_TYPE_GENERAL_ERROR_OFFSET_Error >> 1])
  {
  case HCT_MSG_ALARM_TYPE_GENERAL_ERROR_PORT_DISCARDED:
    if (Alarm_Length != (HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE + HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED_SIZE))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: GENERAL_ALARM_PORT_DISCARDED: Invalid message length %d, expected %d.\n"),
          __FUNCTION__, Alarm_Length, HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE + HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED_SIZE);
    }
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: GENERAL_ALARM_PORT_DISCARDED: Octets Discarded %d.\n"),
        __FUNCTION__, pui16[HCT_MSG_ALARM_TYPE_GENERAL_ERROR_PORT_DISCARDED_OFFSET_Octets_Discarded >> 1]);
    break;
  case HCT_MSG_ALARM_TYPE_GENERAL_ERROR_OUT_OF_MEMORY:
    if (Alarm_Length != (HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE + HCT_MSG_ALARM_TYPE_GENERAL_ERROR_OUT_OF_MEMORY_SIZE))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: GENERAL_ALARM_OUT_OF_MEMORY: Invalid message length %d, expected %d.\n"),
          __FUNCTION__, Alarm_Length, HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE);
    }
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: GENERAL_ALARM_OUT_OF_MEMORY.\n"), __FUNCTION__);
    break;
  case HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED:
    if (Alarm_Length != (HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE + HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED_SIZE))
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: GENERAL_ALARM_MESSAGE_DISCARDED: Invalid message length %d, expected %d.\n"),
          __FUNCTION__, Alarm_Length, HCT_MSG_ALARM_TYPE_GENERAL_ERROR_SIZE + HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED_SIZE);
    }
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: GENERAL_ALARM_MESSAGE_DISCARDED: Type 0x%02x Tag 0x%02x, Msg Len 0x%04x.\n"),
          __FUNCTION__,
          pui16[HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED_OFFSET_Msg_Type_Tag >> 1] >> 8,
          pui16[HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED_OFFSET_Msg_Type_Tag >> 1] & 0xff,
          pui16[HCT_MSG_ALARM_TYPE_GENERAL_ERROR_MESSAGE_DISCARDED_OFFSET_Msg_Len >> 1]
          );
    break;
  default:
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Unknown GENERAL_ALARM type 0x%04x, len %d.\n"),
          __FUNCTION__, Alarm_Data[HCT_MSG_ALARM_TYPE_GENERAL_ERROR_OFFSET_Error >> 1], Alarm_Length);
    break;
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_CL_RELEASE_or_DETACH_handler
*
* DESCRIPTION:
*       CL_RELEASE or DETACH handler.
*
* Return Value:       none
* Input Parameters:   HCT_MSG_BUFFER_t *  message payload
*                     UINT16        message payload length
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_CL_RELEASE_or_DETACH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_CL_LLC)
  {
    ECA_AppEmu_CL_RELEASE_handler(pmsg, msglen);
  }
  else if ( ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node) || ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc) )
  {
    ECA_AppEmu_DETACH_handler(pmsg, msglen);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Warning! Unexpected CL_RELEASE or DETACH reply!\n"));
  }

  MyLocalFree(pmsg);
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEMU_CL_Release
*
* DESCRIPTION:
*       LLC CL_RELEASE.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEMU_CL_Release(UINT16 DestAddr)
{
  HCT_MSG_BUFFER_t *pmsg;
  int payload_len;

  HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Sending CL_RELEASE Address %d.\n"), DestAddr);

  payload_len = HCT_MSG_REQ_CL_RELEASE_SIZE;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, payload_len);
  if (pmsg)
  {
    *((UINT16 *) &pmsg[HCT_MSG_REQ_CL_RELEASE_OFFSET_Destination_Address >> 1]) = DestAddr;

    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_CL_RELEASE, (UINT8 *) pmsg, HCT_MSG_REQ_CL_RELEASE_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to send CL_Release.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_CL_RELEASE_handler
*
* DESCRIPTION:
*       LLC CL_RELEASE confirmation handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_CL_RELEASE_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  UINT16 *pui16;

  if (msglen < HCT_MSG_RPY_CL_RELEASE_SIZE_Status)
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_CL_RELEASE_SIZE_Status, msglen);
  }
  else
  {
    pui16 = (UINT16 *) &pmsg[HCT_MSG_RPY_CL_RELEASE_OFFSET_Status >> 1];
    if (*pui16 != 0)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
        TEXT("%S: Error %#x returned from CL_RELEASE!\n"),
            __FUNCTION__, *pui16
      );
    }
    else
    {
      pui16 = (UINT16 *) &pmsg[HCT_MSG_RPY_CL_RELEASE_OFFSET_Destination_Address >> 1];
      if (*pui16 != ECA_AppEmu_mgh.Destination_Address)
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
          TEXT("%S: ERROR! Destination Address mismatch, received %#x, expected %#x\n"),
              __FUNCTION__, *pui16, ECA_AppEmu_mgh.Destination_Address
        );
      }
      else
      {
        HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
          TEXT("CL_RELEASE from Dest Addr %#x Success!\n"), ECA_AppEmu_mgh.Destination_Address
        );
      }
    }
  }

  ECA_AppEmu_mgh.Destination_Address = -1;

  ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Disconnect]++;
  ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;
  ECA_AppEmu_mgh.flags &= ~(0
      | ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS
      | ECA_APPEMU_FLAGS_CONNECTED
      | ECA_APPEMU_FLAGS_DATA_SEND
      | ECA_APPEMU_FLAGS_INACTIVITY_TIMEOUT
  );

  ECA_AppEmu_mgh.connHandle = INVALID_CONNECTION_HANDLE;
  ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_None;
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEMU_DETACH
*
* DESCRIPTION:
*       G3 DETACH.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEMU_DETACH(UINT16 *pExtended_Address)
{
  HCT_MSG_BUFFER_t *pmsg;
  int payload_len;
  UINT8 *pui8;

  payload_len = HCT_MSG_REQ_DETACH_G3_SIZE;

  pmsg = (HCT_MSG_BUFFER_t *) LocalAlloc(LPTR, payload_len);
  if (pmsg)
  {
    ECA_AppEmu_mgh.flags |= ECA_APPEMU_FLAGS_DETACHING;
    if (pExtended_Address)
    {
      memcpy(&pmsg[HCT_MSG_REQ_DETACH_G3_OFFSET_Extended_Address >> 1], pExtended_Address, HCT_MSG_REQ_DETACH_G3_SIZE_Extended_Address);
    }
    else
    {
      /* detach ourself */
      memset(&pmsg[HCT_MSG_REQ_DETACH_G3_OFFSET_Extended_Address >> 1], 0x00, HCT_MSG_REQ_DETACH_G3_SIZE_Extended_Address);
    }
    pui8 = (UINT8 *) &pmsg[HCT_MSG_REQ_DETACH_G3_OFFSET_Extended_Address >> 1];
    
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Sending DETACH Address %#x.%#x.%#x.%#x.%#x.%#x.%#x.%#x.\n"),
      pui8[0], pui8[1], pui8[2], pui8[3], pui8[4], pui8[5], pui8[6], pui8[7]
    );

		if (pui8[0] != 0x77)
		{
			int i = 0;
		}

    HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_DETACH, (UINT8 *) pmsg, HCT_MSG_REQ_DETACH_G3_SIZE, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
  }
  else
  {
    ECA_OutOfMemory();
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Unable to send DETACH.\n"), __FUNCTION__);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DETACH_handler
*
* DESCRIPTION:
*       G3 DETACH confirmation handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_DETACH_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  if (msglen == HCT_MSG_RPY_DETACH_G3_SIZE)
  {
    ECA_AppEmu_DETACH_Confirm_handler(pmsg, msglen);
  }
  else if (msglen == HCT_MSG_IND_DETACH_G3_SIZE)
  {
    ECA_AppEmu_DETACH_Indicate_handler(pmsg, msglen);
  }
  else
  {
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d or %d, received %d!\n"),
          __FUNCTION__, HCT_MSG_RPY_DETACH_G3_SIZE, HCT_MSG_IND_DETACH_G3_SIZE, msglen);
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DETACH_handler
*
* DESCRIPTION:
*       G3 DETACH confirmation handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_DETACH_Confirm_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  UINT16 *pui16;
  UINT8 *pui8;

  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) != HostAppEMU_G3_Node)
  {
    //debug shu testing WARNING DEBUG
    //HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_WARNING, TEXT("Warning! Unexpected DETACH.Confirm!\n"));
    //return;
  }

  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
	{
    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_DETACHING;
	}

  pui16 = (UINT16 *) &pmsg[HCT_MSG_RPY_DETACH_G3_OFFSET_Status >> 1];
  // if (*pui16 != 0)
  // {
  //   HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: Status %#x returned from DETACH!\n"),
  //         __FUNCTION__, *pui16);
  // }
  // else
  // {
    pui8 = (UINT8 *) &pmsg[HCT_MSG_RPY_DETACH_G3_OFFSET_Extended_Address >> 1];
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
      TEXT("%S: DETACH status %#x: %#x.%#x.%#x.%#x.%#x.%#x.%#x.%#x\n"),
          __FUNCTION__, *pui16, pui8[0], pui8[1], pui8[2], pui8[3], pui8[4], pui8[5], pui8[6], pui8[7]);
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
      TEXT("%S: Packets Sent: %d  Packets Received: %d\n\n"), __FUNCTION__, 
        ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Loop_Start], ECA_AppEmu_mgh.stats.recv_pkt[ECA_APPEMU_STEP_Loop_Start]);
  //}

	if (pui8[0] != 0x77 && pui8[0] != 0x55)
	{
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,TEXT("Detach Confirm Failure - Returned bad address\n"));
	}
  if (*pui16 != 0)
  {
    //
    // Detach failed try again..
    //
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,TEXT("Detach Confirm Failure - Resending command\n"));
    ECA_AppEMU_DETACH((UINT16 *)pui8) ;
  }
  else if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
  {
    ECA_AppEmu_mgh.Destination_Address = -1;

    ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Disconnect]++;
    ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;

#if 0 // ATTACH state
    ECA_AppEmu_mgh.flags &= ~(0
        | ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS
        | ECA_APPEMU_FLAGS_CONNECTED
        | ECA_APPEMU_FLAGS_DATA_SEND
        | ECA_APPEMU_FLAGS_INACTIVITY_TIMEOUT
    );
#else // DISCOVER state
      ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_STATE_MASK;
#endif

    ECA_AppEmu_mgh.connHandle = INVALID_CONNECTION_HANDLE;
    ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_None;
  }
  else if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
  {
#if 0
    UINT8 *pea;

    if (msglen != HCT_MSG_IND_DETACH_G3_SIZE)
    {
      HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("%S: ERROR! Message Size Mismatch, expected %d, received %d!\n"),
            __FUNCTION__, HCT_MSG_IND_DETACH_G3_SIZE, msglen);
      return;
    }

    pea = (UINT8 *) &pmsg[HCT_MSG_IND_DETACH_G3_OFFSET_Extended_Address >> 1];
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("DETACH Indication, EA:%02x%02x%02x%02x%02x%02x%02x%02x.\n\n"),
      pea[0], pea[1], pea[2], pea[3], pea[4], pea[5], pea[6], pea[7]
    );

    ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECTED;
#endif
  }
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_DETACH_handler
*
* DESCRIPTION:
*       G3 DETACH confirmation handler.
*
* Return Value:       none
* Input Parameters:   none
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_DETACH_Indicate_handler(HCT_MSG_BUFFER_t *pmsg, UINT16 msglen)
{
  if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Node)
  {
    ECA_AppEmu_mgh.Destination_Address = -1;

    ECA_AppEmu_mgh.stats.send_pkt[ECA_APPEMU_STEP_Disconnect]++;
    ECA_AppEmu_mgh.task_start = ECA_AppEmu_mgh.time_of_day;
    ECA_AppEmu_mgh.flags &= ~(0
        | ECA_APPEMU_FLAGS_CONNECT_IN_PROGRESS
        | ECA_APPEMU_FLAGS_CONNECTED
        | ECA_APPEMU_FLAGS_DATA_SEND
        | ECA_APPEMU_FLAGS_INACTIVITY_TIMEOUT
    );

    ECA_AppEmu_mgh.connHandle = INVALID_CONNECTION_HANDLE;
    ECA_AppEmu_mgh.curr_step = ECA_APPEMU_STEP_None;
  }
  //else if ((HostAppEMU_Flags & HostAppEMU_MODE_MASK) == HostAppEMU_G3_Conc)
  //{
    UINT8 *pea;

    pea = (UINT8 *) &pmsg[HCT_MSG_IND_DETACH_G3_OFFSET_Extended_Address >> 1];
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("DETACH Indication, EA:%02x%02x%02x%02x%02x%02x%02x%02x.\n\n"),
      pea[0], pea[1], pea[2], pea[3], pea[4], pea[5], pea[6], pea[7] );

    //remove for g3 bn
    //ECA_AppEmu_mgh.flags &= ~ECA_APPEMU_FLAGS_CONNECTED;
  //}
}

UINT32 pageSize = 0;
UINT32 imageSize = 0;
UINT32 imageCRC = 0xFF;
UINT32 hostCRC = 0;
UINT8  * pImage = NULL;
UINT8  * pPagesReceived = NULL;
UINT32 numberOfPages = 0;
UINT32 currentPageCount = 0;

const char * imageDirectory = "\\upgrade";
const char * imageFilename  = "\\UpgradeImage.srec";
const char * activeFilename = "\\ActiveImage.srec";

void ReleaseFlashMemory()
{
	if (pImage != NULL)
	{
		LocalFree(pImage);
		pImage = NULL;
	}
	if (pPagesReceived != NULL)
	{
		LocalFree(pPagesReceived);
		pPagesReceived = NULL;
	}

	pageSize = 0;
	imageSize = 0;
	imageCRC = 0xFF;
	hostCRC = 0;
	numberOfPages = 0;
}

void IntializeUpgrade(HCT_MSG_BUFFER_t *msg)
{
	UINT8 *pmsg;
	UINT32 msglen = 0;
	UINT16 status = HCT_MSG_FIRMWARE_SUCCESS;
	//
	// release any memory from previous attempts;
	//
	ReleaseFlashMemory();

	//
	// PLC requesting firmware upgrade.
	//
	HCT_MSG_REQ_FIRMWARE_UPGRADE_s * upgradeReq = (HCT_MSG_REQ_FIRMWARE_UPGRADE_s *)msg;
	//
	// Save off the necessary data..
	//
	pageSize  =((HCT_MSG_REQ_FIRMWARE_UPGRADE_s *)msg)->Page_Size;
	imageSize = ((HCT_MSG_REQ_FIRMWARE_UPGRADE_s *)msg)->Image_Size;
	imageCRC  = ((HCT_MSG_REQ_FIRMWARE_UPGRADE_s *)msg)->Image_CRC32;
  
	HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received Firmware Upgrade Request: Page Size=%d, ImageSize=%d, Image CRC=%x.\n\n"),
    (int)pageSize, (int)imageSize, (unsigned int)imageCRC);

	pPagesReceived = NULL;
	pImage = NULL;

	numberOfPages = 0;
	currentPageCount = 0;
	hostCRC = 0;
	//
	// validate the page size
	//
	if (pageSize != 32 && pageSize != 64 && pageSize != 128 && pageSize != 192)
	{
		status = HCT_MSG_FIRMWARE_INVALID_CONTENTS_LEN;
	}
	else
	{
		//
		// create an array to store the page contents
		// round the number of pages up if there is a remainder..
		//
		numberOfPages = (imageSize + pageSize - 1) / pageSize;   
		pPagesReceived = (UINT8 *)LocalAlloc(LPTR, numberOfPages);
		//
		// create space for the image
		//
		pImage = (UINT8 *) LocalAlloc(LPTR, numberOfPages*pageSize);

		if (pImage == NULL || pPagesReceived == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
			status = HCT_MSG_STATUS_GENERAL_FAILURE;
		}
		if (pPagesReceived != NULL)
		{
			memset(pPagesReceived, 0, sizeof(pPagesReceived));
		}
	}
	//
	// now respond back with the success status..
	//
	msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
	pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
	
	if (pmsg == NULL)
	{
		//
		// memory allocation failure.
		//
		ECA_OutOfMemory();
		status = HCT_MSG_STATUS_GENERAL_FAILURE;
	}
	else
	{
		//
		// Handshake with the PLC
		//
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = status;
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_INITIATE_UPGRADE_SUBTYPE;
		
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
	}
}

void ReceivedFirmwarePacket(HCT_MSG_BUFFER_t *msg)
{	
	UINT32 pageIndex;
	UINT8 *pmsg = NULL;
	UINT32 msglen = 0;
	UINT16 status = HCT_MSG_FIRMWARE_SUCCESS;
	UINT16 crcStatus = 0;
	UINT32 crc32 = 0;
	UINT16 state = 0;
	UINT32 index;
	//
	// Recevied page packet..
	// Save off the necessary data..
	//
	pageIndex =((HCT_MSG_REQ_FIRMWARE_MESSAGE_s *)msg)->Page_Index;

	if (pImage == NULL || pPagesReceived == NULL)
	{
		//
		// if pImage or pPageReceived have not been initialized we are not ready
		//
		HostAppEMU_DbgPrint(TRUE, 
			HostAppEMU_DBG_LEVEL_ERROR, 
			TEXT("Host unprepared to receive firmware upgrade packet: Page Index=%d.\n\n"),
			(int)pageIndex);

		status = HCT_MSG_FIRMWARE_HOST_NOT_PREPARED;
	}
	else
	{
		if ( pageIndex < numberOfPages)
		{
			pPagesReceived[pageIndex]++;
			if(pPagesReceived[pageIndex] == 1)
			{
				currentPageCount++;
			}
			else if (pPagesReceived[pageIndex] > 1)
			{
				HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received duplicate page index: %d.\n"),(int)pageIndex);
			}

			if (currentPageCount %100 == 0)
			{
				HostAppEMU_DbgPrint(TRUE, 
					HostAppEMU_DBG_LEVEL_ERROR, 
					TEXT("Received Firmware Upgrade Packet: Count=%d Page Index=%d.\n"),
					(int)currentPageCount,(int)pageIndex);
			}
			//
			// save the data
			//
			UINT32 offset = pageIndex * pageSize;

			memcpy(&pImage[offset], &((HCT_MSG_REQ_FIRMWARE_MESSAGE_s *)msg)->Page_Contents, pageSize);
		}
		else
		{
			status = HCT_MSG_FIRMWARE_INVALID_PAGE_INDEX;
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received invalid page index: max pages= %d - page index= %d.\n"),(int)numberOfPages,(int)pageIndex);
		}
	}
	//
	// now respond back with the success status..
	//
	msglen = HCT_MSG_RPY_FIRMWARE_MESSAGE_SIZE;
	pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
	
	if (pmsg == NULL)
	{
		//
		// memory allocation failure.
		//
		ECA_OutOfMemory();
		status = HCT_MSG_STATUS_GENERAL_FAILURE;
	}
	else
	{
		if (currentPageCount >= numberOfPages)
		{
			for(index = 0; index < numberOfPages; index++)
			{
				if (pPagesReceived[index] == 0)
				{
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ReceivedFirmwarePacket::Host missing firmware page: %d.\n"),(int)index);
					//
					// If missing packets don't do the crc check..
					//
					state = 1;
				}
			}
			if (status == HCT_MSG_FIRMWARE_SUCCESS && state == 0)
			{
				crcStatus = 1;
				//crc32 = CRC32_BlockChecksum(pImage, imageSize);
				crc32 = crcSlow(pImage, imageSize);
				//
				// Check the segment CRC's
				//
				status = CheckSegmentCRC(pImage);
			}
		}
		//
		// Handshake with the PLC
		//
		((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->Status = status;
		((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_MESSAGE_SUBTYPE;
		((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->Page_Index = pageIndex;
		((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->ValidCRC = crcStatus;
		((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->CRC32 = crc32;
		
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
	}
}

void ReceivedCRCRequest(HCT_MSG_BUFFER_t *msg)
{
	UINT8 *pmsg = NULL;
	UINT32 msglen = 0;
	UINT16 index = 0;
	UINT16 status = HCT_MSG_FIRMWARE_SUCCESS;

	if (pImage == NULL || pPagesReceived == NULL)
	{
		//
		// if pImage or pPageReceived have not been initialized we are not ready
		//
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Host unprepared to calculate CRC.\n\n"));
		status = HCT_MSG_FIRMWARE_HOST_NO_IMAGE;
	}
	else
	{
		//
		// do image checks..
		// see if all the pages have arrived
		//
		for(index = 0; index < numberOfPages ; index++)
		{
			if (pPagesReceived[index] == 0)
			{
				status = HCT_MSG_FIRMWARE_HOST_IMAGE_INCOMPLETE;
				HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Host missing page: %d\n"), (int)index);
			}
		}
		//
		// If we think we have a valid image, check the CRC
		//
		if (status == HCT_MSG_FIRMWARE_SUCCESS)
		{
      //hostCRC = CRC32_BlockChecksum(pImage, imageSize);
			hostCRC = crcSlow(pImage, imageSize);
			//
			// do the crc's compare?
			//
			if (imageCRC != hostCRC)
			{
				status = HCT_MSG_FIRMWARE_HOST_IMAGE_FAILED_CRC;
			}
		}
	}

	HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received Firmware CRC request. Status= %X  PLC CRC= %X  Cal CRC= %X\n\n"),
		(UINT32)status, imageCRC, hostCRC);
	//
	// now respond back with the success status..
	//
	msglen = HCT_MSG_RPY_FIRMWARE_CRC_MESSAGE_SIZE;
	pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
	
	if (pmsg == NULL)
	{
		//
		// memory allocation failure.
		//
		ECA_OutOfMemory();
		status = HCT_MSG_STATUS_GENERAL_FAILURE;
	}
	else
	{
		//
		// Handshake with the PLC
		//
		((HCT_MSG_RPY_FIRMWARE_CRC_s *)pmsg)->Status = status;
		((HCT_MSG_RPY_FIRMWARE_CRC_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_REQ_CRC_SUBTYPE;
		((HCT_MSG_RPY_FIRMWARE_CRC_s *)pmsg)->CRC_Value = hostCRC;
		
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
	}
}

void ReceivedFlashRequest(HCT_MSG_BUFFER_t *msg, int len)
{
	UINT8 *pmsg = NULL;
	UINT32 msglen = 0;
	UINT16 index = 0;
	UINT16 status = HCT_MSG_FIRMWARE_SUCCESS;

	char * currentWorkingDirectory; 

	char * filePath;
	FILE * imageFile = NULL;
	int imageFileWrite = 0;

	struct _stat fileStats; 
	UINT8  *pCompareImage;

	if (pImage == NULL || pPagesReceived == NULL)
	{
		//
		// if pImage or pPageReceived have not been initialized we are not ready
		//
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Host unprepared to flash firmware.\n\n"));
		status = HCT_MSG_FIRMWARE_HOST_NOT_PREPARED;
	}
	else
	{
		//
		// do image checks..
		// see if all the pages have arrived
		//
		for(index = 0; index < numberOfPages; index++)
		{
			if (pPagesReceived[index] == 0)
			{
				HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Host missing page: %d\n"), (int)index);
				status = HCT_MSG_FIRMWARE_HOST_IMAGE_INCOMPLETE;
			}
		}
	}
	//
	// Have all the message, does the CRC check..
	//
	if (status == HCT_MSG_FIRMWARE_SUCCESS)
	{
		//
		// do the crc's compare? if the hostCRC = 0 then it may not have been requested
		//
		if (imageCRC != hostCRC && hostCRC == 0)
		{
				hostCRC = crcSlow(pImage, imageSize);
		}

		if (imageCRC != hostCRC)
		{
			//
			// bad crc
			//
			status = HCT_MSG_FIRMWARE_HOST_IMAGE_FAILED_CRC;
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Bad Firmware CRC. Status= %u PLC= %X  Host= %X\n\n"),
				(UINT32)status, imageCRC, hostCRC);
		}
		else
		{
			//
			// Check the Segment CRC's
			//
			status = CheckSegmentCRC(pImage);
		}
	}
	//
	// if everything is still ok save the image to a file..
	//
	if (status == HCT_MSG_FIRMWARE_SUCCESS)
	{
		currentWorkingDirectory = _getcwd(NULL,256);
		if (currentWorkingDirectory == NULL)
		{
			ECA_OutOfMemory();
			status = HCT_MSG_STATUS_GENERAL_FAILURE;
		}
		else
		{
			filePath = (char *)LocalAlloc(LPTR, strlen(currentWorkingDirectory) + strlen(imageDirectory) + strlen(imageFilename) + 1);
			if (filePath == NULL)
			{
				ECA_OutOfMemory();
				status = HCT_MSG_STATUS_GENERAL_FAILURE;
			}
			else
			{
				strcpy(filePath, currentWorkingDirectory);
				strcat(filePath, imageDirectory);
				//
				// dir may exist so just ignore any error..
				//
				_mkdir(filePath);
				//
				// add the temp file name..
				//
				strcat(filePath, imageFilename);
				//
				// write the file out..
				//
				imageFile = fopen(filePath, "wb");
				if (imageFile == NULL)
				{
					status = HCT_MSG_STATUS_GENERAL_FAILURE;
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unable to create temp image file: %s.\n\n"),filePath);
				}
				else
				{
					imageFileWrite = fwrite(pImage, sizeof(UINT8), imageSize, imageFile);
					if (imageFileWrite != imageSize)
					{
						status = HCT_MSG_STATUS_GENERAL_FAILURE;
						HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unable to write temp image file: %s.\n\n"),filePath);
					}
					fclose(imageFile);
				}
				LocalFree(filePath);
			}
			free(currentWorkingDirectory);
		}
	}		
	//
	// now respond back with the status..
	//
	msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
	pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
	
	if (pmsg == NULL)
	{
		//
		// memory allocation failure.
		//
		ECA_OutOfMemory();
		status = HCT_MSG_STATUS_GENERAL_FAILURE;
	}
	else
	{
		//
		// Handshake with the PLC
		//
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = status;
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_REQ_FLASH_SUBTYPE;
		
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
	}
	//
	// if everything has gone well and we wish to compare
	// the downloaded file to a optional file,
	// do the compare now.
	// We will not flash the device if we do this.
	//
	if (status == HCT_MSG_FIRMWARE_SUCCESS && HostAppEMU_FirmwareFilename == NULL)
	{
		//
		// intialize the flash state and call the flash routine the first time..
		//
		Sleep(1000); //xraf delay for octave test
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Starting Flash Process\r\n"));

		SendFlashStatusPacket(FLASH_BOF);
		flashState = ACK_FLASH_BOF;		//
		//
		// Firmware flash frees the memory so set to NULL here so we don't do it again.
		//
		msg = NULL;
	}
	else if (status == HCT_MSG_FIRMWARE_SUCCESS && HostAppEMU_FirmwareFilename != NULL )
	{
		//
		// Compare the specified file to the flash image..
		//
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Comparing %s to %s.\n\n"),HostAppEMU_FirmwareFilename, filePath );

		_stat(HostAppEMU_FirmwareFilename, &fileStats); 
		msglen = (UINT32)fileStats.st_size;
		
		if (msglen != imageSize)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Image and compare file are different sizes: %s.\n\n"),HostAppEMU_FirmwareFilename);
		}
		else
		{
			imageFile = fopen(HostAppEMU_FirmwareFilename, "rb");
			if (imageFile == NULL)
			{
				HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unable to open file for compare: %s.\n\n"),HostAppEMU_FirmwareFilename);
			}
			else
			{
				pCompareImage = (UINT8 *) LocalAlloc(LPTR, msglen);
				if (pCompareImage == NULL)
				{
					ECA_OutOfMemory();
					status = HCT_MSG_STATUS_GENERAL_FAILURE;
				}
				else
				{
					if (fread(pCompareImage, 1, msglen, imageFile) != msglen)
					{
						HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unable to read compare file: %s.\n\n"),HostAppEMU_FirmwareFilename);
					}
					else
					{
						if (memcmp(pCompareImage, pImage, msglen) != 0)
						{
							HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Images do not compare.\n\n"));
						}
						else
						{
							HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Images compare.\n\n"));
						}
					}
					LocalFree(pCompareImage);
				}
				fclose(imageFile);
			}
		}
		//
		// If comparing files then exit, do not attempt to flash
		//
		unsigned long flags = HostAppEMU_Exit;
		HostAppEMU_SetFlags(flags);
	}
}

void ReceivedAbort(HCT_MSG_BUFFER_t *msg)
{
	UINT16 flag = ((HCT_MSG_REQ_ABORT_s *)msg)->Flag;
	UINT16 status = HCT_MSG_FIRMWARE_SUCCESS;
	UINT8 *pmsg = NULL;
	UINT32 msglen = 0;
	FILE * imageFile = NULL;
	char * activeImageFilePath = NULL;
	char * currentWorkingDirectory = NULL;
	//
	// Handshake with the PLC
	//
	msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
	pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
	
	if (pmsg == NULL)
	{
		//
		// memory allocation failure.
		//
		ECA_OutOfMemory();
		status = HCT_MSG_STATUS_GENERAL_FAILURE;
	}
	else
	{
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = status;
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_ABORT_SUBTYPE;
		
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
	}

	if (flag == HCT_MSG_DO_NOT_FLASH)
	{
		//
		// don't flash, just wait until it tries again..
		//
	}
	else
	{
		if (pImage != NULL)
		{
			LocalFree(pImage);

		}
		//
		// Reflash original image..
		//
		currentWorkingDirectory = _getcwd(NULL,256);
		if (currentWorkingDirectory == NULL)
		{
			ECA_OutOfMemory();
			status = HCT_MSG_STATUS_GENERAL_FAILURE;
		}
		else
		{

			activeImageFilePath = (char *)LocalAlloc(LPTR, strlen(currentWorkingDirectory) + strlen(imageDirectory) + strlen(activeFilename) + 1);
			if (activeImageFilePath == NULL)
			{
				ECA_OutOfMemory();
				status = HCT_MSG_STATUS_GENERAL_FAILURE;
			}
			else
			{
				strcpy(activeImageFilePath, currentWorkingDirectory);
				strcat(activeImageFilePath, imageDirectory);
				strcat(activeImageFilePath, activeFilename);

				imageFile = fopen(activeImageFilePath, "rb");
				if (imageFile == NULL)
				{
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ReceivedAbort::Unable to open file to restore image: %s.\n\n"),activeImageFilePath);
				}
				else
				{
					pImage = (UINT8 *) LocalAlloc(LPTR, msglen);
					if (pImage == NULL)
					{
						ECA_OutOfMemory();
						status = HCT_MSG_STATUS_GENERAL_FAILURE;
					}
					else
					{
						if (fread(pImage, 1, msglen, imageFile) != msglen)
						{
							HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("ReceivedAbort::Unable to read file: %s.\n\n"),activeImageFilePath);
						}
						else
						{
							//
							// Resend the last active image and flash this image
							// This will remove the new image downloaded
							//
					    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Reflashing original firmware\n"));
							SendFlashStatusPacket(FLASH_BOF);
							flashState = ACK_FLASH_BOF;		//
						}
					}
					fclose(imageFile);
				}
			}
		}
	}
	if (activeImageFilePath != NULL)
	{
		LocalFree(activeImageFilePath);
	}
	if (currentWorkingDirectory != NULL)
	{
		free(currentWorkingDirectory);
	}
}

void TestRpyMessages(int testValue)
{
	UINT8 *pmsg = NULL;
	UINT32 msglen = 0;

	if (testValue == HCT_MSG_REQ_FIRMWARE_ABORT_SUBTYPE)
	{
		msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
		pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);

		if (pmsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
		}
		else
		{
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = 1;
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_ABORT_SUBTYPE;
			
			HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
		}
	}	
	//
	// now respond back with the status..
	//
	
	if (testValue == HCT_MSG_REQ_FIRMWARE_REQ_FLASH_SUBTYPE)
	{
		msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
		pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
		
		if (pmsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
		}
		else
		{
			//
			// Handshake with the PLC
			//
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = 2;
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_REQ_FLASH_SUBTYPE;
			
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
		}
	}

	if (testValue == HCT_MSG_REQ_FIRMWARE_REQ_CRC_SUBTYPE)
	{
		msglen = HCT_MSG_RPY_FIRMWARE_CRC_MESSAGE_SIZE;
		pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);

		if (pmsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
		}
		else
		{
			//
			// Handshake with the PLC
			//
			((HCT_MSG_RPY_FIRMWARE_CRC_s *)pmsg)->Status = 3;
			((HCT_MSG_RPY_FIRMWARE_CRC_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_REQ_CRC_SUBTYPE;
			((HCT_MSG_RPY_FIRMWARE_CRC_s *)pmsg)->CRC_Value = 0xDEAD;
			
			HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
		}
	}
	
	if (testValue == HCT_MSG_REQ_FIRMWARE_MESSAGE_SUBTYPE)
	{
		msglen = HCT_MSG_RPY_FIRMWARE_MESSAGE_SIZE;
		pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
		
		if (pmsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
		}
		else
		{
			//
			// Handshake with the PLC
			//
			((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->Status = 4;
			((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_MESSAGE_SUBTYPE;
			((HCT_MSG_RPY_FIRMWARE_MESSAGE_s *)pmsg)->Page_Index = 0xF;
			
			HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
		}
	}
	//
	// now respond back with the success status..
	//
	if (testValue == HCT_MSG_REQ_FIRMWARE_INITIATE_UPGRADE_SUBTYPE)
	{
		msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
		pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
		
		if (pmsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
		}
		else
		{
			//
			// Handshake with the PLC
			//
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = 5;
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_INITIATE_UPGRADE_SUBTYPE;
			
			HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
		}
	}
	//
	// now respond back with the success status..
	//
	if (testValue == HCT_MSG_REQ_FIRMWARE_ACTIVATE_REQ_SUBTYPE)
	{
		msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
		pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
		
		if (pmsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
		}
		else
		{
			//
			// Handshake with the PLC
			//
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = 6;
			((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_ACTIVATE_REQ_SUBTYPE;
			
			HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
		}
	}
	//
	// wait a couple of seconds then exit.
	//
	_sleep(10000);
	_exit(testValue);
}

void ReceivedActivateRequest(HCT_MSG_BUFFER_t *msg)
{
	struct _stat fileStats; 
	time_t lTime;
	struct tm today;

	char * currentWorkingDirectory = NULL;
	char * filePath = NULL;
	char * imageFilePath = NULL;
	char * tempFilePath = NULL;

	char temp[20];

	
	UINT8 *pmsg = NULL;
	UINT32 msglen = 0;

	UINT16 status = HCT_MSG_FIRMWARE_SUCCESS;
  UINT16 index;

	if (pImage == NULL)
	{
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Activation Failed: No firmware image.\n\n"));
		status = HCT_MSG_FIRMWARE_HOST_NO_IMAGE;
	}
	else if (pPagesReceived == NULL)
	{
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Activation Failed: No firmware image.\n\n"));
		status = HCT_MSG_FIRMWARE_HOST_NO_IMAGE;
	}
	else
	{
		//
		// check to see if all pages have been downloaded.
		//
		for(index = 0; index < numberOfPages; index++)
		{
			if (pPagesReceived[index] == 0)
			{
				status = HCT_MSG_FIRMWARE_HOST_IMAGE_INCOMPLETE;
				HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Activation Failed: Incomplete download. Missing page %d\n"), (int)index);
			}
		}
	}

	if (status == HCT_MSG_FIRMWARE_SUCCESS)
	{
		if (imageCRC != hostCRC )
		{
			status = HCT_MSG_FIRMWARE_HOST_IMAGE_FAILED_CRC;
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Activation Failed: Bad Image CRC.\n\n"));
		}
		else if (flashState != FLASH_LOAD_DONE)
		{
			status = HCT_MSG_FIRMWARE_NOT_FLASHED;	
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Activation Failed: PLC has not been flashed.\n\n"));
		}
	}

	if (status == HCT_MSG_FIRMWARE_SUCCESS )
	{
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Activating new image.\n\n"));
		currentWorkingDirectory = _getcwd(NULL,256);
		if (currentWorkingDirectory == NULL)
		{
			ECA_OutOfMemory();
			status = HCT_MSG_STATUS_GENERAL_FAILURE;
		}
		else
		{
			//
			// Rename the current active file to archive it.
			// Add the file create time to the end of the file name.
			//
			filePath = (char *)LocalAlloc(LPTR, strlen(currentWorkingDirectory) + strlen(imageDirectory) + strlen(activeFilename) + 1);
			if (filePath == NULL)
			{
				ECA_OutOfMemory();
				status = HCT_MSG_STATUS_GENERAL_FAILURE;
			}
			else
			{
				strcpy(filePath, currentWorkingDirectory);
				strcat(filePath, imageDirectory);
				//
				// dir probably exists so just ignore any error..
				//
				_mkdir(filePath);
				
				strcat(filePath, activeFilename);
				//
				// rename or delete the old file but only if it exists
				//
				if (_stat(filePath, &fileStats) == -1)
				{
					//HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Previous active file not found: %s.\n\n"),filePath);
				}
				else
				{
					time(&lTime);
					_localtime64_s(&today, &lTime);
					strftime(temp, sizeof(temp), ".%Y%m%d%H%M%S", &today);

					tempFilePath = (char *) LocalAlloc(LPTR, strlen(filePath) + strlen(temp) + 1);
					strcpy(tempFilePath, filePath);
					strcat(tempFilePath, temp);

					if (rename(filePath, tempFilePath) != 0)
					{
						//
						// can't rename so try deleting
						//
						if (_unlink(filePath) != 0)
						{
							HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unable to delete file: %s.\n\n"), filePath);
							status = HCT_MSG_FIRMWARE_UNABLE_TO_ACTIVATE;
						}
					}
				}
				//
				// Now rename the new image
				//
				if (status == HCT_MSG_FIRMWARE_SUCCESS)
				{
					imageFilePath = (char *)LocalAlloc(LPTR, strlen(currentWorkingDirectory) + strlen(imageDirectory) + strlen(imageFilename) + 1);
					if (imageFilePath == NULL)
					{
						ECA_OutOfMemory();
						status = HCT_MSG_STATUS_GENERAL_FAILURE;
					}
					else
					{
						strcpy(imageFilePath, currentWorkingDirectory);
						strcat(imageFilePath, imageDirectory);
						strcat(imageFilePath, imageFilename);

						if (rename(imageFilePath, filePath) != 0)
						{
							HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unable to rename new image file to activate: %s to %s.\n\n"),imageFilePath, filePath);
							status = HCT_MSG_FIRMWARE_UNABLE_TO_ACTIVATE;
						}
					}
				}
			}
		}
	}
	msglen = HCT_MSG_RPY_FIRMWARE_STATUS_MESSAGE_SIZE;
	pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);

	if (pmsg == NULL)
	{
		//
		// memory allocation failure.
		//
		ECA_OutOfMemory();
	}
	else
	{
		//
		// Handshake with the PLC
		//
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->Status = status;
		((HCT_MSG_RPY_FIRMWARE_s *)pmsg)->SubType = HCT_MSG_REQ_FIRMWARE_ACTIVATE_REQ_SUBTYPE;
		
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_UPGRADE, pmsg, msglen, false);
	}

	if (currentWorkingDirectory != NULL)
	{
		free(currentWorkingDirectory);
	}
	if (tempFilePath != NULL)
	{
		LocalFree(tempFilePath);
	}
	if (filePath != NULL)
	{
		LocalFree(filePath);
	}
	if (imageFilePath)
	{
		LocalFree(imageFilePath);
	}
}

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_FirmwareUpgrade
*
* DESCRIPTION:
*       Receive the flash file from the PLC
*
* Return Value:       none
* Input Parameters:   msg - pointer to message buffer
*											len - length of message buffer
   
* Output Parameters:  none
******************************************************************************/
void ECA_AppEmu_FirmwareUpgrade(HCT_MSG_BUFFER_t *msg, UINT16 len)
{
	//
	// Get the message type
	//
	UINT16 subtype = ((UINT16) msg[HCT_MSG_REQ_FIRMWARE_SUBTYPE_OFFSET >> 1]);

	if (subtype == HCT_MSG_REQ_FIRMWARE_INITIATE_UPGRADE_SUBTYPE) 
	{
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("HostAppEmu.FirmwareUpgrade.Init\r\n"));
		flashComplete = false;
		IntializeUpgrade(msg);
	}
	else if (subtype == HCT_MSG_REQ_FIRMWARE_MESSAGE_SUBTYPE)
	{
		ReceivedFirmwarePacket(msg);
	}
	else if (subtype == HCT_MSG_REQ_FIRMWARE_REQ_CRC_SUBTYPE)
	{
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("HostAppEmu.FirmwareUpgrade.ReceivedCRC request\r\n"));
		ReceivedCRCRequest(msg);
	}
	else if (subtype == HCT_MSG_REQ_FIRMWARE_REQ_FLASH_SUBTYPE)
	{
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("HostAppEmu.FirmwareUpgrade.ReceivedFlash request\r\n"));
		ReceivedFlashRequest(msg, len);
	}
	else if (subtype == HCT_MSG_REQ_FIRMWARE_ACTIVATE_REQ_SUBTYPE)
	{
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("HostAppEmu.FirmwareUpgrade.ActivateNewImage request\r\n"));
		ReceivedActivateRequest(msg);
		//
		// Done so release resources..
		//
		ReleaseFlashMemory();
	}
	else if (subtype == HCT_MSG_REQ_FIRMWARE_ABORT_SUBTYPE)
	{
    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("HostAppEmu.FirmwareUpgrade.ReceivedAbort request\r\n"));
		ReceivedAbort(msg);
		//
		// Going to start over so release resources.
		//
		ReleaseFlashMemory();
	}
	else
	{
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Invalid Firmware Upgrade Subtype: %u.\n\n"),(int)subtype);
		//
		// quit
		//
		ReleaseFlashMemory();

		unsigned long flags = HostAppEMU_Exit;
		HostAppEMU_SetFlags(flags);
	}

	if (msg != NULL)
	{
		LocalFree(msg);
		msg = NULL;
	}
}

UINT16 packetNumber;
UINT32 imageIndex; 
UINT32 startImageIndex;
BOOL   binaryFlash = false;
// max packet size is 256
// first four bytes is address.
// next two is the data length
//
UINT16 currentSectionNumber = 0;
UINT32 currentSectionLength = 0;
UINT32 currentSectionAddress = 0;
UINT32 currentSectionCRC = 0;
UINT32 currentSectionOffset = 0;
UINT32 currentLength = 0;

class SRECList
{
	static SRECList * first;
	static SRECList * last;
	SRECList * next;

	UINT32 msgLength;
	UINT32 offset;
	UINT16 packetNumber;

	SRECList::SRECList(UINT16 packetnum, UINT32 textLength, UINT32 textOffset)
	{
		packetNumber = packetnum;
		offset = textOffset;
		msgLength = textLength;

		next = NULL;

		if (first == NULL)
		{
			first = this;
			last = this;
			next = NULL;
		}
		else
		{
			last->next = this;
			last = this;
		}
	}

public:
	static void Delete(void)
	{
		SRECList* next = first;
		while (next != NULL)
		{
			SRECList * temp = next->next;
			delete next;
			next = temp;
		}
    first = NULL;
    last = NULL;
	}

	static void Add(UINT16 packetNum, UINT16 length, UINT32 msgOffset)
	{
		SRECList * srl = new SRECList(packetNum, length, msgOffset);
	}

	static SRECList * GetPacket(UINT16 packet)
	{
		SRECList * srl = first;
		while (srl != NULL && srl->packetNumber != packet)
		{
			srl = srl->next;
		}
		return srl;
	}

	static void ResendPacket(UINT16 packet)
	{
		SRECList * srl = GetPacket(packet);
		if (srl != NULL)
		{
			UINT8 *	pmsg = (UINT8 *) LocalAlloc(LPTR, HCT_MSG_FLASH_MESSAGE_SIZE);
			//
			// Set the packet and payload type
			//
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PacketNumber = srl->packetNumber;
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PayloadType = SREC_PAYLOAD_TYPE;
			
			memcpy( &((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload, &pImage[srl->offset], srl->msgLength);
			
			HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_FIRMWARE_FLASH, pmsg, HCT_MSG_FLASH_MESSAGE_PAYLOAD_OFFSET+srl->msgLength , false);
			flashState = ACK_FLASH_PACKET;
		}
	}
};


class BinaryRecordList
{
	static BinaryRecordList * first;
	static BinaryRecordList * last;
	BinaryRecordList * next;

	UINT32 msgLength;
	UINT16 packetNumber;
	UINT32 startingAddress;
	UINT16 recordLength;
	UINT32 offset;

	BinaryRecordList::BinaryRecordList(UINT16 packetNum, UINT32 address, UINT16 length, UINT32 msgOffset)
	{
		packetNumber = packetNum;
		startingAddress = address;
		msgLength = length;
		offset = msgOffset;
		next = NULL;

		if (first == NULL)
		{
			first = this;
			last = this;
			next = NULL;
		}
		else
		{
			last->next = this;
			last = this;
		}
	}
	
public:
	static void Delete(void)
	{
		BinaryRecordList* next = first;
		while (next != NULL)
		{
			BinaryRecordList * temp = next->next;
			delete next;
			next = temp;
		}
    first = NULL;
    last = NULL;
	}

	static void Add(UINT16 packetNum, UINT32 address, UINT16 length, UINT32 msgOffset)
	{
		BinaryRecordList * brl = new BinaryRecordList(packetNum, address, length, msgOffset);
	}

	static BinaryRecordList * GetPacket(UINT16 packet)
	{
		BinaryRecordList * brl = first;
		while (brl != NULL && brl->packetNumber != packet)
		{
			brl = brl->next;
		}
		return brl;
	}

	static void ResendPacket(UINT16 packet)
	{
		BinaryRecordList * brl = GetPacket(packet);
		if (brl != NULL)
		{
			UINT8 *	pmsg = (UINT8 *) LocalAlloc(LPTR, HCT_MSG_FLASH_MESSAGE_SIZE);
			//
			// Set the packet and payload type
			//
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PacketNumber = brl->packetNumber;
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PayloadType = SREC_BINARY_PAYLOAD_TYPE;
			//
			// copy starting address...
			//
			*((DWORD *) &((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_ADDRESS_OFFSET]) = (DWORD)brl->startingAddress;
			//
			// copy the length.
			// the length is in words not bytes
			//
			UINT16 tempLength = (UINT16)(brl->msgLength >> 1);
			*((WORD *) &((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_LENGTH_OFFSET]) = tempLength;
			//
			// copy data..
			//
			for (unsigned int i = 0; i < brl->msgLength; i += 2)
			{
				/* byte swap for firmware */
				((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_DATA_OFFSET+i] = pImage[brl->offset+i+1];
				((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_DATA_OFFSET+i+1] = pImage[brl->offset+i];
			}

			//	unsigned char crc = CRC8_BlockChecksum( &pImage[startImageIndex], currentLength);
			//	((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_DATA_OFFSET+currentLength] = crc;

			HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_FIRMWARE_FLASH, pmsg, HCT_MSG_FLASH_MESSAGE_PAYLOAD_OFFSET + HCT_MSG_FLASH_MESSAGE_DATA_OFFSET + brl->msgLength , false);
			
			flashState = ACK_FLASH_PACKET;
		}
	}
};

BinaryRecordList * BinaryRecordList::first = NULL;
BinaryRecordList * BinaryRecordList::last = NULL;
SRECList * SRECList::first = NULL;
SRECList * SRECList::last = NULL;

/******************************************************************************
* FUNCTION NAME: ECA_AppEmu_FirmwareFlash
*
* DESCRIPTION:
*       Flashing the PLC 
*
* Return Value:       none
* Input Parameters:   msg - pointer to message buffer
*											len - length of message buffer
   
* Output Parameters:  none
******************************************************************************/

void ECA_AppEmu_FirmwareFlash(HCT_MSG_BUFFER_t *msg, UINT16 len)
{
	UINT8 *pmsg = NULL;
	UINT32 msglen = 0;
	UINT16 status = 0;
	UINT16 tempPacketNumber=0;

	switch (flashState)
	{
	case ACK_FLASH_BOF:
		BinaryRecordList::Delete();
		SRECList::Delete();

		status = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->Status;
		tempPacketNumber = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->PacketNumber;

		if (status == FLASH_STATUS_SUCCESS && tempPacketNumber == FLASH_BOF)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("Received Ack_Flash_BOF - Sending first Packet\r\n"));
			//
			// start sending data
			//
			packetNumber = 1;
			imageIndex = 0;
			startImageIndex = 0;
			if (pImage[0] == 'S')
			{
				binaryFlash = false;
				SendFlashPacket();
			}
			else
			{
				binaryFlash = true;
				//
				// Skip the number of segments..
				//
				UINT32 offset = sizeof(UINT16);
				memcpy(&currentSectionNumber, &pImage[offset], sizeof(currentSectionNumber));
				offset += sizeof(currentSectionNumber);
				memcpy(&currentSectionAddress, &pImage[offset], sizeof(currentSectionAddress));
				offset += sizeof(currentSectionAddress);
				memcpy(&currentSectionLength, &pImage[offset], sizeof(currentSectionLength));
				offset += sizeof(currentSectionLength);
				memcpy(&currentSectionCRC, &pImage[offset], sizeof(currentSectionCRC));
				startImageIndex = offset + sizeof(currentSectionCRC);

				currentSectionOffset = 0;

				SendBinaryFlashPacket();
			}
		}
		else if (status == FLASH_STATUS_BAD_CRC)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_BOF - Bad CRC: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// resend the BOF
			//
			SendFlashStatusPacket(FLASH_BOF);
			flashState = ACK_FLASH_BOF;
		}
		else if (status == FLASH_STATUS_ABORT)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_BOF - Abort: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			
			flashState = FLASH_LOAD_DONE;
		}
		else if (status == FLASH_STATUS_UTILTIY_MISSING)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_BOF - Utility_Missing: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			
			flashState = FLASH_LOAD_DONE;
		}
		else if (status == FLASH_STATUS_ERASE_PENDING || status == FLASH_STATUS_ERASE_COMPLETE)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_BOF - Status only: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			
			flashState = ACK_FLASH_BOF;
		}
		else
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_BOF - Unknown Status: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// Unknown status
			//
			SendFlashStatusPacket(FLASH_ABORT);
			flashState = ACK_FLASH_ABORT;
		}

		break;

	case ACK_FLASH_PACKET:
		status = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->Status;
		tempPacketNumber = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->PacketNumber;

		if (status == FLASH_STATUS_SUCCESS)
		{
			if (packetNumber % 100 == 0)
			{
		    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
					TEXT("At Ack_Flash_Packet - Success: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			}

			packetNumber++;
			
			if (binaryFlash == false)
			{
				startImageIndex = imageIndex+1;
				SendFlashPacket();
			}
			else
			{
				currentSectionOffset += currentLength;
				startImageIndex += currentLength;

				if (startImageIndex >= imageSize)
				{
					HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("At Sending Flash EOF\r\n"));
					SendFlashStatusPacket(FLASH_EOF);
					flashState = ACK_FLASH_EOF;
				}
				else
				{
					currentLength = 0;
					if (currentSectionOffset >= currentSectionLength)
					{
						UINT32 offset = startImageIndex;
						memcpy(&currentSectionNumber, &pImage[offset], sizeof(currentSectionNumber));
						offset += sizeof(currentSectionNumber);
						memcpy(&currentSectionAddress, &pImage[offset], sizeof(currentSectionAddress));
						offset += sizeof(currentSectionAddress);
						memcpy(&currentSectionLength, &pImage[offset], sizeof(currentSectionLength));
						offset += sizeof(currentSectionLength);
						memcpy(&currentSectionCRC, &pImage[offset], sizeof(currentSectionCRC));
						
						startImageIndex = offset + sizeof(currentSectionCRC);
						currentSectionOffset = 0;;
				    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
							TEXT("Starting new section: Address %0x  Length= %u\r\n"), currentSectionAddress, currentSectionLength);
					}
					SendBinaryFlashPacket();
				}
			}
		}
		else if (status == FLASH_STATUS_BAD_CRC)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_Packet - Bad CRC: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// resend last packet
			//
			if (binaryFlash == TRUE)
			{
				SendBinaryFlashPacket();
			}
			else
			{
				SendFlashPacket();
			}
		}
		else if (status == FLASH_STATUS_MISSING_PACKET)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_Packet - Missing Packet: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// missing packet
			//
			if (binaryFlash == TRUE)
			{
				//
				// This function is currently a empty function..
				//
				BinaryRecordList::ResendPacket(tempPacketNumber);
			}
			else
			{
				SRECList::ResendPacket(tempPacketNumber);
			}
		}
		else if (status == FLASH_STATUS_ERASE_PENDING || status == FLASH_STATUS_ERASE_COMPLETE)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_PACKET - Status only: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			flashState = ACK_FLASH_PACKET;
		}
		else if (status == FLASH_STATUS_ABORT)
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_Packet - Received Abort: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// Any other status send abort.
			//
			flashState = FLASH_ERROR;
		}
		else
		{
	    HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_Packet - Bad status: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// Any other status send abort.
			//
			flashState = FLASH_ERROR;
		}
		break;

	case SEND_FLASH_EOF:
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("At Sending Flash EOF\r\n"));
		SendFlashStatusPacket(FLASH_EOF);
		flashState = ACK_FLASH_EOF;
		break;
		
	case ACK_FLASH_EOF:
		status = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->Status;
		tempPacketNumber = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->PacketNumber;

		if (status == FLASH_STATUS_SUCCESS && tempPacketNumber == FLASH_EOF )
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, 
				TEXT("At Ack_Flash_EOF: Success= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			flashState = FLASH_LOAD_DONE;
			//
			// New code added for PLC with NO EEPROM
			//
			CreateThread(NULL, 0, HostAppEMU_FirmwareReset, 0, 0, NULL);
			//HostAppEMU_FirmwareReset(NULL);
			BinaryRecordList::Delete();
			SRECList::Delete();
		}
		else if (status == FLASH_STATUS_BAD_CRC)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("At Ack_Flash_EOF: Bad CRC= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// resend the EOF
			//
			SendFlashStatusPacket(FLASH_EOF);
			flashState = ACK_FLASH_EOF;
		}
		else if (status == FLASH_STATUS_ABORT)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("At Ack_Flash_EOF: Received Abort= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			flashState = FLASH_LOAD_DONE;
		}
		else if (status == FLASH_STATUS_ABORT)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("At Ack_Flash_Packet - Received Abort: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// Any other status send abort.
			//
			flashState = FLASH_ERROR;
		}
		else if (status == FLASH_STATUS_ERASE_PENDING || status == FLASH_STATUS_ERASE_COMPLETE)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("At Ack_Flash_Eof - Status only: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			flashState = ACK_FLASH_EOF;
		}
		else
		{
			//
			// Any other status send abort.
			//
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("At Ack_Flash_EOF: Unknown Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			flashState = FLASH_ERROR;
		}

		break;

	case ACK_FLASH_ABORT:
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,TEXT("At Ack_Flash_Abort\r\n"));
		flashState = FLASH_LOAD_DONE;
		break;

	case FLASH_LOAD_DONE:   
		status = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->Status;
		tempPacketNumber = ((HCT_MSG_FLASH_MESSAGE_RPY_s *)msg)->PacketNumber;

		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
			TEXT("At Flash Load Done: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
		
		if (status == FLASH_STATUS_ERASE_PENDING || status == FLASH_STATUS_ERASE_COMPLETE)
		{
			flashState = FLASH_LOAD_DONE;
		}
		else if (status == FLASH_STATUS_SUCCESS)
		{
			flashState = FLASH_LOAD_DONE;
		}
		else if (status == FLASH_STATUS_ERASE_PENDING || status == FLASH_STATUS_ERASE_COMPLETE)
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("At Flash_Load_Done - Status only: Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			flashState = FLASH_LOAD_DONE;
		}
		else
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("At Flash Load Done: Unknown Status= %x  PacketNumber= %u\r\n"), (int)status, (int)tempPacketNumber);
			//
			// Any other status send abort.
			//
			flashState = FLASH_ERROR;
		}
		break;

	default:
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,
				TEXT("Unknown State= %x  PacketNumber= %u\r\n"), (int)flashState, (int)tempPacketNumber);
		//
		// Unknow state - abort..
		//
		flashState = FLASH_ERROR;
		break;
	}

	if (flashState == FLASH_ERROR)
	{
		BinaryRecordList::Delete();
		SRECList::Delete();
		//
		// Send abort
		//
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR,TEXT("Flash Error - Sending Abort\r\n"));

		SendFlashStatusPacket(FLASH_ABORT);
		flashState = ACK_FLASH_ABORT;
	}

	if (msg != NULL)
	{
		LocalFree(msg);
		msg = NULL;
	}
}
DWORD WINAPI HostAppEMU_RemoteReset(LPVOID lpParam)
{
	ECA_ResetDevice();
	Sleep(2000);

	bool status = false;
	int count = 0;
	while (status == false && count < 3)
	{
		Sleep(2000);
		status = ECA_Reinitialize();
		count++;
	}
	return 0;
}
DWORD WINAPI HostAppEMU_FirmwareReset(LPVOID lpParam)
{
	bool status = false;
	int count = 0;
	while (status == false && count < 3)
	{
		Sleep(2000);
		status = ECA_SetSystemConfig();
		count++;
	}
	ECA_ResetDevice();
	Sleep(2000);
	HostAppEMU_SendSyncMsgToPLC(HCT_MSG_TYPE_GET_SYSTEM_INFO, NULL, 0, (ECA_AppEmu_mgh.flags & ECA_APPEMU_FLAGS_USE_RPY) ? TRUE : FALSE);
	ECA_SetPhyTxParams();
	ECA_SetPhyRxParams();

	return 0;
}

void SendFlashStatusPacket( UINT16 status )
{
	UINT32 msglen = HCT_MSG_FLASH_MESSAGE_SIZE;
	UINT8 * pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);

	memset(pmsg, 0, msglen);
	
	if (pmsg == NULL)
	{
		//
		// memory allocation failure.
		//
		ECA_OutOfMemory();
		status = FLASH_ERROR;
	}
	else
	{
		//
		// Start flash process
		//
		((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PacketNumber = status;
		if (binaryFlash == TRUE)
		{
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PayloadType = SREC_BINARY_PAYLOAD_TYPE; 
		}
		else
		{
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PayloadType = SREC_PAYLOAD_TYPE;
		}

		memcpy(&(((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload), &imageSize, sizeof(imageSize));
		HostAppEMU_SendMsgToPLC( HCT_MSG_TYPE_FIRMWARE_FLASH, pmsg, HCT_MSG_FLASH_MESSAGE_PAYLOAD_OFFSET+ sizeof(UINT32), false);
	}
}

void SendBinaryFlashPacket(void)
{
	UINT32 msglen = 0;
	UINT8 * pmsg = NULL;
	BOOL foundRec = FALSE;

	msglen = HCT_MSG_FLASH_MESSAGE_SIZE;
	pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
	memset(pmsg, 0, msglen);
	//
	// get the max len of the data payload
	//
	currentLength = HTC_MSG_FLASH_PAYLOAD_MAX_SIZE - HCT_MSG_FLASH_MESSAGE_DATA_OFFSET-16; 

	UINT32 temp = currentLength + currentSectionOffset;
	//
	// If there is less than the current length of data left adjust the length to be sent
	//
	if (temp > currentSectionLength)
	{
		currentLength = currentSectionLength - currentSectionOffset;
	}
	//
	// Figure out the new starting address..
	//
	UINT32 startingAddress = currentSectionAddress + (currentSectionOffset >> 1);
	//
	// Set the packet and payload type
	//
	((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PacketNumber = packetNumber;
	((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PayloadType = SREC_BINARY_PAYLOAD_TYPE;
	//
	// copy starting address...
	//
  *((DWORD *) &((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_ADDRESS_OFFSET]) = (DWORD)startingAddress;
	//
	// copy the length.
	// the length is in words not bytes
	//
	UINT16 tempLength = (UINT16)(currentLength >> 1);
  *((WORD *) &((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_LENGTH_OFFSET]) = tempLength;
	//
	// copy data..
	//
	unsigned int i;
	for (i = 0; i < currentLength; i += 2)
	{
		/* byte swap for firmware */
		((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_DATA_OFFSET+i] = pImage[startImageIndex+i+1];
		((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_DATA_OFFSET+i+1] = pImage[startImageIndex+i];
	}

	BinaryRecordList::Add(packetNumber, startingAddress, currentLength, startImageIndex);
	
//	unsigned char crc = CRC8_BlockChecksum( &pImage[startImageIndex], currentLength);
//	((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload[HCT_MSG_FLASH_MESSAGE_DATA_OFFSET+currentLength] = crc;

	HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_FIRMWARE_FLASH, pmsg, HCT_MSG_FLASH_MESSAGE_PAYLOAD_OFFSET + HCT_MSG_FLASH_MESSAGE_DATA_OFFSET + currentLength , false);
	
	flashState = ACK_FLASH_PACKET;
}

void SendFlashPacket(void)
{
	UINT32 msglen = 0;
	UINT8 * pmsg = NULL;
	BOOL foundRec = FALSE;
	UINT16 len = 0;
	//
	// Find the first 0x0a and send the record;;
	//
	for(imageIndex = startImageIndex; imageIndex < imageSize && imageIndex - startImageIndex < HTC_MSG_FLASH_PAYLOAD_MAX_SIZE; imageIndex++)
	{
		len++;
		if (pImage[imageIndex]== 0x0A)
		{
			if (pImage[startImageIndex] != 'S')
			{
				//
				// Unknown record type, skip
				//
				startImageIndex = imageIndex+1;
				len = 0;
			}
			else
			{
				//
				// found a good rec...
				//
				foundRec = TRUE;
				break;
			}
		}
	}
	if (packetNumber >= FLASH_EOF)
	{
		flashState = FLASH_ERROR;
		HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Images is too large- packet #= %d - size= %u\r\n"), (int)packetNumber, imageSize );
	}
	else if (foundRec == FALSE)
	{
		SendFlashStatusPacket(FLASH_EOF);
		flashState = ACK_FLASH_EOF;
	}
	else
	{
		//
		// don't send the /n /r ..
		//
		len -= 2;

		msglen = HCT_MSG_FLASH_MESSAGE_SIZE;
		pmsg = (UINT8 *) LocalAlloc(LPTR, msglen);
		memset(pmsg, 0, msglen);

		if (pmsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
			flashState = FLASH_ERROR;
		}
		else
		{
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PacketNumber = packetNumber;
			((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->PayloadType = SREC_PAYLOAD_TYPE;
			memcpy( &((HCT_MSG_FLASH_MESSAGE_s *)pmsg)->Payload, &pImage[startImageIndex], len);
			
			HostAppEMU_SendMsgToPLC(HCT_MSG_TYPE_FIRMWARE_FLASH, pmsg, HCT_MSG_FLASH_MESSAGE_PAYLOAD_OFFSET+len , false);
			flashState = ACK_FLASH_PACKET;

			if (imageIndex >= imageSize-1)
			{
				flashState = SEND_FLASH_EOF;
			}

			SRECList::Add(packetNumber,len, startImageIndex);
		}
	}
}
void ECA_AppEMU_Get_MAC_PIB(HCT_MSG_BUFFER_t * msg, UINT16 len)
{
	UINT8 * pmsg = (UINT8 *)msg;
	UINT16 status = *((UINT16 *) &pmsg[HCT_MSG_RPY_GET_MAC_PIB_OFFSET_Status]);
	UINT16 attributeId = *((UINT16 *) &pmsg[HCT_MSG_RPY_GET_MAC_PIB_OFFSET_Attr_ID]);
	UINT16 length = *((UINT16 *) &pmsg[HCT_MSG_RPY_GET_MAC_PIB_OFFSET_Length]);
	UINT8  value = *((UINT8 *) &pmsg[HCT_MSG_RPY_GET_MAC_PIB_OFFSET_Value]);
	HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received MAC PIC 0x24: Status = %d  Value = %d\r\n"), (int)status, (int)value);

	LocalFree(msg);
}

HCT_MSG_BLOB_s * pFWBlobArray[4] = { NULL, NULL, NULL, NULL };

void ECA_AppEmu_WriteBlob(HCT_MSG_BUFFER_t *msg, UINT16 len)
{
	UINT8 * pmsg = (UINT8 *) msg;
	UINT16 blobId   = *((UINT16 *) &pmsg[HCT_MSG_WRITE_BLOB_MESSAGE_BLOB_ID_OFFSET]);
	UINT16 blobLength = *((UINT16 *)&pmsg[HCT_MSG_WRITE_BLOB_MESSAGE_BLOB_LENGTH_OFFSET]);

	HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received Blob Write: Id= %d - size= %d\r\n"), (int)blobId, (int)blobLength);

	UINT16 status = HCT_MSG_STATUS_OK;
	
	int blobAllocateSize = sizeof(HCT_MSG_BLOB_s);// + blobLength;
	UINT8 * pBlob = (UINT8 *)LocalAlloc(LPTR, blobAllocateSize);
	if (pBlob == NULL)
	{
		ECA_OutOfMemory();
	}
	else
	{
		((HCT_MSG_BLOB_s*)pBlob)->Blob_Id = blobId;
		((HCT_MSG_BLOB_s*)pBlob)->Blob_Length = blobLength;
		if (blobLength > 0)
		{
			memcpy(&pBlob[HCT_MSG_WRITE_BLOB_MESSAGE_BLOB_DATA_OFFSET], &pmsg[HCT_MSG_WRITE_BLOB_MESSAGE_BLOB_DATA_OFFSET], blobLength);
			printf("Wrtie Blob id %d - 0x%p - ", blobId, pBlob);
			for(int index= 0; index < blobLength; index++)
			{
				printf("0x%X ", pmsg[HCT_MSG_WRITE_BLOB_MESSAGE_BLOB_DATA_OFFSET+index]);
			}
			printf("\n");
		}

		if (blobId <= BLOB_ID_PMEM)
		{
			if (pFWBlobArray[blobId] != NULL)
			{
				LocalFree(pFWBlobArray[blobId]);
			}
			pFWBlobArray[blobId] = (HCT_MSG_BLOB_s *)pBlob;
		}
		else
		{
			HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Unknown Blob: #= %d\r\n"), (int)blobId);
			status = ERR_UNKNOWN_BLOB;
		}

		int msglen = HCT_MSG_WRITE_BLOB_MESSAGE_RPY_SIZE;
		UINT8 * pRpyMsg = (UINT8 *) LocalAlloc(LPTR, msglen);

		if (pRpyMsg == NULL)
		{
			//
			// memory allocation failure.
			//
			ECA_OutOfMemory();
		}
		else
		{
//			memset(pRpyMsg, 0, msglen);
			((HCT_MSG_WRITE_BLOB_MESSAGE_RPY_s *)pRpyMsg)->Status = status;
			HostAppEMU_SendMsgToPLC(HCT_MGG_TYPE_FIRMWARE_WRITE_BLOB, pRpyMsg, HCT_MSG_WRITE_BLOB_MESSAGE_RPY_SIZE , false);
		}
	}
	LocalFree(msg);
}
UINT16 CheckSegmentCRC(UINT8 * pImage)
{
	UINT16 status = HCT_MSG_FIRMWARE_SUCCESS;
	UINT16 numberOfSegments;
	memcpy(&numberOfSegments, pImage, 2);
	int offset = 2;
	for(int segNum=0; segNum < numberOfSegments; segNum++)
	{
		//
		// read the segment header.
		//
		UINT16 currentSectionNumber;
	  UINT32 currentSectionAddress;
		UINT32 currentSectionLength;
		UINT32 currentSectionCRC;

		memcpy(&currentSectionNumber, &pImage[offset], 2);
		offset += 2;
		memcpy(&currentSectionAddress, &pImage[offset], 4);
		offset += 4;
		memcpy(&currentSectionLength, &pImage[offset], 4);
		offset += 4;
		memcpy(&currentSectionCRC, &pImage[offset], 4);
		offset += 4;

		if (currentSectionCRC != 0)
		{
			//
			// Compute crc32
			//
			UINT32 sectionCRC = crcSlow(&pImage[offset], currentSectionLength);
			if (sectionCRC != currentSectionCRC)
			{
				status = HCT_MSG_FIRMWARE_HOST_IMAGE_BAD;
				HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Bad Firmware Segment CRC32. Segment Num: %u Status= %u PLC= %X  Host= %X\n\n"),
				(UINT32)currentSectionNumber, (UINT32)status, imageCRC, hostCRC);
				break;
			}
		}
		offset += (int)currentSectionLength;
	}
	return status;
}
void ECA_AppEmu_ReadBlob(HCT_MSG_BUFFER_t *msg, UINT16 len)
{
	UINT8 * pmsg = (UINT8 *) msg;
	UINT16 blobId     =*((UINT16 *)&pmsg[HCT_MSG_WRITE_BLOB_MESSAGE_BLOB_ID_OFFSET]);
	UINT16 blobLength =*((UINT16 *)&pmsg[HCT_MSG_WRITE_BLOB_MESSAGE_BLOB_LENGTH_OFFSET]);

	HostAppEMU_DbgPrint(TRUE, HostAppEMU_DBG_LEVEL_ERROR, TEXT("Received Blob Read: Id= %d - size= %d\r\n"), (int)blobId, (int)blobLength);

	UINT16 status = HCT_MSG_STATUS_OK;

	UINT8 * pBlob = NULL;

	if (blobId <= BLOB_ID_PMEM)
	{
		pBlob = (UINT8 *)pFWBlobArray[blobId];
	}

	if (pBlob != NULL)
	{
		blobLength = ((HCT_MSG_BLOB_s *) pBlob)->Blob_Length;
	}

	int msglen = HCT_MSG_READ_BLOB_MESSAGE_RPY_SIZE + blobLength;
	UINT8 * pRpyMsg = (UINT8 *) LocalAlloc(LPTR, msglen);
	memset(pRpyMsg, 0, msglen);

	if (pRpyMsg == NULL)
	{
	//
	// memory allocation failure.
	//
		ECA_OutOfMemory();
	}
	else
	{
		((HCT_MSG_READ_BLOB_MESSAGE_RPY_s *)pRpyMsg)->Blob_Id = blobId;
		((HCT_MSG_READ_BLOB_MESSAGE_RPY_s *)pRpyMsg)->Blob_Length = blobLength;

		if (pBlob != NULL)
		{
			memcpy(& pRpyMsg[HCT_MSG_READ_BLOB_MESSAGE_RPY_BLOB_DATA_OFFSET], & pBlob[HCT_MSG_READ_BLOB_MESSAGE_RPY_BLOB_DATA_OFFSET], blobLength);

			printf("Blob id %d - 0x%p - ", blobId, pBlob);
			for(int index= 0; index < blobLength; index++)
			{
				printf("0x%X ", pBlob[HCT_MSG_READ_BLOB_MESSAGE_RPY_BLOB_DATA_OFFSET+index]);
			}
			printf("\n");

		}
		HostAppEMU_SendMsgToPLC(HCT_MGG_TYPE_FIRMWARE_READ_BLOB, pRpyMsg, HCT_MSG_READ_BLOB_MESSAGE_RPY_SIZE + blobLength , false);
	}
	LocalFree(msg);
}
