/******************************************************************************
* FILE PURPOSE: OS_FIFO implementation.
*******************************************************************************
*
* FILE NAME: os_fifo.c
*
* DESCRIPTION:
*       This file contains all functions for implementing an OS_FIFO object.
*
* HISTORY:
* 09/01/2005    stephen smith           initial revision
* 10/08/2007    stephen smith           ported from DTVAPE to use in AppEMUC.
* 01/17/2008    stephen smith           enhanced to allow multiple processes to use
* 12/08/2009    stephen smith           port from TIWiMAXWrapper DLL
*
* LIST OF FUNCTIONS:
*   os_fifo_create
*   os_fifo_delete
*   os_fifo_open
*   os_fifo_close
*   os_fifo_put
*   os_fifo_get
*   os_fifo_peek
*   os_fifo_pop
*   os_fifo_count
*   os_fifo_stats
*   os_fifo_reset_stats
*
* Copyright (c) 2005-2009 Texas Instruments Inc.  All rights reserved.
******************************************************************************/
#include "stdafx.h"

#include <stdlib.h>
#include <strsafe.h>

#include "hostappemu.h"
#include "os_fifo.h"

static void os_fifo_temp_name(LPTSTR buf, size_t bufsize);

/******************************************************************************
* FUNCTION NAME: os_fifo_temp_name
*
* DESCRIPTION:
*       Generates a temporary name.
*
* Return Value:       none
* Input Parameters:   LPTSTR  buffer to place tempname into
*                     size_t  size of buffer
* Output Parameters:  none
******************************************************************************/
static void os_fifo_temp_name(LPTSTR buf, size_t bufsize)
{
  static DWORD dw = 0;

  if (dw == 0)
  {
    srand(GetTickCount());
  }
  dw = ((rand() << 16) | rand());
  StringCbPrintf(buf, bufsize, TEXT("%x"), dw);
}

/******************************************************************************
* FUNCTION NAME: os_fifo_create
*
* DESCRIPTION:
*       Creates an OS_FIFO object.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
*                    PCHAR       pointer to name of FIFO object
*                    UINT        size of a FIFO element
*                    UINT        count of FIFO elements
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_create(POS_FIFO_CONTROL pfifo_control, TCHAR *name_ptr, UINT element_size, UINT element_count)
{
   memset(pfifo_control, 0, sizeof(*pfifo_control));

   if (name_ptr && wcslen(name_ptr) > 0)
   {
      StringCbCopy(pfifo_control->name, sizeof(pfifo_control->name), name_ptr);
   }
   else
   {
      os_fifo_temp_name(pfifo_control->name, sizeof(pfifo_control->name));
   }

   pfifo_control->active        = TRUE;
   pfifo_control->itemsize      = element_size;
   pfifo_control->count         = element_count;
   pfifo_control->fifo_bufsize  = LEN32(element_size) * element_count;

   pfifo_control->c_access   = 0;

   return OS_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_delete
*
* DESCRIPTION:
*       Deletes an OS_FIFO object.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_delete(POS_FIFO pfifo)
{
  if (pfifo->pfifo_control)
  {
    WaitForSingleObject(pfifo->mutex, INFINITE);
    pfifo->pfifo_control->active = FALSE;
    ReleaseMutex(pfifo->mutex);

    for (;;)
    {
      UINT32 i;

      WaitForSingleObject(pfifo->mutex, INFINITE);
      i = pfifo->pfifo_control->c_access;
      ReleaseMutex(pfifo->mutex);
      if (i == 0)
      {
        break;
      }

      // signal all pending
      ReleaseSemaphore(pfifo->item_in, 1, NULL);
      ReleaseSemaphore(pfifo->item_out, 1, NULL);

      Sleep(10);
    }
  }

  os_fifo_close(pfifo);

  return OS_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_open
*
* DESCRIPTION:
*       Opens a previously created OS_FIFO object.  This is useful when another
*       process needs access to the fifo.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure to open
*                    POS_FIFO    pointer to opened FIFO control structure
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_open(POS_FIFO pfifo, POS_FIFO_CONTROL pfifo_control, PVOID fifo_buffer)
{
  TCHAR buf[OS_MAX_NAME + 20];
  OS_STATUS status;

  pfifo->pfifo_control = pfifo_control;

  if (fifo_buffer)
  {
    pfifo->fifo_buf = fifo_buffer;
  }
  else
  {
    pfifo->fifo_buf = (PVOID) GlobalAlloc(GMEM_FIXED, pfifo_control->fifo_bufsize);
    if (!pfifo->fifo_buf)
    {
      return OS_STATUS_NO_MEMORY;
    }
  }

  StringCbPrintf(buf, sizeof(buf), TEXT("%s.mutex"), pfifo_control->name);
  pfifo->mutex = CreateMutex(NULL, FALSE, buf);
  if (!pfifo->mutex)
  {
    HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: CreateMutex failed!\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
  }

  StringCbPrintf(buf, sizeof(buf), TEXT("%s.semai"), pfifo_control->name);
  pfifo->item_in = CreateSemaphore(NULL, pfifo->pfifo_control->count, pfifo->pfifo_control->count, buf);
  if (!pfifo->item_in)
  {
    HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: CreateSemaphore failed!\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
  }

  StringCbPrintf(buf, sizeof(buf), TEXT("%s.semao"), pfifo_control->name);
  pfifo->item_out = CreateSemaphore(NULL, 0, pfifo->pfifo_control->count, buf);
  if (!pfifo->item_out)
  {
    HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: CreateSemaphore failed!\n"), __FUNCTION__);
    HostAppEMU_DbgPrintLastError();
  }

  if ((!pfifo->item_out) || (!pfifo->item_out) || (!pfifo->item_out))
  {
    os_fifo_close(pfifo);
    status = OS_STATUS_ERROR;
  }
  else
  {
    status = OS_STATUS_SUCCESS;
  }

  return status;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_close
*
* DESCRIPTION:
*       Closes a previously opened OS_FIFO object.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure to open
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_close(POS_FIFO pfifo)
{
  CloseHandle(pfifo->item_in);
  pfifo->item_in = INVALID_HANDLE_VALUE;
  CloseHandle(pfifo->item_out);
  pfifo->item_out = INVALID_HANDLE_VALUE;
  CloseHandle(pfifo->mutex);
  pfifo->mutex = INVALID_HANDLE_VALUE;

  pfifo->fifo_buf = 0;

  return OS_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_put
*
* DESCRIPTION:
*       Puts an item into the OS_FIFO object.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
*                    PVOID       pointer to buffer that contains item to store
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_put(POS_FIFO pfifo, PVOID pdata)
{
   SINT32 inext;
   PVOID p;
   DWORD dw;

   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   pfifo->pfifo_control->c_access++;
   ReleaseMutex(pfifo->mutex);

   /* wait for an available slot */
   dw = WaitForSingleObject(pfifo->item_in, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      if (dw != WAIT_ABANDONED)
      {
        HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
        HostAppEMU_DbgPrintLastError();
      }
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   /* lock the fifo from outside changes */
   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      if (dw != WAIT_ABANDONED)
      {
        HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
        HostAppEMU_DbgPrintLastError();
      }
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   if (!pfifo->pfifo_control->active)
   {
      pfifo->pfifo_control->c_access--;
      ReleaseMutex(pfifo->mutex);             // unlock the fifo
      return OS_STATUS_INTERRUPTED;
   }

   p = ((UINT8 *) pfifo->fifo_buf) + (pfifo->pfifo_control->ihead * LEN32(pfifo->pfifo_control->itemsize));
   memcpy(p, pdata, pfifo->pfifo_control->itemsize);

   inext = pfifo->pfifo_control->ihead + 1;
   if (inext == pfifo->pfifo_control->count)
   {
      inext = 0;
   }
   pfifo->pfifo_control->ihead = inext;

   pfifo->pfifo_control->stats.curr_items++;
   if (pfifo->pfifo_control->stats.curr_items > (UINT32) pfifo->pfifo_control->count)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: BROKEN!!!\n"), __FUNCTION__);
   }

   // update stats
   if (pfifo->pfifo_control->stats.max_items < pfifo->pfifo_control->stats.curr_items)
   {
     pfifo->pfifo_control->stats.max_items = pfifo->pfifo_control->stats.curr_items;
   }

   // notify of slot available
   if (!ReleaseSemaphore(pfifo->item_out, 1, NULL))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseSemaphore failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   pfifo->pfifo_control->c_access--;

   // unlock the fifo
   if (!ReleaseMutex(pfifo->mutex))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseMutex failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   return OS_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_get
*
* DESCRIPTION:
*       Gets an item from the OS_FIFO object.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
*                    PVOID       pointer to buffer to store item into
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_get(POS_FIFO pfifo, PVOID pdata)
{
   INT inext;
   PVOID p;
   DWORD dw;

   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   pfifo->pfifo_control->c_access++;
   ReleaseMutex(pfifo->mutex);

   /* wait for an available slot */
   dw = WaitForSingleObject(pfifo->item_out, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   /* lock the fifo from outside changes */
   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      pfifo->pfifo_control->c_access--;
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   if (!pfifo->pfifo_control->active)
   {
     pfifo->pfifo_control->c_access--;
     ReleaseMutex(pfifo->mutex);                // unlock the fifo
     return OS_STATUS_INTERRUPTED;
   }

   p = ((UINT8 *) pfifo->fifo_buf) + (pfifo->pfifo_control->itail * LEN32(pfifo->pfifo_control->itemsize));
   memcpy(pdata, p, pfifo->pfifo_control->itemsize);

   inext = pfifo->pfifo_control->itail + 1;
   if (inext == pfifo->pfifo_control->count)
   {
      inext = 0;
   }
   pfifo->pfifo_control->itail = inext;

   pfifo->pfifo_control->stats.curr_items--;
   if (pfifo->pfifo_control->stats.curr_items > (UINT32) pfifo->pfifo_control->count)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: BROKEN!!!\n"), __FUNCTION__);
   }

   // notify of slot available
   if (!ReleaseSemaphore(pfifo->item_in, 1, NULL))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseSemaphore failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   pfifo->pfifo_control->c_access--;

   // unlock the fifo
   if (!ReleaseMutex(pfifo->mutex))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseMutex failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   return OS_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_peek
*
* DESCRIPTION:
*       Gets an item from the OS_FIFO object.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
*                    PVOID       pointer to buffer to store item into
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_peek(POS_FIFO pfifo, PVOID pdata)
{
   PVOID p;
   DWORD dw;
   OS_STATUS status;

   /* lock the fifo from outside changes */
   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   if (!pfifo->pfifo_control->active)
   {
      ReleaseMutex(pfifo->mutex);               // unlock the fifo
      return OS_STATUS_INTERRUPTED;
   }

   if (pfifo->pfifo_control->stats.curr_items > 0)
   {
      p = ((UINT8 *) pfifo->fifo_buf) + (pfifo->pfifo_control->itail * LEN32(pfifo->pfifo_control->itemsize));
      memcpy(pdata, p, pfifo->pfifo_control->itemsize);
   }
   else
   {
      status = OS_STATUS_EOF;
   }

   // unlock the fifo
   if (!ReleaseMutex(pfifo->mutex))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseMutex failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   return status;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_pop
*
* DESCRIPTION:
*       Pops an item off the fifo.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_pop(POS_FIFO pfifo)
{
   INT inext;
   DWORD dw;
   OS_STATUS status;

   /* lock the fifo from outside changes */
   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   if (!pfifo->pfifo_control->active)
   {
      ReleaseMutex(pfifo->mutex);               // unlock the fifo
      return OS_STATUS_INTERRUPTED;
   }

   if (pfifo->pfifo_control->stats.curr_items > 0)
   {
      inext = pfifo->pfifo_control->itail + 1;
      if (inext == pfifo->pfifo_control->count)
      {
         inext = 0;
      }
      pfifo->pfifo_control->itail = inext;

      pfifo->pfifo_control->stats.curr_items--;

      if (!ReleaseSemaphore(pfifo->item_in, 1, NULL))
      {
         HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseSemaphore failed %#x!\n"), __FUNCTION__, dw);
         HostAppEMU_DbgPrintLastError();
      }

      status = OS_STATUS_SUCCESS;
   }
   else
   {
      status = OS_STATUS_EOF;
   }

   // unlock the fifo
   if (!ReleaseMutex(pfifo->mutex))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseMutex failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   return status;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_count
*
* DESCRIPTION:
*       Retrieves the count of items in the fifo.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
*                    PSINT32     pointer to receive current items count
*                    PSINT32     pointer to receive count
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_count(POS_FIFO pfifo, PSINT32 pcount, PSINT32 pmaxcount)
{
   DWORD dw;

   if ( (!pcount) || (!pmaxcount) )
   {
      return OS_STATUS_BAD_PARAM;
   }

   /* lock the fifo from outside changes */
   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   *pcount = (SINT32) pfifo->pfifo_control->stats.curr_items;
   *pmaxcount = (SINT32) pfifo->pfifo_control->count;

   // unlock the fifo
   if (!ReleaseMutex(pfifo->mutex))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseMutex failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   return OS_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_stats
*
* DESCRIPTION:
*       Retrieves the stats of this fifo.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
*                    APE_MSG_FIFO_STATS pointer to FIFO stats
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_stats(POS_FIFO pfifo, POS_FIFO_STATS pstats)
{
   DWORD dw;

   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   memcpy(pstats, &pfifo->pfifo_control->stats, sizeof(pfifo->pfifo_control->stats));

   // unlock the fifo
   if (!ReleaseMutex(pfifo->mutex))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseMutex failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   return OS_STATUS_SUCCESS;
}

/******************************************************************************
* FUNCTION NAME: os_fifo_reset_stats
*
* DESCRIPTION:
*       Resets the stats of this fifo.
*
* Return Value:      OS_STATUS   OS_SUCCESS or errorlevel
* Input Parameters:  POS_FIFO    pointer to FIFO control structure
* Output Parameters: none
* Functions Called:
******************************************************************************/
OS_STATUS os_fifo_reset_stats(POS_FIFO pfifo)
{
   DWORD dw;

   dw = WaitForSingleObject(pfifo->mutex, INFINITE);
   if (dw != WAIT_OBJECT_0)
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: WaitForSingleObject failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
      return OS_STATUS_SEMAPHORE_ERROR;
   }

   pfifo->pfifo_control->stats.max_items            = 0;
   pfifo->pfifo_control->stats.max_wait_put_item    = 0;
   pfifo->pfifo_control->stats.max_wait_put_fifo    = 0;
   pfifo->pfifo_control->stats.max_wait_get_fifo    = 0;

   // unlock the fifo
   if (!ReleaseMutex(pfifo->mutex))
   {
      HostAppEMU_DbgPrint(TRUE, 0, TEXT("%S: ReleaseMutex failed %#x!\n"), __FUNCTION__, dw);
      HostAppEMU_DbgPrintLastError();
   }

   return OS_STATUS_SUCCESS;
}
