/**************************************************************************************************************************************************
*       init_ADS1x9x.c  -  Set's up an ADS1x9x device.  All initialization calls are initiated from main.c                                      *
*                                                                                                                                                 *
*       Author:             Luke Duncan                                                                                                         *
*                                                                                                                                                 *
*       Revision Date:      June 2009                                                                                                             *
*                                                                                                                                                 *
*       Revision Level:     1st pass                                                                                                              *
*                                                                                                                                                 *
*       For Support:        https://e2e.ti.com/support/development_tools/mavrk/default.aspx                                                       *
*                                                                                                                                                 *
***************************************************************************************************************************************************
*       Copyright  2009-2010 Texas Instruments Incorporated - http://www.ti.com/                                                                 *
***************************************************************************************************************************************************
*  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met: *
*                                                                                                                                                 *
*    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.                 *
*                                                                                                                                                 *
*    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the        *
*    documentation and/or other materials provided with the distribution.                                                                         *
*                                                                                                                                                 *
*    Neither the name of Texas Instruments Incorporated nor the names of its contributors may be used to endorse or promote products derived      *
*    from this software without specific prior written permission.                                                                                *
*                                                                                                                                                 *
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT          *
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT     *
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT         *
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY    *
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE      *
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                                                           *
***************************************************************************************************************************************************
*                                 MODULE CHANGE LOG                                                                                               *
*                                                                                                                                                 *
*       Date Changed:                            Developer:                                                                                       *
*       Change Description:                                                                                                                       *
*                                                                                                                                                 *
**************************************************************************************************************************************************/
/**************************************************************************************************************************************************
*                                 Included Headers                                                                                                *
**************************************************************************************************************************************************/
#include "stdef.h"
#include "ADS1x9x.h"
#include "F5438_Modular_EVM_IO.h"
#include "F5438_usci.h"
#include "ECG_PC_Interface.h"
#include "UART_State_Machine.h"

/**************************************************************************************************************************************************
*                                 Global Variables                                                                                                *
**************************************************************************************************************************************************/
unsigned char ADS1x9x_SPI_data;

// volatile register_name_type *register_name = (volatile register_name_type*)0xADD;
volatile unsigned char ADS1x9x_Config_1;
volatile ADS1x9x_Config_1_Register_type *ADS1x9x_Config_1_register = (volatile ADS1x9x_Config_1_Register_type*) &ADS1x9x_Config_1;
volatile unsigned char ADS1x9x_Config_2;
volatile ADS1x9x_Config_2_Register_type *ADS1x9x_Config_2_register = (volatile ADS1x9x_Config_2_Register_type*) &ADS1x9x_Config_2;
volatile unsigned char ADS1x9x_Config_3;
volatile ADS1x9x_Config_3_Register_type *ADS1x9x_Config_3_register = (volatile ADS1x9x_Config_3_Register_type*) &ADS1x9x_Config_3;
volatile unsigned char ADS1x9x_Config_4;
volatile ADS1x9x_Config_4_Register_type *ADS1x9x_Config_4_register = (volatile ADS1x9x_Config_4_Register_type*) &ADS1x9x_Config_4;

unsigned char ADS1x9x_Default_Register_Settings [22] = 
{
  0x86, // Config1
  0x20, // Config2
  0xC0, // Config3
  0x00, // loff
  0x00, // ch1set
  0x00, // ch2set
  0x00, // ch3set
  0x00, // ch4set
  0x00, // ch5set
  0x00, // ch6set
  0x00, // ch7set
  0x00, // ch8set
  0x00, // rld_sensp
  0x00, // rld_sensm
  0x00, // loff_sensp
  0x00, // loff_sensm
  0x00, // loff_flip
  0x00, // gpio
  0x00, // resp
  0x00, // pace
  0x00  // config4
};

const unsigned char *ADS1x9x_download_pointer = ADS1x9x_Default_Register_Settings;

volatile unsigned char ADS1x9x_Lead_Off_Control;
volatile ADS1x9x_Lead_Off_Control_Register_type *ADS1x9x_Lead_Off_Control_Register = (volatile ADS1x9x_Lead_Off_Control_Register_type*)&ADS1x9x_Lead_Off_Control;

volatile unsigned char ADS1x9x_Channel_Settings;
volatile ADS1x9x_Channel_Settings_Register_type *ADS1x9x_Channel_Settings_Register = (volatile ADS1x9x_Channel_Settings_Register_type*)&ADS1x9x_Channel_Settings;

volatile unsigned char ADS_1298_Channel_Stack [8] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

unsigned char ADS1x9x_SPI_Settings [6] = {LOW_POLARITY, RETARDED_DATA, _8MHZ_MAX_CLOCK, AFE_ADS1x9x, 0x00, SLOT_NUMBER_1};

volatile unsigned char* ADS1x9x_SPI_TX_BUFFER [4];
volatile unsigned char* ADS1x9x_SPI_RX_BUFFER [4];
volatile USCI_Interrupt_Flag_Status_type* ADS1x9x_SPI_Interrupt_Flags[4];

volatile unsigned char ADS1x9x_Version_ID_Number;


/**************************************************************************************************************************************************
*                                 External Variables                                                                                              *
**************************************************************************************************************************************************/
extern volatile unsigned char Print_UART_Message_Busy_Flag;

extern volatile unsigned char ECG_Num_Channels;                                 // Number of ECG channels to use

/**************************************************************************************************************************************************
*    Initialize the ADS1x9x device                                                                                                                *
**************************************************************************************************************************************************/
unsigned char init_ADS1x9x (unsigned char device_slot)
{
    unsigned char Verify_Check = CLEAR;
    unsigned char Module_Present = CLEAR;
    unsigned char number_of_retries = 10;
    unsigned char i;

    init_ADS1x9x_Data_Ready_Interrupt (device_slot);                            // Set up the ADS1x9x Interrupt pin
    enable_ADS1x9x_Interrupt (device_slot);                                     // Interrupts required for SPI communication
    
    init_ADS1x9x_IO (device_slot);    
    
    Power_Up_ADS1x9x (device_slot);                                             // Power up Digital portion of the ADS1x9x
                                                     
    POR_Reset_ADS1x9x (device_slot);

    Stop_Read_Data_Continuous (device_slot);

    while (!Module_Present)                                                     // Wait for Module to be present
    {
        if (number_of_retries)
        {
          ADS1x9x_Version_ID_Number = ADS1x9x_Read_Version (device_slot);
    
          if (ADS1x9x_Version_ID_Number == ADS1x9x_VERSION_ID)                  // (0x22 for old board, 0x42 for new one)                 
          {
              Module_Present = SET;  

#ifdef TERMINAL

              while (Print_UART_Message_Busy_Flag);                             // Prevent overwrite of UART FIFO  
              UART_PrintF( "\r\n ADS1x9x Found ....", 20);

#endif  /* Terminal */

          }
          number_of_retries--;
        }
        else
        {

#ifdef TERMINAL
            
              while (Print_UART_Message_Busy_Flag);                             // Prevent overwrite of UART FIFO  
              UART_PrintF ("\r\n ADS1x9x Not Found ....", 23);

#endif  /* Terminal */
              
              return ADS_1x9x_NOT_FOUND;
        }
    } 
   
    init_ADS1x9x_Via_Constant_Table (device_slot, (unsigned char*) ADS1x9x_download_pointer);
    
    Verify_Check = verify_ADS1x9x_Registers (device_slot, (unsigned char*) ADS1x9x_download_pointer);
    
    Verify_Check = Initialize_ADS1x9x_Data_Rate (device_slot, MODULATION_FREQUENCY_DIVIDED_BY_1024);    // DEFAULT_MODULATION_FREQUENCY_DIVIDED_BY_16
                                                                                                        // MODULATION_FREQUENCY_DIVIDED_BY_32
                                                                                                        // MODULATION_FREQUENCY_DIVIDED_BY_64 
                                                                                                        // MODULATION_FREQUENCY_DIVIDED_BY_128
                                                                                                        // MODULATION_FREQUENCY_DIVIDED_BY_256
                                                                                                        // MODULATION_FREQUENCY_DIVIDED_BY_512
                                                                                                        // MODULATION_FREQUENCY_DIVIDED_BY_1024
    
    Verify_Check = Initialize_ADS1x9x_Mode (device_slot, HIGH_RESOLUTION_MODE);                         // DEFAULT_LOW_POWER_MODE, HIGH_RESOLUTION_MODE
     
    for (i = 0; i < ECG_Num_Channels; i++)
    {
        Verify_Check = Initialize_ADS1x9x_Channel                               //  Context Save will store the previous setting of the channel 
        (
            device_slot, 
            i + 1,                                                              // references channels 1 - 8
            DEFAULT_ADS1x9x_ELECTRODE_INPUT,                                    // DEFAULT_ADS1x9x_ELECTRODE_INPUT, ADS1x9x_INPUT_SHORTED, ADS1x9x_RIGHT_LEG_DETECT, ADS1x9x_ONE_HALF_DIGITAL_SUPPLY
                                                                                // ADS1x9x_TEMPERATURE_SENSOR, ADS1x9x_CALIBRATION_SIGNAL, ADS1x9x_RIGHT_LEG_DETECT_POSITIVE, ADS1x9x_RIGHT_LEG_DETECT_NEGATIVE
            DEFAULT_GAIN_OF_6,                                                  // DEFAULT_GAIN_OF_6, GAIN_OF_1, GAIN_OF_2, GAIN_OF_3, GAIN_OF_4, GAIN_OF_8, GAIN_OF_12
            DEFAULT_DISABLE_POWER_DOWN,                                         // DEFAULT_DISABLE_POWER_DOWN, ENABLE_POWER_DOWN
            IGNORE_PREVIOUS_STATE                                               // CONTEXT_SAVE_CHANNEL, IGNORE_PREVIOUS_STATE
        ); 
        
        if (Verify_Check == ADS_1x9x_VERIFY_ERROR)
        {
            break;                                                              // exit loop and report verify error
        }
    }
    
    disable_ADS1x9x_Interrupt (device_slot);
    
    return Verify_Check;
}

/**********************************************************************************************************
*               ADS1x9x Data Rate                                                                         *
**********************************************************************************************************/
unsigned char Initialize_ADS1x9x_Data_Rate (unsigned char device_slot, unsigned char Modulation_Frequency_Divider)
{
    unsigned char Verify_status = ADS_1x9x_INIT_SUCCESS;                                    // Error state set Clear
    
    // Load Values set in the ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);        // Read Device ID, Single Byte the Part Number
    ADS1x9x_Config_1 = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                    // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)

    //  Modify Values for the ADS1x9x
    ADS1x9x_Config_1_register->Output_Data_Rate = Modulation_Frequency_Divider;             // DEFAULT_MODULATION_FREQUENCY_DIVIDED_BY_16
                                                                                            // MODULATION_FREQUENCY_DIVIDED_BY_32
                                                                                            // MODULATION_FREQUENCY_DIVIDED_BY_64 
                                                                                            // MODULATION_FREQUENCY_DIVIDED_BY_128
                                                                                            // MODULATION_FREQUENCY_DIVIDED_BY_256
                                                                                            // MODULATION_FREQUENCY_DIVIDED_BY_512
                                                                                            // MODULATION_FREQUENCY_DIVIDED_BY_1024

    // Program Values into ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, WRITE_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);      // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, ADS1x9x_Config_1);                                // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)
  
  
#ifdef VERIFY
    //  Read Back Register    
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);       // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                   // Read the Value from the SPI port   
    if (ADS1x9x_SPI_data != ADS1x9x_Config_1)
    {
        Verify_status = ADS_1x9x_VERIFY_ERROR;
    }        
//  -----------------------------------
#endif /* VERIFY */  
    return Verify_status;   
}
/*********************************************************************************************************/
/**********************************************************************************************************
*                ADS1x9x High_Performance LOW_POWER Mode                                                  *
**********************************************************************************************************/
unsigned char Initialize_ADS1x9x_Mode (unsigned char device_slot, unsigned char ADC_Power_Mode)
{
    unsigned char Verify_status = ADS_1x9x_INIT_SUCCESS;                                                // Error state set Clear
    
    // Load Values set in the ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);       // Read Device ID, Single Byte the Part Number
    ADS1x9x_Config_1 = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                   // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)

    //  Modify Values for the ADS1x9x
    ADS1x9x_Config_1_register->Power_Resolution_Optimization = ADC_Power_Mode;                          // Default_Low_Power_Mode, High_Resolution_Mode 

    // Program Values into ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, WRITE_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);      // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, ADS1x9x_Config_1);                                // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)
  
  
#ifdef VERIFY
    //  Read Back Register    
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);        // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                    // Read the Value from the SPI port   
    if (ADS1x9x_SPI_data != ADS1x9x_Config_1)
    {
        Verify_status = ADS_1x9x_VERIFY_ERROR;
    }        
//  -----------------------------------
#endif /* VERIFY */  
    return Verify_status;   
}
/*********************************************************************************************************/


/**********************************************************************************************************
*                ADS1x9x Control Register                                                                 *
**********************************************************************************************************/
unsigned char Initialize_ADS1x9x_Registers (unsigned char device_slot)
{
    unsigned char Verify_status = ADS_1x9x_INIT_SUCCESS;                                                    // Error state set Clear
  
    // Load Values set in the ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);           // Read Device ID, Single Byte the Part Number
    ADS1x9x_Config_1 = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                       // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)

    //  Modify Values for the ADS1x9x
    ADS1x9x_Config_1_register->Oscillator_Clock_Output = DEFAULT_DISABLE;                                   // DEFAULT_DISABLE, ENABLE
    ADS1x9x_Config_1_register->Readback_Mode = DEFAULT_DAISY_CHAIN_MODE;                                    // DEFAULT_DAISY_CHAIN_MODE, MULTIPLE_READBACK_MODE

    // Program Values into ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, WRITE_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);         // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, ADS1x9x_Config_1);                                   // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)
  
  
#ifdef VERIFY
    //  Read Back Register    
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_1_REGISTER, SINGLE_BYTE_READ_WRITE);          // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                      // Read the Value from the SPI port   
    if (ADS1x9x_SPI_data != ADS1x9x_Config_1)
    {
        Verify_status = ADS_1x9x_VERIFY_ERROR;
    }        
//  -----------------------------------
#endif /* VERIFY */  
    
// Load Values set in the ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_3_REGISTER, SINGLE_BYTE_READ_WRITE);   // Read Device ID, Single Byte the Part Number
    ADS1x9x_Config_3 = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                               // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)

//  Modify Values for the ADS1x9x
    ADS1x9x_Config_3_register->Disable_Right_Leg_Detect_Buffer = DEFAULT_DISABLE;             // DEFAULT_DISABLE, ENABLE
    ADS1x9x_Config_3_register->Right_Leg_Detect_Reference_Source = DEFAULT_IS_FED_EXTERNALLY; // DEFAULT_IS_FED_EXTERNALLY, IS_FED_INTERNALLY
    ADS1x9x_Config_3_register->Right_Leg_Detect_Signal_Route = DEFAULT_IS_OPEN;               // DEFAULT_IS_OPEN, ROUTED_TO_MUX_SETTING_VREF
    ADS1x9x_Config_3_register->Reference_Voltage = DEFAULT_IS_SET_TO_2_4_VOLTS;               // DEFAULT_IS_SET_TO_2_4_VOLTS, IS_SET_TO_4_VOLTS
    ADS1x9x_Config_3_register->Internal_Reference_Buffer = ENABLE;                            // DEFAULT_DISABLE, ENABLE

    ADS1x9x_Config_3_register->Config_3_Reserved_1 = SET;                                     // Preview Silicon Only
      
// Program Values into ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, WRITE_CONFIG_3_REGISTER, SINGLE_BYTE_READ_WRITE);  // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, ADS1x9x_Config_3);                            // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)
  
  
#ifdef VERIFY
    //  Read Back Register    
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_3_REGISTER, SINGLE_BYTE_READ_WRITE);   // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                               // Read the Value from the SPI port   
    if (ADS1x9x_SPI_data != ADS1x9x_Config_3)
    {
        Verify_status = ADS_1x9x_VERIFY_ERROR;
    }        
//  -----------------------------------
#endif /* VERIFY */  
    
    // Load Values set in the ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_LEAD_OFF_CONTROL_REGISTER, SINGLE_BYTE_READ_WRITE);   // Read Device ID, Single Byte the Part Number
    ADS1x9x_Lead_Off_Control = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                               // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)

//  Modify Values for the ADS1x9x
    ADS1x9x_Lead_Off_Control_Register->Lead_Off_Frequency = DEFAULT_LEAD_OFF_DETECTION_DISABLED;    // DEFAULT_LEAD_OFF_DETECTION_DISABLED
                                                                                                    // ONE_HALF_THE_OUTPUT_DATA_RATE
                                                                                                    // ONE_FOURTH_THE_OUTPUT_DATA_RATE
                                                                                                    // DC_LEAD_OFF_DETECT
    ADS1x9x_Lead_Off_Control_Register->Lead_Off_Current = DEFAULT_12_5_NA;                          // DEFAULT_12_5_NA, _25_NA, _35_5NA, _50NA
    ADS1x9x_Lead_Off_Control_Register->Lead_Off_Detection_Mode = DEFAULT_CURRENT_MODE;              // DEFAULT_CURRENT_MODE, VOLTAGE_MODE
    ADS1x9x_Lead_Off_Control_Register->Lead_Off_Comparator_Threshold = DEFAULT_55_PERCENT;          // DEFAULT_55_PERCENT, _60_PERCENT, _65_PERCENT, _70_PERCENT
                                                                                                    // _75_PERCENT, _80_PERCENT, _85_PERCENT, _90_PERCENT
      
// Program Values into ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, WRITE_LEAD_OFF_CONTROL_REGISTER, SINGLE_BYTE_READ_WRITE);  // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, ADS1x9x_Lead_Off_Control);                            // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)
  
#ifdef VERIFY
    //  Read Back Register    
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_LEAD_OFF_CONTROL_REGISTER, SINGLE_BYTE_READ_WRITE);   // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                       // Read the Value from the SPI port   
    if (ADS1x9x_SPI_data != ADS1x9x_Lead_Off_Control)
    {
        Verify_status = ADS_1x9x_VERIFY_ERROR;
    }        
//  -----------------------------------
#endif /* VERIFY */  
    
// Load Values set in the ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_4_REGISTER, SINGLE_BYTE_READ_WRITE);       // Read Device ID, Single Byte the Part Number
    ADS1x9x_Config_4 = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                   // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)

    ADS1x9x_Config_4_register->Config_4_Reserved_2 = SET;                                               // Preview Device 
    
// Program Values into ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, WRITE_CONFIG_4_REGISTER, SINGLE_BYTE_READ_WRITE);      // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, ADS1x9x_Config_4);                                // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)
#ifdef VERIFY
    //  Read Back Register  
    ADS1x9x_SPI_Address_Byte_Count (device_slot, READ_CONFIG_4_REGISTER, SINGLE_BYTE_READ_WRITE);       // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                                   // Read the Value from the SPI port   
    if (ADS1x9x_SPI_data != ADS1x9x_Config_4)
    {
        Verify_status = ADS_1x9x_VERIFY_ERROR;
    }
#endif /* VERIFY */   
    return Verify_status;
}
/*********************************************************************************************************/

/**********************************************************************************************************
*                ADS1x9x Channel Initialization                                                           *
**********************************************************************************************************/
unsigned char Initialize_ADS1x9x_Channel 
( 
  unsigned char device_slot,
  unsigned char Channel_Number,
  unsigned char Input_Type,
  unsigned char Input_Gain,
  unsigned char Enable_Power_Setting,
  unsigned char Save_Previous_Setting
)
{
    unsigned char SPI_Read_Register = 0x00;
    unsigned char SPI_Write_Register = 0x00;
    unsigned char Verify_status = ADS_1x9x_INIT_SUCCESS;                        // Error state set Clear
    
    // Set SPI Address to Read and Write
    switch (Channel_Number)
    {
    	case 1:
    	    SPI_Read_Register = READ_CHANNEL_1_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_1_SET_REGISTER;
    	    break;
    	case 2:
    	    SPI_Read_Register = READ_CHANNEL_2_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_2_SET_REGISTER;
    	    break;
    	case 3:
    	    SPI_Read_Register = READ_CHANNEL_3_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_3_SET_REGISTER;
    	    break;    	    
        case 4:
    	    SPI_Read_Register = READ_CHANNEL_4_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_4_SET_REGISTER;
    	    break;    
        case 5:
    	    SPI_Read_Register = READ_CHANNEL_5_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_5_SET_REGISTER;
    	    break;    
    	case 6:
    	    SPI_Read_Register = READ_CHANNEL_6_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_6_SET_REGISTER;
    	    break;    
    	case 7:
    	    SPI_Read_Register = READ_CHANNEL_7_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_7_SET_REGISTER;
    	    break;    
    	case 8:
    	    SPI_Read_Register = READ_CHANNEL_8_SET_REGISTER;
    	    SPI_Write_Register = WRITE_CHANNEL_8_SET_REGISTER;
    	    break; 
    	default:
    	    break;
    }
       
    // Load Values set in the ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, SPI_Read_Register, SINGLE_BYTE_READ_WRITE);    // Read Device ID, Single Byte the Part Number
    ADS1x9x_Channel_Settings = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                   // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)

    if (Save_Previous_Setting == CONTEXT_SAVE_CHANNEL)
    {
        ADS_1298_Channel_Stack [Channel_Number - 1] = ADS1x9x_Channel_Settings;             // Context Save Channel Setting
    }
    
    //  Modify Values for the ADS1x9x
    ADS1x9x_Channel_Settings_Register->Channel_Input_Is = Input_Type;                       // DEFAULT_ADS1x9x_ELECTRODE_INPUT, ADS1x9x_INPUT_SHORTED
                                                                                            // ADS1x9x_RIGHT_LEG_DETECT, ADS1x9x_ONE_HALF_DIGITAL_SUPPLY
                                                                                            // ADS1x9x_TEMPERATURE_SENSOR, ADS1x9x_CALIBRATION_SIGNAL
                                                                                            // ADS1x9x_RIGHT_LEG_DETECT_POSITIVE, ADS1x9x_RIGHT_LEG_DETECT_NEGATIVE
    ADS1x9x_Channel_Settings_Register->Programmable_Gain_Setting = Input_Gain;              // DEFAULT_GAIN_OF_6, GAIN_OF_1, GAIN_OF_2, GAIN_OF_3, 
                                                                                            // GAIN_OF_4, GAIN_OF_8, GAIN_OF_12
    ADS1x9x_Channel_Settings_Register->Power_Down_Channel = Enable_Power_Setting;           // DEFAULT_DISABLE_POWER_DOWN, ENABLE_POWER_DOWN, ENABLE

    // Program Values into ADS1x9x
    ADS1x9x_SPI_Address_Byte_Count (device_slot, SPI_Write_Register, SINGLE_BYTE_READ_WRITE);   // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, ADS1x9x_Channel_Settings);                // Send Dummy variable (0x55) to return the part number (Chip Select Cleared automatically)
  
  
#ifdef VERIFY
    //  Read Back Register    
    ADS1x9x_SPI_Address_Byte_Count (device_slot, SPI_Read_Register, SINGLE_BYTE_READ_WRITE);    // Read Device ID, Single Byte the Part Number
    ADS1x9x_SPI_data = ADS1x9x_SPI_Data (device_slot, SPI_TEST_DATA);                           // Read the Value from the SPI port   
    if (ADS1x9x_SPI_data != ADS1x9x_Channel_Settings)
    {
        Verify_status = ADS_1x9x_VERIFY_ERROR;
    }        
//  -----------------------------------
#endif /* VERIFY */    
    return Verify_status;
}  
/*********************************************************************************************************/
/**********************************************************************************************************
* Initialize the ADS1x9x using a Constants Table of known good settings                        *
**********************************************************************************************************/
void init_ADS1x9x_Via_Constant_Table (unsigned char device_slot, unsigned char* constant_pointer)
{
    unsigned char i, j;

    ADS1x9x_SPI_Address_Byte_Count (device_slot, DEFAULT_WRITE_NUMBER_OF_REGISTERS, ADS1x9x_TOP_REGISTER_SIZE);
    
    for (i = 0; i <= ADS1x9x_SPI_WRITE_DELAY; i++);                             // Delay added between SPI Writes 
                                    
    for (i = 0; i <= ADS1x9x_TOP_REGISTER_SIZE; i++)                            // Loop through registers to load the hex data value pairs
    {                              
        ADS1x9x_SPI_data = ADS1x9x_SPI_Burst (device_slot, constant_pointer[i]);
        for (j = 0; j <= ADS1x9x_SPI_WRITE_DELAY; j++);                         // Delay added between SPI Writes 
    }

    Clear_ADS1x9x_Chip_Enable (device_slot);                                    // Clear the Chip ENABLE to terminate any previous SPI transactions

    ADS1x9x_SPI_Address_Byte_Count (device_slot, (DEFAULT_WRITE_NUMBER_OF_REGISTERS + ADS1x9x_TOP_REGISTER_SIZE + 2), ADS1x9x_BOTTOM_REGISTER_SIZE);

    for (i = 0; i <= ADS1x9x_SPI_WRITE_DELAY; i++);                             // Delay added between SPI Writes 

    for (i = 0; i < ADS1x9x_BOTTOM_REGISTER_SIZE; i++)                          // Loop through registers to load the hex data value pairs
    {                              
        ADS1x9x_SPI_data = ADS1x9x_SPI_Burst (device_slot, constant_pointer[ADS1x9x_REGISTER_OFFSET]);
        for (j = 0; j <= ADS1x9x_SPI_WRITE_DELAY; j++);                         // Delay added between SPI Writes 
    }

    Clear_ADS1x9x_Chip_Enable (device_slot);                                    // Clear the Chip ENABLE to terminate any previous SPI transactions
}


/*********************************************************************************************************/

