
#include "ads.h"
#include "anyl.h"
#include "joystick.h"
#include "lcd.h"
#include "config.h"
#include "uart.h"

#include <signal.h>


static xTaskHandle _directmonitor;
static xTaskHandle _monitor;

/* Structure for flaging information such as:
	newdata = 0, not new
	        = 1, new and so process
	cal = 1    , set system to calibrate
	cal = 0    , system calibrated readings are valid
	
	newdata flag is set low on every 'set' command function.
*/
static volatile struct {
	unsigned int newdata:1,
		cal:1;
} _flags;
static s32 _code;

#define GEN_OUT_FUNC(NAME,FUNC,PORT,PIN) \
void NAME##FUNC##H(void) { P##PORT##FUNC |=(1<<PIN); } \
void NAME##FUNC##L(void) { P##PORT##FUNC &=~(1<<PIN); } \
int NAME##FUNC##R(void) { return P##PORT##FUNC &(1<<PIN); }

#define GEN_PIN_FUNCS(NAME,PORT,PIN) \
int NAME##IN(void) { return P##PORT##IN & (1<<PIN); } \
GEN_OUT_FUNC(NAME,OUT,PORT,PIN) \
GEN_OUT_FUNC(NAME,DIR,PORT,PIN) \
GEN_OUT_FUNC(NAME,SEL,PORT,PIN)

#define GEN_INT_FUNCS(NAME,PORT,PIN) \
GEN_PIN_FUNCS(NAME,PORT,PIN) \
GEN_OUT_FUNC(NAME,IFG,PORT,PIN) \
GEN_OUT_FUNC(NAME,IES,PORT,PIN) \
GEN_OUT_FUNC(NAME,IE,PORT,PIN)

GEN_INT_FUNCS(DRDY,2,2)

GEN_PIN_FUNCS(GPIO0,3,7)
GEN_PIN_FUNCS(GPIO1,3,6)
GEN_PIN_FUNCS(GPIO2,2,1)
GEN_PIN_FUNCS(GPIO3,4,6)
GEN_PIN_FUNCS(GPIO4,4,7)
GEN_PIN_FUNCS(GPIO5,2,3)
GEN_PIN_FUNCS(CNTL,2,0)
GEN_PIN_FUNCS(CLKX,4,5)
GEN_PIN_FUNCS(CLKR,4,0)
GEN_PIN_FUNCS(FSX,4,1)
GEN_PIN_FUNCS(FSR,4,2)
GEN_PIN_FUNCS(DX,4,3)
GEN_PIN_FUNCS(DR,4,4)
// Same as interrupt GEN_PIN_FUNCS(INTN,2,2)
GEN_PIN_FUNCS(TOUT,1,2)
GEN_PIN_FUNCS(DAC,3,2)




static int gain,_pga;



void ads_init(void)
{
	/* These pins will need to be configured for the evm placed on the MMB3.
	The direction of the pins and default value will need to be adjusted as 
	appropriate.  Direction settings:
		*DIRH = Output mode on pins
		*DIRL = Input mode on pins
	*/
	
	// init pins
	
	GPIO1OUTH();	// Reset not enabled
	GPIO1DIRH();
	CLKXOUTL();	// CLK as an output LOW state
	CLKXDIRH();
	GPIO0OUTH();	// GPIO0 is an output HIGH state
	GPIO0DIRH();
	gain=1;
	_pga=0;
	CNTLOUTL();	// CS held low
	CNTLDIRH();
	GPIO2OUTL();	// Set GPIO2 output low
	GPIO2DIRH();
	GPIO3OUTH();	// power up at 5V vs. +/- 2.5V
	GPIO3DIRH();
	FSROUTH();	// Monitor DRDY
	FSRDIRL();
	DRDYDIRL();	// Set DRDY as the INT for the converter
	DRDYIFGL();
	DRDYIEL();
	DRDIRL();
	
/* Below section is for setting up SPI mode */
	// set up SPI
	U1CTL|=1; 	// hold in reset
	// U1CTL: 00 I2C=0 CHAR=1 LISTEN=0 SYNC=1 MM=1 SWRST=x
	// 00010110 = 0X16
	U1CTL=0x17;
	// U1TCTL: CKPH=0 CKPL=0 SSEL0=SMCLK (10) 00 STC=1 TXEPT=1
	// 00100011 = 0x23
	U1TCTL=0x23;
	// U1RCTL = read only
	// Baud rate = /4
	U1BR0=4;
	U1BR1=0;
	U1MCTL=0;	// no modulation
	ME2|=USPIE1;
	DRSELH();
	DXSELH();
	CLKXSELH();
	U1CTL&=~1;

/* Below section for setting up interrupts */	
	// set up interrupts
	// -- start disabled though
	DRDYIESH(); 	// trig on falling edge
	
	// other stuff
	_directmonitor=0;
	_code=0x123456;

	
}

/* task monitor setting so should not need changing */

void ads_set_monitor(xTaskHandle mon)
{
	_monitor=mon;
}

void ads_set_directmonitor(xTaskHandle mon)
{
	_directmonitor=mon;
}


	
void ads_set_gpio0(int buf)
{
	if (buf) GPIO0OUTH();
	else GPIO0OUTL();
	_flags.newdata=0;
}

int ads_get_gpio0(void)
{
	return GPIO0IN()?1:0;
}
void ads_set_gpio1(int buf)
{
	if (buf) GPIO1OUTH();
	else GPIO1OUTL();
	_flags.newdata=0;
}

int ads_get_gpio1(void)
{
	return GPIO1IN()?1:0;
}


void ads_set_gpio2(int spd)
{
	if (spd) GPIO2OUTH();
	else GPIO2OUTL();
	_flags.newdata=0;
}

int ads_get_gpio2(void)
{
	return GPIO2IN()?1:0;
}
void ads_set_gpio3(int spd)
{
	if (spd) GPIO3OUTH();
	else GPIO3OUTL();
	_flags.newdata=0;
}

int ads_get_gpio3(void)
{
	return GPIO3IN()?1:0;
}
void ads_set_gpio4(int spd)
{
	if (spd) GPIO4OUTH();
	else GPIO4OUTL();
	_flags.newdata=0;
}

int ads_get_gpio4(void)
{
	return GPIO4IN()?1:0;
}
void ads_set_gpio5(int spd)
{
	if (spd) GPIO5OUTH();
	else GPIO5OUTL();
	_flags.newdata=0;
}

int ads_get_gpio5(void)
{
	return GPIO5IN()?1:0;
}


void ads_set_pga(int pga){
	
	/* In sys0 register, pga is upper four bits and speed is lower four bits, so 
	strip out the lower four bits and add them to the new pga settings after shifting their
	value to the upper four bits of the byte. Also, the gain factor needs to be set as it is 
	used to calculate the voltage. */
	switch(pga){
		case 0:
			_pga=0;
			gain=1;
			break;
		case 1:
			_pga=1;
			gain=2;
			break;
		case 2:
			_pga=2;
			gain=4;
			break;
		case 3:
			_pga=3;
			gain=8;
			break;
		case 4:
			_pga=4;
			gain=16;
			break;
		case 5:
			_pga=5;
			gain=32;
			break;
		case 6:
			_pga=6;
			gain=64;
			break;
		case 7:
			_pga=7;
			gain=128;
			break;
		default:
			break;
	}
	
	rparms.sys0 = (u8)pga<<4 | (rparms.sys0 & 0x0f );
	
}

int ads_get_gain(void)
{	/* Used for setting correct voltage when displaying codes as voltages */
	return (gain);
}
int ads_get_pga(void)
{	/* pga is upper four bits of the sys0 register */
	return (int)(rparms.sys0 >> 4);

}

void ads_set_speed(int spd){
	/* In sys0 register, pga is upper four bits and speed is lower four bits, so 
	strip out the upper four bits and add them to the new speed settings after stripping out the 
	value to the upper four bits of the byte */
	rparms.sys0 = (rparms.sys0 & 0xf0) | ((u8)spd & 0x0f );
	
	
}

int ads_get_speed(void)
{
	/* speed is the lower four bits of the sys0 register */
	return (int)(rparms.sys0 & 0x0f);

}



void ads_set_ch(int chp, int chm)
{
	rparms.mux0 = (rparms.mux0 & 0xc0) | ((u8)(chp) << 3) | ((u8)(chm) &0x7);
}

int ads_get_ch(int chm)
{
	/* channel + is bits 3-5 of the MUX0 register  and - is bits 0-2 */
	if (chm==0) return (rparms.mux0 >> 3) &0x7;
	return (rparms.mux0 & 0x07);
}

void ads_set_bo(int bo)
{
	rparms.mux0 = ((u8)(bo) <<6) |(rparms.mux0 & 0x3f);
}

int ads_get_bo(void)
{
	/* burnout current source is bit 7 and 6 of the MUX0 register */
	return (rparms.mux0 >> 6 & 0x03);
}

void ads_set_bias(int bias)
{
	rparms.vbias = ((u8)(bias));
	
}

int ads_get_bias(void)
{
	/* bias voltage is bit 7 through 0 of the VBIAS register */
	return (rparms.vbias);
}

void ads_set_gconfig(int gpio)
{
	rparms.gpiocfg = ((u8)(gpio));
	
}

int ads_get_gconfig(void)
{
	/* GPIO ports is configured using bit 7 through 0 of the GPIOCFG register */
	return (rparms.gpiocfg);
}
void ads_set_gio(int gpio)
{
	rparms.gpiodir = ((u8)(gpio));
	
}

int ads_get_gio(void)
{
	/* GPIO data direction is configured using 7 through 0 of the GPIODIR register */
	return (rparms.gpiodir);
}
void ads_set_gdat(int gpio)
{
	rparms.gpiodat = ((u8)(gpio));
	
}

int ads_get_gdat(void)
{
	/* GPIO read inputs-write output is accomplished through bit 7 through 0 of the GPIODAT register */
	
	return (rparms.gpiodat);
}

int ads_get_gin(void)
{
	/* GPIO read inputs-write output is accomplished through bit 7 through 0 of the GPIODAT register */
	ads_reg_read();
	return (aparms.gread);
}


void ads_set_vref(int vref)
{
	rparms.mux1 = ((u8)(vref<<3))|(rparms.mux1&0xe7);
	
}

int ads_get_vref(void)
{
	/* reference voltage select is bit 3 and 4 of the MUX1 register */
	return (rparms.mux1>>3&0x03);
}

void ads_set_supply(int supply)
{
	//if (aparms.supply==supply) return;
	if (supply) {
		aparms.supply=1;
		GPIO3OUTH();
	} else {
		aparms.supply=0;
		GPIO3OUTL();
			}
}

int ads_get_supply(void)
{
	return (aparms.supply);
}

void ads_set_reset(void)
{
	
	
		GPIO1OUTL();
		vTaskDelay(10);
		GPIO1OUTH();
			
}



void ads_set_pwrdn(int pwrdn)
{
	
	if (pwrdn) {
		aparms.pwrdn=1;
		GPIO0OUTH();
	} else {
		aparms.pwrdn=0;
		GPIO0OUTL();
			}
}

int ads_get_pwrdn(void)
{
	return (aparms.pwrdn);
}

void ads_set_iref(int iref)
{
	rparms.mux1 = ((u8)(iref<<5))|(rparms.mux1&0x9f);
	
}

int ads_get_iref(void)
{
	/* internal voltage reference is bit 5 and 6 of the MUX1 register */
	return (rparms.mux1>>5&0x03);
}
void ads_set_idac0(int idac)
{
	/* Due to register map need to make 10 a 12*/
	if (idac == 10) idac=12;
	rparms.idac1 = ((u8)(idac)&0x0f)|(rparms.idac1&0xf0);
	
}

int ads_get_idac0(void)
{
	/* internal current source IDAC0 is bit 0 to 3 of the IDAC1 register.
	However due to register settings 12 should be 10 so a sequence can be
	established*/
	int i;
	i = (int)(rparms.idac1&0x0f);
	if (i>10) i=10;
	return i;
}
void ads_set_idac1(int idac)
{
	/* Due to register map need to make 10 a 12*/
	if (idac == 10) idac=12;
	rparms.idac1 = ((u8)(idac<<4))|(rparms.idac1&0x0f);
	
}

int ads_get_idac1(void)
{
	/* internal current source IDAC0 is bit 4 to 7 of the IDAC1 register.
	However due to register settings 12 should be 10 so a sequence can be
	established*/
	int i;
	i = (int)(rparms.idac1>>4);
	if (i>10) i=10;
	return i;
}
void ads_set_current(int idac)
{
	rparms.idac0 = ((u8)(idac));
	
}

int ads_get_current(void)
{
	/* IDAC current is set in the IDAC0 register */
	return (rparms.idac0&0x07);
}
void ads_set_mux(int mux)
{
	rparms.mux1 = ((u8)(mux))|(rparms.mux1&0xf8);
	
}

int ads_get_mux(void)
{
	/* MUX position for A/D reading is set in the MUX1 register */
	return (rparms.mux1&0x07);
}

/* start collecting data */
void ads_go(void)
{	/* This function may be different depending on the connection to device for
	START continous conversions. */
	u8 dummy;
	GPIO0OUTH();	// Start pin goes high
	CNTLOUTL();	// CS goes low
	DRDYIFGL();	// INT is enabled
	DRDYIEH();
	
	// Command to start taking continous readings
	while(!(IFG2&UTXIFG1));
		U1TXBUF=0x14;
	while(!(IFG2&URXIFG1));
		dummy=U1RXBUF;
	vTaskDelay(20);	
	
}

/* stop collecting data */
void ads_stop(void)
{
	u8 dummy;
	// stop interrupts
	DRDYIEL();
	
	// Reset SPI
	CNTLOUTH();	// CS goes high
	vTaskDelay(30);
	CNTLOUTL();	// CS goes low
	// Command to stop taking continous readings
	
	while(!(IFG2&UTXIFG1));
		U1TXBUF=0x17;
	while(!(IFG2&URXIFG1));
		dummy=U1RXBUF;
	vTaskDelay(20);	
	
	
}

/* calibrate the system */
/* This needs work....initial parts calibration was broken*/
void ads_cal(void)
{	u8 dummy;
	_flags.cal=1;
	portTickType xLastWakeTime;
	portTickType xFrequency ;
	
	switch (ads_get_speed()){
		case 0:
		xFrequency = portTICK_RATE_MS * 2500;
		break;
		case 1:
		xFrequency = portTICK_RATE_MS * 800;
		break;
		default:
		xFrequency = portTICK_RATE_MS * 200;
		break;
	}
	xLastWakeTime = xTaskGetTickCount();
	anyl_stop();
	
	
	lcd_showstr("OCAL");
	
		//CNTLOUTH();
		CNTLOUTL();
		//GPIO0OUTH();
		
		while(!(IFG2&UTXIFG1));
		U1TXBUF=0x62;
		while(!(IFG2&URXIFG1));
		dummy=U1RXBUF;
		CNTLOUTH();
		//CNTLOUTL();
		//GPIO0OUTL();
		xLastWakeTime = xTaskGetTickCount();
		vTaskDelayUntil(&xLastWakeTime, xFrequency);
		
	if (!aparms.console) anyl_go();
}

/* go into standby mode */
/* This part needs work....never setup */
void ads_standby(void)
{
	ads_stop();
	/* control pins may be different here depending on device */
	GPIO0OUTL();
	
}
/* go into powerdown mode */
/* This part needs work.....never setup */
void ads_powerdown(void)
{
	ads_stop();
	/* control pins may be different here depending on device for 
	//stop continous conversions. */
	GPIO0OUTL();
}

/* go ahead and get the data stored for analysis (interrupt driven from converter to '_code') */
void ads_get(s32 *c)
{
	taskENTER_CRITICAL();	// Critical code section so don't swap tasks until completed
	DRDYIEL();
	
	*c=_code;
	
	_flags.newdata=0;
	DRDYIEH();
	taskEXIT_CRITICAL();
}

int ads_newdata(void)
{
	return _flags.newdata;
}

/* Read the data from the device registers */
void ads_reg_read(void)
{
	u8 dummy;
	int i;
	u8 reg[15];		// Device has 15 registers
	
	
	anyl_stop();		// Stop getting data
	//DRDYIEL();
	vTaskDelay(40);
	CNTLOUTH();
	vTaskDelay(10);
	CNTLOUTL();		// Reset the SPI bus with CS to low 
	
	
	
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x20;		// Command for register read
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x0E;		// Command for number of registers to read
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	
	for (i=0; i<15; ++i) {
	//vTaskDelay(10);
	U1TXBUF=0xff;
	while(!(IFG2&URXIFG1));
	reg[i]=U1RXBUF;
	}
	
	CNTLOUTH();		// Drive CS back high
	rparms.mux0=reg[0];	// BCS + MUXP + MUXM
	rparms.vbias=reg[1];	// GABIAS
	rparms.mux1=reg[2];	// CLKSTAT + VREFCON + REFSELT + MUXCAL
	rparms.sys0=reg[3];	// PGA + DOR
	rparms.ofc0=reg[4];	// Offset coefficients
	rparms.ofc1=reg[5];
	rparms.ofc2=reg[6];
	rparms.fsc0=reg[7];	// Full scale coeffients
	rparms.fsc1=reg[8];
	rparms.fsc2=reg[9] ;
	rparms.idac0=reg[10];	// IDAC control
	rparms.idac1=reg[11];
	rparms.gpiocfg=reg[12]; // GPIO config and control
	rparms.gpiodir=reg[13];
	rparms.gpiodat=reg[14];

	
	aparms.gread=reg[14];

	if (!aparms.console) anyl_go();		// Restart the analysis
	
}
/* Write the parameters to the device registers */

void ads_write(void)
{
	int i;
	u8 dummy;
	u8 reg[15];		// Device has 15 registers
	
	reg[0]= rparms.mux0;	// BCS + MUXP + MUXM
	reg[1]= rparms.vbias;	// GABIAS
	reg[2]= rparms.mux1;	// CLKSTAT + VREFCON + REFSELT + MUXCAL
	reg[3]= rparms.sys0;	// PGA + DOR
	reg[4]= rparms.ofc0;	// Offset coefficients
	reg[5]= rparms.ofc1;
	reg[6]= rparms.ofc2;
	reg[7]= rparms.fsc0;	// Full scale coeffients
	reg[8]= rparms.fsc1;
	reg[9]= rparms.fsc2 ;
	reg[10]= rparms.idac0;	// IDAC control
	reg[11]= rparms.idac1;
	reg[12]= rparms.gpiocfg;// GPIO config and control
	reg[13]= rparms.gpiodir;
	reg[14]= rparms.gpiodat;
	anyl_stop();		// Stop gathering data
	// disable int before shifting
	DRDYIEL();
	
	// Reset SPI and reenable
	CNTLOUTL();
	
	/* We really don't want to overwrite the calibration settings so skip them 
	by writiing the first 4 registers and then write the last 5 registers */
	// Start at address 0
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x40;
	// And write 4 registers
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x03;
	
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	for (i=0;i<4;++i){
	while(!(IFG2&UTXIFG1));
	U1TXBUF=reg[i];
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	}
	// Now repeat the process for the last 5 registers starting with registerA
	CNTLOUTH();    	// Reset SPI
	vTaskDelay(10);
	CNTLOUTL();
	// Start at address A
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x4A;
	// And write 5 registers
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x04;
	
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	for (i=10;i<15;++i){
	while(!(IFG2&UTXIFG1));
	U1TXBUF=reg[i];
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	}
	
	// Disable communication
	CNTLOUTH();		
	if (!aparms.console) {
	// clear and reenable int
	DRDYIFGL();
	DRDYIEH();
	}
	
	
}

/* Writes changes to the offset calibration coefficients */

void ads_write_off(void)
{
	int i;
	u8 dummy;
	u8 reg[3];		// Calibration has 3 registers
	
	reg[0]= rparms.ofc0;	// Offset coefficients
	reg[1]= rparms.ofc1;
	reg[2]= rparms.ofc2;
	
	anyl_stop();		// Stop gathering data
	// disable int before shifting
	DRDYIEL();
	
	// Reset SPI and reenable
	CNTLOUTL();
	
	// Start at address 4
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x44;
	// And write 3 registers
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x02;
	
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	for (i=0;i<3;++i){
	while(!(IFG2&UTXIFG1));
	U1TXBUF=reg[i];
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	}
	
	// Disable communication
	CNTLOUTH();		
	
	
	if (!aparms.console){
		// clear and reenable int
	DRDYIFGL();
	DRDYIEH();
	 anyl_go();		// Restart the analysis
	}
	
}

/* Writes changes to the full scale calibration coefficients */

void ads_write_full(void)
{
	int i;
	u8 dummy;
	u8 reg[3];		// Calibration has 3 registers
	
	reg[0]= rparms.fsc0;	// Offset coefficients
	reg[1]= rparms.fsc1;
	reg[2]= rparms.fsc2;
	
	anyl_stop();		// Stop gathering data
	// disable int before shifting
	DRDYIEL();
	
	// Reset SPI and reenable
	CNTLOUTL();
	
	// Start at address 7
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x47;
	// And write 3 registers
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	while(!(IFG2&UTXIFG1));
	U1TXBUF=0x02;
	
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	for (i=0;i<3;++i){
	while(!(IFG2&UTXIFG1));
	U1TXBUF=reg[i];
	while(!(IFG2&URXIFG1));
	dummy=U1RXBUF;
	}
	
	// Disable communication
	CNTLOUTH();		
	
	if (!aparms.console){
		// clear and reenable int
	DRDYIFGL();
	DRDYIEH();
	 anyl_go();		// Restart the analysis
	}
	
	
}



/* Creates a buffer and converts hex data byte to ASCII characters
		in the buffer string....then displays it on the console along with
		a CR and LF*/
static void printhex(s32 *d)
{
		
	char buf[7];
	#ifndef ADS1147
	hex2asc(d,6,2,buf);
	#else
	hex2asc(d,4,2,buf);
	#endif
	uart_txstr(buf);
	uart_txc('\r');
	uart_txc('\n');
}
/* interrupt service routine may require different controls and different methods 
of reading from buffers depending on length of word and specific communication protocol */
interrupt (PORT2_VECTOR) ads_isr(void)
{
	u8 *cptr=(u8 *)(&_code);
	
	
	// disable int before shifting
	DRDYIEL();
	
	#ifndef ADS1147
	// get 3 bytes as data word is 24 bits
	U1TXBUF=0xff;
	while(!(IFG2&URXIFG1));
	cptr[2]=U1RXBUF;

	U1TXBUF=0xff;
	while(!(IFG2&URXIFG1));
	cptr[1]=U1RXBUF;

	U1TXBUF=0xff;
	while(!(IFG2&URXIFG1));
	cptr[0]=U1RXBUF;
	#else
	U1TXBUF=0xff;
	while(!(IFG2&URXIFG1));
	cptr[1]=U1RXBUF;

	U1TXBUF=0xff;
	while(!(IFG2&URXIFG1));
	cptr[0]=U1RXBUF;
	
	#endif
	if (_flags.cal) {
		// if we need to calibrate, do that first before storing any data
		
	} else {
		// sign-extend
		#ifndef ADS1147
		cptr[3]=(cptr[2]&0x80)?0xff:0;
		#else
		if (cptr[1]&0x80) {
			cptr[3]=0xff;
			cptr[2]=0xff;
		} else {
			cptr[3]=0;
			cptr[2]=0;
		}
		#endif
	}

			// clear and reenable int
	DRDYIFGL();
	DRDYIEH();

	// signal tasks
	if (_flags.cal) // don't output data on cal
		_flags.cal=0;
	else {
		
		// _directmonitor refers to streaming data to console
		if (_directmonitor){	// if streaming
			// Print the code to the console
			printhex(&_code);
			// Give up processing this task and go to the next event
			taskYIELD();
			
		} else _flags.newdata=1;
		
		

	}
	
}
