/****************************************************************/
/* File:        main.c                                          */
/****************************************************************/
/* Author:     Data Converter Plug-In Development Team          */
/*             Richard Oed <dcp-support@list.ti.com>            */
/*                                                              */
/* Created:    2005  By Texas Instruments Deutschland GmbH     */
/*                                                              */
/* Version:    1.0  /  December 2005                            */
/****************************************************************/
/* Description: Demo program for the ADS1258 connected to a     */
/*              C5510 DSK                                       */
/*                                                              */
/*              The program samples BUFFER_SIZE values and      */
/*              places the values into the r_buffer array       */
/*              The variable Ads1258_1.iXferInProgress can be   */
/*              used to track the progress of the transfer (it  */
/*              will be 1 or 2 while a transfer is in           */
/*              progress).                                      */
/*                                                              */
/*              The H/W interface between the ASD1258 and the   */
/*              DSP is defined using the Data Converter Support */
/*              tool in Code Composer Studio, which creates the */
/*              software driver for the low level data transfer */
/*                                                              */
/*              The following settings are made in the DSP/BIOS */
/*              configuration tool:                             */
/*              - Mapping of the interrupt for the transfer DMA */
/*                channel to:                                   */
/*                    _ADS1258_rblock_finished                  */
/*                                                              */
/*              The data read can be viewed by setting a probe- */
/*              point at the dc_read(); line in the main        */
/*              routine and connecting it to a graph window     */
/*              with r_buffer as the start address, a length of */
/*              BUFFERSIZE (for both read and display size) and */
/*              a data size of 32 bit signed integer            */
/****************************************************************/
/* Jumper settings for the 5-6K Interface Board (rev. B):       */
/*              W1: open      W8:  1-2                          */
/*              W2: 2-3       W9:  1-2                          */
/*              W3: 2-3       W10: 1-2                          */
/*              W4: 2-3       W11: 1-2                          */
/*              W5: 2-3       W12: 1-2                          */
/*              W6: 1-2       W13: 1-2                          */
/*              W7: 1-2                                         */
/*                                                              */
/*              J13: INTb        J14: TOUTb                     */
/*                                                              */
/* Power connections on the 5-6K Interface Board:               */
/*              J2: +/- 5V                                      */
/*                                                              */
/* Jumper settings for the ADS1258 EVM:                         */
/*              J3: Connect GPIO0 to J6A pin 1 (START)          */
/*                                                              */
/* Analog signal connections on the ADS8371 EVM:                */
/*              Apply signal to J1 pin 1 & 2                    */
/*              Analog signal settings:                         */
/*               - Frequency: 70Hz                              */
/*               - Amplitude: 3Vpp                              */
/*               - Offset:    0.7V                              */
/*                                                              */
/* All the H/W settings above translate to the following        */
/* settings in the data converter support tool:                 */
/*              DSP Tab:     DSP-Type:           TMS320C5510PG2X*/
/*                           DSP Clock:          200 MHz        */
/*              ADS1258 Tab: SysClock Source:    Internal       */
/*                           CLKIO pin:          Output         */
/*                           Conversion Mode:    Auto Scan      */
/*                           Chopper:            Disabled       */
/*                           Switch Time Delay:  0              */
/*                           Data Rate:          23430 kHz      */
/*                           ADC Source:         Internal       */
/*                                               Multiplexer    */
/*                           Bias Current Src:   Off            */
/*                           Idle Mode:          Sleep          */
/*                           SPI Reset Timer:    4096 Cycles    */
/*                           Status Byte:        Not embedded   */
/*                           DSP Serial Port:    McBSP1         */
/*                           DSP Interrupt:      Ext Int 1      */
/*                           DMA Channel:        Channel 2      */
/*                           Channel Selection:  Differential   */
/*                                               AIN0/1         */
/****************************************************************/

/* icnlude the header files for the data converter              */
#include "dc_conf.h"
#include "t1258_fn.h"

/* include the standard C header files                          */
#include <stdio.h>
#include <stdlib.h>

/* define the buffer size                                       */
#define BUFFER_SIZE (1024)

/* create the buffer                                            */
long r_buffer[BUFFER_SIZE];

/* create semaphore for end of transfer                         */
volatile int AdsBusy;


/****************************************************************/
/* callback(): Will be called at the end of the transfer and    */
/*             will reset the semaphore for the transfer        */
/****************************************************************/
void callback(void *pDC)
{ 
   TADS1258 *pADS = pDC;
   volatile unsigned int size;
   volatile unsigned long address;
   size = ADS1258_getFinishedBufferSize(pADS);
   address = ADS1258_getFinishedBufferAddress(pADS);
   AdsBusy = 0;               /* reset semaphore for transfer   */
}   


/****************************************************************/
/* main(): Configures the interface to the ADS1605              */
/*         Finally, issues a block read command to the converter*/
/*         and waits for the completion                         */
/****************************************************************/
int main(void)
{
    int iStatus;
    unsigned int i;

    /* mae the McBSPs available externally                      */
    *((unsigned char*)0x300006) |= 0x03;

    for (i = 0; i < BUFFER_SIZE; i++)
        *(r_buffer+i) = 0;

    /* call the configuration of the ADS and the interface      */
    iStatus = dc_configure(&Ads1258_1);

    if (!(iStatus == TIDC_NO_ERR))  /* something went wrong     */
    {
        fprintf(stderr, "\nError: ADS1605 init failed!\n");
        abort();
    }

    IRQ_globalEnable();           /* enable interrupts globally */

    while(1)                      /* loop forever               */
    {
        /* make sure there is no transfer in progress           */
        while(Ads1258_1.iXferInProgress == 1);

        /* issue a read block command to the converter          */
        iStatus = dc_readblock(&Ads1258_1, /* data converter object */
                               r_buffer, /* address of the target buffer*/
                               BUFFER_SIZE, /* size of the target buffer  */
                               callback); /* callback function  */
        if (!(iStatus == TIDC_NO_ERR))  /* something went wrong */
        {
            fprintf(stderr, "\nError: ADS1605 read failed!\n");
            abort();
        }

    }
}
