/*

TAS5711 I2C sub routing code for device init. Key function, etc...

File Date : 2010-05-28

Target Device: TAS5711 Support 2.0&2.1 channel output

By Ryan Wang
ryan-wang@ti.com

*/


#include "msp430x21x2.h"
#include "TI_TAS5711_MCU_EVM.H"
#include "TI_TAS5711_LIB.H"

unsigned char drc_on[]={0x00, 0x00, 0x00, 0x03};
unsigned char drc_off[]={0x00, 0x00, 0x00, 0x00};
unsigned char Output_Type;
unsigned char POWER_STATUS = POWER_OFF;
unsigned char MUTE_STATUS = UN_MUTE;
unsigned char EQ_STATUS = OFF;
unsigned char DRC_STATUS = OFF;
unsigned char LD_3D_STATUS = OFF;
unsigned char Vol_Counter = VOL_DEFAULT;
unsigned char Vol_Map[21] = {0x0c,0x0d,0x0e,0x10,0x13,0x14,0x16,0x1e,0x26,0x2e,0x36,0x3e,0x46,0x4e,0x56,0x5e,0x66,0x6e,0x7e,0x8e,0xff};


//Flat EQ
unsigned char EQ_Flat[]  	= {0x00,  0x80,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00,  0x00};

//1.0CH Filters for Subwoofer
unsigned char bq_LP_CH10[]   = {0x00,  0x00,  0x08,  0x92,  0x00,  0x00,  0x11,  0x25,  0x00,  0x00,  0x08,  0x92,  0x00,  0xFA,  0x13,  0x89,  0x0F,  0x85,  0xCA,  0x2B}; //250Hz lowPass
unsigned char bq_HP_CH10[]   = {0x00,  0x7F,  0x86,  0xE2,  0x0F,  0x00,  0xF2,  0x3B,  0x00,  0x7F,  0x86,  0xE2,  0x00,  0xFF,  0x0D,  0x52,  0x0F,  0x80,  0xF1,  0xC8};//40Hz highPass

//2.0CH EQ
unsigned char bq_HP_CH20[]  	= {0x00,  0x7F,  0x86,  0xE2,  0x0F,  0x00,  0xF2,  0x3B,  0x00,  0x7F,  0x86,  0xE2,  0x00,  0xFF,  0x0D,  0x52,  0x0F,  0x80,  0xF1,  0xC8}; //40Hz highPass
unsigned char bq_EQ_CH20[]  	= {0x00,  0x80,  0xA5,  0xD4,  0x0F,  0x01,  0xAD,  0xB9,  0x00,  0x7D,  0xB0,  0x04,  0x00,  0xFE,  0x52,  0x47,  0x0F,  0x81,  0xAA,  0x27}; //EQ 80Hz 5dB 100Hz 

//2.1CH EQ

unsigned char bq_HP_CH21[]       = {0x00,  0x7D,  0x12,  0x57,  0x0F,  0x05,  0xDB,  0x51,  0x00,  0x7D,  0x12,  0x57,  0x00,  0xFA,  0x13,  0x89,  0x0F,  0x85,  0xCA,  0x2B}; //250Hz highPass
unsigned char bq_EQ_CH21[]       = {0x00,  0x80,  0x6D,  0xA1,  0x0F,  0x03,  0x71,  0x77,  0x00,  0x7C,  0x43,  0x8A,  0x00,  0xFC,  0x8E,  0x89,  0x0F,  0x83,  0x4E,  0xD4}; //EQ 250Hz 2dB 100Hz

unsigned char bq_Sub_LP_CH21[]   = {0x00,  0x00,  0x08,  0x92,  0x00,  0x00,  0x11,  0x25,  0x00,  0x00,  0x08,  0x92,  0x00,  0xFA,  0x13,  0x89,  0x0F,  0x85,  0xCA,  0x2B}; //250Hz lowPass
unsigned char bq_Sub_HP_CH21[]   = {0x00,  0x7F,  0x86,  0xE2,  0x0F,  0x00,  0xF2,  0x3B,  0x00,  0x7F,  0x86,  0xE2,  0x00,  0xFF,  0x0D,  0x52,  0x0F,  0x80,  0xF1,  0xC8};//40Hz highPass
unsigned char bq_Sub_EQ_CH21[]   = {0x00,  0x80,  0xA5,  0xD4,  0x0F,  0x01,  0xAD,  0xB9,  0x00,  0x7D,  0xB0,  0x04,  0x00,  0xFE,  0x52,  0x47,  0x0F,  0x81,  0xAA,  0x27};//EQ 80Hz 5dB 100Hz

//DRC1 Coff.********************************************************************
unsigned char DRC1_ae[] = {0x00, 0x00, 0x06, 0xD3, 0x00, 0x7F, 0xF9, 0x2C}; //100ms
unsigned char DRC1_aa[] = {0x00, 0x00, 0x44, 0x32, 0x00, 0x7F, 0xBB, 0xCD}; //10ms
unsigned char DRC1_ad[] = {0x00, 0x00, 0x06, 0xD3, 0x00, 0x7F, 0xF9, 0x2C}; //100ms


//unsigned char DRC1_T[] = {0xF9,  0xDA,  0xBC,  0x20}; //-50dB
//unsigned char DRC1_T[] = {0xFA,  0xAF,  0x56,  0x98}; //-40dB
//unsigned char DRC1_T[] = {0xFB,  0x83,  0xF1,  0x11}; //-30dB
unsigned char DRC1_T[] = {0xFC,  0x58,  0x8B,  0x89}; //-20dB
//unsigned char DRC1_T[] = {0xFD,  0x2D,  0x26,  0x01}; //-10dB
unsigned char DRC1_K[] = {0x0F,  0x84,  0x44,  0x45}; //30
unsigned char DRC1_O[] = {0x00,  0x08,  0x42,  0x10}; //0

//DRC2 Coff.********************************************************************
unsigned char DRC2_ae[] = {0x00, 0x00, 0x06, 0xD3, 0x00, 0x7F, 0xF9, 0x2C}; //100ms
unsigned char DRC2_aa[] = {0x00, 0x00, 0x44, 0x32, 0x00, 0x7F, 0xBB, 0xCD}; //10ms
unsigned char DRC2_ad[] = {0x00, 0x00, 0x06, 0xD3, 0x00, 0x7F, 0xF9, 0x2C}; //100ms


//unsigned char DRC2_T[] = {0xF9,  0xDA,  0xBC,  0x20}; //-50dB
//unsigned char DRC2_T[] = {0xFA,  0xAF,  0x56,  0x98}; //-40dB
//unsigned char DRC2_T[] = {0xFB,  0x83,  0xF1,  0x11}; //-30dB
unsigned char DRC2_T[] = {0xFC,  0x58,  0x8B,  0x89}; //-20dB
//unsigned char DRC2_T[] = {0xFD,  0x2D,  0x26,  0x01}; //-10dB
unsigned char DRC2_K[] = {0x0F,  0x84,  0x44,  0x45}; //30
unsigned char DRC2_O[] = {0x00,  0x08,  0x42,  0x10}; //0


//------------------------------------------------------------------------------
// void Blink_Power_LED ()
// This function provide Power LED Blink function
//
// Input: tmp => if tmp == ON, Start Blinking
//		 if tmp == OFF, Stop Blinking
//------------------------------------------------------------------------------
void Blink_Power_LED(unsigned char tmp)
{
  if (tmp == ON)
  {
  P2DIR |= 0x10;                           // P2.4 output
  P2SEL |= 0x10;                           // P2.4  TA2 options
  TA0CCR0 = 65535;                         // PWM Period
  TA0CCTL2 = OUTMOD_4;                     // TA0CCR2 togle
  TA0CCR2 = 30000;                         // TA0CCR2 PWM duty cycle
  TA0CTL = TASSEL_2 + MC_1 + ID_3;         // SMCLK, up mode
  }
  else if(tmp == OFF)
  {
    P2SEL &= ~0x10;                            // P2.4 I/O options
    TA0CTL = TASSEL_2 + MC_1 + ID_3;
    POWER_LED_ON;
  }

}


//------------------------------------------------------------------------------
// void Delay_1ms (unsigned char time)
//
// This function provide delay function
//
// IN:   time       =>  loop until time decrease to 0
// test 1 = 1.1ms
//------------------------------------------------------------------------------
void Delay_1ms(unsigned int time)
{

while(time--)

  {
    WDTCTL = WDT_MDLY_8;				//Set Watchdog interval MCLK=8MHz Divder=8	
    while (!(IFG1 & WDTIFG));  				//Wait until time out
    IFG1 &=~WDTIFG;
  }

WDTCTL = WDTPW + WDTHOLD;

}

//------------------------------------------------------------------------------
// void OUTPUT_CONFIG ()
//
// This function initializes check P3.4 and P3.5 input leve to determin which output type is selected.
// 		       P3.4		P3.5
//2.0			0		0 (0x00)
//2.1			1		0 (0x10)
//Reserved		0		1 (0x20)
//1.0   		1		1 (0x30)
//------------------------------------------------------------------------------
unsigned char OUTPUT_CONFIG_CHECK()

{

// check P3.4 and P3.5

	if ((P3IN & 0x30) == 0x00)
	{return(CH2_0);}
	
	else if ((P3IN & 0x30) == 0x10)
        {return(CH2_1);}
	
	else if ((P3IN & 0x30) == 0x30)
	{return(CH1_0);}

  	else 
	{return(CH2_0);}

}


//------------------------------------------------------------------------------
// void TI_EVM_BOARD_Init ()
//
// This function initializes the I/O and reset DIR9001 and TAS57xx.
//
// IN:   None
// OUT:  None
//------------------------------------------------------------------------------
void TI_EVM_BOARD_Init()
{
// MSP430 Init. I/O ports////////////////////////////////////////
// P1.1 3D/LDN_SW
// P1.2 EQ_SW
// P1.3 DRC_SW
// P1.4 POWER_SW
// P1.5 MUTE_SW
// P1.6 VOL_SW+
// P1.7 VOL_SW-
//
// P2.4 POWER_LED
// P2.3 EQ_LED
// P3.7 3D/LDN_LED
// P3.6 DRC_LED

// P2.0  /RST_EVM
// P2.1  /PDN_EVM
// P2.2  /MUTE_EVM
// P3.0  /RST_DIR9001
////////////////////////////////////////////////////////////////


//Port 1 Interrupt config

P1DIR  = 0x01;                          // P1.0 output all else input	
P1REN |= 0xFE;				// P1 pullup
P1OUT  = 0xFE;	
P1IE  |= 0xFE;				// P1 interrupt enabled (Only POWER ON/OFF Enabled here)
P1IES |= 0xFF;				// P1 Hi/lo edge
P1IFG  = 0x00;				// P1 IFG cleared

//Port 2,3 Output default value set
P2OUT &= 0x07;		                //Output level init.
P2DIR |= 0x1F;		                //Port 2 output direction, /RST, /PDN, /MUTE, /RST_DIR9001, POWER_LED, EQ_LED

P3REN  = 0x30;
P3DIR |= 0xC1;		                //Port 3 output direction, LDNS_LED, DRC_LED
P3OUT &= 0xC0;		                //Output level init.



//LED Blink---------------------------------------------------------------------
POWER_LED_ON;
Delay_1ms(300);
POWER_LED_OFF;

LD_3D_LED_ON;
Delay_1ms(300);
LD_3D_LED_OFF;

EQ_LED_ON;
Delay_1ms(300);
EQ_LED_OFF;

DRC_LED_ON;
Delay_1ms(300);
DRC_LED_OFF;


//Reset DIR9001-----------------------------------------------------------------
DIR9001_RST_ON;

Delay_1ms(1);

DIR9001_RST_OFF;

Delay_1ms(1);             //Delay for Clock Stable

//Reset TAS57xx-----------------------------------------------------------------
TAS57xx_PDN_OFF;            //Drive PDN = 1

TAS57xx_RST_ON;             //Drive RST = 0

Delay_1ms(1);             //RST device and keep at least 100us,while PDN_OFF

TAS57xx_RST_OFF;            //Driver RST = 1

Delay_1ms(14);              //Finish RST device and wait at least for 13.5ms


// Check output type set by user, 2.0CH or 2.1CH output-------------------------
Output_Type = OUTPUT_CONFIG_CHECK();

}



//------------------------------------------------------------------------------
// void Power_Key_Press ()
// This function provide sub for Power Key press
//
// Register: 0x05; Readback before program;
//------------------------------------------------------------------------------
void Power_Key_Press()
{

    unsigned char temp;
    temp = TI_USCI_I2C_ReadOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_POWER_CONTROL);
	
    if(POWER_STATUS == POWER_OFF)
	  {
	      POWER_STATUS = POWER_ON;
	      POWER_LED_ON;
	      temp = temp&0xBF;
	      TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_POWER_CONTROL,temp); //Power ON 

	  }
    
    else if(POWER_STATUS == POWER_ON)
	  {
	      if(EQ_STATUS == ON)
		{EQ_Key_Press();}//Turn off EQ
	      if(DRC_STATUS == ON)
		{DRC_Key_Press();}//Turn off DRC
	      if(LD_3D_STATUS == ON)
		{LD_3D_Key_Press();}//Turn off 3D
	      if(MUTE_STATUS == MUTE)
		{Mute_Key_Press();}//Turn off Mute
	      
	      temp = temp|0x40;
	      Delay_1ms(100); // Prevent Very Fast On/Off Pop
	      TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_POWER_CONTROL,temp); //Power OFF
	      POWER_STATUS = POWER_OFF;
	      POWER_LED_OFF;
	  }
	
}

//------------------------------------------------------------------------------
// void Mute_Key_Press ()
// This function provide sub for Mute Key press
//
// Register: 0x06
// 0xFF --- Mute All Channels; 0x00 --- UnMute All Channels;
//------------------------------------------------------------------------------
void Mute_Key_Press()
{

 if(MUTE_STATUS == UN_MUTE)
      {
        TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_SOFTWARE_MUTE,0xFF);
        MUTE_STATUS = MUTE;
        Blink_Power_LED(ON);
      }
      else if(MUTE_STATUS == MUTE)
      {
        TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_SOFTWARE_MUTE,0x00);
        MUTE_STATUS = UN_MUTE;
        Blink_Power_LED(OFF);
      }

}


//------------------------------------------------------------------------------
// void VOL_UP_Key_Press ()
//
// This function provide sub for Vol_up Key press
//
//------------------------------------------------------------------------------
void VOL_UP_Key_Press()
{
  if(Vol_Counter != MAX_VOL)
      {
        Vol_Counter = Vol_Counter - 1;
        TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_MASTER_VOL,Vol_Map[Vol_Counter]);
      }
      else
      {
        Blink_Power_LED(ON);
        Delay_1ms(100);
        Blink_Power_LED(OFF);
      }

}

//------------------------------------------------------------------------------
// void VOL_DOWN_Key_Press ()
//
// This function provide sub for Vol_Down Key press
//
//------------------------------------------------------------------------------
void VOL_DOWN_Key_Press()
{
   if(Vol_Counter != MIN_VOL)
      {
        Vol_Counter = Vol_Counter + 1;
        TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_MASTER_VOL,Vol_Map[Vol_Counter]);
      }
      else
      {
        Blink_Power_LED(ON);
        Delay_1ms(100);
        Blink_Power_LED(OFF);
      }

}

//------------------------------------------------------------------------------
// void EQ_Key_Press ()
//
// This function provide sub for EQ Key press
//
//------------------------------------------------------------------------------
void EQ_Key_Press()
{
    unsigned char temp[4];
    TI_USCI_I2C_ReadMultiBytes(TI_TAS5711_I2C_ADDR, TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
    // mute
    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_MASTER_VOL,0xFF);
    Delay_1ms(50);
       
    if(EQ_STATUS == OFF) //if EQ is off, turn on EQ
      {
        EQ_LED_ON;
        EQ_STATUS = ON;
	temp[3]=temp[3]&0x7F;//EQ Control bit set to 0
        TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
      }

    else if(EQ_STATUS == ON)//if EQ is on, turn off EQ
      {
        EQ_LED_OFF;
        EQ_STATUS = OFF;
	temp[3]=temp[3]|0x80;//EQ Control bit set to 1
        TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
      }
	
    Delay_1ms(50);
    //Unmute
    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_MASTER_VOL,Vol_Map[Vol_Counter]);
     
}




//------------------------------------------------------------------------------
// void DRC_Key_Press ()
//
// This function provide sub for DRC Key press
//
//------------------------------------------------------------------------------
void DRC_Key_Press()

{

    if(DRC_STATUS == OFF)
      {
        DRC_LED_ON;
        DRC_STATUS = ON;
        //DRC_ON
        unsigned char temp[]={0x00, 0x00, 0x00, 0x03};
        TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC_CONTROL,sizeof(temp),temp);

      }
      else if(DRC_STATUS == ON)
      {
        DRC_LED_OFF;
        DRC_STATUS = OFF;
        //DRC_OFF
        unsigned char temp[]={0x00, 0x00, 0x00, 0x00};
        TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC_CONTROL,sizeof(temp),temp);
      }

}

//------------------------------------------------------------------------------
// void LDN_3D_Key_Press ()
//
// This function provide sub for DRC Key press
//
//------------------------------------------------------------------------------
void LD_3D_Key_Press()

{
      if((Output_Type == CH2_0)||(Output_Type == CH2_1))//EQ and 3D can't take effect at the same time.
      {
	 
	  // mute
    	  TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_MASTER_VOL,0xFF);
	  Delay_1ms(50);

	  if(LD_3D_STATUS == OFF)
	    {
	        LD_3D_LED_ON;
		LD_3D_STATUS = ON;

		//apply Delay EQ
		unsigned char bq_3D_delay[]={0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
   		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_3D_BQ0,sizeof(bq_3D_delay),bq_3D_delay);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_3D_BQ1,sizeof(bq_3D_delay),bq_3D_delay);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_3D_BQ0,sizeof(bq_3D_delay),bq_3D_delay);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_3D_BQ1,sizeof(bq_3D_delay),bq_3D_delay);
		
		//Programme Filter for (L-R) Channel
		unsigned char bq_3D_HP[]={0x00, 0x7E, 0x82, 0x70, 0x0F, 0x81, 0x7D, 0x90, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7D, 0x04, 0xE0, 0x00, 0x00, 0x00, 0x00};
		unsigned char bq_3D_EQ[]={0x00, 0x86, 0x24, 0xA3, 0x0F, 0x1F, 0xB7, 0xAD, 0x00, 0x5C, 0x13, 0x21, 0x00, 0xE0, 0x48, 0x53, 0x0F, 0x9D, 0xC8, 0x3B};
		
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_PSEUDO_BQ1,sizeof(bq_3D_HP),bq_3D_HP);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_PSEUDO_BQ2,sizeof(bq_3D_EQ),bq_3D_EQ);
		
		//Set Mixer value
		unsigned char CH1_INPUT_Mixer[]={0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x0E, 0xC0, 0x00, 0x00 };
		unsigned char CH2_INPUT_Mixer[]={0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x01, 0x40, 0x00, 0x00 };
		unsigned char CH3_INPUT_Mixer[]={0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x0F, 0xC0, 0x00, 0x00 };
	
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CH1_INPUT_Mixer,sizeof(CH1_INPUT_Mixer),CH1_INPUT_Mixer);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CH2_INPUT_Mixer,sizeof(CH2_INPUT_Mixer),CH2_INPUT_Mixer);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CH3_INPUT_Mixer,sizeof(CH3_INPUT_Mixer),CH3_INPUT_Mixer);
	
		//Enable Delay EQ
		unsigned char temp[4];
		TI_USCI_I2C_ReadMultiBytes(TI_TAS5711_I2C_ADDR, TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
		temp[3]=temp[3]&0x7F;//EQ Control bit set to 0
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
		
		
		
	    }
	  else if(LD_3D_STATUS == ON)
	    {
		LD_3D_LED_OFF;
		LD_3D_STATUS = OFF;
		
		//Remove Delay EQ
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_3D_BQ0,sizeof(EQ_Flat),EQ_Flat);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_3D_BQ1,sizeof(EQ_Flat),EQ_Flat);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_3D_BQ0,sizeof(EQ_Flat),EQ_Flat);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_3D_BQ1,sizeof(EQ_Flat),EQ_Flat);
		
		//Remove (L-R) Channel Filters
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_PSEUDO_BQ1,sizeof(EQ_Flat),EQ_Flat);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_PSEUDO_BQ2,sizeof(EQ_Flat),EQ_Flat);
		
		//Set Mixer to defualt
		unsigned char temp1_Mixer[]={0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00 };
		unsigned char temp2_Mixer[]={0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  };
	
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CH1_INPUT_Mixer,sizeof(temp1_Mixer),temp1_Mixer);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CH2_INPUT_Mixer,sizeof(temp1_Mixer),temp1_Mixer);
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CH3_INPUT_Mixer,sizeof(temp2_Mixer),temp2_Mixer);
      
		//Disable Delay EQ
		unsigned char temp[4];
		TI_USCI_I2C_ReadMultiBytes(TI_TAS5711_I2C_ADDR, TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
		temp[3]=temp[3]|0x80;//EQ Control bit set to 1
		TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
		
		
	    }
	  
    Delay_1ms(50);
    //Unmute
    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR,TI_TAS5711_MASTER_VOL,Vol_Map[Vol_Counter]);
      }


}

//------------------------------------------------------------------------------
// void TI_TAS5711_Init (unsigned char Device_I2C_Address, unsigned char Output_Type)
//
// This function reset device, configure output type, EQ, DRC parameters and waiting for power up command.
//
// IN:  Output_Type			=> Output type selected by user
// IN:  Device_I2C_Address              => I2C address of the device been controlled
// Out: None
//------------------------------------------------------------------------------
void TI_TAS5711_Init(unsigned char Device_I2C_Address)
{
	//Trim the internal oscillator (Write 0x00 to 0x1B)
	TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_OSCILLATOR_TRIM,0x00);
	
	//Wait 50ms while the part acquires lock
	Delay_1ms(50);  

	//Configure Mix and Mux accordign to Target output type
	TI_TAS5711_SET_OUTPUT(Output_Type);
	
	//DRC parameters initialization
	TI_TAS5711_DRC_Init();
	
	//EQ parameters initialization
	TI_TAS5711_EQ_Init(Output_Type);
	
	//3D parameters initialization
	//TI_TAS5711_3D_Init();
	
	//Serial Audio Data Format Select				
	TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_I2S_MODES,0x05);   //i2s format
		
	//Modulation Limitation Setup(add:0x10; Default:97.7%:0x02) //Change to smaller if needed
	TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_MODULATION_LIMIT,0x02);
	
	//Defualt Master volume 0x30 = 0dB
	TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_MASTER_VOL,Vol_Map[VOL_DEFAULT]);	
		
}

//------------------------------------------------------------------------------
// void TI_TAS5711_DRC_Init (unsigned char Output_Type)
//
// This funtion initializes DRC parameters for devices, DRC1 and DRC2 are all programmed.
//
// In:None
// Out: None
//------------------------------------------------------------------------------
void TI_TAS5711_DRC_Init()

{
	
	//DRC1 Init. for SE outputs
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC1_AE,sizeof(DRC1_ae),DRC1_ae);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC1_AA,sizeof(DRC1_aa),DRC1_aa);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC1_AD,sizeof(DRC1_ad),DRC1_ad);
	
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC1_T,sizeof(DRC1_T),DRC1_T);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC1_O,sizeof(DRC1_O),DRC1_O);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC1_K,sizeof(DRC1_K),DRC1_K);

	//DRC2 Init. for subwoofer output
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC2_AE,sizeof(DRC2_ae),DRC2_ae);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC2_AA,sizeof(DRC2_aa),DRC2_aa);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC2_AD,sizeof(DRC2_ad),DRC2_ad);

	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC2_T,sizeof(DRC2_T),DRC2_T);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC2_O,sizeof(DRC2_O),DRC2_O);
	TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_DRC2_K,sizeof(DRC2_K),DRC2_K);
    
}


//------------------------------------------------------------------------------
// void TI_TAS5711_SET_OUTPUT (unsigned char Output_Type)
//
// This funtion initializes 2.0CH output or 2.1CH output
//
// IN:  Output_Type			=> Output type selected by user
// Out: None
//------------------------------------------------------------------------------

void TI_TAS5711_SET_OUTPUT(unsigned char Output_Type)

  {
	switch (Output_Type)
		{
		  case CH1_0: // 1.0CH output.Please make sure Hardware PBTL connection is correct.
			{
			    unsigned char Input_Mix_Data[] = {0x00,0x01,0x77,0x72}; //Defualt Value
			    unsigned char DownMix_Data[]   = {0x00,0x00,0x43,0x03}; //Defualt Value
			    unsigned char PWM_Mux_Data[]   = {0x01,0x10,0x32,0x45}; //Value for 1.0CH		
			    
			     //Input Mux Setup(0x20)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_INPUT_MUX,4,Input_Mix_Data);

			    //Downmix Input Mux Setup(0x21)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_INPUT_DOWN_MIX,4,DownMix_Data);

			    //PWM Mux Setup(0x25)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_PWM_OUTPUT_MUX,4,PWM_Mux_Data);

			    //Start and Stop period, SE output: 0x95 for 703.1ms delay; BTL output: 0x14 for 538.5ms delay
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_START_STOP_PERIOD,0x14);
			    
			    //Shutdown group
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_SHUTDOWN_GROUP_REGISTER,0x3a);
			
			 /* //IC Channel Delay Setup(0x11 ~ 0x16)
			    //Please check datasheet for these settings, defualt is for AD mode
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM1,0xAC);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM2,0x54);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM3,0xAC);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM4,0x54);*/
			    
			}break;
			    
		case CH2_0: // 2.0CH output, default PWM output mode is AD mode
			{

			    unsigned char Input_Mix_Data[] = {0x00,0x01,0x77,0x72}; //Defualt Value for 2.0CH
			    unsigned char DownMix_Data[]   = {0x00,0x00,0x43,0x03}; //Defualt Value for 2.0CH
			    unsigned char PWM_Mux_Data[]   = {0x01,0x02,0x13,0x45}; //Defualt Value for 2.0CH			

			    //Input Mux Setup(0x20)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_INPUT_MUX,4,Input_Mix_Data);

			    //Downmix Input Mux Setup(0x21)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_INPUT_DOWN_MIX,4,DownMix_Data);
			
			    //PWM Mux Setup(0x25)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_PWM_OUTPUT_MUX,4,PWM_Mux_Data);

			    //Start and Stop period, SE output: 0x95 for 703.1ms delay; BTL output: 0x14 for 538.5ms delay
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_START_STOP_PERIOD,0x14);
			    
			    //Shutdown group
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_SHUTDOWN_GROUP_REGISTER,0x30);
			
			   /* //IC Channel Delay Setup(0x11 ~ 0x16)
			    //Please check datasheet for these settings, defualt is for AD mode
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM1,0xAC);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM2,0x54);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM3,0xAC);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM4,0x54);*/
				
				                                				
			}break;
		
		case CH2_1: // A SE and B SE and C.D BTL ; 2.1 BTL@AD MODE
			{
			    unsigned char Input_Mix_Data[] = {0x00,0x01,0x77,0x72};//Defualt Value
			    unsigned char DownMix_Data[]   = {0x00,0x00,0x42,0x03};//Value for 2.1CH
			    unsigned char PWM_Mux_Data[]   = {0x01,0x01,0x32,0x45};//Value for 2.1CH

			    //Input Mux Setup(0x20)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_INPUT_MUX,4,Input_Mix_Data);
			
			    //Downmix Input Mux Setup(0x21)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_INPUT_DOWN_MIX,4,DownMix_Data);
			
			    //PWM Mux Setup(0x25)
			    TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_PWM_OUTPUT_MUX,4,PWM_Mux_Data);
			    
			    //Change Subchannel Volume to 0x0A for control,Defualt is 0x91, 2.1CH require 0xD1
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_VOLUME_CONFIG,0xD1);
			    
			    //Increase SubChannel Volume
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_Sub_VOL,0x20);
			    
			    //Set TAS5711 to 2.1CH
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_POWER_CONTROL,0xC4);
			
			    //Start and Stop period, SE output: 0x95 for 703.1ms delay; BTL output: 0x14 for 538.5ms delay
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_START_STOP_PERIOD,0x95);
			    			    
			    //Shutdown group
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_SHUTDOWN_GROUP_REGISTER,0x00);
			  
			
			 /* //IC Channel Delay Setup(0x11 ~ 0x14)
			    //Please check datasheet for these settings, defualt is for AD mode
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM1,0x3C);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM2,0xC4);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM3,0x00);
			    TI_USCI_I2C_WriteOneByte(TI_TAS5711_I2C_ADDR, TI_TAS5711_IC_DELAY_PWM4,0x00);*/
				
			}break;

		}
  }


//------------------------------------------------------------------------------
// void TI_TAS5711_EQ_Init (unsigned char Output_Type)
//
// This funtion initializes 2.0CH or 2.1CH EQ data, if device support FAST_EQ, all
// parameters are writen to device at initilization. if device don't support FAST_EQ
// parameters are update and cleared when EQ_Key_Pressed.
// Only TAS5706 don't support FAST_EQ
//
// IN:  Output_Type			=> Output type selected by user
// Out: None
//------------------------------------------------------------------------------


void TI_TAS5711_EQ_Init(unsigned char Output_Type)
  {

//******************************Turn off EQ ************************************
        unsigned char temp[]={0x0F, 0x70, 0x80, 0x80};//Turn off EQ
        TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_EQ_CONTROL,sizeof(temp),temp);
//******************************************************************************
         switch (Output_Type)
	      
	   {		
	     case CH1_0: // A.B short and C.D short
		  {	      
		      //PBTL mode use Channel 1 BQ as EQ, example is 250Hz low pass
		      //Write EQ
		     TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_BQ2,sizeof(bq_LP_CH10),bq_LP_CH10);
		     TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_BQ3,sizeof(bq_HP_CH10),bq_HP_CH10);
		  }break;
	      
	      
	     case CH2_0: // A.B BTL and C.D BTL
		  
	       {
		   //Channel 1
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_BQ2,sizeof(bq_HP_CH20),bq_HP_CH20);
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_BQ3,sizeof(bq_EQ_CH20),bq_EQ_CH20);
		   //Channel 2
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_BQ2,sizeof(bq_HP_CH20),bq_HP_CH20);
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_BQ3,sizeof(bq_EQ_CH20),bq_EQ_CH20);
		 
	       }break;
		
	     case CH2_1: // A SE and B SE and C.D BTL
	   
	       {
		   //Channel 1
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_BQ2,sizeof(bq_HP_CH21),bq_HP_CH21);
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL1_BQ3,sizeof(bq_EQ_CH21),bq_EQ_CH21);
		   //Channel 2
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_BQ2,sizeof(bq_HP_CH21),bq_HP_CH21);
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_CHANNEL2_BQ3,sizeof(bq_EQ_CH21),bq_EQ_CH21);
		   //Subwoofer
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_Sub_BQ0,sizeof(bq_Sub_LP_CH21),bq_Sub_LP_CH21);
		   TI_USCI_I2C_WriteMultiBytes(TI_TAS5711_I2C_ADDR,TI_TAS5711_Sub_BQ1,sizeof(bq_Sub_EQ_CH21),bq_Sub_EQ_CH21);
	       
	       }break;
	      
	   }

  }