/*
I2C Tx/Rx library based on USCI

Device :  MPS430 MCU with USCI module

File Date : 2009-10-1

By Ryan Wang

ryan-wang@ti.com

*/


#include "msp430x21x2.h"
#include "TI_TAS5711_MCU_EVM.H"
#include "TI_TAS5711_LIB.H"

unsigned char Tx_ByteCounter;
unsigned char Rx_ByteCounter;
unsigned char *Tx_Pointer_Temp;
unsigned char *Rx_Pointer_Temp;

//------------------------------------------------------------------------------
// void TI_USCI_I2C_Init (unsigned char i2c_master_speed)
//
// This function initializes the USCI module for master-transmit operation.
//
// IN:   unsigned char i2c_speed        =>  SCL clock adjustment
//------------------------------------------------------------------------------
void TI_USCI_I2C_Init(unsigned char i2c_master_speed)
{
  P3SEL |= SDA_PIN + SCL_PIN;						// Assign I2C pins to USCI_B0
  UCB0CTL1 = UCSWRST;								// Enable SW reset
  UCB0CTL0 = UCMST + UCMODE_3 + UCSYNC;				// I2C Master, synchronous mode
  UCB0CTL1 = UCSSEL_2 + UCSWRST;					// Use SMCLK, keep SW reset
  UCB0BR0 = i2c_master_speed;						// Set I2C master speed
  UCB0BR1 = 0;										// Set I2C master speed
  UCB0CTL1 &= ~UCSWRST;								// Clear SW reset, resume operation
}


//------------------------------------------------------------------------------
// void TI_USCI_I2C_WriteMultiBytes(unsigned char slave_address, unsigned char register_address, unsigned char write_length, unsigned char *Tx_array)
//
// This function is used to write multi bytes to I2C slaver.
//
//
//IN: 	unsigned char slaver_address			=>	i2c address of slaver
//		unsigned char register_address			=>	slaver register address to be writen
//		unsigned char write_length				=>	how many bytes to be writen
//		unsigned char *Tx_array			 		=>	value array pointer to be writen to slave
//------------------------------------------------------------------------------
void TI_USCI_I2C_WriteMultiBytes(unsigned char slave_address, unsigned char register_address, unsigned char write_length, unsigned char *Tx_array)
{
	
  while (UCB0STAT & UCBBUSY);

  Tx_ByteCounter = write_length;

  Tx_Pointer_Temp = Tx_array;

  UCB0I2CSA = slave_address;

  UCB0CTL1 |= UCTR + UCTXSTT;                	 // I2C TX, start condition

  UCB0TXBUF = register_address; 				  //transferring register_address

  while (UCB0CTL1 & UCTXSTT);					 // waiting for slaver address was transferred

  while ((IFG2 & UCB0TXIFG) != UCB0TXIFG);

  while (Tx_ByteCounter)
  	{
	  if (IFG2 & UCB0TXIFG)
	  {
              UCB0TXBUF = *Tx_Pointer_Temp;

              while((IFG2 & UCB0TXIFG) != UCB0TXIFG);

              Tx_Pointer_Temp++;

              Tx_ByteCounter--;
          }
	}
  UCB0CTL1 |= UCTXSTP;                    // I2C stop condition

  IFG2 &= ~UCB0TXIFG;                     // Clear USCI_B0 TX int flag

	
}



//------------------------------------------------------------------------------
// void TI_USCI_I2C_WriteOneByte(unsigned char slave_address, unsigned char register_address, unsigned char register_value)
//
// This function is used to write one byte to I2C slaver.
//
// IN:	unsigned char slaver_address  		=>  i2c address of slaver
//		unsigned char register_address		=>  slaver register address to be writen
//		unsigned char register_value     	=>  value to be writen to slave
//------------------------------------------------------------------------------

void TI_USCI_I2C_WriteOneByte(unsigned char slave_address, unsigned char register_address, unsigned char register_value)
{

  while (UCB0STAT & UCBBUSY);

  UCB0I2CSA = slave_address;

  UCB0CTL1 |= UCTR + UCTXSTT;                                   // I2C TX, start condition

  UCB0TXBUF = register_address;					//transferring register_address

  while (UCB0CTL1 & UCTXSTT);					 // waiting for slaver address was transferred

  while ((IFG2 & UCB0TXIFG) != UCB0TXIFG);

  UCB0TXBUF = register_value;					//transferring one byte value

  while((IFG2 & UCB0TXIFG) != UCB0TXIFG);

  UCB0CTL1 |= UCTXSTP;                    // I2C stop condition

  IFG2 &= ~UCB0TXIFG;                     // Clear USCI_B0 TX int flag

}

//------------------------------------------------------------------------------
// void TI_USCI_I2C_ReadMultiBytes(unsigned char slave_address, unsigned char register_address, unsigned char read_length, unsigned char *Rx_array)
//
// This function is used to read multi bytes from I2C slaver.
//
// IN:	unsigned char slaver_address  		=>  i2c address of slaver
//		unsigned char register_address    	=>  slaver register address to be wrote
//		unsigned char read_length			=>	how many bytes to be read from slaver
//		unsigned char *Rx_array     			=>  Rrray for storing reading bytes
//------------------------------------------------------------------------------

void TI_USCI_I2C_ReadMultiBytes(unsigned char slave_address, unsigned char register_address, unsigned char read_length, unsigned char *Rx_array)
{

  while (UCB0STAT & UCBBUSY);

  Rx_ByteCounter = read_length;

  Rx_Pointer_Temp = Rx_array;

  UCB0I2CSA = slave_address;

  UCB0CTL1 |= UCTR + UCTXSTT;                                  // I2C TX, start condition

  while ((IFG2 & UCB0TXIFG) != UCB0TXIFG);

  UCB0TXBUF = register_address;					//transferring register_address

  while ((IFG2 & UCB0TXIFG) != UCB0TXIFG);

  UCB0CTL1 &= ~UCTR;

  UCB0CTL1 |= UCTXSTT;                                      // I2C RX, restart condition

  while (UCB0CTL1 & UCTXSTT);                               // Loop until I2C STT is sent

  while (Rx_ByteCounter)
  {

    if (Rx_ByteCounter == 1)

    {

      UCB0CTL1 |= UCTXSTP;

      while ((IFG2 & UCB0RXIFG) != UCB0RXIFG);

     *Rx_Pointer_Temp = UCB0RXBUF;                 // Move RX data to address Rx_Buffer

      Rx_ByteCounter--;
    }

    else

    {

       while ((IFG2 & UCB0RXIFG) != UCB0RXIFG);

      *Rx_Pointer_Temp = UCB0RXBUF;                 // Move RX data to address Rx_Buffer

       Rx_Pointer_Temp++;

       Rx_ByteCounter--;

     }

  }

 IFG2 &= ~UCB0RXIFG;                     // Clear USCI_B0 TX int flag
 IFG2 &= ~UCB0TXIFG;

}


//------------------------------------------------------------------------------
// unsigned char TI_USCI_I2C_ReadOneBytes(unsigned char slave_address, unsigned char register_address)
//
// This function is used to read multi bytes from I2C slaver.
//
// IN:	unsigned char slaver_address  		=>  i2c address of slaver
//		unsigned char register_address    	=>  slaver register address to be wrote
//------------------------------------------------------------------------------

unsigned char TI_USCI_I2C_ReadOneByte(unsigned char slave_address, unsigned char register_address)
{

  unsigned char Rx_Buffer;

  while (UCB0STAT & UCBBUSY);

  UCB0I2CSA = slave_address;

  UCB0CTL1 |= UCTR + UCTXSTT;                               // I2C TX, start condition

  while ((IFG2 & UCB0TXIFG) != UCB0TXIFG);

  UCB0TXBUF = register_address;					//transferring register_address

  while ((IFG2 & UCB0TXIFG) != UCB0TXIFG);

  UCB0CTL1 &= ~UCTR;

  UCB0CTL1 |= UCTXSTT;                                   // I2C RX, restart condition

  while (UCB0CTL1 & UCTXSTT);                         // Loop until I2C STT is sent

  UCB0CTL1 |= UCTXSTP;

  while ((IFG2 & UCB0RXIFG) != UCB0RXIFG);

  Rx_Buffer = UCB0RXBUF;                 // Move RX data to Rx_Data

  IFG2 &= ~UCB0TXIFG;

  return(Rx_Buffer);

}

