/**
 *  \file   mdio.c
 *
 *  \brief  MDIO APIs.
 *
 *   This file contains the device abstraction layer APIs for MDIO.
 */

/* Copyright (C) 2013 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 */


#include "hw_reg_access.h"
#include "mdio.h"

/*******************************************************************************
*                       INTERNAL MACRO DEFINITIONS
*******************************************************************************/
#define PHY_REG_MASK                             (0x1FU)
#define PHY_ADDR_MASK                            (0x1FU)
#define PHY_DATA_MASK                            (0xFFFFU)
#define PHY_REG_SHIFT                            (21U)
#define PHY_ADDR_SHIFT                           (16U)

/*******************************************************************************
*                        API FUNCTION DEFINITIONS
*******************************************************************************/

/**
 * \brief   Reads a PHY register using MDIO.
 *
 * \param   baseAddr      Base Address of the MDIO Module Registers.
 * \param   phyAddr       PHY Address.
 * \param   regNum        Register Number to be read.
 * \param   dataPtr       Pointer where the read value shall be written.
 *
 * \return  status of the read \n
 *          TRUE - read is successful.\n
 *          FALSE - read is not acknowledged properly.
 *
 **/
uint32 MDIOPhyRegRead(uint32 baseAddr, uint32 phyAddr,
                            uint32 regNum, volatile uint16 * dataPtr)
{
    /* Wait till transaction completion if any */
    while((HWREG(baseAddr + MDIO_USERACCESS0) & MDIO_USERACCESS0_GO) == MDIO_USERACCESS0_GO)
    { 
	} /* Wait */

    HWREG(baseAddr + MDIO_USERACCESS0)
                           = (MDIO_USERACCESS0_READ | MDIO_USERACCESS0_GO
                              |((regNum & PHY_REG_MASK) << PHY_REG_SHIFT)
                              |((phyAddr & PHY_ADDR_MASK) << PHY_ADDR_SHIFT));

    /* wait for command completion */
    while((HWREG(baseAddr + MDIO_USERACCESS0) & MDIO_USERACCESS0_GO) == MDIO_USERACCESS0_GO)
    { 
	} /* Wait */

    /* Store the data if the read is acknowledged */
    if(((HWREG(baseAddr + MDIO_USERACCESS0)) & MDIO_USERACCESS0_ACK) == MDIO_USERACCESS0_ACK)
    {
        *dataPtr = (uint16)((HWREG(baseAddr + MDIO_USERACCESS0))
                                    & PHY_DATA_MASK);
        return TRUE;
    }

    return FALSE;
}

/**
 * \brief   Writes a PHY register using MDIO.
 *
 * \param   baseAddr      Base Address of the MDIO Module Registers.
 * \param   phyAddr       PHY Address.
 * \param   regNum        Register Number to be read.
 * \param   RegVal        Value to be written.
 *
 * \return  None
 *
 **/
void MDIOPhyRegWrite(uint32 baseAddr, uint32 phyAddr,
                     uint32 regNum, uint16 RegVal)
{
    /* Wait till transaction completion if any */
    while((HWREG(baseAddr + MDIO_USERACCESS0) & MDIO_USERACCESS0_GO) == MDIO_USERACCESS0_GO)
    { 
	} /* Wait */

    HWREG(baseAddr + MDIO_USERACCESS0)
                               = (MDIO_USERACCESS0_WRITE | MDIO_USERACCESS0_GO
                                  |((regNum & PHY_REG_MASK) << PHY_REG_SHIFT)
                                  |((phyAddr & PHY_ADDR_MASK) << PHY_ADDR_SHIFT)
                                  | RegVal);

    /* wait for command completion*/
    while((HWREG(baseAddr + MDIO_USERACCESS0) & MDIO_USERACCESS0_GO) == MDIO_USERACCESS0_GO)
    { 
	} /* Wait */
}
/**
 * \brief   Reads the alive status of all PHY connected to this MDIO.
 *          The bit corresponding to the PHY address will be set if the PHY
 *          is alive.
 *
 * \param   baseAddr      Base Address of the MDIO Module Registers.
 *
 * \return  MDIO alive register state
 *
 **/
uint32 MDIOPhyAliveStatusGet(uint32 baseAddr)
{
    return (HWREG(baseAddr + MDIO_ALIVE));
}

/**
 * \brief   Reads the link status of all PHY connected to this MDIO.
 *          The bit corresponding to the PHY address will be set if the PHY
 *          link is active.
 *
 * \param   baseAddr      Base Address of the MDIO Module Registers.
 *
 * \return  MDIO link register state
 *
 **/
uint32 MDIOPhyLinkStatusGet(uint32 baseAddr)
{
    return (HWREG(baseAddr + MDIO_LINK));
}

/**
 * \brief   Initializes the MDIO peripheral. This enables the MDIO state
 *          machine, uses standard pre-amble and set the clock divider value.
 *
 * \param   baseAddr       Base Address of the MDIO Module Registers.
 * \param   mdioInputFreq  The clock input to the MDIO module
 * \param   mdioOutputFreq The clock output required on the MDIO bus
 * \return  None
 *
 **/
void MDIOInit(uint32 baseAddr, uint32 mdioInputFreq,
              uint32 mdioOutputFreq)
{
   uint32 clkDiv =  (mdioInputFreq/mdioOutputFreq) - 1;
   HWREG(baseAddr + MDIO_CONTROL) = ((clkDiv & MDIO_CONTROL_CLKDIV)
                                     | MDIO_CONTROL_ENABLE 
                                     | MDIO_CONTROL_PREAMBLE
                                     | MDIO_CONTROL_FAULTENB);
}

/***************************** End Of File ***********************************/
