// FlashTestMain.c
// Runs on TM4C1294
// Test the functions provided by FlashProgram.c by writing
// constant values to the flash memory and then finding
// them in the memory viewer in the debugger.
// Daniel Valvano
// October 16, 2014

/* This example accompanies the book
   "Embedded Systems: Real Time Interfacing to Arm Cortex M Microcontrollers",
   ISBN: 978-1463590154, Jonathan Valvano, copyright (c) 2014
   "Embedded Systems: Real-Time Operating Systems for ARM Cortex-M Microcontrollers",
   ISBN: 978-1466468863, Jonathan Valvano, copyright (c) 2014

 Copyright 2014 by Jonathan W. Valvano, valvano@mail.utexas.edu
    You may use, edit, run or distribute this file
    as long as the above copyright notice remains
 THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 VALVANO SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL,
 OR CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 For more information about my classes, my research, and my books, see
 http://users.ece.utexas.edu/~valvano/
 */

#include <stdint.h>
#include "FlashProgram.h"
#include "PLL.h"
#include "Scoreboard.h"
#include "UART.h"

#define GPIO_PORTN10            (*((volatile uint32_t *)0x4006400C))
#define GPIO_PORTN_DIR_R        (*((volatile uint32_t *)0x40064400))
#define GPIO_PORTN_AFSEL_R      (*((volatile uint32_t *)0x40064420))
#define GPIO_PORTN_DEN_R        (*((volatile uint32_t *)0x4006451C))
#define GPIO_PORTN_AMSEL_R      (*((volatile uint32_t *)0x40064528))
#define GPIO_PORTN_PCTL_R       (*((volatile uint32_t *)0x4006452C))
#define SYSCTL_RCGCGPIO_R       (*((volatile uint32_t *)0x400FE608))
#define SYSCTL_RCGCGPIO_R12     0x00001000  // GPIO Port N Run Mode Clock
                                            // Gating Control
#define SYSCTL_PRGPIO_R         (*((volatile uint32_t *)0x400FEA08))
#define SYSCTL_PRGPIO_R12       0x00001000  // GPIO Port N Peripheral Ready
// Flash ROM addresses must be 16k byte aligned, e.g., 0x8000, 0xC000, 0x10000...
#define FLASH                   0x00008000  // location in flash to write; make sure no program code is in this block
#define SCOREBOARD              0x00010000  // location in flash for scoreboard

void DisableInterrupts(void); // Disable interrupts
void EnableInterrupts(void);  // Enable interrupts
long StartCritical (void);    // previous I bit, disable interrupts
void EndCritical(long sr);    // restore I bit to previous value
void WaitForInterrupt(void);  // low power mode

void printscores(void){
  SBEType* Scoreboard;                              // pointer to array of ScoreboardElements holding top scores
  int i;
  UART_OutString("High Scores:\r\n");
  Scoreboard = Scoreboard_Get();
  for(i=0; i<SCOREBOARDSIZE; i=i+1){
    UART_OutChar(Scoreboard[i].first);              // print first initial
    UART_OutChar(Scoreboard[i].middle);             // print middle initial
    UART_OutChar(Scoreboard[i].last);               // print last initial
    UART_OutChar(' ');
    UART_OutChar(' ');
    UART_OutUDec(Scoreboard[i].score);              // print the score
    UART_OutChar('\r');
    UART_OutChar('\n');
  }
}

void getnewscore(void){
  char first, middle, last;
  uint32_t score;
  UART_OutString("Enter three letters: ");
  // get the first initial
  first = UART_InChar();
  if(first >= ' '){
    UART_OutChar(first);                            // echo the character
  }
  while((first < 'A') || (('Z' < first ) && (first < 'a')) || ('z' < first)){
    if(first >= ' '){
      UART_OutChar(BS);                             // backspace
    }
    first = UART_InChar();                          // repeat until a letter is pressed
    if(first >= ' '){
      UART_OutChar(first);                          // echo the character
    }
  }
  if(('a' <= first) && (first <= 'z')){
    first = first - ' ';                            // convert to upper case
  }
  // get the middle initial
  middle = UART_InChar();
  if(middle >= ' '){
    UART_OutChar(middle);                           // echo the character
  }
  while((middle < 'A') || (('Z' < middle ) && (middle < 'a')) || ('z' < middle)){
    if(middle >= ' '){
      UART_OutChar(BS);                             // backspace
    }
    middle = UART_InChar();                         // repeat until a letter is pressed
    if(middle >= ' '){
      UART_OutChar(middle);                         // echo the character
    }
  }
  if(('a' <= middle) && (middle <= 'z')){
    middle = middle - ' ';                          // convert to upper case
  }
  // get the last initial
  last = UART_InChar();
  if(last >= ' '){
    UART_OutChar(last);                             // echo the character
  }
  while((last < 'A') || (('Z' < last ) && (last < 'a')) || ('z' < last)){
    if(last >= ' '){
      UART_OutChar(BS);                             // backspace
    }
    last = UART_InChar();                           // repeat until a letter is pressed
    if(last >= ' '){
      UART_OutChar(last);                           // echo the character
    }
  }
  if(('a' <= last) && (last <= 'z')){
    last = last - ' ';                              // convert to upper case
  }
  // get the score
  UART_OutString("\r\nEnter your score as a number and hit ENTER: ");
  score = UART_InUDec();
  UART_OutChar('\r');
  UART_OutChar('\n');
  Scoreboard_Record(first, middle, last, score);
}

uint32_t DataArray[10];
uint32_t volatile *FlashPtr = (uint32_t volatile*)FLASH;
int SuccessfulWrites;
int main(void){
  uint32_t errors;
  int i;
  PLL_Init();                                       // set system clock to 120 MHz
  UART_Init();                                      // initialize UART
  SYSCTL_RCGCGPIO_R |= SYSCTL_RCGCGPIO_R12;         // activate clock for Port N
  while((SYSCTL_PRGPIO_R&SYSCTL_PRGPIO_R12) == 0){};// allow time for clock to stabilize
  GPIO_PORTN_DIR_R |= 0x03;                         // make PN1-0 out (PN1-0 built-in LED1-2)
  GPIO_PORTN_AFSEL_R &= ~0x03;                      // disable alt funct on PN1-0
  GPIO_PORTN_DEN_R |= 0x03;                         // enable digital I/O on PN1-0
                                                    // configure PN1-0 as GPIO
  GPIO_PORTN_PCTL_R = (GPIO_PORTN_PCTL_R&0xFFFFFF00)+0x00000000;
  GPIO_PORTN_AMSEL_R &= ~0x03;                      // disable analog functionality on PN1-0
  DataArray[0] = 0x00001111;
  DataArray[1] = 0x00022220;
  DataArray[2] = 0x00333300;
  DataArray[3] = 0x04444000;
  DataArray[4] = 0x55550000;
  DataArray[5] = 0x66600007;
  DataArray[6] = 0x77000088;
  DataArray[7] = 0x80000999;
  DataArray[8] = 0x0000AAAA;
  DataArray[9] = 0x000BBBB0;
  Flash_Erase(FLASH);                               // erase 0x00008000 through 0x0000BFFC
  Flash_Write(FLASH + 0, 0x10101010);               // write to location 0x00008000
  Flash_Write(FLASH + 4, 0x0BADBEEF);               // write to location 0x00008004
  Flash_Write(FLASH + 8, 0xBEEFF00D);               // write to location 0x00008008
  Flash_Write(FLASH + 0x4000, 0x45464153);          // write to location 0x0000C000; this value should persist after this program is run (see note at bottom of file)
  Flash_Write(FLASH + 0x3FFC, 0x454E4F47);          // write to location 0x0000BFFC
  Flash_Write(FLASH + 0x3FF8, 0x0FEDCBA0);          // write to location 0x0000BFF8
  GPIO_PORTN10 = 0x00;                              // LED1 off; LED2 off
  SuccessfulWrites = Flash_WriteArray(DataArray, FLASH + 9, 10); // invalid address
  GPIO_PORTN10 = 0x02;                              // LED1 on; LED2 off
  SuccessfulWrites = Flash_WriteArray(DataArray, FLASH + 12, 10);// use scope to measure PN1 high time (1,472 usec)
  GPIO_PORTN10 = 0x00;                              // LED1 off; LED2 off
  SuccessfulWrites = Flash_FastWrite(DataArray, FLASH + 124, 10); // invalid address
  GPIO_PORTN10 = 0x01;                              // LED1 off; LED2 on
  SuccessfulWrites = Flash_FastWrite(DataArray, FLASH + 128, 10); // use scope to measure PN0 high time (732 usec)
  GPIO_PORTN10 = 0x00;                              // LED1 off; LED2 off
  // memory test
  errors = 0;
  UART_OutString("\r\n");
  if(FlashPtr[0] != 0x10101010){
    UART_OutString("Error at 0x");
    UART_OutUHex(FLASH + 0);
    UART_OutString(": expected 0x10101010 but actually 0x");
    UART_OutUHex(FlashPtr[0]);
    UART_OutString(".\r\n");
    errors = errors + 1;
  }
  if(FlashPtr[1] != 0x0BADBEEF){
    UART_OutString("Error at 0x");
    UART_OutUHex(FLASH + 4);
    UART_OutString(": expected 0x0BADBEEF but actually 0x");
    UART_OutUHex(FlashPtr[1]);
    UART_OutString(".\r\n");
    errors = errors + 1;
  }
  if(FlashPtr[2] != 0xBEEFF00D){
    UART_OutString("Error at 0x");
    UART_OutUHex(FLASH + 8);
    UART_OutString(": expected 0xBEEFF00D but actually 0x");
    UART_OutUHex(FlashPtr[2]);
    UART_OutString(".\r\n");
    errors = errors + 1;
  }
  for(i=0; i<10; i=i+1){
    if(FlashPtr[i+3] != DataArray[i]){
      UART_OutString("Error at 0x");
      UART_OutUHex(FLASH + 12 + 4*i);
      UART_OutString(": expected 0x");
      UART_OutUHex(DataArray[i]);
      UART_OutString(" but actually 0x");
      UART_OutUHex(FlashPtr[i+3]);
      UART_OutString(".\r\n");
      errors = errors + 1;
    }
  }
  for(i=0; i<10; i=i+1){
    if(FlashPtr[i+32] != DataArray[i]){
      UART_OutString("Error at 0x");
      UART_OutUHex(FLASH + 128 + 4*i);
      UART_OutString(": expected 0x");
      UART_OutUHex(DataArray[i]);
      UART_OutString(" but actually 0x");
      UART_OutUHex(FlashPtr[i+32]);
      UART_OutString(".\r\n");
      errors = errors + 1;
    }
  }
  if(FlashPtr[4094] != 0x0FEDCBA0){
    UART_OutString("Error at 0x");
    UART_OutUHex(FLASH + 0x3FF8);
    UART_OutString(": expected 0x0FEDCBA0 but actually 0x");
    UART_OutUHex(FlashPtr[4094]);
    UART_OutString(".\r\n");
    errors = errors + 1;
  }
  if(FlashPtr[4095] != 0x454E4F47){
    UART_OutString("Error at 0x");
    UART_OutUHex(FLASH + 0x3FFC);
    UART_OutString(": expected 0x454E4F47 but actually 0x");
    UART_OutUHex(FlashPtr[4095]);
    UART_OutString(".\r\n");
    errors = errors + 1;
  }
  UART_OutString("Memory test complete.  ");
  UART_OutUDec(errors);
  UART_OutString(" errors.\r\n\r\n");
  // scoreboard test
  Scoreboard_Init(SCOREBOARD);
  GPIO_PORTN10 = 0x02;                              // LED1 on; LED2 off
  while(1){
    printscores();
    getnewscore();
    GPIO_PORTN10 ^= 0x03;                           // toggle the LEDs
  }
}

// Note on writing to location 0x0000C000:
// The way flash memory works is that an erase sets all bits
// in a block to 1, and writing clears desired bits to 0.
// You cannot "write" individual bits back to 1's.  When
// FlashTestMain is compiled and run, there may already be
// data stored at 0x0000C000.  After the line
// Flash_Write(FLASH + 0x4000, 0x45464153);
// the contents of flash location 0x0000C000 will be the
// bitwise AND of the previous data contents and 0x45464153.
//
// To deal with this issue, you should erase the block once. E.g., 
// Flash_Erase(0x0000C000);   // erase 0x0000C000 through 0x0000FFFF
  
// Another way to erase Flash is to use the Keil debugger
// 1) Open the "Flash" menu and select "Configure Flash Tools..."
// 2) Select the "Utilities" sub-tab
// 3) Click the "Setting" button near the pull-down menu that says
//  "Stellaris ICDI"
// 4) Click the radio button that says "Mass Erase (erase entire Flash)"
// 5) Click both "OK" buttons
// 6) Clear the flash and download your program with the regular
//  "Download" button.  In my experience, the debugger failed
//  after the full erase finished.  If this happens, repeat
//  Steps 1, 2, and 3.  Then in Step 4, click the radio button
//  that says "Page Erase (erase only necessary pages)".  You
//  can also try the ultra-scientific process of repeatedly
//  disconnecting and re-connecting the USB cable or restarting.
//
// This process guarantees that the flash memory at 0x0000.C000 is clear
//  (set to all 1's) in case a previously loaded program left data there.
